#ifndef QL_API_QL_INIT_H
#define QL_API_QL_INIT_H

namespace QL
{
    class arg_spec;
    class enum_spec;

    class symtab;

    using function_t = void (...);

    QL_API arg_spec	 ctxt_arg();

    QL_API symtab	&add_module(symtab	&s,
				    const char	*name);

    QL_API void	add_object_type(symtab		&s,
				const char	*object_type,
				const char	*superclass	= 0,
				const char	*category	= 0);

    QL_API void add_enum_type(symtab		&s,
			      const char	*type_name,
			      int_t		 n,
			      const enum_spec	*spec,
			      const char	*category,
			      bool		 class_enum);

    // Only for ms-abi
    QL_API void add_enum_type(symtab		&s,
			      const char	*type_name,
			      int_t		 n,
			      const enum_spec	*spec,
			      const char	*category	= 0);

    QL_API void	add_type_name(symtab		&s,
			      const char	*type_name,
			      const char	*type,
			      const char	*category	= 0);

    QL_API void	add_func(symtab			&s,
			 function_t		*f,
			 const char		*name,
			 const char		*category,
			 const arg_spec		&ret_type,
			 int_t			 n_args,
			 const arg_spec		*args,
			 const char		*com_name	= 0);

    QL_API void add_constructor(symtab		&s,
				function_t	*f,
				const char	*object_type,
				int_t		 n_args,
				const arg_spec	*args,
				bool		 create_func,
				const char	*cf_com_name	= 0);

    QL_API void	add_mem_func(symtab		&s,
			     function_t		*f,
			     const char		*object_type,
			     const char		*name,
			     const arg_spec	&ret_type,
			     int_t		 n_args,
			     const arg_spec	*args,
			     const char		*com_name	= 0,
			     bool		 hidden		= false);


    QL_API void add_dyncast_func(symtab		&s,
				 function_t	*dyncast,
				 const char	*object_type);

    template <typename T>
    inline handle dyncast(object &o)
    {
	return handle(dynamic_cast<T*>(&o));
    }


    QL_API void set_qlc_marshalling(symtab	&s,
				    const char	*object_type,
				    handle (*marshal_object)(object&));

    QL_API void seal_object_type(symtab		&s,
				 const char	*object_type);

    QL_API void add_global_var(symtab		&s,
			       const char	*name,
			       const char	*category,
			       const char	*type_name);

    QL_API void add_constant(symtab		&s,
			     const char		*name,
			     const char		*category,
			     const char		*type_name,
			     num_val_t		 n_val,
			     object		*o_val);

    QL_API void add_simple_enum(symtab			&s,
				const char		*enum_type,
				int_t			 n,
				const char		*prefix,
				const char *const	*str,
				const char *const	*desc,
				const char		*category	= 0);

    QL_API void add_simple_enum(symtab			&s,
				const char		*enum_type,
				int_t			 n,
				const char		*prefix,
				const char *const	*str,
				const char		*category	= 0);

    QL_API void allow_undeclared_mfuncs(symtab		&s,
					const char	*object_type);
}

class QL::arg_spec
{
public:
    explicit arg_spec(const char *type		= 0,
		      const char *name		= 0,
		      const char *default_value	= 0)
	: type(type),
	  name(name),
	  default_value(default_value),
	  hidden(false),
	  is_ctxt(false),
	  pad1(0),
	  nullable(false),
	  pad2(0),
	  out(false),
	  pad3(0),
	  pad4(0)
    {}

public:
    const char	*type;
    const char	*name;
    const char	*default_value;
    unsigned	 hidden			: 1;
    unsigned	 is_ctxt		: 1;
    unsigned	 pad1			: 6;
    unsigned	 nullable		: 1;
    unsigned	 pad2			: 7;
    unsigned	 out			: 1;
    unsigned	 pad3			: 7;
    unsigned	 pad4			: 8;
};

class QL::enum_spec
{
public:
    explicit enum_spec(const char	*id	= 0,
		       const char	*str	= 0,
		       const char	*desc	= 0)
	: id(id),
	  str(str),
	  desc(desc)
    {}

public:
    const char	*id;
    const char	*str;
    const char	*desc;
};

namespace QL
{
    inline arg_spec R(const char		*type)
    {
	return arg_spec(type);
    }

    inline arg_spec A(const char		*type,
		      const char		*name,
		      const char		*default_value	= 0)
    {
	return arg_spec(type, name, default_value);
    }

    inline arg_spec A_0(const char		*type,
			const char		*name,
			const char		*default_value	= 0)
    {
	arg_spec	a(type, name, default_value);

	a.nullable = true;

	return a;
    }

    inline arg_spec A_out(const char		*type,
			  const char		*name,
			  const char		*default_value	= 0)
    {
	arg_spec	a(type, name, default_value);

	a.out = a.nullable = true;

	return a;
    }

    inline arg_spec A_out_not0(const char	*type,
			       const char	*name,
			       const char	*default_value	= 0)
    {
	arg_spec	a(type, name, default_value);

	a.out = true;
	a.nullable = false;

	return a;
    }

    inline arg_spec A_ctxt()
    {
	return ctxt_arg();
    }

    inline arg_spec A_pos()
    {
	arg_spec	a("integer", "pos", "$pos");

	a.hidden = true;

	return a;
    }

    /**/

    inline arg_spec arg_nullable(const char *type,
				 const char *name,
				 const char *default_value = 0)
    {
	return A_0(type, name, default_value);
    }

    inline arg_spec arg_out(const char *type,
			    const char *name,
			    const char *default_value = 0)
    {
	return A_out(type, name, default_value);
    }
}

#include <initializer_list>

namespace QL
{
    using arg_spec_list = std::initializer_list<arg_spec>;

    template <typename Ret, typename ...Args>
    inline void
    add_func(symtab			 &s,
	     Ret			(*f)(Args...),
	     const char			 *name,
	     const char			 *category,
	     const arg_spec		 &ret_type,
	     arg_spec_list		  args		= arg_spec_list(),
	     const char			 *com_name	= 0)
    {
	add_func(s, reinterpret_cast<function_t*>(f), name, category,
		 ret_type, args.size(), args.begin(), com_name);
    }

    template <typename Ret, typename ...Args>
    inline void
    add_constructor(symtab		 &s,
		    Ret			(*f)(Args...),
		    const char		 *object_type,
		    arg_spec_list	  args		= arg_spec_list(),
		    bool		  create_func	= false,
		    const char		 *cf_com_name	= 0)
    {
	add_constructor(s, reinterpret_cast<function_t*>(f), object_type,
			args.size(), args.begin(), create_func, cf_com_name);
    }

    template <typename Ret, typename ...Args>
    inline void
    add_mem_func(symtab			 &s,
		 Ret			(*f)(Args...),
		 const char		 *object_type,
		 const char		 *name,
		 const arg_spec		 &ret_type,
		 arg_spec_list		  args		= arg_spec_list(),
		 const char		 *com_name	= 0,
		 bool			  hidden	= false)
    {
	add_mem_func(s, reinterpret_cast<function_t*>(f), object_type, name,
		     ret_type, args.size(), args.begin(), com_name, hidden);
    }

    template <typename T>
    inline void
    set_qlc_marshalling(symtab		 &s,
			const char	 *object_type,
			handle_t<T>	(*marshal)(T&))
    {
	set_qlc_marshalling(s, object_type,
			    reinterpret_cast<handle (*)(object&)>(marshal));
    }

    template <typename T>
    inline void
    set_qlc_marshalling(symtab		 &s,
			const char	 *object_type,
			handle_t<T>	(*marshal)(const T&))
    {
	set_qlc_marshalling(s, object_type,
			    reinterpret_cast<handle (*)(object&)>(marshal));
    }
}

#endif /* QL_API_QL_INIT_H */
