#ifndef QL_API_QL_OBJECT_H
#define QL_API_QL_OBJECT_H

#if defined(__clang__) && !defined(_MSC_VER)
#  define QL_NATIVE_API		1
#  define QL_TRIVIAL_ABI	[[clang::trivial_abi]]
#else
#  define QL_NATIVE_API		0
#  define QL_TRIVIAL_ABI
#endif

#include <utility>

namespace QL
{
    using int32_t	= int;
    using int64_t	= long long;

#ifdef _WIN64
    using int_t		= int64_t;
#else
    using int_t		= int32_t;
#endif

    using number_t	= double;
    using logical_t	= signed char;
    using date_t	= int32_t;
    using month_t	= int32_t;
    using timestamp_t	= int64_t;
    using enum_t	= int32_t;

    class object;
    class QL_TRIVIAL_ABI handle;
    template <typename T> class QL_TRIVIAL_ABI handle_t;

    union num_val_t
    {
    public:
	int_t		 i;
	number_t	 n;
	logical_t	 l;
	date_t		 d;
	month_t		 m;
	timestamp_t	 t;
	enum_t		 e;
    };

    union ref_t
    {
    public:
	int_t		*i;
	number_t	*n;
	logical_t	*l;
	date_t		*d;
	month_t		*m;
	timestamp_t	*t;
	enum_t		*e;
	handle		*o;
    };

    enum error_t
    {
	E_NONE,
	E_UNSPECIFIC,
	E_CONSTRAINT,
	E_NULL,
	E_RANGE,
	E_ENUM,
	E_INVALID_ARG,
	E_NAME_LOOKUP,
	E_DATABASE,
	E_CALC,
	E_NO_DATA,
	E_INIT,
	E_REALTIME,
	E_IO,
	E_PARSE,
	E_TIMEOUT,
	E_N,
	/* These are not catchable in qlang. */
	E_ABORTED = E_N,
	E_OOM
    };
}

#ifndef QL_OBJ_API
#  define QL_OBJ_API QL_API
#endif

class QL::object
{
protected:
    QL_OBJ_API object() noexcept;
    QL_OBJ_API virtual ~object();

public:
    QL_OBJ_API static void do_delete(object*) noexcept;

#if QL_NATIVE_API
    static void	add_ref(object *o) noexcept
    { if (o) __atomic_fetch_add(&o->n_refs, 1, __ATOMIC_SEQ_CST); }

    static void	release(object *o) noexcept
    { if (o && __atomic_fetch_sub(&o->n_refs, 1, __ATOMIC_SEQ_CST) - 1 == 0)
	    do_delete(o); }
#else
    // Bogons for rt proxy
    static void	add_ref(object*) noexcept;
    static void	release(object*) noexcept;
#endif

private:
    object(const object&) = delete;
    object &operator=(const object&) = delete;

private:
    int		 n_refs;
public:
    unsigned	 aux_bits; // private
};

class QL_TRIVIAL_ABI QL::handle
{
    template <typename>
    friend class handle_t;

public:
    handle() noexcept
	: o_(0)
	{}

    ~handle()
    { set_o(0); }

    handle(const handle &that) noexcept
	: handle()
    { *this = that; }

    handle &operator=(const handle &that) noexcept
    { set_o(that.o_); return *this; }

    handle(handle &&that) noexcept
	: o_(that.o_)
    { that.o_ = 0; }

    handle &operator=(handle &&that) noexcept
    {
	if (this != &that) {
	    set_o(0);
	    o_ = that.o_;
	    that.o_ = 0;
	}

	return *this;
    }

    template <typename O>
    handle(const handle_t<O> &that) noexcept;

    template <typename O>
    handle &operator=(const handle_t<O>&) noexcept;

    template <typename O>
    handle(handle_t<O>&&) noexcept;

    template <typename O>
    handle &operator=(handle_t<O>&&) noexcept;

    explicit handle(object *o) noexcept
	: handle()
    { *this = o; }

    handle &operator=(object *o) noexcept
    { set_o(o); return *this; }

    bool operator!() const noexcept		{ return !o_; }
    explicit operator bool() const noexcept	{ return !!*this; }

#if QL_NATIVE_API
    void set_o(object *o) noexcept
    {
	object::add_ref(o);
	object::release(o_);
	o_ = o;
    }
#else
    QL_API void	 set_o(object*) noexcept;
#endif

    object	*get_o() const noexcept		{ return o_; }

    void	 set(object *o) noexcept	{ set_o(o); }
    object	*get() const noexcept		{ return o_; }

    object	&operator*() const noexcept	{ return *o_; }

    object *steal() noexcept
    { object *o = o_; o_ = 0; return o; }

    static handle unsteal(object *o) noexcept
    { handle h; h.o_ = o; return h; }

private:
    object	*o_;
};

template <typename O>
class QL_TRIVIAL_ABI QL::handle_t
{
    friend class handle;

    template <typename>
    friend class handle_t;

public:
    handle_t() noexcept
    {}

    handle_t(const handle_t &that) noexcept
	: h_(that.h_)
    {}

    handle_t &operator=(const handle_t &that) noexcept
    { h_ = that.h_; return *this; }

    handle_t &operator=(handle_t &&that) noexcept
    { h_ = std::move(that.h_); return *this; }

    handle_t(handle_t &&that) noexcept
	: h_(std::move(that.h_))
    {}

    template <typename O2>
    handle_t(const handle_t<O2> &that) noexcept
	: h_(that.h_)
    { O *o = that.get(); (void)o; }

    template <typename O2>
    handle_t &operator=(const handle_t<O2> &that) noexcept
    { O *o = that.get(); (void)o; h_ = that.h_; return *this; }

    template <typename O2>
    handle_t(handle_t<O2> &&that) noexcept
	: h_(std::move(that.h_))
    { O *o = that.get(); (void)o; }

    template <typename O2>
    handle_t &operator=(handle_t<O2> &&that) noexcept
    { O *o = that.get(); (void)o; h_ = std::move(that.h_); return *this; }

    explicit handle_t(O *o) noexcept
	: h_(o)
    {}

    handle_t &operator=(O *o) noexcept
    { h_ = o; return *this; }

    bool operator!() const noexcept		{ return !h_; }
    explicit operator bool() const noexcept	{ return !!*this; }

    void	 set(O *o) noexcept	{ h_.set(o); }
    O		*get() const noexcept	{ return static_cast<O*>(h_.get()); }

    O		&operator*() const noexcept	{ return *get(); }
    O		*operator->() const noexcept	{ return  get(); }

    O *steal() noexcept
    { return static_cast<O*>(h_.steal()); }

    static handle_t unsteal(O *o) noexcept
    { handle_t h; h.h_ = handle::unsteal(o); return h; }

 private:
    handle	 h_;
};

template <typename O>
inline QL::handle::handle(const handle_t<O> &that) noexcept
    : handle(that.h_)
{}

template <typename O>
inline QL::handle &QL::handle::operator=(const handle_t<O> &that) noexcept
{ return *this = that.h_; }

template <typename O>
inline QL::handle::handle(handle_t<O> &&that) noexcept
    : handle(std::move(that.h_))
{}

template <typename O>
inline QL::handle &QL::handle::operator=(handle_t<O> &&that) noexcept
{ return *this = std::move(that.h_); }

namespace QL
{
    inline bool operator==(const handle &p1, const handle &p2) noexcept
    { return p1.get() == p2.get(); }

    inline bool operator!=(const handle &p1, const handle &p2) noexcept
    { return p1.get() != p2.get(); }

    inline bool operator<(const handle &p1, const handle &p2) noexcept
    { return p1.get() < p2.get(); }

    template <typename O1, typename O2>
    inline bool operator==(const handle_t<O1> &p1,
			   const handle_t<O2> &p2) noexcept
    { return p1.get() == p2.get(); }

    template <typename O1, typename O2>
    inline bool operator!=(const handle_t<O1> &p1,
			   const handle_t<O2> &p2) noexcept
    { return p1.get() != p2.get(); }

    template <typename O1, typename O2>
    inline bool operator<(const handle_t<O1> &p1,
			  const handle_t<O2> &p2) noexcept
    { return p1.get() < p2.get(); }
}

#endif /* QL_API_QL_OBJECT_H */
