/** Module event
 *
 * For handling events in workspace via LOCAL source
 *
 * Requires:
 *  - local3x.qrt named 'LOCAL'
 *  - logging.ql (found in common_ql/logging)
 *
 * Idea:
 * To control execution of out functions attached to tables and graphs. Calling
 * is done from an out function to the next by referring to the next function's
 * name. A function can call as many other functions as desired.
 *
 * All functions to be involved must register themselves with their name, so
 * they can be called from another function. Function calling is always done by
 * the name of the out function to run.
 *
 * NOTE: The out functions should never throw exceptions, doing this will
 *       break the chain of LOCAL source triggers. Instead of throwing
 *       exception, use log_message(LOG_ERROR, ...) or return the error
 *       if possible.


 * Example:
 *
 * out void update_swap()
 * {
 *   event.this_function("update_swap"); // Always start with this line
 *   ...
 *   event.call_function("show_swaps"); // Show swaps again (they have changed...)
 *   event.call_function("show_prices"); // Show (recalculated) prices again after update
 *   ...
 *   event.next();  // Always do this before end of function and any return statement in the function.
 * }
 *
 * out matrix(string) show_prices()
 * {
 *   event.this_function("show_prices"); // Always, so it can be called. Here it is called from update_swap()
 *   ...
 *   event.next(); // Even if this function has no event.call_function(...) it must call event.next(), since there could be other calls waiting. Here the call to show_swaps is probably waiting to happen.
 * }
 *

 * -------------------------------------------
 * Best practice for workspaces with event lib
 * -------------------------------------------
 *
 * Let the code expression in the workspace only contain short out
 * functions. Put all implementation in a lib file, this will make
 * all git work significantly easier, since qlw files are a hassle.
 * The lib file puts everything in module `wsp`.
 *
 * Here is a typical out function:
 *
 *

// How to wrap a void function
out void update_swap(logical with_foo)
{
	// Always start by telling event lib that a function starts.
	event.this_function("update_swap");
	// Put the call to implementation in a try catch, since we
	// can not ever allow exceptions from the out functions.
	try {
		// This is the call to the implementation.
		wsp.update_swap(with_foo);
	} catch {
		// Ask the event lib to notify user about the
		// error. This is typically done via log_message.
		event.err_void(err);
	}
	// Call functions that should always run after
	// this function. (event.call_function is also often called
	// in the implementation function wsp.update_swap, since
	// it may depend on what happens there.
	event.call_function("show_swaps");
	event.call_function("show_prices");
	// This is always the last thing we do in an out function.
	event.next();
}

// How to wrap a non-void function
out matrix(string) show_prices()
{
	// Always start by telling event lib that a function starts.
	event.this_function("show_prices");

	// Create a variable for what we will return.
	matrix(string) ret;
	// Put the call to implementation in a try catch, since we
	// can not ever allow exceptions from the out functions.
	try {
		// This is the call to the implementation.
		ret = wsp.show_prices();
	} catch {
		// Ask the event lib to set the return variable.
		// "m_s" means matrix(string), choose an appropriate
		// function for your return type. The event lib will
		// put the error in the return value if it can, otherwise
		// return something empty and log the error.
		// Note that this is only for your convenience, you are
		// very welcome to construct your own return value.
		ret = event.err_m_s(err);
	}
	// This is always the last thing we do in an out function, before
	// returning.
	event.next();
	// Now we can return from the function.
	return ret;
}


 * Debugging events in your workspace:
 *
 * A convenient way of seening what the lib does in your workspace is to enable
 * history keeping by adding this line to your expression:
 *
 *   event.preset_obj __e = event.remember();
 *
 * and attaching this out function to a table:
 *
 * out matrix(text_rgb) history_matrix(logical reversed, logical auto_trigger, logical auto_clear)
 * {
 *     return event.history_matrix(reversed, auto_trigger, auto_clear);
 * }
 *
 * It will produce a formatted table of events in your workspace. `reversed`
 * puts newest events on top, `auto_trigger` redraws the table after each
 * event, `auto_clear` makes all invocations show only what has happened since
 * the last invocation. It's discouraged to combine auto_clear and
 * auto_trigger.
 *

 * With logging:
 *  Add loggers with context from event.logging_context().


 * Details:
 * The function calls are kept as a queue, so the next function to be called by event.next()
 * will be the last one requested by event.call_function(...). In the example
 * above show_prices is called first, then any function requested by
 * show_prices, and then show_swaps.
 *
 * No call to a function which is not yet registered will be made. A function
 * is registered by calling event.this_function(func_name). This is to prevent
 * calls to functions that aren't event attached to any table (calling those
 * would bring the chain of calls to event.next() to a halt since the
 * unattached function would not be run).
 *
 */

module event {

// This is stuff copied from qu.ql, so we don't need qu.ql anymore.
module __qu_stuff__ {

logical equal_nbl(string option(nullable) lhs, string option(nullable) rhs)
{
	if (null(lhs) && null(rhs))
		return true;
	if (null(lhs) || null(rhs))
		return false;
	return lhs == rhs;
}

class queue_base {
public:
	logical empty();
	integer size();

protected:
	queue_base();

	integer start_;
	integer end_;
	integer buf_size_;

	logical __full();
	integer __advance(integer vec_ptr);

private:
	virtual void __expand() = 0;
};
integer queue_initial_size = 10;

queue_base.queue_base()
: start_(0), end_(0), buf_size_(queue_initial_size)
{}

logical queue_base.empty() = start_ == end_;

integer queue_base.size()
{
	if (start_ <= end_) {
		return end_ - start_;
	} else {
		return (buf_size_ - start_) + end_;
	}
}

logical queue_base.__full()
{
	integer end_advanced = __advance(end_);
	return end_advanced == start_;
}
integer queue_base.__advance(integer vec_ptr)
{
	if (vec_ptr == buf_size_ - 1) {
		return 0;
	} else {
		return vec_ptr + 1;
	}
}

public class queue_int : public queue_base option(category: "Utilities Enhanced/Stack and Queue") {
public:
	queue_int();
	virtual void push(integer x);
	virtual integer pop();
	integer top();
	vector(integer) contents();
	vector(integer) contents_reverse();
	vector(string) impl_content();

protected:
	void exec_push(integer x);
	integer exec_pop();

private:
	vector(integer) buffer_;

	virtual void __expand();
};
queue_int.queue_int()
{
	resize(buffer_, buf_size_);
}
void queue_int.push(integer val) { exec_push(val); }
integer queue_int.pop() = exec_pop();
integer queue_int.top()
{
	if (this.empty()) {
		throw (E_UNSPECIFIC, "queue is empty");
	}
	return buffer_[start_];
}
vector(integer) queue_int.contents()
{
	if (this.empty()) {
		return vector(i:0;0);
	} else if (start_ < end_) {
//		return buffer_[start_:end_-1];
		return sub_vector(buffer_, start_, end_-start_);
	} else if (end_ == 0) {
		//return buffer_[start_:buf_size_-1];
		return sub_vector(buffer_, start_, buf_size_-start_);
	} else {
		//return concat(buffer_[start_:buf_size_-1], buffer_[0:end_-1]);
		return concat(sub_vector(buffer_, start_, buf_size_-start_),
					  sub_vector(buffer_, 0, end_));
	}
}
vector(integer) queue_int.contents_reverse()
{
	if (this.empty()) {
		return vector(i:0;0);
	} else if (start_ < end_) {
		integer n = this.size();
		vector(integer) r[n];
		for (integer i = 0; i < n; i++) {
			r[i] = buffer_[end_-i-1];
		}
		return r;
	} else if (end_ == 0) {
		integer n = this.size();
		vector(integer) r[n];
		for (integer i = 0; i < n; i++) {
			r[i] = buffer_[buf_size_-i-1];
		}
		return r;
	} else {
		integer n_last = end_;
		integer n_first = buf_size_- start_;
		vector(integer) r[n_last + n_first];
		for (integer i = 0; i < n_last; i++) {
			r[i] = buffer_[end_ - i - 1];
		}
		for (integer i = 0; i < n_first; i++) {
			r[i+n_last] = buffer_[buf_size_ - i - 1];
		}
		return r;
	}
}
vector(string) queue_int.impl_content() =
	concat(
		v_strcat().cat(["start", "end"]).cat(": ").cat(str([start_, end_])).get(),
		str(buffer_));
void queue_int.exec_push(integer val)
{
	if (__full()) {
		__expand();
	}
	buffer_[end_] = val;
	end_ = __advance(end_);
}
integer queue_int.exec_pop()
{
	if (this.empty()) {
		throw (E_UNSPECIFIC, "queue is empty");
	}
	integer val = buffer_[start_];
	start_ = __advance(start_);
	return val;
}
void queue_int.__expand()
{
	vector(integer) content = this.contents();
	buf_size_ *= 2;
	resize(buffer_, buf_size_);
	integer content_sz = v_size(content);
	for (integer i = 0; i < content_sz; i++) {
		buffer_[i] = content[i];
	}
	start_ = 0;
	end_ = v_size(content);
}


public class queue_str : public queue_base option(category: "Utilities Enhanced/Stack and Queue") {
public:
	queue_str();
	virtual void push(string option(nullable) s);
	virtual string pop();
	string top();
	vector(string) contents();
	vector(string) contents_reverse();
	vector(string) impl_content();

protected:
	void exec_push(string option(nullable) s);
	string exec_pop();

private:
	vector(string) buffer_;

	virtual void __expand();
};
queue_str.queue_str()
{
	resize(buffer_, buf_size_);
}
void queue_str.push(string option(nullable) s) { exec_push(s); }
string queue_str.pop() = exec_pop();
string queue_str.top()
{
	if (this.empty()) {
		throw (E_UNSPECIFIC, "queue is empty");
	}
	return buffer_[start_];
}
vector(string) queue_str.contents()
{
	if (this.empty()) {
		return vector(i:0;"");
	} else if (start_ < end_) {
//		return buffer_[start_:end_-1];
		return sub_vector(buffer_, start_, end_-start_);
	} else if (end_ == 0) {
		//return buffer_[start_:buf_size_-1];
		return sub_vector(buffer_, start_, buf_size_-start_);
	} else {
		//return concat(buffer_[start_:buf_size_-1], buffer_[0:end_-1]);
		return concat(sub_vector(buffer_, start_, buf_size_-start_),
					  sub_vector(buffer_, 0, end_));
	}
}
vector(string) queue_str.contents_reverse()
{
	if (this.empty()) {
		return vector(i:0;"");
	} else if (start_ < end_) {
		integer n = this.size();
		vector(string) r[n];
		for (integer i = 0; i < n; i++) {
			r[i] = buffer_[end_-i-1];
		}
		return r;
	} else if (end_ == 0) {
		integer n = this.size();
		vector(string) r[n];
		for (integer i = 0; i < n; i++) {
			r[i] = buffer_[buf_size_-i-1];
		}
		return r;
	} else {
		integer n_last = end_;
		integer n_first = buf_size_- start_;
		vector(string) r[n_last + n_first];
		for (integer i = 0; i < n_last; i++) {
			r[i] = buffer_[end_ - i - 1];
		}
		for (integer i = 0; i < n_first; i++) {
			r[i+n_last] = buffer_[buf_size_ - i - 1];
		}
		return r;
	}
}
vector(string) queue_str.impl_content() = concat(v_strcat().cat(["start", "end"]).cat(": ").cat(str([start_, end_])).get(), clone_vector(buffer_));
void queue_str.exec_push(string option(nullable) s)
{
	if (__full()) {
		__expand();
	}
	buffer_[end_] = s;
	end_ = __advance(end_);
}
string queue_str.exec_pop()
{
	if (this.empty()) {
		throw (E_UNSPECIFIC, "queue is empty");
	}
	string s = buffer_[start_];
	start_ = __advance(start_);
	return s;
}
void queue_str.__expand()
{
	vector(string) content = this.contents();
	buf_size_ *= 2;
	resize(buffer_, buf_size_);
	integer content_sz = v_size(content);
	for (integer i = 0; i < content_sz; i++) {
		buffer_[i] = content[i];
	}
	start_ = 0;
	end_ = v_size(content);
}


class queue_str_unique : public queue_str {
public:
	queue_str_unique();
	virtual void push(string option(nullable) s);
	virtual string pop();

private:
	set_str strings_queued_;
	logical null_queued_;
};
queue_str_unique.queue_str_unique()
: queue_str(), strings_queued_(set_str()), null_queued_(false)
{}
void queue_str_unique.push(string option(nullable) s)
{
	logical stop_push = false;
	if (null(s)) {
		if (null_queued_) {
			stop_push = true;
		} else {
			null_queued_ = true;
		}
	} else {
		if (strings_queued_.includes(s)) {
			stop_push = true;
		} else {
			strings_queued_.add(s);
		}
	}

	if (!stop_push) {
		exec_push(s);
	}
}
string queue_str_unique.pop()
{
	string s = exec_pop();
	if (null(s)) {
		null_queued_ = false;
	} else {
		strings_queued_.remove(s);
	}
	return s;
}
}

// This is copied from stopwatch.ql, which should be a lib file,
// but is commited to some repos. It replaces the timing functions
// of qu.ql, and is better structured that those functions.
module stopwatch {
module __impl__ {

logical		   __Disabled;
map_str_num	   __Name2Index;
vector(string)	__Names;
vector(timestamp) __Starts;
vector(integer)   __Counts; // Only include finished timings
vector(number)	__Totals; // Only include finished timings
vector(number)	__Parents; // Index of timing event to be started before
number			__Current; // Index of last started
integer		   __Size;

logical		   __PathEnabled;
counter_str	  __PathTotal;
counter_str	  __PathCount;
string			__PathDelim option(constant) = "|";

number __SelfCost;
integer __SelfInvocations;

string human_time(number ms)
{
	if (ms < 1000) {
		return strcat(str(ms, 0), " ms");
	} else if (ms < 60000) {
		return strcat(str(ms/1000, 3), " s");
	} else if (ms < 3600000) {
		return strcat(str(ms/60000, 3), " min");
	} else {
		return strcat(str(ms/3600000, 3), " h");
	}
}

string sub_string_end(string s, integer from_pos)
{
	integer n = strlen(s);
	if (from_pos == n) {
		return "";
	}
	integer actual_len;
	integer actual_from_pos;
	if (from_pos > n) {
		throw (E_UNSPECIFIC, strcat(["string '", s, "' has length ", str(n), ", can't create sub string from position ", str(from_pos)]));
	} else if (from_pos < 0) {
		actual_len = -from_pos;
		actual_from_pos = n+from_pos;
		if (actual_len > n) {
			throw (E_UNSPECIFIC, strcat(["string '", s, "' has length ", str(n), ", can't create sub string of letters after first ", str(-from_pos)]));
		}
	} else {
		actual_len = n - from_pos;
		actual_from_pos = from_pos;
	}
	return sub_string(s, actual_from_pos, actual_len);
}

string sub_string_start(string s, integer to_pos)
{
	integer n = strlen(s);
	if (to_pos == 0 || -to_pos == n) {
		return "";
	}
	integer actual_len;
	if (to_pos > n) {
		throw (E_UNSPECIFIC, strcat(["string '", s, "' has length ", str(n), ", can't create sub string of size ", str(to_pos)]));
	} else if (to_pos < 0) {
		actual_len = n+to_pos;
		if (actual_len < 0) {
			throw (E_UNSPECIFIC, strcat(["string '", s, "' has length ", str(n), ", can't create sub string of letters up to last ", str(-to_pos)]));
		}
	} else {
		actual_len = to_pos;
	}
	return sub_string(s, 0, actual_len);
}

vector(number) str_find_all(string haystack, string needle)
{
	vector(number) matches[0];
	if (needle == "") {
		return matches;
	}
	integer start_pos = 0;
	number local_match = str_find(haystack, needle);
	while (!null(local_match)) {
		push_back(matches, start_pos + local_match);
		start_pos = start_pos + integer(local_match) + 1;
		string haystack_end_part = sub_string_end(haystack, start_pos);
		local_match = str_find(haystack_end_part, needle);
	}
	return matches;
}

void reset()
{
	if (null(__Disabled))
		__Disabled = false;
	__Name2Index = map_str_num();
	resize(__Names, 0);
	resize(__Starts, 0);
	resize(__Counts, 0);
	resize(__Totals, 0);
	resize(__Parents, 0);
	__Current = null<number>;
	__Size = 0;

	if (null(__PathEnabled))
		__PathEnabled = true;

	__PathTotal = new counter_str();
	__PathCount = new counter_str();

	__SelfCost = 0;
	__SelfInvocations = 0;
}

void maybe_init()
{
	if (null(__Name2Index))
		reset();
}

void add_self_cost(timestamp since)
{
	__SelfCost += now() - since;
	__SelfInvocations++;
}

integer index(string name)
{
	if (!null(str_find(name, __PathDelim)))
		throw (E_UNSPECIFIC, strcat([__PathDelim, " is illegal in stopwatch name"]));

	number ix = __Name2Index.find(name);
	if (null(ix)) {
		ix = __Size;
		push_back(__Names, name);
		push_back(__Starts, null<timestamp>);
		push_back(__Counts, 0);
		push_back(__Totals, 0);
		push_back(__Parents, null<number>);
		__Name2Index.add(name, ix);
		__Size++;
	}
	return integer(ix);
}

vector(integer) path_indices(integer from_ix)
{
	vector(integer) indices[0];
	number ix = from_ix;
	while (!null(ix)) {
		push_back(indices, integer(ix));
		ix = __Parents[integer(ix)];
	}

	// reverse indices vector
	integer lo = 0;
	integer hi = v_size(indices)-1;
	while (lo < hi) {
		integer tmp = indices[hi];
		indices[hi] = indices[lo];
		indices[lo] = tmp;
		lo++;
		hi--;

	}

	return indices;
}

void start(integer ix, timestamp start_time)
{
	if (__Disabled) return;
	__Starts[ix] = start_time;
	__Parents[ix] = __Current;
	__Current = ix;
}

void stop(integer ix, timestamp stop_time)
{
	if (__Disabled) return;
	if (null(__Starts[ix])) {
		throw (E_UNSPECIFIC, strcat(["timing internal: tried to stop ", str(ix), " but it has no start time"]));
	}
	if (null(__Current)) {
		throw (E_UNSPECIFIC, "timing internal: tried to stop though nothing runs");
	}
	if (ix != __Current) {
		throw (E_UNSPECIFIC, strcat(["timing internal: tried to stop ", str(ix), " but current is ", str(__Current)]));
	}
	number time = stop_time - __Starts[ix];
	if (__PathEnabled) {
		string path = str_join(__Names[path_indices(ix)], __PathDelim);
		__PathCount.increase(path);
		__PathTotal.increase(path, time);
	}
	__Current = __Parents[ix];
	__Totals[ix] += time;
	__Starts[ix] = null<timestamp>;
	__Parents[ix] = null<number>;
	__Counts[ix]++;
}

void stop(string name, logical inclusive, timestamp stop_time)
{
	if (__Disabled) return;
	vector(integer) stop_list[0];
	number ix = __Current;
	logical name_found = false;
	while (!name_found && !null(ix)) {
		if (__Names[integer(ix)] == name) {
			name_found = true;
			if (!inclusive) {
				break;
			}
		}
		push_back(stop_list, integer(ix));
		ix = __Parents[integer(ix)];
	}
	if (!name_found) {
		// "name" not running
		return;
	}
	integer stop_list_sz = v_size(stop_list);
	for (integer i = 0; i < stop_list_sz; ++i) {
		stop(stop_list[i], stop_time);
	}
}

void start(string name, timestamp start_time)
{
	if (__Disabled) return;
	stop(name, true, start_time);
	integer ix = index(name);
	start(ix, start_time);
}

void stop_all(timestamp stop_time)
{
	if (__Disabled) return;
	while (!null(__Current)) {
		stop(integer(__Current), stop_time);
	}
}

void start_top(string name, timestamp start_time)
{
	if (__Disabled) return;
	stop_all(start_time);
	start(name, start_time);
}

logical is_disabled() = __Disabled;
void disable() { __Disabled = true; }
void enable() { __Disabled = false; }

vector(string) timer_list()
{
	return __Name2Index.get_keys();
}

number timer_time(string name)
{
	return __Totals[index(name)];
}

integer timer_count(string name)
{
	return __Counts[index(name)];
}


void track_paths(logical do_or_not)
{
	__PathEnabled = do_or_not;
}

vector(string) path_list()
{
	return __PathTotal.get_keys();
}

number path_time(string path)
{
	return __PathTotal.get_count(path);
}

integer path_count(string path)
{
	return integer(__PathCount.get_count(path));
}

integer path_depth(string path)
{
	return v_size(str_find_all(path, __impl__.__PathDelim));
}


number self_cost() = __SelfCost;
integer self_invocations() = __SelfInvocations;

vector(string) inspect_state()
{
	vector(string) state[0];
	for (integer i = 0; i < __Size; ++i) {
		number pix = __Parents[i];
		push_back(
			state,
			strcat([
					   __Names[i], " @ ", str(__Starts[i]),
					   null(pix) ?
					   " -> " :
					   strcat([" -> ", __Names[integer(pix)], " [", str(pix), "] "]),
					   "Sum: ", str(__Totals[i]), " / ", str(__Counts[i])]));
	}
	if (null(__Current)) {
		push_back(state, "Current: <none>");
	} else {
		push_back(
			state,
			strcat(["Current: ", __Names[integer(__Current)], " [", str(__Current), "]"]));
	}
	return state;
}

vector(string) running_timers_data(timestamp check_time)
{
	vector(string) timer_data[0];
	number ix = __Current;
	while (!null(ix)) {
		push_back(
			timer_data,
			strcat([
					   "'", __Names[integer(ix)], "' since ", str(__Starts[integer(ix)]),
					   " (", human_time(check_time - __Starts[integer(ix)]), ")"]));
		ix = __Parents[integer(ix)];
	}
	return timer_data;
}

}

string path_delimiter() = __impl__.__PathDelim;

// Starts timer "name" (if it is running, stops it and child timers first)
void start(string name)
{
	timestamp t = now();
	__impl__.maybe_init();
	__impl__.start(name, now());
	__impl__.add_self_cost(t);
}

// stops last started timer and starts timer "name"
void stop_start(string stop_name, string start_name)
{
	timestamp t = now();
	__impl__.maybe_init();
	__impl__.stop(stop_name, true, t);
	__impl__.start(start_name, t);
	__impl__.add_self_cost(t);
}

// stops all running timers and starts timer "name"
void start_top(string name)
{
	timestamp t = now();
	__impl__.maybe_init();
	__impl__.start_top(name, now());
	__impl__.add_self_cost(t);
}

// stops timer "name" and all child timers of it
void stop(string name)
{
	timestamp t = now();
	__impl__.maybe_init();
	__impl__.stop(name, true, now());
	__impl__.add_self_cost(t);
}

// stops child timers to "name", but not the timer "name"
void stop_after(string name)
{
	timestamp t = now();
	__impl__.maybe_init();
	__impl__.stop(name, false, now());
	__impl__.add_self_cost(t);
}

// stops all timers
void stop_all()
{
	timestamp t = now();
	__impl__.maybe_init();
	__impl__.stop_all(now());
	__impl__.add_self_cost(t);
}

logical is_disabled()
{
	__impl__.maybe_init();
	return __impl__.is_disabled();
}

// Ignore all calls to start or stop timers
void disable()
{
	__impl__.maybe_init();
	__impl__.disable();
}

// Stop ignoring calls to start or stop timers
void enable()
{
	__impl__.maybe_init();
	__impl__.enable();
}

// returns a list with the names of all timers, running or not
vector(string) timer_list()
{
	timestamp t = now();
	__impl__.maybe_init();
	vector(string) r = __impl__.timer_list();
	__impl__.add_self_cost(t);
	return r;
}

number timer_time(string name)
{
	timestamp t = now();
	__impl__.maybe_init();
	number r = __impl__.timer_time(name);
	__impl__.add_self_cost(t);
	return r;
}

integer timer_count(string name)
{
	timestamp t = now();
	__impl__.maybe_init();
	integer r = __impl__.timer_count(name);
	__impl__.add_self_cost(t);
	return r;
}



// If individual timers aren't enough, enable paths! Paths are the timer hierarchies.
// Examples (calc_all, prepare_calc, fetch_data, check_data are all names of timers):
// calc_all -> prepare_calc -> fetch_data
// calc_all -> prepare_calc -> check_data
// prepare_calc -> fetch_data
// prepare_calc -> check_data
// See? You get prepare calc split on when it is part of calc_all and when it is
// free standing.
void track_paths(logical do_or_not)
{
	__impl__.maybe_init();
	__impl__.track_paths(do_or_not);
}

// returns a list with the names of all timer paths, running or not
vector(string) path_list()
{
	timestamp t = now();
	__impl__.maybe_init();
	vector(string) r = __impl__.path_list();
	__impl__.add_self_cost(t);
	return r;
}

number path_time(string name)
{
	timestamp t = now();
	__impl__.maybe_init();
	number r = __impl__.path_time(name);
	__impl__.add_self_cost(t);
	return r;
}

integer path_count(string name)
{
	timestamp t = now();
	__impl__.maybe_init();
	integer r = __impl__.path_count(name);
	__impl__.add_self_cost(t);
	return r;
}


/* Cannot use interpolation - it requires instrument dll or something...
// Colors from colorbrewer2.org, 10 points, diverging RdYlGn - skipping last 2 green colors and first red
interpolation red_ip = interpolation(ip_linear(), qu.v_range(7)/7, [102, 166, 217, 254, 253, 244, 215]);
interpolation green_ip = interpolation(ip_linear(), qu.v_range(7)/7, [189, 217, 239, 224, 174, 109, 48]);
interpolation blue_ip = interpolation(ip_linear(), qu.v_range(7)/7, [99, 106, 139, 139, 97, 67, 39]);
*/
number interpolate(number low_x,
				   number low_y,
				   number high_x,
				   number high_y,
				   number at_x)
{
	// solve y = kx+m
	number k = (high_y - low_y) / (high_x - low_x);
	number m = low_y - k*low_x;
	number y_at_x = k*at_x + m;
	return y_at_x;
}


number severity_color(number val, number max_val)
{
	number severity = val/max_val;
	/*
	  number r = round(red_ip.eval(severity));
	  number g = round(green_ip.eval(severity));
	  number b = round(blue_ip.eval(severity));
	*/
	integer r = integer(round(severity < 0.5 ? interpolate(0, 102, 0.5, 254, severity) : interpolate(0.5, 254, 1, 215, severity), 0));
	integer g = integer(round(severity < 0.4 ? interpolate(0, 189, 0.4, 240, severity) : interpolate(0.4, 240, 1, 48, severity), 0));
	integer b = integer(round(severity < 0.4 ? interpolate(0, 99, 0.4, 139, severity) : interpolate(0.4, 139, 1, 39, severity), 0));
	return rgb(r,g,b);
}

vector(number) rgb_sequential_green_blue(number n_colors)
{
/*
	-- This section requires interpolation, but that's not generally available
	vector(number) red_y = [247, 224, 204, 168, 123, 78, 43, 8];
	vector(number) green_y = [252, 243, 235, 221, 204, 179, 140, 88];
	vector(number) blue_y = [240, 219, 197, 181, 196, 211, 190, 158];
	if (n_colors < 1) {
		return empty_num_vec();
	}
	if (n_colors == 1) {
		return rgb(red_y[0], green_y[0], blue_y[0]);
	}
	number scaling = (v_size(red_y)-1) / (n_colors-1);
	vector(number) xs = v_range(n_colors) .* scaling;
	red = round(ip_linear(xs, k_x, rb));
	green = round(ip_linear(xs, k_x, gb));
	blue = round(ip_linear(xs, k_x, bb));
*/
	switch (n_colors) {
	case 1: return [rgb(224,243,219)];
	case 2: return [rgb(224,243,219), rgb(67,162,202)];
	case 3: return [rgb(224,243,219), rgb(168,221,181), rgb(67,162,202)];
	case 4: return [rgb(240,249,232), rgb(186,228,188), rgb(123,204,196), rgb(43,140,190)];
	case 5: return [rgb(240,249,232), rgb(186,228,188), rgb(123,204,196), rgb(67,162,202), rgb(8,104,172)];
	case 6: return [rgb(240,249,232), rgb(204,235,197), rgb(168,221,181), rgb(123,204,196), rgb(67,162,202), rgb(8,104,172)];
	case 7: return [rgb(240,249,232), rgb(204,235,197), rgb(168,221,181), rgb(123,204,196), rgb(78,179,211), rgb(43,140,190), rgb(8,88,158)];
	case 8: return [rgb(247,252,240), rgb(224,243,219), rgb(204,235,197), rgb(168,221,181), rgb(123,204,196), rgb(78,179,211), rgb(43,140,190), rgb(8,88,158)];
	default: throw (E_UNSPECIFIC, "rgb_sequential_green_blue can only produce 1-8 colors");
	}
}


matrix(text_rgb) table(logical track_paths, logical reset_afterwards)
{
	__impl__.maybe_init();
	vector(string) timers = __impl__.timer_list();
	vector(number) timer_times = __impl__.timer_time(timers);
	vector(integer) timer_counts = __impl__.timer_count(timers);
	vector(number) timer_avgs = timer_times / timer_counts;
	sort(timers, -timer_times); // "-" for reverse sort
	sort(timer_counts, -timer_times);
	sort(timer_avgs, -timer_times);
	sort(timer_times, -timer_times);
	number max_timer_time = v_max(timer_times);
	number max_timer_avg = v_max(timer_avgs);

	vector(string) paths = __impl__.path_list();
	vector(number) path_times = __impl__.path_time(paths);
	vector(integer) path_counts = __impl__.path_count(paths);
	vector(number) path_avgs = path_times / path_counts;
	vector(number) path_depths = __impl__.path_depth(paths);
	number max_path_time = v_max(path_times);
	number max_path_avg = v_max(path_avgs);

	integer n_timers = v_size(timers);
	integer n_paths = v_size(paths);

	number header_color = rgb(200,200,200);

	integer n_rows = 1 + (__impl__.__PathEnabled ? ..max(n_timers, n_paths) : n_timers);
	integer n_cols = __impl__.__PathEnabled ? 9 : 5;
	integer cost_column = n_cols - 1;

	matrix(text_rgb) ret_mx[n_rows, n_cols];
	ret_mx[0,0] = text_rgb("Timer", null<number>, header_color);
	ret_mx[0,1] = text_rgb("Count", null<number>, header_color);
	ret_mx[0,2] = text_rgb("Total", null<number>, header_color);
	ret_mx[0,3] = text_rgb("Avg", null<number>, header_color);

	if (__impl__.__PathEnabled) {
		ret_mx[0,4] = text_rgb("Path", null<number>, header_color);
		ret_mx[0,5] = text_rgb("Count", null<number>, header_color);
		ret_mx[0,6] = text_rgb("Total", null<number>, header_color);
		ret_mx[0,7] = text_rgb("Avg", null<number>, header_color);
	}

	ret_mx[0, cost_column] = text_rgb(strcat("Stopwatch cost: ", __impl__.human_time(__impl__.self_cost())));

	for (integer i = 0; i < n_timers; ++i) {
		ret_mx[i+1, 0] = text_rgb(timers[i]);
		ret_mx[i+1, 1] = text_rgb(str(timer_counts[i]));
		ret_mx[i+1, 2] = text_rgb(__impl__.human_time(timer_times[i]), null<number>, severity_color(timer_times[i], max_timer_time));
		ret_mx[i+1, 3] = text_rgb(__impl__.human_time(timer_avgs[i]), null<number> , severity_color(timer_avgs[i], max_timer_avg));
	}
	if (__impl__.__PathEnabled &&  n_paths > 0) {
		vector(integer) path_depths_capped = integer(..min(path_depths, 7)); // We know rgb_sequential_green_blue cannot produce more than 8 colors
		integer max_path_depth = integer(v_max(path_depths_capped));
		vector(number) colors = rgb_sequential_green_blue(max_path_depth+1);
		for (integer i = 0; i < n_paths; ++i) {
			number color = colors[path_depths_capped[i]];
			ret_mx[i+1, 4] = text_rgb(paths[i], null<number>, color, false);
			ret_mx[i+1, 5] = text_rgb(str(path_counts[i]), null<number>, color, false);
			ret_mx[i+1, 6] = text_rgb(__impl__.human_time(path_times[i]), null<number>, severity_color(path_times[i], max_path_time));
			ret_mx[i+1, 7] = text_rgb(__impl__.human_time(path_avgs[i]), null<number>, severity_color(path_avgs[i], max_path_avg));
		}
	}

	if (reset_afterwards) {
		__impl__.reset();
	}

	__impl__.track_paths(track_paths);

	return ret_mx;
}

// Function below are for debugging, or other non-standard actions, and they don't
// count towards the self cost
number self_cost()
{
	__impl__.maybe_init();
	return __impl__.self_cost();
}

integer self_invocations()
{
	__impl__.maybe_init();
	return __impl__.self_invocations();
}

vector(string) inspect_state()
{
	__impl__.maybe_init();
	return __impl__.inspect_state();
}

vector(string) running_timers_data()
{
	__impl__.maybe_init();
	return __impl__.running_timers_data(now());
}

// Resets everything.
// Including self cost, settings (such as disabling paths)
void reset()
{
	__impl__.maybe_init();
	__impl__.reset();
}

} // module stopwatch

integer COLOR_ERROR option(constant) = 13158655; // rgb(255,200,200)

string EVENT_HISTORY option(constant) = "#<event>$@history";

counter_str __RIC_COUNT = new counter_str();
string __LOCAL_EVENT_NUMBER_FID = "event_number";
void __conditional_publish(string ric) {
	if (__RIC_COUNT.get_count(ric) == 0) {
		local.publish('LOCAL', ric, [__LOCAL_EVENT_NUMBER_FID], ['0']);
		sleep(2); // To ensure ric is ready for reading
		__RIC_COUNT.increase(ric);
	}
}
void __local_event(string ric) {
	__RIC_COUNT.increase(ric);
	number num = __RIC_COUNT.get_count(ric);
	local.publish('LOCAL', ric, [__LOCAL_EVENT_NUMBER_FID], [str(num)]);
}
void __listen_to_event(string ric, string option(nullable) func_name = null<string>) {
	__conditional_publish(ric);
	rt.get(ric, __LOCAL_EVENT_NUMBER_FID, 'LOCAL');
}


logging.Context __logging_context__;
logging.Context logging_context()
{
	if (null(__logging_context__))
		__logging_context__ = logging.context("EventManager");
	return __logging_context__;
}

	// Dummy class to allow making settings in compile stage.
	// Do it like this:
	// event.preset_obj __e1 = event.remember();

	class preset_obj {};
	preset_obj __preset_obj__;

	public enum event_type option(category: "Workspace") {
		ET_REQUEST option(str: "request"),
			ET_SKIPPED_CALL option(str: "skipped call"),
			ET_CALL option(str: "call"),
			ET_RUN_PLANNED option(str: "run"),
			ET_RUN_UNPLANNED option(str: "run (unplanned)"),
			ET_SKIPPED_NEXT option(str: "skipped next"),
			ET_BLOCK_CALLS option(str: "dont call me"),
			ET_RUN option(str: "unused") // Exists for backward COMPILE compability - no events will be of this type
	};

	class call_state {
	public:
		call_state(integer call_number, set_str option(nullable) event_flags, logical by_event);

		integer call_number();
		set_str event_flags();
		logical by_event();

	private:
		integer call_number_;
		set_str event_flags_;
		logical by_event_;

		void __dbg_print(__dbg_label l);
	};

	call_state.call_state(integer call_number, set_str option(nullable) event_flags, logical by_event)
		: call_number_(call_number), event_flags_(event_flags), by_event_(by_event)
	{
	}

	void call_state.__dbg_print(__dbg_label l)
	{
		string flagstr = null(event_flags_) ? "" : strcat([", flags: ", str_join(event_flags_.get_content(true), ", ")]);
		l.set_text(strcat(["call_state { #", str(call_number_), by_event_ ? " (by event)" : " (unplanned)", flagstr, " }"]));
	}

	logical call_state.by_event() = by_event_;
	integer call_state.call_number() = call_number_;
	set_str call_state.event_flags() = event_flags_;

	public class Event option(category: "Workspace") {
	public:
		Event(string func_name, timestamp time, event_type type);
		string func_name();
		timestamp time();
		event_type type();

		void add_info(string info);
		string extra_info();

		void set_state_desc(string desc);
		string state_desc();

	private:
		string func_name_;
		timestamp time_;
		event_type type_;
		vector(string) extra_info_;
		string state_desc_;
	};
	Event.Event(string func_name, timestamp time, event_type type) :
		func_name_(func_name),
		time_(time),
		type_(type)
		{}
	string Event.func_name() = func_name_;
	timestamp Event.time() = time_;
	event_type Event.type() = type_;
	void Event.add_info(string info) { push_back(extra_info_, info); }
	string Event.extra_info() = str_join(extra_info_, ", ");
	void Event.set_state_desc(string desc) { state_desc_ = desc; }
	string Event.state_desc() = state_desc_;




	number RL_NOTHING = 0;
	number RL_INFO = 1;
	number RL_DEBUG = 2;

	public class EventManager option(category: "Workspace") {
	public:
		EventManager();
		call_state enter_function(string func_name, logical flags_allowed);
		void push(string func_name, vector(string) option(nullable) flags = null);
		void next();

		string current_function();
		void block_calls();

		set_str get_flags(string func_name, logical clear_flags);

		void remember();
		void remember_debug();
		void forget(logical disable_history = true);
		logical user_controlled_history();
		void enable();
		void disable();

		void enable_timing();
		void disable_timing();

		vector(string) event_func_names();
		vector(timestamp) event_times();
		vector(event_type) event_types();
		vector(string) event_info();
		vector(string) event_states();
		vector(string) event_requesters(); // Backward compatibility

	private:
		logical enabled_;

		__qu_stuff__.queue_str_unique call_queue_;
		string called_function_; // called but not yet run
		counter_str registered_func_names_count_;
		set_str blocked_func_names_;
		number remember_level_;
		string last_registered_; // set by enter_function(), unset by next(), remembered along with pushes
		Event last_start_event_; // set by enter_function(), unset by next()
		vector(Event) events_;
		map_str_obj<set_str> flags_;
		logical user_controlled_history_; // True if a history controlling function has been called
		logical use_timing_; // True if this_function() and next() should start and stop timers

		void __add_history_event(Event e);

		void __remember();
		void __remember_debug();
		void __forget(logical disable_history);
		string __state_description();
	};
	EventManager.EventManager() :
		enabled_(true),
		call_queue_(new __qu_stuff__.queue_str_unique()),
		registered_func_names_count_(new counter_str()),
		blocked_func_names_(null<set_str>), // becomes non-null when non-empty
		remember_level_(RL_NOTHING),
		flags_(new map_str_obj<set_str>()),
		user_controlled_history_(false),
		use_timing_(false)
	{
		__forget(true);
	}

	void EventManager.__add_history_event(Event e)
	{
		e.set_state_desc(__state_description());
		push_back(events_, e);
		__local_event(EVENT_HISTORY);
	}

	call_state EventManager.enter_function(string func_name, logical flags_allowed)
	{
		if (!enabled_) return null<call_state>;

		if (use_timing_) stopwatch.start_top(func_name);
		if (!null(blocked_func_names_)) {
			blocked_func_names_.remove(func_name);
			if (blocked_func_names_.size() == 0)
				blocked_func_names_ = null;
		}
		__listen_to_event(func_name);
		logical planned = __qu_stuff__.equal_nbl(func_name, called_function_);
		registered_func_names_count_.increase(func_name);
		last_registered_ = func_name;
		if (remember_level_ >= RL_INFO) {
			last_start_event_ = new Event(func_name, now(), planned ? ET_RUN_PLANNED : ET_RUN_UNPLANNED);
			__add_history_event(last_start_event_);
		}
		if (planned) {
			called_function_ = null<string>;
		}
		// Clear out queued calls to this function

		set_str event_flags = null;

		if (flags_allowed)
			event_flags = this.get_flags(func_name, true);
		else if (!null(flags_.find(func_name)))
			throw (E_UNSPECIFIC, "event: Someone is passing flags to a function that does not take them");

		if (remember_level_ >= RL_INFO) {
			last_start_event_ = new Event(func_name, now(), planned ? ET_RUN_PLANNED : ET_RUN_UNPLANNED);
			if (!null(event_flags))
				last_start_event_.add_info(strcat(["{", str_join(event_flags.get_content(true), ", "), "}"]));
			__add_history_event(last_start_event_);
		}

		if (logging.info(logging_context())) {
			string flag_str = null(event_flags) ? "" : strcat([ "flag_str: ", str_join(event_flags.get_content(true), ", ")]);
			logging.info(logging_context(), ["<event> Run '", func_name, "' ", planned ? "" : " (unplanned)", flag_str]);
		}

		return new call_state(integer(registered_func_names_count_.get_count(func_name)), event_flags, planned);
	}

	void EventManager.push(string func_name, vector(string) option(nullable) flags)
	{
		if (!enabled_) return;

		if (!null(blocked_func_names_) && blocked_func_names_.includes(func_name)) {
			if (remember_level_ >= RL_INFO) {
				Event event = new Event(func_name, now(), ET_REQUEST);
				event.add_info("[Ignored (don't call me)]");
				event.add_info(strcat(["By ", last_registered_]));
				if (!null(flags))
					event.add_info(strcat(["{", str_join(flags, ", "), "}"]));
				__add_history_event(event);
			}
			return;
		}

		call_queue_.push(func_name);

		if (!null(flags)) {
			set_str fl = flags_.find(func_name);
			if (null(fl)) {
				fl = set_str();
				flags_.add(func_name, fl);
			}
			fl.add(flags);
		}

		if (remember_level_ >= RL_INFO) {
			Event event = new Event(func_name, now(), ET_REQUEST);
			event.add_info(strcat(["By ", last_registered_]));
			if (!null(flags))
				event.add_info(strcat(["{", str_join(flags, ", "), "}"]));
			__add_history_event(event);
		}
	}

	void EventManager.next()
	{
		if (!enabled_) return;
		if (use_timing_) stopwatch.stop(last_registered_);
		string func_to_call = null<string>;
		if (null(called_function_)) {
			while (null(func_to_call) && !call_queue_.empty()) {
				func_to_call = call_queue_.pop();
				if (registered_func_names_count_.get_count(func_to_call) == 0) {
					logging.info(logging_context(), ["<event> Skipping call to '", func_to_call, "' (not registered)."]);
					if (remember_level_ >= RL_DEBUG) {
						Event event = new Event(func_to_call, now(), ET_SKIPPED_CALL);
						__add_history_event(event);
					}
					func_to_call = null<string>;
				} else if (registered_func_names_count_.get_count(func_to_call) == 1 && this.current_function() == func_to_call) {
					logging.info(logging_context(), ["<event> Skipping call to '", func_to_call, "' (cannot self-call at first invocation)."]);
					if (remember_level_ >= RL_DEBUG) {
						Event event = new Event(func_to_call, now(), ET_SKIPPED_CALL);
						__add_history_event(event);
					}
					func_to_call = null<string>;
				} else if (!null(blocked_func_names_) && blocked_func_names_.includes(func_to_call)) {
					logging.info(logging_context(), ["<event> Skipping call to '", func_to_call, "' (don't call me)."]);
					if (remember_level_ >= RL_DEBUG) {
						Event event = new Event(func_to_call, now(), ET_SKIPPED_CALL);
						__add_history_event(event);
					}
					func_to_call = null<string>;
				}
			}
		} else {
			if (remember_level_ >= RL_DEBUG) {
				Event e = new Event(last_registered_, now(), ET_SKIPPED_NEXT);
				e.add_info(strcat(["Waiting for ", called_function_]));
				__add_history_event(e);
			}
		}

		if (null(func_to_call)) {
			logging.info(logging_context(), "<event> No functions to call.");
		} else {
			logging.info(logging_context(), ["<event> Calling '", func_to_call, "'."]);
			__local_event(func_to_call);
			called_function_ = func_to_call;
			if (remember_level_ >= RL_INFO) {
				Event event = new Event(func_to_call, now(), ET_CALL);
				__add_history_event(event);
			}
		}
		if (remember_level_ >= RL_INFO) {
			last_start_event_.add_info(strcat([str(now() - last_start_event_.time()), " ms"]));
			last_start_event_ = null<Event>;
		}
		last_registered_ = null<string>;
	}

	string EventManager.current_function() = last_registered_;

	void EventManager.block_calls()
	{
		string func = this.current_function();
		if (null(func))
			throw (E_UNSPECIFIC, "No function is running");

		if (null(blocked_func_names_))
			blocked_func_names_ = set_str();
		blocked_func_names_.add(func);
		if (remember_level_ >= RL_INFO) {
			Event event = new Event(func, now(), ET_BLOCK_CALLS);
			__add_history_event(event);
		}
	}

	set_str EventManager.get_flags(string func_name, logical clear_flags)
	{
		set_str fl = flags_.find(func_name);
		if (null(fl))
			return set_str();
		if (clear_flags)
			flags_.remove(func_name);
		return fl;
	}

	void EventManager.remember()
	{
		user_controlled_history_ = true;
		__remember();
	}

	void EventManager.remember_debug()
	{
		user_controlled_history_ = true;
		__remember_debug();
	}

	void EventManager.forget(logical disable_history)
	{
		user_controlled_history_ = true;
		__forget(disable_history);
	}

	logical EventManager.user_controlled_history() = user_controlled_history_;
	void EventManager.enable() { enabled_ = true; }
	void EventManager.disable() { enabled_ = false; }

	void EventManager.enable_timing() { use_timing_ = true; }
	void EventManager.disable_timing() { use_timing_ = false; }
	vector(string) EventManager.event_func_names() = clone_vector(events_.func_name());
	vector(timestamp) EventManager.event_times() = clone_vector(events_.time());
	vector(event_type) EventManager.event_types() = clone_vector(events_.type());
	vector(string) EventManager.event_info() = clone_vector(events_.extra_info());
	vector(string) EventManager.event_states() = clone_vector(events_.state_desc());

	// Backward compatibility
	vector(string) EventManager.event_requesters() = this.event_info();

	void EventManager.__remember() { remember_level_ = RL_INFO; }
	void EventManager.__remember_debug() { remember_level_ = RL_DEBUG; }
	void EventManager.__forget(logical disable_history)
	{
		last_registered_ = null<string>;
		resize(events_, 0);
		__local_event(EVENT_HISTORY);
		if (disable_history)
			remember_level_ = RL_NOTHING;
	}

	string EventManager.__state_description()
	{
		if (!enabled_) {
			return "disabled";
		}
		string desc = "";
		if (remember_level_ < RL_DEBUG) {
			return desc;
		}

		if (!null(called_function_)) {
			desc = strcat(["Expects ", called_function_, ". "]);
		}

		desc = strcat([desc, "Queue: [", str_join(call_queue_.contents(), ", "), "]"]);
		desc = strcat([desc, " Current: ", last_registered_]);
		return desc;
	}


	EventManager __event_manager__ = new EventManager();

	logical history_shown = false;

	matrix(text_rgb) __history_matrix(logical reversed, logical auto_trigger, logical auto_clear)
	{
		if (!__event_manager__.user_controlled_history()) {
			// If settings function hasn't been called, user hasn't taken
			// control of history handling. Since the user calls this
			// function, it is obvious that history is desired, so
			// start remembering!
			// Unfortunately, at the first call to this function won't
			// no history will exist - but that can't be fixed since we
			// don't want to record history without the user wanting it.
			__event_manager__.remember();
		}

		if (auto_trigger)
			__listen_to_event(EVENT_HISTORY);

		vector(string) fnames = __event_manager__.event_func_names();
		vector(timestamp) times = __event_manager__.event_times();
		vector(event_type) types = __event_manager__.event_types();
		vector(string) infos = __event_manager__.event_info();
		vector(string) states = __event_manager__.event_states();

		if (auto_clear)
			__event_manager__.forget(false);

		integer n_rows = v_size(fnames);
		integer n_cols = 5;

		if (n_rows == 0) {
			matrix(text_rgb) ret_mx[1, n_cols];
			return ret_mx;
		}

		matrix(text_rgb) ret_mx[n_rows, n_cols];

		for (integer row = 0; row < n_rows; ++row) {
			integer index = reversed ? n_rows - row - 1 : row;
			string fname = fnames[index];
			string time  = str(times[index]);
			string type  = string(types[index]);
			string info  = null(infos[index]) ? "" : infos[index];
			string state = states[index];

			number color = null<number>;
			switch (types[index]) {
				case ET_REQUEST:
					color = rgb(240, 230, 140);
					break;
				case ET_SKIPPED_CALL:
					color = rgb(200, 200, 200);
					break;
				case ET_CALL:
					color = rgb(150, 250, 190);
					break;
				case ET_RUN_PLANNED:
					color = rgb(170, 210, 250);
					break;
				case ET_RUN_UNPLANNED:
					color = rgb(190, 225, 250);
					break;
				case ET_SKIPPED_NEXT:
					color = rgb(230, 230, 230);
					break;
			}

			ret_mx[row, 0] = text_rgb(time, null<number>, color, false);
			ret_mx[row, 1] = text_rgb(type, null<number>, color, false);
			ret_mx[row, 2] = text_rgb(fname, null<number>, color, false);
			ret_mx[row, 3] = text_rgb(info, null<number>, color, false);
			ret_mx[row, 4] = text_rgb(state, null<number>, color, false);
		}

		history_shown = true;
		return ret_mx;
	}


	void __settings(logical events_disabled, logical history_enabled, logical keep_shown_history, logical debug_level)
	{
		if (events_disabled) {
			__event_manager__.disable();
		} else {
			__event_manager__.enable();
		}

		if (history_enabled) {
			if (!keep_shown_history && history_shown) {
				__event_manager__.forget();
			}
			if (debug_level) {
				__event_manager__.remember_debug();
			} else {
				__event_manager__.remember();
			}
		} else {
			__event_manager__.forget();
		}
		history_shown = false;
	}

	integer __err_print__ = 0;
	void set_error_message_style(logical with_position)
	{
		__err_print__ = with_position ? 1 : 0;
	}
	string __errmsg(error e) = __err_print__ == 0 ? e.message() : e.print();
	void __err_log(error e, integer pos) { log_message(LOG_ERROR, strcat([__event_manager__.current_function(), ": ", __errmsg(e)]), pos); }
	void __err_void(error e, integer pos) { __err_log(e, pos); }
	vector(string) __err_v_s(error e) = ["Error:", __errmsg(e)];
	vector(string) __err_v1_s(error e) = [strcat("Error: ", __errmsg(e))];
	matrix(string) __err_m_s(error e, integer n_cols)
	{
		if (n_cols <= 1)
			return [["Error:"], [__errmsg(e)]];
		matrix(string) ret[2, n_cols];
		ret[0, 0] = "Error:";
		ret[1, 0] = __errmsg(e);
		return ret;
	}
	vector(text_rgb) __err_v_t(error e) = text_rgb(__err_v_s(e), -1, COLOR_ERROR);
	matrix(text_rgb) __err_m_t(error e, integer n_cols = -1) = text_rgb(__err_m_s(e, n_cols), -1, COLOR_ERROR);
	vector(number) __err_v_n(error e, integer pos)
	{
		__err_log(e, pos);
		return vector(i:0;0);
	}
	matrix(number) __err_m_n(error e, integer n_cols, integer pos)
	{
		__err_log(e, pos);
		matrix(number) ret[0, n_cols > 1 ? n_cols : 1];
		return ret;
	}



	/**** PUBLIC INTERFACE BELOW ****/


	/** Mandatory functions for correct behavior **/
	// Always call at start of out function
	// by_event is set to true if the function was triggered by event lib,
	// and false if not (typically triggered by user action, or realtime source).
	public call_state this_function(string func, logical use_event_flags = false) option(category: "Workspace")
	{
		return __event_manager__.enter_function(func, use_event_flags);
	}

	// Call whenever to put another function in call queue
	public void call_function(string func) option(category: "Workspace")
	{
		__event_manager__.push(func);
		logging.info(logging_context(), ["<event> Call '", func, "'."]);
	}

	public void call_function(string func, vector(string) flags) option(category: "Workspace")
	{
		__event_manager__.push(func, flags);
		logging.info(logging_context(), ["<event> Call '", func, "' with flags ", str_join(flags, ", "), "."]);
	}

	// Always call before leaving out function
	public void next() option(category: "Workspace") { __event_manager__.next(); }

	// Current function cannot be called until it is run manually again.
	public void block_calls() option(category: "Workspace") { __event_manager__.block_calls(); }

	/** Totally disable events (and re-enable) **/
	public preset_obj disable_events() { __event_manager__.disable(); return __preset_obj__; }
	public preset_obj enable_events() { __event_manager__.enable(); return __preset_obj__; }

	/** History keeping **/
	public preset_obj remember() { __event_manager__.remember(); return __preset_obj__; }
	public preset_obj remember_debug() { __event_manager__.remember_debug(); return __preset_obj__; }
	public preset_obj forget(logical disable_history = true) { __event_manager__.forget(disable_history); return __preset_obj__; }

	/** Timing **/
	public preset_obj enable_timing() { __event_manager__.enable_timing(); return __preset_obj__; }
	public preset_obj disable_timing() { __event_manager__.disable_timing(); return __preset_obj__; }

	/** To use as out functions in workspace tables **/
	// Get a neat matrix with recorded event history. History recording must be enabled, either through settings function below or explicit call to remember or remember_debug above.
	public matrix(text_rgb) history_matrix(
		logical reversed = false,
		logical auto_trigger = false,
		logical auto_clear = false) =
		__history_matrix(reversed, auto_trigger, auto_clear);
	// Settings, if desired
	public void settings(logical events_disabled, logical history_enabled, logical keep_shown_history, logical debug_level) { __settings(events_disabled, history_enabled, keep_shown_history, debug_level); }

	/** Help functions for exception handling **/
	void err_void(error e, string error_to_ignore = "") { if (e.message() != error_to_ignore) __err_void(e, e.pos()); }
	vector(string) err_v_s(error e, string error_to_ignore = "") = e.message() == error_to_ignore ? vector(i:0;"") : __err_v_s(e); // Returns vector of size > 1
	vector(string) err_v1_s(error e, string error_to_ignore = "") = e.message() == error_to_ignore ? vector(i:1;"") : __err_v1_s(e); // Returns vector of size 1
	matrix(string) err_m_s(error e, string error_to_ignore = "", integer n_cols = -1) = e.message() == error_to_ignore ? [[""]] : __err_m_s(e, n_cols);
	vector(text_rgb) err_v_t(error e, string error_to_ignore = "") = e.message() == error_to_ignore ? vector(i:0;null<text_rgb>) : __err_v_t(e);
	matrix(text_rgb) err_m_t(error e, string error_to_ignore = "", integer n_cols = -1) = e.message() == error_to_ignore ? [[text_rgb("")]] : __err_m_t(e, n_cols);
	vector(number) err_v_n(error e, string error_to_ignore = "") = e.message() == error_to_ignore ? vector(i:0;0) : __err_v_n(e, e.pos());
	matrix(number) err_m_n(error e, string error_to_ignore = "", integer n_cols = -1) = e.message() == error_to_ignore ? [[null<number>]] : __err_m_n(e, n_cols, e.pos());


	/*** DEPRECATED - will be removed ***/
	EventManager get_manager() = __event_manager__;

}
