

class matrix_holder
{

public:

	matrix_holder(integer n_rows, integer n_cols);

	matrix(number) m;
};


matrix_holder.matrix_holder(integer n_rows, integer n_cols)
{
	matrix(number) tmp[n_rows, n_cols];

	m = tmp;
}


series<date>(number) create_date_series(integer size, date start, integer date_dx)
{
	date d = start + (size - 1) * date_dx;

	series<date>(number) s = series(t: start, d, date_dx; null<number>);

	return s;
}


void var_vect_fill(out vector(number) h,
			       vector(number) x_t, 
                   vector(number) r, 
                   number var0, 
                   integer p, 
                   integer q)
{
	integer size = v_size(h);

	for (integer t = 0; t < size; t++)
	{
	    h[t] = r[0];
	    
	    for (integer i = 1; i <= p; i++) 
	    {
			if ((t-i) < 0)
		    	h[t] += r[i] * var0;
			else
		    	h[t] += r[i] * x_t[t-i] * x_t[t-i];
	    }
	    
	    for (integer i = 1; i <= q; i++)
	    {
			if ((t-i) < 0)
				h[t] += r[p+i] * var0;
			else
				h[t] += r[p+i] * h[t-i];
	    }
	}
}


void series_wash(out vector(number)   ret,
                 series<date>(number) s,
                 calendar             cal,
                 logical              fh)
{
	date    s_start = s_x0(s);
	integer  date_dx = s_dx(s);

	resize(ret, 0);

    // Korrigerar inkommande data for helgdagar

    date    d    = s_start;
    integer size = s_size(s);

    for (integer k = 0; k < size; k++)
	{
		number s_k = s[k];
	
		if (!null(s_k) && s_k > 0.1)
			throw(E_INVALID_ARG, "Extreme Value in series"); // why is 0.1 necessarily extreme..?
	
		if (!cal.is_holiday(d))
			if (!null(s_k) || fh == true) 
				push_back(ret, s_k);

		d += date_dx;
    }

    number mean_value = v_average(ret);

	integer retsize = v_size(ret);

    for (integer i = 0; i < retsize; i++)
	{
		if (!null(ret[i]))
	    	ret[i] -= mean_value;
		else
	    	ret[i] = 0; 
	}   
}


// f_v is the output..
// Seems to be the generated squared volatilities.
vector(number) forecast_engine(integer n, /// Kolla vad n står för, kan den väljas oberoende av resten av argumenten?
                     vector(number) coeffs, // constant term, p coefficients for squared log-returns and q coefficients for squared vol
                     vector(number) x_t,  // Given historical log-returns. Has to be p of them I guess?
                     vector(number) h,    // Given historical squared vols. Has to be q of them I guess?
                     integer p, 
                     integer q)
{
	integer size = v_size(x_t);

	vector(number) f_v = one_vector(n,0);

    for (integer i = 1; i <= n; i++ )
	{
		f_v[i-1] = coeffs[0];
	
		for (integer j = 1; j <= p; j++)
		{
			if (i - j < 1)
				f_v[i - 1] += coeffs[j] * x_t[size - 1 + (i - j)] * x_t[size - 1 + (i - j)];
			else
				f_v[i - 1] += coeffs[j] * f_v[i - 1 - j]; // Seems to be the previously prognostizied log return except the innovation sequence number?
		}

		for (integer j = 1; j <= q; j++)
		{
			if (i - j < 1)
				f_v[i - 1] += coeffs[p + j] * h[size - 1 + (i - j)];
			else
				f_v[i - 1] += coeffs[p + j] * f_v[i - 1 - j];  // Seems to be the previously prognostizied log return except the innovation sequence number?
		}
    }

	return f_v;
}

class garch_fit_result
	option(category: 'Mathematics/Statistics/Garch/Garch Calibration')
{

public: 

	garch_fit_result(number a0,
                     vector(number) a,
                     vector(number) b,
                     number        dof = -1);

	number a0();
	vector(number) a();
	vector(number) b();

	number p();
	number q();
	number DegreeOfFreedom();

	vector(number)       var_estimate(vector(number) v);
	series<date>(number) var_estimate(series<date>(number) s, calendar cal, logical fill_hole = false);

	vector(number)       var_forecast(integer nr_periods, vector(number) v);
	series<date>(number) var_forecast(integer nr_periods, series<date>(number) s, calendar cal, logical fill_hole = false);

	number long_range_variance();

private:

	number a0_;
	vector(number)  a_;
	vector(number)  b_;
	integer	    p_;
	integer	    q_;

	number dof_;
};


garch_fit_result.garch_fit_result(number a0,
                     vector(number) a,
                     vector(number) b,
                     number        dof): a0_(a0), a_(a), b_(b), p_(v_size(a)), q_(v_size(b)), dof_(dof)
{
}

number garch_fit_result.a0()
{
	return a0_;
}

vector(number) garch_fit_result.a()
{
	return a_;
}

vector(number) garch_fit_result.b()
{
	return b_;
}

number garch_fit_result.p()
{
	return p_;
}

number garch_fit_result.q()
{
	return q_;
}

number garch_fit_result.DegreeOfFreedom()
{
	return dof_;
}

number garch_fit_result.long_range_variance()
{
	number lr;
	number asum = 0;
	number bsum = 0;

    for (integer i = 0; i < p_; i++)
		asum += a_[i];

    for (integer i = 0; i < q_; i++)
		bsum += b_[i];

    lr = (a0_) / (1 - asum - bsum);
    return lr;
}


vector(number) garch_fit_result.var_estimate(vector(number) v)
{
	integer length = v_size(v);

	vector(number) h[length];
	vector(number) coeffs[0];

	number var0 = v_variance(v);

    // Fyll i koefficientvektorn
     
	push_back(coeffs, a0_);

    for (integer i = 0; i < p_; i++)
		push_back(coeffs, a_[i]);

    for (integer i = 0; i < q_; i++)
		push_back(coeffs, b_[i]);

    var_vect_fill(h, v,  coeffs, var0, p_, q_); 

	return h;
}

series<date>(number) garch_fit_result.var_estimate(series<date>(number) s, calendar cal, logical fill_hole)
{
   integer size_s = s_size(s);
    
	date   s_start = s_x0(s);
	integer date_dx = s_dx(s);

    series<date>(number) var_series = create_date_series(size_s, s_start, date_dx);

    // Korrigerar inkommande data for helgdagar

	vector(number) s_t;
    series_wash(s_t, s, cal, fill_hole);

	integer length = v_size(s_t);

	vector(number) h[length];
	vector(number) coeffs;

	number var0 = v_variance(s_t);

    // Fyller i koefficientvektorn
    
    push_back(coeffs, a0_);

    for (integer i = 0; i < p_; i++)
		push_back(coeffs, a_[i]);

    for (integer i = 0; i < q_; i++)
		push_back(coeffs, b_[i]);

    var_vect_fill(h, s_t, coeffs, var0, p_, q_); 
    
    // Create the returning var_serie

	integer iter = 0;
	date  d    = s_start;

    for (integer k = 0; k < size_s; k++)
	{
		if (!cal.is_holiday(d))
		{
			if (!null(s[k]) || fill_hole == true)
				var_series[k] = h[iter++];
		}

		d += date_dx;
    }    

    return var_series;

}

vector(number) garch_fit_result.var_forecast(integer nr_periods, vector(number) v)
{
	integer length = v_size(v);

	vector(number)	x_t[length];
	vector(number)	h[length];
	vector(number)	coeffs;
    
    if (nr_periods < 0)
		throw(E_INVALID_ARG, "nr_periods must be >= 0");

	x_t = clone_vector(v);

	number var0 = v_variance(x_t);

    // Fylla i koefficientvektorn
    
    push_back(coeffs, a0_);

    for (integer i = 0; i < p_; i++)
		push_back(coeffs, a_[i]);

    for (integer i = 0; i < q_; i++)
		push_back(coeffs, b_[i]);

    var_vect_fill(h, x_t, coeffs, var0, p_, q_); 

    vector(number) f_v = forecast_engine(nr_periods, coeffs, x_t, h, p_, q_);
    
    // var_vect är utdata som består av skattade varatiliteterna plus de n st
    // prognosticerade värdena.
    
	vector(number) var_vect = one_vector(v_size(h) + nr_periods);

	integer hsize = v_size(h);

    for (integer i = 0; i < hsize; i++) 
		var_vect[i] = h[i];

    for (integer i = 0; i < nr_periods ; i++)
		var_vect[hsize + i] = f_v[i];

    return var_vect;

}

series<date>(number) garch_fit_result.var_forecast(integer nr_periods, series<date>(number) s, calendar cal, logical fill_hole)
{
   if (nr_periods < 0)
		throw(E_INVALID_ARG, "nr_periods must be >= 0");
    
	integer size_s  = s_size(s);
    date    s_start = s_x0(s);
    integer  date_dx = s_dx(s);

    series<date>(number) var_series = create_date_series(size_s + nr_periods, s_start, date_dx);

	vector(number) s_t;
    series_wash(s_t, s, cal, fill_hole);

	number var0 = v_variance(s_t);

	vector(number) h[v_size(s_t)];
	vector(number) coeffs;

    push_back(coeffs, a0_);

    for (integer i = 0; i < p_; i++)
		push_back(coeffs, a_[i]);

    for (integer i = 0; i < q_; i++)
		push_back(coeffs, b_[i]);

    var_vect_fill(h, s_t, coeffs, var0, p_, q_);
    
    vector(number) f_v = forecast_engine(nr_periods, coeffs, s_t, h, p_, q_);
    
    // create return_series
	integer iter = 0;
	date  d    = s_start;

    for (integer k = 0; k < size_s; k++)
	{
		if(!cal.is_holiday(d))
			if(!null(s[k]) || fill_hole == true)
				var_series[k] = h[iter++];

		d += date_dx;
    }

    for (integer i = 0 ; i < nr_periods ; i++)
		var_series[size_s + i] = f_v[i];

    return var_series;

}


void shift(out vector(number) v, number temp) 
{
	integer n = v_size(v);

    for (integer i = n - 1 ; i > 0 ; i--)
	{
		v[i] = v[i - 1];
	}

    v[0] = temp;
}

void shift_row(out matrix(number) m, integer row_idx, number temp) 
{
	integer n = n_cols(m);

    for (integer i = n - 1 ; i > 0 ; i--)
	{
		m[row_idx, i] = m[row_idx, i - 1];
	}

    m[row_idx, 0] = temp;
}





// The first argument v is the output...
vector(number) likelyhood_derivatives(vector(number) x0,
                                      vector(number) x,
                                      number var0,
                                      vector(number) h,
                                      number my,
                                      integer p, 
                                      integer q) 
{
    // x0 = [a0,a1,a2,...,ap,b1,...,bq,u1,u2,u3,...,u(2+p),u(2+p+1),...,u(2+p+q)] // No idea what u is yet...

    integer vsize = 2 * (p + q) + 3;
    integer xsize = v_size(x);

	vector(number) v = one_vector(vsize, 0);

	vector(number) dda0[q+1];
	matrix(number) ddak[p,q+1];
	matrix(number) ddbk[q,q+1];

	number temp = 1;
    
    dda0[0]=1;
    for (integer i = 1; i <= q; i++)
		dda0[i] = 0;

    for (integer i = 0; i < p; i++)
	{
		ddak[i, 0] = var0;

		for (integer j = 1; j <= q; j++)
			ddak[i, j] = 0;
    }

    for (integer i = 0; i < q; i++)
	{
		ddbk[i, 0] = var0;

		for (integer j = 1; j <= q; j++)
			ddbk[i, j] = 0;
    }
  
    for(integer i = 0; i < xsize; i++) 
    {
		number talj = 0.5 * (1 / h[i] - (x[i] / h[i]) * (x[i] / h[i]));
	
		v[0] += talj * dda0[0];
	
		temp = 1;
		for (integer k = 1; k <= q; k++)
			temp += x0[p+k] * dda0[k-1];

		shift(dda0, temp);
	
		//a1..ap
		for (integer j = 1; j <= p; j++)
			v[j] += talj * ddak[j-1, 0];

		for (integer j = 0; j < p; j++)
		{
			if((i-j) < 0)
				temp = var0;
			else 
				temp = x[i-j] * x[i-j];

			for (integer k = 1; k <= q; k++)
				temp += x0[p+k] * ddak[j, k-1];

			shift_row(ddak, j, temp);
		}


		//b1..bq
		for (integer j = 1; j <= q; j++)
			v[p+j] += talj * ddbk[j-1, 0];

		for (integer j = 0; j < q; j++)
		{	    
			if((i-j) < 0)
				temp = var0;
			else
				temp = h[i-j];

			for (integer k = 1; k <= q; k++)
				temp += x0[p+k] * ddbk[j, k-1];

			shift_row(ddbk, j, temp);
		}
    }

    //
    v[0] = v[0] - x0[p+q+2];
    
    for (integer i = 1; i <= p+q; i++)
	{
		v[i]	    += x0[p+q+1] - x0[p+q+2+i];
		v[p+q+1]    += x0[i];
    }
    
    // sumak + sumbk < 1 (this is a side condition. What does this have to do with the below?)

    v[p+q+1] += -1; 
    v[p+q+1] *= x0[p+q+1]; 
    v[p+q+1] += my;
    
    // övriga bivillkor (what is is doing?)

    for (integer i = 0; i <= p+q; i++) 
		v[p+q+2+i] = x0[p+q+2+i] * (-x0[i]) + my;

	return v;
}


vector(number) likelyhood_derivatives_student_t(vector(number) x0,
                                                vector(number) x,
                                                number var0,
                                                vector(number) h,
                                                number my,
                                                integer p, 
                                                integer q,
                                                number degrees_of_freedom) 
{
    // x0 = [a0,a1,a2,...,ap,b1,...,bq,u1,u2,u3,...,u(2+p),u(2+p+1),...,u(2+p+q)] // No idea what u is yet...

	number mu = degrees_of_freedom;

    integer vsize = 2 * (p + q) + 3;
    integer xsize = v_size(x);

	vector(number) v = one_vector(vsize, 0);

	vector(number) dda0[q+1];
	matrix(number) ddak[p,q+1];
	matrix(number) ddbk[q,q+1];

	number temp = 1;
    
    dda0[0]=1;
    for (integer i = 1; i <= q; i++)
		dda0[i] = 0;

    for (integer i = 0; i < p; i++)
	{
		ddak[i, 0] = var0;

		for (integer j = 1; j <= q; j++)
			ddak[i, j] = 0;
    }

    for (integer i = 0; i < q; i++)
	{
		ddbk[i,0] = var0;

		for (integer j = 1; j <= q; j++)
			ddbk[i, j] = 0;
    }
  
    for(integer i = 0; i < xsize; i++) 
    {
		number talj = -0.5 * ((mu + 1) * x[i] * x[i] / ((mu - 2.0) * h[i] * h[i] + h[i] * x[i] * x[i]) - 1.0 / h[i]);
	
		v[0] += talj * dda0[0];
	
		temp = 1;
		for (integer k = 1; k <= q; k++)
			temp += x0[p+k] * dda0[k-1];

		shift(dda0, temp);
	
		//a1..ap
		for (integer j = 1; j <= p; j++)
			v[j] += talj * ddak[j-1, 0];

		for (integer j = 0; j < p; j++)
		{
			if((i-j) < 0)
				temp = var0;
			else 
				temp = x[i-j] * x[i-j];

			for (integer k = 1; k <= q; k++)
				temp += x0[p+k] * ddak[j, k-1];

			shift_row(ddak, j, temp);
		}


		//b1..bq
		for (integer j = 1; j <= q; j++)
			v[p+j] += talj * ddbk[j-1, 0];

		for (integer j = 0; j < q; j++)
		{	    
			if((i-j) < 0)
				temp = var0;
			else
				temp = h[i-j];

			for (integer k = 1; k <= q; k++)
				temp += x0[p+k] * ddbk[j, k-1];

			shift_row(ddbk, j, temp);
		}
    }

    //
    v[0] = v[0] - x0[p+q+2];
    
    for (integer i = 1; i <= p+q; i++)
	{
		v[i]	    += x0[p+q+1] - x0[p+q+2+i];
		v[p+q+1]    += x0[i];
    }
    
    // sumak + sumbk < 1 (this is a side condition. What does this have to do with the below?)

    v[p+q+1] += -1; 
    v[p+q+1] *= x0[p+q+1]; 
    v[p+q+1] += my;
    
    // övriga bivillkor (what is is doing?)

    for (integer i = 0; i <= p+q; i++) 
		v[p+q+2+i] = x0[p+q+2+i] * (-x0[i]) + my;

	return v;
}



matrix(number) likelyhood_second_derivatives(vector(number) x0,
                                             vector(number) x,
                                             number         var0, 
                                             vector(number) h,
                                             integer p, 
                                             integer q)
{   
    integer msize = 2 * (p + q) + 3;
	integer xsize = v_size(x);    

	matrix(number) m[msize, msize];

	for (integer i = 0; i < msize; i++)
	{
		for (integer j = 0; j < msize; j++)
			m[i, j] = 0;
	}

	number temp;
	vector(number) dda0[q+1];
	matrix(number) ddak[p,q+1];
	matrix(number) ddbk[q,q+1];
	matrix(number) ddbka0[q,q+1];

	vector(matrix_holder) ddbkak[p];
	vector(matrix_holder) ddbkbk[q];

    dda0[0] = 1;
    for (integer i = 1; i <= q; i++)
		dda0[i] = 0;
  
    for (integer i = 0; i < p; i++)
	{
		ddak[i, 0] = var0;
		for (integer j = 1; j <= q; j++)
			ddak[i, j] = 0;
    }
    
    for (integer i = 0; i < q; i++)
	{
		ddbk[i, 0] = var0;
		for (integer j = 1; j <= q; j++)
			ddbk[i, j] = 0;
    }

	for (integer i = 0; i < q; i++)
	{
		for (integer j = 0; j < q + 1; j++)
			ddbka0[i, j] = 0;
	}
    
    for (integer i = 0; i < p; i++)
	{	
		ddbkak[i] = new matrix_holder(q,q+1);

		for (integer j = 0; j < q; j++)
			for (integer k = 0; k < q+1; k++)
				ddbkak[i].m[j, k] = 0;
    }
    
    for (integer i = 0; i < q; i++)
	{
		ddbkbk[i] = new matrix_holder(q,q+1);
	
		for (integer j = 0; j < q; j++) 
			for (integer k = 0; k < q+1; k++)
				ddbkbk[i].m[j, k] = 0;
	
    }

  
	for (integer i = 0; i < xsize; i++)
	{
		number talj1 = 0.5 * (1 / h[i] - (x[i] / h[i]) * (x[i] / h[i]));
		number talj2 = 0.5 * (-1 / (h[i] * h[i]) + 2 * x[i] * x[i] / (h[i] * h[i] * h[i]));

		//dda0a0
		m[0, 0] += talj2 * dda0[0] * dda0[0];

		//ddaka0
		for (integer j = 1; j <= p; j++)
			m[0, j] += talj2 * ddak[j - 1, 0] * dda0[0];

		//ddbka0
		for (integer j = 1; j <= q; j++)
			m[0, p + j] += talj1 * ddbka0[j - 1, 0] + talj2 * ddbk[j - 1, 0] * dda0[0];

		//ddakak
		for (integer j = 1; j <= p; j++)
		{
			for (integer k = 1; k <= p; k++)
				m[j, k] += talj2 * ddak[j - 1, 0] * ddak[k - 1, 0];
		}

		//ddbkak
		for (integer j = 1; j <= p; j++)
		{
			for (integer k = 1; k <= q; k++)
				m[j, p + k] += talj1 * ddbkak[j - 1].m[k - 1, 0] + talj2 * ddak[j - 1, 0] * ddbk[k - 1, 0];
		}

		//ddbkbk
		for (integer j = 1; j <= q; j++)
		{
			for (integer k = 1; k <= q; k++)
				m[p + j, p + k] += talj1 * ddbkbk[j - 1].m[k - 1, 0] + talj2 * ddbk[j - 1, 0] * ddbk[k - 1, 0];
		}

		//uppdatera dda0
		temp = 1;
		for (integer k = 1; k <= q; k++)
			temp += x0[p + k] * dda0[k - 1];

		shift(dda0, temp);

		//uppdatera ddak
		for (integer j = 0; j < p; j++)
		{

			if ((i - j) < 0)
				temp = var0;
			else
				temp = x[i - j] * x[i - j];

			for (integer k = 1; k <= q; k++)
				temp += x0[p + k] * ddak[j, k - 1];

			shift_row(ddak, j, temp);
		}

		//uppdatera ddbk
		for (integer j = 0; j < q; j++)
		{
			if ((i - j) < 0)
				temp = var0;
			else
				temp = h[i - j];

			for (integer k = 1; k <= q; k++)
				temp += x0[p + k] * ddbk[j, k - 1];

			shift_row(ddbk, j, temp);
		}

		//uppdatera ddbka0
		for (integer j = 0; j < q; j++)
		{
			temp = dda0[j + 1];

			for (integer k = 1; k <= q; k++)
				temp += x0[p + k] * ddbka0[j, k - 1];

			shift_row(ddbka0, j, temp);
		}


		//uppdatera ddbkak
		for (integer j = 0; j < p; j++)
		{
			for (integer k = 0; k < q; k++)
			{
				temp = ddak[j, k + 1];

				for (integer l = 0; l < q; l++)
					temp += x0[p + l + 1] * ddbkak[j].m[k, l];

				shift_row(ddbkak[j].m, k, temp);
			}
		}


		//uppdatera ddbkbk
		for (integer j = 0; j < q; j++)
		{
			for (integer k = 0; k < q; k++)
			{
				temp = ddbk[j, k + 1] + ddbk[k, j + 1];

				for (integer l = 0; l < q; l++)
					temp += x0[p + l + 1] * ddbkbk[j].m[k, l];

				shift_row(ddbkbk[j].m, k, temp);
			}
		}
    }


    //ddaka0=dda0ak
	for (integer i = 1; i <= p; i++)
	    m[i, 0] = m[0, i];
    
    //ddbka0 = dda0bk
	for (integer i = 1; i <= q; i++)
	    m[p+i, 0] = m[0, p+i];

    //ddakbk = ddbkak
	for (integer i = 1; i<= p; i++) 
	    for (integer j = 1; j <= q; j++)
			m[p+j, i] = m[i, p+j];
	

    //dduka0
	m[0, p+q+2] = -1;
	
    //ddukak och ddukbk
	for (integer i = 1; i <= p+q; i++)
	{
	    m[i, p+q+1]	    = 1;
	    m[i, p+q+2+i]   = -1;
	}

    //ddaku1 och ddbku1 och ddu1u1
	for (integer i = 1; i <= p+q; i++)
	{    
	    m[p+q+1, i]	    = x0[p+q+1];
	    m[p+q+1, p+q+1] += x0[i];
	}
	
	m[p+q+1, p+q+1] += -1;

    //dda0uk och ddakuk och ddbkuk;
	for (integer i = 0; i <= p+q; i++)
	{
	    
	    m[p+q+2+i, i]       = -x0[p+q+2+i];
	    m[p+q+2+i, p+q+2+i] = -x0[i];	
	}

	return m;
}


matrix(number) likelyhood_second_derivatives_student_t(vector(number) x0,
                                                            vector(number) x,
                                                            number         var0, 
                                                            vector(number) h,
                                                            integer p, 
                                                            integer q,
                                                            number degrees_of_freedom)
{   
    integer msize = 2 * (p + q) + 3;
	integer xsize = v_size(x);    

	number mu = degrees_of_freedom;

	matrix(number) m[msize, msize];

	for (integer i = 0; i < msize; i++)
	{
		for (integer j = 0; j < msize; j++)
			m[i, j] = 0;
	}

	number temp;
	vector(number) dda0[q+1];
	matrix(number) ddak[p,q+1];
	matrix(number) ddbk[q,q+1];
	matrix(number) ddbka0[q,q+1];

	vector(matrix_holder) ddbkak[p];
	vector(matrix_holder) ddbkbk[q];

    dda0[0] = 1;
    for (integer i = 1; i <= q; i++)
		dda0[i] = 0;
  
    for (integer i = 0; i < p; i++)
	{
		ddak[i, 0] = var0;
		for (integer j = 1; j <= q; j++)
			ddak[i, j] = 0;
    }
    
    for (integer i = 0; i < q; i++)
	{
		ddbk[i, 0] = var0;
		for (integer j = 1; j <= q; j++)
			ddbk[i, j] = 0;
    }

	for (integer i = 0; i < q; i++)
	{
		for (integer j = 0; j < q + 1; j++)
			ddbka0[i, j] = 0;
	}
    
    for (integer i = 0; i < p; i++)
	{	
		ddbkak[i] = new matrix_holder(q,q+1);

		for (integer j = 0; j < q; j++)
			for (integer k = 0; k < q+1; k++)
				ddbkak[i].m[j, k] = 0;
    }
    
    for (integer i = 0; i < q; i++)
	{
		ddbkbk[i] = new matrix_holder(q,q+1);
	
		for (integer j = 0; j < q; j++) 
			for (integer k = 0; k < q+1; k++)
				ddbkbk[i].m[j, k] = 0;
	
    }

  
	for (integer i = 0; i < xsize; i++)
	{
		number talj1 = -0.5 * ((mu + 1) * x[i] * x[i] / ((mu - 2.0) * h[i] * h[i] + h[i] * x[i] * x[i]) - 1.0 / h[i]);
		number talj2 = (((mu + 1) / 2) * x[i] * x[i] / ((mu - 2) * h[i] + x[i] * x[i]) - 0.5) / (h[i] * h[i]);
		talj2 += (((mu + 1) / 2) * (mu - 2) * x[i] * x[i] / (((mu - 2) * h[i] + x[i] * x[i]) * ((mu - 2) * h[i] + x[i] * x[i]))) / h[i];

		//dda0a0
		m[0, 0] += talj2 * dda0[0] * dda0[0];

		//ddaka0
		for (integer j = 1; j <= p; j++)
			m[0, j] += talj2 * ddak[j - 1, 0] * dda0[0];

		//ddbka0
		for (integer j = 1; j <= q; j++)
			m[0, p + j] += talj1 * ddbka0[j - 1, 0] + talj2 * ddbk[j - 1, 0] * dda0[0];

		//ddakak
		for (integer j = 1; j <= p; j++)
		{
			for (integer k = 1; k <= p; k++)
				m[j, k] += talj2 * ddak[j - 1, 0] * ddak[k - 1, 0];
		}

		//ddbkak
		for (integer j = 1; j <= p; j++)
		{
			for (integer k = 1; k <= q; k++)
				m[j, p + k] += talj1 * ddbkak[j - 1].m[k - 1, 0] + talj2 * ddak[j - 1, 0] * ddbk[k - 1, 0];
		}

		//ddbkbk
		for (integer j = 1; j <= q; j++)
		{
			for (integer k = 1; k <= q; k++)
				m[p + j, p + k] += talj1 * ddbkbk[j - 1].m[k - 1, 0] + talj2 * ddbk[j - 1, 0] * ddbk[k - 1, 0];
		}

		//uppdatera dda0
		temp = 1;
		for (integer k = 1; k <= q; k++)
			temp += x0[p + k] * dda0[k - 1];

		shift(dda0, temp);

		//uppdatera ddak
		for (integer j = 0; j < p; j++)
		{

			if ((i - j) < 0)
				temp = var0;
			else
				temp = x[i - j] * x[i - j];

			for (integer k = 1; k <= q; k++)
				temp += x0[p + k] * ddak[j, k - 1];

			shift_row(ddak, j, temp);
		}

		//uppdatera ddbk
		for (integer j = 0; j < q; j++)
		{
			if ((i - j) < 0)
				temp = var0;
			else
				temp = h[i - j];

			for (integer k = 1; k <= q; k++)
				temp += x0[p + k] * ddbk[j, k - 1];

			shift_row(ddbk, j, temp);
		}

		//uppdatera ddbka0
		for (integer j = 0; j < q; j++)
		{
			temp = dda0[j + 1];

			for (integer k = 1; k <= q; k++)
				temp += x0[p + k] * ddbka0[j, k - 1];

			shift_row(ddbka0, j, temp);
		}


		//uppdatera ddbkak
		for (integer j = 0; j < p; j++)
		{
			for (integer k = 0; k < q; k++)
			{
				temp = ddak[j, k + 1];

				for (integer l = 0; l < q; l++)
					temp += x0[p + l + 1] * ddbkak[j].m[k, l];

				shift_row(ddbkak[j].m, k, temp);
			}
		}


		//uppdatera ddbkbk
		for (integer j = 0; j < q; j++)
		{
			for (integer k = 0; k < q; k++)
			{
				temp = ddbk[j, k + 1] + ddbk[k, j + 1];

				for (integer l = 0; l < q; l++)
					temp += x0[p + l + 1] * ddbkbk[j].m[k, l];

				shift_row(ddbkbk[j].m, k, temp);
			}
		}
    }


    //ddaka0=dda0ak
	for (integer i = 1; i <= p; i++)
	    m[i,0] = m[0,i];
    
    //ddbka0 = dda0bk
	for (integer i = 1; i <= q; i++)
	    m[p+i, 0] = m[0, p+i];

    //ddakbk = ddbkak
	for (integer i = 1; i<= p; i++) 
	    for (integer j = 1; j <= q; j++)
			m[p+j, i] = m[i, p+j];
	

    //dduka0
	m[0, p+q+2] = -1;
	
    //ddukak och ddukbk
	for (integer i = 1; i <= p+q; i++)
	{
	    m[i, p+q+1]	    = 1;
	    m[i, p+q+2+i]   = -1;
	}

    //ddaku1 och ddbku1 och ddu1u1
	for (integer i = 1; i <= p+q; i++)
	{    
	    m[p+q+1, i]	    = x0[p+q+1];
	    m[p+q+1, p+q+1] += x0[i];
	}
	
	m[p+q+1, p+q+1] += -1;

    //dda0uk och ddakuk och ddbkuk;
	for (integer i = 0; i <= p+q; i++)
	{
	    
	    m[p+q+2+i, i]       = -x0[p+q+2+i];
	    m[p+q+2+i, p+q+2+i] = -x0[i];	
	}

	return m;
}


number norm_cal(vector(number) ders) // L2 norm of vector ders
{
	number	norm = 0;

	integer size = v_size(ders);
	
	for (integer i = 0; i < size; i++)
	    norm += ders[i] * ders[i];
	    
	norm = sqrt(norm);
	
	return norm;
}

void r_init(out vector(number) r, integer p, integer q, number my) 
{
    number alfa_init_sum = 0.15; 
    number beta_init_sum = 0.75; 
    
    r[0] = 0.00001; //a0 initieras
    
    for (integer i = 1; i <= p ; i++)
		r[i] = alfa_init_sum / p; //alfa_1 tom alfa_p

    for (integer i = 1; i <= q; i++)
		r[p + i] = beta_init_sum / q; //beta_1 tom beta_q
    
    r[p + q + 1] = my / (1 - alfa_init_sum-beta_init_sum); //Summan av alla variabler ar mindre an 1.
    r[p + q + 2] = my / r[0]; //u1;

    for (integer i = 1; i <= p; i++)
		r[p + q + 2 +i] = my / r[i];

    for (integer i = 1; i <= q; i++)
		r[p + q + 2 + p + i] = my / r[p+i];

}

//Den här funktionen tar en logserie och tar bort tomma helgdagar och andra hål samt medelvärdesjusterar 
//Alltihop sparas i en vektor där man kan se vilka data som egentligen skattas.
vector(number) garch_prepare_series (series<date>(number) s,
                                     calendar  cal,
                                     logical fh = false)
	option(category: 'Mathematics/Statistics/Garch/Garch Calibration')
{
	vector(number) r_v;
    
    series_wash(r_v, s, cal, fh);

    return r_v;
}


garch_fit_result garch_fit(vector(number) v, // historical log-returns
                           integer p,
                           integer q,
                           number tol = 0.000001,
                           number my0_in = 1.0)
	option(category: 'Mathematics/Statistics/Garch/Garch Calibration') 
{
	vector(number) x_t = clone_vector(v); // Maybe not necessary to clone, but whatever

	integer vsize = v_size(v);
   
    vector(number) h[vsize];	   // Variansvektor
	number var0 = v_variance(x_t); // Empiriska variansen
    
    number my0;
    
	// This seems very strange. Why make a sudden radical shift in the input value of my0_in when it is exactly equal to 1?
	// This will most likely introduce chaotic behaviour.
    if (my0_in == 1 && p == 1 && q == 1)
		my0 = 0.1;
    else
		my0 = my0_in;
    
	number my1 = my0;    
	number my  = my1;
    
    // Initierar parametervektorn
    vector(number) r[2 * (p + q) + 3];

    r_init(r, p, q, my);
    
	integer rsize = v_size(r);

    vector(number) first_derivatives[rsize];
    matrix(number) second_derivatives[rsize, rsize];

	number norm;
	number eps = tol;
	number alfa;
	number alfatemp;
	number r_sum;
	number grad_sum;

	integer max_iter   = 2000;
	integer counter	   = 1;
	logical did_accept = false;

    for (;;) // main loop to calibrate parameters.
    {
        // Fill the vector h with the variance 
        // h[t] = a0+alfa_1*x[t-1]^2+alfa_2*x[t-2]^2+...+beta_1*h[t-1]+beta_2*h[t-2]+...
	
		var_vect_fill(h, x_t, r, var0, p, q); 
		first_derivatives = likelyhood_derivatives(r, x_t, var0, h, my, p, q); 
	
		norm = norm_cal(first_derivatives); // Euclidean L2 norm
	
		if (counter++ > max_iter)
		{ 	 
			throw(E_CALC, "Function is not converging.Try another set of p and q or another part of the series");	    
		}

		if (!did_accept && norm < my) 
		{
			my /= 10;
			did_accept = true;
	    
			continue;
		}
	
		did_accept = false;

		if (my < eps)
			break;

        second_derivatives = likelyhood_second_derivatives(r, x_t, var0, h, p, q);
		first_derivatives  = solve_linear(second_derivatives, first_derivatives); // We store the results in the same variable.

		alfa     = 10;
		r_sum    = 0;
		grad_sum = 0;

		for (integer i = 0; i < p+q+1 ; i++)
		{
			if (first_derivatives[i] > 0)
			{   
				alfatemp = (r[i]) / first_derivatives[i];

				if (alfatemp < alfa) 
					alfa = alfatemp;
			}
		
			if (i > 0)
			{
				r_sum += r[i];
				grad_sum += first_derivatives[i];
			}   
		}
	    
		alfatemp = (r_sum - 1) / grad_sum;
	
		if (grad_sum < 0 && alfatemp < alfa)
			alfa = alfatemp;

		alfa *= 0.95;

		if (alfa < 0.1)
		{
			my1 += 5*my0;
	   
			r_init(r, p, q, my1);
			my = my1;
	        
			continue;
		}
		else if (alfa > 1)
			alfa = 1;

		for (integer j = 0 ; j < rsize ; j++) 
			r[j] -= alfa * first_derivatives[j];		
    } 

	number a0 = r[0];

	vector(number) a[p];
	vector(number) b[q];


    for (integer i = 1 ; i <= p; i++)
		a[i-1] = r[i]; 

    for (integer i = 1 ; i <= q; i++)
		b[i-1] = r[p + i];

    return new garch_fit_result(a0, a, b);
}


garch_fit_result garch_fit_student_t(vector(number) v, // historical log-returns
                                     integer p,
                                     integer q,
                                     number degrees_of_freedom = -1,
                                     number  tol = 0.000001,
                                     number  my0_in = 1.0) 
	option(category: 'Mathematics/Statistics/Garch/Garch Calibration')
{

	number mu = degrees_of_freedom;

	if (mu <= 2 && mu >= 0)
		throw(E_CALC, "The degree of freedom for the student t distribution should be greater than 2, to ensure finite first two moments.");

	vector(number) x_t = clone_vector(v);

	if (mu < 0) // This will happen is mu is not given, so then we calibrate it.
	{
		// We calibrate the parameter for the degree of freedom using kurtosis match :
		// sample kurtosis = 3 + 6 / (mu - 4)

		integer n = v_size(x_t);

		number m = 0;
		number CenteredSumPow2 = 0;
		number CenteredSumPow4 = 0;

		for (integer i = 0; i < n; i++)
		{
			m += x_t[i];
		}

		m /= n;

		number tmp;

		for (integer i = 0; i < n; i++)
		{
			tmp = (x_t[i] - m) * (x_t[i] - m);

			CenteredSumPow2 += tmp;
			CenteredSumPow4 += tmp * tmp;
		} 

		CenteredSumPow2 /= n;
		CenteredSumPow4 /= n;

		number kurtosis = CenteredSumPow4 / (CenteredSumPow2 * CenteredSumPow2);

		mu = 6 / (kurtosis - 3) + 4;

		if (mu <= 2)
			mu = 3; // If we end up here something is wrong. Defaulting to something "reasonable"
	}
	
	integer vsize = v_size(v);
   
    vector(number) h[vsize];	   // Variansvektor
	number var0 = v_variance(x_t); // Empiriska variansen
    
    number my0;
    
	// This seems very strange. Why make a sudden radical shift in the input value of my0_in when it is exactly equal to 1?
	// This will most likely introduce chaotic behaviour.
    if (my0_in == 1 && p == 1 && q == 1)
		my0 = 0.1;
    else
		my0 = my0_in;
    
	number my1 = my0;    
	number my  = my1;
    
    // Initierar parametervektorn
    vector(number) r[2 * (p + q) + 3];

    r_init(r, p, q, my);
    
	integer rsize = v_size(r);

    vector(number) first_derivatives[rsize];
    matrix(number) second_derivatives[rsize, rsize];

	number norm;
	number eps = tol;
	number alfa;
	number alfatemp;
	number r_sum;
	number grad_sum;

	integer max_iter   = 2000;
	integer counter	   = 1;
	logical did_accept = false;

    for (;;) // main loop to calibrate parameters.
    {
        // Fill the vector h with the variance 
        // h[t] = a0+alfa_1*x[t-1]^2+alfa_2*x[t-2]^2+...+beta_1*h[t-1]+beta_2*h[t-2]+...
	
		var_vect_fill(h, x_t, r, var0, p, q); 
		first_derivatives = likelyhood_derivatives_student_t(r, x_t, var0, h, my, p, q, mu); 
	
		norm = norm_cal(first_derivatives); // Euclidean L2 norm
	
		if (counter++ > max_iter)
		{ 	 
			throw(E_CALC, "Function is not converging.Try another set of p and q or another part of the series");	    
		}

		if (!did_accept && norm < my) 
		{
			my /= 10;
			did_accept = true;
	    
			continue;
		}
	
		did_accept = false;

		if (my < eps)
			break;

        second_derivatives = likelyhood_second_derivatives_student_t(r, x_t, var0, h, p, q, mu);
		first_derivatives  = solve_linear(second_derivatives, first_derivatives); // We store the results in the same variable.

		alfa     = 10;
		r_sum    = 0;
		grad_sum = 0;

		for (integer i = 0; i < p+q+1 ; i++)
		{
			if (first_derivatives[i] > 0)
			{   
				alfatemp = (r[i]) / first_derivatives[i];

				if (alfatemp < alfa) 
					alfa = alfatemp;
			}
		
			if (i > 0)
			{
				r_sum += r[i];
				grad_sum += first_derivatives[i];
			}   
		}
	    
		alfatemp = (r_sum - 1) / grad_sum;
	
		if (grad_sum < 0 && alfatemp < alfa)
			alfa = alfatemp;

		alfa *= 0.95;

		if (alfa < 0.1)
		{
			my1 += 5*my0;
	   
			r_init(r, p, q, my1);
			my = my1;
	        
			continue;
		}
		else if (alfa > 1)
			alfa = 1;

		for (integer j = 0 ; j < rsize ; j++) 
			r[j] -= alfa * first_derivatives[j];		
    } 

	number a0 = r[0];

	vector(number) a[p];
	vector(number) b[q];


    for (integer i = 1 ; i <= p; i++)
		a[i-1] = r[i]; 

    for (integer i = 1 ; i <= q; i++)
		b[i-1] = r[p + i];

    return new garch_fit_result(a0, a, b, mu);
}


///////////////// Classes used to simulate garch process //////////

//
// Basic class to generate the random variables used for the innovation sequence in garch (i.e the randomness to get the next value..).
// The generated random variable should have expected value = 0 and variance = 1.
//
// Typically we use normal random variables or student t variables so those variants are implemented as subclasses
//

class innovation_generator 
{
public :

	innovation_generator();

	void set_seed(number seed);

	virtual number gen_value() = 0;

protected :

	rng rand_;
	number seed_;	
};



innovation_generator.innovation_generator(){}

void innovation_generator.set_seed(number seed)
{
	seed_ = seed;
	rand_ = rng(millisecond(now()));
}



//////// Student t innovation generator ///////////////////

class t_generator : public innovation_generator
	option(category: 'Mathematics/Statistics/Garch/Garch Simulation')
{
public:

	t_generator(number dof, number option(nullable) seed = null<number>);
	virtual number gen_value();

private:

	number dof_;
};

t_generator.t_generator(number dof, number option(nullable) seed) : dof_(dof)
{
	if(null(seed))
		seed_ = millisecond(now());
	else
		seed_ = seed;

	rand_ = rng(round(seed_));
}

number t_generator.gen_value()
{
	number X = t_inv(rand_.uniform(), dof_);

	X *= sqrt((dof_ - 2) / dof_);

	return X;
}


//////// Normal variable generator //////////////////////////////

class normal_generator : public innovation_generator
	option(category: 'Mathematics/Statistics/Garch/Garch Simulation')
{
public:

	normal_generator(number option(nullable) seed = null<number>);
	virtual number gen_value();

private:

	rng rand_;
	number dof_;
};

normal_generator.normal_generator(number option(nullable) seed) 
{
	if(null(seed))
		seed_ = millisecond(now());
	else
		seed_ = seed;

	rand_ = rng(round(seed_));
}

number normal_generator.gen_value()
{
	return rand_.gauss();
}


///////////////////////////////////////////////////////////////
//// Main class to generate a Garch process for simulation. ///
///////////////////////////////////////////////////////////////


class garch_simulator
	option(category: 'Mathematics/Statistics/Garch/Garch Simulation')
{
public:

	// garch_simulator : /////////////////////////////////////////////////////////////////////////////////
	//             
	// a0 is the constant garch term. 
	//
	// a is a vector of coefficients (length = p) for the previous logreturns (vector y) 
	// squared in garch.
	//
	// b is the coefficients (length = q) 
	// of the previous variance terms (vector h).
	//
	// That is : y_[k] = a0 + a[0]*y^2_{k-1} + ... a[p-1]*y^2_{k-p} + b[0]*h_{k-1} + ... b[p-1]*h_{k-q}
	//
	// g is the generator object for the innovation sequence. See definition above.
	//
	// NOTE : The initial/historical values for the garch process will be choosen as 0 for the logreturns
	//        and 0.0001 for all variances, unless the init_start_values function is used to specify
	//        the values explicitly.
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	garch_simulator(number a0, vector(number) a, vector(number) b, innovation_generator g);


	// init_start_values : //////////////////////////////////////////////////////////////////////////////
	//
	// Initialize the starting/historical values in the garch process.
	// All previous logreturns will be initialized to 0 and all variances to the given value 
	// (recommended to use the long term var)
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void init_start_values(number long_term_var);


	// init_start_values : //////////////////////////////////////////////////////////////////////////////
	//
	// Initialize the starting/historical values in the garch process by given the earlier values 
	// explicitly. 
	// If there are fewer given values than needed, we create the necessary earlier values by constant 
	// extrapolation backwards.
	// If there are more values given than needed, the unnessecary values are ignored.
	//
    // NOTE : The input vectors should be in increasing time order as usual with the last element 
	// representing the value in the most recent historical step.
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	void init_start_values(vector(number) hist_log_returns, vector(number) hist_variances);


	// sim_logreturns : /////////////////////////////////////////////////////////////////////////////////
	//
	// Simulate a path of logreturns of length n_steps in the garch process. 
	// Optional starting seed, otherwise the seed will be constructed using the current timestamp.
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	vector(number) sim_logreturns(integer n_steps, number option(nullable) seed = null<number>);        


	// sim_var : ////////////////////////////////////////////////////////////////////////////////////////
	//
	// Simulate a path of variances of length n_steps in the garch process. 
	// Optional starting seed, otherwise the seed will be constructed using the current timestamp.
	/////////////////////////////////////////////////////////////////////////////////////////////////////
	vector(number) sim_var(integer n_steps, number option(nullable) seed = null<number>);     


	// sim_logreturns_and_var : /////////////////////////////////////////////////////////////////////////////////
	//
	// Simulate a path of both logreturns and variances of length n_steps simulataneously in the garch process.
	// The result at each timestep is represented as pairs (x,y) where  x = logreturn, y = variance
	//
	// Optional starting seed, otherwise the seed will be constructed using the current timestamp.
	/////////////////////////////////////////////////////////////////////////////////////////////////////           
	vector(point_number) sim_logreturns_and_var(integer n_steps, number option(nullable) seed = null<number>); 


	// sim_final_logreturns : /////////////////////////////////////////////////////////////////////////////////
	//
	// Simulate a sample of size n_paths of the final distribution of logreturns after n_steps.
	// Optional starting seed, otherwise the seed will be constructed using the current timestamp.
	///////////////////////////////////////////////////////////////////////////////////////////////////////////
	vector(number) sim_final_logreturns(integer n_steps, integer n_paths, number option(nullable) seed = null<number>);


	// sim_final_var : /////////////////////////////////////////////////////////////////////////////////
	//
	// Simulate a sample of size n_paths of the final distribution of variances after n_steps.
	// Optional starting seed, otherwise the seed will be constructed using the current timestamp.
	///////////////////////////////////////////////////////////////////////////////////////////////////////////
	vector(number)  sim_final_var(integer n_steps, integer n_paths, number option(nullable) seed = null<number>); 


	// sim_final_logreturns_and_var : /////////////////////////////////////////////////////////////////////////////////
	//
	// Simulate the final distributions of both logreturns and variances of length n_steps simulataneously in the garch process.
	// The result is represented as vector of pairs (x,y) where  x = logreturn, y = variance
	//
	// Optional starting seed, otherwise the seed will be constructed using the current timestamp.
	///////////////////////////////////////////////////////////////////////////////////////////////////// ////////////// 
	vector(point_number) sim_final_logreturns_and_var(integer n_steps, integer n_paths, number option(nullable) seed = null<number>); 


private:

	number a0_;
	vector(number) a_;
	vector(number) b_;

	vector(number) hist_logreturns_;
	vector(number) hist_variances_;

	innovation_generator g_;
};


garch_simulator.garch_simulator(number a0, vector(number) a, vector(number) b, innovation_generator g): a0_(a0), a_(a), b_(b), g_(g)
{
	number default_vol = 0.0001;

	hist_logreturns_ = one_vector(v_size(a_), 0);
	hist_variances_  = one_vector(v_size(b_), default_vol);

}

void garch_simulator.init_start_values(number long_term_var)
{
	hist_logreturns_ = one_vector(v_size(a_), 0);
	hist_variances_  = one_vector(v_size(b_), long_term_var);
}


void garch_simulator.init_start_values(vector(number) hist_log_returns, vector(number) hist_variances)
{
	integer p = v_size(a_);
	integer q = v_size(b_);

	integer p_input = v_size(hist_log_returns);
	integer q_input = v_size(hist_variances);

	hist_logreturns_ = one_vector(p, 0);
	hist_variances_  = one_vector(q, 0);

	if(p_input < p)
	{
		if(p_input > 0)
		{
			for(integer i=0; i<p_input; i++)
			{
				hist_logreturns_[p - 1 - i] = hist_log_returns[p_input - 1 - i];
			}

			for(integer i=p_input; i<p; i++)
			{
				hist_logreturns_[p - 1 - i] = hist_log_returns[0];
			}

		}
	}
	else
	{
		for(integer i=0; i<p; i++)
		{
			hist_logreturns_[p - 1 - i] = hist_log_returns[p_input - 1 - i];
		}	
	}

	if(q_input < q)
	{
		if(q_input > 0)
		{
			for(integer i=0; i<q_input; i++)
			{
				hist_variances_[q - 1 - i] = hist_variances[q_input - 1 - i];
			}

			for(integer i=q_input; i<q; i++)
			{
				hist_variances_[q - 1 - i] = hist_variances[0];
			}
		}
		else
		{
			hist_variances_ = one_vector(q, 0.0001);
		}	
	}
	else
	{
		for(integer i=0; i<q; i++)
		{
			hist_variances_[q - 1 - i] = hist_variances[q_input - 1 - i];
		}	
	}


}

vector(point_number) garch_simulator.sim_logreturns_and_var(integer n_steps, number option(nullable) seed)
{
	if(!null(seed))
		g_.set_seed(seed);

	vector(number) logreturns[n_steps];
	vector(number) variances[n_steps];

	vector(point_number) res[n_steps];

	integer p = v_size(a_);
	integer q = v_size(b_);


	for(integer i=0; i<n_steps; i++)
	{
		number sum = a0_;

		for(integer j=0; j<p; j++)
		{
			if(i-j >= 1)
			{
				sum += a_[j] * logreturns[i-j-1] * logreturns[i-j-1];
			}
			else
			{
				sum += a_[j] * hist_logreturns_[p - 1 + i - j] * hist_logreturns_[p - 1 + i - j];
			}			
		}

		for(integer j=0; j<q; j++)
		{
			if(i-j >= 1)
			{
				sum += b_[j] * variances[i-j-1];
			}
			else
			{
				sum += b_[j] * hist_variances_[q - 1 + i - j];				
			}
		}

		number X = g_.gen_value();

		logreturns[i] = sqrt(sum) * X;
		variances[i]  = sum;

		res[i] = point(logreturns[i], variances[i]);
	}

	return res;
}

vector(number) garch_simulator.sim_logreturns(integer n_steps, number option(nullable) seed)
{
	vector(point_number) res = sim_logreturns_and_var(n_steps, seed);

	vector(number) r = res.x();

	return r;
}

vector(number) garch_simulator.sim_var(integer n_steps, number option(nullable) seed)
{
	return sim_logreturns_and_var(n_steps, seed).y();
}



vector(point_number) garch_simulator.sim_final_logreturns_and_var(integer n_steps, integer n_paths, number option(nullable) seed)
{
	if(!null(seed))
		g_.set_seed(seed);

	vector(point_number) res[n_paths];
	vector(point_number) tmp;

	for(integer k=0; k<n_paths; k++)
	{
		tmp    = sim_logreturns_and_var(n_steps);
		res[k] = tmp[n_steps - 1];
	}

	return res;
}

vector(number) garch_simulator.sim_final_logreturns(integer n_steps, integer n_paths, number option(nullable) seed)
{
	return sim_final_logreturns_and_var(n_steps, n_paths, seed).x();
}

vector(number) garch_simulator.sim_final_var(integer n_steps, integer n_paths, number option(nullable) seed)
{
	return sim_final_logreturns_and_var(n_steps, n_paths, seed).y();
}


