option (null:hard);

module http_api {

string default_status_message(integer status_code)
{
	switch (status_code) {
	case 100: return "Continue";
	case 101: return "Switching Protocols";
	case 102: return "Processing";
	case 103: return "Early Hints";
	case 200: return "OK";
	case 201: return "Created";
	case 202: return "Accepted";
	case 203: return "Non-Authoritative Information";
	case 204: return "No Content";
	case 205: return "Reset Content";
	case 206: return "Partial Content";
	case 207: return "Multi-Status";
	case 208: return "Already Reported";
	case 226: return "IM Used";
	case 300: return "Multiple Choices";
	case 301: return "Moved Permanently";
	case 302: return "Found";
	case 303: return "See Other";
	case 304: return "Not Modified";
	case 305: return "Use Proxy";
	case 306: return "Switch Proxy";
	case 307: return "Temporary Redirect";
	case 308: return "Permanent Redirect";
	case 400: return "Bad Request";
	case 401: return "Unauthorized";
	case 402: return "Payment Required";
	case 403: return "Forbidden";
	case 404: return "Not Found";
	case 405: return "Method Not Allowed";
	case 406: return "Not Acceptable";
	case 407: return "Proxy Authentication Required";
	case 408: return "Request Timeout";
	case 409: return "Conflict";
	case 410: return "Gone";
	case 411: return "Length Required";
	case 412: return "Precondition Failed";
	case 413: return "Payload Too Large";
	case 414: return "URI Too Long";
	case 415: return "Unsupported Media Type";
	case 416: return "Range Not Satisfiable";
	case 417: return "Expectation Failed";
	case 418: return "I'm a teapot";
	case 421: return "Misdirected Request";
	case 422: return "Unprocessable Entity";
	case 423: return "Locked";
	case 424: return "Failed Dependency";
	case 425: return "Too Early";
	case 426: return "Upgrade Required";
	case 428: return "Precondition Required";
	case 429: return "Too Many Requests";
	case 431: return "Request Header Fields Too Large";
	case 451: return "Unavailable For Legal Reasons";
	case 500: return "Internal Server Error";
	case 501: return "Not Implemented";
	case 502: return "Bad Gateway";
	case 503: return "Service Unavailable";
	case 504: return "Gateway Timeout";
	case 505: return "HTTP Version Not Supported";
	case 506: return "Variant Also Negotiates";
	case 507: return "Insufficient Storage";
	case 508: return "Loop Detected";
	case 510: return "Not Extended";
	case 511: return "Network Authentication Required";
	default: throw (E_UNSPECIFIC, strcat("Non standard status code: ", str(status_code)));
	}
}

http.response mk_response(integer status_code)
{
	return new http.response(status_code, default_status_message(status_code));
}

http.response __mk_json_response(integer status_code, string body, logical deflate)
{
	http.response resp = new http.response(status_code, default_status_message(status_code));
	resp.set_header("Content-Type", "application/json");

	if (deflate) {
		resp.set_header("Content-Encoding", "gzip");
		resp.set_body(zlib.deflate(new blob(body), zlib.format.FMT_GZIP));
	} else {
		resp.set_body(body);
	}

	return resp;
}

http.response mk_deflated_response(integer status_code, json.node jn, integer indent = -1) =
	__mk_json_response(status_code, indent < 0 ? jn.print() : jn.print(indent), true);

http.response mk_deflated_response(integer status_code, json.builder jb) =
	__mk_json_response(status_code, jb.get(), true);

http.response mk_response(integer status_code, json.node jn, integer indent = -1) =
	__mk_json_response(status_code, indent < 0 ? jn.print() : jn.print(indent), false);

http.response mk_response(integer status_code, json.builder jb) =
	__mk_json_response(status_code, jb.get(), false);

http.response mk_response_raw(
	integer status_code,
	string option(nullable) status_msg,
	string option(nullable) content_type,
	string option(nullable) body)
{
	if (null(status_msg))
		status_msg = default_status_message(status_code);

	http.response resp = new http.response(status_code, status_msg);

	if (!null(content_type))
		resp.set_header("Content-Type", content_type);

	if (!null(body))
		resp.set_body(body);

	return resp;
}

http.response mk_error_response(integer status_code, string msg)
{
	json.builder jb = new json.builder();
	jb.begin_object();
	jb.begin_member("error");
	jb.append(msg);
	jb.end_object();
	return __mk_json_response(status_code, jb.get(), false);
}

http.response mk_method_not_allowed_response(vector(string) allowed_methods)
{
	if (v_size(allowed_methods) == 0)
		throw (E_UNSPECIFIC, "Must allow some method for mk_method_not_allowed_response");

	integer n = v_size(allowed_methods);
	vector(http.method) methods[n];
	for (integer i = 0; i < n; ++i) {
		str_to_enum(allowed_methods[i], methods[i]);
	}

	integer status_code = 405;
	http.response resp = new http.response(status_code, default_status_message(status_code));
	resp.set_header("Allow", str_join(enum_id(methods), ", "));
	return resp;
}

void __apply_cors_headers_if_origin(
	http.response           resp,
	string option(nullable) origin)
{
	if (null(origin))
		return;
	string methods = "POST, GET, PUT, PATCH";
	string headers = "Authorization";
	resp.set_header("Access-Control-Allow-Origin",  origin);
	resp.set_header("Access-Control-Allow-Methods", methods);
	resp.set_header("Access-Control-Allow-Headers", headers);
}

http.response mk_origin_not_allowed_response()
{
	http.response resp =
		mk_response_raw(203, "Origin not allowed", "text/plain", null<string>);
	__apply_cors_headers_if_origin(resp, "");
	return resp;
}

logical origin_allowed(
	http.request req,
	vector(regex) allowed_origins)
{
	string origin = req.header("Origin");
	if (!null(origin))
		return v_size(v_find(allowed_origins.match(origin), [true])) > 0;
	return true;
}

void enforce_cors_if_origin(
	out http.response resp,
	http.request req,
	vector(regex) allowed_origins)
{
	if (origin_allowed(req, allowed_origins))
		__apply_cors_headers_if_origin(resp, req.header("Origin"));
	else
		resp = mk_origin_not_allowed_response();
}

json.builder mk_json(string name, string value, integer indent = 0)
{
	json.builder jb = new json.builder(indent);
	jb.begin_object();
	jb.begin_member(name);
	jb.append(value);
	jb.end_object();
	return jb;
}
json.builder mk_json(string name, vector(string) value, integer indent = 0)
{
	json.builder jb = new json.builder(indent);
	jb.begin_object();
	jb.begin_member(name);
	jb.append(value);
	jb.end_object();
	return jb;
}
json.builder mk_json(string name, vector(number) value, integer indent = 0)
{
	json.builder jb = new json.builder(indent);
	jb.begin_object();
	jb.begin_member(name);
	jb.append(value);
	jb.end_object();
	return jb;
}
json.builder mk_json(string name, vector(integer) value, integer indent = 0)
{
	json.builder jb = new json.builder(indent);
	jb.begin_object();
	jb.begin_member(name);
	jb.append(value);
	jb.end_object();
	return jb;
}
json.builder mk_json(string name, vector(date) value, integer indent = 0)
{
	json.builder jb = new json.builder(indent);
	jb.begin_object();
	jb.begin_member(name);
	jb.append(value);
	jb.end_object();
	return jb;
}


number get_json_number(json.node node, string member, number option(nullable) dflt)
{
	if (!node.has_object_member(member))
		return dflt;
	return node.get_object_member(member).get_number();
}

number get_json_number(json.node node, string member)
{
	return node.get_object_member(member).get_number();
}

string get_json_string(json.node node, string member, string option(nullable) dflt)
{
	if (!node.has_object_member(member))
		return dflt;
	return node.get_object_member(member).get_string();
}

string get_json_string(json.node node, string member)
{
	return node.get_object_member(member).get_string();
}

class json_masker {
public:
	void add_mask(vector(string) member_path, json.node replacement);
	json.node mask_values(json.node jn);
	vector(json.node) get_replaced_values();

private:
	vectors_str paths_;
	vector(json.node) replacements_;
	vector(json.node) replaced_;

	json.node _mask_value_recursive(json.builder jb, json.node jn, vector(string) path, json.node mask);
};

void json_masker.add_mask(vector(string) member_path, json.node replacement)
{
	if (null(paths_))
		paths_ = new vectors_str();

	paths_.push_back(member_path);
	push_back(replacements_, replacement);
}

json.node json_masker._mask_value_recursive(json.builder jb, json.node jn, vector(string) path, json.node mask)
{
	string sought = path[0];
	vector(string) new_path = v_size(path) > 1 ? path[1:] : vector(i:0; "");

	json.node replaced_node;

	if (str_startswith(sought, "[") && str_endswith(sought, "]")) {
		integer array_sz = jn.get_array_size();
		string number_list = sub_string_start(sub_string_end(sought, 1), -1);
		vector(integer) array_ixs;
		if (str_trim(number_list) == "*") {
			array_ixs = vector(i:array_sz; i);
		} else if (str_trim(number_list) == "") {
			array_ixs = vector(i:0; 0);
		} else {
			try {
				array_ixs = str_to_integer(str_tokenize(number_list, ","));
			} catch {
				throw (E_INVALID_ARG, "Invalid array index");
			}

			integer prev_sought_ix = -1;
			for (ix : array_ixs) {
				if (ix <= prev_sought_ix || ix >= array_sz)
					throw (E_INVALID_ARG, "Invalid array index");
				prev_sought_ix = ix;
			}
		}

		json.builder replaced_jb = new json.builder();
		replaced_jb.begin_array();
		jb.begin_array();

		integer array_ixs_pos = 0;

		for (integer ix = 0; ix < array_sz; ++ix) {
			if (array_ixs_pos < v_size(array_ixs) && ix == array_ixs[array_ixs_pos]) {
				if (v_size(new_path) == 0) {
					jb.append(mask);
					replaced_jb.append(jn.get_array_element(ix));
				} else {
					json.node replaced_elem = _mask_value_recursive(jb, jn.get_array_element(ix), new_path, mask);
					replaced_jb.append(replaced_elem);
				}
				array_ixs_pos++;
			} else {
				jb.append(jn.get_array_element(ix));
			}
		}
		jb.end_array();
		replaced_jb.end_array();
		replaced_node = json.parse(replaced_jb.get());
	} else {
		jb.begin_object();
		vector(string) members = jn.get_object_members();
		for (m : members) {
			jb.begin_member(m);
			if (m == sought) {
				if (!null(replaced_node))
					throw (E_UNSPECIFIC, strcat("Multiple members: ", m));
				if (v_size(new_path) == 0) {
					jb.append(mask);
					replaced_node = jn.get_object_member(m);
				} else {
					replaced_node = _mask_value_recursive(jb, jn.get_object_member(m), new_path, mask);
				}
			} else {
				jb.append(jn.get_object_member(m));
			}
		}
		jb.end_object();
	}

	return replaced_node;
}

json.node json_masker.mask_values(json.node jn)
{
	resize(replaced_, 0);
	json.node new_node = json.parse(jn.print());
	for (integer i = 0; i < paths_.size(); ++i) {
		json.builder jb = new json.builder();
		json.node replaced_node = _mask_value_recursive(jb, new_node, paths_.get(i), replacements_[i]);
		push_back(replaced_, replaced_node);
		new_node = json.parse(jb.get());
	}

	return new_node;
}

vector(json.node) json_masker.get_replaced_values() = replaced_;

} // module http_api
