option(null: hard);

module http_api {

void url_percent_decode(out string s)
{
	// https://www.urlencoder.org/
	if (null(s))
		return;

	vector(string) pct_splitted = str_tokenize(s, "%");
	integer n = v_size(pct_splitted);
	for (integer i = 1; i < n; ++i) {
		string hex_and_more = pct_splitted[i];
		string hex = sub_string(hex_and_more, 0, 2);
		integer ch = str_hex_to_integer(hex);
		string decoded_ch = str_build([ch]);
		pct_splitted[i] = strcat(decoded_ch, sub_string_end(hex_and_more, 2));
	}

	if (n > 1)
		s = strcat(pct_splitted);
}

void url_percent_decode(vector(string) v)
{
	integer n = v_size(v);
	for (integer i = 0; i < n; ++i) {
		try {
			url_percent_decode(v[i]);
		} catch {
			throw (E_UNSPECIFIC, strcat(["Invalid URL percent encoding: ", v[i]]));
		}
	}
}

class URL {
public:
	URL(string url);

	logical like(string compare_url);
	integer size();
	string at(integer pos);

	string path();

	vector(string) params();
	integer param_count(string param);
	string param_value(string param, string option(nullable) dflt = null<string>); // Exception if count > 1
	string param_value_req(string param); // Exception if count != 1
	vector(string) param_multi_value(string param); // Never exception, works for missing and single params too

	vector(string) unused_params();

private:
	string orig_;

	vector(string) parts_;

	map_istr_v_str params_;
	set_str unused_params_;

	void __dbg_print(__dbg_label l);
};

URL.URL(string url)
{
	if (!str_startswith(url, "/")) {
		throw (E_UNSPECIFIC, "Does not start with slash");
	}

	url = sub_string_end(url, 1);

	orig_ = url;

	vector(string) base_params = str_tokenize(url, "?");
	parts_ = str_tokenize(base_params[0], "/");
	url_percent_decode(parts_);

	params_ = new map_istr_v_str();
	unused_params_ = set_str();

	if (v_size(base_params) == 1) {
		// no params
	} else if (v_size(base_params) == 2) {
		vector(string) query_params = str_tokenize(base_params[1], "&");
		integer nq = v_size(query_params);
		for (integer i = 0; i < nq; ++i) {
			string qp = query_params[i];
			string key;
			string val;
			integer equal_pos = str_search(qp, "=");
			if (equal_pos < 0) {
				key = qp;
			} else {
				key = sub_string_start(qp, equal_pos);
				val = sub_string_end(qp, equal_pos+1);
			}

			url_percent_decode(key);
			if (!null(val) && str_search(val, "+") >= 0)
				val = str_replace(val, "+", " ");
			url_percent_decode(val);

			unused_params_.add(key);
			vector(string) param_vals = params_.emplace(key);
			push_back(param_vals, val);
		}
	} else {
		throw (E_UNSPECIFIC, "Multiple question marks in URL");
	}
}

logical URL.like(string compare_url)
{
	if (!str_startswith(compare_url, "/")) {
		throw (E_UNSPECIFIC, "Compare URL does not start with slash");
	}

	compare_url = sub_string_end(compare_url, 1);

	vector(string) compare_parts = str_tokenize(compare_url, "/");

	if (v_size(compare_parts) != v_size(parts_)) {
		return false;
	}

	integer n = v_size(parts_);
	for (integer i = 0; i < n; ++i) {
		if (compare_parts[i] == "*") {
			continue;
		}

		string part = parts_[i];
		vector(string) vcmp = str_tokenize(compare_parts[i], "|");
		integer m = v_size(vcmp);
		integer j = 0;
		for ( ; j < m; ++j) {
			if (part == vcmp[j])
				break;
		}

		if (j == m)
			return false; // didn't find part in vcmp
	}

	return true;
}

integer URL.size() = v_size(parts_);

string URL.at(integer pos) = parts_[pos];

string URL.path() = strcat(["/", str_join(parts_, "/")]);

vector(string) URL.params() = params_.get_keys();

integer URL.param_count(string param)
{
	unused_params_.remove(param);
	return v_size(params_.find(param));
}

string URL.param_value(string param, string option(nullable) dflt)
{
	unused_params_.remove(param);
	vector(string) vals = params_.find(param);
	if (v_size(vals) == 0) {
		return dflt;
	} else if (v_size(vals) == 1) {
		return vals[0];
	} else {
		throw (E_UNSPECIFIC, strcat(["Param '", param, "' has multiple values"]));
	}
}

string URL.param_value_req(string param)
{
	if (null(params_.find(param)))
		throw (E_UNSPECIFIC, strcat(["Param '", param, "' expected in URL"]));
	return this.param_value(param);
}

vector(string) URL.param_multi_value(string param)
{
	unused_params_.remove(param);
	vector(string) vals = params_.find(param);
	if (null(vals)) {
		return vector(i:0; "");
	} else {
		return vals;
	}
}

vector(string) URL.unused_params() = unused_params_.get_content();

void URL.__dbg_print(__dbg_label l)
{
	vector(string) param_strs[0];
	vector(string) params = this.params();
	integer n = v_size(params);
	for (integer i = 0; i < n; ++i) {
		vector(string) vals = params_.find(params[i]);
		if (null(vals))
			resize(vals, 0);
		if (v_size(vals) == 1) {
			push_back(param_strs, strcat([params[i], "=", vals[0]]));
		} else {
			push_back(param_strs, strcat([params[i], "=[", str_join(vals, ","), "]"]));
		}
	}

	l.set_text(strcat(["URL { /", str_join(parts_, "/"), " { ", str_join(param_strs, ", "), " } }"]));
}

http.response unused_params_response(URL url)
{
	if (v_size(url.unused_params()) > 0) {
		return http_api.mk_error_response(400, strcat(["Unknown query parameters: ", str_join(url.unused_params(), ", ")]));
	} else {
		return null<http.response>;
	}
}

module __param {
module path {
string userid option(constant) = "USERID";
}
}


/*
 * context is just a base class. A server will have a subclass of context.
 *
 */

class context {
public:
	void set_handler_id(string id);
	string handler_id();

private:
	string handler_id_;
};

void context.set_handler_id(string id)
{
	handler_id_ = id;
}

string context.handler_id() = handler_id_;

/*
 * permissions_base is a base class for permissions
 * Create your own subclass, and use provided functions
 * in client_info to get/set permissions.
 */
class permissions_base {
public:
	void set_load_time(timestamp t);
	timestamp get_load_time();

	virtual logical has_any_rights() = 0;
	virtual string description() = 0;

private:
	timestamp load_time_;
};

void permissions_base.set_load_time(timestamp t)
{
	load_time_ = t;
}

timestamp permissions_base.get_load_time() = load_time_;

/*
 * client_info holds information about the connected client, such as user
 * identity, connection information and permissions.
 */

class client_info {
public:
	client_info(http.server_conn conn);
	client_info(http.server_conn_ssl conn);

	string addr();
	integer port();
	string name();

	logical set_user_id(string user_id, logical force = false, string option(nullable) token = null<string>);
	string user_id();

	logical has_token();
	string get_token();

	void set_permissions(permissions_base p);
	// Subclass of client_info is meant to use this in a get_permissions() that
	// returns subclass of permissions_base.
	permissions_base get_permissions_base();

private:
	string addr_;
	integer port_;
	string name_;

	string user_id_;
	string token_;
	permissions_base perm_;

	void _init(http.server_conn conn);

	void __dbg_print(__dbg_label l);
};

client_info.client_info(http.server_conn conn)
{
	_init(conn);
}

client_info.client_info(http.server_conn_ssl conn)
{
	_init(conn);
}

void client_info._init(http.server_conn conn)
{
	conn.get_peer(addr_, port_, name_);
	user_id_ = "";
	perm_ = null<permissions_base>;
}

void client_info.__dbg_print(__dbg_label l)
{
	l.set_text(strcat(["client_info { ", user_id_ == "" ? "<no user>" : user_id_, " ", str(port_), "@", addr_, " }"]));
}

string client_info.addr() = addr_;
integer client_info.port() = port_;
string client_info.name() = name_;

logical client_info.set_user_id(string user_id, logical force, string option(nullable) token)
{
	if (null(user_id_) || user_id != user_id_ || force) {
		// Permissions must be fetched again if connection changed user
		perm_ = null<permissions_base>;
		user_id_ = user_id;
		token_ = token;
		return true;
	} else {
		token = token_;
		return false;
	}
}

string client_info.user_id() = user_id_;

logical client_info.has_token() = !null(token_);

string client_info.get_token()
{
	if (null(token_))
		throw (E_UNSPECIFIC, "Token is not known");
	return token_;
}

void client_info.set_permissions(permissions_base p)
{
	perm_ = p;
}

permissions_base client_info.get_permissions_base() = perm_;


/*
 * url_path is a URL where some parts are fixed and some parts are to be filled
 * to access correct data. Example path /users/{user}/accounts/{accountName} is
 * will match all four part URLs where part one is "users" and part three is
 * "accounts". Parts two and four are called 'path params' and will often be
 * found in maps with the keys being "user" and "accountName" respectively.
 *
 * The url_path class also makes it easy to test API calls by just passing in
 * the path params.
 */

class url_path {
public:
	url_path(string path);

	string path();
	string pattern(); // "/items/{item}/properties/{property}  =>  "/items/*/properties/*"
	vector(string) url_parts();  // "/items/{item}/properties/{property}"  =>  ["items", "*", "properties", "*"];
	vector(string) path_params();
	string desc_str();
	string mk_url(map_str_str path_param_values);
	string get_path_param(URL url, string element);

private:
	string path_;
	string pattern_;
	vector(string) url_parts_;
	vector(string) path_params_;
	vector(integer) path_param_positions_;

	void __dbg_print(__dbg_label l);
};

void url_path.__dbg_print(__dbg_label l)
{
	l.set_text(strcat(["url_path { ", path_, " }"]));
}

url_path.url_path(string path)
: path_(path)
{
	if (!str_startswith(path, "/"))
		throw (E_UNSPECIFIC, "URL path must start with /");
	if (str_endswith(path, "/"))
		throw (E_UNSPECIFIC, "URL path may not end with /");

	set_str params_seen = set_str();
	resize(path_params_, 0);
	resize(path_param_positions_, 0);

	// modified by loop below
	url_parts_ = str_tokenize(path_, "/")[1:];

	integer n = v_size(url_parts_);
	for (integer i = 0; i < n; ++i) {
		string part = url_parts_[i];
		if (part == "")
			throw (E_UNSPECIFIC, "URL path may not contain \"//\"");
		if (part == "*")
			throw (E_UNSPECIFIC, "URL path may not contain \"/*/\"");
		if (str_startswith(part, "{") && str_endswith(part, "}")) {
			string path_param = sub_string_start(sub_string_end(part, 1), -1);
			if (path_param == "")
				throw (E_UNSPECIFIC, "Path param may not be empty string");
			if (params_seen.includes(path_param))
				throw (E_UNSPECIFIC, strcat("Duplicate path param: ", path_param));
			params_seen.add(path_param);
			push_back(path_params_, path_param);
			push_back(path_param_positions_, i);
			url_parts_[i] = "*";
		}
	}

	pattern_ = strcat("/", str_join(url_parts_, "/"));
}

string url_path.path() = path_;

string url_path.pattern() = pattern_;

vector(string) url_path.url_parts() = url_parts_;

vector(string) url_path.path_params() = path_params_;

string url_path.desc_str() = path_;

string url_path.mk_url(map_str_str path_param_values)
{
	if (v_size(path_params_) == 0)
		return path_;

	vector(string) url_parts = str_tokenize(path_, "/");
	integer n = v_size(url_parts);
	integer path_param_ix = 0;
	for (integer i = 0; i < n; ++i) {
		if (path_param_positions_[path_param_ix] == i) {
			string path_param_key = path_params_[path_param_ix];
			string path_param_val = path_param_values.find(path_param_key);
			if (null(path_param_val)) {
				throw (E_UNSPECIFIC, strcat(["url_path '", this.desc_str(), "' requires query element '", path_param_key, "'"]));
			}
			url_parts[i] = path_param_val;
		}
	}

	return str_join(url_parts, "/");
}

string url_path.get_path_param(URL url, string element)
{
	integer n = v_size(path_params_);
	for (integer i = 0; i < n; ++i) {
		if (path_params_[i] == element) {
			return url.at(path_param_positions_[i]);
		}
	}
	throw (E_UNSPECIFIC, strcat(["No path param '", element, "' in path ", path_]));
}


class request_values {
public:
	request_values();

	string get_path_param(string name);
	string get_query_param(string name);
	vector(string) get_query_multi_param(string name);
	json.node get_body();

	void set_path_param(string name, string value);
	void set_query_param(string name, string option(nullable) value);
	void set_query_multi_param(string name, vector(string) value);
	void set_body(json.node option(nullable) body);

	string params_json();

private:
	map_str_str path_params_;
	map_str_str query_params_;
	map_str_v_str query_multi_params_;
	json.node body_;
};

request_values.request_values()
: path_params_(map_str_str()), query_params_(map_str_str()), query_multi_params_(new map_str_v_str())
{
}

string request_values.get_path_param(string name){
	string value = path_params_.find(name);
	if (null(value))
		throw (E_UNSPECIFIC, strcat(["Internal: No such path param '", name, "'"]));
	return value;
}

string request_values.get_query_param(string name) = query_params_.find(name);

vector(string) request_values.get_query_multi_param(string name) = query_multi_params_.find(name);

json.node request_values.get_body() = body_;

void request_values.set_path_param(string name, string value) { path_params_.add(name, value); }

void request_values.set_query_param(string name, string option(nullable) value)
{
	if (null(value))
		query_params_.remove(name);
	else
		query_params_.add(name, value);
}

void request_values.set_query_multi_param(string name, vector(string) value) { query_multi_params_.add(name, value); }

void request_values.set_body(json.node option(nullable) body) { body_ = body; }

string request_values.params_json()
{
	json.builder jb = new json.builder();
	jb.begin_object();
	{
		jb.begin_member("path_params");
		jb.begin_object();
		vector(string) name, value;
		path_params_.get_content(name, value);
		sort(value, name);
		sort(name);
		for (integer i = 0; i < v_size(name); ++i) {
			jb.begin_member(name[i]);
			jb.append(value[i]);
		}
		jb.end_object();
	}
	{
		jb.begin_member("query_params");
		jb.begin_object();
		vector(string) name, value;
		query_params_.get_content(name, value);
		sort(value, name);
		sort(name);
		for (integer i = 0; i < v_size(name); ++i) {
			jb.begin_member(name[i]);
			jb.append(value[i]);
		}
		jb.end_object();
	}
	{
		jb.begin_member("query_multi_params");
		jb.begin_object();
		vector(string) name = query_multi_params_.get_keys();
		for (integer i = 0; i < v_size(name); ++i) {
			vector(string) values = query_multi_params_.find(name[i]);
			jb.begin_member(name[i]);
			jb.append(values);
		}
		jb.end_object();
	}
	jb.end_object();
	return jb.get();
}

// We wrap http.response in our own class, to future-proof stuff. In the future we
// may want to allow returning more that just a http.response from handler functions.
class response {
public:
	response(http.response resp, string option(nullable) handler_id = null);

	http.response get();

	string handler_id();
	void set_handler_id(string handler_id);

private:
	http.response resp_;
	string handler_id_;
};

response.response(http.response resp, string option(nullable) handler_id)
	: resp_(resp), handler_id_(handler_id)
{
}

http.response response.get() = resp_;

string response.handler_id() = handler_id_;

void response.set_handler_id(string handler_id)
{
	handler_id_ = handler_id;
}


// Lib users should create classes that inherit from response_body, and pass
// instances of these classes to mk_response / create_response functions to get
// the body json created from the classes.
class response_body {
};


// Wrappers for mk_response functions that return our response class.
response create_response(integer status_code) =
	new response(mk_response(status_code));
response create_deflated_response(integer status_code, json.node jn, integer indent = -1) =
	new response(mk_deflated_response(status_code, jn, indent));
response create_deflated_response(integer status_code, json.builder jb) =
	new response(mk_deflated_response(status_code, jb));
response create_response(integer status_code, json.node jn, integer indent = -1) =
	new response(mk_response(status_code, jn, indent));
response create_response(integer status_code, json.builder jb) =
	new response(mk_response(status_code, jb));
response create_response(integer status_code, response_body rb, integer indent = -1) =
	new response(__mk_json_response(status_code, indent < 0 ? json.print(rb) : json.print(rb, indent), false));
response create_response_raw(integer status_code, string option(nullable) status_msg, string option(nullable) content_type, string option(nullable) body) =
	new response(mk_response_raw(status_code, status_msg, content_type, body));
response create_error_response(integer status_code, string msg) =
	new response(mk_error_response(status_code, msg));
response create_error_response(integer status_code, vector(string) errors, integer indent = -1)
{
	json.builder jb = new json.builder();
	jb.begin_object();
	jb.begin_member("errors");
	jb.append(errors);
	jb.end_object();
	return create_response(status_code, json.parse(jb.get()), indent);
}
response create_method_not_allowed_response(vector(string) allowed_methods) =
	new response(mk_method_not_allowed_response(allowed_methods));

string HANDLER_ID_LOGIN option(constant) = "__login";
string HANDLER_ID_FAILURE option(constant) = "__failure";
string HANDLER_ID_OPEN_API option(constant) = "__open_api";
string HANDLER_ID_POST_SAVE_REQ_RESP option(constant) = "__post_save_req_resp";
string HANDLER_ID_DELETE_SAVE_REQ_RESP option(constant) = "__delete_save_req_resp";

// New constant here? Add to function below!

set_str handler_id_start_set()
{
	set_str set = set_str();
	set.add(
		[HANDLER_ID_LOGIN
		 , HANDLER_ID_FAILURE
		 , HANDLER_ID_OPEN_API
	]);
	return set;
}

typedef
response function(
	context ctxt,
	client_info client,
	request_values req_vals)
	handle_func_t;

typedef
string function(
	context ctxt,
	client_info client,
	request_values req_vals)
	permission_func_t;


class request_handler {
public:
	request_handler(
		http.method method,
		string url,
		handle_func_t handle_func,
		permission_func_t permission_func,
		string summary);

	void set_id(string id);
	string id();
	void set_description(string desc);

	void check_complete();

	void set_path_param_desc(string param, open_api.schema sch, string desc);

	void add_query_param(string name, string option(nullable) dflt);
	void add_query_multi_param(string name);

	void set_body(logical required, open_api.schema option(nullable) schema);

	void add_response(open_api.response_desc);

	// Core stuff
	http.method method();
	url_path path();
	handle_func_t func();
	permission_func_t permission_func();
	string summary();
	string desc();

	// Path params
	string path_param_desc(string param);
	open_api.schema path_param_schema(string param);

	// Query params
	vector(string) query_params();
	logical query_param_is_multi(string param_name);
	open_api.schema query_param_schema(string param_name);
	string query_param_default(string param_name);

	// Request body
	logical takes_body();
	logical requires_body();
	open_api.schema request_schema();

	// Responses
	vector(open_api.response_desc) responses();

	// Is authentication disabled?
	logical requires_authentication();
	void set_no_authentication_required();

private:
	// Core stuff
	string id_;
	http.method method_;
	url_path path_;
	handle_func_t func_;
	permission_func_t permission_func_;
	string summary_;
	string desc_;
	map_str_str path_param_descs_;
	map_str_obj<open_api.schema> path_param_schema_;

	// Query params
	set_str query_params_;
	map_str_obj<open_api.schema> query_param_schema_;
	map_str_str query_param_default_;
	set_str query_multi_params_;

	// Request body
	logical takes_body_;
	logical requires_body_;
	open_api.schema request_schema_;

	// Responses
	vector(open_api.response_desc) responses_;

	logical authentication_required_;

	void __dbg_print(__dbg_label l);
};

void request_handler.__dbg_print(__dbg_label l)
{
	if (null(id_))
		l.set_text(strcat(["request_handler { ", enum_id(method_), " ", path_.path(), " }"]));
	else
		l.set_text(strcat(["request_handler { \"", id_, "\" ", enum_id(method_), " ", path_.path(), " }"]));
}


request_handler.request_handler(
	http.method method,
	string url,
	handle_func_t handle_func,
	permission_func_t permission_func,
	string summary)
:   method_(method), path_(new url_path(url))
	, func_(handle_func), permission_func_(permission_func), summary_(summary)
	, path_param_descs_(map_str_str()), path_param_schema_(new map_str_obj<open_api.schema>())
	, query_params_(set_str()), query_param_schema_(new map_str_obj<open_api.schema>())
	, query_param_default_(map_str_str()), query_multi_params_(set_str())
	, takes_body_(false), requires_body_(false), request_schema_(null<open_api.schema>)
	, authentication_required_(true)
{
}

void request_handler.set_id(string id)
{
	regex re = new regex("[a-zA-Z_][a-zA-Z0-9_]*");
	if (!re.match(id))
		throw (E_UNSPECIFIC, "Handler id must be a valid identifier");
	id_ = id;
}

string request_handler.id() = id_;

void request_handler.set_description(string desc)
{
	desc_ = desc;
}

void request_handler.check_complete()
{
	string err_prefix = strcat([enum_id(method_), " ", path_.path(), ": "]);
	if (v_size(responses_) == 0)
		throw (E_UNSPECIFIC, strcat(err_prefix, "No responses"));

	vector(string) path_params = path_.path_params();
	integer n = v_size(path_params);
	for (integer i = 0; i < n; ++i) {
		if (null(path_param_descs_.find(path_params[i]))) {
			throw (E_UNSPECIFIC, strcat([err_prefix, "Need desc for path param '", path_params[i], "'"]));
		}
	}
}

void request_handler.set_path_param_desc(string param, open_api.schema sch, string desc)
{
	vector(string) path_params = path_.path_params();
	integer n = v_size(path_params);
	for (integer i = 0; i < n; ++i) {
		if (path_params[i] == param) {
			path_param_descs_.add(param, desc);
			path_param_schema_.add(param, sch);
			return;
		}
	}
	throw (E_UNSPECIFIC, strcat(["No such path param: ", param]));
}

void request_handler.add_query_param(string name, string option(nullable) dflt)
{
	if (query_params_.includes(name))
		throw (E_UNSPECIFIC, strcat("Duplicate query param: ", name));
	query_params_.add(name);
	open_api.schema_builder schb = new open_api.schema_builder().set_type_string(dflt, dflt);
	query_param_schema_.add(name, schb.get_schema());
	if (!null(dflt))
		query_param_default_.add(name, dflt);
}

void request_handler.add_query_multi_param(string name)
{
	if (query_params_.includes(name))
		throw (E_UNSPECIFIC, strcat("Duplicate query param: ", name));
	query_params_.add(name);
	open_api.schema_builder schb = new open_api.schema_builder().set_type_string(null<string>, null<string>);
	query_multi_params_.add(name);
	query_param_schema_.add(name, schb.get_schema());
}

void request_handler.set_body(logical required, open_api.schema option(nullable) schema)
{
	switch (method_) {
	case http.CONNECT:
	case http.DELETE:
	case http.GET:
	case http.HEAD:
		throw (E_UNSPECIFIC, "Method does not allow body");
	}

	takes_body_ = true;
	requires_body_ = required;
	request_schema_ = schema;
}

void request_handler.add_response(open_api.response_desc resp)
{
	integer n = v_size(responses_);
	for (integer i = 0; i < n; ++i) {
		if (responses_[i].status_code == resp.status_code)
			throw (E_UNSPECIFIC, strcat("Duplicate response for status code: ", str(resp.status_code)));
	}
	push_back(responses_, resp);
}

http.method request_handler.method() = method_;
url_path request_handler.path() = path_;
handle_func_t request_handler.func() = func_;
permission_func_t request_handler.permission_func() = permission_func_;
string request_handler.summary() = summary_;
string request_handler.desc() = desc_;
string request_handler.path_param_desc(string path_param) = path_param_descs_.find(path_param);
open_api.schema request_handler.path_param_schema(string path_param) = path_param_schema_.find(path_param);

vector(string) request_handler.query_params() = query_params_.get_content();

logical request_handler.query_param_is_multi(string param_name)
{
	if (!query_params_.includes(param_name))
		throw (E_UNSPECIFIC, strcat("Unknown query param: ", param_name));
	return query_multi_params_.includes(param_name);
}

open_api.schema request_handler.query_param_schema(string param_name)
{
	if (!query_params_.includes(param_name))
		throw (E_UNSPECIFIC, strcat("Unknown query param: ", param_name));
	return query_param_schema_.find(param_name);
}

string request_handler.query_param_default(string param_name) = query_param_default_.find(param_name);

logical request_handler.takes_body() = takes_body_;
logical request_handler.requires_body() = requires_body_;
open_api.schema request_handler.request_schema() = request_schema_;

vector(open_api.response_desc) request_handler.responses() = responses_;

logical request_handler.requires_authentication() = authentication_required_;
void request_handler.set_no_authentication_required()
{
	authentication_required_ = false;
}

void add_path_params(
	request_values req_vals,
	request_handler handler,
	URL url)
{
	url_path path = handler.path();
	vector(string) path_params = path.path_params();
	integer n = v_size(path_params);
	for (integer i = 0; i < n; ++i) {
		string path_param = path_params[i];
		string value = path.get_path_param(url, path_param);
		req_vals.set_path_param(path_param, value);
	}
}

void add_query_params(
	request_values req_vals,
	request_handler handler,
	URL url)
{
	vector(string) accepted_params = handler.query_params();
	integer n = v_size(accepted_params);
	for (integer i = 0; i < n; ++i) {
		string param = accepted_params[i];
		if (handler.query_param_is_multi(param)) {
			req_vals.set_query_multi_param(param, url.param_multi_value(param));
		} else {
			string dflt = handler.query_param_default(param);
			req_vals.set_query_param(param, url.param_value(param, dflt));
		}
	}

	vector(string) unused_params = url.unused_params();
	if (v_size(unused_params) > 5)
		throw (E_UNSPECIFIC, "Several unexpected query params in URL");
	if (v_size(unused_params) > 0)
		throw (E_UNSPECIFIC, strcat(["Unexpected query params: ", str_join(unused_params, ", ")]));
}

void add_body(
	request_values req_vals,
	request_handler handler,
	blob option(nullable) body)
{
	if (!handler.takes_body()) {
		if (!null(body))
			throw (E_UNSPECIFIC, "Unexpected body");
	} else {
		if (null(body) && handler.requires_body()) {
			throw (E_UNSPECIFIC, "Body required");
		}

		try {
			json.node json_body = null(body) ? null<json.node> : json.parse(body.get_string());
			req_vals.set_body(json_body);
		} catch {
			throw (E_UNSPECIFIC, strcat("Malformed body: ", err.message()));
		}
	}
}

} // module http_api

module open_api {

void append_params(json.builder jb, http_api.request_handler h)
{
	vector(string) path_params = h.path().path_params();
	vector(string) query_params = h.query_params();
	if (v_size(path_params) > 0 || v_size(query_params) > 0) {
		jb.begin_member("parameters");
		jb.begin_array();
		integer npp = v_size(path_params);
		for (integer pp = 0; pp < npp; ++pp) {
			string path_param = path_params[pp];
			jb.begin_object();
			jb.begin_member("name");
			jb.append(path_param);
			string desc = h.path_param_desc(path_param);
			schema sch = h.path_param_schema(path_param);
			if (!null(desc)) {
				jb.begin_member("description");
				jb.append(desc);
			}
			if (!null(sch)) {
				jb.begin_member("schema");
				jb.append_node(sch.mk_json_node());
			}
			jb.begin_member("required");
			jb.append(true);
			jb.begin_member("in");
			jb.append("path");
			jb.end_object();
		}
		integer nqp = v_size(query_params);
		for (integer qp = 0; qp < nqp; ++qp) {
			string query_param = query_params[qp];
			schema sch = h.query_param_schema(query_param);
			jb.begin_object();
			jb.begin_member("name");
			jb.append(query_param);
			jb.begin_member("in");
			jb.append("query");
			if (!null(sch)) {
				jb.begin_member("schema");
				jb.append_node(sch.mk_json_node());
			}
			jb.end_object();
		}
		jb.end_array();
	}
}

void append_responses(json.builder jb, http_api.request_handler h)
{
	vector(response_desc) responses = h.responses();
	jb.begin_member("responses");
	jb.begin_object();
	responses.mk_json(jb);
	jb.end_object(); // responses
}

void append_method(json.builder jb, http_api.request_handler h)
{
	string method = str_to_lower(enum_id(h.method()));
	jb.begin_member(method);
	jb.begin_object();
	if (!null(h.desc())) {
		jb.begin_member("description");
		jb.append(h.desc());
	}

	append_params(jb, h);

	append_responses(jb, h);

	jb.end_object(); // method
}

} // module open_api

module http_api {

module dsattr {
string http_user_pwd option(constant) = "http_user_pwd";
string http_sessions option(constant) = "http_sessions";
}

class user_handling_data {
public:
	logical log_events;
	logging.Context logging_context;
	qlc.hce.dataset user_dataset;
};

class login_data {
public:
	string login_path;
	integer session_max_length_minutes;
	integer session_logout_after_minutes_idle;
	permissions_base function(string user_id, context ctxt) permission_load_func;
	integer permission_lifetime_minutes;
	void function(client_info client, context ctxt) new_user_func;
};


class save_req_resp_data {
public:
	save_req_resp_data(string root_directory);

	string root_directory;
	logical disable;
};

save_req_resp_data.save_req_resp_data(string root_directory)
	: root_directory(root_directory), disable(false)
{
}

class save_req_resp_json_obj {
public:
	integer timeout_seconds;
};


/* For handlers defined by the http_api library itself, this subclass to
   context is used. The external user of the http_api library will define its
   own subclass to context.
*/
class internal_context : public context {
public:
	user_handling_data user;
	login_data login;
	save_req_resp_data save_req_resp;

	void advance_counter();
	integer read_counter();

private:
	integer counter_;
};

void internal_context.advance_counter()
{
	counter_++;
}

integer internal_context.read_counter() = counter_;

response __login_user(
	internal_context ctxt,
	client_info client,
	blob option(nullable) body,
	out logical save_req_resp)
{
	save_req_resp = false;
	json.node jn;
	try {
		if (null(body))
			throw (E_UNSPECIFIC, "Expected body");

		jn = json.parse(body.get_string());
	} catch {
		return new response(mk_error_response(400, err.message()));
	}

	string uid, pwd;
	try {
		uid = jn.get_object_member("username").get_string();
		pwd = jn.get_object_member("password").get_string();
	} catch {
		return new response(mk_error_response(400, err.message()));
	}

	qlc.hce.writer user_wr;

	try {
		user_wr = ctxt.user.user_dataset.begin_write();

		map_str_str uid2hash;
		user_wr.get_global_obj(dsattr.http_user_pwd, uid2hash);
		string hash_pwd = uid2hash.find(user_identity.normalize(uid));
		if (null(hash_pwd))
			return new response(http_api.mk_error_response(401, "Invalid username or password"));

		blob db_hash = base64_decode(hash_pwd);
		if (!user_identity.verify_password(pwd, db_hash))
			return new response(http_api.mk_error_response(401, "Invalid username or password"));
	} catch {
		return new response(mk_error_response(500, "Error checking password"));
	}

	string token;
	try {
		user_identity.user_set user_sessions;
		user_wr.get_global_obj(dsattr.http_sessions, user_sessions);
		token = user_sessions.login_user(uid, ctxt.login.session_max_length_minutes, ctxt.login.session_logout_after_minutes_idle);
		user_wr.set_global_attrib(dsattr.http_sessions, user_sessions);
		client.set_user_id(uid, true);

		save_req_resp = user_sessions.should_save_req_resp(uid);

	} catch {
		return new response(mk_error_response(500, "Error creating session"));
	}

	user_wr.commit();

	json.builder jb = new json.builder();
	jb.begin_object();
	jb.begin_member("token");
	jb.append(token);
	jb.end_object();

	return new response(http_api.mk_response(201, jb));
}

void write_req_to_file(http.request req, string req_path)
{
	out_stream fh = out_stream_file(req_path);
	fh.write(string(req.method()));
	fh.write("\r\n");
	fh.write(req.url());
	fh.write("\r\n\r\nHEADERS\r\n");
	for (header : req.headers())
		fh.write([header, ": ", req.header(header), "\r\n"]);
	fh.write("\r\n");
	if (!null(req.body())) {
		fh.write("BODY\r\n");
		fh.write(req.body().get_string());
	}
	fh.close();
}

void write_resp_to_file(http.response resp, string resp_path)
{
	out_stream fh = out_stream_file(resp_path);
	fh.write([string(resp.status_code()), " ", resp.status_msg(), "\r\n\r\n"]);
	fh.write("HEADERS\r\n");
	for (header : resp.headers())
		fh.write([header, ": ", resp.header(header), "\r\n"]);
	fh.write("\r\n");
	if (!null(resp.body())) {
		fh.write("BODY\r\n");
		fh.write(resp.body().get_string());
	}
	fh.close();
}

void __save_req_resp_log(
	string root_directory,
	client_info client,
	integer req_count,
	string suffix,
	http.request option(nullable) req,
	http.response option(nullable) resp)
{
	string reqresp_dir = path_join(root_directory, "req_resp");
	if (!path_exists(reqresp_dir))
		create_directory(reqresp_dir);

	string target_dir = path_join(reqresp_dir, client.user_id());
	if (!path_exists(target_dir))
		create_directory(target_dir);

	string ts = str(now());
	ts = str_replace(ts, "-", "");
	ts = str_replace(ts, " ", "_");
	ts = str_replace(ts, ":", "");
	ts = str_replace(ts, ".", "_");

	string file_start = strcat([ts, "_p", str(client.port()), "_n", str(req_count)]);

	if (!null(req)) {
		string req_path = path_join(target_dir, strcat([file_start, "_req_", suffix]));
		write_req_to_file(req, req_path);
	}

	if (!null(resp)) {
		string resp_path = path_join(target_dir, strcat([file_start, "_resp_", suffix]));
		write_resp_to_file(resp, resp_path);
	}
}

void save_req_log(
	http.request req,
	internal_context ctxt,
	client_info client)
{
	if (ctxt.save_req_resp.disable)
		return;

	integer qpos = str_search(req.url(), "?");
	integer hpos = str_search(req.url(), "#");
	integer pos;
	if (qpos < 0 && hpos < 0)
		pos = qpos; // Neither found
	else if (qpos < 0)
		pos = hpos;
	else if (hpos < 0)
		pos = qpos;
	else
		pos = min(qpos, hpos);

	string url = pos < 0 ? req.url() : sub_string_start(req.url(), pos);
	string suffix = strcat([string(req.method()), "_", str_replace(url, "/", "!")]);

	try {
		__save_req_resp_log(
			ctxt.save_req_resp.root_directory,
			client,
			ctxt.read_counter(),
			suffix,
			req,
			null);
	} catch {
		logging.warning(ctxt.user.logging_context, ["Error saving request to file: ", err.message(), "."]);
		logging.warning(ctxt.user.logging_context, ["Disabling saving to file for this connection."]);
		ctxt.save_req_resp.disable = true;
	}
}

void save_resp_log(
	http_api.response resp,
	internal_context ctxt,
	client_info client)
{
	if (ctxt.save_req_resp.disable)
		return;

	try {
		__save_req_resp_log(
			ctxt.save_req_resp.root_directory,
			client,
			ctxt.read_counter(),
			null(resp.handler_id()) ? "X" : resp.handler_id(),
			null,
			resp.get());
	} catch {
		logging.warning(ctxt.user.logging_context, ["Error saving response to file: ", err.message(), "."]);
		logging.warning(ctxt.user.logging_context, ["Disabling saving to file for this connection."]);
		ctxt.save_req_resp.disable = true;
	}
}

response __handle_post_save_req_resp(context wctxt, client_info client, request_values req_vals)
{
	internal_context ctxt = dynamic_cast<internal_context>(wctxt);

	save_req_resp_json_obj rb;
	try {
		json.get_object(req_vals.get_body(), rb);
	} catch {
		return create_error_response(400, "Malformed body");
	}

	if (rb.timeout_seconds <= 0)
		return create_error_response(400, "timeout_seconds must be positive");

	qlc.hce.writer wr = ctxt.user.user_dataset.begin_write();
	user_identity.user_set user_sessions;
	wr.get_global_obj(dsattr.http_sessions, user_sessions);
	user_sessions.enable_save_req_resp(req_vals.get_path_param(__param.path.userid), rb.timeout_seconds);
	wr.set_global_attrib(dsattr.http_sessions, user_sessions);
	wr.commit();

	return create_response(200);
}

response __handle_delete_save_req_resp(context wctxt, client_info client, request_values req_vals)
{
	internal_context ctxt = dynamic_cast<internal_context>(wctxt);

	qlc.hce.writer wr = ctxt.user.user_dataset.begin_write();
	user_identity.user_set user_sessions;
	wr.get_global_obj(dsattr.http_sessions, user_sessions);
	if (user_sessions.disable_save_req_resp(req_vals.get_path_param(__param.path.userid))) {
		wr.set_global_attrib(dsattr.http_sessions, user_sessions);
		wr.commit();
		return create_response(200);
	} else {
		return create_error_response(404, "User has not been asked to save req/resp");
	}
}

typedef map_str_obj<map_str_obj<request_handler>> handler_map_t;

class api {
public:
	api(string title, string version);

	void add_schema(string schema_name, open_api.schema schema);

	void add_handler(request_handler h, string option(nullable) id = null<string>);

	response process(http.request req, context ctxt, client_info client);

	string create_open_api();

	// Login and session handling
	void enable_login(
		string login_path,
		logical log_events,
		string option(nullable) logging_context,
		qlc.hce.dataset user_dataset,
		integer session_max_length_minutes,
		integer session_logout_after_minutes_idle,
		permissions_base function(string user_id, context ctxt) permission_load_func,
		integer permission_lifetime_minutes,
		void function(client_info client, context ctxt) new_user_func);

	void enable_save_req_resp(
		string url_path,
		string root_directory,
		permission_func_t permission_func);

	void add_users(map_str_str uid2pwd_hash);
	void add_users(odbc.connection conn);
	void remove_all_users();
	void logout_users(vector(string) user_ids);
	void logout_all();

	// Set some behaviour

	// Uncaught exceptions in handled functions result in 500 response, normally
	// without any further information. Call this function to make the response include
	// print() of the exception, which includes message, file, and line.
	// Makes sense for debug, but maybe not in released product.
	void set_activate_debug_response_on_exception(logical activate);

private:
	string title_;
	string version_;
	handler_map_t pattern2method2handler_;
	map_str_str pattern2path_;
	set_str handler_ids_;

	map_str_obj<open_api.schema> schemas_;

	internal_context internal_ctxt_;

	logical activate_debug_response_on_exception_;

	void _check_ambiguity(url_path path);
	map_str_obj<request_handler> _find_path_handlers(URL url);

	void _load_permissions(client_info client, context ctxt);
	response _resp_if_auth_fail(
		http.request req,
		context ctxt,
		client_info client,
		out logical save_req_resp);
	response _resp_if_login_req(
		http.method method, URL url,
		blob option(nullable) body,
		context ctxt,
		client_info client,
		out logical save_req_resp);
	response _resp_if_open_api_req(http.method method, URL url);
	void _open_api_append_schemas(json.builder jb);
	void _open_api_append_paths(json.builder jb);
	string _create_open_api();
	response _process(http.request req, context ctxt, client_info client);

	void __dbg_print(__dbg_label l);
};

void api.__dbg_print(__dbg_label l)
{
	counter_str method_counts = new counter_str();
	vector(string) patterns = pattern2method2handler_.get_keys();
	integer n = v_size(patterns);
	for (integer i = 0; i < n; ++i) {
		vector(string) methods = pattern2method2handler_.find(patterns[i]).get_keys();
		method_counts.increase(methods);
	}

	vector(string) methods;
	vector(number) counts;
	method_counts.get_content(methods, counts);
	sort(counts, methods);
	sort(methods);
	vector(string) method_count_strs = v_strcat().cat(methods).cat(": ").cat(str(counts)).get();

	l.set_text(strcat(["api { ", str(pattern2path_.size()), " paths, ", str_join(method_count_strs, "; "), " }"]));
}

api.api(string title, string version)
: title_(title), version_(version)
	, pattern2method2handler_(new handler_map_t())
	, pattern2path_(map_str_str())
	, handler_ids_(set_str())
	, schemas_(new map_str_obj<open_api.schema>())
	, internal_ctxt_(new internal_context())
	, activate_debug_response_on_exception_(false)
{
}

void api.add_schema(string schema_name, open_api.schema schema)
{
	schemas_.add(schema_name, schema);
}

void api.add_handler(request_handler h, string option(nullable) id)
{
	if (null(id)) {
		if (!null(handler_ids_) && handler_ids_.size() > 0)
			throw (E_UNSPECIFIC, "All handlers must have id if any handler has it");
		handler_ids_ = null<set_str>; // This api doesn't use handler id's
	} else {
		if (null(handler_ids_))
			throw (E_UNSPECIFIC, "All handlers must have id if any handler has it");
		if (handler_ids_.size() == 0)
			handler_ids_ = handler_id_start_set();
		h.set_id(id);
	}

	h.check_complete();

	_check_ambiguity(h.path());

	if (!null(handler_ids_)) {
		if (handler_ids_.includes(h.id()))
			throw (E_UNSPECIFIC, strcat("Duplicate or invalid handler id: ", h.id()));
		handler_ids_.add(h.id());
	}

	string url_pattern = h.path().pattern();
	string method_str = enum_id(h.method());
	map_str_obj<request_handler> method2handler = pattern2method2handler_.find(url_pattern);

	if (null(method2handler)) {
		method2handler = new map_str_obj<request_handler>();
		pattern2method2handler_.add(url_pattern, method2handler);
		pattern2path_.add(url_pattern, h.path().path());
	} else {
		// Check that the path is not changed, i.e. we do not accept handlers
		// for url_path "/customer/{NAME}", and for url_path
		// "/customer/{ID}". The handlers must have the same path parameters.
		string existing_path = pattern2path_.find(url_pattern);
		if (h.path().path() != existing_path)
			throw (E_UNSPECIFIC, strcat(["URL path \"", h.path().path(), "\" collides with \"", existing_path, "\""]));

		// Check that handler is unique.
		// This will catch URL's that are similar like this:
		// "/items/{id}" and "/items/{source}",
		// but it will not catch URL's that are ambigous like this:
		// "/items/{id}" and "/{type}/list", since the placeholders
		// are at different positions in the URL.
		request_handler pre_existing = method2handler.find(method_str);
		if (!null(pre_existing))
			throw (E_UNSPECIFIC, strcat(["Already have handler for ", method_str, " ", url_pattern]));
	}

	method2handler.add(method_str, h);
}

void api.enable_login(
	string login_path,
	logical log_events,
	string option(nullable) logging_context,
	qlc.hce.dataset user_dataset,
	integer session_max_length_minutes,
	integer session_logout_after_minutes_idle,
	permissions_base function(string user_id, context ctxt) permission_load_func,
	integer permission_lifetime_minutes,
	void function(client_info client, context ctxt) new_user_func)
{
	internal_ctxt_.user = new user_handling_data();
	internal_ctxt_.user.log_events = log_events;
	internal_ctxt_.user.logging_context = null(logging_context) ? null<logging.Context> : logging.context(logging_context);
	internal_ctxt_.user.user_dataset = user_dataset;

	internal_ctxt_.login = new login_data();
	internal_ctxt_.login.login_path = login_path;
	internal_ctxt_.login.session_max_length_minutes = session_max_length_minutes;
	internal_ctxt_.login.session_logout_after_minutes_idle = session_logout_after_minutes_idle;
	internal_ctxt_.login.permission_load_func = permission_load_func;
	internal_ctxt_.login.permission_lifetime_minutes = permission_lifetime_minutes;
	internal_ctxt_.login.new_user_func = new_user_func;

	{
		qlc.hce.writer wr = internal_ctxt_.user.user_dataset.begin_write();
		if (!wr.has_global_attrib(dsattr.http_sessions)) {
			wr.add_global_attrib_obj(dsattr.http_sessions, null<user_identity.user_set>);
			wr.set_global_attrib(dsattr.http_sessions, new user_identity.user_set());
		}
		if (!wr.has_global_attrib(dsattr.http_user_pwd)) {
			wr.add_global_attrib_obj(dsattr.http_user_pwd, null<map_str_str>);
			wr.set_global_attrib(dsattr.http_user_pwd, map_str_str());
		}
		wr.commit();
	}
}

void api.enable_save_req_resp(string base_url, string root_directory, permission_func_t permission_func)
{
	if (null(internal_ctxt_.login) || null(internal_ctxt_.user))
		throw (E_UNSPECIFIC, "Login must be enabled before save req/resp is");

	internal_ctxt_.save_req_resp = new save_req_resp_data(root_directory);

	string url = strcat([base_url, "/{", __param.path.userid, "}"]);

	{
		request_handler h = new request_handler(
			http.POST, strcat(base_url, "/{USERID}"),
			&__handle_post_save_req_resp,
			permission_func,
			"Enable saving request and response to file for a while");
		h.set_path_param_desc(__param.path.userid, new open_api.schema_builder().set_type_string().get_schema(), "User Id");
		h.set_body(true, new open_api.schema_builder().start_type_object().add_property("timeout_seconds", true).set_type_integer().end_type_object().get_schema());
		h.add_response(new open_api.response_desc(200, "Saving enabled"));
		this.add_handler(h, HANDLER_ID_POST_SAVE_REQ_RESP);
	}

	{
		request_handler h = new request_handler(
			http.DELETE, url,
			&__handle_delete_save_req_resp,
			permission_func,
			"Stop saving request and response to file");
		h.set_path_param_desc(__param.path.userid, new open_api.schema_builder().set_type_string().get_schema(), "User Id");
		h.add_response(new open_api.response_desc(200, "Saving disabled for user"));
		h.add_response(new open_api.response_desc(404, "User had no saving requested"));
		this.add_handler(h, HANDLER_ID_DELETE_SAVE_REQ_RESP);
	}

}

void api.add_users(map_str_str uid2pwd_hash)
{
	if (null(internal_ctxt_.login))
		throw (E_UNSPECIFIC, "Login is not enabled");

	qlc.hce.writer wr = internal_ctxt_.user.user_dataset.begin_write();
	map_str_str dataset_uid2pwd_hash;
	wr.get_global_obj(dsattr.http_user_pwd, dataset_uid2pwd_hash);
	vector(string) new_user_ids;
	vector(string) new_pwd_hashes;
	uid2pwd_hash.get_content(new_user_ids, new_pwd_hashes);
	new_user_ids = user_identity.normalize(new_user_ids);
	dataset_uid2pwd_hash.add(new_user_ids, new_pwd_hashes);
	wr.set_global_attrib(dsattr.http_user_pwd, dataset_uid2pwd_hash);
	wr.commit();
}

void api.add_users(odbc.connection conn)
{
	map_str_str new_users = user_identity.fetch_user_data(conn);
	this.add_users(new_users);
}

void api.remove_all_users()
{
	if (null(internal_ctxt_.login))
		throw (E_UNSPECIFIC, "Login is not enabled");

	qlc.hce.writer wr = internal_ctxt_.user.user_dataset.begin_write();
	map_str_str empty = map_str_str();
	wr.set_global_attrib(dsattr.http_user_pwd, empty);
	wr.commit();
}

void api.logout_users(vector(string) user_ids)
{
	if (null(internal_ctxt_.login))
		throw (E_UNSPECIFIC, "Login is not enabled");

	qlc.hce.writer wr = internal_ctxt_.user.user_dataset.begin_write();
	user_identity.user_set user_sessions;
	wr.get_global_obj(dsattr.http_sessions, user_sessions);
	for (uid : user_ids) {
		try {
			user_sessions.logout_user(user_identity.normalize(uid));
		} catch {
			// Users was not logged in
		}
	}
	wr.set_global_attrib(dsattr.http_sessions, user_sessions);
	wr.commit();
}

void api.logout_all()
{
	if (null(internal_ctxt_.login))
		throw (E_UNSPECIFIC, "Login is not enabled");

	qlc.hce.writer wr = internal_ctxt_.user.user_dataset.begin_write();
	user_identity.user_set user_sessions = new user_identity.user_set();
	wr.set_global_attrib(dsattr.http_sessions, user_sessions);
	wr.commit();
}

void api.set_activate_debug_response_on_exception(logical activate)
{
	activate_debug_response_on_exception_ = activate;
}

map_str_obj<request_handler> api._find_path_handlers(URL url)
{
	vector(string) url_patterns = pattern2method2handler_.get_keys(false);
	integer n = v_size(url_patterns);
	for (integer i = 0; i < n; ++i) {
		string url_pattern = url_patterns[i];
		if (url.like(url_pattern))
			return pattern2method2handler_.find(url_pattern);
	}

	return null<map_str_obj<request_handler>>;
}

response api._resp_if_login_req(
	http.method method,
	URL url,
	blob option(nullable) body,
	context ctxt,
	client_info client,
	out logical save_req_resp)
{
	if (method != http.POST)
		return null<response>;

	if (null(internal_ctxt_.login) || url.path() != internal_ctxt_.login.login_path)
		return null<response>;

	save_req_resp = false;

	try {
		response resp = __login_user(
			internal_ctxt_,
			client,
			body,
			save_req_resp);

		resp.set_handler_id(HANDLER_ID_LOGIN);

		if (resp.get().status_code() != 201)
			return resp; // Login failed

		internal_ctxt_.login.new_user_func(client, ctxt);
		permissions_base pb = internal_ctxt_.login.permission_load_func(client.user_id(), ctxt);
		client.set_permissions(pb);

		if (internal_ctxt_.user.log_events) {
			logging.info(internal_ctxt_.user.logging_context, ["User '", client.user_id(), "' logged in."]);
			logging.info(internal_ctxt_.user.logging_context, ["Permissions: ", client.get_permissions_base().description(), "."]);
		}

		return resp;
	} catch {
		if (activate_debug_response_on_exception_)
			return new response(mk_error_response(500, err.print()), HANDLER_ID_LOGIN);
		else
			return new response(mk_error_response(500, "Unhandled exception"), HANDLER_ID_LOGIN);
	}
}

void api._load_permissions(client_info client, context ctxt)
{
	permissions_base pb = client.get_permissions_base();
	if (!null(pb)) {
		timestamp expire_time = pb.get_load_time() + internal_ctxt_.login.permission_lifetime_minutes * 60000;
		if (now_ut() < expire_time)
			return; // No need to update permissions
	}

	pb = internal_ctxt_.login.permission_load_func(client.user_id(), ctxt);
	pb.set_load_time(now_ut());
	client.set_permissions(pb);
}

response api._resp_if_auth_fail(http.request req, context ctxt, client_info client, out logical save_req_resp)
{
	save_req_resp = false;

	if (null(internal_ctxt_.login))
		return null<response>;

	string auth = req.header("Authorization");
	if (null(auth))
		return new response(mk_error_response(401, "Access token missing"), HANDLER_ID_FAILURE);

	string token = str_replace_casei(auth, "bearer ", "");

	user_identity.user_set user_sessions;
	try {
		qlc.hce.reader user_rd = internal_ctxt_.user.user_dataset.begin_read();
		user_rd.get_global_obj(dsattr.http_sessions, user_sessions);
	} catch {
		return new response(mk_error_response(500, "Error finding sessions"), HANDLER_ID_FAILURE);
	}

	logical changed_user;
	try {
		logical last_access_updated;
		string user_id = user_sessions.get_user_info(token, last_access_updated);

		if (last_access_updated) {
			// We must update the dataset, since enough time has passed to change "last access"
			qlc.hce.writer user_wr = internal_ctxt_.user.user_dataset.begin_write();
			user_wr.get_global_obj(dsattr.http_sessions, user_sessions);
			user_id = user_sessions.get_user_info(token, last_access_updated); // Updates last access again
			user_wr.set_global_attrib(dsattr.http_sessions, user_sessions);
			user_wr.commit();
		}

		// Most often user id in client_info remains the same, but a connection can pass a token
		// for another user. We must be prepared for this, in particular since permissions stored
		// in client_info must be reset when it happens.
		changed_user = client.set_user_id(user_id);
	} catch {
		return new response(mk_error_response(401, strcat("Authorization fail: ", err.message())), HANDLER_ID_FAILURE);
	}

	save_req_resp = user_sessions.should_save_req_resp(client.user_id());

	if (changed_user) {
		if (internal_ctxt_.user.log_events)
			logging.info(internal_ctxt_.user.logging_context, ["User is now '", client.user_id(), "'."]);

		try {
			_load_permissions(client, ctxt);
			if (!client.get_permissions_base().has_any_rights())
				return new response(mk_error_response(403, "User has no access rights"), HANDLER_ID_FAILURE);
			if (internal_ctxt_.user.log_events)
				logging.info(internal_ctxt_.user.logging_context, ["Permissions: ", client.get_permissions_base().description(), "."]);
		} catch {
			return new response(mk_error_response(500, "Error loading permissions"), HANDLER_ID_FAILURE);
		}
	}

	if (null(client.get_permissions_base()))
		throw (E_UNSPECIFIC, "Error getting permissions");

	// User has successfully passed authorization
	return null<response>;
}

response api._resp_if_open_api_req(http.method method, URL url)
{
	if (method != http.GET)
		return null<response>;

	if (url.path() != open_api.get_url) {
		return null<response>;
	}

	string open_api_json = this.create_open_api();
	json.node jn = json.parse(open_api_json);
	return new response(mk_response(200, jn), HANDLER_ID_OPEN_API);
}

void api._check_ambiguity(url_path path)
{
	vector(string) path_elements = path.url_parts();
	integer n_elems = v_size(path_elements);

	vector(string) url_patterns = pattern2method2handler_.get_keys(false);
	integer n = v_size(url_patterns);
	for (integer i = 0; i < n; ++i) {
		vector(string) compare_path_elements = str_tokenize(url_patterns[i], "/")[1:];
		if (v_size(compare_path_elements) != n_elems)
			continue;

		if (url_patterns[i] == path.pattern())
			continue;

		// To be able to always differentiate two paths, they must either
		//   - not have the same number of elements (checked above)
		//   - have at least one element position where both paths have fixed string, and these are different
		// E.g. "/items/{item}" and "/{model}/info" are ambiguous for URL "/items/info",
		// but "/items/{item}/price and "/{model}/info/id" can always be distinguished on the third part.
		// However "items/{item}/id" and "/{model}/info/id" are ambiguous.
		logical found_not_ambiguous = false;
		for (integer k = 0; k < n_elems; ++k) {
			string elem = path_elements[k];
			string cmp_elem = compare_path_elements[k];
			if (elem != "*" && cmp_elem != "*" && elem != cmp_elem) {
				found_not_ambiguous = true;
				break;
			}
		}

		if (!found_not_ambiguous)
			throw (E_UNSPECIFIC, strcat(["URL paths \"", path.path(), "\" and \"", pattern2path_.find(url_patterns[i]), "\" are ambiguous"]));
	}
}

response api._process(http.request req, context ctxt, client_info client)
{
	internal_ctxt_.advance_counter();
	URL req_url = new URL(req.url());

	logical save_req_resp;
	{ // Login request check (only active if login_path is set)
		response login_resp = _resp_if_login_req(req.method(), req_url, req.body(), ctxt, client, save_req_resp);
		if (!null(login_resp)) {
			if (save_req_resp)
				// N.B. It is intentional not to save the request, since
				// that would save passwords to file!
				save_resp_log(login_resp, internal_ctxt_, client);
			return login_resp;
		}
	}

	response resp;

	{ // Request for OPEN API spec?
		resp = _resp_if_open_api_req(req.method(), req_url);
		if (!null(resp))
			return resp;
	}

	request_handler handler;
	{
		map_str_obj<request_handler> method2handlers = _find_path_handlers(req_url);
		if (null(method2handlers)) {
			resp = new response(mk_response(404), HANDLER_ID_FAILURE);
		} else {
			handler = method2handlers.find(enum_id(req.method()));
			if (null(handler)) {
				resp = new response(mk_method_not_allowed_response(method2handlers.get_keys()), HANDLER_ID_FAILURE);
			}
		}
	}

	// If we didn't find any handler, we prioritize returning 401 over 404 or 405.
	if (null(handler) || handler.requires_authentication())
	{ // Authorization check (only active if login_path is set)
		response auth_fail_resp = _resp_if_auth_fail(req, ctxt, client, save_req_resp);
		if (!null(auth_fail_resp)) {
			if (save_req_resp) {
				save_req_log(req, internal_ctxt_, client);
				save_resp_log(auth_fail_resp, internal_ctxt_, client);
			}
			return auth_fail_resp;
		}
	} else {
		save_req_resp = false; // No authentication -> We cannot save req/resp
	}

	if (save_req_resp)
		save_req_log(req, internal_ctxt_, client);

	request_values req_vals;
	if (null(resp)) {
		if (!null(handler.id()))
			ctxt.set_handler_id(handler.id());

		req_vals = new request_values();
		try {
			add_path_params(req_vals, handler, req_url);
			add_query_params(req_vals, handler, req_url);
			add_body(req_vals, handler, req.body());
		} catch {
			resp = new response(mk_error_response(400, err.message()), handler.id());
		}
	}

	if (null(resp)) {
		string forbidden_reason = handler.permission_func()(ctxt, client, req_vals);
		if (!null(forbidden_reason))
			resp = new response(mk_error_response(403, forbidden_reason), handler.id());
	}

	if (null(resp)) {
		try {
			if (!null(handler.id()) &&
				(handler.id() == HANDLER_ID_POST_SAVE_REQ_RESP || handler.id() == HANDLER_ID_DELETE_SAVE_REQ_RESP))
			{
				resp = handler.func()(internal_ctxt_, client, req_vals);
			} else {
				resp = handler.func()(ctxt, client, req_vals);
			}

			if (!null(handler.id()))
				resp.set_handler_id(handler.id());
		} catch {
			if (activate_debug_response_on_exception_)
				resp = new response(mk_error_response(500, err.print()), handler.id());
			else
				resp = new response(mk_error_response(500, "Unhandled exception"), handler.id());
		}
	}

	if (save_req_resp)
		save_resp_log(resp, internal_ctxt_, client);

	return resp;
}

response api.process(http.request req, context ctxt, client_info client)
{
	try {
		return _process(req, ctxt, client);
	} catch {
		throw (E_UNSPECIFIC, err.print());
	}
}

void api._open_api_append_schemas(json.builder jb)
{
	vector(string) schema_names;
	vector(open_api.schema) schemas;
	schemas_.get_content(schema_names, schemas);
	integer n = v_size(schemas);
	if (n == 0)
		return;

	sort(schemas, schema_names);
	sort(schema_names);

	jb.begin_member("components");
	jb.begin_object();
	jb.begin_member("schemas");
	jb.begin_object();
	for (integer i = 0; i < n; ++i) {
		jb.begin_member(schema_names[i]);
		jb.append_node(schemas[i].mk_json_node());
	}
	jb.end_object(); // schemas
	jb.end_object(); // components
}


void api._open_api_append_paths(json.builder jb)
{
	jb.begin_member("paths");
	jb.begin_object();

	jb.begin_member(open_api.get_url);
	jb.begin_object();
	jb.begin_member("get");
	jb.begin_object();
	jb.begin_member("description");
	jb.append("Get the json for the OPEN API specification");
	jb.begin_member("responses");
	jb.begin_object();
	jb.begin_member("200");
	jb.begin_object();
	jb.begin_member("description");
	jb.append(default_status_message(200));
	jb.end_object();
	jb.end_object();
	jb.end_object();
	jb.end_object();

	vector(string) url_patterns;
	vector(map_str_obj<request_handler>) method2handlers;
	pattern2method2handler_.get_content(url_patterns, method2handlers);
	sort(method2handlers, url_patterns);
	sort(url_patterns);
	integer n = v_size(url_patterns);
	for (integer i = 0; i < n; ++i) {
		string url_pattern = url_patterns[i];
		string path = pattern2path_.find(url_pattern);
		jb.begin_member(path);
		jb.begin_object();
		vector(string) methods = method2handlers[i].get_keys();
		vector(request_handler) handlers = method2handlers[i].find(methods);
		integer m = v_size(methods);
		for (integer j = 0; j < m; ++j) {
			request_handler h = handlers[j];
			open_api.append_method(jb, h);
		}
		jb.end_object(); // path
	}

	jb.end_object(); // paths
}

string api._create_open_api()
{
	json.builder jb = new json.builder();
	jb.begin_object();
	jb.begin_member("openapi");
	jb.append("3.0.1");

	jb.begin_member("info");
	jb.begin_object();
	jb.begin_member("title");
	jb.append(title_);
	jb.begin_member("version");
	jb.append(version_);
	jb.end_object(); // info

	_open_api_append_schemas(jb);

	_open_api_append_paths(jb);

	/* Appearently consumes and produces can't be in the root
	// api only supports json everywhere
	jb.begin_member("consumes");
	jb.begin_array();
	jb.append("application/json");
	jb.end_array();

	jb.begin_member("produces");
	jb.begin_array();
	jb.append("application/json");
	jb.end_array();
	*/

	jb.end_object(); // root

	return jb.get();
}

string api.create_open_api()
{
	try {
		return _create_open_api();
	} catch {
		// It is too dangerous to throw exception here. Ideally, impossible API
		// should have been caught earlier. Other devs cannot be expected to
		// put call to create_open_api in a try-catch clause, and the server
		// should not fail due to OPEN api json not being possible.
		return "{ \"error\": \"Could not create Open API json\" }";
	}
}


void get_user_info(
	qlc.hce.reader user_dataset_rd,
	out vector(string) user_ids,
	out user_identity.user_set user_sessions)
{
	map_str_str uid2hash;
	user_dataset_rd.get_global_obj(dsattr.http_user_pwd, uid2hash);
	user_ids = uid2hash.get_keys(true);

	user_dataset_rd.get_global_obj(dsattr.http_sessions, user_sessions);
}

}
