option(null: hard);

module open_api {

string get_url option(constant) = "/docs";

module __impl__ {

enum schema_element { object_prop, object_type, array_type_empty, array_type_filled };

class element_stack {
public:
	void push(schema_element e);
	schema_element pop();
	schema_element top();
	logical is_empty();

private:
	vector(schema_element) stack_;
};

void element_stack.push(schema_element e)
{
	push_back(stack_, e);
}

schema_element element_stack.pop()
{
	integer n = v_size(stack_);
	if (n == 0)
		throw (E_UNSPECIFIC, "Stack is empty");
	schema_element e = stack_[n-1];
	if (n == 1)
		stack_ = null<vector(schema_element)>;
	else
		stack_ = stack_[0:n-2];
	return e;
}

schema_element element_stack.top()
{
	integer n = v_size(stack_);
	if (n == 0)
		throw (E_UNSPECIFIC, "Stack is empty");
	return stack_[n-1];
}

logical element_stack.is_empty() = v_size(stack_) == 0;

}

class enum dtype {
	str option(str: "string")
		, num option(str: "number")
		, bool option(str: "boolean")
		, int option(str: "integer")
};

class schema {
public:
	schema(json.builder jb);
	string mk_json();
	json.node mk_json_node();

private:
	string json_;
};

schema.schema(json.builder jb)
: json_(jb.get())
{
}

string schema.mk_json() = json_;

json.node schema.mk_json_node() = json.parse(json_);


class schema_builder {
public:
	schema_builder();

	schema_builder set_type_boolean(logical option(nullable) dflt = null<logical>, logical option(nullable) example = null<logical>);
	schema_builder set_type_integer();
	schema_builder set_type_integer_w_default(integer dflt);
	schema_builder set_type_integer_w_example(integer example);
	schema_builder set_type_integer(integer dflt, integer example);
	schema_builder set_type_number(number option(nullable) dflt = null<number>, number option(nullable) example = null<number>);
	schema_builder set_type_string(string option(nullable) dflt = null<string>, string option(nullable) example = null<string>);
	schema_builder set_type_ref(string schema_builder_name);

	schema_builder start_type_object();
	schema_builder add_property(string name, logical requried);
	schema_builder end_type_object();

	schema_builder start_type_array();
	schema_builder end_type_array();

	schema get_schema();

private:
	json.builder jb_;
	logical finished_;
	logical broken_;
	__impl__.element_stack stack_;

	vector(string) required_;

	schema_builder _set_type(
		dtype t,
		string option(nullable) str_dflt,
		string option(nullable) str_ex,
		number option(nullable) num_dflt,
		number option(nullable) num_ex,
		logical option(nullable) logical_dflt,
		logical option(nullable) logical_ex);

	void _check_modify_allowed();
	void _throw(string msg);

	void _require(string name);

	void _check_property_allowed();
	void _check_type_allowed();

	void _type_ended();
};

schema_builder.schema_builder()
{
	jb_ = new json.builder();
	finished_ = false;
	broken_ = false;
	stack_ = new __impl__.element_stack();
	required_ = [""];
}

void schema_builder._check_modify_allowed()
{
	if (broken_)
		throw (E_UNSPECIFIC, "schema_builder has thrown exception and is not usable");
	if (finished_)
		throw (E_UNSPECIFIC, "schema_builder is finished, and cannot be modified");
}

void schema_builder._throw(string msg)
{
	broken_ = true;
	throw (E_UNSPECIFIC, msg);
}

void schema_builder._check_property_allowed()
{
	_check_modify_allowed();
	if (stack_.is_empty() || stack_.top() != __impl__.object_type)
		_throw("Not in object");
}

void schema_builder._check_type_allowed()
{
	_check_modify_allowed();
	if (stack_.is_empty())
		return;

	if (stack_.top() != __impl__.object_prop && stack_.top() != __impl__.array_type_empty)
		_throw("Cannot set type at this point");
}

void schema_builder._type_ended()
{
	if (stack_.is_empty()) {
		finished_ = true;
	} else if (stack_.top() == __impl__.object_prop) {
		stack_.pop();
	} else if (stack_.top() == __impl__.array_type_empty) {
		stack_.pop();
		stack_.push(__impl__.array_type_filled);
	}
}

void schema_builder._require(string name)
{
	integer k = v_size(required_)-1;
	required_[k] = strcat([required_[k], "|", name]);
}

schema_builder schema_builder._set_type(
	dtype t,
	string option(nullable) str_dflt,
	string option(nullable) str_ex,
	number option(nullable) num_dflt,
	number option(nullable) num_ex,
	logical option(nullable) logical_dflt,
	logical option(nullable) logical_ex)
{
	_check_type_allowed();
	jb_.begin_object();
	jb_.begin_member("type");
	jb_.append(string(t));

	logical has_default =
		(t == dtype.str && !null(str_dflt)) ||
		(t == dtype.num && !null(num_dflt)) ||
		(t == dtype.int && !null(num_dflt)) ||
		(t == dtype.bool && !null(logical_dflt));

	if (has_default) {
		jb_.begin_member("default");
		switch (t) {
		case dtype.str:
			jb_.append(str_dflt);
			break;
		case dtype.num:
		case dtype.int:
			jb_.append(num_dflt);
			break;
		case dtype.bool:
			jb_.append(logical_dflt);
			break;
		}
	}

	jb_.end_object();
	_type_ended();
	return this;
}

schema_builder schema_builder.set_type_string(string option(nullable) dflt, string option(nullable) example)
{
	return _set_type(dtype.str, dflt, example, null<number>, null<number>, null<logical>, null<logical>);
}

schema_builder schema_builder.set_type_boolean(logical option(nullable) dflt, logical option(nullable) example)
{
	return _set_type(dtype.bool, null<string>, null<string>, null<number>, null<number>, dflt, example);
}

schema_builder schema_builder.set_type_integer()
{
	return _set_type(dtype.int, null<string>, null<string>, null<number>, null<number>, null<logical>, null<logical>);
}

schema_builder schema_builder.set_type_integer_w_default(integer dflt)
{
	return _set_type(dtype.int, null<string>, null<string>, dflt, null<number>, null<logical>, null<logical>);
}

schema_builder schema_builder.set_type_integer_w_example(integer example)
{
	return _set_type(dtype.int, null<string>, null<string>, null<number>, example, null<logical>, null<logical>);
}

schema_builder schema_builder.set_type_integer(integer dflt, integer example)
{
	return _set_type(dtype.int, null<string>, null<string>, dflt, example, null<logical>, null<logical>);
}

schema_builder schema_builder.set_type_number(number option(nullable) dflt, number option(nullable) example)
{
	return _set_type(dtype.num, null<string>, null<string>, dflt, example, null<logical>, null<logical>);
}

schema_builder schema_builder.start_type_object()
{
	_check_type_allowed();
	jb_.begin_object();
	jb_.begin_member("properties");
	jb_.begin_object();
	push_back(required_, "");
	stack_.push(__impl__.object_type);
	return this;
}

schema_builder schema_builder.add_property(string name, logical requried)
{
	_check_property_allowed();
	stack_.push(__impl__.object_prop);
	jb_.begin_member(name);
	if (requried)
		_require(name);
	return this;
}

schema_builder schema_builder.end_type_object()
{
	_check_modify_allowed();
	if (stack_.is_empty() || stack_.pop() != __impl__.object_type)
		_throw("Not in object");
	jb_.end_object(); // end "properties"
	integer nr = v_size(required_);
	vector(string) req_here = str_tokenize(required_[nr-1], "|");
	if (v_size(req_here) > 1) {
		jb_.begin_member("required");
		jb_.append(req_here[1:]);
	}
	jb_.end_object();
	required_ = required_[0:nr-2];
	_type_ended();
	return this;
}

schema_builder schema_builder.start_type_array()
{
	_check_type_allowed();
	jb_.begin_object();
	jb_.begin_member("type");
	jb_.append("array");
	jb_.begin_member("items");
	stack_.push(__impl__.array_type_empty);
	return this;
}

schema_builder schema_builder.set_type_ref(string schema_builder_name)
{
	_check_type_allowed();
	jb_.begin_object();
	jb_.begin_member("$ref");
	jb_.append(strcat("#/components/schema_builders/", schema_builder_name));
	jb_.end_object();
	_type_ended();
	return this;
}

schema_builder schema_builder.end_type_array()
{
	_check_modify_allowed();
	if (stack_.is_empty() || stack_.pop() != __impl__.array_type_filled)
		_throw("Cannot end array here");
	jb_.end_object(); // end "items"
	_type_ended();
	return this;
}

schema schema_builder.get_schema()
{
	if (!finished_)
		throw (E_UNSPECIFIC, "Schema is incomplete");

	if (broken_)
		throw (E_UNSPECIFIC, "Schema is broken");

	return new schema(jb_);
}


class response_desc {
public:
	response_desc(integer status_code, string desc);
	response_desc(integer status_code, string desc, schema sch);

	void mk_json(json.builder jb);

	void _init(integer status_code, string desc, schema option(nullable) sch);

	integer status_code;
	string desc;
	schema sch;
};

response_desc.response_desc(integer status_code, string desc)
{
	_init(status_code, desc, null<schema>);
}

response_desc.response_desc(integer status_code, string desc, schema sch)
{
	_init(status_code, desc, sch);
}

void response_desc._init(integer status_code, string desc, schema option(nullable) sch)
{
	http_api.default_status_message(status_code); // check that status code is valid
	this.status_code = status_code;
	this.desc = desc;
	this.sch = sch;
}

void response_desc.mk_json(json.builder jb)
{
	jb.begin_member(str(status_code));
	jb.begin_object();
	jb.begin_member("description");
	jb.append(null(desc) ? http_api.default_status_message(status_code) : desc);
	if (!null(this.sch)) {
		jb.begin_member("content");
		jb.begin_object();
		jb.begin_member("application/json");
		jb.begin_object();
		jb.begin_member("schema");
		jb.append_node(sch.mk_json_node());
		jb.end_object();
		jb.end_object();
	}
	jb.end_object();
}


class method {
public:
	string summary;
	string description;
	vector(response_desc) response_descs;
};

class path {
public:
	method get;
	method put;
	method post;
	method delete;
};


module generate_ql {

json.node find_ref_node(json.node root, string ref)
{
	if (!str_startswith(ref, "#/"))
		throw (E_UNSPECIFIC, "ref type not supported");

	vector(string) path = str_tokenize(ref, "/");
	json.node ret;
	for (m : path; ix) {
		if (ix == 0)
			ret = root;
		else
			ret = ret.get_object_member(m);
	}

	return ret;
}

string path_part_to_module(string path_part)
{
	if (path_part == "")
		return "root";

	if (str_startswith(path_part, "{") && str_endswith(path_part, "}"))
		path_part = str_to_upper(sub_string(path_part, 1, strlen(path_part)-2));

	vector(integer) ch_azAZ09_ = str_split("azAZ09_");
	vector(integer) v_ch = str_split(path_part);
	for (out ch : v_ch) {
		if (ch >= ch_azAZ09_[0] && ch <= ch_azAZ09_[1])
			continue; // a-z
		if (ch >= ch_azAZ09_[2] && ch <= ch_azAZ09_[3])
			continue; // A-Z
		if (ch >= ch_azAZ09_[4] && ch <= ch_azAZ09_[5])
			continue; // 0-9
		if (ch == ch_azAZ09_[6])
			continue; // _
		ch = ch_azAZ09_[6]; // set to _
	}

	return str_build(v_ch);
}

vector(string) path_to_modules(string path)
{
	if (path == "/")
		return ["root"];

	vector(string) path_parts = str_tokenize(path, "/");
	vector(string) modules = path_part_to_module(path_parts[1:]);
	return modules;
}

class ql_generator {
public:
	ql_generator(json.node root, string indent);
	void begin_path(string path);
	void end_path();
	void begin_method(string method);
	void end_method();
	void add_schema(string resp_code, json.node schema_node);
	void print(string top_module, out_stream fh, string eol = "\n");

private:
	json.node root_;
	string indent_;

	map_str_str ref2class_;

	vector(string) current_path_modules_;
	vector(string) def_lines_;
	vector(string) schema_lines_;

	string _class_name_from_ref(string ref);
	void _mk_enum(string enum_name, json.node values_node);
	string _decode_ref(string ref);
	string _handle_array(json.node items_node, string option(nullable) ref, string option(nullable) prop_name);
	string _mk_class(json.node prop_node, string class_name, logical is_component);
	string _mk_type(json.node prop_def, string option(nullable) ref, string option(nullable) prop_name);
};

ql_generator.ql_generator(json.node root, string indent)
	: root_(root), indent_(indent), ref2class_(map_str_str())
{
	resize(current_path_modules_, 0);
	resize(def_lines_, 0);
	resize(schema_lines_, 0);
}

void ql_generator._mk_enum(string enum_name, json.node values_node)
{
	integer n_values = values_node.get_array_size();
	vector(string) values = vector(i:n_values; values_node.get_array_element(i).get_string());

	// We always put enums in the components section, even if they aren't found
	// there in the open api document. It's easier than keeping track of where
	// the enum should be.

	push_back(def_lines_, strcat(["class enum ", enum_name, " {"]));
	if (v_size(values) <= 5) {
		push_back(def_lines_, strcat([indent_, str_join(values, ", ")]));
	} else {
		push_back(def_lines_, strcat([indent_, values[0]]));
		for (val : values[1:])
			push_back(def_lines_, strcat([indent_, ", ", val]));
	}
	push_back(def_lines_, "};");
	push_back(def_lines_, "");
}

string ql_generator._class_name_from_ref(string ref)
{
	if (!str_startswith(ref, "#"))
		throw (E_UNSPECIFIC, "Cannot make class name out of ref");

	vector(string) ref_parts = str_tokenize(ref, "/");

	return strcat(ref_parts[v_size(ref_parts)-1], "_t");
}
	
string ql_generator._decode_ref(string ref)
{
	string member_class_name = ref2class_.find(ref);
	if (!null(member_class_name))
		return member_class_name;

	json.node ref_node = find_ref_node(root_, ref);
	member_class_name = _mk_type(ref_node, ref, null<string>);
	ref2class_.add(ref, member_class_name);
	return member_class_name;
}

string ql_generator._handle_array(json.node items_node, string option(nullable) ref, string option(nullable) prop_name)
{
	string type_name = _mk_type(items_node, ref, prop_name);
	return strcat(["vector(", type_name, ")"]);
}

string ql_generator._mk_type(json.node prop_def, string option(nullable) ref, string option(nullable) prop_name)
{
	if (null(ref) && null(prop_name))
		throw (E_INVALID_ARG, "Need ref or prop_name");

	string custom_type = null(ref) ? strcat(prop_name, "_t") : _class_name_from_ref(ref);

	// Descend into allOf by assuming first element is what we want, and the
	// rest is unimportant.  This will not work if the rest is important,
	// e.g. someone is trying to use inheritance.
	if (prop_def.has_object_member("allOf")) {
		prop_def = prop_def.get_object_member("allOf").get_array_element(0);
	}

	if (prop_def.has_object_member("$ref")) {
		return _decode_ref(prop_def.get_object_member("$ref").get_string());
	}

	if (prop_def.has_object_member("additionalProperties"))
		logging.warning(["additionalProperties is not supported"]);

	if (prop_def.has_object_member("properties")) {
		string class_name = null(ref) ? strcat(prop_name, "_t") : _class_name_from_ref(ref);
		return _mk_class(prop_def.get_object_member("properties"), class_name, !null(ref));
	}

	if (!prop_def.has_object_member("type"))
		throw (E_UNSPECIFIC, "Expected $ref or type");

	switch (prop_def.get_object_member("type").get_string()) {
	case "string":
		if (prop_def.has_object_member("enum")) {
			if (null(ref) || null(ref2class_.find(custom_type))) {
				_mk_enum(custom_type, prop_def.get_object_member("enum"));
				if (!null(ref))
					ref2class_.add(ref, custom_type);
			}
			return custom_type;
		} else {
			return "string";
		}
	case "boolean":
		return "logical";
	case "integer":
		return "integer";
	case "number":
		return "number";
	case "array":
		return _handle_array(prop_def.get_object_member("items"), ref, prop_name);
	case "object":
		return "string"; // Just survive additionalProperties only object
	default:
		throw (E_UNSPECIFIC, "Unhandled type");
	}
}

string ql_generator._mk_class(json.node prop_node, string class_name, logical is_component)
{
	vector(string) class_lines = [
		strcat(["class ", class_name, " {"]),
		"public:"];

	for (member_prop_name : prop_node.get_object_members()) {
		json.node prop_def = prop_node.get_object_member(member_prop_name);
		string member_type = _mk_type(prop_def, null<string>, member_prop_name);
		push_back(class_lines, strcat([indent_, member_type, " ", member_prop_name, ";"]));
	}

	push_back(class_lines, "};");
	push_back(class_lines, "");

	if (is_component)
		for (line : class_lines)
			push_back(def_lines_, line);
	else
		for (line : class_lines)
			push_back(schema_lines_, line);

	return class_name;
}

void ql_generator.begin_path(string path)
{
	current_path_modules_ = path_to_modules(path);
	for (m : current_path_modules_)
		push_back(schema_lines_, strcat(["module ", m, " {"]));
}

void ql_generator.end_path()
{
	for (m : current_path_modules_)
		push_back(schema_lines_, strcat("} // ", m));
	push_back(schema_lines_, "");
}

void ql_generator.begin_method(string method)
{
	push_back(schema_lines_, strcat(["module ", method, " {"]));
}

void ql_generator.end_method()
{
	push_back(schema_lines_, "}");
}

void ql_generator.add_schema(string resp_code, json.node schema_node)
{
	if (resp_code == "default")
		resp_code = "default_code"; // default can't be used as identifier in ql code

	if (schema_node.has_object_member("$ref")) {
		string type = _decode_ref(schema_node.get_object_member("$ref").get_string());
		push_back(schema_lines_, strcat(["class resp_", resp_code, " : public components.", type, " {};"]));
		push_back(schema_lines_, "");
	} else {
		if (schema_node.get_object_member("type").get_string() != "object")
			throw (E_UNSPECIFIC, "Schema must be $ref or object");
		_mk_class(schema_node.get_object_member("properties"), resp_code, false);
	}
}

void ql_generator.print(string top_module, out_stream fh, string eol)
{
	fh.write(["option(null: hard);", eol, eol]);
	fh.write(["module ", top_module, " {", eol, eol]);

	fh.write(["module components {", eol, eol]);

	for (line : def_lines_)
		fh.write([line, eol]);

	fh.write(["} // components", eol, eol]);

	for (line : schema_lines_)
		fh.write([line, eol]);

	fh.write(["} // ", top_module, eol]);
}

void open_api_to_ql(string open_api_file, string option(nullable) ql_file, string top_module, string indent)
{
	if (null(ql_file))
		ql_file = str_replace(open_api_file, "json", "ql");

	if (equal_casei(open_api_file, ql_file))
		throw (E_UNSPECIFIC, "ql file must be different from open api file");

	json.node api_root = json.parse(file_read(open_api_file));

	ql_generator ql_gen = new ql_generator(api_root, indent);

	for (path : api_root.get_object_member("paths").get_object_members()) {
		json.node path_node = api_root.get_object_member("paths").get_object_member(path);

		ql_gen.begin_path(path);

		for (method : path_node.get_object_members()) {
			ql_gen.begin_method(method);

			for (resp_status : path_node.get_object_member(method).get_object_member("responses").get_object_members()) {
				json.node resp_node = path_node.get_object_member(method).get_object_member("responses").get_object_member(resp_status);

				json.node schema_node;
				if (resp_node.has_object_member("schema")) {
					schema_node = resp_node.get_object_member("schema");
				} else if (resp_node.has_object_member("content")) {
					if (resp_node.get_object_member("content").has_object_member("application/json")) {
						if (resp_node.get_object_member("content").get_object_member("application/json").has_object_member("schema")) {
							schema_node = resp_node.get_object_member("content").get_object_member("application/json").get_object_member("schema");
						}
					}
				}

				if (null(schema_node))
					logging.warning(["Could not find schema for ", method, " ", path, " resp ", resp_status, "."]);
				else
					ql_gen.add_schema(resp_status, schema_node);
			}
			ql_gen.end_method();
		}

		ql_gen.end_path();
	}

	out_stream fh = out_stream_file(ql_file);
	ql_gen.print(top_module, fh);
}

} // generate_ql

} // module open_api
