module logging {

integer MAX_IMPORTANCE option(constant) = 1000;
integer MIN_IMPORTANCE option(constant) = 0;
// All importances i must satisfy MIN_IMPORTANCE < i < MAX_IMPORTANCE

// *** LEVELS ***

// Users handle the enums, which are converted to
// premade instances of the class internally.

// To add a level, ensure you update everwhere, where it's needed, and add
// functions to log as well (logging.info(...) etc)

enum LogLevel option(category: "Logging/Logging") {
    EVERYTHING option(str: "EVERYTHING"),
    DEBUG option(str: "DEBUG"),
    INFO option(str: "INFO"),
    WARNING option(str: "WARNING"),
    ERROR option(str: "ERROR"),
    FATAL option(str: "FATAL"),
    NOTHING option(str: "NOTHING")
};


integer fixlen_len option(constant) = 5;
class LogLevelObj option(category: "Logging/Logging") {
public:
    // Don't call the constructor to create new levels, use create_level function below!
    LogLevelObj(
        string name,
        string fixlen_name,
        string short_name,
        integer importance,
        LogLevel level_enum,
        integer index,
        logical special);

    // Special log levels cannot be logged to, they are only used to control
    // what levels are logged.
    logical is_special();
    string name();
    string fixlen_name();
    string short_name();
    integer importance();
    LogLevel get_enum();
    integer index();

private:
    logical special_;
    string name_;
    string fixlen_name_;
    string short_name_;
    integer importance_;
    LogLevel level_enum_;
    integer index_;
};

// Don't call the constructor willy nilly, do it and everything in init_loglevels()
LogLevelObj.LogLevelObj(
    string name,
    string fixlen_name,
    string short_name,
    integer importance,
    LogLevel level_enum,
    integer index,
    logical special)
    :
    special_(special),
    name_(name),
    fixlen_name_(fixlen_name),
    short_name_(short_name),
    importance_(importance),
    level_enum_(level_enum),
    index_(index)
{
    if (string(level_enum_) != name_) {
        throw (E_CONSTRAINT, strcat(["Enum ", string(level_enum_), " does not match name ", name_]));
    }
    if (!special_ && (importance <= MIN_IMPORTANCE || importance >= MAX_IMPORTANCE)) {
        throw (E_CONSTRAINT, strcat(["Importance ", str(importance), " for ", name, " is out of range"]));
    }
    if (strlen(fixlen_name) != fixlen_len) {
        throw (E_CONSTRAINT, strcat(["fixlen_name for LogLevelObj must be ", str(fixlen_len), " chars long"]));
    }
}
logical LogLevelObj.is_special() = special_;
string LogLevelObj.name() = name_;
string LogLevelObj.fixlen_name() = fixlen_name_;
string LogLevelObj.short_name() = short_name_;
integer LogLevelObj.importance() = importance_;
LogLevel LogLevelObj.get_enum() = level_enum_;
integer LogLevelObj.index() = index_;


module __impl__ {

vector(LogLevelObj) _LogLevelObjs;
LogLevelObj _EVERYTHING;
LogLevelObj _DEBUG;
LogLevelObj _INFO;
LogLevelObj _WARNING;
LogLevelObj _ERROR;
LogLevelObj _FATAL;
LogLevelObj _NOTHING;

void init_loglevels()
{
    if (!null(_LogLevelObjs))
        return;

    integer index = 0;

    _EVERYTHING = new LogLevelObj("EVERYTHING", "EVTH ", "0", MIN_IMPORTANCE, EVERYTHING, index++, true);
    _DEBUG = new LogLevelObj("DEBUG", "DEBUG", "D", 10, DEBUG, index++, false);
    _INFO = new LogLevelObj("INFO", "INFO ", "I", 20, INFO, index++, false);
    _WARNING = new LogLevelObj("WARNING", "WARN ", "W", 30, WARNING, index++, false);
    _ERROR = new LogLevelObj("ERROR", "ERROR", "E", 40, ERROR, index++, false);
    _FATAL = new LogLevelObj("FATAL", "FATAL", "F", 50, FATAL, index++, false);
    _NOTHING = new LogLevelObj("NOTHING", "NOTH ", "1", MAX_IMPORTANCE, NOTHING, index++, true);

    _LogLevelObjs = [
        _EVERYTHING,
        _DEBUG,
        _INFO,
        _WARNING,
        _ERROR,
        _FATAL,
        _NOTHING
        ];

    // Check the following:
    // * LogLevelObj.index() is correct
    // * First and last elements have MIN_IMPORTANCE and MAX_IMPORTANCE
    // * Importances are in strict ascending order
    // * No names, fixlen_names or shortnames are repeated
    // * First and last elements, and only these, are special
    integer _LogLevelObjs_sz = v_size(_LogLevelObjs);
    set_istr names = set_istr();
    set_istr fixnames = set_istr();
    set_istr shortnames = set_istr();
    for (integer i = 0; i < _LogLevelObjs_sz; ++i) {
        LogLevelObj l = _LogLevelObjs[i];
        if (l.index() != i)
            throw (E_UNSPECIFIC, "Internal: Place in vector is not index");
        if (i == 0 && l.importance() != MIN_IMPORTANCE)
            throw (E_UNSPECIFIC, "Internal: First must be MIN_IMPORTANCE");
        if (i == _LogLevelObjs_sz - 1 && l.importance() != MAX_IMPORTANCE)
            throw (E_UNSPECIFIC, "Internal: First must be MIN_IMPORTANCE");
        if (i > 0 && l.importance() <= _LogLevelObjs[i-1].importance())
            throw (E_UNSPECIFIC, "Internal: Importances not in strict ascending order");
        if (names.includes(l.name()) ||
            fixnames.includes(l.fixlen_name()) ||
            shortnames.includes(l.short_name()))
        {
            throw (E_UNSPECIFIC, "Internal: Repeated name, fixlen_name or short_name");
        }

        names.add(l.name());
        fixnames.add(l.fixlen_name());
        shortnames.add(l.short_name());

        if (i == 0 || i == _LogLevelObjs_sz-1) {
            if (!l.is_special()) {
                throw (E_UNSPECIFIC, "Internal: First and last LogLevelObj must be special");
            }
        } else {
            if (l.is_special()) {
                throw (E_UNSPECIFIC, "Internal: Only first and last LogLevelObj may be special");
            }
        }
    }
}

vector(LogLevelObj) levels()
{
    init_loglevels();
    return clone_vector(_LogLevelObjs);
}

} // End module __impl__

LogLevelObj to_obj(LogLevel e)
    option(category: "Logging/Logging")
{
    __impl__.init_loglevels();
    switch (e) {
    case EVERYTHING: return __impl__._EVERYTHING;
    case DEBUG: return __impl__._DEBUG;
    case INFO: return __impl__._INFO;
    case WARNING: return __impl__._WARNING;
    case ERROR: return __impl__._ERROR;
    case FATAL: return __impl__._FATAL;
    case NOTHING: return __impl__._NOTHING;
    default: throw (E_UNSPECIFIC, "Internal: Unhandled enum in to_obj");
    }
}

vector(LogLevel) get_levels() = __impl__.levels().get_enum();

integer num_log_levels() = v_size(__impl__.levels());

LogLevelObj str_to_log_level_obj(string level_name)
{
    vector(LogLevelObj) levels = __impl__.levels();
    integer n = v_size(levels);
    for (integer i = 0; i < n; ++i) {
        LogLevelObj l = levels[i];
        if (equal_casei(l.name(), level_name)) {
            return l;
        }
    }

    throw (E_UNSPECIFIC, strcat(["The string ", level_name, " is no LogLevelObj"]));
}

LogLevel str_to_log_level(string level_name) =
    str_to_log_level_obj(level_name).get_enum();



// *** CONTEXT ***

// A context is an object of its own to avoid unintentional use, e.g. by
// forgetting [ ] in a call:
// logging.info("User ID: ", uid);   <- context = "User ID:",  message = <uid>
// probably wanted this:
// logging.info(["User ID: ", uid]);   <- context = null, message = "User ID: <uid>"

module __impl__ {
    string _MasterContext option(constant) = ")M[";
}

class Context {
public:
    Context(string name);
    string name();

private:
    string name_;

    void __dbg_print(__dbg_label l);
};
Context.Context(string name)
    : name_(name)
{
    if (name_ == "") {
        throw (E_UNSPECIFIC, "A context can not have an empty name");
    }
    if (name_ == __impl__._MasterContext) {
        throw (E_UNSPECIFIC, strcat(
            ["'",__impl__._MasterContext,
             "' happens to be a reserved name and cannot be the name of a Context."]));
    }
}
string Context.name() = name_;
void Context.__dbg_print(__dbg_label l) { l.set_text(strcat(["Context { ", name_, " }"])); }


// *** UNIT TO LOG ***
class AbstractLogUnit option(category: "Logging/Logging") {
public:
    AbstractLogUnit(Context option(nullable) context, integer pos);
    string context();
	integer position();
    Context context_obj();

    virtual logical is_newline() = 0;

protected:
    Context context_;
	integer pos_;
};
AbstractLogUnit.AbstractLogUnit(Context option(nullable) context, integer pos)
: context_(context),
  pos_(pos)
{}

string AbstractLogUnit.context() = context_.name();

integer AbstractLogUnit.position() = pos_;

Context AbstractLogUnit.context_obj() = context_;

class LogUnit : public AbstractLogUnit option(category: "Logging/Logging") {
public:
    LogUnit(vector(string) msg_parts, timestamp time, LogLevelObj l, Context option(nullable) context, integer pos);
    vector(string) message_parts();
    string message();
    timestamp time();
    LogLevelObj level();

    override logical is_newline();

private:
    vector(string) msg_parts_;
    timestamp time_;
    LogLevelObj level_;

    void __dbg_print(__dbg_label l);
};
LogUnit.LogUnit(vector(string) msg_parts, timestamp time, LogLevelObj l, Context option(nullable) context, integer pos)
: AbstractLogUnit(context, pos), msg_parts_(msg_parts), time_(time), level_(l)
{}
void LogUnit.__dbg_print(__dbg_label l)
{
    if (null(context_)) {
        l.set_text(strcat(["LogUnit { ", str(time_), " ", level_.fixlen_name(), " ", this.message(), " }"]));
    } else {
        l.set_text(strcat(["LogUnit { ", str(time_), " <", context_.name(), "> ", level_.fixlen_name(), " ", this.message(), " }"]));
    }
}
vector(string) LogUnit.message_parts() { return msg_parts_; }
string LogUnit.message() = strcat(msg_parts_);
timestamp LogUnit.time() = time_;
LogLevelObj LogUnit.level() = level_;
logical LogUnit.is_newline() = false;


class NewlineUnit : public AbstractLogUnit option(category: "Logging/Logging") {
public:
    NewlineUnit(integer n_newlines, Context option(nullable) context, integer pos);

    integer n_newlines();
    override logical is_newline();

private:
    integer n_newlines_;

    void __dbg_print(__dbg_label l);
};

NewlineUnit.NewlineUnit(integer n_newlines, Context option(nullable) context, integer pos)
: AbstractLogUnit(context, pos), n_newlines_(n_newlines)
{}

integer NewlineUnit.n_newlines() = n_newlines_;

logical NewlineUnit.is_newline() = true;

void  NewlineUnit.__dbg_print(__dbg_label l)
{
    l.set_text(strcat(["NewlineUnit { n=", str(n_newlines_), " }"]));
}



// *** Counter of Log Messages, split on level and context ***
class LogCount option(category: "Logging/Logging") {
public:
    LogCount();

    /* Any Context */
    integer get(); // Grand total
    integer get(LogLevel level); // Only on this level
    // Between these levels (inclusive)
    integer get(LogLevel from_level, LogLevel to_level);

    /* One Context (null = master context) */
    integer get(Context option(nullable) ctxt); // Grand total for one Context
    integer get(Context option(nullable) ctxt, LogLevel level); // Only on this level
    // Between these levels (inclusive)
    integer get(Context option(nullable) ctxt, LogLevel from_level, LogLevel to_level);

    vector(string) context_names(); // Contexts with any logging

    void add(Context option(nullable) ctxt, LogLevel level);

private:
    vector(string) contexts_;
    vector(map_int_int) level_counts_;
};

LogCount.LogCount()
: contexts_([__impl__._MasterContext]), level_counts_([map_int_int()])
{}

integer LogCount.get()
{
    integer total = 0;
    integer contexts__sz = v_size(contexts_);
    for (integer i = 0; i < contexts__sz; ++i) {
        map_int_int map = level_counts_[i];
        vector(integer) importances = map.get_keys();
        integer importances_sz = v_size(importances);
        for (integer j = 0; j < importances_sz; ++j) {
            integer count = map.find(importances[j]);
            if (count > 0)
                total += count;
        }
    }

    return total;
}

integer LogCount.get(LogLevel level)
{
    integer total = 0;
    integer contexts__sz = v_size(contexts_);
    for (integer i = 0; i < contexts__sz; ++i) {
        map_int_int map = level_counts_[i];
        integer count = map.find(to_obj(level).importance());
        if (count > 0)
            total += count;
    }

    return total;
}

integer LogCount.get(LogLevel from_level, LogLevel to_level)
{
    LogLevelObj from = to_obj(from_level);
    LogLevelObj to = to_obj(to_level);
    if (from.importance() > to.importance())
        throw (E_UNSPECIFIC, "Did you mixup from_level and to_level (from_level must be <= to_level)?");
    integer total = 0;
    integer contexts__sz = v_size(contexts_);
    for (integer i = 0; i < contexts__sz; ++i) {
        map_int_int map = level_counts_[i];
        vector(integer) importances = map.get_keys();
        integer importances_sz = v_size(importances);
        for (integer j = 0; j < importances_sz; ++j) {
            if (importances[j] < from.importance())
                continue;
            if (importances[j] > to.importance())
                break;

            integer count = map.find(importances[j]);
            if (count > 0)
                total += count;
        }
    }

    return total;
}

integer LogCount.get(Context option(nullable) ctxt)
{
    string ctxt_name = null(ctxt) ? __impl__._MasterContext : ctxt.name();
    integer total = 0;
    integer contexts__sz = v_size(contexts_);
    for (integer i = 0; i < contexts__sz; ++i) {
        if (contexts_[i] != ctxt_name)
            continue;
        map_int_int map = level_counts_[i];
        vector(integer) importances = map.get_keys();
        integer importances_sz = v_size(importances);
        for (integer j = 0; j < importances_sz; ++j) {
            integer count = map.find(importances[j]);
            if (count > 0)
                total += count;
        }
        break;
    }

    return total;
}

integer LogCount.get(Context option(nullable) ctxt, LogLevel level)
{
    string ctxt_name = null(ctxt) ? __impl__._MasterContext : ctxt.name();
    integer total = 0;
    integer contexts__sz = v_size(contexts_);
    for (integer i = 0; i < contexts__sz; ++i) {
        if (contexts_[i] != ctxt_name)
            continue;
        map_int_int map = level_counts_[i];
        integer count = map.find(to_obj(level).importance());
        if (count > 0)
            total += count;
        break;
    }

    return total;
}

integer LogCount.get(Context option(nullable) ctxt, LogLevel from_level, LogLevel to_level)
{
    LogLevelObj from = to_obj(from_level);
    LogLevelObj to = to_obj(to_level);
    if (from.importance() > to.importance())
        throw (E_UNSPECIFIC, "Did you mixup from_level and to_level (from_level must be <= to_level)?");
    string ctxt_name = null(ctxt) ? __impl__._MasterContext : ctxt.name();
    integer total = 0;
    integer contexts__sz = v_size(contexts_);
    for (integer i = 0; i < contexts__sz; ++i) {
        if (contexts_[i] != ctxt_name)
            continue;
        map_int_int map = level_counts_[i];
        vector(integer) importances = map.get_keys();
        integer importances_sz = v_size(importances);
        for (integer j = 0; j < importances_sz; ++j) {
            if (importances[j] < from.importance())
                continue;
            if (importances[j] > to.importance())
                break;

            integer count = map.find(importances[j]);
            if (count > 0)
                total += count;
        }
        break;
    }

    return total;
}

vector(string) LogCount.context_names()
{
    if (v_size(contexts_) == 1) {
        vector(string) v[0];
        return v;
    } else {
        return contexts_[1:];
    }
}

void LogCount.add(Context option(nullable) ctxt, LogLevel level)
{
    map_int_int map;
    string ctxt_name = null(ctxt) ? __impl__._MasterContext : ctxt.name();
    integer contexts__sz = v_size(contexts_);
    for (integer i = 0; i < contexts__sz; ++i) {
        if (contexts_[i] == ctxt_name) {
            map = level_counts_[i];
            break;
        }
    }

    if (null(map)) {
        map = map_int_int();
        push_back(contexts_, ctxt_name);
        push_back(level_counts_, map);
    }

    LogLevelObj l = to_obj(level);
    integer current_count = map.find(l.importance());
    if (current_count == -1)
        current_count = 0;

    map.add(l.importance(), current_count+1);
}


// *** Display requesting ***
enum display_request { DR_DEFAULT, DR_SHOW, DR_HIDE };
class DisplayRequestSet {
public:
        DisplayRequestSet();
        void show_timestamp();
        void hide_timestamp();
        void show_level();
        void hide_level();
        void show_context();
        void hide_context();
        void show_message();
        void hide_message();
        display_request timestamp_request();
        display_request level_request();
        display_request context_request();
        display_request message_request();

private:
        display_request dr_timestamp_;
        display_request dr_level_;
        display_request dr_context_;
        display_request dr_message_;
};
DisplayRequestSet.DisplayRequestSet()
        : dr_timestamp_(DR_DEFAULT), dr_level_(DR_DEFAULT), dr_context_(DR_DEFAULT), dr_message_(DR_DEFAULT)
{}
void DisplayRequestSet.show_timestamp() { dr_timestamp_ = DR_SHOW; }
void DisplayRequestSet.hide_timestamp() { dr_timestamp_ = DR_HIDE; }
void DisplayRequestSet.show_level() { dr_level_ = DR_SHOW; }
void DisplayRequestSet.hide_level() { dr_level_ = DR_HIDE; }
void DisplayRequestSet.show_context() { dr_context_ = DR_SHOW; }
void DisplayRequestSet.hide_context() { dr_context_ = DR_HIDE; }
void DisplayRequestSet.show_message() { dr_message_ = DR_SHOW; }
void DisplayRequestSet.hide_message() { dr_message_ = DR_HIDE; }
display_request DisplayRequestSet.timestamp_request() = dr_timestamp_;
display_request DisplayRequestSet.level_request() = dr_level_;
display_request DisplayRequestSet.context_request() = dr_context_;
display_request DisplayRequestSet.message_request() = dr_message_;



// *** Colorizer ***
/* Used by Logger implementations to decide color from LogLevelObj. Using this
 * makes it easy to let the user specify his own color scheme, and the Logger
 * class will handle user added LogLevelObjs without problem.
 *
 * See the TableLogger class for how to use the Colorizer
 */

class Colorizer {
public:
    Colorizer();
    Colorizer(vector(integer) thresholds, vector(integer) colors);
    integer color(LogLevel level);
    integer color(LogLevelObj level);

private:
    vector(integer) color_thresholds_;
    vector(integer) colors_;

    void _init(vector(integer) thresholds, vector(integer) colors);
};

Colorizer.Colorizer()
{
    // The default color scheme
    _init(
        [0, 15, 25, 35, 45],
        [rgb(230, 230, 230), // gray for debug (10)
         rgb(255, 255, 255), // white for info (20)
         rgb(255, 250, 190), // pale yellow for warning (30)
         rgb(255, 180, 180), // pale red for error (40)
         rgb(235, 30, 20)]); // deep red for fatal (50)
}

Colorizer.Colorizer(vector(integer) thresholds, vector(integer) colors)
{
    _init(thresholds, colors);
}

integer Colorizer.color(LogLevel level) =
    this.color(to_obj(level));

integer Colorizer.color(LogLevelObj level)
{
    integer color = -1;
    integer color_thresholds__sz = v_size(color_thresholds_);
    integer importance = level.importance();
    for (integer i = 0; i < color_thresholds__sz; ++i) {
        if (importance >= color_thresholds_[i]) {
            color = colors_[i];
        } else {
            break;
        }
    }
    return color;
}

void Colorizer._init(vector(integer) thresholds, vector(integer) colors)
{
    integer n = v_size(thresholds);
    if (v_size(colors) != n) {
        throw (E_UNSPECIFIC, "Mismatching vector sizes");
    }
    sort(colors, thresholds);
    sort(thresholds);
    color_thresholds_ = clone_vector(thresholds);
    colors_ = clone_vector(colors);
}




class StatusTracker {
public:
    StatusTracker();

    // set_status and clear_status returns previous status
    // If previous status was the same as given status, null is returned
    // If previous status was null, empty string is returned
    // Confusing, isn't it?
    // The point is to use a simple null check to see if status was changed,
    // and let a change be a bit more complicated to handle
    // With this in mind, get_status will never return null.

    string set_status(Context option(nullable) ctxt, vector(string) key_parts, string status);
    string clear_status(Context option(nullable) ctxt, vector(string) key_parts);
    string get_status(Context option(nullable) ctxt, vector(string) key_parts);

    void data_count(out integer n, out integer key_strlen, out integer value_strlen);

private:
    map_str_str statuses_;

    string _key_str(Context option(nullable) ctxt, vector(string) key_parts);
};

StatusTracker.StatusTracker() : statuses_(map_str_str())
{}

string StatusTracker._key_str(Context option(nullable) ctxt, vector(string) key_parts)
{
    if (null(ctxt)) {
        return strcat("$", str_join(key_parts, "|"));
    } else {
        return strcat([ctxt.name(), "$", str_join(key_parts, "|")]);
    }
}

string StatusTracker.set_status(Context option(nullable) ctxt, vector(string) key_parts, string status)
{
    string key_str = _key_str(ctxt, key_parts);

    string old_status = statuses_.find(key_str);
    if (null(old_status)) {
        old_status = "";
    }

    if (status == old_status) {
        return null<string>;
    }

    if (status == "") {
        statuses_.remove(key_str);
    } else {
        statuses_.add(key_str, status);
    }

    return old_status;
}

string StatusTracker.clear_status(Context option(nullable) ctxt, vector(string) key_parts)
{
    return this.set_status(ctxt, key_parts, "");
}

string StatusTracker.get_status(Context option(nullable) ctxt, vector(string) key_parts)
{
    string key_str = _key_str(ctxt, key_parts);
    string status = statuses_.find(key_str);
    return null(status) ? "" : status;
}

void StatusTracker.data_count(out integer n, out integer key_strlen, out integer value_strlen)
{
    vector(string) keys, values;
    statuses_.get_content(keys, values);
    n = v_size(keys);
    key_strlen = integer(v_sum(strlen(keys)));
    value_strlen = integer(v_sum(strlen(values)));
}

// *** LOGGER BASE CLASS ***

// Instruction for subclasses to Logger:
//
// Must define the function
// write_impl(LogUnit)
// to do the actual logging.
//
// To adhere calls for newline, redefine
// write_newline_impl(integer)
// which by default is a no-op. The argument guaranteed to be a positive integer.
//
// To respect settings for what elements (timestamp, level, context, message) to log,
// add those applicable of the following as public functions:
//  void show_timestamp() { _show_timestamp(); }
//  void hide_timestamp() { _hide_timestamp(); }
//  void show_level() { _show_level(); }
//  void hide_level() { _hide_level(); }
//  void show_context() { _show_context(); }
//  void hide_context() { _hide_context(); }
//  void show_message() { _show_message(); }
//  void hide_message() { _hide_message(); }
// to allow users to show/hide the element.
// Call applicable of these functions in write_impl to check user setting:
//  display_request timestamp_request()
//  display_request level_request()
//  display_request context_request()
//  display_request message_request()
// where display_request is an enum with the values DR_SHOW, DR_HIDE and DR_DEFAULT.
// Returns DR_DEFAULT if user hasn't called show or hide, otherwise whatever was
// called last.

class Logger option(category: "Logging/Logging") {
public:
    void set_level(LogLevel level); // Will log calls of importance >= level.importance()
    LogLevel get_level();
    integer get_level_importance();

    // Will handle log calls of level "from" as log calls of level "to"
    // Possible usage: lib logs "error", but to you it's not that bad, only a "warning"
    void set_level_translation(LogLevel from, LogLevel to);

    logical uses_utc();
    void set_use_utc(logical use_utc);
    void set_use_utc_if_null(logical use_utc);

    // Called by logging module itself
    void write_newline(integer n_newlines = 1);
    void log(
        Context option(nullable) context,
        vector(string) msg_parts,
        LogLevel level,
        timestamp option(nullable) time,
        integer pos);

    // Not used by logging module, provided for those who want to
    // log directly to a logger object
    void log(AbstractLogUnit lu);

    void debug(string msg, integer pos = $pos);
    void info(string msg, integer pos = $pos);
    void warning(string msg, integer pos = $pos);
    void error(string msg, integer pos = $pos);
    void fatal(string msg, integer pos = $pos);

    void debug(Context option(nullable) context, string msg, integer pos = $pos);
    void info(Context option(nullable) context, string msg, integer pos = $pos);
    void warning(Context option(nullable) context, string msg, integer pos = $pos);
    void error(Context option(nullable) context, string msg, integer pos = $pos);
    void fatal(Context option(nullable) context, string msg, integer pos = $pos);

    void debug(vector(string) msg_parts, integer pos = $pos);
    void info(vector(string) msg_parts, integer pos = $pos);
    void warning(vector(string) msg_parts, integer pos = $pos);
    void error(vector(string) msg_parts, integer pos = $pos);
    void fatal(vector(string) msg_parts, integer pos = $pos);

    void debug(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos);
    void info(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos);
    void warning(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos);
    void error(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos);
    void fatal(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos);

protected:
    virtual LogLevel default_level(); // Subclasses can have different default level by implementing default_level() as e.g "return WARNING;"
    virtual void write_impl(LogUnit lu) = 0; // Subclasses must define how to log
    virtual void write_newline_impl(integer n_newlines); // Subclasses can handle newlines (ignored by default)

    void _show_timestamp();
    void _hide_timestamp();
    void _show_level();
    void _hide_level();
    void _show_context();
    void _hide_context();
    void _show_message();
    void _hide_message();
    display_request _timestamp_request();
    display_request _level_request();
    display_request _context_request();
    display_request _message_request();

    LogLevelObj min_level_;

private:
    void _level_log(Context option(nullable) context, vector(string) msg_parts, LogLevel l, timestamp time, integer pos);
    LogLevelObj _translate_level(LogLevelObj l);
    LogLevelObj _get_level();
    logical translation_exists_;
    vector(LogLevelObj) level_translations_;
    DisplayRequestSet display_request_set_;
    DisplayRequestSet _display_request_set(); // Creates display_request_set_ upon first call
    logical use_utc_;
};

void Logger.set_level(LogLevel l) { min_level_ = to_obj(l); }
LogLevel Logger.get_level() = _get_level().get_enum();
integer Logger.get_level_importance() = _get_level().importance();

void Logger.set_level_translation(LogLevel from, LogLevel to)
{
    LogLevelObj from_level = to_obj(from);
    LogLevelObj to_level = to_obj(to);
    if (from_level.is_special() || to_level.is_special()) {
        throw (E_CONSTRAINT, "Translation between special levels not allowed");
    }
    if (null(level_translations_)) {
        resize(level_translations_, num_log_levels());
        translation_exists_ = true;
    }
    level_translations_[from_level.index()] = to_level;
}

LogLevelObj Logger._get_level()
{
    if (null(min_level_)) {
        min_level_ = to_obj(default_level());
    }
    return min_level_;
}

logical Logger.uses_utc()
{
    return !null(use_utc_) && use_utc_;
}

void Logger.set_use_utc(logical use_utc)
{
    use_utc_ = use_utc;
}

void Logger.set_use_utc_if_null(logical use_utc)
{
    if (null(use_utc_))
        use_utc_ = use_utc;
}

void Logger.write_newline(integer n_newlines) {
    if (n_newlines < 1) {
        return;
    }
    this.write_newline_impl(n_newlines);
}

void Logger.log(
    Context option(nullable) context,
    vector(string) msg_parts,
    LogLevel level,
    timestamp option(nullable) time,
    integer pos)
{
    if (null(time))
        time = this.uses_utc() ? now_ut() : now();
    _level_log(context, msg_parts, level, time, pos);
}

void Logger.log(AbstractLogUnit alu)
{
    if (alu.is_newline()) {
        this.write_newline(dynamic_cast<NewlineUnit>(alu).n_newlines());
    } else {
        LogUnit lu = dynamic_cast<LogUnit>(alu);
        this.log(lu.context_obj(), lu.message_parts(), lu.level().get_enum(), lu.time(), lu.position());
    }
}

void Logger.debug(string msg, integer pos) { this.log(null<Context>, [msg], DEBUG, null<timestamp>, pos); }
void Logger.info(string msg, integer pos) { this.log(null<Context>, [msg], INFO, null<timestamp>, pos); }
void Logger.warning(string msg, integer pos) { this.log(null<Context>, [msg], WARNING, null<timestamp>, pos); }
void Logger.error(string msg, integer pos) { this.log(null<Context>, [msg], ERROR, null<timestamp>, pos); }
void Logger.fatal(string msg, integer pos) { this.log(null<Context>, [msg], FATAL, null<timestamp>, pos); }

void Logger.debug(Context option(nullable) context, string msg, integer pos) { this.log(context, [msg], DEBUG, null<timestamp>, pos); }
void Logger.info(Context option(nullable) context, string msg, integer pos) { this.log(context, [msg], INFO, null<timestamp>, pos); }
void Logger.warning(Context option(nullable) context, string msg, integer pos) { this.log(context, [msg], WARNING, null<timestamp>, pos); }
void Logger.error(Context option(nullable) context, string msg, integer pos) { this.log(context, [msg], ERROR, null<timestamp>, pos); }
void Logger.fatal(Context option(nullable) context, string msg, integer pos) { this.log(context, [msg], FATAL, null<timestamp>, pos); }

void Logger.debug(vector(string) msg_parts, integer pos) { this.log(null<Context>, msg_parts, DEBUG, null<timestamp>, pos); }
void Logger.info(vector(string) msg_parts, integer pos) { this.log(null<Context>, msg_parts, INFO, null<timestamp>, pos); }
void Logger.warning(vector(string) msg_parts, integer pos) { this.log(null<Context>, msg_parts, WARNING, null<timestamp>, pos); }
void Logger.error(vector(string) msg_parts, integer pos) { this.log(null<Context>, msg_parts, ERROR, null<timestamp>, pos); }
void Logger.fatal(vector(string) msg_parts, integer pos) { this.log(null<Context>, msg_parts, FATAL, null<timestamp>, pos); }

void Logger.debug(Context option(nullable) context, vector(string) msg_parts, integer pos) { this.log(context, msg_parts, DEBUG, null<timestamp>, pos); }
void Logger.info(Context option(nullable) context, vector(string) msg_parts, integer pos) { this.log(context, msg_parts, INFO, null<timestamp>, pos); }
void Logger.warning(Context option(nullable) context, vector(string) msg_parts, integer pos) { this.log(context, msg_parts, WARNING, null<timestamp>, pos); }
void Logger.error(Context option(nullable) context, vector(string) msg_parts, integer pos) { this.log(context, msg_parts, ERROR, null<timestamp>, pos); }
void Logger.fatal(Context option(nullable) context, vector(string) msg_parts, integer pos) { this.log(context, msg_parts, FATAL, null<timestamp>, pos); }

LogLevelObj Logger._translate_level(LogLevelObj l)
{
    if (null(translation_exists_) || !translation_exists_) {
        return l;
    }
    LogLevelObj new_level = level_translations_[l.index()];
    return null(new_level) ? l : new_level;
}

void Logger._level_log(Context option(nullable) context, vector(string) msg_parts, LogLevel l, timestamp time, integer pos) {
    LogLevelObj level = to_obj(l);
    if (level.is_special()) {
        throw (E_CONSTRAINT, "Internal: tried to log with special level");
    }
    level = _translate_level(level);
    if (level.importance() >= this.get_level_importance()) {
        write_impl(new LogUnit(msg_parts, time, level, context, pos));
    }
}
// Default newline implementation
void Logger.write_newline_impl(integer n_newlines) {/* no op */}
// Default logging level
LogLevel Logger.default_level() = INFO;

// Display requests
void Logger._show_timestamp() { _display_request_set().show_timestamp(); }
void Logger._hide_timestamp() { _display_request_set().hide_timestamp(); }
void Logger._show_level() { _display_request_set().show_level(); }
void Logger._hide_level() { _display_request_set().hide_level(); }
void Logger._show_context() { _display_request_set().show_context(); }
void Logger._hide_context() { _display_request_set().hide_context(); }
void Logger._show_message() { _display_request_set().show_message(); }
void Logger._hide_message() { _display_request_set().hide_message(); }
display_request Logger._timestamp_request() { return _display_request_set().timestamp_request(); }
display_request Logger._level_request() { return _display_request_set().level_request(); }
display_request Logger._context_request() { return _display_request_set().context_request(); }
display_request Logger._message_request() { return _display_request_set().message_request(); }
DisplayRequestSet Logger._display_request_set()
{
    if (null(display_request_set_)) {
        display_request_set_ = new DisplayRequestSet();
    }
    return display_request_set_;
}




// *** MODULE KEEPING TRACK OF LOGGERS IN APP ***
module __impl__ {
    // All loggers in a context can be found in the chain of that context.
    // The loggers are stored in _Loggers, and for every logger, the next
    // logger in the chain is found by looking at the corresponding index
    // in _LoggerLinks. If there are no more loggers in the chain, -1 will
    // be found there. _Loggers[0] is the first logger of the master context,
    // the first logger of other contexts are stored in _Context2Index.

    vector(logging.Logger)  _Loggers;
    // Links the logger chain. _Loggers[i] is followed by _Loggers[_LoggerLinks[i]].
    // _LoggerLinks[i] == -1 means chain ends here.
    integer                 _NoLink option(constant) = -1;
    logging.Logger          _NoLogger = null<logging.Logger>;
    vector(integer)         _LoggerLinks;
    // Maps from a context to the first logger in the logger chain
    map_str_int             _Context2Index = map_str_int();
    // Map from a context to the associated level
    map_str_int             _Context2Level = map_str_int();
    // Logging counter, activated by user
    LogCount                _LogCounter = null<LogCount>;
    // status Tracker, used by status_info, status_clear etc
    StatusTracker            _StatusTracker = new StatusTracker();

    // If first master doesn't exist, we will act differently in some cases.
    // It's not beautiful to handle it with this flag, but it'll have to do.
    logical                 _FirstMasterExists = false;

    logical                 _UseUTC = false;

    void initialize()
    {
        // Clear, and reserve first position for first master logger
        _Loggers = [_NoLogger];
        _LoggerLinks = [_NoLink];
        _Context2Index = map_str_int();
        _Context2Level = map_str_int();
        _LogCounter = null<LogCount>;
        _StatusTracker = new StatusTracker();
        _FirstMasterExists = false;
        _UseUTC = false;
    }

    void maybe_initialize()
    {
        if (null(_Loggers)) {
            initialize();
        }
    }

    logical uses_utc() = _UseUTC;
    void set_use_utc(logical use_utc)
    {
        maybe_initialize();
        _UseUTC = use_utc;
    }

    // Returns link index of last logger in chain
    integer follow_chain(integer from_index)
    {
        integer ix = from_index;
        while (_LoggerLinks[ix] != _NoLink) {
            ix = _LoggerLinks[ix];
        }
        return ix;
    }

    vector(integer) get_chain(integer from_index)
    {
        integer ix = from_index;
        vector(integer) ixes = [ix];
        while (_LoggerLinks[ix] != _NoLink) {
            ix = _LoggerLinks[ix];
            push_back(ixes, ix);
        }
        return ixes;
    }

    void delete_chain(integer from_index)
    {
        vector(integer) chain = get_chain(from_index);
        integer chain_sz = v_size(chain);
        for (integer i = 0; i < chain_sz; ++i) {
            _Loggers[chain[i]] = _NoLogger;
            _LoggerLinks[chain[i]] = _NoLink;
        }
    }

    integer add_logger(logging.Logger l, integer chain_index)
    {
        l.set_use_utc_if_null(uses_utc());

        // Put logger last in vector
        integer ix = v_size(_Loggers);
        push_back(_Loggers, l);
        push_back(_LoggerLinks, _NoLink);

        // Find end of chain and link to new logger
        if (chain_index != _NoLink) {
            integer last_link_index = follow_chain(chain_index);
            _LoggerLinks[last_link_index] = ix;
        }
        return ix;
    }

    void set_logger(logging.Logger l)
    {
        maybe_initialize();
        delete_chain(0);
        l.set_use_utc_if_null(uses_utc());
        _Loggers[0] = l;
        _FirstMasterExists = true;
    }

    logging.Logger get_logger()
    {
        maybe_initialize();
        return _Loggers[0];
    }

    void set_logger(logging.Context context, logging.Logger l)
    {
        maybe_initialize();

        integer ix = _Context2Index.find(context.name());
        if (ix != _NoLink) {
            // Start new chain in place of old chain
            delete_chain(ix);
            _Loggers[ix] = l;
        } else {
            // Start a brand new chain
            ix = add_logger(l, _NoLink);
            _Context2Index.add(context.name(), ix);
        }
    }

    logging.Logger get_logger(logging.Context option(nullable) context, logical master_fallback)
    {
        logical use_master = null(context);
        if (!use_master) {
            integer ix = _Context2Index.find(context.name());
            if (ix != _NoLink) {
                return _Loggers[ix];
            }
            use_master = master_fallback;
        }
        if (use_master) {
            return get_logger();
        }
        return null<logging.Logger>;
    }


    // Logger chains
    void add_logger(logging.Logger l)
    {
        maybe_initialize();

        if (!_FirstMasterExists) {
            set_logger(l);
        } else {
            add_logger(l, 0);
        }
    }

    void add_logger(logging.Context context, logging.Logger l)
    {
        maybe_initialize();

        integer ix = _Context2Index.find(context.name());
        if (ix != _NoLink) {
            add_logger(l, ix);
        } else {
            ix = add_logger(l, _NoLink);
            _Context2Index.add(context.name(), ix);
        }
    }

    vector(logging.Logger) get_loggers()
    {
        maybe_initialize();

        if (!_FirstMasterExists) {
            vector(logging.Logger) empty[0];
            return empty;
        }
        vector(integer) ixes = get_chain(0);
        return _Loggers[ixes];
    }

    vector(logging.Logger) get_loggers(logging.Context context, logical master_fallback)
    {
        integer ix = _Context2Index.find(context.name());
        if (ix != _NoLink) {
            vector(integer) ixes = get_chain(ix);
            return _Loggers[ixes];
        } else {
            if (master_fallback) {
                return get_loggers();
            } else {
                vector(logging.Logger) empty[0];
                return empty;
            }
        }
    }

    integer lowest_importance(vector(Logger) option(nullable) loggers)
    {
        vector(integer) importances = loggers.get_level_importance();
        if (v_size(importances) == 0) {
            return logging.MAX_IMPORTANCE;
        } else {
            return integer(v_min(importances));
        }
    }

    integer lowest_importance() = lowest_importance(get_loggers());
    integer lowest_importance(logging.Context option(nullable) context)
    {
        if (null(context)) {
            return lowest_importance(get_loggers());
        } else {
            return lowest_importance(get_loggers(context, true));
        }
    }

    logical any_would_log(LogLevel level, vector(Logger) loggers)
    {
        integer level_importance = to_obj(level).importance();
        integer loggers_sz = v_size(loggers);
        for (integer i = 0; i < loggers_sz; ++i) {
            if (loggers[i].get_level_importance() <= level_importance) {
                return true;
            }
        }
        return false;
    }


    // Context levels
    void set_context_level(string context, LogLevel level)
    {
        maybe_initialize();
        _Context2Level.add(context, to_obj(level).importance());
    }

    integer get_context_level(string context, logical master_fallback)
    {
        maybe_initialize();
        integer level = _Context2Level.find(context); // Will return -1 for unset context
        if (level == -1) {
            if (master_fallback) {
                return _Context2Level.find(_MasterContext);
            } else {
                return -1;
            }
        } else {
            return level;
        }
    }

    // Returns the loggers only if level is at least context level

    vector(Logger) get_loggers_if_context_level(LogLevel l)
    {
        maybe_initialize();
        integer master_level = get_context_level(_MasterContext, false);
        if (to_obj(l).importance() < master_level) {
            vector(Logger) v[0];
            return v;
        } else {
            return get_loggers();
        }
    }

    vector(Logger) get_loggers_if_context_level(
        Context option(nullable) context, LogLevel l, logical master_fallback)
    {
        maybe_initialize();
        if (null(context)) {
            return get_loggers_if_context_level(l); // null context means master logger
        }
        integer context_level = get_context_level(context.name(), master_fallback);
        if (to_obj(l).importance() < context_level) {
            vector(Logger) v[0];
            return v;
        } else {
            return get_loggers(context, master_fallback);
        }
    }


    logical any_would_log(LogLevel level, Context option(nullable) context)
    {
        return any_would_log(level, get_loggers_if_context_level(context, level, true));
    }


    // Single point of entry for all logging to the __impl__ module
    // Fixates timestamp for logging.
    void log(Context option(nullable) ctxt, vector(string) msg_parts, LogLevel level, integer pos, timestamp option(nullable) time = null<timestamp>)
    {
        if (!null(_LogCounter))
            _LogCounter.add(ctxt, level);

        vector(Logger) loggers = get_loggers_if_context_level(ctxt, level, true);

        integer n = v_size(loggers);
        if (n == 0)
            return;

        timestamp utc_time = null(time) ? now_ut() : time;
        timestamp local_time = null(time) ? now() : time;

        for (integer i = 0; i < n; ++i) {
            Logger logger = loggers[i];
            if (logger.uses_utc()) {
                logger.log(ctxt, msg_parts, level, utc_time, pos);
            } else {
                logger.log(ctxt, msg_parts, level, local_time, pos);
            }
        }
    }


    LogCount get_log_counter()
    {
        maybe_initialize();
        return _LogCounter;
    }

    void delete_log_counter()
    {
        maybe_initialize();
        _LogCounter = null<LogCount>;
    }

    void create_log_counter()
    {
        maybe_initialize();
        _LogCounter = new LogCount();
    }
}
}

module logging {

/*
 * By default, logging uses system time, as given by now().
 *
 * Individual loggers can use UTC time given by now_ut() by calling the method
 * set_use_utc(true).
 *
 * If logging.use_utc() is called, all loggers added henceforth will use UTC,
 * unless they have been explicitly set to not use UTC.
 *
 * Note that some loggers, such as QlbLogger do not use the time provided by
 * logging module, and will not be affected by UTC setting (they may or may not
 * use UTC themselves).
 */

public void use_utc()
{
    __impl__.set_use_utc(true);
}



/* All function should be in one alternative without context and one with (as
 * first argument):
 *   public <return_type> <function_name>(<arguments>) option(category: "Logging/Control Logging") { <body> }
 *   public <return_type> <function_name>(string option(nullable) context, <arguments>) option(category: "Logging/Control Logging") { <body> }
 */

// Master logger
public void set_logger(Logger logger) option(category: "Logging/Control Logging") { __impl__.set_logger(logger); }
public Logger get_logger() option(category: "Logging/Control Logging") { return __impl__.get_logger(); }
public void add_logger(Logger logger) option(category: "Logging/Control Logging") { __impl__.add_logger(logger); }
public vector(Logger) get_loggers() option(category: "Logging/Control Logging") { return __impl__.get_loggers(); }

// Context loggers (returns Master logger if no context logger exists)
public void set_logger(Context context, Logger logger) option(category: "Logging/Control Logging") { __impl__.set_logger(context, logger); }
public Logger get_logger(Context option(nullable) context) option(category: "Logging/Control Logging") { return __impl__.get_logger(context, true); }
public void add_logger(Context context, Logger logger) option(category: "Logging/Control Logging") { __impl__.add_logger(context, logger); }
public vector(Logger) get_loggers(Context option(nullable) context) option(category: "Logging/Control Logging") { return __impl__.get_loggers(context, true); }

// Control Logging
/* Setting level like this is no longer the recommended way. It has the
 * maybe surprising behaviour of only setting the first logger, and not
 * affecting future loggers on the context. To remedy this, new functions are
 * introduced:
 *     set_master_level: Set the minimum level for all logging
 *     set_context_level: Set the minimum level for a particular context, which
 *                        is then no longer affected by master level.
 * In the future set_level may either:
 * (1) Change behaviour to same as set_master_level and set_context_level
 * (2) Be removed
 * (1) would make the cleanest API, but introduce a change in behaviour that
 * will sneak in under the radar. (2) would break compilation but also force
 * users to decide how to set level and make them aware of the change. Which
 * path to choose is yet to be decided.
 *
 *
 * You will in the future still be able to set level directly on Loggers by
 * calling get_loggers(...).set_level(...). Note that a logging action must
 * pass both the context level and the Logger level to be logged.
 *
 * As a side note, set_level and set_level_translation differs from get_logger
 * etc, in that they do not fallback to master context if the given context has
 * no logger set, but only affects loggers that actually are added to the
 * context.
 *
 */
public void set_level(LogLevel l) option(category: "Logging/Control Logging") { __impl__.get_logger().set_level(l); }
public void set_level(Context context, LogLevel l) option(category: "Logging/Control Logging") { __impl__.get_logger(context, false).set_level(l); }
public void set_level_translation(LogLevel from, LogLevel to) option(category: "Logging/Control Logging") { __impl__.get_logger().set_level_translation(from, to); }
public void set_level_translation(Context context, LogLevel from, LogLevel to) option(category: "Logging/Control Logging") { __impl__.get_logger(context, false).set_level_translation(from, to); }

// Level on context, if set, a LogUnit must pass both this and the level of the Loggers
public void set_master_level(LogLevel l) option(category: "Logging/Control Logging") { __impl__.set_context_level(__impl__._MasterContext, l); }
public void set_context_level(Context context, LogLevel l) option(category: "Logging/Control Logging") { __impl__.set_context_level(context.name(), l); }

// Do logging
public void debug(string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, [msg], DEBUG, pos); }
public void debug(vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, msg_parts, DEBUG, pos); }
public void info(string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, [msg], INFO, pos); }
public void info(vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, msg_parts, INFO, pos); }
public void warning(string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, [msg], WARNING, pos); }
public void warning(vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, msg_parts, WARNING, pos); }
public void error(string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, [msg], ERROR, pos); }
public void error(vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, msg_parts, ERROR, pos); }
public void fatal(string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, [msg], FATAL, pos); }
public void fatal(vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(null<Context>, msg_parts, FATAL, pos); }

// Specify level and optionally time as argument
public void log(string msg, LogLevel level, timestamp option(nullable) time = null<timestamp>, integer pos = $pos)
    option(category: "Logging/Do Logging")
{
    __impl__.log(null<Context>, [msg], level, pos, time);
}

public void log(vector(string) msg_parts, LogLevel level, timestamp option(nullable) time = null<timestamp>, integer pos = $pos)
    option(category: "Logging/Do Logging")
{
    __impl__.log(null<Context>, msg_parts, level, pos, time);
}

// Specify level and optionally time as argument
public void log(string msg, LogLevelObj level, timestamp option(nullable) time = null<timestamp>, integer pos = $pos)
    option(category: "Logging/Do Logging")
{
    __impl__.log(null<Context>, [msg], level.get_enum(), pos, time);
}

public void log(vector(string) msg_parts, LogLevelObj level, timestamp option(nullable) time = null<timestamp>, integer pos = $pos)
    option(category: "Logging/Do Logging")
{
    __impl__.log(null<Context>, msg_parts, level.get_enum(), pos, time);
}

public void debug(Context option(nullable) context, string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, [msg], DEBUG, pos); }
public void debug(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, msg_parts, DEBUG, pos); }
public void info(Context option(nullable) context, string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, [msg], INFO, pos); }
public void info(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, msg_parts, INFO, pos); }
public void warning(Context option(nullable) context, string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, [msg], WARNING, pos); }
public void warning(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, msg_parts, WARNING, pos); }
public void error(Context option(nullable) context, string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, [msg], ERROR, pos); }
public void error(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, msg_parts, ERROR, pos); }
public void fatal(Context option(nullable) context, string msg, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, [msg], FATAL, pos); }
public void fatal(Context option(nullable) context, vector(string) msg_parts, integer pos = $pos) option(category: "Logging/Do Logging") { __impl__.log(context, msg_parts, FATAL, pos); }

// Specify level as argument
public void log(Context option(nullable) context, string msg, LogLevel level, timestamp option(nullable) time = null<timestamp>, integer pos = $pos)
    option(category: "Logging/Do Logging")
{
    __impl__.log(context, [msg], level, pos, time);
}

public void log(Context option(nullable) context, vector(string) msg_parts, LogLevel level, timestamp option(nullable) time = null<timestamp>, integer pos = $pos)
    option(category: "Logging/Do Logging")
{
    __impl__.log(context, msg_parts, level, pos, time);
}

public void log(Context option(nullable) context, string msg, LogLevelObj level, timestamp option(nullable) time = null<timestamp>, integer pos = $pos)
    option(category: "Logging/Do Logging")
{
    __impl__.log(context, [msg], level.get_enum(), pos, time);
}

public void log(Context option(nullable) context, vector(string) msg_parts, LogLevelObj level, timestamp option(nullable) time = null<timestamp>, integer pos = $pos)
    option(category: "Logging/Do Logging")
{
    __impl__.log(context, msg_parts, level.get_enum(), pos, time);
}


public void newline(integer n_newlines = 1) option(category: "Logging/Do Logging") { get_loggers().write_newline(n_newlines); }
public void newline(Context option(nullable) context, integer n_newlines = 1) option(category: "Logging/Do Logging") { get_loggers(context).write_newline(n_newlines); }

module __impl__ {


public void loop_log(
    LogLevel level,
    Context option(nullable) context,
    vector(string) msg_parts,
    integer i,
    integer n,
	integer pos)
{
    /*
     * Meant for logging progress in loops that adapts to all magnitudes of loop sizes.
     *
     * Log only if (i+1) ("human i") is 1,2,5,10,20,50,100,... or (i+1) = n
     *
     * Gives logging like this with the part in paranthesis appended by this function:
     * "My message (1 of 689)"
     * "My message (2 of 689)"
     * "My message (5 of 689)"
     * "My message (10 of 689)"
     * "My message (20 of 689)"
     * "My message (50 of 689)"
     * "My message (100 of 689)"
     * "My message (200 of 689)"
     * "My message (500 of 689)"
     * "My message (689 of 689)"
     *
     * for a loop like this:
     * integer n = v_size(v); // n = 689
     * for (integer i = 0; i < n; ++i) {
     *     // do stuff with v[i]
     *     logging.loop_info(["My message"], i, n);
     * }
     */

    if (i < 0) {
        throw (E_UNSPECIFIC, "logging loop log requires i >= 0");
    }

    if (i >= n) {
        throw (E_UNSPECIFIC, "logging loop log requires i < n");
    }

    integer human_i = i+1; // humans count from 1

    // Find magnitude
    integer next_magnitude = 10;
    while (human_i >= next_magnitude) {
        next_magnitude *= 10;
    }

    integer magnitude = integer(next_magnitude/10);

    if (human_i == n || human_i == magnitude || human_i == 2*magnitude || human_i == 5*magnitude)
        log(context, concat(msg_parts, [" (", str(human_i), " of ", str(n), ")"]), level, pos);
}

}

public void loop_debug(
    Context option(nullable) context,
    vector(string) msg_parts,
    integer i,
    integer n,
	integer pos = $pos)
{
    __impl__.loop_log(DEBUG, context, msg_parts, i, n, pos);
}

public void loop_info(
    Context option(nullable) context,
    vector(string) msg_parts,
    integer i,
    integer n,
	integer pos = $pos)
{
    __impl__.loop_log(INFO, context, msg_parts, i, n, pos);
}

public void loop_debug(
    vector(string) msg_parts,
    integer i,
    integer n,
	integer pos = $pos)
{
    __impl__.loop_log(DEBUG, null<Context>, msg_parts, i, n, pos);
}

public void loop_info(
    vector(string) msg_parts,
    integer i,
    integer n,
	integer pos = $pos)
{
    __impl__.loop_log(INFO, null<Context>, msg_parts, i, n, pos);
}



public void log(AbstractLogUnit alu)
{
    if (null(alu.context())) {
        get_loggers().log(alu);
    } else {
        get_loggers(alu.context_obj()).log(alu);
    }
}



/* Do status Change logging
 * A `key` and `status` is passed with the message. If `status` for `key` is not
 * changed, nothing is logged. If `status` has changed, the message is logged,
 * and `status` is remembered until next time.
 * Clearing the status (with logging.status_clear) will write an INFO message
 * mentioning what the previous status was.
 *
 * Typical use case is repeated attempts it I/O and wanting to log if something
 * goes wrong, but not wanting to log every following attempt running into the
 * same problem. You'd use the error message as `status`, and maybe as message
 * too.
 *
 * Key is a vector, since it should contain information both about what was
 * attempted and for what object, e.g. ["GET-REALTIME", "BB", "SE0000493852"].
 *
 * Context is automatically part of the key.
 */

public void status_log(Context option(nullable) ctxt, vector(string) key, string status, vector(string) message, LogLevel level, integer pos = $pos)
{
    if (!null(__impl__._StatusTracker.set_status(ctxt, key, status))) {
        log(ctxt, message, level, null<timestamp>, pos);
    }
}

public void status_clear(Context option(nullable) ctxt, vector(string) key)
{
    string prev_status = __impl__._StatusTracker.clear_status(ctxt, key);
    if (!null(prev_status)) {
        log(ctxt, ["[", str_join(key, ", "), "] status cleared (was: ", prev_status, ")"], INFO);
    }
}
public void status_clear(vector(string) key) { status_clear(null<Context>, key); }

public void status_debug(vector(string) key, string status, vector(string) message) { status_log(null<Context>, key, status, message, DEBUG); }
public void status_info(vector(string) key, string status, vector(string) message) { status_log(null<Context>, key, status, message, INFO); }
public void status_warning(vector(string) key, string status, vector(string) message) { status_log(null<Context>, key, status, message, WARNING); }
public void status_error(vector(string) key, string status, vector(string) message) { status_log(null<Context>, key, status, message, ERROR); }
public void status_fatal(vector(string) key, string status, vector(string) message) { status_log(null<Context>, key, status, message, FATAL); }

public void status_debug(Context option(nullable) ctxt, vector(string) key, string status, vector(string) message) { status_log(ctxt, key, status, message, DEBUG); }
public void status_info(Context option(nullable) ctxt, vector(string) key, string status, vector(string) message) { status_log(ctxt, key, status, message, INFO); }
public void status_warning(Context option(nullable) ctxt, vector(string) key, string status, vector(string) message) { status_log(ctxt, key, status, message, WARNING); }
public void status_error(Context option(nullable) ctxt, vector(string) key, string status, vector(string) message) { status_log(ctxt, key, status, message, ERROR); }
public void status_fatal(Context option(nullable) ctxt, vector(string) key, string status, vector(string) message) { status_log(ctxt, key, status, message, FATAL); }

public void status_data_count(out integer n, out integer key_strlen, out integer value_strlen)
{
    __impl__._StatusTracker.data_count(n, key_strlen, value_strlen);
}

// Find lowest active importance level (e.g. debug is lower than info, which is lower than warning)
/*
 * These functions are removed from the interface because all uses cases as of
 * now are better handled with the test functions below (debug(), info(),
 *  info(Context) etc). Reintroduce lowest_importance if a new use case calls for it.
public integer lowest_importance() option(category: "Logging/Control Logging") { return __impl__.lowest_importance(); }
public integer lowest_importance(Context option(nullable) context) option(category: "Logging/Control Logging") { return __impl__.lowest_importance(context); }
*/

// Test if a given log command has any loggers logging at this level
// Typical usage:
// if (logging.debug()) {
//     string msg = expensive_log_message_creation();
//     logging.debug(msg);
// }
public logical debug() option(category: "Logging/Do Logging") { return __impl__.any_would_log(DEBUG, null<Context>); }
public logical info() option(category: "Logging/Do Logging") { return __impl__.any_would_log(INFO, null<Context>); }
public logical warning() option(category: "Logging/Do Logging") { return __impl__.any_would_log(WARNING, null<Context>); }
public logical error() option(category: "Logging/Do Logging") { return __impl__.any_would_log(ERROR, null<Context>); }
public logical fatal() option(category: "Logging/Do Logging") { return __impl__.any_would_log(FATAL, null<Context>); }
public logical any_logging(LogLevel level) option(category: "Logging/Do Logging") { return __impl__.any_would_log(level, null<Context>); }
public logical debug(Context option(nullable) context) option(category: "Logging/Do Logging") { return __impl__.any_would_log(DEBUG, context); }
public logical info(Context option(nullable) context) option(category: "Logging/Do Logging") { return __impl__.any_would_log(INFO, context); }
public logical warning(Context option(nullable) context) option(category: "Logging/Do Logging") { return __impl__.any_would_log(WARNING, context); }
public logical error(Context option(nullable) context) option(category: "Logging/Do Logging") { return __impl__.any_would_log(ERROR, context); }
public logical fatal(Context option(nullable) context) option(category: "Logging/Do Logging") { return __impl__.any_would_log(FATAL, context); }
public logical any_logging(Context option(nullable) context, LogLevel level) option(category: "Logging/Do Logging") { return __impl__.any_would_log(level, context); }

// Fast context creation
public Context context(string c) option(category: "Logging/Convenience") { return new Context(c); }

// Count logging calls
public void start_counting() option(category: "Logging/Logging")
{
    LogCount ret = __impl__.get_log_counter();
    if (!null(ret))
        throw (E_UNSPECIFIC, "Logging counting is already active");
    __impl__.create_log_counter();
}

public LogCount stop_counting() option(category: "Logging/Logging")
{
    LogCount ret = __impl__.get_log_counter();
    if (null(ret))
        throw (E_UNSPECIFIC, "Logging counting is not active");
    __impl__.delete_log_counter();
    return ret;
}

public logical is_counting() option(category: "Logging/Logging") = !null(__impl__.get_log_counter());


// Reset all logging
public void reset() option(category: "Logging/Control Logging") { __impl__.initialize(); }

} // End module logging



/********************************************************************************
 * Core Loggers:
 * - MemoryLogger: Keep messages in memory
 * - WarningLogger: Log to Quantlab warning window
 * - LogMessageLogger: Log with Quantlab log_message function
 * - TableLogger: Show log in text_rgb table
 * - FileLogger: Log to file
 * - StdoutLogger, StderrLogger: Log to stdout, stderr
 * - NullLogger: Trash messages immediately (e.g. to disable a context)
 ********************************************************************************/

module logging {

    class enum flush_rule {
        never, always, above_debug
    };

    void maybe_flush(out_stream stream, flush_rule flush, LogLevelObj level_obj)
    {
        switch (flush) {
        case flush_rule.never:
            break;
        case flush_rule.always:
            stream.flush();
            break;
        case flush_rule.above_debug:
            if (level_obj.importance() > to_obj(DEBUG).importance())
                stream.flush();
            break;
        default:
            throw (E_UNSPECIFIC, "Logging internal: Unhandled flush_rule in maybe_flush");
        }
    }

    // *** MemoryLogger ***
    // Simplest logger -- remembers logging, so you can get it whenever you want
    public class MemoryLogger : public Logger option(category: "Logging/Loggers") {
      public:
        MemoryLogger(number option(nullable) max_num_lines = null<number>);
        vector(LogUnit) dump_log(); // returns log lines, empties buffer

        integer size();
        LogUnit peek();

        // Before any logging and after call to dump_log, functions below will
        // return this: num_missed - 0; last_line - null<string>

        integer num_missed(); // number of lines dropped due to reaching max_num_lines
        LogUnit last_log_unit(); // If num_missed > 0, the last log unit is still
        // saved, and can be retrieved here

      protected:
        virtual void write_impl(LogUnit lu);

      private:
        void _reset_buffer();
        vector(LogUnit) log_units_;
        LogUnit last_log_unit_;
        number max_num_lines_;
        integer num_stored_;
        integer num_missed_;

        void __dbg_print(__dbg_label l);
    };
    MemoryLogger.MemoryLogger(number option(nullable) max_num_lines) {
        _reset_buffer();
        max_num_lines_ = max_num_lines;
    }

    void MemoryLogger.__dbg_print(__dbg_label l)
    {
        l.set_text(strcat(["MemoryLogger { n=", str(this.size()), ", max=", null(max_num_lines_) ? "" : str(max_num_lines_), " }"]));
    }

    vector(LogUnit) MemoryLogger.dump_log() {
        integer n_units = v_size(log_units_);
        vector(LogUnit) copy_of_units[n_units];
        if (n_units > 0) for (i: 0, n_units - 1) {
                copy_of_units[i] = log_units_[i];
            }
        _reset_buffer();
        return copy_of_units;
    }

    integer MemoryLogger.size() = v_size(log_units_);

    LogUnit MemoryLogger.peek()
    {
        if (v_size(log_units_) > 0) {
            return log_units_[v_size(log_units_)-1];
        } else {
            return null<LogUnit>;
        }
    }

    integer MemoryLogger.num_missed() { return num_missed_; }
    LogUnit MemoryLogger.last_log_unit() { return last_log_unit_; }
    void MemoryLogger.write_impl(LogUnit lu) {
        if (null(max_num_lines_) || num_stored_ < max_num_lines_) {
            push_back(log_units_, lu);
            num_stored_++;
        } else {
            num_missed_++;
        }
        last_log_unit_ = lu;
    }
    void MemoryLogger._reset_buffer() {
        resize(log_units_, 0);
        last_log_unit_ = null<LogUnit>;
        num_stored_ = 0;
        num_missed_ = 0;
    }


    // Relieve user from doing dynamic casting
    public MemoryLogger get_memory_logger() option(category: "Logging/Loggers")
    {
        MemoryLogger ret;
        vector(Logger) loggers = get_loggers();
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<MemoryLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }

    public MemoryLogger get_memory_logger(Context option(nullable) context) option(category: "Logging/Loggers")
    {
        MemoryLogger ret;
        vector(Logger) loggers = get_loggers(context);
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<MemoryLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }


    // *** WarningLogger ***
    // NOTE: DEPRECATED - please use LogMessageLogger instead
    // Sends messages to warning window

    public class WarningLogger : public Logger option(category: "Logging/Loggers") {
      public:
        WarningLogger();
        void enable_level();
        void enable_context();

      protected:
        virtual void write_impl(LogUnit lu);

      private:
        logical show_level_;
        logical show_context_;
    };
    WarningLogger.WarningLogger() : show_level_(false), show_context_(false) {}
    void WarningLogger.enable_level() { show_level_ = true; }
    void WarningLogger.enable_context() { show_context_ = true; }
    void WarningLogger.write_impl(LogUnit lu) {
        string msg = lu.message();
        if (show_level_) {
            msg = strcat(["(", lu.level().name(), ") ", msg]);
        }
        if (show_context_) {
            msg = strcat(["[", lu.context(), "] ", msg]);
        }

        ..warning(msg, lu.position());
    }

    // Relieve user from doing dynamic casting
    public WarningLogger get_warning_logger() option(category: "Logging/Loggers")
    {
        WarningLogger ret;
        vector(Logger) loggers = get_loggers();
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<WarningLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }

    public WarningLogger get_warning_logger(Context option(nullable) context) option(category: "Logging/Loggers")
    {
        WarningLogger ret;
        vector(Logger) loggers = get_loggers(context);
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<WarningLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }




    // *** LogMessageLogger ***
    // Uses Quantlab's log_message function

    ..log_level importance_to_ql_level(integer importance);

    public class LogMessageLogger : public Logger option(category: "Logging/Loggers") {
      public:
        LogMessageLogger();
        void enable_level();
        void enable_context();

      protected:
        virtual void write_impl(LogUnit lu);
        virtual void write_newline_impl(integer n_newlines);

      private:
        logical show_level_;
        logical show_context_;
    };
    LogMessageLogger.LogMessageLogger() : show_level_(false), show_context_(false) {}
    void LogMessageLogger.enable_level() { show_level_ = true; }
    void LogMessageLogger.enable_context() { show_context_ = true; }
    void LogMessageLogger.write_impl(LogUnit lu) {
        string msg = lu.message();
        if (show_level_) {
            msg = strcat(["(", lu.level().name(), ") ", msg]);
        }
        if (show_context_) {
            msg = strcat(["[", lu.context(), "] ", msg]);
        }

        ..log_level log_message_ql_level = importance_to_ql_level(lu.level().importance());

        log_message(log_message_ql_level, msg, lu.position());
    }

    void LogMessageLogger.write_newline_impl(integer n_newlines)
    {
        for (integer i = 0; i < n_newlines; ++i) {
            log_message(LOG_INFO, "");
        }
    }

    // Relieve user from doing dynamic casting
    public LogMessageLogger get_log_message_logger() option(category: "Logging/Loggers")
    {
        LogMessageLogger ret;
        vector(Logger) loggers = get_loggers();
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<LogMessageLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }

    public LogMessageLogger get_log_message_logger(Context option(nullable) context) option(category: "Logging/Loggers")
    {
        LogMessageLogger ret;
        vector(Logger) loggers = get_loggers(context);
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<LogMessageLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }


    ..log_level importance_to_ql_level(integer importance)
    {
        if (importance >= to_obj(ERROR).importance()) {
            return ..LOG_ERROR;
        } else if (importance >= to_obj(WARNING).importance()) {
            return ..LOG_WARNING;
        } else if (importance >= to_obj(INFO).importance()) {
            return ..LOG_INFO;
        } else {
            return ..LOG_DEBUG;
        }
    }



    // *** TableLogger ***
    // Stores log messages (like MemoryLogger) but returns them nicely formatted for a Table
    public class TableLogger : public Logger option(category: "Logging/Loggers") {
      public:
        TableLogger(integer max_num_messages = 100);
        void clear_log();
        void enable_context();
        void set_colors(vector(LogLevel) levels, vector(integer) colors);
        void set_colors(vector(integer) thresholds, vector(integer) colors);
        void disable_colors();

        logical is_update_pending();
        matrix(text_rgb) log_matrix(); // Returns what to show

      protected:
        virtual void write_impl(LogUnit lu);

      private:
        integer max_units_;
        vector(LogUnit) units_;
        logical show_context_;
        integer start_index_;
        integer end_index_;
        integer vector_size_;
        logical update_pending_;
        logical multi_context_;
        string context_seen_;
        Colorizer colors_;

        void __dbg_print(__dbg_label l);
    };
    TableLogger.TableLogger(integer max_num_messages)
    : max_units_(max_num_messages), show_context_(false), multi_context_(false), colors_(new Colorizer())
    {
        this.clear_log();
    }

    void TableLogger.__dbg_print(__dbg_label l)
    {
        l.set_text(strcat(["TableLogger { n=", str(v_size(units_)), ", max=", str(max_units_), " }"]));
    }

    void TableLogger.enable_context() { show_context_ = true; }

    void TableLogger.set_colors(vector(LogLevel) levels, vector(integer) colors)
    {
        set_colors(to_obj(levels).importance(), colors);
    }

    void TableLogger.set_colors(vector(integer) thresholds, vector(integer) colors)
    {
        colors_ = new Colorizer(thresholds, colors);
    }

    void TableLogger.disable_colors() { colors_ = new Colorizer([0], [-1]); }

    void TableLogger.clear_log()
    {
        vector_size_ = 2*max_units_;
        resize(units_, vector_size_);
        start_index_ = 0;
        end_index_ = 0;
        update_pending_ = true;
    }
    logical TableLogger.is_update_pending() = update_pending_;
    matrix(text_rgb) TableLogger.log_matrix()
    {
        integer n_units;
        logical wraps = end_index_ < start_index_;
        integer n_from_start_index;
        integer n_from_vector_start;
        if (wraps) {
            n_from_start_index = vector_size_ - start_index_;
            n_from_vector_start = end_index_;
        } else {
            n_from_start_index = end_index_ - start_index_;
            n_from_vector_start = 0;
        }
        n_units = n_from_start_index + n_from_vector_start;

        logical context_on = show_context_ || multi_context_;
        integer num_cols = context_on ? 3 : 2;
        matrix(text_rgb) trgb[n_units, num_cols];

        // i is pos in vector to return
        // ix is pos in data buffer
        integer ix = start_index_;
        integer i = 0;
        if (wraps) {
            while (ix < vector_size_) {
                LogUnit lu = units_[ix];
                number bg_color = colors_.color(lu.level());
                trgb[i, 0] = text_rgb(str(lu.time()), null<number>, bg_color, false);
                if (context_on) {
                    trgb[i, 1] = text_rgb(lu.context(), null<number>, bg_color, false);
                }
                trgb[i, num_cols-1] = text_rgb(lu.message(), null<number>, bg_color, false);
                i++;
                ix++;
            }
            ix = 0;
            while (ix < end_index_) {
                LogUnit lu = units_[ix];
                number bg_color = colors_.color(lu.level());
                trgb[i, 0] = text_rgb(str(lu.time()), null<number>, bg_color, false);
                if (context_on) {
                    trgb[i, 1] = text_rgb(lu.context(), null<number>, bg_color, false);
                }
                trgb[i, num_cols-1] = text_rgb(lu.message(), null<number>, bg_color, false);
                i++;
                ix++;
            }
        } else {
            while (ix < end_index_) {
                LogUnit lu = units_[ix];
                number bg_color = colors_.color(lu.level());
                trgb[i, 0] = text_rgb(str(lu.time()), null<number>, bg_color, false);
                if (context_on) {
                    trgb[i, 1] = text_rgb(lu.context(), null<number>, bg_color, false);
                }
                trgb[i, num_cols-1] = text_rgb(lu.message(), null<number>, bg_color, false);
                i++;
                ix++;
            }
        }

        update_pending_ = false;
        return trgb;
    }
    void TableLogger.write_impl(LogUnit lu)
    {
        // Insert at current end index
        units_[end_index_] = lu;

        // Advance end, possibly set max_reached
        end_index_++;

        // Advance start if max is reached
        if (end_index_ < start_index_ || end_index_ > start_index_ + max_units_) {
            start_index_++;
        }

        // Fix index starting over at beginning
        if (start_index_ == vector_size_) {
            start_index_ = 0;
        }
        if (end_index_ == vector_size_) {
            end_index_ = 0; // start filling in start of vector
        }

        // Remember contexts seen
        string lu_ctxt = lu.context();
        if (null(lu_ctxt)) {
            lu_ctxt = ".$N/U/L/L%;"; // Let's hope nobody uses a context like this
        }

        if (null(context_seen_)) {
            context_seen_ = lu_ctxt;
        } else {
            if (context_seen_ != lu_ctxt) {
                multi_context_ = true;
            }
        }

        update_pending_ = true;
    }

    public TableLogger get_table_logger() option(category: "Logging/Loggers")
    {
        TableLogger ret;
        vector(Logger) loggers = get_loggers();
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<TableLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }

    public TableLogger get_table_logger(Context option(nullable) context) option(category: "Logging/Loggers")
    {
        TableLogger ret;
        vector(Logger) loggers = get_loggers(context);
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<TableLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }



// *** StdoutLogger, StderrLogger ***
module _streamlogger {

class StdstreamLogger : public Logger {
public:
    StdstreamLogger(logical stderr, flush_rule flush);
    void set_flush_rule(flush_rule flush);
    void flush_buffer();
    void show_timestamp();
    void hide_timestamp();
    void show_level();
    void hide_level();
    void show_context();
    void hide_context();

protected:
    override void write_impl(LogUnit lu);
    override void write_newline_impl(integer n_newlines);

private:
    out_stream stream_;
    flush_rule flush_;
};

StdstreamLogger.StdstreamLogger(logical stderr, flush_rule flush)
{
    try {
        stream_ = stderr ? out_stream_stderr() : out_stream_stdout();
        stream_.write("");
    } catch {
        throw (E_UNSPECIFIC, strcat(["Could not use ", stderr ? "stderr" : "stdout", ": ", err.message()]));
    }
    flush_ = flush;
}

void StdstreamLogger.set_flush_rule(flush_rule flush) { flush_ = flush; }
void StdstreamLogger.flush_buffer() { stream_.flush(); }
void StdstreamLogger.show_timestamp() { _show_timestamp(); }
void StdstreamLogger.hide_timestamp() { _hide_timestamp(); }
void StdstreamLogger.show_level() { _show_level(); }
void StdstreamLogger.hide_level() { _hide_level(); }
void StdstreamLogger.show_context() { _show_context(); }
void StdstreamLogger.hide_context() { _hide_context(); }

void StdstreamLogger.write_impl(LogUnit lu)
{
    vector(string) pre_msg[0];
    if (_timestamp_request() != DR_HIDE) {
        push_back(pre_msg, [str(lu.time()), " "]);
    }
    if (_level_request() != DR_HIDE) {
        push_back(pre_msg, [lu.level().fixlen_name(), " "]);
    }
    if (_context_request() == DR_SHOW) {
        push_back(pre_msg, ["[", lu.context(), "] "]);
    }

    stream_.write(pre_msg);
    stream_.write(lu.message_parts());
    stream_.write("\r\n");

    maybe_flush(stream_, flush_, lu.level());
}

void StdstreamLogger.write_newline_impl(integer n_newlines)
{
    for (integer i = 0; i < n_newlines; ++i) {
        stream_.write("\r\n");
    }
}
}

class StdoutLogger : public _streamlogger.StdstreamLogger option(category: "Logging/Loggers") {
public:
    StdoutLogger();

};
StdoutLogger.StdoutLogger()
    : _streamlogger.StdstreamLogger(false, flush_rule.always)
{}

// Relieve user from doing dynamic casting
public StdoutLogger get_stdout_logger() option(category: "Logging/Loggers")
{
    StdoutLogger ret;
    vector(Logger) loggers = get_loggers();
    for (integer i = 0; i < v_size(loggers); ++i) {
        ret = dynamic_cast<StdoutLogger>(loggers[i]);
        if (!null(ret))
            break;
    }
    return ret;
}

public StdoutLogger get_stdout_logger(Context option(nullable) context) option(category: "Logging/Loggers")
{
    StdoutLogger ret;
    vector(Logger) loggers = get_loggers(context);
    for (integer i = 0; i < v_size(loggers); ++i) {
        ret = dynamic_cast<StdoutLogger>(loggers[i]);
        if (!null(ret))
            break;
    }
    return ret;
}



class StderrLogger : public _streamlogger.StdstreamLogger option(category: "Logging/Loggers") {
public:
    StderrLogger();

};
StderrLogger.StderrLogger()
: _streamlogger.StdstreamLogger(true, flush_rule.always)
{}

// Relieve user from doing dynamic casting
public StderrLogger get_stderr_logger() option(category: "Logging/Loggers")
{
    StderrLogger ret;
    vector(Logger) loggers = get_loggers();
    for (integer i = 0; i < v_size(loggers); ++i) {
        ret = dynamic_cast<StderrLogger>(loggers[i]);
        if (!null(ret))
            break;
    }
    return ret;
}

public StderrLogger get_stderr_logger(Context option(nullable) context) option(category: "Logging/Loggers")
{
    StderrLogger ret;
    vector(Logger) loggers = get_loggers(context);
    for (integer i = 0; i < v_size(loggers); ++i) {
        ret = dynamic_cast<StderrLogger>(loggers[i]);
        if (!null(ret))
            break;
    }
    return ret;
}




    // *** FileLogger ***
    // Log to a file

    class file_name_context { };

    /*
     * You can change the name of the log file for a FileLogger during
     * usage. To do this, you provide a function to the FileLogger that says
     * what the file name should be. This function will be called every time
     * the FileLogger is about to log something. To keep state in your
     * function, inherit from class logging.file_name_context. When you set the
     * file_name_function, you can give it a file_name_context too, and
     * FileLogger will keep it between calls to file_name_function for you.
     *
     * See the test "file_name_function" for an example using context. Here is
     * an example not using state, that appends today's date to the log file
     * names.
     *
     string create_log_file_name(
         string base_file_name,
         logging.file_name_context option(nullable) ctxt)
	 {
         return strcat([base_file_name, "_", str(today()), ".log"]);
     }

     out void main()
     {
         logging.FileLogger fl = new logging.FileLogger("path\to\logfile");
         fl.set_file_name_function(&create_log_file_name, null<logging.file_name_context>);
         logging.set_logger(fl);

         // Will write to path\\to\logfile_2018-12-05.log (if that is today's date)
         logging.info("My message");
     }

    */

    public class FileLogger : public Logger option(category: "Logging/Loggers") {
      public:
        FileLogger(string file_name);
        string file_name();
        logical clear_file();
        logical open(); // Do not change flush rule, return false if failed
        logical open(out string error);
        logical open(flush_rule flush);
        logical open(flush_rule flush, out string error);
        logical open(logical option(nullable) auto_flush); // Backward compability
        void close();
        // keep_open makes FileLogger keep the file open from next write until next close
        void keep_open();
        void keep_open(flush_rule flush);
        void keep_open(logical option(nullable) auto_flush); // Backward compability
        logical is_open();
        void flush_buffer();
        void set_field_separator(string s);
        void hide_timestamp();
        void hide_level();
        void show_context();

        // For controlling file_name, used for log rotations, adding date to filename etc
        void set_file_name_function(
            string function(string base_file_name, file_name_context option(nullable) ctxt) f,
            file_name_context option(nullable) ctxt = null<file_name_context>);

      protected:
        virtual void write_impl(LogUnit lu); // Never throws
        virtual void write_newline_impl(integer n_newlines); // Never throws

      private:
        // If file_name_function is used, base_file_name is the original file name
        // sent to the constructor of FileLogger
        string base_file_name_;
        string file_name_;
        flush_rule flush_;
        logical keep_open_;
        string field_sep_;
        out_stream fh_;

        string function(string base_file_name, file_name_context option(nullable) ctxt) file_name_function_;
        file_name_context file_name_context_;

        out_stream _begin_writing();
        void _end_writing(out_stream fh, LogLevelObj level_obj);
        void _do_file_name_check();
        logical _open(out string error);

        void __dbg_print(__dbg_label l);
    };
    FileLogger.FileLogger(string file_name)
    : base_file_name_(file_name), file_name_(file_name), flush_(flush_rule.always), keep_open_(false), field_sep_(" ")
    {}

    void FileLogger.__dbg_print(__dbg_label l)
    {
        l.set_text(strcat(["FileLogger { fn=\"", file_name_, "\"", null(fh_) ? "" : " <open>", null(file_name_function_) ? "" : " <func-generated-name>", " }"]));
    }

    string FileLogger.file_name() = file_name_;
    logical FileLogger.clear_file()
    {
        logical kept_open = !null(fh_);
        logical success = true;
        if (kept_open) {
            this.close();
        }

        try {
            file_write(file_name_, "");
        } catch {
            success = false;
            // Just leave it, logging shouldn't throw exceptions
        }

        if (kept_open) {
            if (!this.open()) {
                success = false;
            }
        }

        return success;
    }

    logical FileLogger._open(out string error)
    {
        if (null(fh_)) {
            try {
                fh_ = out_stream_file_append(file_name_);
                error = null<string>;
            } catch {
                error = err.message();
                return false;
            }
        }
        return true;
    }

    logical FileLogger.open()
    {
        string error;
        return this.open(error);
    }

    logical FileLogger.open(out string error)
    {
        return _open(error);
    }

    logical FileLogger.open(logical option(nullable) auto_flush)
    {
        if (null(auto_flush))
            return open();
        else if (auto_flush)
            return open(flush_rule.always);
        else
            return open(flush_rule.never);
    }

    logical FileLogger.open(flush_rule flush)
    {
        string error;
        return this.open(flush, error);
    }

    logical FileLogger.open(flush_rule flush, out string error)
    {
        flush_ = flush;
        return _open(error);
    }

    void FileLogger.close()
    {
        fh_.close();
        fh_ = null<out_stream>;
        keep_open_ = false;
    }

	void FileLogger.keep_open()
    {
        keep_open_ = true;
    }

    void FileLogger.keep_open(logical option(nullable) auto_flush)
    {
        if (null(auto_flush))
            keep_open();
        else if (auto_flush)
            keep_open(flush_rule.always);
        else
            keep_open(flush_rule.never);
    }

    void FileLogger.keep_open(flush_rule flush)
    {
        flush_ = flush;
        keep_open();
    }

    logical FileLogger.is_open() = !null(fh_);
	void FileLogger.flush_buffer()
    {
        if (!null(fh_))
            fh_.flush();
    }
    void FileLogger.set_field_separator(string s) { field_sep_ = s; }
    void FileLogger.hide_timestamp() { _hide_timestamp(); }
    void FileLogger.hide_level() { _hide_level(); }
    void FileLogger.show_context() { _show_context(); }
    void FileLogger.write_impl(LogUnit lu) // Never throws
    {
        out_stream fh = _begin_writing();
        if (_timestamp_request() != DR_HIDE) {
            fh.write(str(lu.time()));
            fh.write(field_sep_);
        }
        if (_level_request() != DR_HIDE) {
            fh.write(lu.level().fixlen_name());
            fh.write(field_sep_);
        }

        if (_context_request() == DR_SHOW && !null(lu.context())) {
            fh.write(["<", lu.context(), ">", field_sep_]);
        }

        vector(string) msg_parts = lu.message_parts();
        integer msg_parts_sz = v_size(msg_parts);
        for (integer i = 0; i < msg_parts_sz; ++i) {
            fh.write(msg_parts[i]);
        }

        fh.write("\r\n");

        _end_writing(fh, lu.level());
    }
    void FileLogger.write_newline_impl(integer n_newlines) // Never throws
    {
        out_stream fh = _begin_writing();
        for (integer i = 0; i < n_newlines; ++i) {
            fh.write("\r\n");
        }
        _end_writing(fh, to_obj(INFO));
    }

    void FileLogger.set_file_name_function(
        string function(string base_file_name, file_name_context option(nullable) ctxt) f,
        file_name_context option(nullable) ctxt)
    {
        file_name_function_ = f;
        file_name_context_ = ctxt;
    }

    out_stream FileLogger._begin_writing()
    {
        _do_file_name_check();

        if (null(fh_)) {
            try {
                if (keep_open_) {
                    fh_ = out_stream_file_append(file_name_);
                    return fh_;
                } else {
                    return out_stream_file_append(file_name_);
                }
            } catch {
                return null<out_stream>; // Just accept it, logging shouldn't throw exceptions
            }
        } else {
            return fh_;
        }
    }

	void FileLogger._end_writing(out_stream fh, LogLevelObj level_obj)
    {
        if (null(fh_)) {
            fh.close(); // It was a temporary file handle
        } else {
            maybe_flush(fh, flush_, level_obj);
        }
    }

    void FileLogger._do_file_name_check()
    {
        if (null(file_name_function_))
            return;

        string new_file_name = file_name_function_(base_file_name_, file_name_context_);
        if (new_file_name != file_name_) {
            logical was_open = this.is_open();

            if (was_open)
                this.close();

            file_name_ = new_file_name;

            if (was_open)
                this.open();
        }
    }

    public FileLogger get_file_logger() option(category: "Logging/Loggers")
    {
        FileLogger ret;
        vector(Logger) loggers = get_loggers();
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<FileLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }

    public FileLogger get_file_logger(Context option(nullable) context) option(category: "Logging/Loggers")
    {
        FileLogger ret;
        vector(Logger) loggers = get_loggers(context);
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<FileLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }



/*
 * PipeLogger writes log messages to a pipe.
 * It can be given a logging context at startup which will be used if no
 * context is provided with the message. If context is provided in both ways,
 * it will result in a concatenation with dot: "pipe-context.message-context".
 *
 * Errors will be counted per type so the client can check.
 */

class PipeLogger : public Logger option(category: "Logging/Loggers") {
public:
    PipeLogger(qlc.pipe_writer pipe, integer timeout);
    PipeLogger(qlc.pipe_writer pipe, Context ctxt, integer timeout);

    void set_message_prefix(string option(nullable) prefix);
    void set_message_suffix(string option(nullable) suffix);

    void log_null(); // Used as poison pill to stop logging thread

    string error_of_latest_log();

protected:
    override void write_impl(LogUnit lu);
	override void write_newline_impl(integer n_newlines);


private:
    qlc.pipe_writer pipe_;
    Context ctxt_;
    integer timeout_;
    string error_of_latest_log_;
    string prefix_;
    string suffix_;
};

PipeLogger.PipeLogger(qlc.pipe_writer pipe, integer timeout)
: pipe_(pipe), ctxt_(null<Context>), timeout_(timeout)
{
}

PipeLogger.PipeLogger(qlc.pipe_writer pipe, Context ctxt, integer timeout)
: pipe_(pipe), ctxt_(ctxt), timeout_(timeout)
{
}

void PipeLogger.set_message_prefix(string option(nullable) prefix)
{
    prefix_ = prefix;
}

void PipeLogger.set_message_suffix(string option(nullable) suffix)
{
    suffix_ = suffix;
}

void PipeLogger.log_null()
{
    pipe_.write(qlc.variant(null<AbstractLogUnit>), timeout_);
}

string PipeLogger.error_of_latest_log() = error_of_latest_log_;

void PipeLogger.write_impl(LogUnit lu)
{
    if (!null(ctxt_) || !null(prefix_) || !null(suffix_)) {
        // LogUnit must be changed

        vector(string) msg_parts = concat([prefix_], lu.message_parts());
        push_back(msg_parts, suffix_);
        Context ctxt =
            null(ctxt_) ? lu.context_obj() :
            null(lu.context()) ? ctxt_ :
            context(strcat([ctxt_.name(), ".", lu.context()]));
		integer pos = lu.position();

        lu = new LogUnit(msg_parts, lu.time(), lu.level(), ctxt, pos);
    }

    try {
		AbstractLogUnit alu = lu;
		pipe_.write(qlc.variant(alu), timeout_);
        error_of_latest_log_ = null<string>;
    } catch {
        error_of_latest_log_ = err.message();
    }
}

void PipeLogger.write_newline_impl(integer n_newlines)
{
	try {
		AbstractLogUnit alu = new NewlineUnit(n_newlines, null<Context>, 0);
		pipe_.write(qlc.variant(alu), timeout_);
		error_of_latest_log_ = null<string>;
	} catch {
		error_of_latest_log_ = err.message();
	}
}

public PipeLogger get_pipe_logger() option(category: "Logging/Loggers")
{
    PipeLogger ret;
    vector(Logger) loggers = get_loggers();
    for (integer i = 0; i < v_size(loggers); ++i) {
        ret = dynamic_cast<PipeLogger>(loggers[i]);
        if (!null(ret))
            break;
    }
    return ret;
}

    // *** NullLogger ***
    // When you don't want any logging, but need a logger (to surpress a context perhaps)
    public class NullLogger : public Logger option(category: "Logging/Loggers") {
      protected:
        virtual void write_impl(LogUnit lu);
    };
    void NullLogger.write_impl(LogUnit lu) {}

    public NullLogger get_null_logger() option(category: "Logging/Loggers")
    {
        NullLogger ret;
        vector(Logger) loggers = get_loggers();
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<NullLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }

    public NullLogger get_null_logger(Context option(nullable) context) option(category: "Logging/Loggers")
    {
        NullLogger ret;
        vector(Logger) loggers = get_loggers(context);
        for (integer i = 0; i < v_size(loggers); ++i) {
            ret = dynamic_cast<NullLogger>(loggers[i]);
            if (!null(ret))
                break;
        }
        return ret;
    }
}
