/* Job Status Control
 *
 * QlbLogger helps you control the status of your QLB job.
 * There are three levels for doing this:
 * - No control (call QlbLogger.disable_status_control() to use this level)
 * - Mild control (default, can also be set by calling QlbLogger.enable_mild_status_control())
 * - Firm control (call QlbLogger.enable_firm_status_control() to use this level)
 *
 * No control: Never set the job status
 *
 * Mild control: Set job status to WARNING if logging is done with level ERROR
 * or higher.
 *
 * Firm control: Set job status to WARNING if logging is done with level
 * WARNING. Set job status to ERROR if logging is done with level ERROR or
 * higher.
 *
 * No matter the control level, you can set job status to ERROR by calling
 * QlbLogger.set_job_status_error().
 *
 * The QlbLogger will not set status WARNING if status ERROR has been set
 * previously.
 */


ahs_message_type qlb_convert_to_ahs_message_type(logging.LogLevelObj l) {
    switch(l.get_enum()) {
        case logging.DEBUG: return AHS_MESSAGE_DEBUG;
        case logging.INFO: return AHS_MESSAGE_INFO;
        case logging.WARNING: return AHS_MESSAGE_WARNING;
        case logging.ERROR: return AHS_MESSAGE_ERROR;
        case logging.FATAL: return AHS_MESSAGE_ERROR;
        default: return AHS_MESSAGE_INFO;
    }
}

enum QlbJobStatus {
    QJS_NOT_SET,
    QJS_WARNING,
    QJS_ERROR
};

void set_qlb_job_status(QlbJobStatus status)
{
    switch (status) {
        case QJS_WARNING:
            qlb_set_status(WARNING);
            break;
        case QJS_ERROR:
            qlb_set_status(ERROR);
    }
}

module logging {

    enum QlbStatusControl {
        QLB_SC_NONE,
        QLB_SC_MILD,
        QLB_SC_FIRM
    };

    public class QlbLogger : public Logger option(category: "Logging/Loggers") {
      public:
        QlbLogger(string default_function = "");
        void set_default_function(string default_function);
        void disable_status_control();
        void enable_mild_status_control();
        void enable_firm_status_control();
        void set_job_status_error();
        void set_job_status_warning();

      protected:
        virtual void write_impl(LogUnit lu);

      private:
        string default_function_;
        QlbJobStatus current_job_status_;
        QlbStatusControl status_control_;
        void _set_job_status(LogLevelObj level);
        void _set_job_status(QlbJobStatus status);
    };
    QlbLogger.QlbLogger(string default_function)
        : default_function_(default_function), current_job_status_(QJS_NOT_SET), status_control_(QLB_SC_MILD)
    {}

    void QlbLogger.set_default_function(string default_function) { default_function_ = default_function; }
    void QlbLogger.disable_status_control() { status_control_ = QLB_SC_NONE; }
    void QlbLogger.enable_mild_status_control() { status_control_ = QLB_SC_MILD; }
    void QlbLogger.enable_firm_status_control() { status_control_ = QLB_SC_FIRM; }
    void QlbLogger.set_job_status_error() { _set_job_status(QJS_ERROR); }
    void QlbLogger.set_job_status_warning() { _set_job_status(QJS_WARNING); }
    void QlbLogger.write_impl(LogUnit lu)
    {
        this._set_job_status(lu.level());

        string current_func = null(lu.context()) ? default_function_ : lu.context();
        ahs_log_report(qlb_convert_to_ahs_message_type(lu.level()), current_func, lu.message());
    }
    void QlbLogger._set_job_status(LogLevelObj level)
    {
        if (status_control_ == QLB_SC_NONE) {
            return;
        } else if (status_control_ == QLB_SC_MILD) {
            if (level.importance() >= to_obj(ERROR).importance()) {
                _set_job_status(QJS_WARNING);
            }
        } else if (status_control_ == QLB_SC_FIRM) {
            if (level.importance() >= to_obj(ERROR).importance()) {
                _set_job_status(QJS_ERROR);
            } else if (level.importance() >= to_obj(WARNING).importance()) {
                _set_job_status(QJS_WARNING);
            }
        }
    }
    void QlbLogger._set_job_status(QlbJobStatus status)
    {
        switch (status) {
            case QJS_ERROR:
                if (current_job_status_ != QJS_ERROR) {
                    set_qlb_job_status(QJS_ERROR);
                    current_job_status_ = QJS_ERROR;
                }
                break;
            case QJS_WARNING:
                if (current_job_status_ == QJS_NOT_SET) {
                    set_qlb_job_status(QJS_WARNING);
                    current_job_status_ == QJS_WARNING;
                }
        }
    }


    public QlbLogger get_qlb_logger() option(category: "Logging/Loggers") { return dynamic_cast<QlbLogger>(get_logger()); }
    public QlbLogger get_qlb_logger(Context option(nullable) context) option(category: "Logging/Loggers") { return dynamic_cast<QlbLogger>(get_logger(context)); }
}
