module mail
{
	void check_str(string option(nullable) s)
	{
		if (null(s))
			throw(E_INVALID_ARG, "Null string");
		if (find_string(s, "\n"))
			throw(E_INVALID_ARG, "Newline in string");
		if (s == ".")
			throw(E_INVALID_ARG, "Dot");
	}

	void check_response(string option(nullable) response, string code_sp)
	{
		if (null(response))
			throw(E_IO, "Null response from SMTP server");
		if (sub_string(response, 0, 4) != code_sp)
			throw(E_IO, strcat([ "Error from SMTP server: ", response ]));
	}

	string fix_address(string address)
	{
		string first_char = sub_string(address,0,1);
		string last_char = sub_string(address,strlen(address)-1,1);
		if (!equal("<", first_char)) {
			address = strcat(["<",address]);
		}
		if (!equal(">", last_char)) {
			address = strcat([address,">"]);
		}

		return address;
	}

	string fix_attachment_content(string content) {
		// Fixes:
		// 1) Newlines will be \r\n
		// 2) '.' on start of line converted to '..'
		// 3) File will end with newline

		// (1)
		// A little inefficient maybe, translates all newlines to \n, then
		// (back) to \r\n. Should be in C++ really
		content = str_replace(content, '\r\n', '\n');
		content = str_replace(content, '\r', '\n');
		content = str_replace(content, '\n', '\r\n');

		// (2)
		if (strlen(content) > 0 && sub_string(content, 0, 1) == '.') {
			content = strcat('.', content); // extra dot if content starts with dot
		}
		content = str_replace(content, '\r\n.', '\r\n..');

		// (3)
		if (strlen(content) > 0 && sub_string(content, strlen(content) - 1, 1) != '\n') {
			content = strcat(content, '\r\n');
		}

		return content;
	}

	// Assumes central european time, and daylight saving Apr to Oct (inclusive).
	// Gives error on timezone from last Sunday of March through end of March and
	// from last Sunday of October through end of October.
	string rfc822_datetime(timestamp t) {
		date d = date(t);
		string dayofmonth = str(day(d));
		vector(string) month_list = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
		string shortmonth = month_list[month(d)-1];
		string shortyear = sub_string(str(year(d)), 2, 2);
		string localtime = strcat([str(hour(t)), ":", str(minute(t)), ":", str(second(t))]);
		logical daylight_saving = month(d) > 3 && month(d) < 11;
		string timezone;
		if (daylight_saving) {
			timezone = "+0200";
		} else {
			timezone = "+0100";
		}

		string s = strcat([dayofmonth, " ", shortmonth, " ", shortyear, " ", localtime, " ", timezone]);
		return s;
	}

	public object attachment {
		string name;
		string content;
		timestamp created;
		timestamp modified;
		string content_type;
	};
	// Create without created/modified timestamps
	 public attachment attachment(string option(nullable) name = null<string>,
								  string option(nullable) content = null<string>,
								  string option(nullable) content_type = null<string>)
	 {
		 attachment att = new attachment;
		 if (null(name)) name = "";
		 att.name = name;
		 if (null(content)) content = "";
		 att.content = fix_attachment_content(content);
		 att.content_type = content_type;
		 return att;
	 }
	 // Create with created/modified timestamps
	public attachment attachment(string option(nullable) name,
								 string option(nullable) content,
								 timestamp option(nullable) created,
								 timestamp option(nullable) modified,
								 string option(nullable) content_type = null<string>)
	{
		attachment att = new attachment;
		if (null(name)) name = "";
		att.name = name;
		if (null(content)) content = "";
		att.content = fix_attachment_content(content);
		att.created = created;
		att.modified = modified;
		att.content_type = content_type;
		return att;
	}

	public object email {
	public:
		void set_content_type(string content_type);
		void set_reply_to(string address);
		void set_from_address(string address);
		void set_from_full_name(string full_name);
		void set_to(vector(string) to_addresses);
		void set_subject(string new_subject);
		void set_body(vector(string) new_body);
		void set_importance(logical important);
		void add_cc(string address);
		void add_bcc(string address);
		void add_attachment(attachment att);
		void send(string host, integer port = 25);

		// Required (set upon creation)
		string from_address;
		string from_full_name;
		vector(string) to;
		string subject;
		vector(string) body;
		// Optional (set by member functions)
		string content_type;
		string reply_to;
		vector(string) cc;
		vector(string) bcc;
		vector(attachment) attachments;
		logical high_importance;
	};
	 public email email(string from, string from_full_name, vector(string) to, string subject, vector(string) body)
	 {
		 email e = new email;
		 e.set_from_address(from);
		 e.set_from_full_name(from_full_name);
		 e.set_to(to);
		 e.set_subject(subject);
		 e.set_importance(false);
		 e.set_body(body);

		 return e;
	 }
	 void email.set_body(vector(string) new_body)
	 {
		 resize(this.body, 0);
		 integer body_sz = v_size(new_body);
		 for (integer i = 0; i < body_sz; ++i) {
			 vector(string) split_body_line = str_tokenize(new_body[i], "\n");
			 if (v_size(split_body_line) > 0) {
				 this.body = concat(this.body, split_body_line);
											  }
											   }
	 }
	 void email.set_from_address(string address)
	 {
		 this.from_address = address;
	 }
	 void email.set_from_full_name(string full_name)
	 {
		 this.from_full_name = full_name;
	 }
	 void email.set_to(vector(string) to_addresses)
	 {
		 this.to = to_addresses;
	 }
	 void email.set_subject(string new_subject)
	 {
		 this.subject = new_subject;
	 }

	 void email.set_content_type(string content_type)
	 {
		 this.content_type = content_type;
	 }
	 void email.set_reply_to(string address)
	 {
		 this.reply_to = address;
	 }
	 void email.add_cc(string address)
	 {
		 push_back(this.cc, address);
	 }
	 void email.add_bcc(string address)
	 {
		 push_back(this.bcc, address);
	 }
	 void email.add_attachment(attachment att)
	 {
		 push_back(this.attachments, att);
	 }
	 void email.set_importance(logical important)
	 {
		 this.high_importance = important;
	 }
	 void email.send(string host, integer port)
	 {
		 integer n_to = v_size(this.to);
		 integer n_cc = v_size(this.cc);
		 integer n_bcc = v_size(this.bcc);
		 integer n_body = v_size(this.body);

		 if (n_to == 0)
			 throw(E_INVALID_ARG, "No recipients");
		 check_str(this.from_address);
		 check_str(this.from_full_name);
		 for (integer i = 0 ; i < n_to ; i++)
			 check_str(this.to[i]);
		 check_str(this.subject);
		 for (integer i = 0 ; i < n_body ; i++)
			 check_str(this.body[i]);

		 this.from_address = fix_address(this.from_address);
		 for (integer i = 0 ; i < n_to ; i++)
			 this.to[i] = fix_address(this.to[i]);
		 for (integer i = 0 ; i < n_cc ; i++)
			 this.cc[i] = fix_address(this.cc[i]);
		 for (integer i = 0 ; i < n_bcc ; i++)
			 this.bcc[i] = fix_address(this.bcc[i]);

		 in_stream s_in;
		 out_stream s_out;

		 socket_connect(s_in, s_out, host, port);
		 check_response(s_in.read_line(), "220 ");

		 s_out.write(strcat([ "helo localhost\r\n" ]));
		 check_response(s_in.read_line(), "250 ");

		 s_out.write(strcat([ "mail from: ", from_address, "\r\n" ]));
		 check_response(s_in.read_line(), "250 ");

		 for (integer i = 0 ; i < n_to ; i++) {
			 s_out.write(strcat([ "rcpt to: ", this.to[i], "\r\n" ]));
			 check_response(s_in.read_line(), "250 ");
		 }
		 for (integer i = 0 ; i < n_cc ; i++) {
			 s_out.write(strcat([ "rcpt to: ", this.cc[i], "\r\n" ]));
			 check_response(s_in.read_line(), "250 ");
		 }
		 for (integer i = 0 ; i < n_bcc ; i++) {
			 s_out.write(strcat([ "rcpt to: ", this.bcc[i], "\r\n" ]));
			 check_response(s_in.read_line(), "250 ");
		 }

		 s_out.write("data\r\n");
		 check_response(s_in.read_line(), "354 ");

		 s_out.write(strcat([ "From: ", this.from_address,
							  " (", this.from_full_name, ")\r\n" ]));
		 for (integer i = 0 ; i < n_to ; i++)
			 s_out.write(strcat([ "To: ", this.to[i], "\r\n" ]));
		 for (integer i = 0 ; i < n_cc ; i++)
			 s_out.write(strcat([ "Cc: ", this.cc[i], "\r\n" ]));
		 for (integer i = 0 ; i < n_bcc ; i++)
			 s_out.write(strcat([ "Bcc: ", this.bcc[i], "\r\n" ]));
		 if (!null(this.reply_to))
			 s_out.write(strcat([ "Reply-To: ", this.reply_to, "\r\n" ]));
		 s_out.write(strcat([ "Subject: ", this.subject, "\r\n" ]));

		 if (this.high_importance){
			 s_out.write("X-Priority: 1\r\n");
			 s_out.write("Importance: high\r\n");
		 }

		 string mime_boundary_id = "nateuhi.apgf,dhieu [98 234 haodi";
		 string mime_boundary = strcat(['--', mime_boundary_id, '\r\n']);
		 string mime_boundary_end = strcat(['--', mime_boundary_id, '--\r\n']);

		 integer n_attachments = v_size(this.attachments);
		 logical mime_multipart = n_attachments > 0;

		 if (mime_multipart) {
			 s_out.write("MIME-Version: 1.0\r\n");
			 s_out.write(strcat(['Content-Type: multipart/mixed; boundary="', mime_boundary_id, '"\r\n']));
			 s_out.write("\r\nThis is a message with multiple parts in MIME format.\r\n");
			 s_out.write(mime_boundary);
		 }


		 if (!null(this.content_type)) {
			 s_out.write(strcat(["Content-Type: ", this.content_type, "\r\n"]));
		 }

		 // Body
		 s_out.write("\r\n");
		 for (integer i = 0 ; i < n_body ; i++)
			 s_out.write(strcat([ this.body[i], "\r\n" ]));

		 // Attachments
		 for (integer i = 0; i < n_attachments; i++) {
			 attachment att = this.attachments[i];
			 s_out.write(mime_boundary);

			 string content_disp_line = strcat(["Content-Disposition: attachment; filename=", att.name]);
			 if (!null(att.created)) {
				 content_disp_line = strcat([content_disp_line, '; creation-date="', rfc822_datetime(att.created), '"']);
		}
		if (!null(att.modified)) {
		content_disp_line = strcat([content_disp_line, '; modification-date="', rfc822_datetime(att.modified), '"']);
		}
		s_out.write(strcat(content_disp_line, "\r\n"));

		if (!null(att.content_type)) {
		s_out.write(strcat(["Content-Type: ", att.content_type, "\r\n"]));
		}
		s_out.write("\r\n");
		string fixed_content = fix_attachment_content(att.content);
		s_out.write(fixed_content);
	}
	if (mime_multipart) {
		s_out.write(mime_boundary_end);
	}

	s_out.write(".\r\n");

	check_response(s_in.read_line(), "250 ");

	s_out.write("quit\r\n");
	check_response(s_in.read_line(), "221 ");
	}



	public void send(string		from,
			 string		from_full_name,
			 vector(string)	to,
			 string		subject,
			 vector(string)	body,
			 vector(attachment) option(nullable) attachments,
			 string		host,
			 integer	port    = 25)
	{
	integer n_to = v_size(to);
	integer n_body = v_size(body);

	if (n_to == 0)
		throw(E_INVALID_ARG, "No recipients");
	check_str(from);
	check_str(from_full_name);
	for (integer i = 0 ; i < n_to ; i++)
		check_str(to[i]);
	check_str(subject);
	for (integer i = 0 ; i < n_body ; i++)
		check_str(body[i]);

	from = fix_address(from);
	for (integer i = 0 ; i < n_to ; i++)
		to[i] = fix_address(to[i]);

	in_stream s_in;
	out_stream s_out;

	socket_connect(s_in, s_out, host, port);
	check_response(s_in.read_line(), "220 ");

	s_out.write(strcat([ "helo localhost\r\n" ]));
	check_response(s_in.read_line(), "250 ");

	s_out.write(strcat([ "mail from: ", from, "\r\n" ]));
	check_response(s_in.read_line(), "250 ");

	for (integer i = 0 ; i < n_to ; i++) {
		s_out.write(strcat([ "rcpt to: ", to[i], "\r\n" ]));
		check_response(s_in.read_line(), "250 ");
	}

	s_out.write("data\r\n");
	check_response(s_in.read_line(), "354 ");

	s_out.write(strcat([ "From: ", from,
				 " (", from_full_name, ")\r\n" ]));
	for (integer i = 0 ; i < n_to ; i++)
		s_out.write(strcat([ "To: ", to[i], "\r\n" ]));
	s_out.write(strcat([ "Subject: ", subject, "\r\n" ]));

	string mime_boundary_id = "nateuhi.apgf,dhieu [98 234 haodi";
	string mime_boundary = strcat(['--', mime_boundary_id, '\r\n']);
	string mime_boundary_end = strcat(['--', mime_boundary_id, '--\r\n']);

	integer n_attachments = v_size(attachments);
	logical mime_multipart = n_attachments > 0;

	if (mime_multipart) {
		s_out.write("MIME-Version: 1.0\r\n");
		s_out.write(strcat(['Content-Type: multipart/mixed; boundary="', mime_boundary_id, '"\r\n']));
		s_out.write("\r\nThis is a message with multiple parts in MIME format.\r\n");
		s_out.write(mime_boundary);
		s_out.write("Content-Type: text/plain\r\n");
	}

	// Body
	s_out.write("\r\n");
	for (integer i = 0 ; i < n_body ; i++)
		s_out.write(strcat([ body[i], "\r\n" ]));

	// Attachments
	for (integer i = 0; i < n_attachments; i++) {
		attachment att = attachments[i];
		s_out.write(mime_boundary);

		string content_disp_line = strcat(["Content-Disposition: attachment; filename=", att.name]);
		if (!null(att.created)) {
		content_disp_line = strcat([content_disp_line, '; creation-date="', rfc822_datetime(att.created), '"']);
		}
		if (!null(att.modified)) {
		content_disp_line = strcat([content_disp_line, '; modification-date="', rfc822_datetime(att.modified), '"']);
			 }
		s_out.write(strcat(content_disp_line, "\r\n"));

		if (!null(att.content_type)) {
		s_out.write(strcat(["Content-Type: ", att.content_type, "\r\n"]));
		}
		s_out.write("\r\n");
		string fixed_content = fix_attachment_content(att.content);
		s_out.write(fixed_content);
	}
	if (mime_multipart) {
		s_out.write(mime_boundary_end);
	}

	s_out.write(".\r\n");

	check_response(s_in.read_line(), "250 ");

	s_out.write("quit\r\n");
	check_response(s_in.read_line(), "221 ");
	}

	public void send(string		from,
			 string		from_full_name,
			 vector(string)	to,
			 string		subject,
			 vector(string)	body,
			 string		host,
			 integer		port = 25)
	{
	send(from,
		 from_full_name,
		 to,
		 subject,
		 body,
		 null<vector(attachment)>,
		 host,
		 port);
	}

	public void send(string		from,
			 string		from_full_name,
			 vector(string)	to,
			 string		subject,
			 vector(string)	body,
			 vector(string) option(nullable) attachment_paths,
			 vector(string) option(nullable) attachment_mime_types,
			 string		host,
			 integer		port = 25)
	{
	vector(attachment) atts;
	integer attachment_paths_sz = v_size(attachment_paths);
	if (!null(attachment_mime_types) &&
		v_size(attachment_mime_types) != attachment_paths_sz) {
		throw (E_INVALID_ARG, "Vector sizes mismatch");
	}
	if (attachment_paths_sz>0) for (i: 0, attachment_paths_sz-1) {
		string path = attachment_paths[i];
		string mime_type;
		if (!null(attachment_mime_types))
		mime_type = attachment_mime_types[i];

		push_back(atts, attachment(path, file_read(path), null<timestamp>, null<timestamp>, mime_type));
	}

	send(from,
		 from_full_name,
		 to,
		 subject,
		 body,
		 atts,
		 host,
		 port);
	}


	void test_attachment(string to_email, string host = 'emmaline.algorithmica.se') {
	string content1 =
		"testline\n"
		". line startswith dot\r\n"
		".\n"
		"line with dot only\r\n"
		"..\r\n"
		"two dots on a line\r\n"
		"ends without newline"
		;
	string content2 = "";

	string content3 = null<string>;
	string content4 = ".";
	string content5 =
		"	tab and mixed newlines\n"
		"\r"
		"\r\n\n"
		"\r"
		"\r\n"
		"a couple of mixed newlines, should be 5 empty lines\n"
		;
	string content6 = "nameless attachment (null)";
	string content7 = "nameless attachment (empty string)";
	send("test_attachment", "test@mail_attachment", [to_email], "test of mail module", ["test of ", "attachments", "7 created attachments"],
		 attachment(["content1", "content2", "content3", "content4", "content5", null<string>, ""],
			[content1, content2, content3, content4, content5, content6, content7]),
			host);
	}
}
