// Contains lots of small improvements to central areas of QLang, such as vector and string handling.
// Some useful larger classes are included as well.

/**** Top Picks ****
- equaln, equaln_casei
    Test equality with null handling.

- value
    Pass around a value that can be of any primitive type.

- str_join
    The inverse of str_tokenize. Printing a vector is now easy to do!

- sub_string_start, sub_string_end
    Covers the most common uses of sub_string with ease, and can also take
    negative arguments.

- path_join
    Join directories and filenames without worrying about directory separators ('/', '\').

- object_map
    Yes, you can now have your own classes as the data in a map!


/**** CONTENTS
[Handling primitive data types]
- equaln, equaln_casei
- maxn, minn -- returns max/min even if either is null
- set_if_null
- value (class)

[Creating strings with other data]
- str_join, v_str_join
- string_generator -- takes values from map and uses with format string to create e.g. item name
- map_strs
- n_digit_string
- condensed_date, condensed_time, condensed_datetime
- xml_print
- string_builder (class)
- to_str
- concat_str_row

[String Testing and Manipulation]
- str_empty
- str_startswith, str_endswith
- sub_string_start, sub_string_end
- str_find_all, str_find_last
- replace_whitespace
- str_ensure_endswith

[Creating useful vectors and matrices]
- v_range
- empty_num_vec, empty_str_vec, empty_date_vec, empty_timestamp_vec, empty_logical_vec, empty_text_rgb_vec
- empty_num_mx, empty_str_mx, empty_date_mx, empty_timestamp_mx, empty_logical_mx, empty_text_rgb_mx
- vector of vector (vectors_str, vectors_int, vectors_num, ...)

[Vector Testing and Manipulation]
- v_equals
- v_find_starting, v_find_containing (+casei)
- v_find_eq, v_find_ne
- v_empty
- copy_sort, copy_sort_casei
- v_reverse, v_reverse_copy
- v_nulls, v_non_nulls, v_null_partition
- v_true, v_false
- v_partition, v_sizes
- v_breaks
- v_concat (returns empty vector instead of null)
- elem_mult, elem_div, elem_add, elem_sub
- v_unionize

[Path Manipulation]
- path_join, path_split
- path_normalize
- path_trim
- path_filepart, path_dirpart
- path_insert_filename_suffix
- path_add_separator

[Map and Set]
- index_map, reverse_map
- object_map (class)
- count_set, count_set_num, count_set_date (class)
- pair
- map_str_pair
- set_pair

[Extras]
- decision_table  - Express complicated if-rules as a table
- ascii_table (Easy creation of tables meant to print and look at)
- counting_alarm
- stack_str
- queue_str
- queue_str_unique
- flag
- timing
- random_order
- file_reader (a wrapper improving in_stream_file)

****/

module qu {
    public logical equaln(number option(nullable) a, number option(nullable) b) option(category: "Utilities Enhanced/Basics")
    {
        if (null(a) && null(b)) {
            return true;
        }
        if (null(a) || null(b)) {
            return false;
        }
        return a == b;
    }
    public logical equaln(string option(nullable) a, string option(nullable) b)  option(category: "Utilities Enhanced/Basics")
    {
        if (null(a) && null(b)) {
            return true;
        }
        if (null(a) || null(b)) {
            return false;
        }
        return a == b;
    }
    public logical equaln_casei(string option(nullable) a, string option(nullable) b)  option(category: "Utilities Enhanced/Basics")
    {
        if (null(a) && null(b)) {
            return true;
        }
        if (null(a) || null(b)) {
            return false;
        }
        return equal_casei(a, b);
    }
    public logical equaln(date option(nullable) a, date option(nullable) b)  option(category: "Utilities Enhanced/Basics")
    {
        if (null(a) && null(b)) {
            return true;
        }
        if (null(a) || null(b)) {
            return false;
        }
        return a == b;
    }
    public logical equaln(timestamp option(nullable) a, timestamp option(nullable) b)  option(category: "Utilities Enhanced/Basics")
    {
        if (null(a) && null(b)) {
            return true;
        }
        if (null(a) || null(b)) {
            return false;
        }
        return a == b;
    }
    public logical equaln(logical option(nullable) a, logical option(nullable) b)  option(category: "Utilities Enhanced/Basics")
    {
        if (null(a) && null(b)) {
            return true;
        }
        if (null(a) || null(b)) {
            return false;
        }
        return a == b;
    }


    public number maxn(number option(nullable) a, number option(nullable) b)
        option(category: "Utilities Enhanced/Basics")
    {
        if (!null(a) && !null(b)) {
            return ..max(a,b);
        } else if (null(b)) {
            return a;
        } else {
            return b;
        }
    }

    public number minn(number option(nullable) a, number option(nullable) b)
        option(category: "Utilities Enhanced/Basics")
    {
        if (!null(a) && !null(b)) {
            return ..min(a,b);
        } else if (null(b)) {
            return a;
        } else {
            return b;
        }
    }

    public date max(date a, date b)
        option(category: "Utilities Enhanced/Basics")
    {
        return a > b ? a : b;
    }

    public date min(date a, date b)
        option(category: "Utilities Enhanced/Basics")
    {
        return a < b ? a : b;
    }

    public date maxn(date option(nullable) a, date option(nullable) b)
        option(category: "Utilities Enhanced/Basics")
    {
        if (!null(a) && !null(b)) {
            return max(a,b);
        } else if (null(b)) {
            return a;
        } else {
            return b;
        }
    }

    public date minn(date option(nullable) a, date option(nullable) b)
        option(category: "Utilities Enhanced/Basics")
    {
        if (!null(a) && !null(b)) {
            return min(a,b);
        } else if (null(b)) {
            return a;
        } else {
            return b;
        }
    }

    public timestamp max(timestamp a, timestamp b)
        option(category: "Utilities Enhanced/Basics")
    {
        return a > b ? a : b;
    }

    public timestamp min(timestamp a, timestamp b)
        option(category: "Utilities Enhanced/Basics")
    {
        return a < b ? a : b;
    }

    public timestamp maxn(timestamp option(nullable) a, timestamp option(nullable) b)
        option(category: "Utilities Enhanced/Basics")
    {
        if (!null(a) && !null(b)) {
            return max(a,b);
        } else if (null(b)) {
            return a;
        } else {
            return b;
        }
    }

    public timestamp minn(timestamp option(nullable) a, timestamp option(nullable) b)
        option(category: "Utilities Enhanced/Basics")
    {
        if (!null(a) && !null(b)) {
            return min(a,b);
        } else if (null(b)) {
            return a;
        } else {
            return b;
        }
    }

    public void set_if_null(out number option(nullable) variable, number value) option(category: "Utilities Enhanced/Basics") {
        if (null(variable)) { variable = value; }
    }
    public void set_if_null(out logical option(nullable) variable, logical value) option(category: "Utilities Enhanced/Basics") {
        if (null(variable)) { variable = value; }
    }
    public void set_if_null(out date option(nullable) variable, date value) option(category: "Utilities Enhanced/Basics") {
        if (null(variable)) { variable = value; }
    }
    public void set_if_null(out timestamp option(nullable) variable, timestamp value) option(category: "Utilities Enhanced/Basics") {
        if (null(variable)) { variable = value; }
    }
    public void set_if_null(out string option(nullable) variable, string value) option(category: "Utilities Enhanced/Basics") {
        if (null(variable)) { variable = value; }
    }


    public enum data_type option(category: "Utilities Enhanced/Basics") {
        DT_NULL option(str: "null"),
            DT_STR option(str: "string"),
            DT_NUM option(str: "number"),
            DT_DATE option(str: "date"),
            DT_TIMESTAMP option(str: "timestamp"),
            DT_LOGICAL option(str: "logical")
    };

    data_type int_to_data_type(integer i)
    {
        switch (i) {
        case 0: return DT_NULL;
        case 1: return DT_STR;
        case 2: return DT_NUM;
        case 3: return DT_DATE;
        case 4: return DT_TIMESTAMP;
        case 5: return DT_LOGICAL;
        default: return null<data_type>;
        }
    }

    data_type str_to_data_type(string s)
    {
        integer i = 0;
        data_type v;
        while (!null(v = int_to_data_type(i++))) {
            if (s == string(v)) {
                return v;
            }
        }
        throw (E_UNSPECIFIC, strcat("Could not convert to data_type: ", s));
    }

    vector(data_type) data_types()
    {
        integer i = 0;
        data_type v;
        vector(data_type) vs[0];
        while (!null(v = int_to_data_type(i++))) {
            push_back(vs, v);
        }
        return vs;
    }


    public class value option(category: "Utilities Enhanced/Basics") {
      public:
        value();
        value(string option(nullable) x);
        value(number option(nullable) x);
        value(date option(nullable) x);
        value(timestamp option(nullable) x);
        value(logical option(nullable) x);

        logical is_null();
		void set_null();
        string repr(string null_repr = "",
                    number option(nullable) n_decimals = null<number>,
                    string true_repr = "true",
                    string false_repr = "false");

        data_type type;
        string s;
        number n;
        date d;
        timestamp t;
        logical l;
    };
    value.value() { type = DT_NULL; }
    value.value(string option(nullable) x) { s = x; type = DT_STR; }
    value.value(number option(nullable) x) { n = x; type = DT_NUM; }
    value.value(date option(nullable) x) { d = x; type = DT_DATE; }
    value.value(timestamp option(nullable) x) { t = x; type = DT_TIMESTAMP; }
    value.value(logical option(nullable) x) { l = x; type = DT_LOGICAL; }

    logical value.is_null()
    {
        switch (type) {
            case DT_NULL: return true;
            case DT_STR: return null(s);
            case DT_NUM: return null(n);
            case DT_DATE: return null(d);
            case DT_TIMESTAMP: return null(t);
            case DT_LOGICAL: return null(l);
        }
        throw (E_UNSPECIFIC, "Unhandled type");
    }
	void value.set_null()
    {
        switch (type) {
            case DT_NULL: return;
            case DT_STR: s = null<string>; break;
            case DT_NUM: n = null<number>; break;
            case DT_DATE: d = null<date>; break;
            case DT_TIMESTAMP: t = null<timestamp>; break;
            case DT_LOGICAL: l = null<logical>; break;
			default: throw (E_UNSPECIFIC, "Unhandled type");
        }
    }
    string value.repr(string null_repr, number option(nullable) n_decimals, string true_repr, string false_repr)
    {
        if (this.is_null()) {
            return null_repr;
        }
        switch (type) {
            case DT_STR: return s;
            case DT_NUM: return null(n_decimals) ? str(n) : str(n, integer(n_decimals));
            case DT_DATE: return str(d);
            case DT_TIMESTAMP: return str(t);
            case DT_LOGICAL: return l ? true_repr : false_repr;
        }
        throw (E_UNSPECIFIC, "Unhandled type");
    }

    value value_from_string(data_type dt, string option(nullable) s)
    {
        switch (dt) {
            case DT_NULL: return new value();
            case DT_STR: return new value(s);
            case DT_NUM: return new value(str_to_number(s));
            case DT_DATE: return new value(str_to_date(s));
            case DT_TIMESTAMP: return new value(str_to_timestamp(s));
            case DT_LOGICAL: return new value(str_to_logical(s));
        }
        throw (E_UNSPECIFIC, strcat("Cannot convert string to qu.value: ", s));
    }

    // Quick creation e.g. qu.qv("test") instead of new qu.value("test")
    // Also good for vector expansion
    value qv(string option(nullable) val) option(category: "Utilities Enhanced/Basics") { return new value(val); }
    value qv(number option(nullable) val) option(category: "Utilities Enhanced/Basics") { return new value(val); }
    value qv(date option(nullable) val) option(category: "Utilities Enhanced/Basics") { return new value(val); }
    value qv(timestamp option(nullable) val) option(category: "Utilities Enhanced/Basics") { return new value(val); }
    value qv(logical option(nullable) val) option(category: "Utilities Enhanced/Basics") { return new value(val); }

    value clone(value v) option(category: "Utilities Enhanced/Basics")
    {
        switch (v.type) {
            case DT_NULL: return new value();
            case DT_STR: return new value(v.s);
            case DT_NUM: return new value(v.n);
            case DT_DATE: return new value(v.d);
            case DT_TIMESTAMP: return new value(v.t);
            case DT_LOGICAL: return new value(v.l);
        }
        throw (E_UNSPECIFIC, "Unhandled value data_type");
    }

    public logical equaln(qu.value a, qu.value b) option(category: "Utilities Enhanced/Basics")
    {
        if (a.type != b.type) {
            return false;
        }
        switch (a.type) {
            case DT_NULL: return true;
            case DT_STR: return equaln(a.s, b.s);
            case DT_NUM: return equaln(a.n, b.n);
            case DT_DATE: return equaln(a.d, b.d);
            case DT_TIMESTAMP: return equaln(a.t, b.t);
            case DT_LOGICAL: return equaln(a.l, b.l);
        }
        throw (E_UNSPECIFIC, "Unhandled value data_type");
    }

    public qu.value copy(qu.value v) option(category: "Utilities Enhanced/Basics")
    {
        switch (v.type) {
            case DT_NULL: return new qu.value();
            case DT_STR: return new qu.value(v.s);
            case DT_NUM: return new qu.value(v.n);
            case DT_DATE: return new qu.value(v.d);
            case DT_TIMESTAMP: return new qu.value(v.t);
            case DT_LOGICAL: return new qu.value(v.l);
        }
        throw (E_UNSPECIFIC, "Unhandled value data_type");
    }


    public logical v_equals(vector(integer) option(nullable) v1, vector(integer) option(nullable) v2) option(category: "Utilities Enhanced/Vector Functions")
    {
        integer v1_sz = null(v1) ? 0 : v_size(v1);
        integer v2_sz = null(v2) ? 0 : v_size(v2);
        if (v1_sz != v2_sz) {
            return false;
        }
        for (integer i = 0; i < v1_sz; ++i) {
            if (v1[i] != v2[i]) {
                return false;
            }
        }
        return true;
    }
    public logical v_equals(vector(number) option(nullable) v1, vector(number) option(nullable) v2) option(category: "Utilities Enhanced/Vector Functions")
    {
        integer v1_sz = null(v1) ? 0 : v_size(v1);
        integer v2_sz = null(v2) ? 0 : v_size(v2);
        if (v1_sz != v2_sz) {
            return false;
        }
        for (integer i = 0; i < v1_sz; ++i) {
            if (!equaln(v1[i], v2[i])) {
                return false;
            }
        }
        return true;
    }
    public logical v_equals(vector(string) option(nullable) v1, vector(string) option(nullable) v2) option(category: "Utilities Enhanced/Vector Functions")
    {
        integer v1_sz = null(v1) ? 0 : v_size(v1);
        integer v2_sz = null(v2) ? 0 : v_size(v2);
        if (v1_sz != v2_sz) {
            return false;
        }
        for (integer i = 0; i < v1_sz; ++i) {
            if (!equaln(v1[i], v2[i])) {
                return false;
            }
        }
        return true;
    }
    public logical v_equals(vector(date) option(nullable) v1, vector(date) option(nullable) v2) option(category: "Utilities Enhanced/Vector Functions")
    {
        integer v1_sz = null(v1) ? 0 : v_size(v1);
        integer v2_sz = null(v2) ? 0 : v_size(v2);
        if (v1_sz != v2_sz) {
            return false;
        }
        for (integer i = 0; i < v1_sz; ++i) {
            if (!equaln(v1[i], v2[i])) {
                return false;
            }
        }
        return true;
    }

    public logical str_empty(string option(nullable) s) option(category: "Utilities Enhanced/String Manipulation") { return null(s) || strlen(s) == 0; }

    public logical str_startswith(string str, string start) option(category: "Utilities Enhanced/String Manipulation") {
        integer len = strlen(start);
        if (strlen(str) < len) {
            return false;
        }
        return equal(sub_string(str, 0, len), start);
    }
    public logical str_startswith_casei(string str, string start) option(category: "Utilities Enhanced/String Manipulation") {
        integer len = strlen(start);
        if (strlen(str) < len) {
            return false;
        }
        return equal_casei(sub_string(str, 0, len), start);
    }
    public logical str_endswith(string str, string end) option(category: "Utilities Enhanced/String Manipulation") {
        integer len = strlen(end);
        integer str_len = strlen(str);
        if (str_len < len) {
            return false;
        }
        return equal(sub_string(str, str_len-len, len), end);
    }
    public logical str_endswith_casei(string str, string end) option(category: "Utilities Enhanced/String Manipulation") {
        integer len = strlen(end);
        integer str_len = strlen(str);
        if (str_len < len) {
            return false;
        }
        return equal_casei(sub_string(str, str_len-len, len), end);
    }

    public string str_ensure_endswith(string str, string end) option(category: "Utilities Enhanced/String Manipulation") {
        if (str_endswith(str, end)) {
            return str;
        } else {
            return strcat(str, end);
        }
    }

    // Convenient sub_string for end or start of string
    /* Think of from_pos and to_pos as positions between letters. Negative
     *   positions are counted from end of string: (string = "abcd")
     *    a  b  c  d
     *   0  1  2  3  4
     *  -4 -3 -2 -1
     * sub_string_start takes string section to the left of given positon
     * sub_string_end takes string section to the right of given position
     * a position whose absolute value exceeds the length of the string
     * generates an exception (here, from_pos/to_pos must be -4 <= pos <= 4)
     * Note that for sub_string_start positive to_pos is equivalent to
     *   sub_string(s, 0, to_pos).
     * Examples:
     * sub_string_start("abcd", 2) => "ab"
     * sub_string_start("abcd", 4) => "abcd"
     * sub_string_start("abcd", -1) => "abc"
     * sub_string_start("abcd", -2) => "ab"
     * sub_string_end("abcd", 2) => "cd"
     * sub_string_end("abcd", 4) => ""
     * sub_string_end("abcd", -1) => "d"
     * sub_string_end("abcd", -2) => "cd"
     */

    public string sub_string_end(string s, integer from_pos) option(category: "Utilities Enhanced/String Manipulation")
    {
        integer n = strlen(s);
        if (from_pos == n) {
            return "";
        }
        integer actual_len;
        integer actual_from_pos;
        if (from_pos > n) {
            throw (E_UNSPECIFIC, strcat(["string '", s, "' has length ", str(n), ", can't create sub string from position ", str(from_pos)]));
        } else if (from_pos < 0) {
            actual_len = -from_pos;
            actual_from_pos = n+from_pos;
            if (actual_len > n) {
                throw (E_UNSPECIFIC, strcat(["string '", s, "' has length ", str(n), ", can't create sub string of letters after first ", str(-from_pos)]));
            }
        } else {
            actual_len = n - from_pos;
            actual_from_pos = from_pos;
        }
        return sub_string(s, actual_from_pos, actual_len);
    }

    public string sub_string_start(string s, integer to_pos) option(category: "Utilities Enhanced/String Manipulation")
    {
        integer n = strlen(s);
        if (to_pos == 0 || -to_pos == n) {
            return "";
        }
        integer actual_len;
        if (to_pos > n) {
            throw (E_UNSPECIFIC, strcat(["string '", s, "' has length ", str(n), ", can't create sub string of size ", str(to_pos)]));
        } else if (to_pos < 0) {
            actual_len = n+to_pos;
            if (actual_len < 0) {
                throw (E_UNSPECIFIC, strcat(["string '", s, "' has length ", str(n), ", can't create sub string of letters up to last ", str(-to_pos)]));
            }
        } else {
            actual_len = to_pos;
        }
        return sub_string(s, 0, actual_len);
    }

    /*
     * If /s/ is at most /to_pos/ characters long, return /s/. Otherwise return
     * /to_pos/ characters at start plus snip_suffix.
     */
    public string sub_string_start_or_all(string s, integer to_pos, string snip_suffix = "")
    {
        if (strlen(s) <= to_pos) {
            return s;
        } else {
            return strcat(sub_string(s, 0, to_pos), snip_suffix);
        }
    }

    public number str_find_last(string haystack, string needle) option(category: "Utilities Enhanced/String Manipulation")
    {
        if (needle == "") {
            return null<number>;
        }
        number local_match = str_find(haystack, needle);
        if (null(local_match)) {
            return null<number>;
        }
        integer start_pos = 0;
        number last_match;
        while (!null(local_match)) {
            last_match = integer(local_match) + start_pos;
            start_pos = integer(last_match) + 1;
            string haystack_end_part = sub_string_end(haystack, start_pos);
            local_match = str_find(haystack_end_part, needle);
        }
        return last_match;
    }

    public vector(number) str_find_all(string haystack, string needle) option(category: "Utilities Enhanced/String Manipulation")
    {
        vector(number) matches[0];
        if (needle == "") {
            return matches;
        }
        integer start_pos = 0;
        number local_match = str_find(haystack, needle);
        while (!null(local_match)) {
            push_back(matches, start_pos + local_match);
            start_pos = start_pos + integer(local_match) + 1;
            string haystack_end_part = sub_string_end(haystack, start_pos);
            local_match = str_find(haystack_end_part, needle);
        }
        return matches;
    }

    public string m_str(matrix(number) m) option(category: "Utilities Enhanced/String Conversion") {
        string s = '\n[';
        for (integer row = 0; row < n_rows(m); row++) {
            if (row != 0) s = strcat(s, '\n');
            s = strcat(s, '[');
            for (integer col = 0; col < n_cols(m); col++) {
                if (col != 0) s = strcat(s, ', ');
                string e = str(m[row,col]);
                s = strcat(s, null(e) ? '<null>' : e);
            }
            s = strcat(s, ']');
        }
        return strcat(s, ']\n');
    }
    public string m_str(matrix(string) m) option(category: "Utilities Enhanced/String Conversion") {
        string s = '\n[';
        for (integer row = 0; row < n_rows(m); row++) {
            if (row != 0) s = strcat(s, '\n');
            s = strcat(s, '[');
            for (integer col = 0; col < n_cols(m); col++) {
                if (col != 0) s = strcat(s, ', ');
                string e = m[row,col];
                s = strcat(s, null(e) ? '<null>' : strcat(['\'', e, '\'']));
            }
            s = strcat(s, ']');
        }
        return strcat(s, ']\n');
    }
    public string m_str(matrix(date) m) option(category: "Utilities Enhanced/String Conversion") {
        string s = '\n[';
        for (integer row = 0; row < n_rows(m); row++) {
            if (row != 0) s = strcat(s, '\n');
            s = strcat(s, '[');
            for (integer col = 0; col < n_cols(m); col++) {
                if (col != 0) s = strcat(s, ', ');
                string e = str(m[row,col]);
                s = strcat(s, null(e) ? '<null>' : e);
            }
            s = strcat(s, ']');
        }
        return strcat(s, ']\n');
    }


    public string str_join(vector(string) parts, string option(nullable) delimiter = null<string>) option(category: "Utilities Enhanced/String Manipulation") {
        string joined_str;
        integer n = v_size(parts);
        if (null(delimiter) || n < 2) {
            joined_str = strcat(parts);
        } else {
            vector(string) join_parts[2*n-1];
            join_parts[0] = parts[0];
            for (integer i = 1; i < n; ++i) {
                join_parts[i*2-1] = delimiter;
                join_parts[i*2] = parts[i];
            }
            joined_str = strcat(join_parts);
        }
        return joined_str;
    }

    public vector(string) v_str_join(matrix(string) mx, string option(nullable) delimiter = null<string>) option(category: "Utilities Enhanced/String Manipulation")
    {
        integer m = n_cols(mx);
        vector(string) joined[m];
        for (integer i = 0; i < m; i++) {
            joined[i] = str_join(mx[:,i], delimiter);
        }
        return joined;
    }

    public vector(string) v_str_join(vector(string) v1, vector(string) v2, string option(nullable) delimiter = null<string>) option(category: "Utilities Enhanced/String Manipulation")
    {
        integer n1 = v_size(v1);
        integer n2 = v_size(v2);
        integer n_min = ..min(n1, n2);
        integer n_max = ..max(n1, n2);
        vector(string) joined[n_max];
        for (integer i = 0; i < n_min; i++) {
            if (null(delimiter)) {
                joined[i] = strcat(v1[i], v2[i]);
            } else {
                joined[i] = strcat([v1[i], delimiter, v2[i]]);
            }
        }
        if (n1 > n_min) {
            for (integer i = n_min; i < n_max; i++) {
                if (null(delimiter)) {
                    joined[i] = v1[i];
                } else {
                    joined[i] = strcat(v1[i], delimiter);
                }
            }
        } else if (n2 > n_min) {
            for (integer i = n_min; i < n_max; i++) {
                if (null(delimiter)) {
                    joined[i] = v2[i];
                } else {
                    joined[i] = strcat(delimiter, v2[i]);
                }
            }
        }
        return joined;
    }

    public vector(string) v_str_full_join(vector(string) v1, vector(string) v2, string option(nullable) delimiter = null<string>) option(category: "Utilities Enhanced/String Manipulation")
    {
        integer n1 = v_size(v1);
        integer n2 = v_size(v2);
        vector(string) joined[n1*n2];
        integer k = 0;
        for (integer i = 0; i < n1; i++) {
            for (integer j = 0; j < n2; j++) {
                if (null(delimiter)) {
                    joined[k++] = strcat(v1[i], v2[j]);
                } else {
                    joined[k++] = strcat([v1[i], delimiter, v2[j]]);
                }
            }
        }
        return joined;
    }


    public string n_digit_string(number x, integer n_digits) option(category: "Utilities Enhanced/String Conversion") {
        number threshold = pow(10, n_digits);
        if (x < 0 || x >= threshold) {
            throw (E_UNSPECIFIC, strcat(['Cannot turn ', str(x), ' into ', str(n_digits), ' digits.']));
        }

        x = floor(x); // lose the decimals

        // Special case x = 0, algorithm below won't work
        if (x == 0) {
            string s = '';
            for (integer i = 0; i < n_digits; i++) {
                s = strcat(s, '0');
            }
            return s;
        }

        string new_str = str(x);

        // pad with zeroes
        number scaled_x = x * 10;
        while (scaled_x < threshold) {
            new_str = strcat(['0', new_str]);
            scaled_x = scaled_x * 10;
        }

        return new_str;
    }

    public string replace_whitespace(string s, string replacement) option(category: "Utilities Enhanced/String Manipulation") {
        string new_s = s;
        new_s = str_replace(new_s, ' ', replacement);
        new_s = str_replace(new_s, '\t', replacement);
        new_s = str_replace(new_s, '\r\n', replacement);
        new_s = str_replace(new_s, '\r', replacement);
        new_s = str_replace(new_s, '\n', replacement);
        return new_s;
    }

    public string condensed_date(date d) option(category: "Utilities Enhanced/String Conversion")
    {
        string ds = str(d);
        return strcat([sub_string(ds,2,2), sub_string(ds,5,2), sub_string(ds,8,2)]);
    }
    public string condensed_time(timestamp ts, logical seconds = true, string option(nullable) separator = null<string>) option(category: "Utilities Enhanced/String Conversion")
    {
        number h = hour(ts);
        string hs = n_digit_string(h,2);
        number m = minute(ts);
        string ms = n_digit_string(m,2);
        string ss = "";
        if (seconds) {
            number s = second(ts);
            ss = n_digit_string(s,2);
        }
        return str_join([hs, ms, ss], separator);
    }
    public string condensed_datetime(timestamp ts, logical seconds = true) option(category: "Utilities Enhanced/String Conversion") {
        return strcat([condensed_date(date(ts)), "_", condensed_time(ts, seconds)]);
    }


    /*
     * Strings are generated from fields and constant strings. There are two
     * formats:
     * (A): Comma separated list of fields or strings. Strings are quoted
     *      with ". The generated string is the values and strings joined with
     *      an underscore (making good item names).
     * (B): Free format. The format string surrounds fields with %, and the
     *      rest is used as given.
     * So (A) is simple and (B) is flexible.
     *
     * Examples (IdCode = 23432 and DueDate = 2015-03-04):
     * "BOND", IdCode, "DUE", DueDate
     * Result: BOND_23432_DUE_2015-03-04
     *
     * BOND %IdCode%-Due:%DueDate%
     * Result: BOND 23432-Due:2015-03-04
     *
     * If the format string contains any % sign, it will be interpreted as (B),
     * otherwise as (A). If you have an explicit % character in a field name or
     * in a constant string, you can provide another character to function as
     * %. If you do, format_string is always interpreted as (B).
     *
     */

    public class string_generator option(category: "Utilities Enhanced/String Manipulation") {
    public:
        string_generator(string format_string);
        string_generator(string format_string, string field_character);

        vector(string) fields();
        string format_string();
        string field_character();

        string generate(map_str_str field_to_val);
        string generate_accept_null(map_str_str field_to_val);

    private:
        string field_char_; // not used internally, but can be returned
        string format_string_; // not used internally, but can be returned
        vector(string) string_parts_;
        vector(integer) field_positions_;
        vector(string) field_names_;

        void _init_list_format(string format_string);
        void _init_free_format(string format_string);
        string _generate(map_str_str field_to_val, logical accept_null);
    };
    string_generator.string_generator(string format_string)
    {
        string default_field_char = "%";
        format_string_ = format_string;

        if (!null(str_find(format_string, default_field_char))) {
            field_char_ = default_field_char;
            _init_free_format(format_string);
        } else {
            _init_list_format(format_string);
        }
    }
    string_generator.string_generator(string format_string, string field_character)
    {
        if (strlen(field_character) != 1) {
            throw (E_UNSPECIFIC, strcat(["field_character must be exactly one character and '", field_character, "' is not"]));
        }
        field_char_ = field_character;
        format_string_ = format_string;
        _init_free_format(format_string);
    }

    string string_generator.generate(map_str_str field_to_val)
    {
        return _generate(field_to_val, false);
    }

    string string_generator.generate_accept_null(map_str_str field_to_val)
    {
        return _generate(field_to_val, true);
    }

    vector(string) string_generator.fields() = clone_vector(field_names_);
    string string_generator.format_string() = format_string_;
    string string_generator.field_character() = field_char_;

    void string_generator._init_list_format(string format_string)
    {
        vector(string) format_units = str_trim(str_tokenize(format_string, ","));
        integer n = v_size(format_units);
        resize(string_parts_, 0);
        resize(field_positions_, 0);
        resize(field_names_, 0);

        for (integer i = 0; i < n; ++i) {
            string unit = format_units[i];
            if (str_startswith(unit, '"') && str_endswith(unit, '"')) {
                push_back(string_parts_, sub_string(unit, 1, strlen(unit)-2));
            } else {
                push_back(string_parts_, null<string>);
                push_back(field_positions_, v_size(string_parts_)-1);
                push_back(field_names_, unit);
            }
            if (i < n-1) {
                push_back(string_parts_, "_");
            }
        }
    }

    void string_generator._init_free_format(string format_string)
    {
        vector(string) format_units = str_tokenize(format_string, field_char_);
        integer n = v_size(format_units);
        resize(string_parts_, 0);
        resize(field_positions_, 0);
        resize(field_names_, 0);

        logical currently_field = false;
        for (integer i = 0; i < n; ++i) {
            string unit = format_units[i];
            if (unit != "") {
                if (currently_field) {
                    push_back(string_parts_, null<string>);
                    push_back(field_positions_, v_size(string_parts_)-1);
                    push_back(field_names_, unit);
                } else {
                    push_back(string_parts_, unit);
                }
            }
            currently_field = !currently_field;
        }
        // Good format strings end with non-field or field+%, so last unit should always be non-field,
        // thus /currently_field/ will be set to true as last action of loop above.
        if (!currently_field) {
            throw (E_UNSPECIFIC, strcat(["Malformatted format string '", format_string, "'"]));
        }
    }

    string string_generator._generate(map_str_str field_to_val, logical accept_null)
    {
        vector(string) elems = clone_vector(string_parts_);
        integer field_positions__sz = v_size(field_positions_);
        for (integer i = 0; i < field_positions__sz; ++i) {
            integer pos = field_positions_[i];
            string fld = field_names_[i];
            string val = field_to_val.find(fld);
            if (null(val)) {
                if (accept_null) {
                    val = "";
                } else {
                    throw (E_UNSPECIFIC, strcat(["No value for '", fld, "' -- cannot generate string"]));
                }
            }
            elems[pos] = val;
        }
        return strcat(elems);
    }

    xml_node find_text_child(vector(xml_node) option(nullable) children)
    {
        if (v_size(children) == 1 && children[0].is_text()) {
            return children[0];
        } else {
            return null<xml_node>;
        }
    }

    string xmlify(string s)
    {
        s = str_replace(s, "&", "&amp;");
        s = str_replace(s, "'", "&apos;");
        s = str_replace(s, '"', "&quot;");
        s = str_replace(s, ">", "&gt;");
        s = str_replace(s, "<", "&lt;");
        return s;
    }


    void print_tag_content(out_stream_string ss, xml_node node)
    {
        ss.write(node.tag());
        vector(string) attrs;
        vector(string) values;
        node.get_attribs(attrs, values);
        integer attrs_sz = v_size(attrs);
        for (integer i = 0; i < attrs_sz; i++) {
            string attr = attrs[i];
            string val = values[i];
            ss.write(strcat([" ", attr, "=\"", xmlify(val), "\""]));
        }
    }

    void print_tag(out_stream_string ss, xml_node node)
    {
        ss.write(node.tag());
    }
    void xml_print(out_stream_string ss, xml_node node, string indent, string current_indent, string newline) option(category: "Utilities Enhanced/XML")
    {
        ss.write(current_indent);
        vector(xml_node) children = node.children();
        integer nch = v_size(children);
            ss.write('<');

        print_tag_content(ss, node);

        xml_node text_child = find_text_child(children);
        if (!null(text_child)) {
            ss.write('>');
            ss.write(xmlify(text_child.text()));
            ss.write('</');
            print_tag(ss, node);
            ss.write(strcat('>', newline));
        } else if (nch == 0) {
            ss.write(strcat('/>', newline));
        } else {
            ss.write(strcat('>', newline));
            string next_indent = strcat(current_indent, indent);
            for (integer i = 0; i < nch; i++) {
                xml_print(ss, children[i], indent, next_indent, newline);
            }
            ss.write(current_indent);
            ss.write('</');
            print_tag(ss, node);
            ss.write(strcat('>', newline));
        }
    }

    public string xml_print(xml_node node, string indent='\t', string newline='\r\n') option(category: "Utilities Enhanced/XML")
    {
        out_stream_string ss = out_stream_string();
        xml_print(ss, node, indent, "", newline);
        return ss.get_string();
    }

    public class string_builder option(category: "Utilities Enhanced/String Conversion") {
      public:
        string_builder(string option(nullable) separator = null<string>);
        string get_string();
        void add(string option(nullable) val);
        void add(number option(nullable) val, number option(nullable) n_decimals = null<number>);
        void add(date option(nullable) val);
        void add(logical option(nullable) val);
        void add(timestamp option(nullable) val);
        void set_separator(string option(nullable) separator);
        void set_logical_repr(string true_repr, string false_repr);
        void set_null_repr(string null_repr);
        void set_decimal_comma(logical comma_instead_of_point);
		void set_date_format(string fmt);
		void set_timestamp_format(string fmt);

      private:
        string list_; // null if str_vec_ is dirty, and this must be recreated
        vector(string) str_vec_;
        logical use_separator_;
        string separator_;
        string true_repr_;
        string false_repr_;
        string null_repr_;
		string date_fmt_;
		string timestamp_fmt_;
        logical comma_instead_of_point_;

        void _add_element(string option(nullable) elem);
    };
    string_builder.string_builder(string option(nullable) separator)
    {
        list_ = null<string>;
        resize(str_vec_, 0);
        true_repr_ = "true";
        false_repr_ = "false";
        null_repr_ = "";
        comma_instead_of_point_ = false;
        this.set_separator(separator);
    }
    string string_builder.get_string()
    {
        if (null(list_)) {
            list_ = strcat(str_vec_);
        }
        return list_;
    }
    void string_builder.add(string option(nullable) val) {
        _add_element(val);
    }
    void string_builder.add(number option(nullable) val, number option(nullable) n_decimals) {
        string repr;
        if (null(n_decimals)) {
            repr = str(val);
        } else {
            repr = str(val, integer(n_decimals));
        }
        if (comma_instead_of_point_) {
            repr = str_replace(repr, '.', ',');
        }
        _add_element(repr);
    }
    void string_builder.add(date option(nullable) val) { _add_element(null(date_fmt_) ? str(val) : date_to_str(val, date_fmt_)); }
    void string_builder.add(logical option(nullable) val) {
        if (null(val)) {
            _add_element(null<string>);
        } else {
            _add_element(val ? true_repr_ : false_repr_);
        }
    }
    void string_builder.add(timestamp option(nullable) val) { _add_element(null(timestamp_fmt_) ? str(val) : timestamp_to_str(val, timestamp_fmt_)); }
    void string_builder.set_separator(string option(nullable) separator)
    {
        separator_ = separator;
        use_separator_ = !null(separator_);
    }
    void string_builder.set_logical_repr(string true_repr, string false_repr) {
        true_repr_ = true_repr;
        false_repr_ = false_repr;
    }
    void string_builder.set_null_repr(string null_repr) { null_repr_ = null_repr; }
    void string_builder.set_decimal_comma(logical comma_instead_of_point) { comma_instead_of_point_ = comma_instead_of_point; }
	void string_builder.set_date_format(string fmt) { date_fmt_ = fmt; }
	void string_builder.set_timestamp_format(string fmt) { timestamp_fmt_ = fmt; }
    void string_builder._add_element(string option(nullable) elem)
    {
        if (null(elem)) {
            elem = null_repr_;
        }
        if (v_size(str_vec_) > 0) {
            push_back(str_vec_, separator_);
        }
        push_back(str_vec_, elem);

        list_ = null<string>;
    }

    string to_str(string option(nullable) x) option(category: "Utilities Enhanced/String Conversion") { return x; }
    string to_str(number option(nullable) x) option(category: "Utilities Enhanced/String Conversion") { return str(x); }
    string to_str(date option(nullable) x) option(category: "Utilities Enhanced/String Conversion") { return str(x); }
    string to_str(timestamp option(nullable) x) option(category: "Utilities Enhanced/String Conversion") { return str(x); }
    string to_str(logical option(nullable) x, string option(nullable) null_repr = null<string>, string true_repr = "true", string false_repr = "false")
        option(category: "Utilities Enhanced/String Conversion")
    {
        return null(x) ? null_repr : x ? true_repr : false_repr;
    }
    string logical_str(logical option(nullable) x, string option(nullable) null_repr = null<string>, string true_repr = "true", string false_repr = "false")
        option(category: "Utilities Enhanced/String Conversion")
    {
        return to_str(x, null_repr, true_repr, false_repr);
    }
    logical str_to_logical_multi(string s, vector(string) true_vals, vector(string) false_vals, logical casei)
    {
        integer true_vals_sz = v_size(true_vals);
        for (integer i = 0; i < true_vals_sz; ++i) {
            if (casei) {
                if (equaln_casei(true_vals[i], s)) {
                    return true;
                }
            } else {
                if (equaln(true_vals[i], s)) {
                    return true;
                }
            }
        }

        integer false_vals_sz = v_size(false_vals);
        for (integer i = 0; i < false_vals_sz; ++i) {
            if (casei) {
                if (equaln_casei(false_vals[i], s)) {
                    return false;
                }
            } else {
                if (equaln(false_vals[i], s)) {
                    return false;
                }
            }
        }

        throw (E_UNSPECIFIC, strcat(["Cannot convert ", s, " to logical"]));
    }

    string human_time(number ms)
    {
        if (ms < 1000) {
            return strcat(str(ms, 0), " ms");
        } else if (ms < 60000) {
            return strcat(str(ms/1000, 3), " s");
        } else if (ms < 3600000) {
            return strcat(str(ms/60000, 3), " min");
        } else {
            return strcat(str(ms/3600000, 3), " h");
        }
    }

    string str_to_fixlen(string s, integer len, string option(nullable) justification)
    {
        integer slen = strlen(s);
        string result;
        if (slen < len) {
            integer n_spaces = len - slen;
            integer space_mid = n_spaces % 2 == 0 ? integer(n_spaces/2) : integer((n_spaces+1)/2);
            vector(string) spaces[n_spaces];
            spaces = " ";
            string space_str = strcat(spaces);
            if (null(justification) || equaln_casei(justification, "l") || equaln_casei(justification, "left")) {
                result = strcat(s, space_str);
            } else if (equaln_casei(justification, "r") || equaln_casei(justification, "right")) {
                result = strcat(space_str, s);
            } else if (equaln_casei(justification, "c") || equaln_casei(justification, "center")) {
                string left_spaces = sub_string(space_str, 0, space_mid);
                string right_spaces = sub_string(space_str, space_mid, n_spaces-space_mid);
                result = strcat([left_spaces, s, right_spaces]);
            } else {
                throw (E_UNSPECIFIC, strcat(["Justification '", justification, "' is not supported"]));
            }
        } else if (slen > len) {
            result = sub_string(s, 0, len);
        } else {
            result = s;
        }
        return result;
    }

    string concat_str_row(
        vector(string) strings,
        vector(integer) widths,
        vector(string) justifications, // vector of size one => use element for all strings
        integer row_length = -1, // -1 => don't trim or pad final string
        string option(nullable) delimiter = null<string>)
    {
        integer n = v_size(strings);

        if (v_size(justifications) == 1) {
            string justification = justifications[0];
            resize(justifications, n);
            justifications = justification;
        }

        if (v_size(widths) != n || v_size(justifications) != n) {
            throw (E_UNSPECIFIC, "Mismatching vectors passed to concat_str_row");
        }
        vector(string) padded_strings[n];
        for (integer i = 0; i < n; ++i) {
            integer width = widths[i] == -1 ? strlen(strings[i]) : widths[i];
            string padded_str;
            if (null(strings[i])) {
                vector(string) spaces[width];
                spaces = " ";
                padded_str = strcat(spaces);
            } else {
                padded_str = str_to_fixlen(strings[i], width, justifications[i]);
            }
            padded_strings[i] = padded_str;
        }
        string untrimmed_result = str_join(padded_strings, delimiter);
        if (row_length == -1) {
            return untrimmed_result;
        } else {
            return str_to_fixlen(untrimmed_result, row_length, "left");
        }
    }


    // **** path manipulation
    string __std_separator_list__ = "\\/";
    integer __PATH_ALL_IS_DIR__ = -1;
    integer __PATH_ALL_IS_FILE__ = -2;
    integer __PATH_EMPTY__ = -3;
    vector(string) __PATH_INVALID_FILENAME_CHARS__ = ['/', '\\', '?', '%', '*', ':', '|', '"', '<', '>'];
    vector(string) __PATH_INVALID_FILENAME_ENDS__ = [' ', '.'];


    // Replaces some common invalid characters by underscore. Does not handle control characters.
    public string path_make_filename(string s)
    {
        integer __PATH_INVALID_FILENAME_CHARS___sz = v_size(__PATH_INVALID_FILENAME_CHARS__);
        for (integer i = 0; i < __PATH_INVALID_FILENAME_CHARS___sz; i++) {
            s = str_replace(s, __PATH_INVALID_FILENAME_CHARS__[i], "_");
        }
        integer __PATH_INVALID_FILENAME_ENDS___sz = v_size(__PATH_INVALID_FILENAME_ENDS__);
        for (integer i = 0; i < __PATH_INVALID_FILENAME_ENDS___sz; i++) {
            if (str_endswith(s, __PATH_INVALID_FILENAME_ENDS__[i])) {
                s = strcat(sub_string_start(s, -1), "_");
            }
        }
        return s;
    }


    // Joins the two paths first and second, by removing separator at end of first or beginning of second,
    // and then joining them with the first character in separators.
    public string path_join(string first, string second, string separators = __std_separator_list__) option(category: "Utilities Enhanced/Path")
    {
        if (strlen(separators) < 1) {
            throw (E_CONSTRAINT, "path_join called without separators");
        }
        integer flen = strlen(first);
        integer slen = strlen(second);
        if (flen > 0) {
            string end_of_first = sub_string(first, flen-1, 1);
            if (find_string(separators, end_of_first)) {
                first = sub_string(first, 0, flen-1);
                flen--;
            }
        }
        if (slen > 0) {
            string start_of_second = sub_string(second, 0, 1);
            if (find_string(separators, start_of_second)) {
                second = sub_string(second, 1, slen-1);
                slen--;
            }
        }
        if (flen == 0 || slen == 0) {
            return strcat(first, second);
        } else {
            return strcat([first, sub_string(separators, 0, 1), second]);
        }
    }
    // Normalizes a path by changing all separators to the first character in separators
    public string path_normalize(string path, string separators = __std_separator_list__) option(category: "Utilities Enhanced/Path")
    {
        integer seplen = strlen(separators);
        if (seplen == 0) {
            throw (E_CONSTRAINT, "path_normalize called without separators");
        }
        string main_sep = sub_string(separators, 0, 1);
        for (integer i = 1; i < seplen; ++i) {
            string other_sep = sub_string(separators, i, 1);
            path = str_replace(path, other_sep, main_sep);
        }
        return path;
    }

    // Remove trailing path separator
    public string path_trim(string path, string separators = __std_separator_list__) option(category: "Utilities Enhanced/Path")
    {
        integer pathlen = strlen(path);
        if (pathlen > 0) {
            string last_char = sub_string(path, pathlen-1, 1);
            if (find_string(separators, last_char)) {
                path = sub_string(path, 0, pathlen-1);
            }
        }
        return path;
    }

    // Ensure trailing path separator exists
    public string path_add_separator(string path, string separators = __std_separator_list__) option(category: "Utilities Enhanced/Path")
    {
        integer seplen = strlen(separators);
        for (integer i = 0; i < seplen; ++i) {
            if (str_endswith(path, sub_string(separators, i, 1))) {
                return path;
            }
        }
        return strcat(path, sub_string(separators, 0, 1));
    }

    // special return values:
    // __PATH_ALL_IS_DIR__
    // __PATH_ALL_IS_FILE__
    // __PATH_EMPTY__
    // Check these first, if neithes, return value is a valid position in path
    integer __find_last_separator(string path, string separators)
    {
        // Normalize for easier treatment
        string main_sep = sub_string(separators, 0, 1);
        path = path_normalize(path, separators);
        number sep_index = -1; // assume separator before start of path
        integer path_len = strlen(path);
        if (path_len == 0) {
            return __PATH_EMPTY__;
        }
        integer len = path_len; // modified in loop
        string file_part = path; // modified in loop
        while (len >= 0 && !null(sep_index)) {
            if (len == 0) {
                return __PATH_ALL_IS_DIR__;
            } else {
                file_part = sub_string(file_part, integer(sep_index)+1, len);
            }
            sep_index = str_find(file_part, main_sep);
            if (!null(sep_index)) {
                len -= integer(sep_index) + 1;
            }
        }

        integer sep_position = path_len - strlen(file_part) - 1;
        if (sep_position < 0) {
            return __PATH_ALL_IS_FILE__;
        } else {
            return sep_position;
        }
    }

    public void path_split(string path, out string dirpart, out string filepart, string separators = __std_separator_list__) option(category: "Utilities Enhanced/Path")
    {
        integer seplen = strlen(separators);
        if (seplen == 0) {
            throw (E_CONSTRAINT, "path_split called without separators");
        }
        integer seppos = __find_last_separator(path, separators);
        if (seppos == __PATH_EMPTY__) {
            dirpart = "";
            filepart = "";
        } else if (seppos == __PATH_ALL_IS_DIR__) {
            dirpart = sub_string(path, 0, strlen(path)-1); // remove trailing separator
            filepart = "";
        } else if (seppos == __PATH_ALL_IS_FILE__) {
            dirpart = "";
            filepart = path;
        } else {
            dirpart = sub_string(path, 0, seppos);
            filepart = sub_string(path, seppos+1, strlen(path)-seppos-1);
        }
    }


    // Extracts file name part, i.e. part of path after last separator
    public string path_filepart(string path, string separators = __std_separator_list__) option(category: "Utilities Enhanced/Path")
    {
        integer seplen = strlen(separators);
        if (seplen == 0) {
            throw (E_CONSTRAINT, "path_filepart called without separators");
        }
        string dp, fp;
        path_split(path, dp, fp, separators);
        return fp;
    }

    // Extracts directory part, i.e. part of path before last separator
    public string path_dirpart(string path, string separators = __std_separator_list__) option(category: "Utilities Enhanced/Path")
    {
        integer seplen = strlen(separators);
        if (seplen == 0) {
            throw (E_CONSTRAINT, "path_dirpart called without separators");
        }
        string dp, fp;
        path_split(path, dp, fp, separators);
        return dp;
    }

    public string path_insert_filename_suffix(string filename, string suffix) option(category: "Utilities Enhanced/Path")
    {
        number dot_pos;
        number next_dot_pos = str_find(filename, '.');
        while (!null(next_dot_pos)) {
            dot_pos = next_dot_pos;
            next_dot_pos = dot_pos + 1 + str_find(sub_string_end(filename, integer(dot_pos)+1), '.');
        }
        if (null(dot_pos)) {
            return strcat(filename, suffix);
        } else {
            return strcat([sub_string_start(filename, integer(dot_pos)), suffix, sub_string_end(filename, integer(dot_pos))]);
        }
    }

    public logical v_empty(vector(number) option(nullable) v) option(category: "Utilities Enhanced/Vector Functions") { return null(v) || v_size(v) == 0; }
    public logical v_empty(vector(string) option(nullable) v) option(category: "Utilities Enhanced/Vector Functions") { return null(v) || v_size(v) == 0; }
    public logical v_empty(vector(date) option(nullable) v) option(category: "Utilities Enhanced/Vector Functions") { return null(v) || v_size(v) == 0; }
    public logical v_empty(vector(timestamp) option(nullable) v) option(category: "Utilities Enhanced/Vector Functions") { return null(v) || v_size(v) == 0; }
    public logical v_empty(vector(logical) option(nullable) v) option(category: "Utilities Enhanced/Vector Functions") { return null(v) || v_size(v) == 0; }
    public logical v_empty(vector(text_rgb) option(nullable) v) option(category: "Utilities Enhanced/Vector Functions") { return null(v) || v_size(v) == 0; }
    public logical v_empty(vector(object<1>) option(nullable) v) option(category: "Utilities Enhanced/Vector Functions") { return null(v) || v_size(v) == 0; }

    public number elem_mult(number a, number b) option(category: "Utilities Enhanced/Vector Functions") = a*b;
    public number elem_div(number a, number b) option(category: "Utilities Enhanced/Vector Functions") = a/b;
    public number elem_add(number a, number b) option(category: "Utilities Enhanced/Vector Functions") = a+b;
    public number elem_sub(number a, number b) option(category: "Utilities Enhanced/Vector Functions") = a-b;


    // *** empty_x -- returns empty vectors and matrices
    public vector(integer) empty_int_vec() option(category: "Utilities Enhanced/Vector Functions") { vector(integer) v[0]; return v; }
    public vector(number) empty_num_vec() option(category: "Utilities Enhanced/Vector Functions") { vector(number) v[0]; return v; }
    public vector(date) empty_date_vec() option(category: "Utilities Enhanced/Vector Functions") { vector(date) v[0]; return v; }
    public vector(timestamp) empty_timestamp_vec() option(category: "Utilities Enhanced/Vector Functions") { vector(timestamp) v[0]; return v; }
    public vector(string) empty_str_vec() option(category: "Utilities Enhanced/Vector Functions") { vector(string) v[0]; return v; }
    public vector(logical) empty_logical_vec() option(category: "Utilities Enhanced/Vector Functions") { vector(logical) v[0]; return v; }
    public vector(text_rgb) empty_text_rgb_vec() option(category: "Utilities Enhanced/Vector Functions") { vector(text_rgb) v[0]; return v; }

    public matrix(number) empty_int_mx() option(category: "Utilities Enhanced/Vector Functions") { matrix(integer) m[0,0]; return m; }
    public matrix(number) empty_num_mx() option(category: "Utilities Enhanced/Vector Functions") { matrix(number) m[0,0]; return m; }
    public matrix(date) empty_date_mx() option(category: "Utilities Enhanced/Vector Functions") { matrix(date) m[0,0]; return m; }
    public matrix(timestamp) empty_timestamp_mx() option(category: "Utilities Enhanced/Vector Functions") { matrix(timestamp) m[0,0]; return m; }
    public matrix(string) empty_str_mx() option(category: "Utilities Enhanced/Vector Functions") { matrix(string) m[0,0]; return m; }
    public matrix(logical) empty_logical_mx() option(category: "Utilities Enhanced/Vector Functions") { matrix(logical) m[0,0]; return m; }
    public matrix(text_rgb) empty_text_rgb_mx() option(category: "Utilities Enhanced/Vector Functions") { matrix(text_rgb) m[0,0]; return m; }

    public class v_compare_str
        option(category: "Utilities Enhanced/Vector Functions")
    {
    public:
        v_compare_str(vector(string) option(nullable),
                      vector(string) option(nullable),
                      vector(string) option(nullable));
        vector(string) only_in_left;
        vector(string) only_in_right;
        vector(string) in_both;
    };

    v_compare_str.v_compare_str(vector(string) option(nullable) l,
                                vector(string) option(nullable) r,
                                vector(string) option(nullable) in_both)
        : only_in_left(l), only_in_right(r), in_both(in_both)
    {}

    // Overload the standard strcmp function to handle strings which are null.
    // Note that "some string" < null<string> because that is how the sort
    // function considers the elements.
    number strcmp(string option(nullable) l, string option(nullable) r) {
      return null(l) && null(r) ? 0
           : null(l)            ? 1
           : null(r)            ? -1
                                : ..strcmp(l, r);
    }

    integer find_next_differing_entry(vector(string) v, integer start, integer size)
    {
        integer i = start;
        while (++i < size && equaln(v[i], v[start]));
        return i;
    }

    // This function expects the input to be sorted to work correctly. If one
    // wants to compare two unsorted vectors, use v_compare which sorts
    // the vectors (after cloning them) and then calls this function.
    public v_compare_str v_compare_sorted(vector(string) option(nullable) left,
                                          vector(string) option(nullable) right)
        option(category: "Utilities Enhanced/Vector Functions")
    {
        integer n_left = v_size(left);
        integer n_right = v_size(right);
        vector(string) only_in_left[0];
        vector(string) only_in_right[0];
        vector(string) in_both[0];
        integer i = 0;
        integer j = 0;
        while (i < n_left && j < n_right) {
            string l = left[i];
            string r = right[j];
            number cmp = strcmp(l, r);
            if (cmp < 0) {
                push_back(only_in_left, l);
                i = find_next_differing_entry(left, i, n_left);
            }
            else if (cmp > 0) {
                push_back(only_in_right, r);
                j = find_next_differing_entry(right, j, n_right);
            }
            else {
                push_back(in_both, l);
                i = find_next_differing_entry(left, i, n_left);
                j = find_next_differing_entry(right, j, n_right);
            }
        }
        while (i < n_left) {
            push_back(only_in_left, left[i]);
            i = find_next_differing_entry(left, i, n_left);
        }
        while (j < n_right) {
            push_back(only_in_right, right[j]);
            j = find_next_differing_entry(right, j, n_right);
        }
        return new v_compare_str(only_in_left, only_in_right, in_both);
    }

    public v_compare_str v_compare(vector(string) option(nullable) left,
                                   vector(string) option(nullable) right)
        option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(string) left_clone = clone_vector(left);
        vector(string) right_clone = clone_vector(right);
        sort(left_clone);
        sort(right_clone);
        return v_compare_sorted(left_clone, right_clone);
    }

    // Find occurences of string part in vector
    public vector(integer) v_find_starting(vector(string) v, string str) option(category: "Utilities Enhanced/Sort and Search")
    {
        vector(integer) matches[0];
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (!null(v[i]) && str_startswith(v[i], str)) {
                push_back(matches, i);
            }
        }
        return matches;
    }

    public vector(integer) v_find_starting_casei(vector(string) v, string str) option(category: "Utilities Enhanced/Sort and Search")
    {
        vector(integer) matches[0];
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (!null(v[i]) && str_startswith_casei(v[i], str)) {
                push_back(matches, i);
            }
        }
        return matches;
    }

    public vector(integer) v_find_containing(vector(string) v, string str) option(category: "Utilities Enhanced/Sort and Search")
    {
        vector(integer) matches[0];
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (!null(v[i]) && find_string(v[i], str)) {
                push_back(matches, i);
            }
        }
        return matches;
    }

    public vector(integer) v_find_containing_casei(vector(string) v, string str) option(category: "Utilities Enhanced/Sort and Search")
    {
        vector(integer) matches[0];
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (!null(v[i]) && !null(str_find_casei(v[i], str))) {
                push_back(matches, i);
            }
        }
        return matches;
    }

    // Get indices of elements in v equal to n (null safe)
    public vector(integer) v_find_eq(vector(number) v, number option(nullable) x) option(category: "Utilities Enhanced/Sort and Search")
    {
        vector(integer) matches[0];
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (equaln(x, v[i])) {
                push_back(matches, i);
            }
        }
        return matches;
    }
    // Get indices of elements in v not equal to n (null safe)
    public vector(integer) v_find_ne(vector(number) v, number option(nullable) x) option(category: "Utilities Enhanced/Sort and Search")
    {
        vector(integer) matches[0];
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (!equaln(x, v[i])) {
                push_back(matches, i);
            }
        }
        return matches;
    }


    public vector(number) v_cum_sum(vector(number) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        integer n = v_size(v);
        vector(number) sv[n];
        number sum = 0;
        for (integer i = 0; i < n; ++i) {
            sum += v[i];
            sv[i] = sum;
        }
        return sv;
    }

    public vector(number) v_cum_prod(vector(number) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        integer n = v_size(v);
        vector(number) pv[n];
        number prod = 1;
        for (integer i = 0; i < n; ++i) {
            prod *= v[i];
            pv[i] = prod;
        }
        return pv;
    }

    public vector(integer) v_range(integer n) option(category: "Utilities Enhanced/Vector Functions") {
        if (n > 0) {
            return integer(s2v(series(i: 0, n-1, 1; i)));
        } else if (n == 0) {
            return empty_int_vec();
        } else {
            return integer(-s2v(series(i: 0, n-1, 1; i)));
        }
    }
    public vector(integer) v_range(integer from_incl, integer to_excl) option(category: "Utilities Enhanced/Vector Functions") {
        integer n = to_excl - from_incl;
        if (n > 0) {
            return integer(s2v(series(i: from_incl, to_excl-1, 1; i)));
        } else if (n == 0) {
            return empty_int_vec();
        } else {
            return integer(s2v(series(i: from_incl, to_excl-1, -1; i)));
        }
    }


    public void v_reverse(out vector(integer) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer lo = 0;
        integer hi = v_size(v)-1;
        while (lo < hi) {
            integer tmp = v[hi];
            v[hi] = v[lo];
            v[lo] = tmp;
            lo++;
            hi--;
        }
    }
    public void v_reverse(out vector(number) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer lo = 0;
        integer hi = v_size(v)-1;
        while (lo < hi) {
            number tmp = v[hi];
            v[hi] = v[lo];
            v[lo] = tmp;
            lo++;
            hi--;
        }
    }
    public void v_reverse(out vector(string) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer lo = 0;
        integer hi = v_size(v)-1;
        while (lo < hi) {
            string tmp = v[hi];
            v[hi] = v[lo];
            v[lo] = tmp;
            lo++;
            hi--;
        }
    }
    public void v_reverse(out vector(date) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer lo = 0;
        integer hi = v_size(v)-1;
        while (lo < hi) {
            date tmp = v[hi];
            v[hi] = v[lo];
            v[lo] = tmp;
            lo++;
            hi--;
        }
    }
    public void v_reverse(out vector(timestamp) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer lo = 0;
        integer hi = v_size(v)-1;
        while (lo < hi) {
            timestamp tmp = v[hi];
            v[hi] = v[lo];
            v[lo] = tmp;
            lo++;
            hi--;
        }
    }
    public void v_reverse(out vector(logical) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer lo = 0;
        integer hi = v_size(v)-1;
        while (lo < hi) {
            logical tmp = v[hi];
            v[hi] = v[lo];
            v[lo] = tmp;
            lo++;
            hi--;
        }
    }
    public void v_reverse(out vector(text_rgb) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer lo = 0;
        integer hi = v_size(v)-1;
        while (lo < hi) {
            text_rgb tmp = v[hi];
            v[hi] = v[lo];
            v[lo] = tmp;
            lo++;
            hi--;
        }
    }
    public void v_reverse(out vector(object<1>) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer lo = 0;
        integer hi = v_size(v)-1;
        while (lo < hi) {
            object<1> tmp = v[hi];
            v[hi] = v[lo];
            v[lo] = tmp;
            lo++;
            hi--;
        }
    }
    public vector(integer) v_reverse_copy(vector(integer) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(integer) rv[n];
        for (integer i = 0; i < n; ++i) {
                rv[n-i-1] = v[i];
            }
        return rv;
    }
    public vector(number) v_reverse_copy(vector(number) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(number) rv[n];
        for (integer i = 0; i < n; ++i) {
                rv[n-i-1] = v[i];
            }
        return rv;
    }
    public vector(string) v_reverse_copy(vector(string) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(string) rv[n];
        for (integer i = 0; i < n; ++i) {
                rv[n-i-1] = v[i];
            }
        return rv;
    }
    public vector(date) v_reverse_copy(vector(date) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(date) rv[n];
        for (integer i = 0; i < n; ++i) {
                rv[n-i-1] = v[i];
            }
        return rv;
    }
    public vector(timestamp) v_reverse_copy(vector(timestamp) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(timestamp) rv[n];
        for (integer i = 0; i < n; ++i) {
                rv[n-i-1] = v[i];
            }
        return rv;
    }
    public vector(logical) v_reverse_copy(vector(logical) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(logical) rv[n];
        for (integer i = 0; i < n; ++i) {
                rv[n-i-1] = v[i];
            }
        return rv;
    }
    public vector(text_rgb) v_reverse_copy(vector(text_rgb) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(text_rgb) rv[n];
        for (integer i = 0; i < n; ++i) {
                rv[n-i-1] = v[i];
            }
        return rv;
    }
    public vector(object<1>) v_reverse_copy(vector(object<1>) v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(object<1>) rv[n];
        for (integer i = 0; i < n; ++i) {
                rv[n-i-1] = v[i];
            }
        return rv;
    }

    public vector(object<1>) v_clone(vector(object<1>) v)
    {
        integer n = v_size(v);
        vector(object<1>) clone[n];
        for (integer i = 0; i < n; i++) {
            clone[i] = v[i];
        }
        return clone;
    }

    public vector(string) copy_sort(vector(string) v) option(category: "Utilities Enhanced/Sort and Search") {
        vector(string) copy = clone_vector(v);
        sort(copy);
        return copy;
    }
    public vector(object<1>) copy_sort(vector(object<1>) v, vector(string) ref_v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(object<1>) copy[n];
        for (integer i = 0; i < n; ++i) {
                copy[i] = v[i];
            }
        sort(copy, ref_v);
        return copy;
    }
    public vector(string) copy_sort_casei(vector(string) v) option(category: "Utilities Enhanced/Sort and Search") {
        vector(string) copy = clone_vector(v);
        sort_casei(copy);
        return copy;
    }
    public vector(object<1>) copy_sort_casei(vector(object<1>) v, vector(string) ref_v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(object<1>) copy[n];
        for (integer i = 0; i < n; ++i) {
                copy[i] = v[i];
            }
        sort_casei(copy, ref_v);
        return copy;
    }
    public vector(object<1>) copy_sort(vector(object<1>) v, vector(number) ref_v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(object<1>) copy[n];
        for (integer i = 0; i < n; ++i) {
                copy[i] = v[i];
            }
        sort(copy, ref_v);
        return copy;
    }
    public vector(object<1>) copy_sort(vector(object<1>) v, vector(date) ref_v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(object<1>) copy[n];
        for (integer i = 0; i < n; ++i) {
                copy[i] = v[i];
            }
        sort(copy, ref_v);
        return copy;
    }
    public vector(object<1>) copy_sort(vector(object<1>) v, vector(timestamp) ref_v) option(category: "Utilities Enhanced/Sort and Search") {
        integer n = v_size(v);
        vector(object<1>) copy[n];
        for (integer i = 0; i < n; ++i) {
            copy[i] = v[i];
        }
        sort(copy, ref_v);
        return copy;
    }



    public void v_null_partition(vector(number) v, out vector(integer) nonnull_indices, out vector(integer) null_indices) option(category: "Utilities Enhanced/Vector Functions")
    {
        resize(nonnull_indices, 0);
        resize(null_indices, 0);
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (null(v[i])) {
                push_back(null_indices, i);
            } else {
                push_back(nonnull_indices, i);
            }
        }
    }
    public void v_null_partition(vector(string) v, out vector(integer) nonnull_indices, out vector(integer) null_indices) option(category: "Utilities Enhanced/Vector Functions")
    {
        resize(nonnull_indices, 0);
        resize(null_indices, 0);
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (null(v[i])) {
                push_back(null_indices, i);
            } else {
                push_back(nonnull_indices, i);
            }
        }
    }
    public void v_null_partition(vector(date) v, out vector(integer) nonnull_indices, out vector(integer) null_indices) option(category: "Utilities Enhanced/Vector Functions")
    {
        resize(nonnull_indices, 0);
        resize(null_indices, 0);
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (null(v[i])) {
                push_back(null_indices, i);
            } else {
                push_back(nonnull_indices, i);
            }
        }
    }
    public void v_null_partition(vector(timestamp) v, out vector(integer) nonnull_indices, out vector(integer) null_indices) option(category: "Utilities Enhanced/Vector Functions")
    {
        resize(nonnull_indices, 0);
        resize(null_indices, 0);
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (null(v[i])) {
                push_back(null_indices, i);
            } else {
                push_back(nonnull_indices, i);
            }
        }
    }
    public void v_null_partition(vector(logical) v, out vector(integer) nonnull_indices, out vector(integer) null_indices) option(category: "Utilities Enhanced/Vector Functions")
    {
        resize(nonnull_indices, 0);
        resize(null_indices, 0);
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (null(v[i])) {
                push_back(null_indices, i);
            } else {
                push_back(nonnull_indices, i);
            }
        }
    }
    public void v_null_partition(vector(object<1>) v, out vector(integer) nonnull_indices, out vector(integer) null_indices) option(category: "Utilities Enhanced/Vector Functions")
    {
        resize(nonnull_indices, 0);
        resize(null_indices, 0);
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (null(v[i])) {
                push_back(null_indices, i);
            } else {
                push_back(nonnull_indices, i);
            }
        }
    }


    public vector(integer) v_nulls(vector(number) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return nulls;
    }

    public vector(integer) v_nulls(vector(string) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return nulls;
    }

    public vector(integer) v_nulls(vector(date) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return nulls;
    }
    public vector(integer) v_nulls(vector(timestamp) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return nulls;
    }
    public vector(integer) v_nulls(vector(logical) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return nulls;
    }
    public vector(integer) v_nulls(vector(object<1>) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return nulls;
    }

    public vector(integer) v_non_nulls(vector(number) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return non_nulls;
    }

    public vector(integer) v_non_nulls(vector(string) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return non_nulls;
    }

    public vector(integer) v_non_nulls(vector(date) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return non_nulls;
    }
    public vector(integer) v_non_nulls(vector(timestamp) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return non_nulls;
    }
    public vector(integer) v_non_nulls(vector(logical) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return non_nulls;
    }
    public vector(integer) v_non_nulls(vector(object<1>) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) non_nulls;
        vector(integer) nulls;
        v_null_partition(v, non_nulls, nulls);
        return non_nulls;
    }

    public vector(integer) v_logical_test(vector(logical) v, logical value_of_interest) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) rv[0];
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
            if (null(v[i])) {
                throw (E_UNSPECIFIC, strcat("Null logical at index ", str(i)));
            }
            if (v[i] == value_of_interest) {
                push_back(rv, i);
            }
        }
        return rv;
    }

    public vector(integer) v_true(vector(logical) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        return v_logical_test(v, true);
    }

    public vector(integer) v_false(vector(logical) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        return v_logical_test(v, false);
    }

    public vector(integer) v_partition(integer n_elems, integer partition_size) option(category: "Utilities Enhanced/Vector Functions")
    {
        if (n_elems == 0) {
            return empty_int_vec();
        }
        integer last_start = n_elems - (n_elems % partition_size);
        if (last_start == n_elems) {
            last_start -= partition_size;
        }
        vector(integer) starts = integer(s2v(series(i: 0, last_start, partition_size; i)));
        return starts;
    }

    public vector(integer) v_partition(vector(object<1>) v, integer partition_size) option(category: "Utilities Enhanced/Vector Functions")
    {
        return v_partition(v_size(v), partition_size);
    }

    public vector(integer) v_sizes(integer n_elems, vector(integer) starts) option(category: "Utilities Enhanced/Vector Functions")
    {
        vector(integer) starts_and_ends = concat(starts, [n_elems]);
        integer starts_and_ends_sz = v_size(starts_and_ends);
        vector(integer) sizes[starts_and_ends_sz-1];
        integer sizes_sz = v_size(sizes);
        for (integer i = 0; i < sizes_sz; ++i) {
            sizes[i] = starts_and_ends[i+1] - starts_and_ends[i];
        }
        return sizes;
    }


    // Returns vector of value-changing indices of given vector, plus size of
    // vector (unless size is 0).  Usage: you have a (sorted) vector and want
    // to know which position range each value spans. Maybe a vector of objects
    // should be processed in groups based on a particular value of the object.
    //
    // You get all indices where a new value is introduced (excluding index 0)
    // and the size of the vector (last element of returned vector). This makes
    // it easy for you to extracts subvectors, or write a nested loop dealing
    // with the groups of values.
    //
    // Examples:
    // v_breaks(<empty vector>) => [] // nowhere a new value
    // v_breaks([1, 1, 3, 5 ,5]) => [2, 3, 5]
    // v_breaks([1]) => [1]
    // v_breaks([1, 2, 3, 4]) => [1, 2, 3, 4]
    // v_breaks([4, 4, 4, 4, 6, 4, 4, 4, 4, 4, 4, 4]) => [4, 5, 12]
    //
    // Here's how to work with subvectors (using qu.v_range):
    // vector(integer) breaks = qu.v_breaks(vec);
    // integer sub_start = 0;
    // for (integer b = 0; b < v_size(breaks); b++) {
    //     integer sub_end = breaks[b];
    //     proccess_subvector(vec[qu.v_range(sub_start, sub_end)]);
    //     sub_start = sub_end;
    // }

    // Here's how to use for nested loop:
    // vector(integer) breaks = v_breaks(vec);
    // for (integer vec_pos = 0, integer break_pos = 0; break_pos < v_size(break_pos), break_pos++) {
    //     // do someting for group in vec
    //     for ( ; vec_pos < breaks[break_pos], ++vec_pos) {
    //         // do something with vec[vec_pos]
    //     }
    // }

    public vector(integer) v_breaks(vector(number) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        integer v_sz = v_size(v);
        if (v_sz == 0) {
            return empty_int_vec();
        }

        vector(integer) break_positions[0];
        number current_value = v[0];
        for (integer i = 1; i < v_sz; ++i) {
            if (!equaln(v[i], current_value)) {
                push_back(break_positions, i);
                current_value = v[i];
            }
        }
        push_back(break_positions, v_sz);
        return break_positions;
    }

    public vector(integer) v_breaks(vector(string) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        integer v_sz = v_size(v);
        if (v_sz == 0) {
            return empty_int_vec();
        }

        vector(integer) break_positions[0];
        string current_value = v[0];
        for (integer i = 1; i < v_sz; ++i) {
            if (!equaln(v[i], current_value)) {
                push_back(break_positions, i);
                current_value = v[i];
            }
        }
        push_back(break_positions, v_sz);
        return break_positions;
    }

    public vector(integer) v_breaks(vector(date) v) option(category: "Utilities Enhanced/Vector Functions")
    {
        integer v_sz = v_size(v);
        if (v_sz == 0) {
            return empty_int_vec();
        }

        vector(integer) break_positions[0];
        date current_value = v[0];
        for (integer i = 1; i < v_sz; ++i) {
            if (!equaln(v[i], current_value)) {
                push_back(break_positions, i);
                current_value = v[i];
            }
        }
        push_back(break_positions, v_sz);
        return break_positions;
    }

    // Tells how a union of v1 and v2 should be created.
    // v1 and v2 are assumed to be sorted.
    // v1_indices and v2_indices will be of the same length, namely the length
    // the union would have. Each element of v1_indices specifies whatelement
    // from v1 that belongs at that position in the union. Ditto for v2_indices
    // and v2. If an element of v1_indices/v2_indices is null, it means that no
    // element from v1/v2 belongs in that position.
    //
    // Example:
    // v1 = ['a', 'b'];
    // v2 = ['b', 'c', 'd'];
    // v1 union v2 is ['a', 'b', 'c', 'd']
    // v1_indices will be [0, 1, null, null] since 'a' and 'b' are in position 0
    // and 1 of v1, and 'c' and 'd' are not in v1 at all.
    // v2_indices will be [null, 0, 1, 2] since 'a' is not in v2 at all and 'b',
    // 'c' and 'd' are in positions 0, 1 and 2 resp.
    public void v_unionize(vector(string) v1, vector(string) v2, out vector(number) v1_indices, out vector(number) v2_indices)
    {
        integer v1_sz = v_size(v1);
        integer v2_sz = v_size(v2);
        integer a_pos = 0;
        integer b_pos = 0;

        resize(v1_indices, 0);
        resize(v2_indices, 0);

        // Both have values
        while (a_pos < v1_sz && b_pos < v2_sz) {
            number cmp = strcmp(v1[a_pos], v2[b_pos]);
            if (cmp == 0) {
                push_back(v1_indices, a_pos);
                push_back(v2_indices, b_pos);
                a_pos++;
                b_pos++;
            } else if (cmp < 0) {
                push_back(v1_indices, a_pos);
                push_back(v2_indices, null<number>);
                a_pos++;
            } else { // cmp > 0
                push_back(v1_indices, null<number>);
                push_back(v2_indices, b_pos);
                b_pos++;
            }
        }

        // Only v1 or v2 has values
        while (a_pos < v1_sz) {
            push_back(v1_indices, a_pos);
            push_back(v2_indices, null<number>);
            a_pos++;
        }
        while (b_pos < v2_sz) {
            push_back(v1_indices, null<number>);
            push_back(v2_indices, b_pos);
            b_pos++;
        }
    }

    // Creates the union of v1 and v2, where the union is already defined in
    // v1_indices and v2_indices (typically created by v_unionize)
    public vector(string) v_make_union(vector(string) v1, vector(string) v2, vector(number) v1_indices, vector(number) v2_indices)
    {
        integer v1_indices_sz = v_size(v1_indices);
        vector(string) result[v1_indices_sz];

        for (integer i = 0; i < v1_indices_sz; ++i) {
            if (!null(v1_indices[i])) {
                result[i] = v1[integer(v1_indices[i])];
            } else {
                result[i] = v2[integer(v2_indices[i])];
            }
        }
        return result;
    }

    // Creates the union of v1 and v2, where the union is already defined in
    // v1_indices and v2_indices (typically created by v_unionize)
    public vector(object<1>) v_make_union(vector(object<1>) v1, vector(object<1>) v2, vector(number) v1_indices, vector(number) v2_indices)
    {
        integer v1_indices_sz = v_size(v1_indices);
        vector(object<1>) result[v1_indices_sz];

        for (integer i = 0; i < v1_indices_sz; ++i) {
            if (!null(v1_indices[i])) {
                result[i] = v1[integer(v1_indices[i])];
            } else {
                result[i] = v2[integer(v2_indices[i])];
            }
        }
        return result;
    }

    // v1 and v2 must be sorted
    // Creates the union of v1 and v2
    public vector(string) v_union(vector(string) v1, vector(string) v2)
    {
        vector(number) v1_indices;
        vector(number) v2_indices;
        v_unionize(v1, v2, v1_indices, v2_indices);

        return v_make_union(v1, v2, v1_indices, v2_indices);
    }

    public vector(object<1>) v_concat(vector(object<1>) option(nullable) u, vector(object<1>) option(nullable) v)
    {
        vector(object<1>) uv[0];
        if (v_size(u) > 0 && v_size(v) > 0) {
            uv = concat(u, v);
        } else if (v_size(u) > 0) {
            uv = clone_vector(u);
        } else if (v_size(v) > 0) {
            uv = clone_vector(v);
        }
        return uv;
    }


    public vector(number) v_concat(vector(number) option(nullable) u, vector(number) option(nullable) v)
    {
        vector(number) uv[0];
        if (v_size(u) > 0 && v_size(v) > 0) {
            uv = concat(u, v);
        } else if (v_size(u) > 0) {
            uv = clone_vector(u);
        } else if (v_size(v) > 0) {
            uv = clone_vector(v);
        }
        return uv;
    }
    public vector(date) v_concat(vector(date) option(nullable) u, vector(date) option(nullable) v)
    {
        vector(date) uv[0];
        if (v_size(u) > 0 && v_size(v) > 0) {
            uv = concat(u, v);
        } else if (v_size(u) > 0) {
            uv = clone_vector(u);
        } else if (v_size(v) > 0) {
            uv = clone_vector(v);
        }
        return uv;
    }
    public vector(string) v_concat(vector(string) option(nullable) u, vector(string) option(nullable) v)
    {
        vector(string) uv[0];
        if (v_size(u) > 0 && v_size(v) > 0) {
            uv = concat(u, v);
        } else if (v_size(u) > 0) {
            uv = clone_vector(u);
        } else if (v_size(v) > 0) {
            uv = clone_vector(v);
        }
        return uv;
    }
    public vector(timestamp) v_concat(vector(timestamp) option(nullable) u, vector(timestamp) option(nullable) v)
    {
        vector(timestamp) uv[0];
        if (v_size(u) > 0 && v_size(v) > 0) {
            uv = concat(u, v);
        } else if (v_size(u) > 0) {
            uv = clone_vector(u);
        } else if (v_size(v) > 0) {
            uv = clone_vector(v);
        }
        return uv;
    }
    public vector(logical) v_concat(vector(logical) option(nullable) u, vector(logical) option(nullable) v)
    {
        vector(logical) uv[0];
        if (v_size(u) > 0 && v_size(v) > 0) {
            uv = concat(u, v);
        } else if (v_size(u) > 0) {
            uv = clone_vector(u);
        } else if (v_size(v) > 0) {
            uv = clone_vector(v);
        }
        return uv;
    }



    public map_str_str reverse_map(map_str_str in_map) option(category: "Utilities Enhanced/Map and Set") {
        map_str_str out_map = map_str_str();
        vector(string) keys = in_map.get_keys();
        integer keys_sz = v_size(keys);
        for (integer i = 0; i < keys_sz; ++i) {
                out_map.add(in_map.find(keys[i]), keys[i]);
            }
        return out_map;
    }
    public map_str_num index_map(vector(string) v) option(category: "Utilities Enhanced/Map and Set") {
        map_str_num m = map_str_num();
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
                m.add(v[i], i);
            }
        return m;
    }
    public map_istr_num index_map_casei(vector(string) v) option(category: "Utilities Enhanced/Map and Set") {
        map_istr_num m = map_istr_num();
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
                m.add(v[i], i);
            }
        return m;
    }
    public map_num_num index_map(vector(number) v) option(category: "Utilities Enhanced/Map and Set") {
        map_num_num m = map_num_num();
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
                m.add(v[i], i);
            }
        return m;
    }
    public map_date_num index_map(vector(date) v) option(category: "Utilities Enhanced/Map and Set") {
        map_date_num m = map_date_num();
        integer v_sz = v_size(v);
        for (integer i = 0; i < v_sz; ++i) {
                m.add(v[i], i);
            }
        return m;
    }

    public vector(string) mapstrs(map_str_str m) option(category: "Utilities Enhanced/Map and Set") {
        vector(string) keys = m.get_keys();
        vector(string) vals = m.find(keys);
        integer keys_sz = null(keys) ? 0 : v_size(keys);
        vector(string) strs[keys_sz];
        for (integer i = 0; i < keys_sz; i++) {
            strs[i] = strcat([keys[i], " => ", vals[i]]);
        }
        return strs;
    }
    public vector(string) mapstrs(map_str_num m) option(category: "Utilities Enhanced/Map and Set") {
        vector(string) keys = m.get_keys();
        vector(string) vals = str(m.find(keys));
        integer keys_sz = null(keys) ? 0 : v_size(keys);
        vector(string) strs[keys_sz];
        for (integer i = 0; i < keys_sz; i++) {
            strs[i] = strcat([keys[i], " => ", vals[i]]);
        }
        return strs;
    }
    public vector(string) mapstrs(map_istr_str m) option(category: "Utilities Enhanced/Map and Set") {
        vector(string) keys = m.get_keys();
        vector(string) vals = m.find(keys);
        integer keys_sz = null(keys) ? 0 : v_size(keys);
        vector(string) strs[keys_sz];
        for (integer i = 0; i < keys_sz; i++) {
            strs[i] = strcat([keys[i], " => ", vals[i]]);
        }
        return strs;
    }
    public vector(string) mapstrs(map_istr_num m) option(category: "Utilities Enhanced/Map and Set") {
        vector(string) keys = m.get_keys();
        vector(string) vals = str(m.find(keys));
        integer keys_sz = null(keys) ? 0 : v_size(keys);
        vector(string) strs[keys_sz];
        for (integer i = 0; i < keys_sz; i++) {
            strs[i] = strcat([keys[i], " => ", vals[i]]);
        }
        return strs;
    }
    public vector(string) mapstrs(map_num_str m) option(category: "Utilities Enhanced/Map and Set") {
        vector(number) keys = m.get_keys();
        vector(string) keys_str = str(keys);
        vector(string) vals = m.find(keys);
        integer keys_sz = null(keys) ? 0 : v_size(keys);
        vector(string) strs[keys_sz];
        for (integer i = 0; i < keys_sz; i++) {
            strs[i] = strcat([keys_str[i], " => ", vals[i]]);
        }
        return strs;
    }
    public vector(string) mapstrs(map_num_num m) option(category: "Utilities Enhanced/Map and Set") {
        vector(number) keys = m.get_keys();
        vector(string) keys_str = str(keys);
        vector(number) vals = m.find(keys);
        vector(string) vals_str = str(vals);
        integer keys_sz = null(keys) ? 0 : v_size(keys);
        vector(string) strs[keys_sz];
        for (integer i = 0; i < keys_sz; i++) {
            strs[i] = strcat([keys_str[i], " => ", vals_str[i]]);
        }
        return strs;
    }



public class stack_str option(category: "Utilities Enhanced/Stack and Queue") {
public:
    stack_str();
    logical empty();
    void push(string option(nullable) s);
    string pop();
    string top();
    vector(string) contents();

private:
    vector(string) stack_;
    integer end_;
    integer max_size_;

    logical _full();
    void _expand();
};
stack_str.stack_str()
{
    max_size_ = 10;
    resize(stack_, max_size_);
    end_ = 0;
}
void stack_str.push(string option(nullable) s)
{
    if (_full()) {
        _expand();
    }
    stack_[end_] = s;
    ++end_;
}
logical stack_str.empty() = end_ == 0;
string stack_str.pop()
{
    if (this.empty()) {
        throw (E_UNSPECIFIC, "stack is empty");
    }
    --end_;
    return stack_[end_];
}
string stack_str.top()
{
    if (this.empty()) {
        throw (E_UNSPECIFIC, "stack is empty");
    }
    return stack_[end_-1];
}
vector(string) stack_str.contents()
{
    if (this.empty()) {
        return empty_str_vec();
    } else {
        return stack_[v_range(end_)];
    }
}
logical stack_str._full() = end_ == max_size_;
void stack_str._expand()
{
    integer new_size = max_size_ * 2;
    vector(string) new_stack[new_size];
    for (integer i = 0; i < end_; i++) {
        new_stack[i] = stack_[i];
    }
    stack_ = new_stack;
    max_size_ = new_size;
}


class queue_base {
public:
    logical empty();
    integer size();

protected:
    queue_base();

    integer start_;
    integer end_;
    integer buf_size_;

    logical _full();
    integer _advance(integer vec_ptr);

private:
    virtual void _expand() = 0;
};
integer queue_initial_size = 10;

queue_base.queue_base()
: start_(0), end_(0), buf_size_(queue_initial_size)
{}

logical queue_base.empty() = start_ == end_;

integer queue_base.size()
{
    if (start_ <= end_) {
        return end_ - start_;
    } else {
        return (buf_size_ - start_) + end_;
    }
}

logical queue_base._full()
{
    integer end_advanced = _advance(end_);
    return end_advanced == start_;
}
integer queue_base._advance(integer vec_ptr)
{
    if (vec_ptr == buf_size_ - 1) {
        return 0;
    } else {
        return vec_ptr + 1;
    }
}

public class queue_int : public queue_base option(category: "Utilities Enhanced/Stack and Queue") {
public:
    queue_int();
    virtual void push(integer x);
    virtual integer pop();
    integer top();
    vector(integer) contents();
    vector(integer) contents_reverse();
    vector(string) impl_content();

protected:
    void exec_push(integer x);
    integer exec_pop();

private:
    vector(integer) buffer_;

    virtual void _expand();
};
queue_int.queue_int()
{
    resize(buffer_, buf_size_);
}
void queue_int.push(integer val) { exec_push(val); }
integer queue_int.pop() = exec_pop();
integer queue_int.top()
{
    if (this.empty()) {
        throw (E_UNSPECIFIC, "queue is empty");
    }
    return buffer_[start_];
}
vector(integer) queue_int.contents()
{
    if (this.empty()) {
        return empty_int_vec();
    } else if (start_ < end_) {
//        return buffer_[start_:end_-1];
        return sub_vector(buffer_, start_, end_-start_);
    } else if (end_ == 0) {
        //return buffer_[start_:buf_size_-1];
        return sub_vector(buffer_, start_, buf_size_-start_);
    } else {
        //return concat(buffer_[start_:buf_size_-1], buffer_[0:end_-1]);
        return concat(sub_vector(buffer_, start_, buf_size_-start_),
                      sub_vector(buffer_, 0, end_));
    }
}
vector(integer) queue_int.contents_reverse()
{
    if (this.empty()) {
        return empty_int_vec();
    } else if (start_ < end_) {
        integer n = this.size();
        vector(integer) r[n];
        for (integer i = 0; i < n; i++) {
            r[i] = buffer_[end_-i-1];
        }
        return r;
    } else if (end_ == 0) {
        integer n = this.size();
        vector(integer) r[n];
        for (integer i = 0; i < n; i++) {
            r[i] = buffer_[buf_size_-i-1];
        }
        return r;
    } else {
        integer n_last = end_;
        integer n_first = buf_size_- start_;
        vector(integer) r[n_last + n_first];
        for (integer i = 0; i < n_last; i++) {
            r[i] = buffer_[end_ - i - 1];
        }
        for (integer i = 0; i < n_first; i++) {
            r[i+n_last] = buffer_[buf_size_ - i - 1];
        }
        return r;
    }
}
vector(string) queue_int.impl_content() =
    concat(
        v_str_join(["start", "end"], str([start_, end_]), ": "),
        str(buffer_));
void queue_int.exec_push(integer val)
{
    if (_full()) {
        _expand();
    }
    buffer_[end_] = val;
    end_ = _advance(end_);
}
integer queue_int.exec_pop()
{
    if (this.empty()) {
        throw (E_UNSPECIFIC, "queue is empty");
    }
    integer val = buffer_[start_];
    start_ = _advance(start_);
    return val;
}
void queue_int._expand()
{
    vector(integer) content = this.contents();
    buf_size_ *= 2;
    resize(buffer_, buf_size_);
    integer content_sz = v_size(content);
    for (integer i = 0; i < content_sz; i++) {
        buffer_[i] = content[i];
    }
    start_ = 0;
    end_ = v_size(content);
}


public class queue_str : public queue_base option(category: "Utilities Enhanced/Stack and Queue") {
public:
    queue_str();
    virtual void push(string option(nullable) s);
    virtual string pop();
    string top();
    vector(string) contents();
    vector(string) contents_reverse();
    vector(string) impl_content();

protected:
    void exec_push(string option(nullable) s);
    string exec_pop();

private:
    vector(string) buffer_;

    virtual void _expand();
};
queue_str.queue_str()
{
    resize(buffer_, buf_size_);
}
void queue_str.push(string option(nullable) s) { exec_push(s); }
string queue_str.pop() = exec_pop();
string queue_str.top()
{
    if (this.empty()) {
        throw (E_UNSPECIFIC, "queue is empty");
    }
    return buffer_[start_];
}
vector(string) queue_str.contents()
{
    if (this.empty()) {
        return empty_str_vec();
    } else if (start_ < end_) {
//        return buffer_[start_:end_-1];
        return sub_vector(buffer_, start_, end_-start_);
    } else if (end_ == 0) {
        //return buffer_[start_:buf_size_-1];
        return sub_vector(buffer_, start_, buf_size_-start_);
    } else {
        //return concat(buffer_[start_:buf_size_-1], buffer_[0:end_-1]);
        return concat(sub_vector(buffer_, start_, buf_size_-start_),
                      sub_vector(buffer_, 0, end_));
    }
}
vector(string) queue_str.contents_reverse()
{
    if (this.empty()) {
        return empty_str_vec();
    } else if (start_ < end_) {
        integer n = this.size();
        vector(string) r[n];
        for (integer i = 0; i < n; i++) {
            r[i] = buffer_[end_-i-1];
        }
        return r;
    } else if (end_ == 0) {
        integer n = this.size();
        vector(string) r[n];
        for (integer i = 0; i < n; i++) {
            r[i] = buffer_[buf_size_-i-1];
        }
        return r;
    } else {
        integer n_last = end_;
        integer n_first = buf_size_- start_;
        vector(string) r[n_last + n_first];
        for (integer i = 0; i < n_last; i++) {
            r[i] = buffer_[end_ - i - 1];
        }
        for (integer i = 0; i < n_first; i++) {
            r[i+n_last] = buffer_[buf_size_ - i - 1];
        }
        return r;
    }
}
vector(string) queue_str.impl_content() = concat(v_str_join(["start", "end"], str([start_, end_]), ": "), clone_vector(buffer_));
void queue_str.exec_push(string option(nullable) s)
{
    if (_full()) {
        _expand();
    }
    buffer_[end_] = s;
    end_ = _advance(end_);
}
string queue_str.exec_pop()
{
    if (this.empty()) {
        throw (E_UNSPECIFIC, "queue is empty");
    }
    string s = buffer_[start_];
    start_ = _advance(start_);
    return s;
}
void queue_str._expand()
{
    vector(string) content = this.contents();
    buf_size_ *= 2;
    resize(buffer_, buf_size_);
    integer content_sz = v_size(content);
    for (integer i = 0; i < content_sz; i++) {
        buffer_[i] = content[i];
    }
    start_ = 0;
    end_ = v_size(content);
}


class queue_str_unique : public queue_str {
public:
    queue_str_unique();
    virtual void push(string option(nullable) s);
    virtual string pop();

private:
    set_str strings_queued_;
    logical null_queued_;
};
queue_str_unique.queue_str_unique()
: queue_str(), strings_queued_(set_str()), null_queued_(false)
{}
void queue_str_unique.push(string option(nullable) s)
{
    logical stop_push = false;
    if (null(s)) {
        if (null_queued_) {
            stop_push = true;
        } else {
            null_queued_ = true;
        }
    } else {
        if (strings_queued_.includes(s)) {
            stop_push = true;
        } else {
            strings_queued_.add(s);
        }
    }

    if (!stop_push) {
        exec_push(s);
    }
}
string queue_str_unique.pop()
{
    string s = exec_pop();
    if (null(s)) {
        null_queued_ = false;
    } else {
        strings_queued_.remove(s);
    }
    return s;
}

// object_map - Keep a map of objects
// Usage:
// * Create object_map passing in the vector you want to use (data in vector will be erased!)
// * Pass this vector to object_map if called for
// * Never access this vector directly, only use it via the object_map
// * Never pass vector and object_map separately to a function - they belong together!
class _base_object_map {
public:
    _base_object_map(out vector(object<1>) option(nullable) omv, logical case_sensitive = true);
    void add(string key, object<1> o, out vector(object<1>) omv);
    void remove(string key);
    logical has(string key);
    object<1> find(string key, vector(object<1>) omv);
    number vec_pos(string key);
    vector(integer) vacancies();

    vector(string) get_keys();

    void state(out integer count, out vector(string) key, out vector(number) pos, out vector(integer) vacancies);

private:
    logical case_sensitive_;
    map_str_num keymap_;
    map_istr_num cikeymap_;
    integer count_;
    queue_int vacant_;
};
_base_object_map._base_object_map(out vector(object<1>) option(nullable) omv, logical case_sensitive)
    : case_sensitive_(case_sensitive), count_(0) {
    resize(omv, 0);
    keymap_ = map_str_num();
    cikeymap_ = map_istr_num();
    vacant_ = new queue_int();
}
vector(string) _base_object_map.get_keys() { return case_sensitive_ ? keymap_.get_keys() : cikeymap_.get_keys(); }
void _base_object_map.add(string key, object<1> o, out vector(object<1>) omv) {
    if (v_size(omv) != count_) {
        throw (E_UNSPECIFIC, "Wrong vector passed to object_map");
    }
    number ix = case_sensitive_ ? keymap_.find(key) : cikeymap_.find(key);
    if (null(ix)) {
        if (vacant_.empty()) {
            ix = count_;
            push_back(omv, o);
            count_++;
        } else {
            ix = vacant_.pop();
            omv[integer(ix)] = o;
        }
    } else {
        omv[integer(ix)] = o;
    }

    if (case_sensitive_) keymap_.add(key, ix);
    else cikeymap_.add(key, ix);
}
void _base_object_map.remove(string key) {
    number ix = case_sensitive_ ? keymap_.find(key) : cikeymap_.find(key);

    if (!null(ix)) {
        vacant_.push(integer(ix));
        if (case_sensitive_) keymap_.remove(key);
        else cikeymap_.remove(key);
    }
}
logical _base_object_map.has(string key) { return !null(case_sensitive_ ? keymap_.find(key) : cikeymap_.find(key)); }
object<1> _base_object_map.find(string key, vector(object<1>) omv) {
    number ix = case_sensitive_ ? keymap_.find(key) : cikeymap_.find(key);
    if (null(ix)) {
        object<1> null_obj;
        return null_obj;
    } else {
        return omv[integer(ix)];
    }
}
number _base_object_map.vec_pos(string key) { return case_sensitive_ ? keymap_.find(key) : cikeymap_.find(key); }
vector(integer) _base_object_map.vacancies() = vacant_.contents();

void _base_object_map.state(
    out integer count, out vector(string) key, out vector(number) pos, out vector(integer) vacancies)
{
    count = count_;
    key = this.get_keys();
    pos = this.vec_pos(key);
    vacancies = vacant_.contents();
}


public class object_map : public _base_object_map option(category: "Utilities Enhanced/Map and Set") {
public:
    object_map(vector(object<1>) option(nullable) omv);
};
object_map.object_map(vector(object<1>) option(nullable) omv)
    : _base_object_map(omv, true)
{}

public class object_map_casei : public _base_object_map option(category: "Utilities Enhanced/Map and Set") {
public:
    object_map_casei(vector(object<1>) option(nullable) omv);
};
object_map_casei.object_map_casei(vector(object<1>) option(nullable) omv)
    : _base_object_map(omv, false)
{}

// map_vec - Vector valued maps.
// Class strings is used internally in the map_vec_str class,
// and not intended to be used in other contexts.
class strings {
public:
    strings(vector(string) v);
    vector(string) get_vector();
    void push_back(string option(nullable) val);
private:
    vector(string) v_;
};

strings.strings(vector(string) v) :
    v_(v) {}
vector(string) strings.get_vector() = v_;
void strings.push_back(string option(nullable) val)
{
    ..push_back(v_, val);
}

public class map_str_vec_str option(category: "Utilities Enhanced/Map and Set") {
public:
    map_str_vec_str();
    void add(string key, vector(string) value);
    void remove(string key);
    logical has_key(string key);
    vector(string) find(string key);
    vector(string) get_keys();
private:
    object_map map_;
    vector(strings) vec_;
};

map_str_vec_str.map_str_vec_str() {
    resize(vec_, 0);
    map_ = new object_map(vec_);
}
void map_str_vec_str.add(string key, vector(string) value) {
    map_.add(key, new strings(value), vec_);
}
void map_str_vec_str.remove(string key) {
    map_.remove(key);
}
logical map_str_vec_str.has_key(string key) = map_.has(key);
vector(string) map_str_vec_str.find(string key) {
    strings value_found = map_.find(key, vec_);
    return null(value_found) ? null<vector(string)> : value_found.get_vector();
}
vector(string) map_str_vec_str.get_keys() = map_.get_keys();


public class map_str_set_str option(category: "Utilities Enhanced/Map and Set") {
public:
    map_str_set_str();
    void add(string key, set_str obj);
    logical has(string key);
    set_str find(string key);
    vector(string) get_keys();
private:
    object_map map_;
    vector(set_str) vec_;
};
map_str_set_str.map_str_set_str()
{
    vector(set_str) v[0];
    vec_ = v;
    map_ = new object_map(vec_);
}
void map_str_set_str.add(string key, set_str obj) { map_.add(key, obj, vec_); }
logical map_str_set_str.has(string key) { return map_.has(key); }
set_str map_str_set_str.find(string key) { return map_.find(key, vec_); }
vector(string) map_str_set_str.get_keys() { return map_.get_keys(); }



public class vectors_str {
public:
    vectors_str();
    vectors_str(integer n_vecs);
    integer num();
    void add_vector();
    void add_vector(vector(string) v);
    void push_back(integer vec, string option(nullable) val);
    vector(string) at(integer vec);

private:
    vector(strings) vecs_;
};
vectors_str.vectors_str() {}
vectors_str.vectors_str(integer n_vecs)
{
    resize(vecs_, n_vecs);
    for (integer i = 0; i < n_vecs; ++i) {
        vecs_[i] = new strings(empty_str_vec());
    }
}
integer vectors_str.num() = v_size(vecs_);
void vectors_str.add_vector()
{
    ..push_back(vecs_, new strings(empty_str_vec()));
}
void vectors_str.add_vector(vector(string) v)
{
    ..push_back(vecs_, new strings(v));
}
void vectors_str.push_back(integer vec, string option(nullable) val)
{
    vecs_[vec].push_back(val);
}
vector(string) vectors_str.at(integer vec) = vecs_[vec].get_vector();


// Class numbers is used internally in the map_vec_str class,
// and not intended to be used in other contexts.
class numbers {
public:
    numbers(vector(number) v);
    vector(number) get_vector();
    void push_back(number option(nullable) val);
private:
    vector(number) v_;
};

numbers.numbers(vector(number) v) :
    v_(v) {}
vector(number) numbers.get_vector() = v_;
void numbers.push_back(number option(nullable) val)
{
    ..push_back(v_, val);
}

public class map_str_vec_num option(category: "Utilities Enhanced/Map and Set"){
public:
    map_str_vec_num();
    void add(string key, vector(number) value);
    void remove(string key);
    logical has_key(string key);
    vector(number) find(string key);
    vector(string) get_keys();
private:
    object_map map_;
    vector(numbers) vec_;
};

map_str_vec_num.map_str_vec_num() {
    resize(vec_, 0);
    map_ = new object_map(vec_);
}
void map_str_vec_num.add(string key, vector(number) value) {
    map_.add(key, new numbers(value), vec_);
}
void map_str_vec_num.remove(string key) {
    map_.remove(key);
}
logical map_str_vec_num.has_key(string key) = map_.has(key);
vector(number) map_str_vec_num.find(string key) {
    numbers value_found = map_.find(key, vec_);
    return null(value_found) ? null<vector(number)> : value_found.get_vector();
}
vector(string) map_str_vec_num.get_keys() = map_.get_keys();

public class vectors_num {
public:
    vectors_num();
    vectors_num(integer n_vecs);
    integer num();
    void add_vector();
    void add_vector(vector(number) v);
    void push_back(integer vec, number option(nullable) val);
    vector(number) at(integer vec);

private:
    vector(numbers) vecs_;
};
vectors_num.vectors_num() {}
vectors_num.vectors_num(integer n_vecs)
{
    resize(vecs_, n_vecs);
    for (integer i = 0; i < n_vecs; ++i) {
        vecs_[i] = new numbers(empty_num_vec());
    }
}
integer vectors_num.num() = v_size(vecs_);
void vectors_num.add_vector()
{
    ..push_back(vecs_, new numbers(empty_num_vec()));
}
void vectors_num.add_vector(vector(number) v)
{
    ..push_back(vecs_, new numbers(v));
}
void vectors_num.push_back(integer vec, number option(nullable) val)
{
    vecs_[vec].push_back(val);
}
vector(number) vectors_num.at(integer vec) = vecs_[vec].get_vector();


// Class integers is used internally in the map_str_vec_int class,
// and not intended to be used in other contexts.
class integers {
public:
    integers(vector(integer) v);
    vector(integer) get_vector();
    void push_back(integer val);
private:
    vector(integer) v_;
};

integers.integers(vector(integer) v) :
    v_(v) {}
vector(integer) integers.get_vector() = v_;
void integers.push_back(integer val)
{
    ..push_back(v_, val);
}

public class map_str_vec_int option(category: "Utilities Enhanced/Map and Set"){
public:
    map_str_vec_int();
    void add(string key, vector(integer) value);
    void remove(string key);
    logical has_key(string key);
    vector(integer) find(string key);
    vector(string) get_keys();
private:
    object_map map_;
    vector(integers) vec_;
};

map_str_vec_int.map_str_vec_int() {
    resize(vec_, 0);
    map_ = new object_map(vec_);
}
void map_str_vec_int.add(string key, vector(integer) value) {
    map_.add(key, new integers(value), vec_);
}
void map_str_vec_int.remove(string key) {
    map_.remove(key);
}
logical map_str_vec_int.has_key(string key) = map_.has(key);
vector(integer) map_str_vec_int.find(string key) {
    integers value_found = map_.find(key, vec_);
    return null(value_found) ? null<vector(integer)> : value_found.get_vector();
}
vector(string) map_str_vec_int.get_keys() = map_.get_keys();

public class vectors_int {
public:
    vectors_int();
    vectors_int(integer n_vecs);
    integer num();
    void add_vector();
    void add_vector(vector(integer) val);
    void push_back(integer vec, integer val);
    vector(integer) at(integer vec);

private:
    vector(integers) vecs_;
};
vectors_int.vectors_int() {}
vectors_int.vectors_int(integer n_vecs)
{
    resize(vecs_, n_vecs);
    for (integer i = 0; i < n_vecs; ++i) {
        vecs_[i] = new integers(empty_int_vec());
    }
}
integer vectors_int.num() = v_size(vecs_);
void vectors_int.add_vector()
{
    ..push_back(vecs_, new integers(empty_int_vec()));
}
void vectors_int.add_vector(vector(integer) v)
{
    ..push_back(vecs_, new integers(v));
}
void vectors_int.push_back(integer vec, integer val)
{
    vecs_[vec].push_back(val);
}
vector(integer) vectors_int.at(integer vec) = vecs_[vec].get_vector();


// Class logicals is used internally in the map_vec_str class,
// and not intended to be used in other contexts.
class logicals {
public:
    logicals(vector(logical) v);
    vector(logical) get_vector();
    void push_back(logical option(nullable) val);
private:
    vector(logical) v_;
};

logicals.logicals(vector(logical) v) :
    v_(v) {}
vector(logical) logicals.get_vector() = v_;
void logicals.push_back(logical option(nullable) val)
{
    ..push_back(v_, val);
}

public class map_str_vec_logical option(category: "Utilities Enhanced/Map and Set"){
public:
    map_str_vec_logical();
    void add(string key, vector(logical) value);
    void remove(string key);
    logical has_key(string key);
    vector(logical) find(string key);
    vector(string) get_keys();
private:
    object_map map_;
    vector(logicals) vec_;
};

map_str_vec_logical.map_str_vec_logical() {
    resize(vec_, 0);
    map_ = new object_map(vec_);
}
void map_str_vec_logical.add(string key, vector(logical) value) {
    map_.add(key, new logicals(value), vec_);
}
void map_str_vec_logical.remove(string key) {
    map_.remove(key);
}
logical map_str_vec_logical.has_key(string key) = map_.has(key);
vector(logical) map_str_vec_logical.find(string key) {
    logicals value_found = map_.find(key, vec_);
    return null(value_found) ? null<vector(logical)> : value_found.get_vector();
}
vector(string) map_str_vec_logical.get_keys() = map_.get_keys();

public class vectors_logical {
public:
    vectors_logical();
    vectors_logical(integer n_vecs);
    integer num();
    void add_vector();
    void add_vector(vector(logical) v);
    void push_back(integer vec, logical option(nullable) val);
    vector(logical) at(integer vec);

private:
    vector(logicals) vecs_;
};
vectors_logical.vectors_logical() {}
vectors_logical.vectors_logical(integer n_vecs)
{
    resize(vecs_, n_vecs);
    for (integer i = 0; i < n_vecs; ++i) {
        vecs_[i] = new logicals(empty_logical_vec());
    }
}
integer vectors_logical.num() = v_size(vecs_);
void vectors_logical.add_vector()
{
    ..push_back(vecs_, new logicals(empty_logical_vec()));
}
void vectors_logical.add_vector(vector(logical) v)
{
    ..push_back(vecs_, new logicals(v));
}
void vectors_logical.push_back(integer vec, logical option(nullable) val)
{
    vecs_[vec].push_back(val);
}
vector(logical) vectors_logical.at(integer vec) = vecs_[vec].get_vector();



// Class dates is used internally in the map_vec_str class,
// and not intended to be used in other contexts.
class dates {
public:
    dates(vector(date) v);
    vector(date) get_vector();
    void push_back(date option(nullable) val);
private:
    vector(date) v_;
};

dates.dates(vector(date) v) :
    v_(v) {}
vector(date) dates.get_vector() = v_;
void dates.push_back(date option(nullable) val)
{
    ..push_back(v_, val);
}

public class map_str_vec_date option(category: "Utilities Enhanced/Map and Set"){
public:
    map_str_vec_date();
    void add(string key, vector(date) value);
    void remove(string key);
    logical has_key(string key);
    vector(date) find(string key);
    vector(string) get_keys();
private:
    object_map map_;
    vector(dates) vec_;
};

map_str_vec_date.map_str_vec_date() {
    resize(vec_, 0);
    map_ = new object_map(vec_);
}
void map_str_vec_date.add(string key, vector(date) value) {
    map_.add(key, new dates(value), vec_);
}
void map_str_vec_date.remove(string key) {
    map_.remove(key);
}
logical map_str_vec_date.has_key(string key) = map_.has(key);
vector(date) map_str_vec_date.find(string key) {
    dates value_found = map_.find(key, vec_);
    return null(value_found) ? null<vector(date)> : value_found.get_vector();
}
vector(string) map_str_vec_date.get_keys() = map_.get_keys();

public class vectors_date {
public:
    vectors_date();
    vectors_date(integer n_vecs);
    integer num();
    void add_vector();
    void add_vector(vector(date) v);
    void push_back(integer vec, date option(nullable) val);
    vector(date) at(integer vec);

private:
    vector(dates) vecs_;
};
vectors_date.vectors_date() {}
vectors_date.vectors_date(integer n_vecs)
{
    resize(vecs_, n_vecs);
    for (integer i = 0; i < n_vecs; ++i) {
        vecs_[i] = new dates(empty_date_vec());
    }
}
integer vectors_date.num() = v_size(vecs_);
void vectors_date.add_vector()
{
    ..push_back(vecs_, new dates(empty_date_vec()));
}
void vectors_date.add_vector(vector(date) v)
{
    ..push_back(vecs_, new dates(v));
}
void vectors_date.push_back(integer vec, date option(nullable) val)
{
    vecs_[vec].push_back(val);
}
vector(date) vectors_date.at(integer vec) = vecs_[vec].get_vector();


// count_set - Managing many counters made simple!
// Usage:
//   1. Create a count_set like this: count_set cs = count_set();
//
//   2. Increase a counter identified by a string like this: cs.increase('ERICb');
//      (If the counter didn't already exist it is created for you)
//
//   3. Read your counters like this: number count = cs.get_count('ERICb');
//      or several counters: vector(number) counts = cs.get_counts(vector_of_strings);
//      (If you ask about a counter that doesn't exist, 0 is returned)
//
//   4. Get all existing counters: vector(string) created_counters = cs.get_keys();
//      and their values: vector(number) created_counters_counts = cs.get_counts();
//
//  For keys of other data types, use count_set_num or count_set_date.

public class count_set option(category: "Utilities Enhanced/Map and Set") {
public:
    count_set();
    void increase(string key, number amount = 1);
    void decrease(string key, number amount = 1);
    vector(string) get_keys();
    vector(number) get_counts();
    vector(number) get_counts(vector(string) keys);
    number get_count(string key);

private:
    map_str_num the_set_;

    void _adjust_value(string key, number adj);

};
count_set.count_set()
: the_set_(map_str_num())
{}
void count_set.increase(string key, number amount) { _adjust_value(key, amount); }
void count_set.decrease(string key, number amount) { _adjust_value(key, -amount); }
vector(string) count_set.get_keys() { return the_set_.get_keys(); }
vector(number) count_set.get_counts()
{
    vector(string) keys = the_set_.get_keys();
    vector(number) counts[v_size(keys)];
    for (integer i = 0; i < v_size(keys); i++) {
        counts[i] = the_set_.find(keys[i]);
    }
    return counts;
}
vector(number) count_set.get_counts(vector(string) keys)
{
    vector(number) counts[v_size(keys)];
    for (integer i = 0; i < v_size(keys); i++) {
        counts[i] = this.get_count(keys[i]);
    }
    return counts;
}
number count_set.get_count(string key)
{
    number c = the_set_.find(key);
    return null(c) ? 0 : c;
}
void count_set._adjust_value(string key, number adj)
{
    number current_value = the_set_.find(key);
    if (null(current_value)) {
        current_value = 0;
    }
    number new_value = current_value + adj;
    the_set_.add(key, new_value);
}


public class count_set_num option(category: "Utilities Enhanced/Map and Set") {
public:
    count_set_num();
    void increase(number key, number amount = 1);
    vector(number) get_keys();
    vector(number) get_counts();
    vector(number) get_counts(vector(number) keys);
    number get_count(number key);
private:
    map_num_num the_set_;
};
count_set_num.count_set_num()
: the_set_(map_num_num())
{}
void count_set_num.increase(number key, number amount) {
    number current_value = the_set_.find(key);
    if (null(current_value)) {
        current_value = 0;
    }
    number new_value = current_value + amount;
    the_set_.add(key, new_value);
}
vector(number) count_set_num.get_keys() { return the_set_.get_keys(); }
vector(number) count_set_num.get_counts() {
    vector(number) keys = the_set_.get_keys();
    vector(number) counts[v_size(keys)];
    for (integer i = 0; i < v_size(keys); i++) {
        counts[i] = the_set_.find(keys[i]);
    }
    return counts;
}
vector(number) count_set_num.get_counts(vector(number) keys) {
    vector(number) counts[v_size(keys)];
    for (integer i = 0; i < v_size(keys); i++) {
        counts[i] = this.get_count(keys[i]);
    }
    return counts;
}
number count_set_num.get_count(number key) {
    number c = the_set_.find(key);
    return null(c) ? 0 : c;
}

public class count_set_date option(category: "Utilities Enhanced/Map and Set") {
public:
    count_set_date();
    void increase(date key, number amount = 1);
    vector(date) get_keys();
    vector(number) get_counts();
    vector(number) get_counts(vector(date) keys);
    number get_count(date key);
private:
    map_date_num the_set_;
};
count_set_date.count_set_date()
: the_set_(map_date_num())
{}
void count_set_date.increase(date key, number amount) {
    number current_value = the_set_.find(key);
    if (null(current_value)) {
        current_value = 0;
    }
    number new_value = current_value + amount;
    the_set_.add(key, new_value);
}
vector(date) count_set_date.get_keys() { return the_set_.get_keys(); }
vector(number) count_set_date.get_counts() {
    vector(date) keys = the_set_.get_keys();
    vector(number) counts[v_size(keys)];
    for (integer i = 0; i < v_size(keys); i++) {
        counts[i] = the_set_.find(keys[i]);
    }
    return counts;
}
vector(number) count_set_date.get_counts(vector(date) keys) {
    vector(number) counts[v_size(keys)];
    for (integer i = 0; i < v_size(keys); i++) {
        counts[i] = this.get_count(keys[i]);
    }
    return counts;
}
number count_set_date.get_count(date key) {
    number c = the_set_.find(key);
    return null(c) ? 0 : c;
}


class pair_str_str {
public:
    pair_str_str(string option(nullable) first, string option(nullable) second);
    string first;
    string second;
    string repr();
};
pair_str_str.pair_str_str(string option(nullable) first, string option(nullable) second)
: first(first), second(second)
{}
string pair_str_str.repr()
{
    string first_repr = null(first) ? "<null>" : first;
    string second_repr = null(second) ? "<null>" : second;
    return strcat(["(", first_repr, ", ", second_repr, ")"]);
}


class set_pair_str_str {
public:
    set_pair_str_str();
    void add(pair_str_str p);
    void remove(pair_str_str p);
    logical includes(pair_str_str p);
    vector(pair_str_str) get_content();

private:
    set_str pair_str_set_;
    string DELIMITER_;
    integer DELIMITER_LEN_;
    string NULL_REPR_;

    string _to_str(pair_str_str p);
    pair_str_str _from_str(string s);
};
set_pair_str_str.set_pair_str_str()
: pair_str_set_(set_str()),
    DELIMITER_("}]..[{"),
    DELIMITER_LEN_(strlen(DELIMITER_)),
    NULL_REPR_("*(*N/U/L/L+]+")
{}
void set_pair_str_str.add(pair_str_str p) { pair_str_set_.add(_to_str(p)); }
void set_pair_str_str.remove(pair_str_str p) { pair_str_set_.remove(_to_str(p)); }
logical set_pair_str_str.includes(pair_str_str p)  = pair_str_set_.includes(_to_str(p));
vector(pair_str_str) set_pair_str_str.get_content() = _from_str(pair_str_set_.get_content());

string set_pair_str_str._to_str(pair_str_str p) = strcat([null(p.first) ? NULL_REPR_ : p.first,
                                                          DELIMITER_,
                                                          null(p.second) ? NULL_REPR_ : p.second
                                                          ]);
pair_str_str set_pair_str_str._from_str(string s)
{
    number delim_pos = str_find(s, DELIMITER_);
    if (null(delim_pos)) {
        throw (E_UNSPECIFIC, strcat("Internal - invalid pair string: ", s));
    }
    string first = qu.sub_string_start(s, integer(delim_pos));
    string second = qu.sub_string_end(s, integer(delim_pos) + DELIMITER_LEN_);
    pair_str_str p = new pair_str_str(first == NULL_REPR_ ? null<string> : first,
                                      second == NULL_REPR_ ? null<string> : second);
    return p;
}

public class counting_alarm option(category: "Utilities Enhanced/Extras") {
public:
    counting_alarm(number first_alarm = 1, number alarm_factor = 2);
    logical increase(); // returns true if alarm is called
    number current_count();
    number next_alarm();

private:
    number next_alarm_;
    number alarm_factor_;
    number count_;
};
counting_alarm.counting_alarm(number first_alarm, number alarm_factor)
: next_alarm_(first_alarm), alarm_factor_(alarm_factor), count_(0)
{}
logical counting_alarm.increase()
{
    count_++;
    if (count_ == next_alarm_) {
        next_alarm_ *= alarm_factor_;
        return true;
    } else {
        return false;
    }
}
number counting_alarm.current_count() = count_;
number counting_alarm.next_alarm() = next_alarm_;


    module flag {

        class FlagManager {
        public:
            FlagManager();
            void set(string flag);
            logical is_set(string flag, string observer);
            void unset(string flag, string observer);

        private:
            map_str_set_str flag_2_unsetters_;
        };
        FlagManager.FlagManager()
            : flag_2_unsetters_(new map_str_set_str())
        {}
        void FlagManager.set(string flag)
        {
            set_str unsetters = set_str();
            flag_2_unsetters_.add(flag, unsetters);
        }
        logical FlagManager.is_set(string flag, string observer)
        {
            set_str unsetters = flag_2_unsetters_.find(flag);
            if (null(unsetters)) {
                return false;
            } else {
                return !unsetters.includes(observer);
            }
        }
        void FlagManager.unset(string flag, string observer)
        {
            set_str unsetters = flag_2_unsetters_.find(flag);
            if (!null(unsetters)) {
                unsetters.add(observer);
            }
        }

        FlagManager __flag_manager__;
        FlagManager __flag_manager()
        {
            if (null(__flag_manager__)) {
                __flag_manager__ = new FlagManager();
            }
            return __flag_manager__;
        }

        public void set(string flag) { __flag_manager().set(flag); }
        public logical is_set(string flag, string observer) { return __flag_manager().is_set(flag, observer); }
        public void unset(string flag, string observer) { __flag_manager().unset(flag, observer); }
    }

    module timing {
        module __impl__ {

            map_str_num       _Name2Index = map_str_num();
            vector(string)    _Names[0];
            vector(timestamp) _Starts[0];
            vector(integer)   _Counts[0]; // Only include finished timings
            vector(number)    _Totals[0]; // Only include finished timings
            vector(number)    _Parents[0]; // Index of timing event to be started before
            number            _Current; // Index of last started
            integer           _Size = 0;

            logical           _PathEnabled = false;
            count_set         _PathTotal = new count_set();
            count_set         _PathCount = new count_set();
            string            _PathDelimWithSpaces = " -> ";

            timestamp _WorkStart;
            number _SelfCost = 0;

            void reset()
            {
                _Name2Index = map_str_num();
                resize(_Names, 0);
                resize(_Starts, 0);
                resize(_Counts, 0);
                resize(_Totals, 0);
                resize(_Parents, 0);
                _Current = null<number>;
                _Size = 0;

                _PathEnabled = false;
                _PathTotal = new count_set();
                _PathCount = new count_set();

                _WorkStart = null<timestamp>;
                _SelfCost = 0;
            }

            integer index(string name)
            {
                number ix = _Name2Index.find(name);
                if (null(ix)) {
                    ix = _Size;
                    push_back(_Names, name);
                    push_back(_Starts, null<timestamp>);
                    push_back(_Counts, 0);
                    push_back(_Totals, 0);
                    push_back(_Parents, null<number>);
                    _Name2Index.add(name, ix);
                    _Size++;
                }
                return integer(ix);
            }

            vector(integer) path_indices(integer from_ix)
            {
                vector(integer) indices[0];
                number ix = from_ix;
                while (!null(ix)) {
                    push_back(indices, integer(ix));
                    ix = _Parents[integer(ix)];
                }
                v_reverse(indices);
                return indices;
            }

            void start(integer ix, timestamp start_time)
            {
                _Starts[ix] = start_time;
                _Parents[ix] = _Current;
                _Current = ix;
            }

            void end(integer ix, timestamp end_time)
            {
                if (null(_Starts[ix])) {
                    throw (E_UNSPECIFIC, strcat(["timing internal: tried to end ", str(ix), " but it has no start time"]));
                }
                if (null(_Current)) {
                    throw (E_UNSPECIFIC, "timing internal: tried to end though nothing runs");
                }
                if (ix != _Current) {
                    throw (E_UNSPECIFIC, strcat(["timing internal: tried to end ", str(ix), " but current is ", str(_Current)]));
                }
                number time = end_time - _Starts[ix];
                if (_PathEnabled) {
                    string path = str_join(_Names[path_indices(ix)], _PathDelimWithSpaces);
                    _PathCount.increase(path);
                    _PathTotal.increase(path, time);
                }
                _Current = _Parents[ix];
                _Totals[ix] += time;
                _Starts[ix] = null<timestamp>;
                _Parents[ix] = null<number>;
                _Counts[ix]++;
            }

            void end(timestamp end_time)
            {
                if (!null(_Current)) {
                    end(integer(_Current), end_time);
                }
            }

            void end(string name, logical inclusive, timestamp end_time)
            {
                vector(integer) end_list[0];
                number ix = _Current;
                logical name_found = false;
                while (!name_found && !null(ix)) {
                    if (_Names[integer(ix)] == name) {
                        name_found = true;
                        if (!inclusive) {
                            break;
                        }
                    }
                    push_back(end_list, integer(ix));
                    ix = _Parents[integer(ix)];
                }
                if (!name_found) {
                    // "name" not running
                    return;
                }
                integer end_list_sz = v_size(end_list);
                for (integer i = 0; i < end_list_sz; ++i) {
                    end(end_list[i], end_time);
                }
            }

            void start(string name, timestamp start_time)
            {
                end(name, true, start_time);
                integer ix = index(name);
                start(ix, start_time);
            }

            void end_all(timestamp end_time)
            {
                while (!null(_Current)) {
                    end(integer(_Current), end_time);
                }
            }

            void start_next(string name, timestamp start_time)
            {
                end(start_time);
                start(name, start_time);
            }

            void start_top(string name, timestamp start_time)
            {
                end_all(start_time);
                start(name, start_time);
            }

            vector(string) timer_list()
            {
                return _Name2Index.get_keys();
            }

            number timer_time(string name)
            {
                return _Totals[index(name)];
            }

            integer timer_count(string name)
            {
                return _Counts[index(name)];
            }


            void track_paths()
            {
                _PathEnabled = true;
            }

            vector(string) path_list()
            {
                return _PathTotal.get_keys();
            }

            number path_time(string path)
            {
                return _PathTotal.get_count(path);
            }

            integer path_count(string path)
            {
                return integer(_PathCount.get_count(path));
            }

            integer path_depth(string path)
            {
                return v_size(str_find_all(path, __impl__._PathDelimWithSpaces));
            }


            number self_cost() = _SelfCost;

            vector(string) inspect_state()
            {
                vector(string) state[0];
                for (integer i = 0; i < _Size; ++i) {
                    number pix = _Parents[i];
                    push_back(
                        state,
                        strcat([
                            _Names[i], " @ ", str(_Starts[i]),
                            null(pix) ?
                                " -> " :
                                strcat([" -> ", _Names[integer(pix)], " [", str(pix), "] "]),
                        "Sum: ", str(_Totals[i]), " / ", str(_Counts[i])]));
                }
                if (null(_Current)) {
                    push_back(state, "Current: <none>");
                } else {
                    push_back(
                        state,
                        strcat(["Current: ", _Names[integer(_Current)], " [", str(_Current), "]"]));
                }
                return state;
            }

            vector(string) running_timers_data(timestamp check_time)
            {
                vector(string) timer_data[0];
                number ix = _Current;
                while (!null(ix)) {
                    push_back(
                        timer_data,
                        strcat([
                            "'", _Names[integer(ix)], "' since ", str(_Starts[integer(ix)]),
                            " (", human_time(check_time - _Starts[integer(ix)]), ")"]));
                    ix = _Parents[integer(ix)];
                }
                return timer_data;
            }

        }

        // Dummy class to allow making settings in compile stage
        class preset_obj {};
        preset_obj __preset_obj__;

        // Usage (enable path tracking example):
        // qu.timing.preset_obj timing_obj = qu.timing.track_paths();

        // Starts timer "name" (if it is running, ends it and child timers first)
        public void start(string name) option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            __impl__.start(name, now());
            __impl__._SelfCost += now() - __impl__._WorkStart;
        }

        // ends last started timer
        public void end() option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            __impl__.end(now());
            __impl__._SelfCost += now() - __impl__._WorkStart;
        }

        // ends last started timer and starts timer "name"
        public void start_next(string name) option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            __impl__.start_next(name, now());
            __impl__._SelfCost += now() - __impl__._WorkStart;
        }

        // ends all running timers and starts timer "name"
        public void start_top(string name) option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            __impl__.start_top(name, now());
            __impl__._SelfCost += now() - __impl__._WorkStart;
        }

        // ends timer "name" and all child timers of it
        public void end(string name)
        {
            __impl__._WorkStart = now();
            __impl__.end(name, true, now());
            __impl__._SelfCost += now() - __impl__._WorkStart;
        }

        // ends child timers to "name", but not the timer "name"
        public void end_after(string name)
        {
            __impl__._WorkStart = now();
            __impl__.end(name, false, now());
            __impl__._SelfCost += now() - __impl__._WorkStart;
        }

        // ends all timers
        public void end_all() option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            __impl__.end_all(now());
            __impl__._SelfCost += now() - __impl__._WorkStart;
        }

        // returns a list with the names of all timers, running or not
        public vector(string) timer_list() option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            vector(string) r = __impl__.timer_list();
            __impl__._SelfCost += now() - __impl__._WorkStart;
            return r;
        }

        public number timer_time(string name) option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            number r = __impl__.timer_time(name);
            __impl__._SelfCost += now() - __impl__._WorkStart;
            return r;
        }

        public integer timer_count(string name) option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            integer r = __impl__.timer_count(name);
            __impl__._SelfCost += now() - __impl__._WorkStart;
            return r;
        }



        // If individual timers aren't enough, enable paths! Paths are the timer hierarchies.
        // Examples (calc_all, prepare_calc, fetch_data, check_data are all names of timers):
        // calc_all -> prepare_calc -> fetch_data
        // calc_all -> prepare_calc -> check_data
        // prepare_calc -> fetch_data
        // prepare_calc -> check_data
        // See? You get prepare calc split on when it is part of calc_all and when it is
        // free standing.
        public preset_obj track_paths() option(category: "Utilities Enhanced/Timing") { __impl__.track_paths(); return __preset_obj__; }

        // returns a list with the names of all timer paths, running or not
        public vector(string) path_list() option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            vector(string) r = __impl__.path_list();
            __impl__._SelfCost += now() - __impl__._WorkStart;
            return r;
        }

        public number path_time(string name) option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            number r = __impl__.path_time(name);
            __impl__._SelfCost += now() - __impl__._WorkStart;
            return r;
        }

        public integer path_count(string name) option(category: "Utilities Enhanced/Timing")
        {
            __impl__._WorkStart = now();
            integer r = __impl__.path_count(name);
            __impl__._SelfCost += now() - __impl__._WorkStart;
            return r;
        }


    /* Cannot use interpolation - it requires instrument dll or something...
        // Colors from colorbrewer2.org, 10 points, diverging RdYlGn - skipping last 2 green colors and first red
        interpolation red_ip = interpolation(ip_linear(), qu.v_range(7)/7, [102, 166, 217, 254, 253, 244, 215]);
        interpolation green_ip = interpolation(ip_linear(), qu.v_range(7)/7, [189, 217, 239, 224, 174, 109, 48]);
        interpolation blue_ip = interpolation(ip_linear(), qu.v_range(7)/7, [99, 106, 139, 139, 97, 67, 39]);
    */
        number interpolate(number low_x,
                           number low_y,
                           number high_x,
                           number high_y,
                           number at_x)
        {
            // solve y = kx+m
            number k = (high_y - low_y) / (high_x - low_x);
            number m = low_y - k*low_x;
            number y_at_x = k*at_x + m;
            return y_at_x;
        }


        number severity_color(number val, number max_val)
        {
            number severity = val/max_val;
            /*
            number r = round(red_ip.eval(severity));
            number g = round(green_ip.eval(severity));
            number b = round(blue_ip.eval(severity));
            */
            integer r = integer(round(severity < 0.5 ? interpolate(0, 102, 0.5, 254, severity) : interpolate(0.5, 254, 1, 215, severity), 0));
            integer g = integer(round(severity < 0.4 ? interpolate(0, 189, 0.4, 240, severity) : interpolate(0.4, 240, 1, 48, severity), 0));
            integer b = integer(round(severity < 0.4 ? interpolate(0, 99, 0.4, 139, severity) : interpolate(0.4, 139, 1, 39, severity), 0));
            return rgb(r,g,b);
        }

        vector(number) rgb_sequential_green_blue(number n_colors)
        {
/*
-- This section requires interpolation, but that's not generally available
            vector(number) red_y = [247, 224, 204, 168, 123, 78, 43, 8];
            vector(number) green_y = [252, 243, 235, 221, 204, 179, 140, 88];
            vector(number) blue_y = [240, 219, 197, 181, 196, 211, 190, 158];
            if (n_colors < 1) {
                return empty_num_vec();
            }
            if (n_colors == 1) {
                return rgb(red_y[0], green_y[0], blue_y[0]);
            }
            number scaling = (v_size(red_y)-1) / (n_colors-1);
            vector(number) xs = v_range(n_colors) .* scaling;
            red = round(ip_linear(xs, k_x, rb));
            green = round(ip_linear(xs, k_x, gb));
            blue = round(ip_linear(xs, k_x, bb));
*/
            switch (n_colors) {
                case 1: return [rgb(224,243,219)];
                case 2: return [rgb(224,243,219), rgb(67,162,202)];
                case 3: return [rgb(224,243,219), rgb(168,221,181), rgb(67,162,202)];
                case 4: return [rgb(240,249,232), rgb(186,228,188), rgb(123,204,196), rgb(43,140,190)];
                case 5: return [rgb(240,249,232), rgb(186,228,188), rgb(123,204,196), rgb(67,162,202), rgb(8,104,172)];
                case 6: return [rgb(240,249,232), rgb(204,235,197), rgb(168,221,181), rgb(123,204,196), rgb(67,162,202), rgb(8,104,172)];
                case 7: return [rgb(240,249,232), rgb(204,235,197), rgb(168,221,181), rgb(123,204,196), rgb(78,179,211), rgb(43,140,190), rgb(8,88,158)];
                case 8: return [rgb(247,252,240), rgb(224,243,219), rgb(204,235,197), rgb(168,221,181), rgb(123,204,196), rgb(78,179,211), rgb(43,140,190), rgb(8,88,158)];
                default: throw (E_UNSPECIFIC, "rgb_sequential_green_blue can only produce 1-8 colors");
            }
        }


        public matrix(text_rgb) table(logical track_paths, logical reset_afterwards)
        {
            vector(string) timers = __impl__.timer_list();
            vector(number) timer_times = __impl__.timer_time(timers);
            vector(integer) timer_counts = __impl__.timer_count(timers);
            vector(number) timer_avgs = timer_times / timer_counts;
            sort(timers, -timer_times); // "-" for reverse sort
            sort(timer_counts, -timer_times);
            sort(timer_avgs, -timer_times);
            sort(timer_times, -timer_times);
            number max_timer_time = v_max(timer_times);
            number max_timer_avg = v_max(timer_avgs);

            vector(string) paths = __impl__.path_list();
            vector(number) path_times = __impl__.path_time(paths);
            vector(integer) path_counts = __impl__.path_count(paths);
            vector(number) path_avgs = path_times / path_counts;
            vector(number) path_depths = __impl__.path_depth(paths);
            number max_path_time = v_max(path_times);
            number max_path_avg = v_max(path_avgs);

            integer n_timers = v_size(timers);
            integer n_paths = v_size(paths);

            number header_color = rgb(200,200,200);

            integer n_rows = 1 + (__impl__._PathEnabled ? ..max(n_timers, n_paths) : n_timers);
            integer n_cols = __impl__._PathEnabled ? 9 : 5;
            integer cost_column = n_cols - 1;

            matrix(text_rgb) ret_mx[n_rows, n_cols];
            ret_mx[0,0] = text_rgb("Timer", null<number>, header_color);
            ret_mx[0,1] = text_rgb("Count", null<number>, header_color);
            ret_mx[0,2] = text_rgb("Total", null<number>, header_color);
            ret_mx[0,3] = text_rgb("Avg", null<number>, header_color);

            if (__impl__._PathEnabled) {
                ret_mx[0,4] = text_rgb("Path", null<number>, header_color);
                ret_mx[0,5] = text_rgb("Count", null<number>, header_color);
                ret_mx[0,6] = text_rgb("Total", null<number>, header_color);
                ret_mx[0,7] = text_rgb("Avg", null<number>, header_color);
            }

            ret_mx[0, cost_column] = text_rgb(strcat("Timing cost: ", human_time(__impl__.self_cost())));

            for (integer i = 0; i < n_timers; ++i) {
                ret_mx[i+1, 0] = text_rgb(timers[i]);
                ret_mx[i+1, 1] = text_rgb(str(timer_counts[i]));
                ret_mx[i+1, 2] = text_rgb(human_time(timer_times[i]), null<number>, severity_color(timer_times[i], max_timer_time));
                ret_mx[i+1, 3] = text_rgb(human_time(timer_avgs[i]), null<number> , severity_color(timer_avgs[i], max_timer_avg));
            }
            if (__impl__._PathEnabled &&  n_paths > 0) {
                vector(integer) path_depths_capped = integer(..min(path_depths, 7)); // We know rgb_sequential_green_blue cannot produce more than 8 colors
                integer max_path_depth = integer(v_max(path_depths_capped));
                vector(number) colors = rgb_sequential_green_blue(max_path_depth+1);
                for (integer i = 0; i < n_paths; ++i) {
                    number color = colors[path_depths_capped[i]];
                    ret_mx[i+1, 4] = text_rgb(paths[i], null<number>, color, false);
                    ret_mx[i+1, 5] = text_rgb(str(path_counts[i]), null<number>, color, false);
                    ret_mx[i+1, 6] = text_rgb(human_time(path_times[i]), null<number>, severity_color(path_times[i], max_path_time));
                    ret_mx[i+1, 7] = text_rgb(human_time(path_avgs[i]), null<number>, severity_color(path_avgs[i], max_path_avg));
                }
            }

            if (reset_afterwards) {
                __impl__.reset();
            }
            if (track_paths) {
                __impl__.track_paths();
            }

            return ret_mx;
        }

        // Function below are for debugging, or other non-standard actions, and they don't
        // count towards the self cost
        public number self_cost() option(category: "Utilities Enhanced/Timing")
        {
            return __impl__.self_cost();
        }

        public vector(string) inspect_state() option(category: "Utilities Enhanced/Timing")
        {
            return __impl__.inspect_state();
        }

        public vector(string) running_timers_data() option(category: "Utilities Enhanced/Timing")
        {
            return __impl__.running_timers_data(now());
        }

        // Resets everything.
        // Including self cost, settings (such as disabling paths)
        public void reset() option(category: "Utilities Enhanced/Timing")
        {
            __impl__.reset();
        }


    }


    public vector(integer) random_order(integer n, rng option(nullable) rand_gen = null<rng>) option(category: "Utilities Enhanced/Extras")
    {
        if (null(rand_gen)) {
            rand_gen = rng();
        }

        vector(integer) v = v_range(n);

        for (integer i = 0; i < n; ++i) {
            integer r = n-i;
            integer offset = rand_gen.integer(r);
            integer pos = i+offset;
            integer tmp = v[pos];
            v[pos] = v[i];
            v[i] = tmp;
        }
        return v;
    }

    public class file_reader option(category: "Utilities Enhanced/IO") {
      public:
        file_reader(string file_name);
        string read_line();
        string peek_line();
        integer line_count();
        void close();

      private:
        in_stream fis_;
        string peeked_line_;
        logical peeked_exists_;
        integer line_count_;
        logical closed_;

        void _throw_if_closed();
    };
    file_reader.file_reader(string file_name)
        : peeked_exists_(false), line_count_(0), closed_(false)
    {
        fis_ = in_stream_file(file_name);
    }
    string file_reader.read_line()
    {
        _throw_if_closed();
        string line;
        if (peeked_exists_) {
            line = peeked_line_;
            peeked_exists_ = false;
        } else {
            line = fis_.read_line();
        }
        if (!null(line)) {
            line_count_++;
        }
        return line;
    }
    string file_reader.peek_line()
    {
        _throw_if_closed();
        if (!peeked_exists_) {
            peeked_line_ = fis_.read_line();
            peeked_exists_ = true;
        }
        return peeked_line_;
    }
    integer file_reader.line_count() = line_count_;
    void file_reader.close()
    {
        _throw_if_closed();
        fis_.close();
        closed_ = true;
    }
    void file_reader._throw_if_closed()
    {
        if (closed_) {
            throw (E_UNSPECIFIC, "File stream has been closed");
        }
    }


    object ascii_table_row {
        vector(string) elems;
    };
    ascii_table_row ascii_table_row(vector(string) elems)
    {
        ascii_table_row obj = new ascii_table_row;
        obj.elems = elems;
        return obj;
    }

    /*
     * ascii_table
     *
     * Easy creating of tables to print to file, or show in workspace.
     *
     * Call set_headers and add_row to fill with data, then print or get_lines
     * to get the result.
     *
     * If you don't want the data stored, you can get the lines one by one with
     * the format-functions.
     *
     * Get good looking table without any hassle
     * -- or --
     * Customize output format in many ways (column widths, column alignments,
     * column separators, row separators).
     *
     * Number of columns can be set at creation, otherwise it is set by first
     * passing of a vector expected to cover all columns (such as set_headers or
     * add_row).
     *
     *
     *  The Default Settings:
     * - Column separator is " | ", that is space-pipe-space
     * - Row separator is "-" (repeated to fill cell, row separator used after header line)
     * - Alignment is left for headers and data
     * - Neither left nor right outer delimiter is used
     */

    class ascii_table {
    public:
        ascii_table(integer n_cols = -1);

        // Fill with data
        void add_row(vector(string) elems);
        void set_headers(vector(string) headers);


        // Customize output format
        void set_column_delimiter(string delim);

        void set_column_width(integer width); // Use width for all columns
        void set_column_width(vector(integer) width); // Use these widths in column order
        void set_column_width(integer col_index, integer width); // Use width for column col_index

        void set_column_align(string align); // Use align for all columns
        void set_column_align(vector(string) align); // Use these aligns in column order
        void set_column_align(integer col_index, string align); // Use align for column col_index

        void set_header_align(string align); // Use align for all headers
        void set_header_align(vector(string) align); // Use these aligns in column order for headers
        void set_header_align(integer col_index, string align); // Use align for header col_index

        void set_left_outer_delimiter(string delim);
        void set_right_outer_delimiter(string delim);
        void set_outer_delimiter(string delim); // Use delim for both left and right

        // sep is repeated to fill column. If null, no row separator will be used
        // below headers.
        void set_row_separator(string option(nullable) sep);

        // Sort table contents
        // (Sorts existing rows, new rows are added at bottom.)
        void sort(integer col_index, logical descending = false);
        void sort_numeric(integer col_index, logical descending = false);

        // Generate rows
        string format_header(vector(string) headers);
        string format_row_separator(string option(nullable) separator = null<string>); // null => default
        string format_line(vector(string) elems); // column widths must be set when calling this
        // Generate full table
        vector(string) get_lines();
        void print(out_stream os, string newline = "\r\n");

    private:
        integer n_cols_;
        vector(string) headers_;
        vector(ascii_table_row) rows_;
        vector(string) aligns_;
        vector(integer) widths_;
        vector(string) header_aligns_;
        string default_align_;
        integer default_width_;
        string default_header_align_;
        string delim_;
        string left_outer_delim_;
        string right_outer_delim_;
        string row_sep_;

        void _check_size(integer n);
        void _check_size(vector(string) v);
        void _check_size(vector(integer) v);
        void _check_column_index(integer col_index);
        void _init_size(integer n);
        vector(integer) _largest_widths();
        string _make_line(vector(string) elems, vector(integer) widths, vector(string) aligns);
        string _make_row_separator(vector(integer) widths, string separator);
        vector(string) _get_lines();
    };
    ascii_table.ascii_table(integer n_cols)
        :
        n_cols_(-1),
        default_align_(null<string>),
        default_width_(-1),
        delim_(" | "),
        left_outer_delim_(null<string>),
        right_outer_delim_(null<string>),
        row_sep_("-")
    {
        if (n_cols_ != -1) {
            _init_size(n_cols);
        }
        resize(rows_, 0);
    }

    void ascii_table.add_row(vector(string) elems)
    {
        _check_size(elems);
        push_back(rows_, ascii_table_row(elems));
    }

    void ascii_table.set_headers(vector(string) headers)
    {
        _check_size(headers);
        headers_ = headers;
    }

    void ascii_table.set_column_delimiter(string delim)
    {
        delim_ = delim;
    }

    void ascii_table.set_column_width(integer width)
    {
        if (!null(widths_)) {
            widths_ = width;
        }
        default_width_ = width;
    }

    void ascii_table.set_column_width(integer col_index, integer width)
    {
        _check_column_index(col_index);
        widths_[col_index] = width;
    }

    void ascii_table.set_column_width(vector(integer) width)
    {
        _check_size(width);
        for (integer i = 0; i < n_cols_; ++i) {
            widths_[i] = width[i];
        }
    }

    void ascii_table.set_column_align(string align)
    {
        if (!null(aligns_)) {
            aligns_ = align;
        }
        default_align_ = align;
    }

    void ascii_table.set_column_align(integer col_index, string align)
    {
        _check_column_index(col_index);
        aligns_[col_index] = align;
    }

    void ascii_table.set_column_align(vector(string) align)
    {
        _check_size(align);
        for (integer i = 0; i < n_cols_; ++i) {
            aligns_[i] = align[i];
        }
    }

    void ascii_table.set_header_align(string align)
    {
        if (!null(header_aligns_)) {
            header_aligns_ = align;
        }
        default_header_align_ = align;
    }

    void ascii_table.set_header_align(integer col_index, string align)
    {
        _check_column_index(col_index);
        header_aligns_[col_index] = align;
    }

    void ascii_table.set_header_align(vector(string) align)
    {
        _check_size(align);
        for (integer i = 0; i < n_cols_; ++i) {
            header_aligns_[i] = align[i];
        }
    }

    void ascii_table.set_outer_delimiter(string delim)
    {
        this.set_left_outer_delimiter(delim);
        this.set_right_outer_delimiter(delim);
    }

    void ascii_table.set_left_outer_delimiter(string delim)
    {
        left_outer_delim_ = delim;
    }

    void ascii_table.set_right_outer_delimiter(string delim)
    {
        right_outer_delim_ = delim;
    }

    void ascii_table.set_row_separator(string option(nullable) sep)
    {
        row_sep_ = sep;
    }

    void ascii_table.sort(integer col_index, logical descending)
    {
        _check_column_index(col_index);
        integer n = v_size(rows_);
        vector(string) vals[n];
        for (integer i = 0; i < n; ++i) {
            vals[i] = rows_[i].elems[col_index];
        }

        ..sort(rows_, vals);
        if (descending) {
            v_reverse(rows_);
        }
    }

    void ascii_table.sort_numeric(integer col_index, logical descending)
    {
        _check_column_index(col_index);
        integer n = v_size(rows_);
        vector(number) vals[n];
        for (integer i = 0; i < n; ++i) {
            try {
                vals[i] = str_to_number(rows_[i].elems[col_index]);
            } catch {
                vals[i] = null<number>;
            }
        }

        ..sort(rows_, vals);
        if (descending) {
            v_reverse(rows_);
        }
    }

    string ascii_table.format_header(vector(string) headers)
    {
        for (integer i = 0; i < n_cols_; ++i) {
            if (widths_[i] == -1) {
                throw (E_UNSPECIFIC, "Columns must have widths set before calling format_line");
            }
        }
        return _make_line(headers, widths_, header_aligns_);
    }

    string ascii_table.format_row_separator(string option(nullable) separator)
    {
        if (null(separator)) {
            separator = row_sep_;
        }
        return _make_row_separator(widths_, separator);
    }

    string ascii_table.format_line(vector(string) elems)
    {
        for (integer i = 0; i < n_cols_; ++i) {
            if (widths_[i] == -1) {
                throw (E_UNSPECIFIC, "Columns must have widths set before calling format_line");
            }
        }
        return _make_line(elems, widths_, aligns_);
    }

    vector(string) ascii_table.get_lines() = _get_lines();

    void ascii_table.print(out_stream os, string newline)
    {
        vector(string) lines = _get_lines();
        integer lines_sz = v_size(lines);
        for (integer i = 0; i < lines_sz; ++i) {
            os.write(lines[i]);
            os.write(newline);
        }
    }

    void ascii_table._check_size(integer n)
    {
        if (n_cols_ == -1) {
            _init_size(n);
        } else if (n != n_cols_) {
            throw (E_UNSPECIFIC, strcat(["Wrong vector size: ", str(n), " (expected ", str(n_cols_), ")"]));
        }
    }

    void ascii_table._check_size(vector(string) v)
    {
        _check_size(v_size(v));
    }

    void ascii_table._check_size(vector(integer) v)
    {
        _check_size(v_size(v));
    }

    void ascii_table._init_size(integer n)
    {
        n_cols_ = n;
        resize(aligns_, n);
        aligns_ = default_align_;
        resize(widths_, n);
        widths_ = default_width_;
        resize(header_aligns_, n);
        header_aligns_ = default_header_align_;
    }

    void ascii_table._check_column_index(integer col_index)
    {
        if (n_cols_ == -1) {
            throw (E_UNSPECIFIC, "Cannot refer to column index when number of columns is unknown");
        }
        if (col_index < 0 || col_index >= n_cols_) {
            throw (E_UNSPECIFIC, strcat("Invalid column index: ", str(col_index)));
        }
    }

    vector(integer) ascii_table._largest_widths()
    {
        if (n_cols_ < 0) {
            vector(integer) v[0];
            return v;
        }

        vector(integer) ret[n_cols_];
        ret = 0;

        if (!null(headers_)) {
            for (integer i = 0; i < n_cols_; ++i) {
                ret[i] = ..max(ret[i], strlen(headers_[i]));
            }
        }

        integer rows__sz = v_size(rows_);
        for (integer r = 0; r < rows__sz; ++r) {
            for (integer i = 0; i < n_cols_; ++i) {
                ret[i] = ..max(ret[i], strlen(rows_[r].elems[i]));
            }
        }

        return ret;
    }

    string ascii_table._make_line(
        vector(string) elems, vector(integer) widths, vector(string) aligns)
    {
        string line = concat_str_row(elems, widths, aligns, -1, delim_);
        if (!null(left_outer_delim_) || !null(right_outer_delim_)) {
            line = strcat([left_outer_delim_, line, right_outer_delim_]);
        }
        return line;
    }

    string ascii_table._make_row_separator(vector(integer) widths, string separator)
    {
        vector(string) elems[n_cols_];
        for (integer i = 0; i < n_cols_; ++i) {
            vector(string) sep_elems[widths[i]];
            sep_elems = separator;
            elems[i] = strcat(sep_elems);
        }
        return _make_line(elems, widths, aligns_);
    }

    vector(string) ascii_table._get_lines()
    {
        vector(integer) widths_to_use = _largest_widths();
        for (integer i = 0; i < n_cols_; ++i) {
            if (widths_[i] != -1) {
                widths_to_use[i] = widths_[i];
            }
        }

        vector(string) lines[0];
        if (!null(headers_)) {
            push_back(lines, _make_line(headers_, widths_to_use, header_aligns_));

            if (!null(row_sep_)) {
                push_back(lines, _make_row_separator(widths_to_use, row_sep_));
            }
        }

        integer rows__sz = v_size(rows_);
        for (integer i = 0; i < rows__sz; ++i) {
            push_back(lines, _make_line(rows_[i].elems, widths_to_use, aligns_));
        }

        return lines;
    }

    ascii_table ascii_table(matrix(string) m)
    {
        ascii_table t = new ascii_table(n_cols(m));
        integer n = n_rows(m);
        for (integer i = 0; i < n; ++i) {
            t.add_row(m[i,:]);
        }
        return t;
    }



/*
 * Decision table
 *
 * Easy to set up and work with. Check out class value above to understand it.
 *
 *
 * Example usage:
    vector(qu.value) big_cats = [new qu.value("LION"), new qu.value("TIGER"), new qu.value("LEOPARD")];
    vector(qu.value) boring_continents = [new qu.value("EUROPE"), new qu.value("NORTH AMERICA")];
    decision_table dtab = new decision_table(DT_LOGICAL, [DT_STR, DT_LOGICAL, DT_STR]);
    dtab.add_rule(
        qu.value(true),
        [new cell_rule(rule_operator.IN, big_cats)
         , new cell_rule(rule_operator.IN, [new qu.value(false)])
         , new cell_rule(rule_operator.NOT_IN, boring_continents)
         ]);
    // more rules...

    // Using it
    qu.value is_exotic_male_big_cat = dtab.decide(
        [new qu.value("BEAR"), new qu.value(true), new qu.value("ASIA")]);

    // l is the logical member in a qu.value, other members are s, n, d and t
    // for string, number, date and timestamp
    if (is_exotic_male_big_cat.l) {
        warning("Watch out!");
    }

 */

module rule_operator
{

enum instance {
    SKIP option(str: "<SKIP>")
    , IN option(str: "<IN>")
    , NOT_IN option(str: "<NOT IN>")
};

instance from_opt(integer i)
{
    switch(i)
    {
        case 0 : return SKIP;
        case 1 : return IN;
        case 2 : return NOT_IN;
        default : return null<instance>;
    }
}

/*
 * NO CHANGES NEEDED BELOW THIS LINE
 */

instance from(integer i)
{
    instance e = from_opt(i);
    if (null(e))
        throw(E_INVALID_ARG, strcat(["No enum found matching the integer ", string(i), "."]));
    return e;
}

instance from(string s)
{
    instance e;
    integer i = 0;
    while (!null(e = from_opt(i++)))
    {
        if (s == string(e))
            return e;
    }
    throw(E_INVALID_ARG, strcat(["No enum found matching the string '", s, "'."]));
}

instance from_casei(string s)
{
    instance e;
    integer i = 0;
    while (!null(e = from_opt(i++)))
    {
        if (equal_casei(s, string(e)))
            return e;
    }
    throw(E_INVALID_ARG, strcat(["No enum found matching the string '", s, "'."]));
}

integer to_int(instance e)
{
     integer i = -1;
     while (from(++i) != e) {
     }
     return i;
}

string to_str(instance e) = string(e);

vector(instance) values()
{
    vector(instance) v_e[0];
    instance e;
    integer i = 0;
    while (!null(e = from_opt(i++)))
    {
        push_back(v_e, e);
    }
    return v_e;
}

}


string rule_operator_to_small_string(rule_operator.instance op)
{
     switch (op) {
     case rule_operator.SKIP: return "-";
     case rule_operator.IN: return "=";
     case rule_operator.NOT_IN: return "not";
     default: throw (E_UNSPECIFIC, "Unknown rule_operator");
     }
}


class cell_rule {
public:
    cell_rule(rule_operator.instance op, logical accept_null);
    cell_rule(rule_operator.instance op, logical accept_null, vector(value) values);
    cell_rule(data_type dt, string rule_str); // Result of export() works

    rule_operator.instance op();
    vector(value) values();

    logical passes(value val, out logical found_match);
    string repr();
    string repr_short();
    string export(); // All info needed to recreate

private:
    rule_operator.instance op_;
    logical accept_null_;
    vector(value) values_;
    integer n_;

    void _init(rule_operator.instance op, logical accept_null, vector(value) values);
};

cell_rule.cell_rule(rule_operator.instance op, logical accept_null)
{
    vector(value) empty[0];
    _init(op, accept_null, empty);
}

cell_rule.cell_rule(rule_operator.instance op, logical accept_null, vector(value) values)
{
    _init(op, accept_null, values);
}

cell_rule.cell_rule(data_type dt, string rule_str)
{
    vector(string) elems = str_tokenize(rule_str, "|");
    rule_operator.instance op = rule_operator.from(elems[0]);
    logical accept_null;
    if (elems[2] == "null") {
        accept_null = true;
    } else if (elems[2] == "not-null") {
        accept_null = false;
    } else {
        throw (E_UNSPECIFIC, strcat("Invalid rule_str: ", rule_str));
    }

    vector(value) values[0];
    if (op != rule_operator.SKIP) {
        values = value_from_string(dt, str_tokenize(elems[1], ","));
    }

    _init(op, accept_null, values);
}

void cell_rule._init(rule_operator.instance op, logical accept_null, vector(value) values)
{
    op_ = op;
    accept_null_ = accept_null;
    values_ = clone_vector(values);
    n_ = v_size(values);

    switch (op_) {
    case rule_operator.SKIP:
        if (n_ != 0) {
            throw (E_UNSPECIFIC, strcat("No values should be given for operator ", string(op_)));
        }
        break;
    case rule_operator.IN:
    case rule_operator.NOT_IN:
        if (n_ == 0) {
            throw (E_UNSPECIFIC, strcat("Empty value vector not allowed for operator ", string(op_)));
        }
        break;
    default:
        throw (E_UNSPECIFIC, "Internal: Unhandled operator in cell_rule constructor");
    }
}

rule_operator.instance cell_rule.op() = op_;
vector(value) cell_rule.values() = clone_vector(values_);

logical cell_rule.passes(value val, out logical found_match)
{
    found_match = false;
    switch (op_) {
    case rule_operator.SKIP:
        return true;
    case rule_operator.IN:
    case rule_operator.NOT_IN:
        for (integer i = 0; i < n_; ++i) {
            if (qu.equaln(val, values_[i])) {
                found_match = true;
                return op_ == rule_operator.IN;
            }
        }
        if (!accept_null_ && val.is_null()) {
            throw (E_UNSPECIFIC, "value is null");
        }
        return op_ == rule_operator.NOT_IN;
    default:
        throw (E_UNSPECIFIC, "Internal: Unhandled operator in cell_rule.passes");
    }
}

string cell_rule.repr()
{
    if (n_ == 0) {
        return string(op_);
    } else {
        return strcat([string(op_), " {", str_join(values_.repr(), ","), "}"]);
    }
}

string cell_rule.repr_short()
{
    if (n_ == 0) {
        return rule_operator_to_small_string(op_);
    } else {
        return strcat([rule_operator_to_small_string(op_), " {", str_join(values_.repr(), ","), "}"]);
    }
}

string cell_rule.export()
{
    return strcat([string(op_), "|", str_join(values_.repr(), ","), "|", accept_null_ ? "null" : "not-null"]);
}


/*
 * Convenience functions for creating cell rules
 */

/* Skip this rule entirely */
cell_rule cell_rule_skip() = new cell_rule(rule_operator.SKIP, true);

/* Equal to a value, exception on null values */
cell_rule cell_rule_eq(string val) = new cell_rule(rule_operator.IN, false, [qv(val)]);
cell_rule cell_rule_eq(number val) = new cell_rule(rule_operator.IN, false, [qv(val)]);
cell_rule cell_rule_eq(logical val) = new cell_rule(rule_operator.IN, false, [qv(val)]);
cell_rule cell_rule_eq(date val) = new cell_rule(rule_operator.IN, false, [qv(val)]);

/* Not equal to a value, exception on null values */
cell_rule cell_rule_ne(string val) = new cell_rule(rule_operator.NOT_IN, false, [qv(val)]);
cell_rule cell_rule_ne(number val) = new cell_rule(rule_operator.NOT_IN, false, [qv(val)]);
cell_rule cell_rule_ne(logical val) = new cell_rule(rule_operator.NOT_IN, false, [qv(val)]);
cell_rule cell_rule_ne(date val) = new cell_rule(rule_operator.NOT_IN, false, [qv(val)]);

/* Equal to any of several values, exception on null values */
cell_rule cell_rule_in(vector(string) vals) = new cell_rule(rule_operator.IN, false, qv(vals));
cell_rule cell_rule_in(vector(number) vals) = new cell_rule(rule_operator.IN, false, qv(vals));
cell_rule cell_rule_in(vector(logical) vals) = new cell_rule(rule_operator.IN, false, qv(vals));
cell_rule cell_rule_in(vector(date) vals) = new cell_rule(rule_operator.IN, false, qv(vals));

/* Any value not in list, exception on null values */
cell_rule cell_rule_not_in(vector(string) vals) = new cell_rule(rule_operator.NOT_IN, false, qv(vals));
cell_rule cell_rule_not_in(vector(number) vals) = new cell_rule(rule_operator.NOT_IN, false, qv(vals));
cell_rule cell_rule_not_in(vector(logical) vals) = new cell_rule(rule_operator.NOT_IN, false, qv(vals));
cell_rule cell_rule_not_in(vector(date) vals) = new cell_rule(rule_operator.NOT_IN, false, qv(vals));



object decision_row {
    value result;
    vector(cell_rule) cell_rules;
};

decision_row decision_row(value result, vector(cell_rule) cell_rules)
{
    decision_row obj = new decision_row;
    obj.result = result;
    obj.cell_rules = cell_rules;
    return obj;
}

string to_str(decision_row row)
{
    return strcat([str_join(row.cell_rules.repr_short(), "; "), " -> ", row.result.repr("<null>")]);
}

object value_vec {
    vector(value) v;
};

class decision_table {
public:
    decision_table(data_type result_type, vector(data_type) attrib_types);
    decision_table(data_type result_type, vector(data_type) attrib_types, vector(string) attrib_names);
    decision_table(matrix(string) table); // Result of export() works here
    void set_attribute_names(vector(string) attrib_names); // For error messages

    void add_rule(value result, vector(cell_rule) cell_rules);

    // Get result the easy way
    value decide(vector(value) values, logical require_unique_hit = false);
    value decide(vector(string) values_as_strs, logical require_unique_hit = false);

    // Get result the difficult way, but you can also find out how the result was reached
    integer decide_rule(vector(value) values, logical require_unique_hit = false);
    integer decide_rule(vector(string) values_as_strs, logical require_unique_hit = false);
    integer n_rules();
    decision_row get_rule(integer i);
    string get_rule_desc(integer i);

    // Optional, but adds a layer of security. Values in rules must be in
    // domain. Values tested must be in rule or domain.
    // You can run test_coverage to see that your decision_table is
    // well-formed.
    void set_domain(integer attrib_index, vector(value) values);
    void check_in_domain(integer attrib_index, value val);

    // Domains for all attribs must have been set.
    // Returns [error, result, attrib val 0, attrib val 1, ...].
    // Either error or result is null.
    vector(value_vec) test_coverage(logical require_unique_hit);


    matrix(string) show(); // Human friendly
    matrix(string) export(); // Computer friendly

private:
    data_type result_type_;
    vector(data_type) attrib_types_;
    vector(string) attrib_names_;
    integer n_attribs_;
    vector(decision_row) rows_;

    vector(value_vec) domains_;

    void _init(
        data_type result_type,
        vector(data_type) attrib_types,
        vector(string) option(nullable) attrib_names);
    string _attrib_pos_str(integer i);
};

decision_table.decision_table(data_type result_type, vector(data_type) attrib_types)
{
    _init(result_type, attrib_types, null<vector(string)>);
}

decision_table.decision_table(
    data_type result_type,
    vector(data_type) attrib_types,
    vector(string) attrib_names)
{
    _init(result_type, attrib_types, attrib_names);
}

decision_table.decision_table(matrix(string) table)
{
    n_attribs_ = n_cols(table)-1;

    resize(attrib_names_, n_attribs_);
    resize(attrib_types_, n_attribs_);

    for (integer j = 0; j < n_attribs_; ++j) {
        attrib_names_[j] = table[0,j];
        attrib_types_[j] = str_to_data_type(table[1,j]);
    }

    result_type_ = str_to_data_type(table[1,n_attribs_]);

    resize(rows_, 0);

    for (integer i = 2; i < n_rows(table); ++i) {
        vector(cell_rule) cell_rules[n_attribs_];
        for (integer j = 0; j < n_attribs_; ++j) {
            cell_rules[j] = new cell_rule(attrib_types_[j], table[i,j]);
        }
        value result = value_from_string(result_type_, table[i, n_attribs_]);
        add_rule(result, cell_rules);
    }
}

void decision_table._init(
    data_type result_type,
    vector(data_type) attrib_types,
    vector(string) option(nullable) attrib_names)
{
    result_type_ = result_type;
    attrib_types_ = clone_vector(attrib_types);
    n_attribs_ = v_size(attrib_types_);
    resize(rows_, 0);
    resize(domains_, n_attribs_);

    if (null(attrib_names)) {
        resize(attrib_names_, n_attribs_);
    } else {
        set_attribute_names(attrib_names);
    }
}


string decision_table._attrib_pos_str(integer i)
{
    if (null(attrib_names_[i])) {
        return strcat(["attribute at position ", str(i)]);
    } else {
        return strcat(["'", attrib_names_[i], "' (position ", str(i), ")"]);
    }
}

void decision_table.set_attribute_names(vector(string) attrib_names)
{
    if (v_size(attrib_names) != n_attribs_) {
        throw (E_UNSPECIFIC, "Wrong vector size");
    }
    attrib_names_ = clone_vector(attrib_names);
}

void decision_table.add_rule(value result, vector(cell_rule) cell_rules)
{
    if (result.type != result_type_) {
        throw (E_UNSPECIFIC, strcat(["Result must be of type ", string(result_type_)]));
    }

    if (v_size(cell_rules) != n_attribs_) {
        throw (E_UNSPECIFIC, strcat(["Rule must have ", str(n_attribs_), " cell rules, got ", str(v_size(cell_rules))]));
    }

    integer cell_rules_sz = v_size(cell_rules);
    for (integer i = 0; i < cell_rules_sz; ++i) {
        vector(value) rule_values = cell_rules[i].values();
        integer rule_values_sz = v_size(rule_values);
        for (integer j = 0; j < rule_values_sz; ++j) {
            this.check_in_domain(i, rule_values[j]);
        }
    }

    push_back(rows_, decision_row(result, cell_rules));
}

value decision_table.decide(vector(value) values, logical require_unique_hit)
{
    return this.get_rule(this.decide_rule(values, require_unique_hit)).result;
}

value decision_table.decide(vector(string) values_as_strs, logical require_unique_hit)
{
    return this.get_rule(decide_rule(values_as_strs, require_unique_hit)).result;
}

integer decision_table.decide_rule(vector(value) values, logical require_unique_hit)
{
    if (v_size(values) != n_attribs_) {
        throw (E_UNSPECIFIC, strcat(["Expected ", str(n_attribs_), " values, got ", str(v_size(values))]));
    }

    integer match_row_index = -1;
    vector(integer) hits[0];

    logical value_in_rule;

    integer rows__sz = v_size(rows_);
    for (integer r = 0; r < rows__sz; ++r) {
        decision_row row = rows_[r];
        logical matched = true;
        try {
            for (integer i = 0; i < n_attribs_; ++i) {
                value val = values[i];

                if (!(val.type == DT_NULL || val.type == attrib_types_[i])) {
                    throw (E_UNSPECIFIC, strcat(
                               ["Expected type ", string(attrib_types_[i]), " for ",
                                _attrib_pos_str(i), ", got ", string(val.type)]));
                }

                try {
                    matched = row.cell_rules[i].passes(values[i], value_in_rule);
                } catch {
                    throw (E_UNSPECIFIC, strcat(["Error for ", _attrib_pos_str(i), ": ", err.message()]));
                }

                if (!value_in_rule) {
                    check_in_domain(i, val);
                }

                if (!matched) {
                    break;
                }
            }
        } catch {
            throw (E_UNSPECIFIC, strcat([err.message(), " [Rule ", str(r), ": ", to_str(row), "]"]));
        }

        if (matched) {
            match_row_index = r;
            push_back(hits, r);
            if (!require_unique_hit) {
                break;
            }
        }
    }

    if (v_size(hits) == 1) {
        return match_row_index;
    } else if (v_size(hits) == 0) {
        throw (
            E_UNSPECIFIC,
            strcat(["None of the ", str(rows__sz), " rules matched ", str_join(values.repr(), "; ")]));
    } else {
        throw (
            E_UNSPECIFIC,
            strcat(["Rules ", str_join(str(hits), ","), " matched ", str_join(values.repr(), "; ")]));
    }
}

integer decision_table.decide_rule(vector(string) values_as_strs, logical require_unique_hit)
{
    vector(value) values[n_attribs_];
    if (v_size(values_as_strs) != n_attribs_) {
        throw (E_UNSPECIFIC, strcat(["Expected ", str(n_attribs_), " values"]));
    }

    for (integer i = 0; i < n_attribs_; ++i) {
        try {
            switch (attrib_types_[i]) {
            case DT_STR:
                values[i] = qv(values_as_strs[i]);
                break;
            case DT_NUM:
                values[i] = qv(str_to_number(values_as_strs[i]));
                break;
            case DT_DATE:
                values[i] = qv(str_to_date(values_as_strs[i]));
                break;
            case DT_TIMESTAMP:
                values[i] = qv(str_to_timestamp(values_as_strs[i]));
                break;
            case DT_LOGICAL:
                if (null(values_as_strs[i]))
                {
                    values[i] = qv(null<logical>);
                } else if (
                    values_as_strs[i] == "1" ||
                    equal_casei(values_as_strs[i], "true") ||
                    equal_casei(values_as_strs[i], "y") ||
                    equal_casei(values_as_strs[i], "yes"))
                {
                    values[i] = qv(true);
                }
                else if (
                    values_as_strs[i] == "0" ||
                    equal_casei(values_as_strs[i], "false") ||
                    equal_casei(values_as_strs[i], "n") ||
                    equal_casei(values_as_strs[i], "no"))
                {
                    values[i] = qv(false);
                }
                else
                {
                    throw (E_UNSPECIFIC, strcat(["Cannot convert '", values_as_strs[i], "' to logical"]));
                }
            }
        } catch {
            throw (E_UNSPECIFIC, strcat(["Error using value '", values_as_strs[i], "' (", str(i), "): ", err.message()]));
        }
    }

    return decide_rule(values, require_unique_hit);
}

decision_row decision_table.get_rule(integer index)
{
    if (!(0 <= index && index < v_size(rows_))) {
        throw (E_UNSPECIFIC, strcat(["Index ", str(index), " is not in the range of rules"]));
    }
    return rows_[index];
}

string decision_table.get_rule_desc(integer index)
{
    decision_row row = this.get_rule(index);
    vector(string) cell_rule_descs[0];
    for (integer i = 0; i < n_attribs_; ++i) {
        cell_rule cr = row.cell_rules[i];
        if (cr.op() == rule_operator.SKIP) {
            continue;
        }
        if (null(attrib_names_[i])) {
            push_back(cell_rule_descs, strcat(["Attrib ", str(i), " ", cr.repr_short()]));
        } else {
            push_back(cell_rule_descs, strcat([attrib_names_[i], " ", cr.repr_short()]));
        }
    }

    return strcat([str_join(cell_rule_descs, "; "), " -> ", row.result.repr("<null>")]);
}

integer decision_table.n_rules() = v_size(rows_);

void decision_table.set_domain(integer attrib_index, vector(value) values)
{
    if (attrib_index < 0 || attrib_index >= n_attribs_) {
        throw (E_UNSPECIFIC, "Invalid attrib_index");
    }
    domains_[attrib_index] = new value_vec;
    domains_[attrib_index].v = clone_vector(values);
}

void decision_table.check_in_domain(integer attrib_index, value val)
{
    if (!null(domains_[attrib_index])) {
        vector(value) domain_values = domains_[attrib_index].v;
        integer domain_values_sz = v_size(domain_values);
        for (integer i = 0; i < domain_values_sz; ++i) {
            if (equaln(val, domain_values[i])) {
                return;
            }
        }
        throw (E_UNSPECIFIC, strcat(
                   [val.repr("<null>"), " not in domain of attrib ", str(attrib_index),
                    " {", str_join(domain_values.repr("<null>"), ","), "}"]));
    }
}

vector(value_vec) decision_table.test_coverage(logical require_unique_hit)
{
    if (n_attribs_ <= 0) {
        vector(value_vec) v[0];
        return v;
    }
    integer n_combinations = 1;
    vector(integer) size[n_attribs_];
    vector(number) divisor[n_attribs_];
    for (integer i = 0; i < n_attribs_; ++i) {
        value_vec v = domains_[i];
        if (null(v)) {
            throw (E_UNSPECIFIC, "Domain not set for all attribs");
        }
        size[i] = v_size(v.v);
        n_combinations *= size[i];
        if (i == 0) {
            divisor[i] = 1;
        } else {
            divisor[i] = divisor[i-1] * size[i-1];
        }
    }

    vector(value) values[n_attribs_];

    vector(value_vec) results[n_combinations];

    for (integer i = 0; i < n_combinations; ++i) {
        for (integer a_ix = 0; a_ix < n_attribs_; ++a_ix) {
            integer index = integer(floor(i/divisor[a_ix])) % size[a_ix];
            values[a_ix] = domains_[a_ix].v[index];
        }
        try {
            value v = decide(values, require_unique_hit);
            results[i] = new value_vec;
            results[i].v = concat([null<qu.value>, v], values);
        } catch {
            value v = new value(strcat("Error: ", err.message()));
            results[i] = new value_vec;
            results[i].v = concat([v, null<qu.value>], values);
        }
    }

    return results;
}


matrix(string) decision_table.show()
{
    integer n = v_size(rows_) + 1;
    integer m = v_size(attrib_types_) + 1;
    matrix(string) ret[n, m];

    ret[0,0] = strcat(["Result (", string(result_type_), ")"]);

    for (integer j = 0; j < m-1; ++j) {
        ret[0, j+1] = strcat([attrib_names_[j], " (", string(attrib_types_[j]), ")"]);
    }

    for (integer i = 1; i < n; ++i) {
        decision_row row = rows_[i-1];
        ret[i, 0] = strcat(row.result.repr(), " <-");
        for (integer j = 1; j < m; ++j) {
            cell_rule rule = row.cell_rules[j-1];

            ret[i, j] = rule.repr_short();
        }
    }

    return ret;
}

matrix(string) decision_table.export()
{
    integer n = v_size(rows_) + 2;
    integer m = v_size(attrib_types_) + 1;
    matrix(string) ret[n, m];

    ret[0,m-1] = strcat(["Result"]);
    ret[1,m-1] = string(result_type_);

    for (integer j = 0; j < m-1; ++j) {
        ret[0, j] = attrib_names_[j];
        ret[1, j] = string(attrib_types_[j]);
    }

    for (integer i = 2; i < n; ++i) {
        decision_row row = rows_[i-2];
        for (integer j = 0; j < m-1; ++j) {
            ret[i, j] = row.cell_rules[j].export();
        }
        ret[i, m-1] = row.result.repr();
    }

    return ret;
}

}
