/*

 * Using TreeData
 *
 * The TreeData class helps you keep label and parent vectors, as well as any
 * extra columns, extra data, tooltips and colors you may want. It is very
 * suited to fill recursively and then pass to tree.init.
 *
tree.TreeData td = new tree.TreeData();
td.set_data_labels(["NodeType"]);

integer mammal_ix = td.add_root("Mammal");
td.set_data_val(mammal_ix, "NodeType", "Class");

integer tiger_ix = td.add_node("Tiger", mammal_ix);
td.set_data_val(tiger_ix, "NodeType", "Species");

tree.init("animals", td);





 * Using with xml_node
 *
 * Create an xml_node and pass it in with p_ix = -1, something like this:

xml_node node = parse_xml(file_read(filename));
xml_to_tree_vectors(node, -1, labels, parents);

 *
 * In normal mode, each tag gets one row. In deep mode, the attributes and text of
 * a tag get their own rows.
 *


void xml_to_tree_vectors(
	xml_node x, integer p_ix,
	out vector(string) labels,
	out vector(integer) parents,
	logical deep = false)
{
	integer my_ix = v_size(labels);

	string tag = x.tag();

	vector(string) attr_names;
	vector(string) attr_vals;
	x.get_attribs(attr_names, attr_vals);

	if (deep) {
		push_back(labels, strcat(["<", tag, ">"]));
		push_back(parents, p_ix);

		for (integer i = 0; i < v_size(attr_names); ++i) {
			push_back(labels, strcat([attr_names[i], "=\"", attr_vals[i], "\""]));
			push_back(parents, my_ix);
		}

		if (v_size(x.children()) == 1 && x.children()[0].is_text()) {
			push_back(labels, x.children()[0].text());
			push_back(parents, my_ix);
		} else {
			xml_to_tree_vectors(x.children(), my_ix, labels, parents, deep);
		}
	} else {
		push_back(labels, "placeholder");
		push_back(parents, p_ix);
		vector(string) label_elems = [strcat(["<", tag])];
		for (integer i = 0; i < v_size(attr_names); ++i) {
			push_back(label_elems, strcat([" ", attr_names[i], "=\"", attr_vals[i], "\""]));
		}
		push_back(label_elems, ">");
		labels[my_ix] = strcat(label_elems);

		if (v_size(x.children()) == 1 && x.children()[0].is_text()) {
			labels[my_ix] = strcat([labels[my_ix], x.children()[0].text(), "</", tag, ">"]);
		} else {
			xml_to_tree_vectors(x.children(), my_ix, labels, parents, deep);
		}
	}
}




 * Using with ahs_api.item_set / ahs_api.sec_set
 *
 * To use tree, convert the set to
 * vector(ahs_api.item_set) and vector(integer) parents
 * call the following function with parent = -1, and sets
 * and parents being empty vectors:
 *

void set_to_tree_data(
	ahs_api.item_set set,
	integer parent,
	out vector(ahs_api.item_set) sets,
	out vector(integer) parents)
{
	integer my_index = v_size(parents);
	push_back(sets, set);
	push_back(parents, parent);
	set_to_tree_data(set.get_children(), my_index, sets, parents);
}


*
* If you want securities too, use function below. The vectors sets and
* securties will both be of the same size as parents, and for every index
* only the approprate vector will contain data, and the other will have a
* null element.
*

void set_to_tree_data(
	ahs_api.sec_set set,
	integer parent,
	out vector(ahs_api.sec_set) sets,
	out vector(ahs_api.security) securities,
	out vector(integer) parents)
{
	integer my_index = v_size(parents);
	push_back(sets, set);
	push_back(securities, null<ahs_api.security>);
	push_back(parents, parent);
	if (v_size(set.get_children()) > 0) {
		set_to_tree_data(set.get_children(), my_index, sets, securities, parents);
	} else {
		vector(ahs_api.security) secs = set.get_securities();
		integer secs_sz = v_size(secs);
		for (integer i = 0; i < secs_sz; ++i) {
			push_back(sets, null<ahs_api.sec_set>);
			push_back(securities, secs[i]);
			push_back(parents, my_index);
		}
	}
}



*
* Using with mixed type data
*
* The simplest way is too keep one vector for every type, let all vectors
* be of the same size (the full size of parents/labels), and let them have
* null values wherever they aren't the source of the tree node. When you get an
* index from the tree, loop through your vectors until you find a non-null
* entry for that index - and there you have it.
*
* If don't want vectors with lots of null in them, e.g. you use them elsewhere,
* here is a general recipe for that:
*
* In addition to the vectors needed for tree, Labels and Parents, create a
* matching vector Type and another matching vector Index. Let Type contain a
* type identifier, for which you have a vector with all elements of that
* type. Let Index contain the index for that vector.
*
* When you get an index from tree, check the type in Type and find the correct
* object in corresponding vector.
*
* Let's say leaf nodes are of type Instrument and other nodes are of type
* Issuer or Currency.
*
* Labels:
* ["SEK", "CHF", "Volvo", "Volvo 304", "Volvo 307", "ABB", "ABB 4"  ]
* Parents:
* [-1,    -1,    0,       2,           2,           1,     5        ]
* Type:
* ["ccy", "ccy", "issuer", "instr",    "instr",     "issuer", "inst"]
* Index:
* [0,     1,     0,       0,           1,           1,     2        ]
* Currency:
* ["SEK", "CHF"]
* Issuer:
* [<Volvo>, <ABB>]
* Instr:
* [<Volvo 304>, <Volvo 307>, <ABB 4>]
*
*/




/*
 * == Recipes ==
 *
 * - Hide subtrees without leaves -
 *
 * 1) Search for empty string among leaves only
 * 2) Filter to result of search
 *
 *
 * - Deselect everything -
 *
 * 1) tree.deselect("treename", tree.get_selected("treename"))
 *
 */


module tree {

	module __impl__ {
		string LOCAL_SOURCE = "LOCAL";

		string spaces(integer n)
		{
			vector(string) v[n];
			v = " ";
			return strcat(v);
		}

		class text_rgb_v {
		public:
			text_rgb_v(vector(text_rgb) v);
			vector(text_rgb) get();

		private:
			vector(text_rgb) v_;
		};
		text_rgb_v.text_rgb_v(vector(text_rgb) v) : v_(v) {}
		vector(text_rgb) text_rgb_v.get() = v_;


		class string_v {
		public:
			string_v(vector(string) v);
			vector(string) get();

		private:
			vector(string) v_;
		};
		string_v.string_v(vector(string) v) : v_(v) {}
		vector(string) string_v.get() = v_;


		logical equaln(string option(nullable) s1, string option(nullable) s2)
		{
			if (null(s1) && null(s2))
				return true;
			if (null(s1) || null(s2))
				return false;
			return s1 == s2;
		}

		logical find_stringn(string option(nullable) haystack, string option(nullable) needle)
		{
			if (null(haystack) && null(needle))
				return true;
			if (null(haystack) || null(needle))
				return false;
			return find_string(haystack, needle);
		}


		enum leafiness {
			L_BRANCH option(str: "branch"),
			L_LEAF option(str: "leaf"),
			L_SUBLEAF option(str: "subleaf")
		};

		class Node {
		public:
			Node(
				string option(nullable) label,
				integer parent,
				integer depth,
				leafiness leaf,
				integer original_index
				);
			string label();
			void set_label(string option(nullable) label);
			integer parent();
			integer depth();
			leafiness get_leafiness();
			logical is_branch();
			logical is_leaf();
			logical is_sub_leaf();
			integer original_index();

			void set_leafiness(leafiness leaf);

			logical selected();
			void set_selected(logical);
			logical collapsed();
			void set_collapsed(logical);
			logical shown();
			void set_shown(logical shown);

			// Set default color to the color the cell should normally have,
			// then use set_color and unset_color to manage temporary color
			// changes (e.g. to highlight search results).
			integer color();
			void set_default_color(integer rgb);
			void set_color(integer rgb);
			void unset_color();

			void set_tooltip(string option(nullable) tooltip);
			string tooltip();

			logical ignore_select_click();
			void set_ignore_select_click(logical ignore);

			string count_type();
			void set_count_type(string option(nullable) count_type);

		private:
			string label_;
			integer parent_;
			integer depth_;
			leafiness leaf_;
			integer original_index_;
			logical selected_;
			logical collapsed_;
			logical shown_;
			integer default_color_;
			integer color_;
			string tooltip_;
			logical ignore_select_click_;
			string count_type_;
		};

		Node.Node(
			string option(nullable) label,
			integer parent,
			integer depth,
			leafiness leaf,
			integer original_index)
			: label_(null(label) ? "" : label)
			, parent_(parent)
			, depth_(depth)
			, leaf_(leaf)
			, original_index_(original_index)
			, selected_(false)
			, collapsed_(true)
			, shown_(true)
			, default_color_(-1)
			, color_(-1)
			, ignore_select_click_(false)
			, count_type_(null<string>)
		{}

		string Node.label() = label_;
		void Node.set_label(string option(nullable) label) { label_ = label; }
		integer Node.parent() = parent_;
		integer Node.depth() = depth_;
		leafiness Node.get_leafiness() = leaf_;
		logical Node.is_branch() = leaf_ == L_BRANCH;
		logical Node.is_leaf() = leaf_ == L_LEAF;
		logical Node.is_sub_leaf() = leaf_ == L_SUBLEAF;
		integer Node.original_index() = original_index_;

		void Node.set_leafiness(leafiness leaf) { leaf_ = leaf; }

		logical Node.selected() = selected_;
		void Node.set_selected(logical selected) { selected_ = selected; }

		logical Node.collapsed() = collapsed_;
		void Node.set_collapsed(logical collapsed) { collapsed_ = collapsed; }

		logical Node.shown() = shown_;
		void Node.set_shown(logical shown) { shown_ = shown; }

		integer Node.color() = color_ == -1 ? default_color_ : color_;
		void Node.set_default_color(integer color) { default_color_ = color; }
		void Node.set_color(integer color) { color_ = color; }
		void Node.unset_color() { color_ = -1; }

		void Node.set_tooltip(string option(nullable) tooltip) { tooltip_ = tooltip; }
		string Node.tooltip() = tooltip_;

		logical Node.ignore_select_click() = ignore_select_click_;
		void Node.set_ignore_select_click(logical ignore) { ignore_select_click_ = ignore; }

		string Node.count_type() = count_type_;
		void Node.set_count_type(string option(nullable) count_type) { count_type_ = count_type; }

		string tree_string(Node node, string branch_string)
		{
			if (node.parent() == -1) {
				return "";
			} else {
				return strcat(spaces((node.depth()-1)*4), branch_string);
			}
		}




		class intvec {
		public:
			intvec();
			vector(integer) get();
			void push_back(integer i);

		private:
			vector(integer) v_;
		};

		intvec.intvec()
		{
			resize(v_, 0);
		}

		vector(integer) intvec.get() = v_;

		void intvec.push_back(integer i) { ..push_back(v_, i); }

		vector(intvec) list_children(vector(integer) parents)
		{
			integer n = v_size(parents);
			vector(intvec) child_vectors[n+1]; // last element is list of root nodes


			for (integer i = 0; i < n+1; ++i) {
				child_vectors[i] = new intvec();
			}

			for (integer i = 0; i < n; ++i) {
				integer p_ix = parents[i];
				if (p_ix < -1 || p_ix >= n) {
					throw (E_UNSPECIFIC, strcat("Bad parent vector: Parent index out of range at element ", str(i)));
				} else if (p_ix == -1) {
					child_vectors[n].push_back(i);
				} else {
					child_vectors[p_ix].push_back(i);
				}
			}

			return child_vectors;
		}

		integer fill_nodes(
			vector(Node) nodes,
			integer root_index,
			vector(string) labels,
			vector(integer) node_pos,
			vector(intvec) child_vectors,
			integer parent_pos,
			integer pos,
			integer depth)
		{
			vector(integer) children = child_vectors[root_index].get();
			integer children_sz = v_size(children);
			Node node = new Node(labels[root_index], parent_pos, depth, children_sz == 0 ? L_LEAF : L_BRANCH, root_index);
			nodes[pos] = node;

			node_pos[root_index] = pos;
			integer next_pos = pos+1;

			for (integer i = 0; i < children_sz; ++i) {
				integer child_index = children[i];
				next_pos = fill_nodes(nodes, child_index, labels, node_pos, child_vectors, pos, next_pos, depth+1);
			}

			return next_pos;
		}


		void make_ordered_nodes(
			vector(string) labels,
			vector(integer) parents,
			out vector(Node) nodes,
			out vector(integer) orig_2_node_index)
		{
			vector(intvec) child_vectors = list_children(parents);

			integer n = v_size(labels);
			resize(nodes, n);

			resize(orig_2_node_index, n);
			orig_2_node_index = -1; // we will check that all are > 0 after nodes are created

			vector(integer) roots = child_vectors[n].get();
			integer roots_sz = v_size(roots);
			integer next_pos = 0;
			for (integer i = 0; i < roots_sz; ++i) {
				integer root_index = roots[i];
				next_pos = fill_nodes(
					nodes, root_index, labels, orig_2_node_index, child_vectors, -1, next_pos, 0);
			}

			vector(integer) missed_elements[0];
			for (integer i = 0; i < n; ++i) {
				if (orig_2_node_index[i] < 0) {
					push_back(missed_elements, i);
				}
			}

			if (v_size(missed_elements) > 0) {
				if (v_size(missed_elements) == 1) {
					throw (E_UNSPECIFIC, strcat(
						["Bad parent vector: Element ", str(missed_elements[0]),
						 " was not reachable from any root"]));
				} else if (v_size(missed_elements) == 2) {
					throw (E_UNSPECIFIC, strcat(
						["Bad parent vector: Elements ", str(missed_elements[0]),
						 " and ", str(missed_elements[1]), " were not reachable from any root"]));
				} else {
					throw (E_UNSPECIFIC, strcat(
						["Bad parent vector: Element ", str(missed_elements[0]), " and ",
						 str(v_size(missed_elements)-1), " others were not reachable from any root"]));
				}
			}
		}
	}

	class count_set {
	public:
		count_set();
		vector(string) get_keys();
		integer count(string key);
		void increase(string key, integer amount = 1);

	private:
		map_str_num counts_;
	};

	count_set.count_set() : counts_(map_str_num()) {}
	vector(string) count_set.get_keys() = counts_.get_keys();

	integer count_set.count(string key)
	{
		number c = counts_.find(key);
		return null(c) ? 0 : integer(c);
	}

	void count_set.increase(string key, integer amount)
	{
		counts_.add(key, this.count(key) + amount);
	}

	void add_counts(count_set to, count_set from)
	{
		vector(string) keys = from.get_keys();
		integer keys_sz = v_size(keys);
		for (integer i = 0; i < keys_sz; ++i) {
			to.increase(keys[i], from.count(keys[i]));
		}
	}

	string show_count_type_counts(count_set selected, count_set leaves, logical all_selected)
	{
		vector(string) keys = leaves.get_keys();
		vector(integer) sel_counts = selected.count(keys);
		vector(integer) leaf_counts = leaves.count(keys);
		integer keys_sz = v_size(keys);
		vector(string) elems[0];
		for (integer i = 0; i < keys_sz; ++i) {
			if (all_selected) {
				push_back(elems, [", ", keys[i], ": ", str(sel_counts[i])]);
			} else if (sel_counts[i] == leaf_counts[i]) {
				push_back(elems, [", ", keys[i], ": All ", str(sel_counts[i])]);
			} else {
				push_back(elems, [", ", keys[i], ": ", str(sel_counts[i]), "/", str(leaf_counts[i])]);
			}
		}
		return strcat(elems);
	}

	enum SelectionMode option(category: "Workspace/Tree") {
		SELECT_NONE option(str: "None")
		, SELECT_ANY option(str: "Any")
		, SELECT_LEAVES option(str: "Leaves")
	};

	enum FilterSelectMethod option(category: "Workspace/Tree/Filter and Select") {
		FS_ADD option(str: "Add")
		, FS_REMOVE option(str: "Remove")
		, FS_SET_TO_SHOWN option(str: "These")
		, FS_SET_TO_HIDDEN option(str: "Others")
	};

	vector(FilterSelectMethod) filter_select_methods() =
		[FS_ADD, FS_REMOVE, FS_SET_TO_SHOWN, FS_SET_TO_HIDDEN];

	FilterSelectMethod str_to_FilterSelectMethod(string s)
	{
		vector(FilterSelectMethod) methods = filter_select_methods();
		integer methods_sz = v_size(methods);
		for (integer i = 0; i < methods_sz; ++i) {
			if (s == string(methods[i])) {
				return methods[i];
			}
		}
		throw (E_UNSPECIFIC, strcat("No such FilterSelectMethod: ", s));
	}

	enum SearchOption option (category: "Workspace/Tree/Filter and Select") {
		SO_CASEI option(str: "case-insensitive")
		, SO_COMPLETE_MATCH option(str: "complete-match")
		, SO_ONLY_LEAVES option(str: "only-leaves")
	};


	module display_element {
		integer BRANCH = 0;
		integer COLLAPSED = 1;
		integer EXPANDED = 2;
		integer LEAF = 3;
		integer NOTSELECTED = 4;
		integer SELECTED = 5;
		integer NUM_ELEMS = 6;

		vector(string) defaults =
			["  ",
			 " [+] ",
			 " [-] ",
			 " ",
			 "[ ] ",
			 "[x] "];

		vector(string) alternatives =
			[str_build([9492, 32]),
			 str_build([9656, 32]),
			 str_build([9662, 32]),
			 str_build([9642, 32]),
			 str_build([9744]),
			 str_build([9746])];
	}

	string COL_LABEL option(constant) option(category: "Workspace/Tree/Interact") = "_Wsp/_Tree::col/Label";
	string COL_SELECT option(constant) option(category: "Workspace/Tree/Interact") = "_Wsp/_Tree::col/Select";
	string COL_DEBUG option(constant) option(category: "Workspace/Tree/Interact") = "_Wsp/_Tree::col/Debug";

	class Tree option(category: "Workspace/Tree/Misc") {
	public:
		Tree(string id, vector(string) labels, vector(integer) parents);
		Tree(string id, vector(string) labels, vector(integer) parents, SelectionMode mode);

		// Basic Interaction
		matrix(text_rgb) display();
		integer click(number row, number col, logical option(nullable) is_single_click);
		vector(integer) get_selected();
		string last_click_column();

		// Header
		void set_header(logical header);
		void set_header_color(integer rgb);

		// Change your mind about the labels used
		void set_label(integer index, string option(nullable) label);

		// Control symbols used in tree
		void set_display_string(integer elem, string s);

		// Extra columns
		void add_column(string id, vector(text_rgb) content);
		void remove_column(string id);
		void hide_column(string id);
		void show_column(string id);
		vector(string) get_shown_columns();
		void set_column_header_color(string id, integer rgb);
		void unset_column_header_color(string id);
		void set_column_val(string id, integer index, text_rgb option(nullable) val);
		text_rgb get_column_val(string id, integer index);

		// Colors based on row context
		void set_node_color(integer rgb);
		void set_leaf_color(integer rgb);
		void set_sub_leaf_color(integer rgb);
		void set_selected_color(integer rgb);
		void set_has_selected_color(integer rgb);
		void set_clicked_color(integer rgb);

		// Individual colors for nodes
		void set_default_colors(vector(integer) rgbs);
		void set_color(integer index, integer rgb);
		void unset_color(integer index);

		// Tooltips
		void set_tooltips(vector(string) tooltips);
		void set_tooltip(integer index, string option(nullable) tooltip);

		// Expanding/Collapsing
		void set_double_click_expansion(logical dbl_clk);
		void expand_to_nodes(
			vector(integer) indices, logical collapse_others, logical expand_given_nodes);
		void set_expansion_depth(integer depth);
		// Returns `depth` from last call to set_expansion_depth. Is not affected
		// by other means of expanding/collapsing the tree. Returns null if
		// set_expansion_depth has not been called at all.
		number get_set_expansion_depth();
		integer get_max_depth();
		void set_auto_expand(logical auto_expand);
		void set_cascade_collapse(logical cascade);

		// Special selection behaviour
		void set_ignore_select_click(integer index, logical ignore);
		void set_single_select(logical single_select);

		// Free form counting of different node types
		void set_count_types(vector(integer) index, vector(string) count_types);

		// Filter tree
		// `shown` is list of indices refering to original init vectors.
		// Nodes matching are shown, as well as nodes that are above listed noden
		// in the tree. Others are completely hidden.
		void set_filter(vector(integer) shown);
		void apply_or_filter(vector(integer) shown);
		void apply_and_filter(vector(integer) shown);
		void clear_filter();
		vector(integer) get_filter_shown();

		// Tree structure information
		integer num_nodes();
		vector(integer) get_children(integer index);
		integer get_parent(integer index);
		// Returns original index for node at this row. Returns -1 for header row and other
		// rows that aren't a row for a node in the tree.
		integer index_of_row(integer row);
		vector(integer) index_of_row(vector(integer) rows);
		integer num_rows(); // Number of rows shown, including header if there is one

		// Programmable selection (as in "not via clicking")
		void select(integer index, logical only_shown);
		void deselect(integer index, logical only_shown);
		void select_by_filter(FilterSelectMethod method);

		// Search
		vector(integer) search(
			vector(string) keys,
			vector(SearchOption) options,
			logical check_label,
			vector(string) option(nullable) col_ids, // null for all columns
			vector(string) option(nullable) data_ids); // null for all data_ids


		/* The tree class is very liberal about what a leaf is. The only rule
		 * is that no node in the subtree rooted at a leaf can be a leaf. So a
		 * node can have zero children and still not be a leaf. And a node can
		 * have children, and still be a leaf.
		 *
		 * The leaf-status is used for selection mode SELECT_LEAVES, for
		 * coloring and counting purposes etc, so it is more of a logical
		 * concept than a tree-technical.
		 *
		 * If a leaf has children, we say that the subtree rooted at the leaf
		 * is a leaf-tree, and the nodes below the leaf are sub-leaf nodes.
		 */
		void set_leaf_status(vector(logical) is_leaf);
		void set_leaves(vector(integer) indices);


		// Keep info that is not shown
		void add_data(string id, vector(string) content);
		string get_data(string id, integer index);
		void set_data(string id, integer index, string option(nullable) data);
		void set_data_recursively(string id, integer index, string option(nullable) data);

		// Debug stuff
		void set_debug_mode(logical val);
		vector(integer) _orig_to_index_vec();

		// Get current settings of tree
		map_str_str get_settings();


	private:
		string id_;
		vector(__impl__.Node) nodes_;
		vector(integer) orig_2_node_index_;
		vector(integer) shown_nodes_;

		// If more settings are added, add them in get_settings() too!
		SelectionMode selection_mode_;
		logical dbl_clk_expansion_;
		logical debug_mode_;
		logical show_header_;
		vector(string) display_elements_;
		logical auto_expand_;
		logical cascade_collapse_;
		logical single_select_;
		number set_expansion_depth_;
		// If more settings are added, add them in get_settings() too!

		vector(__impl__.text_rgb_v) extra_cols_;
		vector(string) extra_col_ids_;
		vector(integer) extra_col_header_colors_;
		vector(integer) extra_cols_shown_;

		vector(__impl__.string_v) extra_data_;
		vector(string) extra_data_ids_;

		integer last_click_node_;
		string last_click_column_;

		integer header_color_;
		integer node_color_;
		integer leaf_color_;
		integer sub_leaf_color_;
		integer selected_color_;
		integer has_selected_color_;
		integer clicked_color_;

		// Calculated display variables
		integer num_cols_; // Total number of matrix columns
		integer label_col_; // Matrix column with node labels
		integer sel_col_; // Matrix column with selection cells
		integer debug_col_; // Matrix column with debug info if debug mode is true
		integer extra_col_; // First matrix column with tree extra column in it

		// cache members
		logical dirty_selection_;
		logical dirty_collapse_;
		vector(integer) cache_num_sub_leaves_;
		vector(count_set) cache_num_sub_leaves_by_count_type_;
		vector(integer) cache_num_sub_selected_;
		vector(count_set) cache_num_sub_selected_by_count_type_;
		vector(integer) cache_selected_;

		void _init(string id, vector(string) labels, vector(integer) parents, SelectionMode mode);
		void _set_collapse_dirty();
		void _set_selection_dirty();
		void _refresh();
		void _set_filter(vector(integer) indices, logical apply_or);
		void _recalc_columns();
		integer _extra_col_index(string id);
		vector(integer) _leaves(integer root_index);
		vector(integer) _sub_nodes(integer root_index);
		void _switch_collapse(integer row);
		void _set_selected(integer node_index, logical new_status, logical only_if_shown);
		void _set_select_by_filter(logical on_filter_status, logical new_status);
		vector(integer) _shown_nodes();
	};

	Tree.Tree(string id, vector(string) labels, vector(integer) parents)
	{
		_init(id, labels, parents, SELECT_NONE);
	}

	Tree.Tree(string id, vector(string) labels, vector(integer) parents, SelectionMode mode)
	{
		_init(id, labels, parents, mode);
	}

	void Tree._init(string id, vector(string) labels, vector(integer) parents, SelectionMode mode)
	{
		id_ = id;
		selection_mode_ = mode;
		dbl_clk_expansion_ = false;
		show_header_ = false;
		dirty_selection_ = true;
		dirty_collapse_ = true;
		debug_mode_ = false;
		resize(extra_cols_, 0);
		resize(extra_col_ids_, 0);
		resize(extra_cols_shown_, 0);
		resize(extra_data_, 0);
		resize(extra_data_ids_, 0);
		display_elements_ = clone_vector(display_element.defaults);

		auto_expand_ = true;
		cascade_collapse_ = false;
		single_select_ = false;
		last_click_node_ = -1;
		header_color_ = rgb(200,200,200);
		node_color_ = -1;
		leaf_color_ = -1;
		sub_leaf_color_ = -1;
		selected_color_ = -1;
		has_selected_color_ = -1;
		clicked_color_ = -1;

		integer n = v_size(labels);
		if (v_size(parents) != n) {
			throw (E_UNSPECIFIC, "Vector size mismatch");
		}

		__impl__.make_ordered_nodes(labels, parents, nodes_, orig_2_node_index_);

		if (v_size(nodes_) != n) {
			throw (E_UNSPECIFIC, "Bad parent vector, wrong number of nodes created");
		}

		_recalc_columns();
	}

	matrix(text_rgb) Tree.display()
	{
		_refresh();

		vector(integer) shown_nodes = _shown_nodes();
		vector(__impl__.Node) nodes_to_show = nodes_[shown_nodes];
		integer n = v_size(nodes_to_show);

		integer num_header_rows = show_header_ ? 1 : 0;
		integer num_rows = n + num_header_rows;
		matrix(text_rgb) M[num_rows, num_cols_];

		integer num_extra_cols = v_size(extra_cols_shown_);


		if (show_header_) {
			M[0, label_col_] = text_rgb("Name", null<number>, header_color_);

			if (sel_col_ >= 0) {
				M[0, sel_col_] = text_rgb(display_elements_[display_element.NOTSELECTED], null<number>, header_color_);
			}

			for (integer i = 0; i < num_extra_cols; ++i) {
				integer extra_col_ix = extra_cols_shown_[i];
				integer bg = extra_col_header_colors_[extra_col_ix] == -1 ? header_color_ : extra_col_header_colors_[extra_col_ix];
				M[0, extra_col_+i] = text_rgb(extra_col_ids_[extra_col_ix], -1, bg, -1, extra_col_ids_[extra_col_ix]);
			}

			if (debug_col_ >= 0) {
				M[0, debug_col_] = text_rgb("Debug info", -1, rgb(220,220,220));
			}
		}

		for (integer i = 0; i < n; ++i) {
			integer row = i + num_header_rows;
			__impl__.Node node = nodes_to_show[i];
			integer real_index = shown_nodes[i];
			__impl__.leafiness leaf = nodes_[real_index].get_leafiness();
			integer num_sel = cache_num_sub_selected_[real_index];
			integer num_leaves = cache_num_sub_leaves_[real_index];


			/* Set correct color (in order of rule prio low->high) */
			integer color = node_color_;

			// Color due to leafiness
			if (leaf_color_ != -1 && node.is_leaf()) {
				color = leaf_color_;
			}
			if (sub_leaf_color_ != -1 && node.is_sub_leaf()) {
				color = sub_leaf_color_;
			}

			// Default color of node
			if (node.color() != -1) {
				color = node.color();
			}

			// Color due to selection
			if (selected_color_ != -1 &&
				node.selected() &&
				(selection_mode_ == SELECT_ANY || (selection_mode_ == SELECT_LEAVES && node.is_leaf()))) {

				color = selected_color_;
			}
			if (has_selected_color_ != -1 &&
				selection_mode_ == SELECT_LEAVES &&
				leaf == __impl__.L_BRANCH &&
				num_sel > 0) {

				color = has_selected_color_;
			}

			// Color due to clicking
			if (clicked_color_ != -1 && real_index == last_click_node_) {
				color = clicked_color_;
			}


			/* Set text in label cell */
			M[row, label_col_] = text_rgb(
				strcat(
					[__impl__.tree_string(node, display_elements_[display_element.BRANCH]),
					 node.is_leaf() ? display_elements_[display_element.LEAF] :
					 node.collapsed() ? display_elements_[display_element.COLLAPSED] : display_elements_[display_element.EXPANDED],
					 node.label()]),
				-1,
				color,
				-1,
				node.tooltip());


			/* Set text in selection cell */
			if (selection_mode_ == SELECT_ANY) {
				M[row, sel_col_] = text_rgb(
					node.selected() ?
						display_elements_[display_element.SELECTED] :
						display_elements_[display_element.NOTSELECTED],
					-1, color);
			} else if (selection_mode_ == SELECT_LEAVES) {
				if (node.is_leaf()) {
					M[row, sel_col_] = text_rgb(
						node.selected() ?
							display_elements_[display_element.SELECTED] :
							display_elements_[display_element.NOTSELECTED],
						-1, color);
				} else {
					vector(string) v;
					if (leaf == __impl__.L_SUBLEAF) {
						v = [""];
					} else if (num_leaves == 0) {
						v = ["( 0 )"];
					} else {
						if (num_sel == num_leaves) {
							v = ["( All ", str(num_leaves)];
						} else {
							v = ["(", str(num_sel), "/", str(num_leaves)];
						}
						string ct_str = show_count_type_counts(
							cache_num_sub_selected_by_count_type_[real_index],
							cache_num_sub_leaves_by_count_type_[real_index],
							num_sel == num_leaves);
						push_back(v, ct_str);
						push_back(v, num_sel == num_leaves ? " )" : ")");
					}
					M[row, sel_col_] = text_rgb(strcat(v), -1, color);
				}
			}

			/* Fill cells for extra columns */
			for (integer c = 0; c < num_extra_cols; ++c) {
				text_rgb t = extra_cols_[extra_cols_shown_[c]].get()[real_index];
				if (t.bg() == -1) {
					t = text_rgb(t.text(), t.fg(), color, false, t.tooltip());
				}
				M[row, extra_col_+c] = t;
			}

			if (debug_col_ >= 0) {
				M[row, debug_col_] = text_rgb(strcat(
					["Orig:", str(node.original_index()),
					 " SubSel:", str(cache_num_sub_selected_[real_index]),
					 "/", str(cache_num_sub_leaves_[real_index]),
					 " P:", str(node.parent())]));
			}
		}

		return M;
	}

	integer Tree.click(number row, number col, logical option(nullable) is_single_click)
	{
		/* Note about single/double clicks.
		 *
		 * In the beginning, there were only double clicks (single clicks were
		 * ignored by the UI and would never reach any QLang code).
		 *
		 * Then, single clicks were let through to the QLang code, and thus,
		 * the tree module. This allows for better behaviour of the tree. The
		 * tree will use single clicks by default to expand/collapse nodes and
		 * select/deselect.
		 *
		 * __on_click functions in newer UI that want to use single clicks must
		 * add the new parameter is_single_click. They shall pass this
		 * parameter to the tree module so tree can do the right thing. If tree
		 * does not get this parameter, tree assumes that single clicks are not
		 * used (it is and older UI version, or the parameter has not been
		 * added to the __on_click function.
		 *
		 * In this function, is_single_click is null when it was not passed to
		 * tree, and true/false if it was. You will see in the code how this
		 * affects behaviour.
		 *
		 * Variables affecting this function:
		 *   dbl_clk_expansion_: If true, expansion/collapsing is done with
		 *   double click, not with single click.
		 */

		integer shown_row = integer(row-1);
		if (show_header_) {
			shown_row--;
		}
		integer real_col = integer(col-1);

		// Check that the actual tree was clicked
		if (shown_row < 0 || real_col >= num_cols_) {
			last_click_column_ = null<string>;
			last_click_node_ = -1;
			return -1;
		}

		if ((null(is_single_click) || is_single_click != dbl_clk_expansion_) && real_col == label_col_) {
			_switch_collapse(shown_row);
		} else if ((null(is_single_click) || is_single_click) && real_col == sel_col_) {
			integer node_index = _shown_nodes()[shown_row];

			logical ignore_click =
				nodes_[node_index].ignore_select_click() ||
				(single_select_ &&
				 selection_mode_ == SELECT_LEAVES &&
				 !nodes_[node_index].is_leaf());

			if (!ignore_click) {
				if (single_select_) {
					// Clear selection elsewhere
					logical clicked_node_is_selected = nodes_[node_index].selected();
					nodes_.set_selected(false);
					if (clicked_node_is_selected) {
						nodes_[node_index].set_selected(true);
					}
					_set_selection_dirty();
				}
				_set_selected(node_index, !nodes_[node_index].selected(), true);
			}
		}

		last_click_column_ = null<string>;
		if (real_col == label_col_) {
			last_click_column_ = COL_LABEL;
		} else if (real_col == sel_col_) {
			last_click_column_ = COL_SELECT;
		} else {
			integer extra_col_index = real_col - extra_col_;
			if (extra_col_index >= 0 && extra_col_index < v_size(extra_cols_shown_)) {
				last_click_column_ = extra_col_ids_[extra_cols_shown_[extra_col_index]];
			}
		}
		last_click_node_ = _shown_nodes()[shown_row];
		return nodes_[_shown_nodes()[shown_row]].original_index();
	}

	vector(integer) Tree.get_selected()
	{
		if (selection_mode_ == SELECT_NONE) {
			throw (E_UNSPECIFIC, "Selection mode is SELECT_NONE");
		}

		_refresh();

		return nodes_[cache_selected_].original_index();
	}

	string Tree.last_click_column() = last_click_column_;

	/*
	 * Header
	 */
	void Tree.set_header(logical header) { show_header_ = header; }
	void Tree.set_header_color(integer rgb) { header_color_ = rgb; }

	/*
	 * Change your mind about the labels used
	 */
	void Tree.set_label(integer index, string option(nullable) label)
	{
		nodes_[orig_2_node_index_[index]].set_label(label);
	}

	/*
	 * Control symbols used in tree
	 */
	void Tree.set_display_string(integer elem, string s)
	{
		if (elem >= display_element.NUM_ELEMS) {
			throw (E_UNSPECIFIC, "elem integer out of range");
		}
		display_elements_[elem] = s;
	}

	/*
	 * Extra columns
	 */
	void Tree.add_column(string id, vector(text_rgb) content)
	{
		integer n = v_size(nodes_);
		if (v_size(content) != n) {
			throw (E_UNSPECIFIC, "Argument content has wrong size");
		}

		vector(text_rgb) node_sorted_vec[n];
		for (integer i = 0; i < n; ++i) {
			text_rgb elem = content[nodes_[i].original_index()];
			if (null(elem)) {
				elem = text_rgb("");
			}
			node_sorted_vec[i] = elem;
		}

		push_back(extra_cols_, new __impl__.text_rgb_v(node_sorted_vec));
		push_back(extra_col_ids_, id);
		push_back(extra_col_header_colors_, -1);
		push_back(extra_cols_shown_, v_size(extra_cols_)-1);
		num_cols_++;
	}

	void Tree.remove_column(string id)
	{
		vector(string) new_extra_col_ids[0];
		vector(__impl__.text_rgb_v) new_extra_cols[0];
		vector(integer) new_extra_cols_shown[0];

		set_str shown = set_str();
		shown.add(extra_col_ids_[extra_cols_shown_]);

		for (integer i = 0; i < v_size(extra_col_ids_); ++i) {
			string col_id = extra_col_ids_[i];
			if (col_id != id) {
				push_back(new_extra_col_ids, col_id);
				push_back(new_extra_cols, extra_cols_[i]);
				if (shown.includes(col_id))
					push_back(new_extra_cols_shown, v_size(new_extra_col_ids)-1);
			} else {
				if (shown.includes(id))
					num_cols_--;
			}
		}

		extra_col_ids_ = new_extra_col_ids;
		extra_cols_ = new_extra_cols;
		extra_cols_shown_ = new_extra_cols_shown;
	}

	void Tree.hide_column(string id)
	{
		/* Sorts columns! Must be fixed if user is allowed to
		 * set column order.
		 */

		integer col_index = _extra_col_index(id);

		set_int shown = set_int();
		shown.add(extra_cols_shown_);

		shown.remove(col_index);

		extra_cols_shown_ = shown.get_content();

		_recalc_columns();
	}

	void Tree.show_column(string id)
	{
		/* Sorts columns! Must be fixed if user is allowed to
		 * set column order.
		 */

		integer col_index = _extra_col_index(id);

		set_int shown = set_int();
		shown.add(extra_cols_shown_);

		shown.add(col_index);

		extra_cols_shown_ = shown.get_content();

		_recalc_columns();
	}

	vector(string) Tree.get_shown_columns()
	{
		vector(string) ret[num_cols_];

		ret[label_col_] = COL_LABEL;

		if (sel_col_ >= 0)
			ret[sel_col_] = COL_SELECT;

		for (integer i = 0; i < v_size(extra_cols_shown_); ++i) {
			ret[extra_col_ + i] = extra_col_ids_[extra_cols_shown_[i]];
		}

		if (debug_col_ >= 0) {
			ret[debug_col_] = COL_DEBUG;
		}

		// Internal sanity check
		for (integer i = 0; i < v_size(ret); ++i) {
			if (null(ret[i]))
				throw (E_UNSPECIFIC, "Internal: Null in get_shown_columns result");
		}

		return ret;
	}

	void Tree.set_column_header_color(string id, integer rgb)
	{
		extra_col_header_colors_[_extra_col_index(id)] = rgb;
	}
	void Tree.unset_column_header_color(string id)
	{
		extra_col_header_colors_[_extra_col_index(id)] = -1;
	}

	void Tree.set_column_val(string id, integer index, text_rgb option(nullable) val)
	{
		integer node_index = orig_2_node_index_[index];
		integer col_index = _extra_col_index(id);

		if (null(val)) {
			val = text_rgb("");
		}

		extra_cols_[col_index].get()[node_index] = val;
	}

	text_rgb Tree.get_column_val(string id, integer index)
	{
		integer node_index = orig_2_node_index_[index];
		integer col_index = _extra_col_index(id);

		return extra_cols_[col_index].get()[node_index];
	}


	/*
	 * Colors based on row context
	 */
	void Tree.set_node_color(integer rgb) { node_color_ = rgb; }
	void Tree.set_leaf_color(integer rgb) { leaf_color_ = rgb; }
	void Tree.set_sub_leaf_color(integer rgb) { sub_leaf_color_ = rgb; }
	void Tree.set_selected_color(integer rgb) { selected_color_ = rgb; }
	void Tree.set_has_selected_color(integer rgb) { has_selected_color_ = rgb; }
	void Tree.set_clicked_color(integer rgb) { clicked_color_ = rgb; }

	/*
	 * Individual colors for nodes
	 */
	void Tree.set_default_colors(vector(integer) rgbs)
	{
		integer nodes__sz = v_size(nodes_);
		if (v_size(rgbs) != nodes__sz) {
			throw (E_UNSPECIFIC, "Argument rgbs has wrong size");
		}
		for (integer i = 0; i < nodes__sz; ++i) {
			__impl__.Node node = nodes_[i];
			node.set_default_color(rgbs[node.original_index()]);
		}
	}
	void Tree.set_color(integer index, integer rgb)
	{
		nodes_[orig_2_node_index_[index]].set_color(rgb);
	}
	void Tree.unset_color(integer index)
	{
		nodes_[orig_2_node_index_[index]].unset_color();
	}

	/*
	 * Tooltips
	 */
	void Tree.set_tooltips(vector(string) tooltips)
	{
		integer nodes__sz = v_size(nodes_);
		if (v_size(tooltips) != nodes__sz) {
			throw (E_UNSPECIFIC, "Argument tooltips has wrong size");
		}
		for (integer i = 0; i < nodes__sz; ++i) {
			__impl__.Node node = nodes_[i];
			node.set_tooltip(tooltips[node.original_index()]);
		}
	}
	void Tree.set_tooltip(integer index, string option(nullable) tooltip)
	{
		nodes_[orig_2_node_index_[index]].set_tooltip(tooltip);
	}


	/*
	 * Expanding and collapsing
	 */
	void Tree.set_double_click_expansion(logical dbl_clk)
	{
		dbl_clk_expansion_ = dbl_clk;
	}

	void Tree.expand_to_nodes(
		vector(integer) indices, logical collapse_others, logical expand_given_nodes)
	{
		if (collapse_others) {
			nodes_.set_collapsed(true);
		}

		integer n = v_size(indices);
		vector(integer) node_indices[n];
		for (integer i = 0; i < n; ++i) {
			node_indices[i] = orig_2_node_index_[indices[i]];
		}
		sort(node_indices);

		vector(integer) current = node_indices;
		vector(integer) next[0];
		while (v_size(current) > 0) {
			integer current_sz = v_size(current);
			for (integer i = 0; i < current_sz; ++i) {
				__impl__.Node node = nodes_[current[i]];
				if (node.parent() != -1) {
					push_back(next, node.parent());
				}
			}

			nodes_[next].set_collapsed(false);
			current = clone_vector(next);
			resize(next, 0);
		}

		if (expand_given_nodes) {
			nodes_[node_indices].set_collapsed(false);
		}
	}

	void Tree.set_expansion_depth(integer depth)
	{
		// Depth = 0 means collapse tree completely
		// Positive depth means expand nodes on this level
		// Depth = -1 means expand all nodes
		// Depth = -2 means expand all nodes except those have only leaves below

		if (depth == -1) {
			nodes_.set_collapsed(false);
		} else if (depth == -2) {
			// Collapse all, then loop through all and for non-leaves, expand parent
			nodes_.set_collapsed(true);
			integer nodes__sz = v_size(nodes_);
			for (integer i = 0; i < nodes__sz; ++i) {
				__impl__.Node node = nodes_[i];
				if (!node.is_leaf()) {
					integer p = node.parent();
					if (p != -1) {
						nodes_[p].set_collapsed(false);
					}
				}
			}
		} else if (depth >= 0) {
			integer nodes__sz = v_size(nodes_);
			for (integer i = 0; i < nodes__sz; ++i) {
				__impl__.Node node = nodes_[i];
				node.set_collapsed(node.depth() >= depth);
			}
		} else {
			throw (E_UNSPECIFIC, "Negative depth other than -1 or -2 not allowed");
		}

		set_expansion_depth_ = depth;

		_set_collapse_dirty();
	}

	number Tree.get_set_expansion_depth()
	{
		return set_expansion_depth_;
	}

	integer Tree.get_max_depth()
	{
		integer max_depth = 0;
		integer nodes__sz = v_size(nodes_);
		for (integer i = 0; i < nodes__sz; ++i) {
			max_depth = max(max_depth, nodes_[i].depth());
		}
		return max_depth;
	}

	void Tree.set_auto_expand(logical auto_expand)
	{
		auto_expand_ = auto_expand;
	}

	void Tree.set_cascade_collapse(logical cascade_collapse)
	{
		cascade_collapse_ = cascade_collapse;
	}


	/*
	 * Special selection behaviour
	 */
	void Tree.set_ignore_select_click(integer index, logical ignore)
	{
		nodes_[orig_2_node_index_[index]].set_ignore_select_click(ignore);
	}

	void Tree.set_single_select(logical single_select)
	{
		single_select_ = single_select;
	}


	/*
	 * Free form counting of different node types
	 */
	void Tree.set_count_types(vector(integer) index, vector(string) count_types)
	{
		nodes_[orig_2_node_index_[index]].set_count_type(count_types);
		_set_selection_dirty();
	}


	/*
	 * Filter tree
	 */
	void Tree.set_filter(vector(integer) shown)
	{
		this.clear_filter();
		this.apply_and_filter(shown);
	}

	void Tree.apply_or_filter(vector(integer) shown)
	{
		_set_filter(shown, true);
	}

	void Tree.apply_and_filter(vector(integer) shown)
	{
		_set_filter(shown, false);
	}

	void Tree.clear_filter()
	{
		nodes_.set_shown(true);
		_set_collapse_dirty();
	}

	vector(integer) Tree.get_filter_shown()
	{
		vector(integer) orig_indices[0];
		integer nodes__sz = v_size(nodes_);
		for (integer i = 0; i < nodes__sz; ++i) {
			if (nodes_[i].shown())
				push_back(orig_indices, nodes_[i].original_index());
		}

		sort(orig_indices);
		return orig_indices;
	}

	/*
	 * Tree structure information
	 */
	integer Tree.num_nodes() = v_size(nodes_);

	vector(integer) Tree.get_children(integer index)
	{
		vector(integer) children[0];

		if (index == -1) {
			// Roots are sought
			integer nodes__sz = v_size(nodes_);
			for (integer i = 0; i < nodes__sz; ++i) {
				if (nodes_[i].parent() == -1)
					push_back(children, nodes_[i].original_index());
			}
		} else if (index >= 0) {
			integer nodes__sz = v_size(nodes_);
			integer node = orig_2_node_index_[index];
			for (integer i = node+1; i < nodes__sz; ++i) {
				if (nodes_[i].parent() == node)
					push_back(children, nodes_[i].original_index());
				else if (nodes_[i].parent() < node)
					// Reached next subtree
					break;
			}
		} else {
			throw (E_UNSPECIFIC, "index must be -1 (to get roots) or >= 0 (to get non-roots)");
		}

		return children;
	}

	integer Tree.get_parent(integer index)
	{
		integer node = orig_2_node_index_[index];
		return nodes_[node].parent();
	}

	integer Tree.index_of_row(integer row) = this.index_of_row([row])[0];

	vector(integer) Tree.index_of_row(vector(integer) rows)
	{
		vector(integer) tree_rows = rows - (show_header_ ? 1 : 0);
		vector(integer) shown = _shown_nodes();
		integer n_shown = v_size(shown);
		integer rows_sz = v_size(rows);
		vector(integer) ret[rows_sz];
		for (integer i = 0; i < rows_sz; ++i) {
			if (tree_rows[i] < 0 || tree_rows[i] >= n_shown) {
				ret[i] = -1;
			} else {
				ret[i] = nodes_[shown[tree_rows[i]]].original_index();
			}
		}
		return ret;
	}

	integer Tree.num_rows()
	{
		return v_size(_shown_nodes()) + (show_header_ ? 1 : 0);
	}


	/*
	 * Programmable selection
	 */
	void Tree.select(integer index, logical only_shown)
	{
		if (index < 0 || index >= v_size(nodes_)) {
			throw (E_UNSPECIFIC, strcat(["Invalid index: ", str(index)]));
		}
		_set_selected(orig_2_node_index_[index], true, only_shown);
	}

	void Tree.deselect(integer index, logical only_shown)
	{
		if (index < 0 || index >= v_size(nodes_)) {
			throw (E_UNSPECIFIC, strcat(["Invalid index: ", str(index)]));
		}
		_set_selected(orig_2_node_index_[index], false, only_shown);
	}

	void Tree.select_by_filter(FilterSelectMethod method)
	{
		switch (method) {
			case FS_ADD:
				_set_select_by_filter(true, true);
				break;
			case FS_REMOVE:
				_set_select_by_filter(true, false);
				break;
			case FS_SET_TO_SHOWN:
				_set_select_by_filter(false, false);
				_set_select_by_filter(true, true);
				break;
			case FS_SET_TO_HIDDEN:
				_set_select_by_filter(true, false);
				_set_select_by_filter(false, true);
				break;
			default:
				throw (E_UNSPECIFIC, "Internal: unhandled FilterSelectMethod");
		}
	}

	/*
	 * Search
	 */
	vector(integer) Tree.search(
		vector(string) keys,
		vector(SearchOption) options,
		logical check_label,
		vector(string) option(nullable) col_ids, // null for all columns
		vector(string) option(nullable) data_ids) // null for all data
	{
		logical case_i = false;
		logical complete_match = false;
		logical only_leaves = false;

		integer options_sz = v_size(options);
		for (integer i = 0; i < options_sz; ++i) {
			switch (options[i]) {
				case SO_CASEI:
					case_i = true;
					break;
				case SO_COMPLETE_MATCH:
					complete_match = true;
					break;
				case SO_ONLY_LEAVES:
					only_leaves = true;
					break;
				default:
					throw (E_UNSPECIFIC, "Internal: unhandled SearchOption");
			}
		}

		integer n_extra_cols = v_size(extra_col_ids_);
		vector(integer) extra_cols_to_check[0];
		if (null(col_ids)) {
			if (n_extra_cols > 0) {
				extra_cols_to_check = integer(s2v(series(i: 0, n_extra_cols-1, 1; i)));
			}
		} else {
			for (integer i = 0; i < n_extra_cols; ++i) {
				for (integer j = 0; j < v_size(col_ids); ++j) {
					if (extra_col_ids_[i] == col_ids[j]) {
						push_back(extra_cols_to_check, i);
						break;
					}
				}
			}
		}
		n_extra_cols = v_size(extra_cols_to_check);

		integer n_extra_datas = v_size(extra_data_ids_);
		vector(integer) extra_data_to_check[0];
		if (null(data_ids)) {
			if (n_extra_datas > 0) {
				extra_data_to_check = integer(s2v(series(i: 0, n_extra_datas-1, 1; i)));
			}
		} else {
			for (integer i = 0; i < n_extra_datas; ++i) {
				for (integer j = 0; j < v_size(data_ids); ++j) {
					if (extra_data_ids_[i] == data_ids[j]) {
						push_back(extra_data_to_check, i);
						break;
					}
				}
			}
		}
		n_extra_datas = v_size(extra_data_to_check);

		vector(integer) matched_nodes[0];
		integer nodes__sz = v_size(nodes_);
		integer keys_sz = v_size(keys);
		for (integer i = 0; i < nodes__sz; ++i) {
			logical matched = false;

			if (only_leaves && !nodes_[i].is_leaf()) {
				continue;
			}

			if (check_label) {
				string label = nodes_[i].label();
				if (case_i) {
					label = str_to_upper(label);
				}
				for (integer k = 0; k < keys_sz; ++k) {
					string key = case_i ? str_to_upper(keys[k]) : keys[k];
					if (complete_match) {
						if (label == key) {
							push_back(matched_nodes, i);
							matched = true;
							break;
						}
					} else {
						if (find_string(label, key)) {
							push_back(matched_nodes, i);
							matched = true;
							break;
						}
					}
				}
			}

			if (matched) {
				continue;
			}

			for (integer e = 0; e < n_extra_cols; e++) {
				string val = extra_cols_[extra_cols_to_check[e]].get()[i].text();
				if (case_i) {
					val = str_to_upper(val);
				}
				for (integer k = 0; k < keys_sz; ++k) {
					string key = case_i ? str_to_upper(keys[k]) : keys[k];
					if (complete_match) {
						if (val == key) {
							matched = true;
							push_back(matched_nodes, i);
							break;
						}
					} else {
						if (find_string(val, key)) {
							matched = true;
							push_back(matched_nodes, i);
							break;
						}
					}
				}
				if (matched) {
					break;
				}
			}

			for (integer e = 0; e < n_extra_datas; e++) {
				string val = extra_data_[extra_data_to_check[e]].get()[i];
				if (case_i) {
					val = str_to_upper(val);
				}
				for (integer k = 0; k < keys_sz; ++k) {
					string key = case_i ? str_to_upper(keys[k]) : keys[k];
					if (complete_match) {
						if (__impl__.equaln(val,key)) {
							matched = true;
							push_back(matched_nodes, i);
							break;
						}
					} else {
						if (__impl__.find_stringn(val, key)) {
							matched = true;
							push_back(matched_nodes, i);
							break;
						}
					}
				}
				if (matched) {
					break;
				}
			}
		}

		vector(integer) matched_orig_indices = nodes_[matched_nodes].original_index();
		sort(matched_orig_indices);
		return matched_orig_indices;
	}


	/*
	 * Play arond with what is leaf and what isn't
	 */
	void Tree.set_leaf_status(vector(logical) is_leaf)
	{
		integer n = v_size(is_leaf);
		if (n != v_size(nodes_)) {
			throw (E_UNSPECIFIC, "Argument is_leaf has wrong size");
		}

		vector(integer) leaf_ixs[0];
		for (integer i = 0; i < n; ++i) {
			if (is_leaf[i])
				push_back(leaf_ixs, i);
		}

		this.set_leaves(leaf_ixs);
	}

	void Tree.set_leaves(vector(integer) leaf_indices)
	{
		// Remember for reverting in case of erronous leaf_indices vector
		vector(__impl__.leafiness) original_leafiness = nodes_.get_leafiness();

		vector(integer) node_ixs = orig_2_node_index_[leaf_indices];

		nodes_.set_leafiness(__impl__.L_BRANCH);
		nodes_[node_ixs].set_leafiness(__impl__.L_LEAF);

		// Check that no leaf has a leaf ancestor
		integer n = v_size(nodes_);
		vector(logical) can_be_leaf[n];
		can_be_leaf = true;
		for (integer i = n-1; i >= 0; --i) {
			__impl__.Node node = nodes_[i];
			logical node_can_be_leaf = can_be_leaf[i];

			if (node.is_leaf() && !node_can_be_leaf) {
				for (integer j = 0; j < n; ++j) {
					// Revert nodes to original leaf status
					nodes_[j].set_leafiness(original_leafiness[j]);
				}
				throw (E_UNSPECIFIC, strcat(["Node ", str(node.original_index()), " '", node.label(), "' cannot both be a leaf and have descendants that are leaves"]));
			}

			if (node.is_leaf() || !node_can_be_leaf) {
				integer p = node.parent();
				if (p != -1)
					can_be_leaf[p] = false;
			}
		}

		// Set correct leafiness on sub-leaf nodes
		integer leaf_depth = -1; // -1: Not in leaf tree, otherwise depth of current leaf
		for (integer i = 0; i < n; ++i) {
			__impl__.Node node = nodes_[i];

			if (node.depth() <= leaf_depth) {
				leaf_depth = -1; // Stepped out of leaf tree
			}

			if (leaf_depth >= 0) {
				node.set_leafiness(__impl__.L_SUBLEAF); // In leaf tree
			} else if (node.is_leaf()) {
				leaf_depth = node.depth(); // Entering a leaf tree
			}
		}

		_set_selection_dirty();
	}

	/*
	 * Keep info that is not shown
	 */
	void Tree.add_data(string id, vector(string) data)
	{
		integer n = v_size(nodes_);
		if (v_size(data) != n) {
			throw (E_UNSPECIFIC, "Argument data has wrong size");
		}

		vector(string) node_sorted_vec[n];
		for (integer i = 0; i < n; ++i) {
			node_sorted_vec[i] = data[nodes_[i].original_index()];
		}

		push_back(extra_data_, new __impl__.string_v(node_sorted_vec));
		push_back(extra_data_ids_, id);
	}

	string Tree.get_data(string id, integer index)
	{
		integer extra_data_ids__sz = v_size(extra_data_ids_);
		for (integer i = 0; i < extra_data_ids__sz; ++i) {
			if (extra_data_ids_[i] == id) {
				return extra_data_[i].get()[orig_2_node_index_[index]];
			}
		}
		throw (E_UNSPECIFIC, strcat(["There is no data with id '", id, "'"]));
	}

	void Tree.set_data(string id, integer index, string option(nullable) data)
	{
		integer extra_data_ids__sz = v_size(extra_data_ids_);
		for (integer i = 0; i < extra_data_ids__sz; ++i) {
			if (extra_data_ids_[i] == id) {
				extra_data_[i].get()[orig_2_node_index_[index]] = data;
				return;
			}
		}
		throw (E_UNSPECIFIC, strcat(["There is no data with id '", id, "'"]));
	}

	void find_all_parents(vector(__impl__.Node) v_node, integer child, out vector(integer) v_parent)
	{
		integer parent = v_node[child].parent();
		if (parent != -1)
		{
			push_back(v_parent, parent);
			find_all_parents(v_node, parent, v_parent);
		}
	}

	void Tree.set_data_recursively(string id, integer index, string option(nullable) data)
	{
		vector(integer) v_index;
		find_all_parents(nodes_, index, v_index);
		push_back(v_index, index);
		set_data(id, v_index, data);
	}

	void Tree.set_debug_mode(logical val)
	{
		if (debug_mode_ != val) {
			debug_mode_ = val;
			_recalc_columns();
		}
	}

	map_str_str Tree.get_settings()
	{
		map_str_str ret = map_str_str();

		ret.add("selection_mode", string(selection_mode_));
		ret.add("debug_mode", string(debug_mode_));
		ret.add("show_header", string(show_header_));
		ret.add("dbl_clk_expansion", string(dbl_clk_expansion_));
		ret.add("auto_expand", string(auto_expand_));
		ret.add("cascade_collapse", string(cascade_collapse_));
		ret.add("set_expansion_depth", string(set_expansion_depth_));
		ret.add("single_select", string(single_select_));
		ret.add("node_color", string(node_color_));
		ret.add("leaf_color", string(leaf_color_));
		ret.add("selected_color", string(selected_color_));
		ret.add("has_selected_color", string(has_selected_color_));
		ret.add("clicked_color", string(clicked_color_));
		for (integer i = 0; i < display_element.NUM_ELEMS; ++i) {
			ret.add(strcat("display_element #", str(i)), display_elements_[i]);
		}
		return ret;
	}

	void Tree._refresh()
	{
		integer n = v_size(nodes_);

		if (dirty_collapse_) {
			resize(shown_nodes_, 0);
			integer collapsed_level = -1;
			for (integer i = 0; i < n; ++i) {
				__impl__.Node node = nodes_[i];
				if (!node.shown() ||
					(collapsed_level >= 0 && node.depth() > collapsed_level)) {
					continue;
				} else {
					push_back(shown_nodes_, i);
					if (node.collapsed()) {
						collapsed_level = node.depth();
					} else {
						collapsed_level = -1;
					}
				}
			}
		}

		if (dirty_selection_) {
			resize(cache_selected_, 0);
			for (integer i = 0; i < n; ++i) {
				if (nodes_[i].selected()) {
					if (selection_mode_ == SELECT_ANY ||
						((selection_mode_ == SELECT_LEAVES && nodes_[i].is_leaf()))) {
						push_back(cache_selected_, i);
					}
				}
			}

			if (selection_mode_ == SELECT_LEAVES) {
				resize(cache_num_sub_selected_, n);
				cache_num_sub_selected_ = 0;
				resize(cache_num_sub_selected_by_count_type_, n);
				resize(cache_num_sub_leaves_, n);
				cache_num_sub_leaves_ = 0;
				resize(cache_num_sub_leaves_by_count_type_, n);

				for (integer i = 0; i < n; ++i) {
					cache_num_sub_selected_by_count_type_[i] = new count_set();
					cache_num_sub_leaves_by_count_type_[i] = new count_set();
				}

				for (integer i = n-1; i >= 0; --i) {
					__impl__.Node node = nodes_[i];
					integer p = node.parent();
					if (p != -1) {

						if (node.is_leaf() && node.selected()) {
							cache_num_sub_selected_[p]++;
							if (!null(node.count_type())) {
								cache_num_sub_selected_by_count_type_[p].increase(
									node.count_type());
							}
						} else {
							cache_num_sub_selected_[p] += cache_num_sub_selected_[i];
							add_counts(
								cache_num_sub_selected_by_count_type_[p],
								cache_num_sub_selected_by_count_type_[i]);
						}

						if (node.is_leaf()) {
							cache_num_sub_leaves_[p]++;
							cache_num_sub_leaves_by_count_type_[p].increase(node.count_type());
						} else {
							cache_num_sub_leaves_[p] += cache_num_sub_leaves_[i];
							add_counts(
								cache_num_sub_leaves_by_count_type_[p],
								cache_num_sub_leaves_by_count_type_[i]);
						}

					}
				}
			}

			dirty_selection_ = false;
		}

	}

	vector(integer) Tree._shown_nodes()
	{
		_refresh();
		return shown_nodes_;
	}

	void Tree._set_selected(integer node_index, logical new_status, logical only_if_shown)
	{
		if (selection_mode_ == SELECT_ANY) {
			__impl__.Node node = nodes_[node_index];
			if (!only_if_shown || node.shown()) {
				if (node.selected() != new_status) {
					node.set_selected(new_status);
					_set_selection_dirty();
				}
			}
		} else if (selection_mode_ == SELECT_LEAVES) {
			if (!only_if_shown || nodes_[node_index].shown()) {
				nodes_[node_index].set_selected(new_status);
				vector(integer) leaf_indices = _leaves(node_index);
				integer leaf_indices_sz = v_size(leaf_indices);
				for (integer i = 0; i < leaf_indices_sz; ++i) {
					integer index = leaf_indices[i];
					if (!only_if_shown || nodes_[index].shown()) {
						nodes_[index].set_selected(new_status);
					}
				}
			}
			_set_selection_dirty();
			if (new_status == true && auto_expand_) {
				vector(integer) sub_node_indices = _sub_nodes(node_index);
				nodes_[sub_node_indices].set_collapsed(false);
				_set_collapse_dirty();
			}
		}
	}

	void Tree._set_filter(vector(integer) indices, logical apply_or)
	{
		set_num shown_orig = set_num();
		shown_orig.add(indices);

		// First pass from top to bottom. Show node if
		// - it was shown to begin with
		// AND / OR (according to apply_or)
		// - it is set to be shown
		//   OR its parent is shown

		// Second pass from bottom to top
		// If node is shown, show its parent

		// Third pass, expand shown nodes

		integer n = v_size(nodes_);

		for (integer i = 0; i < n; ++i) {
			__impl__.Node node = nodes_[i];

			logical match =
				(node.parent() != -1 && nodes_[node.parent()].shown()) ||
				shown_orig.includes(node.original_index());

			if (apply_or) {
				node.set_shown(node.shown() || match);
			} else {
				node.set_shown(node.shown() && match);
			}
		}

		for (integer i = n-1; i >= 0; --i) {
			__impl__.Node node = nodes_[i];
			if (node.shown() &&
				node.parent() != -1)
			{
				nodes_[node.parent()].set_shown(true);
			}
		}

		if (auto_expand_) {
			for (integer i = 0; i < n; ++i) {
				__impl__.Node node = nodes_[i];
				if (node.collapsed() && node.shown()) {
					node.set_collapsed(false);
				}
			}
		}

		_set_collapse_dirty();
	}

	void Tree._recalc_columns()
	{
		label_col_ = selection_mode_ == SELECT_NONE ? 0 : 1;
		sel_col_ = selection_mode_ == SELECT_NONE ? -5 : 0; // -5 so we never identify clicks there
		extra_col_ = label_col_ + 1;
		debug_col_ = debug_mode_ ? extra_col_ + v_size(extra_cols_) : -5;

		num_cols_ =
			1 +
			(selection_mode_ == SELECT_NONE ? 0 : 1) +
			v_size(extra_cols_shown_) +
			(debug_mode_ ? 1 : 0);
	}

	integer Tree._extra_col_index(string id)
	{
		for (integer i = 0; i < v_size(extra_col_ids_); ++i) {
			if (extra_col_ids_[i] == id)
				return i;
		}
		throw (E_UNSPECIFIC, strcat("No such column: ", id));
	}

	vector(integer) Tree._leaves(integer root_index)
	{
		vector(integer) leaf_indices[0];

		if (nodes_[root_index].is_leaf()) {
			return [root_index];
		}

		vector(integer) sub_nodes = _sub_nodes(root_index);
		integer sub_nodes_sz = v_size(sub_nodes);
		for (integer i = 0; i < sub_nodes_sz; ++i) {
			if (nodes_[sub_nodes[i]].is_leaf()) {
				push_back(leaf_indices, sub_nodes[i]);
			}
		}

		return leaf_indices;
	}

	vector(integer) Tree._sub_nodes(integer root_index)
	{
		vector(integer) indices = [root_index];

		// By starting at the root index and going forward in the node vector
		// until we reach a node with depth <= the depth of the root index, we
		// can be sure we have traversed the whole sub tree, and nothing but
		// the sub tree, of the root.

		integer root_depth = nodes_[root_index].depth();
		integer n = v_size(nodes_);
		for (integer index = root_index+1; index < n; ++index) {
			if (nodes_[index].depth() <= root_depth) {
				break;
			}
			push_back(indices, index);
		}

		return indices;
	}

	void Tree._switch_collapse(integer row)
	{
		integer index = _shown_nodes()[row];
		__impl__.Node node = nodes_[index];
		node.set_collapsed(!node.collapsed());
		if (cascade_collapse_ && node.collapsed()) {
			nodes_[_sub_nodes(index)].set_collapsed(true);
		}
		_set_collapse_dirty();
	}

	void Tree._set_select_by_filter(logical on_filter_status, logical new_status)
	{
		integer nodes__sz = v_size(nodes_);
		for (integer i = 0; i < nodes__sz; ++i) {
			__impl__.Node node = nodes_[i];
			if (node.shown() == on_filter_status) {
				node.set_selected(new_status);
			}
		}
		_set_selection_dirty();
	}

	void Tree._set_collapse_dirty()
	{
		dirty_collapse_ = true;
	}

	void Tree._set_selection_dirty()
	{
		dirty_selection_ = true;
	}

	vector(integer) Tree._orig_to_index_vec() = clone_vector(orig_2_node_index_);



class TreeData {
public:
	TreeData();
	void set_data_labels(vector(string) data_labels);
	void set_column_labels(vector(string) column_labels);

	integer add_root(string label);
	integer add_node(string label, integer parent);

	void set_color(integer index, integer rgb);
	void set_color_by_data(string data_label, string data_value, integer rgb);

	void set_tooltip(integer index, string option(nullable) tooltip);

	void set_data(integer index, string data_label, string data_value);
	void set_column(integer index, string column_label, string column_value, integer rgb = -1);

	Tree create_tree(string id, SelectionMode selection_mode);

private:
	vector(string) labels_;
	vector(integer) parents_;
	vector(integer) colors_;
	vector(string) tooltips_;

	vector(string) data_labels_;
	map_str_v_str data_;
	vector(string) column_labels_;
	map_str_v_str column_;
	map_str_v_int column_colors_;

	integer _add_node(string label, integer parent);
};

TreeData.TreeData()
	: data_(new map_str_v_str()), column_(new map_str_v_str()), column_colors_(new map_str_v_int())
{
	resize(labels_, 0);
	resize(parents_, 0);
	resize(colors_, 0);
	resize(tooltips_, 0);
	resize(data_labels_, 0);
	resize(column_labels_, 0);
}

void TreeData.set_data_labels(vector(string) data_labels)
{
	if (v_size(labels_) > 0)
		throw (E_UNSPECIFIC, "This function cannot be called after the first root has been added");

	data_labels_ = clone_vector(data_labels);
	data_ = new map_str_v_str();
	for (integer i = 0; i < v_size(data_labels_); ++i) {
		if (!null(data_.find(data_labels_[i])))
			throw (E_UNSPECIFIC, strcat("Repeated data_label: ", data_labels_[i]));
		data_.emplace(data_labels_[i]);
	}
}

void TreeData.set_column_labels(vector(string) column_labels)
{
	if (v_size(labels_) > 0)
		throw (E_UNSPECIFIC, "This function cannot be called after the first root has been added");

	column_labels_ = clone_vector(column_labels);
	column_ = new map_str_v_str();
	column_colors_ = new map_str_v_int();
	for (integer i = 0; i < v_size(column_labels_); ++i) {
		if (!null(column_.find(column_labels_[i])))
			throw (E_UNSPECIFIC, strcat("Repeated column_label: ", column_labels_[i]));
		column_.emplace(column_labels_[i]);
		column_colors_.emplace(column_labels_[i]);
	}
}

integer TreeData.add_root(string label) = _add_node(label, -1);
integer TreeData.add_node(string label, integer parent) = _add_node(label, parent);

void TreeData.set_color(integer index, integer rgb)
{
	colors_[index] = rgb;
}

void TreeData.set_color_by_data(string data_label, string data_value, integer rgb)
{
	vector(string) values = data_.find(data_label);
	if (null(values))
		throw (E_UNSPECIFIC, strcat("data_label does not exist: ", data_label));

	for (integer i = 0; i < v_size(values); ++i) {
		if (values[i] == data_value)
			colors_[i] = rgb;
	}
}

void TreeData.set_tooltip(integer index, string option(nullable) tooltip)
{
	tooltips_[index] = tooltip;
}

void TreeData.set_data(integer index, string data_label, string data_value)
{
	vector(string) values = data_.emplace(data_label);
	if (v_size(values) == 0 && v_size(labels_) > 0)
		throw (E_UNSPECIFIC, strcat("data_label does not exist: ", data_label));

	values[index] = data_value;
}

void TreeData.set_column(integer index, string column_label, string column_value, integer rgb)
{
	vector(string) values = column_.emplace(column_label);
	if (v_size(values) == 0 && v_size(labels_) > 0)
		throw (E_UNSPECIFIC, strcat("column_label does not exist: ", column_label));

	vector(integer) colors = column_colors_.emplace(column_label);

	values[index] = column_value;
	colors[index] = rgb;
}

Tree TreeData.create_tree(string id, SelectionMode selection_mode)
{
	Tree t = new Tree(id, labels_, parents_, selection_mode);

	for (integer i = 0; i < v_size(data_labels_); ++i) {
		t.add_data(data_labels_[i], data_.find(data_labels_[i]));
	}

	for (integer i = 0; i < v_size(column_labels_); ++i) {
		vector(text_rgb) contents = text_rgb(
			column_.find(column_labels_[i]),
			-1,
			column_colors_.find(column_labels_[i]));

		t.add_column(column_labels_[i], contents);
	}

	t.set_default_colors(colors_);
	t.set_tooltips(tooltips_);

	return t;
}

integer TreeData._add_node(string label, integer parent)
{
	if (parent >= v_size(labels_))
		throw (E_UNSPECIFIC, strcat("Invalid parent index: ", str(parent)));
	integer ix = v_size(labels_);
	push_back(labels_, label);
	push_back(parents_, parent);
	push_back(colors_, -1);
	push_back(tooltips_, null<string>);

	for (integer i = 0; i < v_size(data_labels_); ++i) {
		vector(string) values = data_.emplace(data_labels_[i]);
		push_back(values, null<string>);
	}

	for (integer i = 0; i < v_size(column_labels_); ++i) {
		vector(string) values = column_.emplace(column_labels_[i]);
		push_back(values, null<string>);
		vector(integer) colors = column_colors_.emplace(column_labels_[i]);
		push_back(colors, -1);
	}

	return ix;
}


/* First column (by `column_delim`) should be labels for the tree. The column
 * should contain full path for tree, separated with `child_delim`.
 *
 * Columns after this are added with column names given in `columns`, and/or
 * added as data with names as in `data`. Null in `data` or `columns` means
 * ignore the column for the purpose.
 *
 * Example with column_delim = '|', child_delim = '/':
 *
 * father|77
 * father/myself|38
 * father/brother|36
 * father/myself/son|6
 * uncle|74
 * uncle/cousin|41
 */
TreeData parse_file(
	string file,
	string column_delim,
	string child_delim,
	vector(string) columns,
	vector(string) data,
	logical sensitive)
{
	in_stream fh = in_stream_file(file);

	string line;

	map_str_int index_map = map_str_int();

	tree.TreeData td = new TreeData();
	td.set_column_labels(columns);

	while (!null(line = fh.read_line())) {
		vector(string) elems = str_trim(str_tokenize(line, column_delim));

		if (elems[0] == "")
			continue;

		if (v_size(elems) != v_size(columns)+1) {
			if (sensitive)
				throw (E_UNSPECIFIC, strcat(["Wrong number of columns in line: ", line]));
			else
				continue;
		}

		vector(string) full_path = str_tokenize(elems[0], child_delim);

		string parent_path = "";
		for (integer i = 0; i < v_size(full_path); ++i) {
			string label = full_path[i];
			string my_path = strcat([parent_path, child_delim, label]);

			integer ix = index_map.find(my_path);
			if (ix == -1) {
				integer parent_ix = index_map.find(parent_path);
				ix = td.add_node(label, parent_ix);

				if (i == v_size(full_path)-1) {
					for (integer c = 0; c < v_size(columns); ++c) {
						if (null(columns[c]))
							continue;
						try {
							td.set_column(ix, columns[c], elems[c+1]);
						} catch {
							if (sensitive)
								throw (E_UNSPECIFIC, strcat(["Error with column ", str(c), " (", columns[c], "): ", err.message()]));
						}
					}
					for (integer d = 0; d < v_size(data); ++d) {
						if (null(data[d]))
							continue;
						try {
							td.set_data(ix, data[d], elems[d+1]);
						} catch {
							if (sensitive)
								throw (E_UNSPECIFIC, strcat(["Error with column ", str(d), " (", data[d], "): ", err.message()]));
						}
					}
				}
				index_map.add(my_path, ix);
			}
		}
	}

	return td;
}

	/*
	 * A bit of a secret function. Prints matrix(text_rgb) to file,
	 * which we use for testing the tree module.
	 */
	void print_matrix(string filename, matrix(text_rgb) T)
	{
		out_stream fh = out_stream_file(filename);
		integer n = n_rows(T);
		integer m = n_cols(T);

		for (integer i = 0; i < n; ++i) {
			for (integer j = 0; j < m; ++j) {
				fh.write(strcat([T[i,j].text(), "|", str(T[i,j].bg())]));
				if (j+1 < m) {
					fh.write("\t");
				}
			}
			fh.write("\r\n");
		}
	}








	/******************
	 *
	 * FREE TREE FUNCTIONS
	 *
	 * These are free functions that take an treename, and finds the tree in an
	 * internal map. Users are recommended to use these and not Tree object
	 * directly. It's easier, and you get free local source trigger for your
	 * tree!
	 *
	 * The trigger is activated by `init`, `click` and `redraw`. If you want
	 * the tree to be redrawn after any other action, call `redraw` in that out
	 * function. The only function to react to the trigger is `show`, if other
	 * functions need to react, add a call to `listen_to_trigger` there
	 * (e.g. if you have `get_selected` somewhere and want it to run on every
	 * selection change). If you want to call show without hooking up to the
	 * trigger, call show_once().
	 *
	 **********************/




	// Free functions for ease of use!

	enum ColorTarget option(category: "Workspace/Tree/Color") {
		C_HEADER option(str: "Header")
		, C_NODE option(str: "Node")
		, C_LEAF option(str: "Leaf")
		, C_SUBLEAF option(str: "Subleaf")
		, C_SELECTED option(str: "Selected")
		// C_HAS_SELECTED is used in SELECT_LEAVES mode, for non-leaves with selected leaves in subtree
		, C_HAS_SELECTED option(str: "Has selected")
		, C_CLICKED option(str: "Clicked")
	};

	map_str_obj<Tree> __treemap__;
	Tree find_or_null(string treename) = __treemap__.find(treename);
	Tree find_or_throw(string treename)
	{
		Tree t = __treemap__.find(treename);
		if (null(t)) {
			throw (E_UNSPECIFIC, strcat(["No tree '", treename, "' exists"]));
		}
		return t;
	}

	string tree_ric(string name) = strcat([":TREE:", name, "//x("]);

	void local_publish_tree(string treename)
	{
		local.publish(__impl__.LOCAL_SOURCE, tree_ric(treename), ['fid'], ['val']);
	}

	void local_get_tree(string treename)
	{
		rt.get(tree_ric(treename), 'fid', __impl__.LOCAL_SOURCE);
	}

	void local_publish_and_get_tree(string treename)
	{
		local_publish_tree(treename);

		integer num_attempts = 10;
		integer sleep_ms = 20;
		for (integer i = 0; i < num_attempts; ++i) {
			try {
				local_get_tree(treename);
				return;
			} catch {
				sleep(sleep_ms);
			}
		}
		throw (E_UNSPECIFIC, strcat(["Timeout publishing tree ", treename, " to local source"]));
	}

	void init(
		string treename,
		vector(string) labels,
		vector(integer) parents,
		SelectionMode mode)
		option(category: "Workspace/Tree")
	{
		if (null(__treemap__)) {
			__treemap__ = new map_str_obj<Tree>();
		}
		Tree t = new Tree(treename, labels, parents, mode);
		__treemap__.add(treename, t);
		local_publish_tree(treename);
	}

	void init(
		string treename,
		vector(string) labels,
		vector(integer) parents)
		option(category: "Workspace/Tree")
	{
		init(treename, labels, parents, SELECT_NONE);
	}

	void init(
		string treename,
		TreeData treedata,
		SelectionMode mode)
		option(category: "Workspace/Tree")
	{
		if (null(__treemap__)) {
			__treemap__ = new map_str_obj<Tree>();
		}
		Tree t = treedata.create_tree(treename, mode);
		__treemap__.add(treename, t);
		local_publish_tree(treename);
	}

	matrix(text_rgb) show_once(string treename)
		option(category: "Workspace/Tree/Interact")
	{
		Tree t = find_or_null(treename);
		if (null(t)) {
			matrix(text_rgb) M[1,2];
			return M;
		} else {
			return t.display();
		}
	}

	matrix(text_rgb) show(string treename)
		option(category: "Workspace/Tree")
	{
		try {
			local_get_tree(treename);
		} catch {
			local_publish_and_get_tree(treename);
		}
		return show_once(treename);
	}

	integer click(string treename, number row, number col, logical option(nullable) is_single_click = null<logical>)
		option(category: "Workspace/Tree")
	{
		Tree t = find_or_null(treename);
		if (!null(t)) {
			integer ret = t.click(row, col, is_single_click);
			local_publish_tree(treename);
			return ret;
		} else {
			return -1;
		}
	}

	void set_double_click_expansion(string treename, logical dbl_clk)
		option(category: "Workspace/Tree")
	{
		find_or_throw(treename).set_double_click_expansion(dbl_clk);
	}

	vector(integer) get_selected(string treename)
		option(category: "Workspace/Tree")
	{
		return find_or_null(treename).get_selected();
	}


	// returns given id for extra columns, wsp.tree.COL_LABEL for label
	// column, wsp.tree.COL_SELECT for select column.
	string last_click_column(string treename) option(category: "Workspace/Tree/Interact") =
		find_or_null(treename).last_click_column();

	void remove(string treename)
		option(category: "Workspace/Tree/Misc")
	{
		__treemap__.remove(treename);
	}

	logical exists(string treename)
		option(category: "Workspace/Tree/Misc")
	{
		return !null(find_or_null(treename));
	}

	Tree get_obj(string treename)
		option(category: "Workspace/Tree/Misc")
	{
		return find_or_throw(treename);
	}

	void redraw(string treename)
		option(category: "Workspace/Tree/Interact")
	{
		local.publish(__impl__.LOCAL_SOURCE, tree_ric(treename), ['fid'], ['val']);
	}

	integer num_nodes(string treename)
		option(category: "Workspace/Tree/Interact")
	{
		return find_or_throw(treename).num_nodes();
	}

	vector(integer) get_children(string treename, integer index)
		option(category: "Workspace/Tree/Interact")
	{
		return find_or_throw(treename).get_children(index);
	}

	/* Color for a row is, in order of priority high->low:
	 * 1) Color set by set_clicked_color() if node was just clicked
	 * 2) Color set by set_selected_color() if node is selected, but not if
	 *    SelectionMode is SELECT_LEAVES and node is not a leaf
	 * 3) Color set by set_has_selected_color() if SelectionMode is
	 *    SELECT_LEAVES, node is not a leaf, and any node in the node subtree
	 *    is selected
	 * 3) Color of node set by set_color()
	 * 4) Default color of node set by set_default_colors()
	 * 5) Color set by set_leaf_color()/set_sub_leaf_color() if node is leaf or subleaf
	 * 6) Color set by set_node_color()
	 */

	void set_color(string treename, ColorTarget where, integer rgb)
		option(category: "Workspace/Tree/Color")
	{
		Tree t = find_or_throw(treename);
		switch (where) {

		case C_HEADER:
			t.set_header_color(rgb);
			break;
		case C_NODE:
			t.set_node_color(rgb);
			break;
		case C_LEAF:
			t.set_leaf_color(rgb);
			break;
		case C_SUBLEAF:
			t.set_sub_leaf_color(rgb);
			break;
		case C_SELECTED:
			t.set_selected_color(rgb);
			break;
		case C_HAS_SELECTED:
			t.set_has_selected_color(rgb);
			break;
		case C_CLICKED:
			t.set_clicked_color(rgb);
			break;
		default:
			throw (E_UNSPECIFIC, "Internal: Unhandled ColorTarget");
		}
	}

	void set_default_colors(string treename, vector(integer) rgbs)
		option(category: "Workspace/Tree/Color")
	{
		find_or_throw(treename).set_default_colors(rgbs);
	}

	void set_color(string treename, integer index, integer rgb)
		option(category: "Workspace/Tree/Color")
	{
		find_or_throw(treename).set_color(index, rgb);
	}

	void unset_color(string treename, integer index)
		option(category: "Workspace/Tree/Color")
	{
		find_or_throw(treename).unset_color(index);
	}

	void set_tooltips(string treename, vector(string) tooltips)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).set_tooltips(tooltips);
	}

	void set_tooltip(string treename, integer index, string option(nullable) tooltip)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).set_tooltip(index, tooltip);
	}

	// Expand tree to nodes given by indices are shown. If collapse_others is
	// true any other expanded nodes will be collapsed. If expand_given_nodes
	// is true the nodes given by indices will also be expandend (not just
	// their ancestors).
	void expand_to_nodes(
		string treename,
		vector(integer) indices,
		logical collapse_others = false,
		logical expand_given_nodes = false)
		option(category: "Workspace/Tree/Interact")
	{
		find_or_throw(treename).expand_to_nodes(indices, collapse_others, expand_given_nodes);
	}

	// Expand all nodes at depth < `depth`, collapse those with depth >= `depth`
	// depth = 0 collapses whole tree, depth = -1 expands whole tree
	void set_expansion_depth(string treename, integer depth)
		option(category: "Workspace/Tree/Interact")
	{
		find_or_throw(treename).set_expansion_depth(depth);
	}

	number get_set_expansion_depth(string treename)
		option(category: "Workspace/Tree/Interact")
	{
		return find_or_throw(treename).get_set_expansion_depth();
	}

	integer get_max_depth(string treename)
		option(category: "Workspace/Tree/Interact")
	{
		return find_or_throw(treename).get_max_depth();
	}

	void set_auto_expand(string treename, logical auto_expand)
		option(category: "Workspace/Tree/Interact")
	{
		find_or_throw(treename).set_auto_expand(auto_expand);
	}

	void set_cascade_collapse(string treename, logical cascade_collapse)
		option(category: "Workspace/Tree/Interact")
	{
		find_or_throw(treename).set_cascade_collapse(cascade_collapse);
	}

	// Tree will ignore clicks in the select column for indices with ignore set
	// to true. Tree will still respect direct calls to select/deselect where
	// the selection is controlled programatically as opposed to done in the table.
	void set_ignore_select_click(string treename, integer index, logical ignore)
		option(category: "Workspace/Tree/Filter and Select")
	{
		find_or_throw(treename).set_ignore_select_click(index, ignore);
	}

	// All nodes will be deselected before a selection click is processed. Keeps
	// selection limited to one node (possibly with its subtree). Tree will not
	// do any deselection upon calls to select/deselect, they will work normally.
	void set_single_select(string treename, logical single_select)
	{
		find_or_throw(treename).set_single_select(single_select);
	}

	void set_count_types(
		string treename, vector(integer) indices, vector(string) count_types)
	{
		find_or_throw(treename).set_count_types(indices, count_types);
	}

	integer index_of_row(string treename, integer row)
		option(category: "Workspace/Tree/Interact")
	{
		return find_or_throw(treename).index_of_row(row);
	}

	vector(integer) index_of_row(string treename, vector(integer) rows)
		option(category: "Workspace/Tree/Interact")
	{
		return find_or_throw(treename).index_of_row(rows);
	}

	integer num_rows(string treename)
		option(category: "Workspace/Tree/Interact")
	{
		return find_or_throw(treename).num_rows();
	}

	void use_alternative_style(string treename)
		option(category: "Workspace/Tree/Content")
	{
		vector(integer) elems = integer(s2v(series(i:0,display_element.NUM_ELEMS-1;i)));
		find_or_throw(treename).set_display_string(elems, display_element.alternatives);
	}

	void enable_header(string treename)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).set_header(true);
	}

	map_str_str get_settings(string treename)
		option(category: "Workspace/Tree/Misc")
	{
		return find_or_throw(treename).get_settings();
	}

	void disable_header(string treename)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).set_header(false);
	}

	void add_column(string treename, string col_id, vector(text_rgb) content)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).add_column(col_id, content);
	}

	void remove_column(string treename, string col_id)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).remove_column(col_id);
	}

	void hide_column(string treename, string col_id)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).hide_column(col_id);
	}

	void show_column(string treename, string col_id)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).show_column(col_id);
	}

	void set_column_header_color(string treename, string col_id, integer rgb)
	{
		find_or_throw(treename).set_column_header_color(col_id, rgb);
	}

	void unset_column_header_color(string treename, string col_id)
	{
		find_or_throw(treename).unset_column_header_color(col_id);
	}

	vector(string) get_shown_columns(string treename)
		option(category: "Workspace/Tree/Content")
	{
		return find_or_throw(treename).get_shown_columns();
	}

	void set_leaf_status(string treename, vector(logical) is_leaf)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).set_leaf_status(is_leaf);
	}

	void set_leaves(string treename, vector(integer) leaf_indices)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).set_leaves(leaf_indices);
	}

	void set_label(string treename, integer index, string option(nullable) label)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).set_label(index, label);
	}

	void set_column_val(string treename, string id, integer index, text_rgb option(nullable) val)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).set_column_val(id, index, val);
	}

	text_rgb get_column_val(string treename, string id, integer index)
		option(category: "Workspace/Tree/Content")
	{
		return find_or_throw(treename).get_column_val(id, index);
	}

	vector(integer) search_label(
		string treename,
		vector(string) keys,
		vector(SearchOption) options)
		option(category: "Workspace/Tree/Filter and Select")
	{
		vector(string) no_extra_cols[0];
		return find_or_throw(treename).search(keys, options, true, no_extra_cols, no_extra_cols);
	}

	vector(integer) search_columns(
		string treename,
		vector(string) keys,
		vector(SearchOption) options,
		logical check_label)
		option(category: "Workspace/Tree/Filter and Select")
	{
		vector(string) no_extra_data[0];
		return find_or_throw(treename).search(keys, options, check_label, null<vector(string)>, no_extra_data);
	}

	vector(integer) search_columns(
		string treename,
		vector(string) keys,
		vector(SearchOption) options,
		logical check_label,
		vector(string) extra_col_ids)
		option(category: "Workspace/Tree/Filter and Select")
	{
		vector(string) no_extra_data[0];
		return find_or_throw(treename).search(keys, options, check_label, extra_col_ids, no_extra_data);
	}

	vector(integer) search_data(
		string treename,
		vector(string) keys,
		vector(SearchOption) options)
		option(category: "Workspace/Tree/Filter and Select")
	{
		vector(string) no_extra_cols[0];
		return find_or_throw(treename).search(keys, options, false, no_extra_cols, null<vector(string)>);
	}

	vector(integer) search_data(
		string treename,
		vector(string) keys,
		vector(SearchOption) options,
		vector(string) extra_data_ids)
		option(category: "Workspace/Tree/Filter and Select")
	{
		vector(string) no_extra_cols[0];
		return find_or_throw(treename).search(keys, options, false, no_extra_cols, extra_data_ids);
	}

	void set_filter(string treename, vector(integer) indices)
		option(category: "Workspace/Tree/Filter and Select")
	{
		Tree t = find_or_throw(treename);
		t.set_filter(indices);
	}

	void apply_or_filter(string treename, vector(integer) indices)
		option(category: "Workspace/Tree/Filter and Select")
	{
		Tree t = find_or_throw(treename);
		t.apply_or_filter(indices);
	}

	void apply_and_filter(string treename, vector(integer) indices)
		option(category: "Workspace/Tree/Filter and Select")
	{
		Tree t = find_or_throw(treename);
		t.apply_and_filter(indices);
	}

	void clear_filter(string treename)
		option(category: "Workspace/Tree/Filter and Select")
	{
		Tree t = find_or_throw(treename);
		t.clear_filter();
	}

	vector(integer) get_filter_shown(string treename)
		option(category: "Workspace/Tree/Filter and Select")
	{
		return find_or_throw(treename).get_filter_shown();
	}

	void select(string treename, integer index, logical only_shown = false)
		option(category: "Workspace/Tree/Filter and Select")
	{
		find_or_throw(treename).select(index, only_shown);
	}

	void deselect(string treename, integer index, logical only_shown = false)
		option(category: "Workspace/Tree/Filter and Select")
	{
		find_or_throw(treename).deselect(index, only_shown);
	}

	void select_by_filter(string treename, FilterSelectMethod method)
		option(category: "Workspace/Tree/Filter and Select")
	{
		find_or_throw(treename).select_by_filter(method);
	}

	void add_data(string treename, string id, vector(string) data)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).add_data(id, data);
	}

	string get_data(string treename, string id, integer index)
		option(category: "Workspace/Tree/Content")
	{
		return find_or_throw(treename).get_data(id, index);
	}

	void set_data(string treename, string id, integer index, string option(nullable) data)
		option(category: "Workspace/Tree/Content")
	{
		find_or_throw(treename).set_data(id, index, data);
	}

	// Get a local source trigger whenever tree is updated
	void listen_to_trigger(string treename)
		option(category: "Workspace/Tree/Misc")
	{
		try {
			local_get_tree(treename);
		} catch {
			local_publish_and_get_tree(treename);
		}
	}

	void set_local_source_name(string name)
		option(category: "Workspace/Tree/Misc")
	{
		__impl__.LOCAL_SOURCE = name;
	}
} // end module tree
