/*
 * ahs_api supplemental
 *
 * This file contains things that should go into the ahs_api, but due to time
 * constraints are implemented in QLang instead. This file is however not the
 * place for "hacks", but for well-thought classes and functions that users can
 * rely on for the future.
 *
 * Everything here should be moved to ahs_api.qll, and when it is be removed
 * from this file.
 *
 * Entities in this file are not as thoroughly tested as entities in the qll
 * are. Interfaces may change, in particular some methods that are public in
 * QLang will be private in C++. If done correctly here, these methods start
 * with an underscore.
 */

module ahs_api {

module __impl__ {

string str_join(vector(string) option(nullable) parts, string option(nullable) delimiter = null<string>)
{
	integer n = v_size(parts);
	if (n >= 2 && !null(delimiter) && delimiter != "") {
		vector(string) join_parts[2*n-1];
		join_parts[0] = parts[0];
		for (integer i = 1; i < n; ++i) {
			join_parts[i*2-1] = delimiter;
			join_parts[i*2] = parts[i];
		}
		return strcat(join_parts);
	} else if (null(parts)) {
		return "";
	} else {
		return strcat(parts);
	}
}

string n_digit_string(number x, integer n_digits)
{
	number threshold = pow(10, n_digits);
	if (x < 0 || x >= threshold) {
		throw (E_UNSPECIFIC, strcat(['Cannot turn ', str(x), ' into ', str(n_digits), ' digits.']));
	}

	x = floor(x); // lose the decimals

	// Special case x = 0, algorithm below won't work
	if (x == 0) {
		string s = '';
		for (integer i = 0; i < n_digits; i++) {
			s = strcat(s, '0');
		}
		return s;
	}

	string new_str = str(x);

	// pad with zeroes
	number scaled_x = x * 10;
	while (scaled_x < threshold) {
		new_str = strcat(['0', new_str]);
		scaled_x = scaled_x * 10;
	}

	return new_str;
}

void v_null_partition(vector(string) v, out vector(integer) nonnull_indices, out vector(integer) null_indices)
{
	resize(nonnull_indices, 0);
	resize(null_indices, 0);
	integer v_sz = v_size(v);
	for (integer i = 0; i < v_sz; ++i) {
		if (null(v[i])) {
			push_back(null_indices, i);
		} else {
			push_back(nonnull_indices, i);
		}
	}
}

vector(integer) v_nulls(vector(string) v)
{
	vector(integer) non_nulls;
	vector(integer) nulls;
	v_null_partition(v, non_nulls, nulls);
	return nulls;
}

vector(integer) v_non_nulls(vector(string) v)
{
	vector(integer) non_nulls;
	vector(integer) nulls;
	v_null_partition(v, non_nulls, nulls);
	return non_nulls;
}

} // End module __impl__



class instrument option(category: "AHS/ahs_api/Supplemental") {
public:
	instrument(string name);
	instrument(integer id, string name);
	integer get_id();
	void _set_id(integer id);
	string get_name();
	string show();

	logical has(string source);
	ahs_api.instrument_item get(string source);
	vector(ahs_api.instrument_item) get();

	void _add_from_db(ahs_api.instrument_item ii);
	void add(ahs_api.instrument_item ii);
//	void remove(ahs_api.instrument_item ii); // not implemented yet
	vector(ahs_api.instrument_item) _get_added();
	vector(ahs_api.instrument_item) _get_removed();
	void _mark_written();

private:
	integer id_;
	string name_;
	map_str_obj<ahs_api.instrument_item> instrument_items_;
	vector(ahs_api.instrument_item) added_instrument_items_;
	vector(ahs_api.instrument_item) removed_instrument_items_;

	void _init(integer id, string name);
};

instrument.instrument(string name)
{
	_init(-1, name);
}

instrument.instrument(integer id, string name)
{
	_init(id, name);
}

void instrument._init(integer id, string name)
{
	id_ = id;
	name_ = name;
	instrument_items_ = new map_str_obj<ahs_api.instrument_item>();
}

integer instrument.get_id() = id_;
void instrument._set_id(integer id) { id_ = id; }
string instrument.get_name() = name_;

string instrument.show()
{
	vector(string) elems = [name_, " (", str(id_), ") ["];

	vector(string) ii_sources = instrument_items_.get_keys();
	integer ii_sources_sz = v_size(ii_sources);
	for (integer i = 0; i < ii_sources_sz; ++i) {
		push_back(
			elems,
			[ii_sources[i], ": ", instrument_items_.find(ii_sources[i]).get_name(),
			 " (", str(instrument_items_.find(ii_sources[i]).get_id()), "); "]);
	}
	push_back(elems, "]");

	return strcat(elems);
}

logical instrument.has(string source) = !null(instrument_items_.find(source));

ahs_api.instrument_item instrument.get(string source)
{
	ahs_api.instrument_item ii = instrument_items_.find(source);
	if (null(ii)) {
		throw (E_UNSPECIFIC, strcat(["Instrument has no item of source ", source]));
	}
	return ii;
}

vector(ahs_api.instrument_item) instrument.get()
{
	return instrument_items_.find(instrument_items_.get_keys());
}

void instrument._add_from_db(ahs_api.instrument_item ii)
{
	if (this.has(ii.get_source())) {
		throw (E_UNSPECIFIC, strcat(["Instrument already has instrument_item of source ", ii.get_source()]));
	}
	instrument_items_.add(ii.get_source(), ii);
}

void instrument.add(ahs_api.instrument_item ii)
{
	_add_from_db(ii);
	push_back(added_instrument_items_, ii);
}

/*
void instrument.remove(ahs_api.instrument_item ii)
{
	if (!this.has(ii.get_source())) {
		throw (E_UNSPECIFIC, strcat(["Instrument has no instrument_item of source ", ii.get_source()]));
	}
	instrument_items_.remove(ii.get_source());
	push_back(removed_instrument_items_, ii);
}
*/

vector(ahs_api.instrument_item) instrument._get_added() = clone_vector(added_instrument_items_);
vector(ahs_api.instrument_item) instrument._get_removed() = clone_vector(removed_instrument_items_);

void instrument._mark_written()
{
	resize(added_instrument_items_, 0);
	resize(removed_instrument_items_, 0);
}



class instrument_item_set option(category: "AHS/ahs_api/Supplemental") {
public:
	instrument_item_set(string name);
	instrument_item_set(integer id, string name);
	integer get_id();
	void _set_id(integer id);
	string get_name();
	string show(logical summary = false);

	vector(ahs_api.instrument_item) get();

	void _add_from_db(ahs_api.instrument_item ii);
	void add(ahs_api.instrument_item ii);
	void remove(ahs_api.instrument_item ii);

	void write(ahs_api.credentials creds);

private:
	integer id_;
	string name_;
	map_int_obj<ahs_api.instrument_item> db_instrument_items_;
	vector(ahs_api.instrument_item) added_instrument_items_;
	vector(ahs_api.instrument_item) removed_instrument_items_;

	vector(integer) db_instrument_item_ids_;

	void _init(integer id, string name);

	void _write_set(odbc.connection conn);
	void _remove_instrument_items(odbc.connection conn);
	void _add_instrument_items(odbc.connection conn);
};

instrument_item_set.instrument_item_set(string name)
{
	_init(-1, name);
}

instrument_item_set.instrument_item_set(integer id, string name)
{
	_init(id, name);
}

void instrument_item_set._init(integer id, string name)
{
	id_ = id;
	name_ = name;
	db_instrument_items_ = new map_int_obj<ahs_api.instrument_item>();
}

integer instrument_item_set.get_id() = id_;
void instrument_item_set._set_id(integer id) { id_ = id; }
string instrument_item_set.get_name() = name_;

string instrument_item_set.show(logical summary)
{

	if (summary) {
		return strcat(
			[name_, " (", str(id_), "): ",
			 str(v_size(db_instrument_items_.get_keys())), " from db, ",
			 str(v_size(added_instrument_items_)), " added, ",
			 str(v_size(removed_instrument_items_)), " removed"]);
	}

	vector(string) elems = [name_, " (", str(id_), ") ["];

	integer added_instrument_items__sz = v_size(added_instrument_items_);
	if (added_instrument_items__sz > 0) {
		push_back(elems, str(added_instrument_items__sz));
		push_back(elems, "+: ");
		vector(string) added_reprs[added_instrument_items__sz];
		for (integer i = 0; i < added_instrument_items__sz; ++i) {
			ahs_api.instrument_item ii = added_instrument_items_[i];
			added_reprs[i] = strcat([ii.get_source(), ":", ii.get_name(), " (", str(ii.get_id()), ")"]);
		}
		push_back(elems, __impl__.str_join(added_reprs, ";"));
		push_back(elems, "] [");
	}

	integer removed_instrument_items__sz = v_size(removed_instrument_items_);
	if (removed_instrument_items__sz > 0) {
		push_back(elems, str(removed_instrument_items__sz));
		push_back(elems, "-: ");
		vector(string) removed_reprs[removed_instrument_items__sz];
		for (integer i = 0; i < removed_instrument_items__sz; ++i) {
			ahs_api.instrument_item ii = removed_instrument_items_[i];
			removed_reprs[i] = strcat([ii.get_source(), ":", ii.get_name(), " (", str(ii.get_id()), ")"]);
		}
		push_back(elems, __impl__.str_join(removed_reprs, ";"));
		push_back(elems, "] [");
	}

	vector(ahs_api.instrument_item) db_iis = db_instrument_items_.find(db_instrument_items_.get_keys());
	integer db_iis_sz = v_size(db_iis);
	vector(string) removed_reprs[db_iis_sz];
	for (integer i = 0; i < db_iis_sz; ++i) {
		ahs_api.instrument_item ii = db_iis[i];
		removed_reprs[i] = strcat([ii.get_source(), ":", ii.get_name(), " (", str(ii.get_id()), ")"]);
	}
	push_back(elems, __impl__.str_join(removed_reprs, ";"));
	push_back(elems, "]");

	return strcat(elems);
}

vector(ahs_api.instrument_item) instrument_item_set.get()
{
	return concat(
		db_instrument_items_.find(db_instrument_items_.get_keys()),
		added_instrument_items_);
}

void instrument_item_set._add_from_db(ahs_api.instrument_item ii)
{
	if (ii.get_id() < 0) {
		throw (E_UNSPECIFIC, strcat(["Invalid id ", str(ii.get_id()), " for instrument_item ", ii.get_name()]));
	}
	db_instrument_items_.add(ii.get_id(), ii);
}

void instrument_item_set.add(ahs_api.instrument_item ii)
{
	push_back(added_instrument_items_, ii);
}

void instrument_item_set.remove(ahs_api.instrument_item ii)
{
	integer id = ii.get_id();
	if (ii.get_id() < 0) {
		throw (E_UNSPECIFIC, strcat(["Cannot remove instrument_item without valid id (", str(ii.get_id()), ", instrument_item ", ii.get_name(), ")"]));
	}
	if (null(db_instrument_items_.find(id))) {
		throw (E_UNSPECIFIC, strcat(["Instrument item ", ii.get_name(), " with id ", str(id), " is not in the set"]));
	}
	db_instrument_items_.remove(id);
	push_back(removed_instrument_items_, ii);
}

void instrument_item_set.write(ahs_api.credentials creds)
{
	odbc.connection conn = new odbc.connection(creds.get_db_dsn());

	ahs_api.write_instrument_items(db_instrument_items_.find(db_instrument_items_.get_keys()), creds);
	ahs_api.write_instrument_items(added_instrument_items_, creds);

	odbc.transaction trans = new odbc.transaction(conn);
	_write_set(conn);
	_remove_instrument_items(conn);
	_add_instrument_items(conn);
	trans.commit();

	db_instrument_items_.add(added_instrument_items_.get_id(), added_instrument_items_);
	resize(added_instrument_items_, 0);
	resize(removed_instrument_items_, 0);
}

void instrument_item_set._write_set(odbc.connection conn)
{
	if (id_ < 0) {
		odbc.statement stmt = new odbc.statement(
			conn,
			" DECLARE @IdOut TABLE ( id INT ) "
			" INSERT ahs2_set (set_type, name) "
			" OUTPUT inserted.id INTO @IdOut "
			" VALUES (?, ?) "
			" SELECT id FROM @IdOut ");
		stmt.set_param(0, 'INSTRUMENT_ITEM');
		stmt.set_param(1, name_);
		stmt.execute();
		stmt.fetch();
		id_ = stmt.get_int(0);
	}
}

void instrument_item_set._add_instrument_items(odbc.connection conn)
{
	integer added_instrument_items__sz = v_size(added_instrument_items_);
	if (added_instrument_items__sz == 0) {
		return;
	}

	date d = today();

	{
		// This is to remove any blocking rows where from_date = to_date
		odbc.statement stmt = new odbc.statement(
			conn,
			" DELETE ahs2_set_instrument_item WHERE "
			" set_id = ? AND instrument_item_id = ? AND from_date = ? AND to_date = ? ",
			added_instrument_items__sz);

		for (integer i = 0; i < added_instrument_items__sz; ++i) {
			integer ii_id = added_instrument_items_[i].get_id();
			stmt.set_param_v(0, i, id_);
			stmt.set_param_v(1, i, ii_id);
			stmt.set_param_v(2, i, d);
			stmt.set_param_v(3, i, d);
		}

		stmt.execute();
	}

	{
		odbc.statement stmt = new odbc.statement(
			conn,
			" INSERT ahs2_set_instrument_item (set_id, instrument_item_id, from_date) "
			" VALUES (?, ?, ?) ",
			added_instrument_items__sz);

		for (integer i = 0; i < added_instrument_items__sz; ++i) {
			integer ii_id = added_instrument_items_[i].get_id();
			stmt.set_param_v(0, i, id_);
			stmt.set_param_v(1, i, ii_id);
			stmt.set_param_v(2, i, d);
		}

		stmt.execute();
	}
}

void instrument_item_set._remove_instrument_items(odbc.connection conn)
{
	integer removed_instrument_items__sz = v_size(removed_instrument_items_);
	if (removed_instrument_items__sz == 0) {
		return;
	}
	odbc.statement stmt = new odbc.statement(
		conn,
		" UPDATE ahs2_set_instrument_item SET to_date = ? "
		" WHERE set_id = ? AND instrument_item_id = ? AND to_date IS NULL ",
		removed_instrument_items__sz);

	date d = today();

	for (integer i = 0; i < removed_instrument_items__sz; ++i) {
		integer ii_id = removed_instrument_items_[i].get_id();
		stmt.set_param_v(0, i, d);
		stmt.set_param_v(1, i, id_);
		stmt.set_param_v(2, i, ii_id);
	}

	stmt.execute();
}


module __impl__ {

object intvec {
	vector(integer) v;
};

string create_temp_table_name()
{
	timestamp t = now();
	date d = date(t);
	return strcat(
		["##ahs_api_temp_",
		 str(year(d)),
		 __impl__.n_digit_string(month(d), 2),
		 __impl__.n_digit_string(day(d), 2),
		 __impl__.n_digit_string(hour(t), 2),
		 __impl__.n_digit_string(minute(t), 2),
		 __impl__.n_digit_string(second(t), 2),
		 __impl__.n_digit_string(millisecond(t), 3),
		 "_",
		 str(get_process_id())]);
}

string create_temp_table_int(odbc.connection conn, vector(integer) vals)
{
	string table_name = create_temp_table_name();

	odbc.statement create_stmt =
		new odbc.statement(conn, strcat(["create table ", table_name, " (val int primary key)"]));
	create_stmt.execute();

	integer vals_sz = v_size(vals);
	if (vals_sz > 0) {
		odbc.statement stmt = new odbc.statement(
			conn, strcat(["insert into ", table_name, " (val) values (?)"]), vals_sz);

		for (integer i = 0; i < vals_sz; ++i) {
			stmt.set_param_v(0, i, vals[i]);
		}

		stmt.execute();
	}

	return table_name;
}

string create_temp_table_str(odbc.connection conn, vector(string) vals)
{
	string table_name = create_temp_table_name();

	vals = vals[__impl__.v_non_nulls(vals)];

	integer vals_sz = v_size(vals);
	integer max_len = vals_sz == 0 ? 0 : integer(v_max(strlen(vals)));
	odbc.statement create_stmt = new odbc.statement(
		conn,
		strcat(
			["create table ", table_name,
			 " (val varchar(", str(max_len+1), ") primary key)"]));
	create_stmt.execute();

	if (vals_sz > 0) {
		odbc.statement stmt = new odbc.statement(
			conn, strcat(["insert into ", table_name, " (val) values (?)"]), vals_sz);

		for (integer i = 0; i < vals_sz; ++i) {
			stmt.set_param_v(0, i, vals[i]);
		}

		stmt.execute();
	}

	return table_name;
}

string create_temp_table_int_str(odbc.connection conn, vector(integer) intvals, vector(string) strvals)
{
	string table_name = create_temp_table_name();

	if (v_size(__impl__.v_nulls(strvals)) > 0) {
		throw (E_UNSPECIFIC, "Null string is not allowed");
	}

	integer strvals_sz = v_size(strvals);
	if (v_size(intvals) != strvals_sz) {
		throw (E_UNSPECIFIC, "Vector size mismatch");
	}
	integer max_len = strvals_sz == 0 ? 0 : integer(v_max(strlen(strvals)));

	odbc.statement create_stmt = new odbc.statement(
		conn,
		strcat(
			["create table ", table_name,
			 " (intval int primary key, strval varchar(", str(max_len+1), ") primary key)"]));
	create_stmt.execute();

	if (strvals_sz > 0) {
		odbc.statement stmt = new odbc.statement(
			conn, strcat(["insert into ", table_name, " (intval, strval) values (?,?)"]), strvals_sz);

		for (integer i = 0; i < strvals_sz; ++i) {
			stmt.set_param_v(0, i, intvals[i]);
			stmt.set_param_v(1, i, strvals[i]);
		}

		stmt.execute();
	}

	return table_name;
}

void drop_table(odbc.connection conn, string table_name)
{
	try {
		odbc.statement stmt = new odbc.statement(conn, strcat("drop table ", table_name));
		stmt.execute();
	} catch {
	}
}

vector(instrument) load_instruments(odbc.statement stmt, ahs_api.credentials creds)
{
	// Assume SELECT <instrument name> <instrument id> <instrument_item id>

	map_int_obj<instrument> instrs = new map_int_obj<instrument>();
	map_int_obj<intvec> instr_to_instr_item_ids = new map_int_obj<intvec>();
	set_int all_ii_ids_set = set_int();

	while (stmt.fetch()) {
		string name = stmt.get_str(0);
		integer id = stmt.get_int(1);
		instrument instr = instrs.find(id);
		if (null(instr)) {
			instr = new instrument(id, name);
			instrs.add(id, instr);
		}
		if (!stmt.is_null_int(2)) {
			intvec v_i = instr_to_instr_item_ids.find(id);
			if (null(v_i)) {
				v_i = new intvec;
				instr_to_instr_item_ids.add(id, v_i);
			}
			push_back(v_i.v, stmt.get_int(2));
			all_ii_ids_set.add(stmt.get_int(2));
		}
	}

	vector(integer) all_ii_ids = all_ii_ids_set.get_content();

	vector(ahs_api.instrument_item) instr_items =
		ahs_api.fetch_instrument_items_by_id(all_ii_ids, creds);

	map_int_obj<ahs_api.instrument_item> instr_item_map = new map_int_obj<ahs_api.instrument_item>;
	instr_item_map.add(instr_items.get_id(), instr_items);


	vector(integer) instr_ids = instrs.get_keys();
	integer instr_ids_sz = v_size(instr_ids);
	vector(instrument) ret[instr_ids_sz];
	for (integer i = 0; i < instr_ids_sz; ++i) {
		instrument instr = instrs.find(instr_ids[i]);
		vector(integer) instr_item_ids[0];
		intvec v_i = instr_to_instr_item_ids.find(instr.get_id());
		if (!null(v_i)) {
			instr_item_ids = clone_vector(v_i.v);
		}
		integer instr_item_ids_sz = v_size(instr_item_ids);
		for (integer j = 0; j < instr_item_ids_sz; ++j) {
			instr._add_from_db(instr_item_map.find(instr_item_ids[j]));
		}
		ret[i] = instr;
	}

	return ret;
}

void create_instrs(odbc.connection conn, vector(instrument) instrs)
{
	vector(instrument) new_instrs[0];
	integer instrs_sz = v_size(instrs);
	for (integer i = 0; i < instrs_sz; ++i) {
		if (instrs[i].get_id() < 0) {
			push_back(new_instrs, instrs[i]);
		}
	}

	if (v_size(new_instrs) == 0) {
		return;
	}

	string table_name = create_temp_table_str(conn, new_instrs.get_name());

	string sql = strcat(
		[" declare @IdentityOutput table ( ID int, NAME varchar(50) ) ",
		 " insert ahs2_instrument (name) ",
		 " output inserted.id, inserted.name into @IdentityOutput ",
		 " select val from ", table_name,
		 " select ID, NAME from @IdentityOutput "]);

	odbc.statement stmt = new odbc.statement(conn, sql);
	stmt.execute();

	map_str_int name_2_id = map_str_int();
	while (stmt.fetch()) {
		name_2_id.add(stmt.get_str(1), stmt.get_int(0));
	}

	drop_table(conn, table_name);

	integer new_instrs_sz = v_size(new_instrs);
	for (integer i = 0; i < new_instrs_sz; ++i) {
		integer id = name_2_id.find(new_instrs[i].get_name());
		new_instrs[i]._set_id(id);
	}
}

string idstr(string source, string name) = strcat([source, "::", name]);
string idstr(ahs_api.instrument_item ii) = idstr(ii.get_source(), ii.get_name());

void create_instr_items(vector(instrument) instrs, ahs_api.credentials creds)
{
	map_str_obj<ahs_api.instrument_item> new_iis_map = new map_str_obj<ahs_api.instrument_item>();
	integer instrs_sz = v_size(instrs);
	for (integer i = 0; i < instrs_sz; ++i) {
		vector(ahs_api.instrument_item) iis = instrs[i].get();
		integer iis_sz = v_size(iis);
		for (integer j = 0; j < iis_sz; ++j) {
			ahs_api.instrument_item ii = iis[j];
			if (ii.get_id() < 0) {
				new_iis_map.add(idstr(ii), ii);
			}
		}

	}

	vector(ahs_api.instrument_item) new_iis = new_iis_map.find(new_iis_map.get_keys());
	ahs_api.write_instrument_items(new_iis, creds);
}

void remove_instr_items(odbc.connection conn, vector(instrument) instrs)
{
	integer instrs_sz = v_size(instrs);
	for (integer i = 0; i < instrs_sz; ++i) {
		instrument instr = instrs[i];
		vector(ahs_api.instrument_item) removed = instr._get_removed();
		integer removed_sz = v_size(removed);
		for (integer i = 0; i < removed_sz; ++i) {
			throw (E_UNSPECIFIC, "remove_instr_items not implemented yet");
		}
	}
}

void add_instr_items(odbc.connection conn, vector(instrument) instrs)
{
	string sql =
		" insert ahs2_instrument_instrument_item (instrument_id, instrument_item_id, from_date) "
		" values (?, ? ,?)";

	vector(integer) instr_ids[0];
	vector(integer) instr_item_ids[0];

	integer instrs_sz = v_size(instrs);
	for (integer i = 0; i < instrs_sz; ++i) {
		instrument instr = instrs[i];
		integer instr_id = instr.get_id();
		if (instr_id < 0) {
			throw (E_UNSPECIFIC, strcat(["Instrument ", instr.show(), " has invalid id"]));
		}
		vector(ahs_api.instrument_item) iis = instr._get_added();
		integer iis_sz = v_size(iis);
		for (integer j = 0; j < iis_sz; ++j) {
			ahs_api.instrument_item ii = iis[j];
			integer ii_id = ii.get_id();
			if (ii_id < 0) {
				throw (E_UNSPECIFIC, strcat(["Instrument item ", ii.get_name(), " (", ii.get_source(), ") has invalid id"]));
			}
			push_back(instr_ids, instr_id);
			push_back(instr_item_ids, ii_id);
		}
	}

	integer n = v_size(instr_ids);

	if (n == 0) {
		return;
	}

	date from_date = today();

	odbc.statement stmt = new odbc.statement(conn, sql, n);

	for (integer i = 0; i < n; ++i) {
		stmt.set_param_v(0, i, instr_ids[i]);
		stmt.set_param_v(1, i, instr_item_ids[i]);
		stmt.set_param_v(2, i, from_date);
	}

	stmt.execute();
}

} // End module __impl__


vector(instrument) fetch_instruments_by_id(vector(integer) ids, ahs_api.credentials creds)
	option(category: "AHS/ahs_api/Supplemental")
{
	set_int ids_set = set_int();
	ids_set.add(ids);
	vector(integer) unique_ids = ids_set.get_content();

	string sql =
		"SELECT i.name, i.id, ii.instrument_item_id "
		"FROM ahs2_instrument i "
		"LEFT OUTER JOIN ahs2_instrument_instrument_item ii ON "
		"	ii.instrument_id = i.id AND ii.from_date < GETDATE() AND ii.to_date IS NULL "
		"WHERE i.id IN (SELECT val FROM ahs_unique_strlist_to_table(?, ?)) ";

	odbc.connection conn = new odbc.connection(creds.get_db_dsn());
	odbc.statement stmt = new odbc.statement(conn, sql);
	stmt.set_param(0, __impl__.str_join(str(unique_ids), "|"));
	stmt.set_param(1, "|");
	stmt.execute();

	vector(instrument) instrs = __impl__.load_instruments(stmt, creds);
	return instrs;
}

vector(instrument) fetch_instruments_by_name(vector(string) names, ahs_api.credentials creds)
	option(category: "AHS/ahs_api/Supplemental")
{
	set_str names_set = set_str();
	names_set.add(names);
	vector(string) unique_names = names_set.get_content();

	string sql =
		"SELECT i.name, i.id, ii.instrument_item_id "
		"FROM ahs2_instrument i "
		"LEFT OUTER JOIN ahs2_instrument_instrument_item ii ON "
		"	ii.instrument_id = i.id AND ii.from_date < GETDATE() AND ii.to_date IS NULL "
		"WHERE i.name IN (SELECT val FROM ahs_unique_strlist_to_table(?, ?)) ";

	odbc.connection conn = new odbc.connection(creds.get_db_dsn());
	odbc.statement stmt = new odbc.statement(conn, sql);
	stmt.set_param(0, __impl__.str_join(unique_names, "|"));
	stmt.set_param(1, "|");
	stmt.execute();

	vector(instrument) instrs = __impl__.load_instruments(stmt, creds);
	return instrs;
}

void write_instruments(vector(instrument) instrs, ahs_api.credentials creds)
	option(category: "AHS/ahs_api/Supplemental")
{
	odbc.connection conn = new odbc.connection(creds.get_db_dsn());
	odbc.transaction trans = new odbc.transaction(conn);
	__impl__.create_instrs(conn, instrs);
	__impl__.create_instr_items(instrs, creds);
	__impl__.remove_instr_items(conn, instrs);
	__impl__.add_instr_items(conn, instrs);
	trans.commit();

	instrs._mark_written();
}

instrument_item_set fetch_instrument_item_set(string name, ahs_api.credentials creds)
	option(category: "AHS/ahs_api/Supplemental")
{
	odbc.connection conn = new odbc.connection(creds.get_db_dsn());

	instrument_item_set set;
	{
		odbc.statement stmt = new odbc.statement(
			conn, " SELECT id, name FROM ahs2_set WHERE name = ? AND set_type = 'INSTRUMENT_ITEM' ");
		stmt.set_param(0, name);
		stmt.execute();

		if (stmt.fetch()) {
			set = new instrument_item_set(stmt.get_int(0), stmt.get_str(1));
			if (stmt.fetch()) {
				throw (E_UNSPECIFIC, strcat(["Set name '", name, "' is not unique"]));
			}
		} else {
			throw (E_UNSPECIFIC, strcat(["No instrument_item set '", name, "' exists"]));
		}
	}

	vector(integer) ii_ids[0];
	{
		odbc.statement stmt = new odbc.statement(
			conn,
			" SELECT instrument_item_id FROM ahs2_set_instrument_item "
			" WHERE set_id = ? "
			" AND (from_date IS NULL OR from_date <= GETDATE()) "
			" AND (to_date IS NULL OR to_date > GETDATE()) ");
		stmt.set_param(0, set.get_id());
		stmt.execute();

		while (stmt.fetch()) {
			push_back(ii_ids, stmt.get_int(0));
		}
	}

	vector(ahs_api.instrument_item) iis = ahs_api.fetch_instrument_items_by_id(ii_ids, creds);

	integer iis_sz = v_size(iis);
	for (integer i = 0; i < iis_sz; ++i) {
		set._add_from_db(iis[i]);
	}

	return set;
}

} // End module ahs_api
