/*	
	Basic test functions 
	Developed by Algorithmica Research, Magnus Nyström
	

	------module dual_boot_test---------------
		 
		Basic help functions for testing/evaluating the "dual" bootstrap framework
				
		function browser information related to dual bootstrap (no help yet!!):
		1.the core bootstrap functions can be viewed under "Yield Curve"|"Bootstrap" 
		2.the disc_func and fwd_func manipulation functions can be viewed under "Yield Curve"|"Discount Function Manipulation"
		3.the new fwd_func object is located under "Yield Curve"
		4.the dual bootstrap classes and wrappers are located under "Yield Curve"|"Dual Bootstrap"  								
		
		
		Generally, creating an (ois) discount function (disc_func) or a forward function (fwd_func) is done is several steps. We create a model object by 
		starting with an object populated with default values (configurable). The defaults can be overriden by using apropriate 'set' functions. 
		Then a curve object is created. The curve object performs several tasks eg. blends together several segments, adds synthetic instruments, etc.
		When these two objects are created it is possible to calculate a disc_func or fwd_func. 
			

		The create functions below for disc_func/fwd_func have many parameters that are hardcoded since they are intended to be used for educational
		purposes.
		
	------------------------------------


	
			
*/
option(null: hard);

module CORE_TEST	
{
	module curve_util
	{
		//-------------------------------------------------------
		//  incl_excl_helper
		//-------------------------------------------------------
		integer rgb_out(){
			return rgb(247, 2, 2); 
		}
		integer rgb_grey(){
			return rgb(228, 228, 228); 
		}
		void incl_excl_helper(vector(date) mat,
							  vector(string) name,
							  vector(date) mat_all,
							  vector(string) name_all,
							  out vector(text_rgb) tr)
		{
			integer s_sh = v_size(mat);
			resize(tr, s_sh);
			tr = text_rgb(name,rgb(0,0,0),rgb_out(),rgb(0,0,0),rgb_out(),strcat("excl: ",str(mat)));
				   
			vector(integer) v_f = v_find( name,name_all);
			for(integer i = 0; i < v_size(v_f); i++){
				tr[v_f[i]] = text_rgb(name[v_f[i]],rgb(0,0,0),rgb_grey(),rgb(0,0,0),rgb_grey(),strcat("incl: ",str(mat[v_f[i]])));
			}
		}
		//-------------------------------------------------------
		//  format_crv_table
		//-------------------------------------------------------
		void format_crv_table(	curve option(nullable) crv_sh,
								curve option(nullable) crv_mi,
								curve option(nullable) crv_lo,
								curve blend_crv,
								out vector(text_rgb) crv_sh_name,
								out vector(text_rgb) crv_mi_name,
								out vector(text_rgb) crv_lo_name)
		{
			resize(crv_sh_name,0);
			resize(crv_mi_name,0);
			resize(crv_lo_name,0);
			vector(instrument) vibl = blend_crv.instruments();
			if(!null(crv_sh)) {
				vector(instrument) vish = crv_sh.instruments();
				incl_excl_helper(vish.maturity(),vish.name(), vibl.maturity(),vibl.name(),crv_sh_name);
			}
			if(!null(crv_mi)) {
				vector(instrument) vimi = crv_mi.instruments();
				incl_excl_helper(vimi.maturity(),vimi.name(), vibl.maturity(),vibl.name(),crv_mi_name);
			}
			if(!null(crv_lo)) {
				vector(instrument) vilo = crv_lo.instruments();
				incl_excl_helper(vilo.maturity(),vilo.name(), vibl.maturity(),vibl.name(),crv_lo_name);
			}
		}
			
		//-------------------------------------------------------
		//  init_curve
		//-------------------------------------------------------
		void init_curve(out curve 		crv,
						out vector(string) crv_name_v,
						out vector(number) crv_yld_v,
						out vector(number) user_yld_v,
						out vector(number) user_sprd_v,
						out curve 		crv_ex_sprd)
		{
			crv_ex_sprd = null;
			
			vector(instrument) vi = null(crv) ? null:crv.instruments() ;//get the instruments
			integer crv_size = v_size(vi);

			vector(instrument) viex[crv_size];
				
			if(v_size(user_sprd_v) != crv_size){
				resize(user_sprd_v, crv_size);
				user_sprd_v = null<number>;  			
			}
			
			if(v_size(user_yld_v) != crv_size){
				resize(user_yld_v, crv_size);
				user_yld_v = null<number>;  //if the user_yld column is out of synch with curve we set all quotes to null
											//(to be sure we don't have a user yld on the wrong instrument)
			}

			error_info ee = new error_info(true,true);
			error_info ees = new error_info(true,false);
			
			if(crv_size>0) {
				logical upd = false;
				logical upd_sprd = false;
							
				for(integer i = 0; i < crv_size; i++){
					if(!null(user_yld_v[i])) {
					   number y = user_yld_v[i]/100;
					   viex[i] 	= vi[i].set_yield(y,ee);
					   
					   if(!null(user_sprd_v[i]) && user_sprd_v[i] != 0){
						   vi[i] = vi[i].set_yield(y + user_sprd_v[i]/10000,ee);
						   upd_sprd = true;
					   }
					   else {
						   vi[i] = vi[i].set_yield(y,ee);
					   }
					   upd = true;
					}
					else if(!null(user_sprd_v[i]) && user_sprd_v[i] != 0) {
						number y 	= vi[i].yield(false,ees);
						if(!null(y)){
							viex[i] 	= vi[i].set_yield(y,ee);
							vi[i] 		= vi[i].set_yield(y + user_sprd_v[i]/10000,ee);
							upd 		= true;
							upd_sprd 	= true;
						}
					}
					else {
						number y 	= vi[i].yield(false,ees);
						if(!null(y))
							viex[i] 	= vi[i].set_yield(y,ee);
					}
				}

				if(upd)
					crv = curve(vi, crv.name());
				if(upd_sprd)
					crv_ex_sprd = curve(viex, crv.name());
			}
				
			if(crv_size>0) {
				crv_name_v = vi.name();
				crv_yld_v = vi.yield(false,ees)*100;		
			}
			else {
				resize(crv_name_v,0);
				resize(crv_yld_v,0);
			}
		}
	}
	
	module dual_boot_test
	{
		//-------------------------------------------------------
		// create_disc_func (generic)
		// shows an example how to utilize some of the built-in functions to create a blended curve and calculate a disc_func		
		//-------------------------------------------------------
		disc_func create_disc_func(	date 					d,
									string option(nullable) z_model,
									string option(nullable) ip_model,
									string option(nullable) ip_rt,	
									curve option(nullable)	disc_sh_crv,//eg. ois,depo 
									curve option(nullable)  disc_mi1_crv,//eg. fra/sps/fut/onfut
									curve option(nullable)  disc_mi2_crv,//eg. fra/sps/fut/onfut
									curve option(nullable)	disc_crv, //eg. ois,swaps
									curve 	option(nullable) extrap_crv,//curve that can be used for extrapolation purpose
									curve option(nullable) 	turn_crv,
									out date  				disc_crv_end,
									out swap_curve_disc_ext disc_curve,
									vector(date) option(nullable) cb_dates = null<vector(date)>)
		{	
			/*--------------------------------------
			Step 1 	create the discount curve input with or 
					without extrapolation via a secondary curve
			-----------------------------------------*/																				
			//blending parameters
			integer blend_buf_days 	= DFLT_BLEND_BUF_DAYS;			//min number of days between curve segment in the curve blending
		
			logical no_overlap_fwd	= false;		//if true, all depos with longer maturity than fra/fut settle is removed
													//if false, all depos with longer maturity than fra/fut maturity is removed													
			
			logical merge_middle	= false;		//if true all fra/fut/sps will be merged into the curve in the blending													
		 
			logical merge_crv		= false;		//if true all segments of the curve are merged together
		
			curve_prio  prio1 		= curve_prio.SHORT;	//priority 1 when blending the 3 curves 
			curve_prio  prio2 		= curve_prio.MIDDLE;		//priority 2 when blending the 3 curves 
					
			curve_blend_parm blend_parm		= curve_blend_parm(prio1,prio2,no_overlap_fwd,merge_middle,merge_crv,blend_buf_days);

			//if we have two "middle"-curves (fra_fut curves) these are blended as a pre-step to the main blending of 3 curves (short,middle,long))
			logical fwd1_fwd2_prio_1 	= true;
			logical fwd1_fwd2_merge 	= false;
			curve disc_mi_crv 			= blend_curve_fwdfwd(disc_mi1_crv,disc_mi2_crv,fwd1_fwd2_prio_1, fwd1_fwd2_merge,blend_buf_days );
							
			vector(number)  short_crv_y_spr, fra_fut_crv_y_spr,  swap_crv_y_spr;
			vector(number)  fut_cvx_adj;//currently no input of convexity adjustments for futures
			swap_curve_disc_parm disc_parm	= swap_curve_disc_parm(	disc_sh_crv,disc_mi_crv,disc_crv,blend_parm,
																	short_crv_y_spr,fra_fut_crv_y_spr,swap_crv_y_spr,fut_cvx_adj);

			

			
			//ceate the base disc curve object
			disc_curve		= disc_curve_create(d,disc_parm);

			//check that the creation of the disc_curve is ok (checking of instr types, currencies, quotes, required fixings, etc)
			string msg;
			logical disc_curve_ok = disc_curve.check(msg);
			QL_REQUIRE(disc_curve_ok,msg);	
				
			cb_dates = disc_curve.check_cb_dates(cb_dates, cb_step_type.EXACT_FIT);	//check and possibly adjust the cb date vector (w.r.t. the blended curve),
																					//cb_step_type.EXACT_FIT: --> only one cb date allowed between instruments
																					//(if more than one is input --> the last one is used and the others are removed
			vector(..on_turns) turnv = disc_curve.check_on_turns(turn_crv);			//check the turn curve, an o/n deposit curve (with spread as quote style)
			
			/*------------------------------
			Step 2 create the discount model		 
			--------------------------------*/
			
			/*---Example procedure/model for the discounting curve ---*/
		
			/*---Example models 
				--> 1. Hagan West (forward monotone convex spline (Interpolation 
									Methods for Curve Construction, Applied Mathematical Finance,2006)---
				--> 2. tension spline spread (Discount curve construction with tension splines,L.Andersen, Review of Derivatives Research,2007)---*/
			
			disc_model_parm disc_p = disc_model_parm(); 		//creates a parameter object for the disc model with all defaults (the defaults are 
																//set in dual_boot_config.ql file (all defaults can be changed by the user))
		
			//disc_p.set_model_boot(DIPT_HAGAN_WEST) ;			//we can override the default model with our choice, Hagan West	
			//disc_p.set_model_tension_w_spread();				//we can override the default model with our choice, tension spline spread
			
			if(!null(z_model))
				disc_p.set_model(z_model) ;		//optional override of the default model with our choice
			
			if(!null(ip_rt))
				disc_p.set_rt(ip_rt) ;			//optional override of the interpolator with our choice, not relevant for all models
			
			if(!null(ip_model))
				disc_p.set_ip(ip_model) ;		//optional override of the rate type with our choice, not relevant for all models
			
			if(!null(cb_dates))
				disc_p.set_cb_dates( cb_dates, cb_step_type.EXACT_FIT) ;//set the cb dates, cb_step_type.EXACT_FIT means that the underlying curve is repriced exactly
																		//Note that depending on the input dates there will also be shifts on the maturity of the instruments
																		//in order to force repricing of the curve instruments
			if(!null(turnv))
				disc_p.set_turns(turnv) ;		//set the turns via the turn curve (a o/n deposit curve), turns will create a disc_func that is shifted between
												//the dates the turn (overnight period), the model is fitted to the curve with the turn effect removed
			
			disc_z_model disc_model = init_disc_model(disc_p);	//create the disc model based on the parameter object	

			//it is possible to extrapolate the disc curve with a swap curve (constant basis) model for extrapolating the disc curve
			//create the extrapolation model; this is an optional model and used only when extending the disc curve
			disc_model_parm extrap_p;
			disc_z_model extrap_model;
			
			if(!null(extrap_crv)){
				extrap_p = disc_model_parm();				
				//extrap_p.set_model_boot(DIPT_HAGAN_WEST) ;		//Hagan West; bootstrap and interpolation-model for extending the disc curve
				//extrap_p.set_model_tension_w_spread() ;			//tension spline spread; bootstrap and interpolation-model for extending the disc curve
				if(!null(z_model))
					extrap_p.set_model(z_model) ;	//optional override of the default model with our choice
				
				if(!null(ip_rt))
					extrap_p.set_rt(ip_rt) ;		//optional override of the interpolator with our choice, not relevant for all models
				
				if(!null(ip_model))
					extrap_p.set_ip(ip_model) ;		//optional override of the rate type with our choice, not relevant for all models
				
				extrap_model = init_disc_model(extrap_p);
			}
			
			/*models for the disc curve, 
			DZ_ON_STEP: 			piecewise flat o/n rate, 
			DZ_BOOT: 				standard bootstrap versions
			DZ_MAX_SMOOTH: 			Adams/Deventer maximum smoothness model.
			DZ_MAX_SMOOTH_SPREAD: 	Adams/Deventer maximum smoothness model, optimize within a spread.
			DZ_ON_STEP_BOOT: 		standard bootstrap but with a step function (for O/N rates) up to the step_end date.
			DZ_TANGGAARD: 			max smoothness with residuals
			DZ_TENSION_W_EXACT: 	tension splines exact fit
			DZ_TENSION_W_SPREAD: 	tension splines - optimize within a spread
			DZ_TENSION_Y_EXACT: 	tension splines exact fit, y = cont.comp. spot rate
			DZ_TENSION_Y_SPREAD:	tension splines - optimize within a spread, y = cont.comp. spot rate
			DZ_TENSION_Z_EXACT: 	tension splines exact fit
			DZ_TENSION_Z_SPREAD: 	tension splines - optimize within a spread
			(tension_z and tension_w have better asymptotic behaviour for large t than tension_y i.e. avoid tension_y for extrapolation)
			DZ_NS: nelson siegel original
			also other classical yield curve models can be used but are not in the above wrapper (ex Nelson Siegel versions, see Fitting | Models)

			Note: if a cb_date vector is input the part of the curve covered by these dates will always be a step-function regardless of the model used

			It also possible to splice together two separate models with a specified cutoff in days
			(see below in Alt. step 3)
			
			*/			 						

			/*interpolation method for discounting
			1. ip_hagan_west(logical force_pos): Hagan/West forward monotone convex spline, 
				if force_pos  = true, the forward rates are enforced to be positive.
			2. ip_hermite_4th_order(args...) : 
			3. ip_hermite_akima(args...):
			4. ip_hermite_catmull_rom(args...) :
			5. ip_hermite_finite_diff (args...):
			6. ip_hermite_fritsch_butland(args...) :
			7. ip_hermite_kruger(args...):
			8. ip_hermite_monotone(args...):
			9. ip_hermite_parabolic(args...):
			10. ip_linear(args...):linear interpolation on spot rates
			11. ip_spline(args...): cubic spline
			12. ip_step(args...): step function interpolation (disc_func--> step on spot r, fwd_func--> step on discrete fwd r).*/
			
			//ip_spline(null<number>, 0, IP_HYMAN) ;	//financial spline with Hyman filter 
			//ip_spline(0, 0); 							//constant to the right and left.
			//ip_spline(null<number>, null<number>); 	//"natural spline" (second derivative = 0)
			//ip_spline(null<number>, 0); 				//"financial spline", --> horizontal rate asymptote, extrapolation ok.
																						
			/*interpolation rate type for discounting (for DZ_BOOT and DZ_STEP_BOOT)
			1. RT_CONT: Continuous compounding.
			2. RT_ON:	 Daily compounding.	
			3. RT_SIMPLE: Simple compounding.	
			4. RT_EFFECTIVE: Annual effective compounding.	
			5. RT_SEMI_ANNUAL: Semi-annual compounding.	
			6. RT_QUARTERLY: Quarterly compounding.	
			7. RT_MONTHLY: Monthly compounding.	
			8. RT_BANKDISC: Bank discount rate.
			9. RT_LOGDF: Log of the discount function							
			(for HaganWest only RT_CONT is applicable)*/

			integer alt_to_use = null(extrap_crv) ? 1 : 2;
			
			/*---------------------
			Step 3:  calculate the disc func
			-----------------------*/
			disc_func df, df_ex_turn;//df_ex_turn is the disc_func with turns removed i.e. the one that is fitted
			if(alt_to_use == 1) {
				df 				= disc_curve.disc_df(disc_model,df_ex_turn);
				disc_crv_end 	= disc_curve.curve_end();	
			}
			else if(alt_to_use == 2) {//create the extrapolated disc curve with the extrap_model	
				swap_curve_disc_ext disc_curve_extrap 	= disc_curve_create(d, disc_curve, extrap_crv, extrap_model);		
				df 										= disc_curve_extrap.disc_df(disc_model,df_ex_turn);	
				disc_crv_end							= disc_curve_extrap.curve_end();	
			}
			else if(alt_to_use == 3) {//illustration only; calc disc func with 2 models (disc_model + disc_model2) with a cutoff
				disc_model_parm disc_p2 	= disc_model_parm();
				disc_p2.set_model("Boot") ;	
				disc_z_model disc_model2 	= init_disc_model(disc_p2);
				integer cutoff_days			= 300;
				df							= disc_curve.disc_df(disc_model,disc_model2,cutoff_days);
				disc_crv_end				= disc_curve.curve_end();	
			}
			else if(alt_to_use == 4) {//illustration only; (df_base must be an input and is not available in this example)
				//calc a disc_func with a given base disc_func and where a spread is fitted on top with a model (maintaining an exact fit)
				//the interpolation model for the spread is here linear on continous rates
				disc_func df_base 			= disc_func_interp([1],[1],ip_linear(),rate_type.RT_CONT);//a fake 0 rate disc_func
				disc_model_parm disc_p_sprd = disc_model_parm();
				disc_p_sprd.set_model_boot(disc_ip_type.DIPT_LINEAR) ;
				disc_p_sprd.set_rt(rate_type.RT_CONT);
				disc_z_model disc_model_sprd = init_disc_model(disc_p_sprd);
				df 							= disc_curve.disc_df(df_base, disc_model_sprd);
				disc_crv_end 				= disc_curve.curve_end();	
			}
			else if(alt_to_use == 5) {//illustration only; a combo of alt 3 and 4
				disc_func df_base 			= disc_func_interp([1],[1],ip_linear(),rate_type.RT_CONT);//a fake 0 rate disc_func
				disc_model_parm disc_p_sprd = disc_model_parm();
				disc_p_sprd.set_model_boot(disc_ip_type.DIPT_LINEAR) ;
				disc_p_sprd.set_rt(rate_type.RT_CONT);
				disc_z_model disc_model_sprd = init_disc_model(disc_p_sprd);
				
				disc_model_parm disc_p2 	= disc_model_parm();
				disc_p2.set_model("Boot") ;	
				disc_z_model disc_model_sprd2 = init_disc_model(disc_p2);
				integer cutoff_days			= 300;
				df 							= disc_curve.disc_df(df_base,disc_model_sprd,disc_model_sprd2,cutoff_days);
			}

			QL_REQUIRE(!null(df),'invalid disc_func');		
			return df;
	
			
		}		
		//-------------------------------------------------------
		// create_disc_func (generic)
		// this version uses ois/ibor basis swap plus a swap curve on the long end --> creates a synthetic fixed leg vs o/n leg	
		//-------------------------------------------------------
		disc_func create_disc_func(	date 					d,
									string option(nullable) z_model,
									string option(nullable) ip_model,
									string option(nullable) ip_rt,	
									curve option(nullable)	disc_sh_crv,//eg. ois,depo 
									curve option(nullable)  disc_mi_crv,//eg. fra/sps/fut
									curve option(nullable)	disc_lo_crv, //eg. ois,swaps
									curve 					lib_swap_crv, //irs, eg. USD Fix vs 3M
									curve 					ois_lib_basis_crv, //ois/ibor basis swap eg. fedfunds/libor basis swap
									curve option(nullable) 	turn_crv,
									logical					extrap_ok,	//extrapolate basis spreads based on lib_swap_crv maturities							
									out date  				disc_crv_end,
									out swap_curve_disc_b1_ext disc_curve,
									vector(date) option(nullable) cb_dates = null<vector(date)>)
		{	
			/*--------------------------------------
			Step 1 	create the discount curve input
			-----------------------------------------*/																				
			//blending parameters
			integer blend_buf_days 	= DFLT_BLEND_BUF_DAYS;			
			logical no_overlap_fwd	= false;				
			logical merge_middle	= false;			 
			logical merge_crv		= false;			
			curve_prio  prio1 		= curve_prio.SHORT;		
			curve_prio  prio2 		= curve_prio.MIDDLE;	
					
			curve_blend_parm blend_parm	= curve_blend_parm(prio1,prio2,no_overlap_fwd,merge_middle,merge_crv,blend_buf_days);
			
			vector(number)  short_crv_y_spr, fra_crv_y_spr,  swap_crv_y_spr;
			vector(number)  lib_swap_crv_y_spr ,ois_lib_basis_crv_q_spr;
			logical allow_fra_swap_adj = true;
			swap_curve_b1_disc_parm disc_parm	= swap_curve_b1_disc_parm(	disc_sh_crv,disc_mi_crv,disc_lo_crv,
																			lib_swap_crv,ois_lib_basis_crv,blend_parm,allow_fra_swap_adj,
																			short_crv_y_spr,fra_crv_y_spr,swap_crv_y_spr,
																			lib_swap_crv_y_spr ,ois_lib_basis_crv_q_spr);

			disc_curve 		= disc_curve_create(d,disc_parm,extrap_ok);//use other constructor when there is a mismatch between basis curve and swap curve
			string msg;
			logical disc_curve_ok = disc_curve.check(msg);
			QL_REQUIRE(disc_curve_ok,msg);
			
			cb_dates 		= disc_curve.check_cb_dates(cb_dates, cb_step_type.EXACT_FIT);
			
			vector(..on_turns) turnv = disc_curve.check_on_turns(turn_crv);
			
			/*------------------------------
			Step 2 create the discount model		 
			--------------------------------*/		
			disc_model_parm disc_p = disc_model_parm(); 
			if(!null(z_model))
				disc_p.set_model(z_model) ;
			if(!null(ip_rt))
				disc_p.set_rt(ip_rt) ;
			if(!null(ip_model))
				disc_p.set_ip(ip_model) ;
			if(!null(cb_dates))
				disc_p.set_cb_dates( cb_dates, cb_step_type.EXACT_FIT) ;
			if(!null(turnv))
				disc_p.set_turns(turnv) ;  	
			
			disc_z_model disc_model = init_disc_model(disc_p);	
			
			/*---------------------
			Step 3 calculate disc func
			-----------------------*/			
			disc_func df_ois 	= disc_curve.disc_df(disc_model);
			disc_crv_end 		= disc_curve.curve_end();		
			
			QL_REQUIRE(!null(df_ois),'invalid df_ois');		
			return df_ois;
		}

		
		//-------------------------------------------------------
		// create_tenor_main_func (generic)
		// main tenor --> create fwd_func directly from a depo/fra/sps/swap curve
		// all instruments in the middle and long curves refer to the same tenor
		// the relevant tenor in the depo_fix_crv is extracted, and used, if it exists (all other tenors are removed)
		//-------------------------------------------------------
		fwd_func create_tenor_main_func(date 					d,
										fwd_func_tenor	option(nullable) fwd_tenor,
										string option(nullable) z_model,
										disc_func				df_ois,
										disc_func option(nullable)	df_synt_base,
										date option(nullable)	synt_end,									
										curve option(nullable)	depo_fix_crv,	//a deposit or deposit_index (fixing) curve	
										curve option(nullable)	fra_fut_crv,	//eg. fra or sps 
										curve option(nullable)	swap_crv,
										curve option(nullable)	turn_crv,
										logical 				fra_fut_prio,
										logical 				incl_synt,
										logical					model_spr,
										out fwd_func 			f_fwd_ex_turns,
										out swap_curve_fwd_ext fwd_crv,
										vector(number) option(nullable) edfut_cvx_adj = null<vector(number)> ) //used for external convexity adjustment for futures
		{
			number  first_fix = null;				//relevant only if the depo_fix_crv contains deposits
			logical  force_null_first_fix = null;	//relevant only if the depo_fix_crv is a curve with fixing instruments
													//(useful because before the fixing time the fixing instrument quote contains an incorrect quote (yesterday's?)
			
			/*------------------------------
			Step 1 create the forwarding input curve (with hardcoded parameters)
			--------------------------------*/	  								
			logical merge_fra 			= false;		//if true all fra's/sps's will be merged into the curve in the blending
			logical merge_crv			= false;		//if true all segments of the curve are merged together						
			integer blend_buf_days 		= DFLT_BLEND_BUF_DAYS;
			logical no_overlap_fwd		= false;								
			curve_prio prio1 			= curve_prio.SHORT;//priority when blending the curves
			curve_prio prio2 			= fra_fut_prio? curve_prio.MIDDLE : curve_prio.LONG;
		
			curve_blend_parm blend_parm	= curve_blend_parm(prio1,prio2,no_overlap_fwd,merge_fra,merge_crv,blend_buf_days);		
			
			vector(number) fra_fut_crv_y_spr, swap_crv_y_spr;//currently having spreads on the short end is not supported (a ToDo)
			
			swap_curve_fwd_f_parm fwd_parm;
			if(!null(depo_fix_crv) && !depo_fix_crv.empty()) {
				instrument inst = depo_fix_crv.instruments()[0];
				if(inst.instr_type() == instr_type.DEPOSIT_INDEX){//need to know if this curve contains fixing instruments or just deposits
					if(null(force_null_first_fix))
						force_null_first_fix = false;
					fwd_parm = swap_curve_fwd_f_parm(fixing_curve(depo_fix_crv), fra_fut_crv, swap_crv, blend_parm,
													 fra_fut_crv_y_spr, swap_crv_y_spr,edfut_cvx_adj,force_null_first_fix);
				}
				else {				
					fwd_parm = swap_curve_fwd_f_parm(depo_fix_crv, fra_fut_crv, swap_crv, blend_parm,
													 fra_fut_crv_y_spr, swap_crv_y_spr,edfut_cvx_adj,first_fix);
				}
			}
			else {
				fwd_parm = swap_curve_fwd_f_parm(null, fra_fut_crv, swap_crv, blend_parm,
												 fra_fut_crv_y_spr, swap_crv_y_spr,edfut_cvx_adj,false);
			}

			
			if(null(fwd_tenor)) {
				fwd_tenor = get_fwd_func_tenor(fwd_parm);
				QL_REQUIRE(!null(fwd_tenor),"invalid fra curve and/or swap curve");
			}
												
			//ceate the forward curve object
			fwd_func fwd_pre = null<fwd_func>;
			date fwd_pre_cut_off = null<date>;
			//Note:	fwd_pre and fwd_pre_cut_off can be used if we already have a fwd_func created up to fwd_pre_cut_off and
			//		wish to use this when creating the long end part of the fwd_curve i.e. it is possible to create a fwd_func
			//		in "pieces"
			
			if(incl_synt){
				//it is possible to include synthetic fras in the short end. they are created from extrapolating (backwards)
				//with a constant basis spread using df_synt_base 
				//the end of the discount curve (synt_end) determines the end of the forward curve when creating the basis spread 
				//curve for synt instruments (no extrapolation on the df_synt_base)
				fwd_crv	= fwd_curve_create_synt(d, df_ois,fwd_tenor, fwd_parm, df_synt_base, synt_end,fwd_pre,fwd_pre_cut_off);
			}
			else {
				fwd_crv	= fwd_curve_create(d, df_ois,fwd_tenor, fwd_parm,fwd_pre,fwd_pre_cut_off);
			}
			
			//check that the creation of the fwd_curve was ok (checking of instr types, currencies, quotes, etc)
			string msg;
			logical fwd_curve_ok = fwd_crv.check(msg);
			QL_REQUIRE(fwd_curve_ok,msg);
			/*------------------------------
			Step 2 create the forwarding model
			--------------------------------*/	
			fwd_model_parm fwd_p 		= fwd_model_parm();		//creates a parameter object for the fwd model with all defaults (the defaults are 
																//set in the dual_boot_config.ql file)

			if(!null(z_model))
				fwd_p.set_model(z_model);						//we override the default model with our choice

			if(!null(turn_crv))
				fwd_p.set_turns(turn_crv);						//set the turns via the turn curve (a fra curve), turns will create a fwd_func that is shifted between
																//the dates the turn is affecting a fwd rate, the model is fitted to the fwd rates with the turn effect removed
																//normally the fra turn curve is implied from the o/n turn curve (see create_disc_func)  
			
			fwd_z_model fwd_model 		= init_fwd_model(fwd_p);//create the fwd model based on the parameter object		
			
			/*---------------------
			Step 3 calc fwd func
			-----------------------*/
			fwd_func f_fwd;
			if(model_spr) {//model the spread on top of the disc_func input (while maintaining exact fit)
				f_fwd = fwd_crv.fwd_spr(df_ois,fwd_model,f_fwd_ex_turns) ;
			}
			else {
				f_fwd = fwd_crv.fwd(fwd_model,f_fwd_ex_turns);
			}
			
			QL_REQUIRE(!null(f_fwd),'invalid f_fwd');
			QL_REQUIRE(!null(f_fwd_ex_turns),'invalid f_fwd_ex_turns');

			return f_fwd;
		}	

		//-------------------------------------------------------
		// create_tenor_b2_func (generic)
		// create a swap_curve_b2_ext via tenor basis swap quoted as a spread between two fixed legs
		//-------------------------------------------------------
		swap_curve_b2_ext create_tenor_b2_func(	date 					d,
												fwd_func_tenor			fwd_tenor,
												//string option(nullable) z_model,
												disc_func				df_ois,									
												date option(nullable)	ois_end,
												curve option(nullable)	fix_crv, 	//for noow only a deposit_index (fixing) curve	
												curve option(nullable)	fra_out_crv, 
												curve 					swap_in_crv,//this swap will be converted with the basis curve to out curve
												curve 					basis_t1t2_crv,												
												logical 				fra_prio,
												logical  				incl_synt,																																		
												fwd_func option(nullable) 	fwd_pre = null<fwd_func>, 
												date option(nullable)		fwd_pre_cut_off = null<date>)
		{	
			//---------------------
			// create fwd curve
			//---------------------										
			logical merge_fra 			= false;
			logical merge_crv			= false;							
			integer blend_buf_days 		= DFLT_BLEND_BUF_DAYS;
			logical no_overlap_fwd		= false;			
			curve_prio prio1 			= curve_prio.SHORT;
			curve_prio prio2 			= fra_prio? curve_prio.MIDDLE : curve_prio.LONG;
			logical x_pol_basis 		= true;	
				
			curve_blend_parm blend_parm	= curve_blend_parm(prio1,prio2,no_overlap_fwd,merge_fra,merge_crv,blend_buf_days);		
			fixing_curve fixing_crv		= null(fix_crv) ? null<fixing_curve>: fixing_curve(fix_crv);
				
			vector(number) fra_crv_y_spr, swap_crv_y_spr, basis_crv_q_spr;
			swap_curve_b2_f_parm parm = swap_curve_b2_f_parm(	fixing_crv,fra_out_crv,swap_in_crv,basis_t1t2_crv,																					
																blend_parm,fra_crv_y_spr,swap_crv_y_spr,basis_crv_q_spr ) ;

			swap_curve_b2_ext fwd_crv;
			if(incl_synt)
				fwd_crv	= fwd_b2_curve_create_synt(	d,df_ois,fwd_tenor, parm, null<disc_func>, ois_end, x_pol_basis,fwd_pre, fwd_pre_cut_off);
			else
				fwd_crv	= fwd_b2_curve_create(	d,df_ois,fwd_tenor, parm, x_pol_basis,fwd_pre, fwd_pre_cut_off);
			
			QL_REQUIRE(!null(fwd_crv),'invalid swap_curve_b2_ext');

			return fwd_crv;
		}
		//-------------------------------------------------------
		// create_tenor_b2_func (generic)
		// create a fwd_func via tenor basis swap quoted as a spread between two fixed legs
		//-------------------------------------------------------
		fwd_func create_tenor_b2_func(date 					d,
									fwd_func_tenor			fwd_tenor,
									string option(nullable) z_model,
									disc_func				df_ois,									
									date option(nullable)	ois_end,
									curve option(nullable)	fix_crv, 	//for noow only a deposit_index (fixing) curve	
									curve option(nullable)	fra_out_crv, 
									curve 					swap_in_crv,//this swap will be converted with the basis curve to out curve
									curve 					basis_t1t2_crv,
									curve option(nullable)	turn_out_crv,
									logical 				fra_prio,
									logical  				incl_synt,										
									out swap_curve_b2_ext 	fwd_crv,													
									fwd_func option(nullable) 	fwd_pre = null<fwd_func>, 
									date option(nullable)		fwd_pre_cut_off = null<date>)
		{
			//---------------------
			// create fwd curve
			//---------------------	
			fwd_crv						= create_tenor_b2_func(	d,fwd_tenor,df_ois,ois_end,fix_crv,fra_out_crv,swap_in_crv,basis_t1t2_crv,
																fra_prio,incl_synt,fwd_pre,fwd_pre_cut_off);
			//---------------------
			//create fwd model 
			//---------------------
			fwd_model_parm fwd_p 		= fwd_model_parm();
			fwd_p.set_model(z_model);
			if(!null(turn_out_crv))
				fwd_p.set_turns(turn_out_crv);						//set the turns via the turn curve (a fra curve)
			fwd_z_model fwd_model 		= init_fwd_model(fwd_p);

			//---------------------
			// calc fwd func
			//---------------------
			fwd_func f_fwd 				= fwd_crv.fwd(fwd_model);
			QL_REQUIRE(!null(f_fwd),'invalid f_fwd');

			return f_fwd;
		}
		//-------------------------------------------------------
		// create_tenor_b2_func (generic)
		// create a fwd_func via tenor basis swap quoted as a spread between two fixed legs
		//-------------------------------------------------------
		fwd_func create_tenor_b2_func(date 					d,
									fwd_func_tenor			fwd_tenor,
									string option(nullable) z_model,
									disc_func				df_ois,									
									date option(nullable)	ois_end,
									curve option(nullable)	fix_crv, //for noow only a deposit_index (fixing) curve		
									curve option(nullable)	fra_out_crv, 
									curve 					swap_in_crv,//this swap will be converted with the basis curve to out curve
									curve 					basis_t1t2_crv,
									curve option(nullable)	turn_out_crv,
									logical 				fra_prio,
									logical  				incl_synt,
									logical					model_spr,
									fwd_func option(nullable)	fwd_func_base,										
									out swap_curve_b2_ext 	fwd_crv,													
									fwd_func option(nullable) 	fwd_pre = null<fwd_func>, 
									date option(nullable)		fwd_pre_cut_off = null<date>)
		{
			//---------------------
			// create fwd curve
			//---------------------	
			fwd_crv						= create_tenor_b2_func(	d,fwd_tenor,df_ois,ois_end,fix_crv,fra_out_crv,swap_in_crv,basis_t1t2_crv,
																fra_prio,incl_synt,fwd_pre,fwd_pre_cut_off);			
			//---------------------
			//create fwd model 
			//---------------------
			fwd_model_parm fwd_p 		= fwd_model_parm();
			fwd_p.set_model(z_model);
			if(!null(turn_out_crv))
				fwd_p.set_turns(turn_out_crv);						//set the turns via the turn curve (a fra curve)
			fwd_z_model fwd_model 		= init_fwd_model(fwd_p);
				
			//---------------------
			// calc fwd func
			//---------------------
			if(model_spr){
				QL_REQUIRE(!null(fwd_func_base), "invalid fwd_func_base (required when modelling the spread)");
				return fwd_crv.fwd_spr(fwd_func_base,fwd_model);
			}
			else
				return fwd_crv.fwd(fwd_model);	
		}

		//-------------------------------------------------------
		// create_tenor_b1_func 
		// create a fwd_func via tenor basis swap
		// in this function the fwd_func for the tenor we're not solving for is exogenous
		//-------------------------------------------------------
		fwd_func create_tenor_b1_func(date 						d,
										string option(nullable) z_model,
										disc_func				df_ois,									
										date option(nullable) 	ois_end,
										fwd_func				fwd_in,
										fwd_func_tenor fwd_tenor_in,
										fwd_func_tenor fwd_tenor_out,			//the "solve-tenor"
										curve option(nullable)	fix_crv, 		//for noow only a deposit_index (fixing) curve												
										curve option(nullable)	fra_out_crv, 	//fra's with "solve-tenor"								
										curve 					basis_t1t2_crv,
										curve option(nullable)	turn_out_crv,
										logical					fra_prio,																							
										logical 				incl_synt,
										logical					model_spr,																														
										out swap_curve_b1_ext 	fwd_crv,
										vector(date) option(nullable)  basis_ip_mat = null<vector(date)>,													
										fwd_func option(nullable) 	fwd_pre = null<fwd_func>, 
										date option(nullable)		fwd_pre_cut_off = null<date>)
		{		
			//---------------------
			//create fwd model 
			//---------------------
			fwd_model_parm fwd_p 		= fwd_model_parm();
			fwd_p.set_model(z_model);
			if(!null(turn_out_crv))
				fwd_p.set_turns(turn_out_crv);						//set the turns via the turn curve (a fra curve)
			fwd_z_model fwd_model 		= init_fwd_model(fwd_p);
		
			//---------------------
			// create fwd curve
			//---------------------
			logical merge_fra 			= false;
			logical merge_crv			= false;							
			integer blend_buf_days 		= DFLT_BLEND_BUF_DAYS;
			logical no_overlap_fwd		= false;	
			curve_prio prio1 			= curve_prio.SHORT;
			curve_prio prio2 			= fra_prio? curve_prio.MIDDLE : curve_prio.LONG;

			curve_blend_parm blend_parm	= curve_blend_parm(prio1,prio2,no_overlap_fwd,merge_fra,merge_crv,blend_buf_days);		
			fixing_curve fixing_crv		= null(fix_crv) ? null<fixing_curve>: fixing_curve(fix_crv);
				
			vector(number) fra_crv_y_spr, basis_crv_q_spr;
			swap_curve_b1_f_parm parm = swap_curve_b1_f_parm(	fixing_crv,fra_out_crv,basis_t1t2_crv,																					
																blend_parm,fra_crv_y_spr,basis_crv_q_spr ) ;
			
			//fwd_model is used in fwd_b1_curve_create only if incl_synt = true because a fwd_model is needed to calculate 
			//the implied swap curve which in turn is needed in the synt instrument calculation
			if(incl_synt)
				fwd_crv  = fwd_b1_curve_create_synt(d,df_ois,fwd_tenor_in,fwd_in, fwd_tenor_out, parm,  
													null<disc_func>,ois_end,fwd_model, basis_ip_mat, fwd_pre, fwd_pre_cut_off);
			else
				fwd_crv  = fwd_b1_curve_create(	d,df_ois,fwd_tenor_in,fwd_in, fwd_tenor_out, parm,
												basis_ip_mat, fwd_pre, fwd_pre_cut_off);
			
			//---------------------
			// calc fwd func
			//---------------------
			fwd_func f_fwd 				= model_spr ? fwd_crv.fwd_spr(fwd_in,fwd_model) : fwd_crv.fwd(fwd_model);
			QL_REQUIRE(!null(f_fwd),'invalid f_fwd');

			return f_fwd;							
		}

		//-------------------------------------------------------
		// create_cb_imp_disc_sprd  
		//-------------------------------------------------------
		disc_func create_cb_imp_disc_sprd(	date 							d,
											fx_spot option(nullable) 		fxs,
											curve  option(nullable) 		fxswap_crv,
											curve  							cc_basis_crv,
											disc_func 						df_flat,
											fwd_func_tenor 					fwd_tenor_flat,
											fwd_func 						fwd_flat,										
											instrument  					fix_instr_flat,

											disc_func option(nullable)		df_sprd,
											fwd_func_tenor 					fwd_tenor_sprd,
											fwd_func 						fwd_sprd,																												
											instrument  					fix_instr_sprd,
											interpolator 					ip,
											rate_type						rt,																										
											out swap_curve_bc_ext  			bc) 
		{																																
			logical	 merge_crv 			= false;
			logical prio_fxswap       	= !null(fxswap_crv);											
			integer	 blend_buf_days		= DFLT_BLEND_BUF_DAYS;

			fixing_curve  fixing_crv_flat = fixing_curve(curve([fix_instr_flat]));
			fixing_curve  fixing_crv_sprd = fixing_curve(curve([fix_instr_sprd]));	

			swap_curve_cb1_f_parm cb1_parm = swap_curve_cb1_f_parm(	fxs,fxswap_crv,fixing_crv_flat,fixing_crv_sprd,											
																	cc_basis_crv,merge_crv,prio_fxswap,blend_buf_days);
				
			bc 							= cb_curve_create(	d,cb1_parm,df_flat, df_sprd,fwd_flat,
															fwd_tenor_flat,fwd_sprd,fwd_tenor_sprd,"noname");

			return bc.adj_disc_df_sprd(ip,rt);
		}

		//-------------------------------------------------------
		// create_cb_imp_disc_flat  
		//-------------------------------------------------------
		disc_func create_cb_imp_disc_flat(	date 							d,
											fx_spot option(nullable) 		fxs,
											curve  option(nullable) 		fxswap_crv,
											curve  							cc_basis_crv,
											disc_func option(nullable)		df_flat,
											fwd_func_tenor 					fwd_tenor_flat,
											fwd_func 						fwd_flat,										
											instrument  					fix_instr_flat,

											disc_func 						df_sprd,
											fwd_func_tenor 					fwd_tenor_sprd,
											fwd_func 						fwd_sprd,																												
											instrument  					fix_instr_sprd,
											interpolator 					ip,
											rate_type						rt,																										
											out swap_curve_bc_ext  bc) 
		{																																
			logical	 merge_crv 			= false;
			logical prio_fxswap       	= !null(fxswap_crv);											
			integer	 blend_buf_days		= DFLT_BLEND_BUF_DAYS;

			fixing_curve  fixing_crv_flat = fixing_curve(curve([fix_instr_flat]));
			fixing_curve  fixing_crv_sprd = fixing_curve(curve([fix_instr_sprd]));	

			swap_curve_cb1_f_parm cb1_parm = swap_curve_cb1_f_parm(	fxs,fxswap_crv,fixing_crv_flat,fixing_crv_sprd,											
																	cc_basis_crv,merge_crv,prio_fxswap,blend_buf_days);
				
			bc 							= cb_curve_create(	d,cb1_parm,df_flat, df_sprd,fwd_flat,
															fwd_tenor_flat,fwd_sprd,fwd_tenor_sprd,"noname");

			return bc.adj_disc_df_flat(ip,rt);
		}

		//-------------------------------------------------------
		// create_cbois_imp_disc_sprd   (for basis swaps with 2 ois legs)
		//-------------------------------------------------------
		disc_func create_cbois_imp_disc_sprd(date 			d,
											 fx_spot option(nullable) fxs,
											curve  option(nullable) fxswap_crv,
											curve  			cc_basis_crv,
											disc_func 		df_flat,										
											disc_func 		df_sprd,
											disc_func		df_fwd_flat,			
											disc_func 		df_fwd_sprd,
											interpolator 	ip,
											rate_type		rt,																										
											out swap_curve_bc_ois  bc) 
		{																															
			logical	 merge_crv 			= false;
			logical prio_fxswap       	= !null(fxswap_crv);											
			integer	 blend_buf_days		= DFLT_BLEND_BUF_DAYS;
			
			swap_curve_cbois1_f_parm cb1_parm = swap_curve_cbois1_f_parm(fxs,fxswap_crv,null<fixing_curve>,cc_basis_crv,merge_crv,prio_fxswap,blend_buf_days);
				
			bc 	= cbois_curve_create(d,cb1_parm,df_flat, df_sprd,df_fwd_flat,df_fwd_sprd,"noname");

			return bc.adj_disc_df_sprd(ip,rt);
		}

		//-------------------------------------------------------
		// create_cbois_imp_disc_flat  (for basis swaps with 2 ois legs)
		//-------------------------------------------------------
		disc_func create_cbois_imp_disc_flat(	date 			d,
												fx_spot option(nullable) fxs,
												curve  option(nullable) fxswap_crv,
												curve  			cc_basis_crv,
												disc_func 		df_flat,										
												disc_func 		df_sprd,
												disc_func		df_fwd_flat,			
												disc_func 		df_fwd_sprd,										 
												interpolator 	ip,
												rate_type		rt,																										
												out swap_curve_bc_ois  bc) 
		{																															
			logical	 merge_crv 			= false;
			logical prio_fxswap       	= !null(fxswap_crv);											
			integer	 blend_buf_days		= DFLT_BLEND_BUF_DAYS;
			
			swap_curve_cbois1_f_parm cb1_parm = swap_curve_cbois1_f_parm(fxs,fxswap_crv,null<fixing_curve>,cc_basis_crv,merge_crv,prio_fxswap,blend_buf_days);
				
			bc 	= cbois_curve_create(d,cb1_parm,df_flat, df_sprd,df_fwd_flat,df_fwd_sprd,"noname");

			return bc.adj_disc_df_flat(ip,rt);
		}
		//-------------------------------------------------------
		//freq_map
		// to convert number to freq enum used in swap-lib
		//-------------------------------------------------------
		coupon_freq freq_map(number n)
		{
			switch(n)
			{
			case 1:
				return CFR_ANN ;
			case 2:
				return CFR_SEMI_ANN ;
			case 4:
				return CFR_QUART ;
			case 12:
				return CFR_MONTH ;
			default:
				return null<coupon_freq> ;
			}
		}
		
		//-------------------------------------------------------
		// out_rate
		// basic output for "eye-balling" implied forward rates
		//-------------------------------------------------------
		void out_rate(	date	 			d,
						integer 			tenor_freq,
						fwd_func			f_fwd,
						calendar			cal,
						out vector(string) 	label,
						out vector(date) 	float_idx0,
						out vector(date) 	float_idx1,																				
						out vector(number) 	imp)
		{
			date settle = cal.move_bus_days (d, 2);
			
			resize(float_idx0,0);
			vector(date) float_pmt[0];
			date out_first,out_last_reg;
			integer out_roll_day;
			logical is_reg_first,is_reg_last,eom_roll;
			integer roll_day = 0;
			integer	 freq = 12;
			date end_date = date_code("5Y").apply_fwd(settle);
			vector(date) fixing_dates;
			
			swap_pmt_dates_flt(settle ,null<date>,null<date>, end_date,roll_day,freq,
							   BD_MOD_FOLLOWING,cal,true,
							   BD_MOD_FOLLOWING,cal,true,
							   2,true,out_first,out_last_reg, out_roll_day ,
							   is_reg_first, is_reg_last, eom_roll,float_idx0,float_pmt,fixing_dates);
			
	/*		fiws.swap_flt_dates(settle ,null<date>,null<date>, end_date,roll_day,freq,
								BD_MOD_FOLLOWING,cal,true, 
								BD_MOD_FOLLOWING,cal,true,true,out_first,out_last_reg,
								out_roll_day,is_reg_first,is_reg_last, eom_roll,float_idx0,float_pmt);*/

			
			imp = f_fwd.fwd((float_idx0 - d)/365.0)*100;
			number m = 12.0/tenor_freq;
			float_idx1 = move_month(float_idx0,m,true,cal,BD_MOD_FOLLOWING,false);			
			resize(label, v_size(imp));
			label[0] = strcat(["Spot",str(m),"m"]);
			for(integer i = 1; i < v_size(imp); i++){
				label[i] = strcat([str(i),'M x ', str(i+m), 'M']);
			}
		}
		//-------------------------------------------------------
		// out_rate
		// basic output for "eye-balling" implied forward rates
		//-------------------------------------------------------
		void out_rate(	date	 			d,
						integer 			tenor_freq,
						disc_func			df_fwd,
						day_count_method 	dc,
						calendar 			cal,
						out vector(string) 	label,
						out vector(date) 	float_idx0,
						out vector(date) 	float_idx1,	
						out vector(number) 	imp)
		{
			date settle = cal.move_bus_days (d, 2);
			
			resize(float_idx0,0);
			vector(date) float_pmt[0];
			date out_first,out_last_reg;
			integer out_roll_day;
			logical is_reg_first,is_reg_last,eom_roll;
			integer roll_day = 0;
			integer	 freq = 12;
			date end_date = date_code("5Y").apply_fwd(settle);

			vector(date) fixing_dates;
			swap_pmt_dates_flt(settle ,null<date>,null<date>, end_date,roll_day,freq,
							   BD_MOD_FOLLOWING,cal,true,
							   BD_MOD_FOLLOWING,cal,true,
							   2,true,out_first,out_last_reg, out_roll_day ,
							   is_reg_first, is_reg_last, eom_roll,float_idx0,float_pmt,fixing_dates);

			
			/*fiws.swap_flt_dates(settle ,null<date>,null<date>, end_date,roll_day,freq,
								BD_MOD_FOLLOWING,cal,true, 
								BD_MOD_FOLLOWING,cal,true,true,out_first,out_last_reg,
								out_roll_day,is_reg_first,is_reg_last, eom_roll,float_idx0,float_pmt);	*/
			
			number m = 12.0/tenor_freq;
			float_idx1 = move_month(float_idx0,m,true,cal,BD_MOD_FOLLOWING,false);
			imp = df_fwd.simple_rate(d, float_idx0,float_idx1, dc)*100;			
			resize(label, v_size(imp));
			label[0] = strcat(["Spot",str(m),"m"]);
			for(integer i = 1; i < v_size(imp); i++){
				label[i] = strcat([str(i),'M x ', str(i+m), 'M']);
			}
		}	
		//-------------------------------------------------------
		// out_swap_rate
		// basic output for "eye-balling" swap rates
		//-------------------------------------------------------
		/*void out_swap_rate(	date	 			d,
							disc_func			df_disc,					
							vector(date) 		maturity,
							date				issue_date,
							integer 			fix_freq,
							day_count_method	fix_dc,			
							integer 			flt_freq,
							day_count_method	flt_dc,
							fwd_func			f_flt_idx,	
							calendar 			cal,				
							out vector(string) 	label,
							out vector(number) 	r)
		{
			date settle = cal.move_bus_days (d, 2);
			//vector(date) fix_date_beg;
			//vector(date) fix_date_end;
			//vector(date) flt_date_beg;
			//vector(date) flt_date_end;
			bd_convention bd = BD_MOD_FOLLOWING;
			integer size = v_size(maturity);
		
			resize(r, size);
			resize(label, size);
			for(integer i = 0; i < size; i++){
				r[i] = fiws.par_swap_rate(d,cal,settle,bd ,settle ,maturity[i] ,flt_freq,fix_freq,
										flt_dc,fix_dc,df_disc,f_flt_idx,2,null<number> )*100;
				label[i] = strcat(str(round(maturity[i]-settle )/365,0),"Y");
			}
		}*/			
		//-------------------------------------------------------
		// on_rates
		//-------------------------------------------------------
		void on_rates(	date 		d, 
						date 		max,
						integer 	interval, 				
						fwd_func 	f_fwd,
						calendar 			cal,				
						out vector(date) 	vd,
						out vector(number) 	vn)
		{
			resize(vd,0);
			resize(vn,0);
		
			date start = d;
			while (start  <= max){				
				number r = f_fwd.fwd((start - d)/365.0)*100;		
				push_back(vd,start);
				push_back(vn,r);	
				start = cal.adjust_date (start  + interval, BD_FOLLOWING);
			}		
			return ;
		}
		//-------------------------------------------------------
		// on_rates
		//-------------------------------------------------------
		void on_rates(	date 		d, 
						date 		max,
						integer 	interval, 				
						disc_func 	df_fwd,
						logical 	true_period,
						day_count_method dc,
						calendar 	cal,
						out vector(date) 	vd,						
						out vector(number) 	vn)
		{
			resize(vd,0);
			resize(vn,0);
		
			date start = d;
			while (start  <= max){
				
				date end;
				
				if(true_period)
					end = move_bus_days (start, 1, cal);
				else
					end = start + 1;
				
				number r =  df_fwd.simple_rate(d, start , end, dc)*100;
				push_back(vd,start);
				push_back(vn,r);
				start  += interval;
				start = cal.adjust_date (start, BD_FOLLOWING);
			}		
			return ;
		}
			
		//-------------------------------------------------------
		// gr_on_pd (always a 1d period)
		//-------------------------------------------------------
		vector(point_date) gr_on_pd(date 		d, 
									date 		max,
									integer 	interval, 				
									disc_func 	df_fwd,
									day_count_method dc,
									calendar 	cal)
		{
			vector(date) vd;
			vector(number) vn;
		
			on_rates(d,max,interval,df_fwd,true,dc,cal,vd,vn);
					
			return  point(vd, vn);
		}
		//-------------------------------------------------------
		// gr_v_pd
		//-------------------------------------------------------
		vector(point_date) gr_v_pd(	date 	d, 
									date 	max,
									integer interval, 
									fwd_func f_fwd)
		{
			vector(date) vd;
			vector(number) vn;
		
			date start = d;
			while (start  <= max){				
				number r =  f_fwd.fwd((start-d)/365.0)*100;
				push_back(vd,start);
				push_back(vn,r);
				start  += interval;
			}		
			return  point(vd, vn);
		}

		//-------------------------------------------------------
		// gr_v_pd_sprd
		//-------------------------------------------------------
		vector(point_date) gr_v_pd_sprd(date 	d, 
										date 	max,
										integer interval, 
										fwd_func f_fwd_base, 
										fwd_func f_fwd)
		{
			vector(date) vd;
			vector(number) vn;
		
			date start = d;
			while (start  <= max){				
				number rb =  f_fwd_base.fwd((start-d)/365.0)*100;
				number r =  f_fwd.fwd((start-d)/365.0)*100;
				push_back(vd,start);
				push_back(vn,(r-rb)*100);
				start  += interval;
			}		
			return  point(vd, vn);
		}
		
		//-------------------------------------------------------
		// set_test_quote
		//-------------------------------------------------------
		void set_test_quote(out curve crv, number q)
		{
			try{
				if(null(crv))
					return;
				vector(number) user_y_pct;
				resize(user_y_pct, v_size(crv.instruments()));
				for(integer i = 0; i < v_size(crv.instruments()); i++)
					user_y_pct[i] = q;
				WSP_UTIL.set_curve_quote(crv,user_y_pct,crv);
			}
			catch{
				warning(err.message());
			}
		}
		//-------------------------------------------------------
		// set_test_quote
		//-------------------------------------------------------
		void set_test_quote(out curve crv, vector(number) q)
		{
			try{
				if(v_size(q) == 1) {
					set_test_quote(crv, q[0]);
					return;
				}
				if(null(crv))
					return;
				vector(number) user_y_pct;
				resize(user_y_pct, v_size(crv.instruments()));
				for(integer i = 0; i < v_size(crv.instruments()); i++)
					user_y_pct[i] = q[i];
				WSP_UTIL.set_curve_quote(crv,user_y_pct,crv);
			}
			catch{
				warning(err.message()); 
			}
		}

		
	
	}
}
	