/*	
	float-ois swap example with a template
	Algorithmica Research, Magnus Nyström

	notes on fltois
	single currency: primary quote leg is leg2 (spread on the ois leg) (to follow the fltflt principle)

	cross ccy:
	- if usd: spread on the non-usd leg,
	- if non usd: spread on leg2 
	
	(ex. nibor3m=leg1, sofr=leg2, spread is on leg1, mtm, 2 day payment delay for interest payments both sides, no delay on principal
	quarterly pmts)

	
*/

option(null: hard); // Throw an error if a null value is used in a non-nullable context

module BP_INST
{
	/*-----------------------------------------------------------------------
	  swap_flt_ois_single_ccy_tmpl
	  ----------------------------------------------------------------------*/	
	swap_fltois swap_flt_ois_single_ccy_tmpl( INSTR_TMPL.swap_fltois_def_tmpl	tmpl,
											  string 							name,
											  date 	option(nullable) 			trade_date,
											  date option(nullable) 			settle_date,
											  date option(nullable) 			start_date,
											  date option(nullable) 			maturity,
											  number 							notional,
											  logical							pay_float,
											  disc_func option(nullable) 		disc_func,
											  number option(nullable) 			flt_spread,//decimal
											  number option(nullable) 			flt_curr_fix ,//decimal
											  fwd_func  option(nullable) 		flt_fwd_func,	
											  number option(nullable) 			ois_spread,//decimal
											  fwd_func  option(nullable) 		ois_fwd_func,											  
											  error_info option(nullable) 		e)
	{
		string fwd_start_code 		= tmpl.fwd_start_code();
		string maturity_code 		= tmpl.maturity_code();
		
		integer sprd_rnd_dec		= -1;//-1 --> no rounding				
		number pv01_not_unit  		= 100000;//rounding unit for notional when implied from pv01					
		number leg_pv01 			= null;//the leg in question depends on which leg is the quote leg (use fltois_parm.primary_quote_is_flt())
		
		if(null(maturity) && null(maturity_code))
		   QL_FAIL("invalid maturity/maturity_code");

		if(null(trade_date) && null(settle_date) && null(start_date) && null(fwd_start_code))
		   QL_FAIL("swap start not valid");//if either tradedate or settledate is input the startdate is implied via spot_code
		
		//create empty parm
		CORE_SWAPLIB.fltois_parm fio = new CORE_SWAPLIB.fltois_parm();

		//update parm with static data using the template
		fio.init_static(tmpl);

		logical q_is_float = fio.primary_quote_is_leg1();
		if(q_is_float) {
			if(null(flt_spread) && (null(disc_func) || null(flt_fwd_func) || null(trade_date)))
				QL_FAIL("flt_spread_rate cannot be implied due to incomplete input");
		}
		else {
			if(null(ois_spread) && ((null(disc_func) && null(ois_fwd_func)) || null(trade_date)))
				QL_FAIL("ois_spread_rate cannot be implied due to incomplete input");
		}
		
		//some checks eg. that we have an appropriate template
		QL_REQUIRE(!fio.is_cross_currency(),"invalid template (must be single currency)");
		
		fio.set_plain_single_ccy(	name,trade_date,settle_date,start_date,fwd_start_code,maturity ,
									maturity_code, pay_float, sprd_rnd_dec,pv01_not_unit, notional,leg_pv01,
									flt_spread, flt_curr_fix, ois_spread );
			
		logical sync_endog_ok 		= true;//if this is true we are in charge of making sure that fwd_func and disc_func are in sync in case we have self discounting
		logical flt_allow_extrap	= false;//do not allow extrap of fwd rates when we have an odd start or odd end period
		swap_fltois swap 			= fio.create_swap(disc_func, flt_fwd_func, null, disc_func, ois_fwd_func,null,sync_endog_ok, flt_allow_extrap);
			
		QL_FAIL_COND(e.is_error(),e.message());
		QL_FAIL_COND(null(swap),"invalid swap");
		return swap;
	}

	/*-----------------------------------------------------------------------
	  swap_flt_ois_cross_ccy_tmpl
	  ----------------------------------------------------------------------*/	
	swap_fltois swap_flt_ois_cross_ccy_tmpl(INSTR_TMPL.swap_fltois_def_tmpl	tmpl,
											string 							name,
											date 	option(nullable) 		trade_date,
											date option(nullable) 			settle_date,
											date option(nullable) 			start_date,
											date option(nullable) 			maturity,
											number option(nullable) 		fx_quote,//as of settledate
											number option(nullable) 		fx_mtm_spot_quote,//as of spot date NOT swap settle
											string 							fx_quote_baseccy,
											string 							fx_quote_priceccy,
											logical 						pv_in_base_ccy,											 
											logical							pay_float,
											//flt
											number 							flt_notional,											  
											disc_func option(nullable) 		flt_disc_func,
											number option(nullable) 		flt_spread,//decimal
											number option(nullable) 		flt_curr_fix ,//decimal
											fwd_func  option(nullable) 		flt_fwd_func,
											//ois
											number 							ois_notional,
											disc_func option(nullable) 		ois_disc_func,
											number option(nullable) 		ois_spread,//decimal
											fwd_func  option(nullable) 		ois_fwd_func,											  
											error_info option(nullable) 	e)
	{
		string fwd_start_code 		= tmpl.fwd_start_code();
		string maturity_code 		= tmpl.maturity_code();
		
		integer imp_sprd_dec		= -1;//-1 --> no rounding				
		number pv01_not_unit  		= 100000;//rounding unit for notional when implied from pv01					
		number flt_pv01 			= null;//the leg in question depends on which leg is the quote leg (use oisois_parm.primary_quote_is_flt())
		
		if(null(maturity) && null(maturity_code))
		   QL_FAIL("invalid maturity/maturity_code");

		if(null(trade_date) && null(settle_date) && null(start_date) && null(fwd_start_code))
		   QL_FAIL("swap start not valid");//if either tradedate or settledate is input the startdate is implied via spot_code
		
		//create empty parm
		CORE_SWAPLIB.fltois_parm fo = new CORE_SWAPLIB.fltois_parm();

		//update parm with static data using the template
		fo.init_static(tmpl);

		logical q_is_float = fo.primary_quote_is_leg1();
		if(q_is_float) {
			if(null(flt_spread) && (null(flt_disc_func) || null(flt_fwd_func) || null(trade_date)))
				QL_FAIL("flt_spread_rate cannot be implied due to incomplete input");
		}
		else {
			if(null(ois_spread) && (null(flt_disc_func) || null(ois_fwd_func) || null(trade_date)))
				QL_FAIL("ois_spread_rate cannot be implied due to incomplete input");
		}	

		stub_type  flt_stub_type 		= null;
		date  flt_eff_date 			= null;
		date  flt_first_cpn_date 		= null;
		date  flt_last_reg_date 		= null;
		vector(date) flt_fixing_dates  = null;	
		vector(number) flt_fixing_rates = null;
		number flt_fix_proxy  			= null;
		logical flt_allow_fwd_fix 		= null;

		stub_type ois_stub_type 		= null;
		date ois_eff_date 				= null;
		date ois_first_cpn_date 		= null;
		date ois_last_reg_date 		= null;
		vector(date) ois_fixing_dates  = null;	
		vector(number) ois_fixing_rates = null;
		number ois_fix_proxy  			= null;
		logical ois_allow_fwd_fix 		= null;
		
		fo.set_plain(	name,trade_date,settle_date,start_date,fwd_start_code,maturity ,
						maturity_code, pay_float, imp_sprd_dec,pv01_not_unit,
						flt_pv01,fx_quote, fx_quote_baseccy, fx_quote_priceccy, pv_in_base_ccy,
						//flt
						flt_spread,
						flt_stub_type,
						flt_eff_date,
						flt_first_cpn_date,
						flt_last_reg_date,
						flt_notional,
						flt_curr_fix,
						flt_fixing_dates ,	
						flt_fixing_rates,
						flt_fix_proxy ,
						flt_allow_fwd_fix,
						//ois
						ois_spread,
						ois_stub_type,
						ois_eff_date,
						ois_first_cpn_date,
						ois_last_reg_date,
						ois_notional,
						ois_fixing_dates ,	
						ois_fixing_rates,
						ois_fix_proxy ,
						ois_allow_fwd_fix);				

		fo.set_mtm_data(fx_mtm_spot_quote);//currently not as an input in set_plain func so need to set it separately
		
		logical sync_endog_ok 		= true;//if this is true we are in charge of making sure that fwd_func and disc_func are in sync in case we have self discounting
		logical flt_allow_extrap	= false;//do not allow extrap of fwd rates when we have an odd start or odd end period
		swap_fltois swap = fo.create_swap(flt_disc_func, flt_fwd_func, null,ois_disc_func, ois_fwd_func,null,sync_endog_ok, flt_allow_extrap);
			
		QL_FAIL_COND(e.is_error(),e.message());
		QL_FAIL_COND(null(swap),"invalid swap");
		return swap;
	}
}