option(null: hard);	

/*	
	instrument wrapper functions
	Developer: Algorithmica Research, Magnus Nyström

	--------------------------------------------------------------------
	 	swap_fixflt create funcs
	--------------------------------------------------------------------			
*/

//--------
// from db
//--------

/*-----------------------------------------------------------------------
  swap_fixflt
  no input of settle_date for db swaps
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt(instrument_name 			instrument_name, 
							date option(nullable) 		trade_date 	= null<date>,
							string option(nullable) 	quote_side 	= null<string>,
							error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{	
	try{			
		__instrument c = __instrument(instrument_name,trade_date,quote_side, null<date> );		
		return create_swap_fixflt(c,error, E_INIT);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt");
		return null<swap_fixflt>;
	}
}

/*-----------------------------------------------------------------------
  swap_fixflt
  no input of settle_date for db swaps
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt(instrument_name 			instrument_name, 
							date option(nullable) 		trade_date ,
							number  option(nullable) 	quote ,
							error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{	
	try{				
		__instrument c = __instrument(instrument_name,trade_date,quote, null<date> );
		return create_swap_fixflt(c,error, E_INIT);							
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt");
		return null<swap_fixflt>;
	}
}


/*-----------------------------------------------------------------------
  swap_fixflt
  no input of settle_date for db swaps
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt(	instr_def 					instr_def, 
							date option(nullable) 		trade_date 	= null<date>,
							string option(nullable) 	quote_side 	= null<string>,
							error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{	
	try{				
		__instrument c = __instrument(instr_def,trade_date,quote_side, null<date> );
		return create_swap_fixflt(c,error, E_INIT);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt");
		return null<swap_fixflt>;
	}
}
/*-----------------------------------------------------------------------
  swap_fixflt
  no input of settle_date for db swaps
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt(	instr_def 					instr_def, 
							date option(nullable) 		trade_date ,
							number  option(nullable) 	quote ,
							error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{	
	try{			
		__instrument c = __instrument(instr_def,trade_date,quote, null<date> );
		return create_swap_fixflt(c,error, E_INIT);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt");
		return null<swap_fixflt>;
	}
}


//----------------
// from db or not
//----------------

/*-----------------------------------------------------------------------
  func: swap_fixflt
  arguments: 
	instr_def: 	can be created either from the db or user defined 				 
				instr_def is copied and modified
				no lookup of quotes
	instrument_name: user defined name

  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt(	instr_def 				instr_def, 
							string 					instrument_name  ,
							date option(nullable) 	trade_date ,
							date option(nullable) 	settle_date ,//=effective_date=start_date							
							date  option(nullable) 	maturity,//maturity has precedence over mat_code
							string option(nullable) maturity_code,
							number option(nullable)	fix_rate,							
							number 					nominal 	= 100,
							error_info option(nullable) error	= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{	
	try{			
		__instrument c;

		if(!null(maturity))
			c = __instrument_fixflt_par_nodb(instr_def,instrument_name,trade_date,settle_date,
														maturity,fix_rate,nominal );
		else if(!null(maturity_code))
			c = __instrument_fixflt_par_nodb(instr_def,instrument_name,trade_date,settle_date,
														maturity_code,fix_rate,nominal );
		else
			QL_FAIL("invalid maturity/maturity_code");			

		return create_swap_fixflt( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt");
		return null<swap_fixflt>;
	}
}
/*-----------------------------------------------------------------------
  swap_fixflt
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt(CORE_SWAPLIB.fixflt_parm 	fifl,
						disc_func option(nullable) 	fix_disc_func,
						disc_func option(nullable) 	flt_disc_func,
						fwd_func option(nullable) 	flt_fwd_func,
						error_info option(nullable) error	= null<error_info> )
{
	try {
		QL_REQUIRE(fifl.is_inited(),"fixflt parameter object not initialized");
		return fifl.create_swap( fix_disc_func, flt_disc_func, flt_fwd_func);
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt");
		return null<swap_fixflt>;
	}
}
/*-----------------------------------------------------------------------
  swap_fixflt
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt(CORE_SWAPLIB.fixflt_parm 	fifl,
						disc_func option(nullable) 	fix_disc_func,
						disc_func option(nullable) 	flt_disc_func,
						disc_func option(nullable) 	flt_fwd_disc_func,
						error_info option(nullable) error	= null<error_info> )
{
	try {
		QL_REQUIRE(fifl.is_inited(),"fixflt parameter object not initialized");
		return fifl.create_swap( fix_disc_func, flt_disc_func, flt_fwd_disc_func);
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt");
		return null<swap_fixflt>;
	}
}

module CORE_INT_FIXFLT
{
	
	/*-----------------------------------------------------------------------
	  func: swap_fixflt_plain_int
	  minimal constructor for single curr "screen"-swap, based on a instr_def
	  ----------------------------------------------------------------------*/
	swap_fixflt swap_fixflt_plain_int(	instr_def option(nullable)		def,//only one of the def, tmpl, sw_base can be input
										swap_fixflt option(nullable)	sw_base,
										INSTR_TMPL.swap_fixflt_def_tmpl	option(nullable) tmpl,
										string 							name,	
										date option(nullable) 			trade_date,
										date option(nullable) 			settle_date,
										string option(nullable) 		spot_code, //not used with template
										date option(nullable) 			start_date,//=effective_date										
										string option(nullable) 		fwd_start_code,
										date 	option(nullable)		maturity,
										string 	option(nullable)		maturity_code,
										number option(nullable)			fix_coupon_rate ,//if null --> will be set to par
										integer 						fix_r_rnd_dec,// < 0 --> no rounding
										number 	option(nullable)		notional,
										number 	 option(nullable)		pv01,
										logical 						pay_fixed,
										..disc_func option(nullable) 	disc_func,								
										fwd_func option(nullable) 		flt_fwd_func,
										..disc_func option(nullable) 	flt_fwd_disc_func,
										number option(nullable) 		curr_fix,										
										error_info option(nullable) 	error )
	{	
		try{
			//basic initial checks of input
			QL_FAIL_COND(!null(flt_fwd_func) && !null(flt_fwd_disc_func),"ambigious input (fwd_func/disc_func)");
			QL_FAIL_COND(null(notional) && null(pv01), "one of notional and pv01 is required", E_INIT);

			//QL_FAIL_COND(!null(start_date) && !null(fwd_start_code), "ambigious input (start_date/fwd_start_code)", E_INIT);
			
			//create empty parm
			CORE_SWAPLIB.fixflt_parm fifl = new CORE_SWAPLIB.fixflt_parm();

			//update parm with init_static()
						
			if(!null(def)) {
				fifl.init_static(def,spot_code);
			}
			else if(!null(sw_base)){
				fifl.init_static(sw_base,spot_code);
			}
			else if(!null(tmpl)){
				fifl.init_static(tmpl);
			}
			else{
				QL_FAIL("err");
			}
			
			QL_FAIL_COND(fifl.is_cross_currency(), "constructor not applicable for currency swap", E_INIT);

			//update parm with current input
			number pv01_not_unit = 100000 ;//rounding unit for notional when implied from pv01
			number flt_spread = 0;
			
			fifl.set_plain_single_ccy(	name,trade_date,settle_date,start_date,fwd_start_code,maturity ,
										maturity_code, pay_fixed, notional,pv01, pv01_not_unit,
										fix_coupon_rate, fix_r_rnd_dec, flt_spread,curr_fix, null,null);
			
			//create swap
			//2 versions: 1) fwd rates via fwd_func and 2) fwd rates via disc_func
			if(!null(flt_fwd_func))
				return swap_fixflt(fifl, disc_func, disc_func, flt_fwd_func);
			else
				return swap_fixflt(fifl,disc_func, disc_func, flt_fwd_disc_func);
		}
		catch {
			CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt_plain");
			return null<swap_fixflt>;
		}
	}

	/*-----------------------------------------------------------------------
	  swap_fixflt_par
	  ----------------------------------------------------------------------*/
	
	/*swap_fixflt swap_fixflt(instr_class_name 	class_name, 
								instrument_name		instrument_name, 									
								date 				trade_date, 
								date				settle_date,
								date 				maturity,
								number				fix_rate,
								out instr_error option(nullable) error)
	
	{
		instr_def i_d 	= instr_def(class_name ,true , error);
		if(error.is_error()) return null<swap_fixflt>;
		swap_fixflt sw 	= ..swap_fixflt(	i_d, instrument_name,trade_date,settle_date, 
											maturity,fix_rate,100, error);
		if(error.is_error()) return null<swap_fixflt>;
	}*/

	/*-----------------------------------------------------------------------
	  Name :  	create_fixflt_par
	  Notes :          
	  Changes:
	  			Date    		Who     Detail
	  			?				MN		for backward compat
	  ----------------------------------------------------------------------*/
	swap_fixflt create_fixflt_par(	instr_class_name 	class_name,
									instrument_name		instrument_name, 									
									date 				trade_date, 
									date				settle_date,
									date 				maturity,
									number				fix_rate)
	{
		try{
			error_info ee = new error_info(true,false);
			instr_def i_d 	= instr_def(class_name , ee);
			
			//QL_FAIL_COND(error.is_error(), error.message());
			ee = new error_info(true,true);
			swap_fixflt sw 	= ..swap_fixflt(i_d, instrument_name,trade_date,settle_date,
												maturity,null<string>,fix_rate,100, ee);
			//QL_FAIL_COND(null(sw.yield()),"invalid yield");
			return sw;
		}
		catch {
			CORE_INT.write_warning_message(err.message());
			return null<swap_fixflt>;
		}
	}
	//-------------------------------------------------------
	// create_class_fixflt  for backward compat
	//-------------------------------------------------------
	swap_fixflt create_class_fixflt(string class_name)
	{
		try {
			//dates not matter since the swap serves only as a class template
			date d 	= #2011-03-12;
			date s 	= #2011-03-14;	
			date m 	= #2011-04-14;
	
			error_info ee = error_info(true,false);
			string name = strcat(class_name,"_");
			instr_def i_d 	= instr_def(class_name , ee);			
			//QL_FAIL_COND(error.is_error(), error.message());
			ee = new error_info(true,true);
			swap_fixflt sw 	= ..swap_fixflt(i_d, name,d,s,m,null<string>,0.01,100, ee);					
			return sw;
		}
		catch {
			CORE_INT.write_warning_message(err.message());
			return null<swap_fixflt>;
		}
	}
}

/*-----------------------------------------------------------------------
  func: swap_fixflt_plain
  minimal constructor for single curr "screen"-swap,
  this version: fwd_func input
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt_plain(	instr_def 					def,
								string 						instrument_name,	
								date option(nullable) 		trade_date,
								date option(nullable) 		settle_date,
								string option(nullable) 	spot_settle_code,
								date option(nullable) 		start_date,			//start_date has precedence over fwd_start_code
								string option(nullable) 	fwd_start_code,
								date option(nullable)		maturity,			//maturity has precedence over mat_code
								string 	option(nullable)	maturity_code,																													
								number option(nullable)		fix_coupon_rate ,	//if null --> will be set to par
								integer 					fix_r_rnd_dec,		// < 0 --> no rounding
								number 	 					notional,																										
								logical 					pay_fixed,
								..disc_func option(nullable) disc_func,								
								fwd_func option(nullable) 	flt_fwd_func,																																			
							    number option(nullable) 	curr_fix,
							    
								error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{
	return CORE_INT_FIXFLT.swap_fixflt_plain_int(def,null,null,instrument_name,trade_date,settle_date,spot_settle_code,start_date,
												 fwd_start_code,maturity,maturity_code,
												fix_coupon_rate ,fix_r_rnd_dec,notional,null, pay_fixed,disc_func,flt_fwd_func, null<disc_func>,
												curr_fix, error );
}

/*-----------------------------------------------------------------------
  func: swap_fixflt_plain
  minimal constructor for single curr "screen"-swap,
  this version: disc_func  input
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt_plain(	instr_def 					def,
								string 						instrument_name,	
								date option(nullable) 		trade_date,
								date option(nullable) 		settle_date,
								string option(nullable) 	spot_settle_code,
								date option(nullable) 		start_date,			//start_date has precedence over fwd_start_code
								string option(nullable) 	fwd_start_code,
								date option(nullable)		maturity,			//maturity has precedence over mat_code
								string 	option(nullable)	maturity_code,																													
								number option(nullable)		fix_coupon_rate ,	//if null --> will be set to par
								integer 					fix_r_rnd_dec,		// < 0 --> no rounding
								number 	 					notional,																										
								logical 					pay_fixed,
								..disc_func option(nullable) disc_func,								
								disc_func option(nullable) 	flt_fwd_disc_func,																																			
							    number option(nullable) 	curr_fix, 
								error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{
	return CORE_INT_FIXFLT.swap_fixflt_plain_int(def,null,null,instrument_name,trade_date,settle_date,spot_settle_code,start_date,fwd_start_code,maturity,maturity_code,
								 fix_coupon_rate ,fix_r_rnd_dec,notional,null, pay_fixed,disc_func,null<fwd_func>, flt_fwd_disc_func,
								 curr_fix, error );
}
/*-----------------------------------------------------------------------
  func: swap_fixflt_plain
  minimal constructor for single curr "screen"-swap,
  this version: template + fwd_func + fwd_start_code + maturity_code input
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt_plain(	INSTR_TMPL.swap_fixflt_def_tmpl	tmpl,
								string 						instrument_name,	
								date option(nullable) 		trade_date,	
								string option(nullable) 	fwd_start_code,		//if tmpl have a code it is ignored
								string option(nullable)		maturity_code,		//if tmpl have a code it is ignored																												
								number option(nullable)		fix_coupon_rate ,	//if null --> will be set to par
								integer 					fix_r_rnd_dec,		// < 0 --> no rounding
								number 	 					notional,																										
								logical 					pay_fixed,
								..disc_func option(nullable) disc_func,								
								fwd_func option(nullable) 	flt_fwd_func,																																			
							    number option(nullable) 	curr_fix, 
								error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{
	try{		
		return CORE_INT_FIXFLT.swap_fixflt_plain_int(null,null,tmpl,instrument_name,trade_date,null,null,null,fwd_start_code,null,maturity_code,
								 fix_coupon_rate ,fix_r_rnd_dec,notional,null, pay_fixed,disc_func,flt_fwd_func, null<disc_func>,
								 curr_fix, error );
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt_plain");
		return null<swap_fixflt>;
	}
}
/*-----------------------------------------------------------------------
  func: swap_fixflt_plain
  minimal constructor for single curr "screen"-swap,
  this version: template + fwd_func + maturity date input
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt_plain(	INSTR_TMPL.swap_fixflt_def_tmpl	tmpl,
								string 						instrument_name,	
								date option(nullable) 		trade_date,
								date option(nullable) 		settle_date,
								date option(nullable) 		start_date,			//if tmpl have a code it is ignored
								date option(nullable)		maturity,			//if tmpl have a code it is ignored																												
								number option(nullable)		fix_coupon_rate ,	//if null --> will be set to par
								integer 					fix_r_rnd_dec,		// < 0 --> no rounding
								number 	 					notional,																										
								logical 					pay_fixed,
								..disc_func option(nullable) disc_func,								
								fwd_func option(nullable) 	flt_fwd_func,																																			
							    number option(nullable) 	curr_fix, 
								error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{
	try{		
		return CORE_INT_FIXFLT.swap_fixflt_plain_int(null,null,tmpl,instrument_name,trade_date,settle_date,null,start_date,null,maturity,null,
								 fix_coupon_rate ,fix_r_rnd_dec,notional,null, pay_fixed,disc_func,flt_fwd_func, null<disc_func>,
								 curr_fix, error );
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt_plain");
		return null<swap_fixflt>;
	}
}
/*-----------------------------------------------------------------------
  func: swap_fixflt_plain
  cross ccy ok
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt_plain(	instr_def 					instr_def, 
								string 						name  ,	
								date option(nullable) 		trade_date ,		
								date option(nullable) 		settle_date ,
								string option(nullable) 	spot_settle_code,
								date  option(nullable) 		start_date,
								string option(nullable) 	fwd_start_code,
								date  option(nullable)		maturity,
								string 	option(nullable)	maturity_code,
								//integer						cpn_roll_day,	//<----------------
								logical 					pay_fixed,			
								number 	option(nullable) 	fx_quote,			//required if instr_def is a cross curr, as of settle	
								string 	option(nullable) 	fx_quote_baseccy,	//required if instr_def is a cross curr
								string 	option(nullable) 	fx_quote_priceccy,	//required if instr_def is a cross curr
								logical option(nullable) 	pv_in_base_ccy,		//defaults to true if null

								stub_type option(nullable)	fix_stub_type,
								date option(nullable) 		fix_eff_date,		
					    		date  option(nullable)		fix_first_cpn_date,	
					    		date  option(nullable)		fix_last_reg_date,													
								number option(nullable)		fix_coupon_rate ,		//will be implied if null
								integer 					fix_r_rnd_dec,			// < 0 --> no rounding
								number 	 					fix_notional,
								//number	option(nullable)	fix_pv01,							
								disc_func option(nullable)	fix_disc_func,		

								stub_type option(nullable)	flt_stub_type,
								date option(nullable) 		flt_eff_date,		
								date  option(nullable)		flt_first_cpn_date,	
					    		date  option(nullable)		flt_last_reg_date,													
								number 	option(nullable)	flt_notional,				
								number						flt_spread,			
								number option(nullable) 	flt_curr_fix,									
								disc_func option(nullable)	flt_disc_func,						
								fwd_func  option(nullable)	flt_fwd_func,					
								vector(date) option(nullable) 	flt_fixing_dates,	
								vector(number) option(nullable) flt_fixing_rates,	
								//number option(nullable) first_fixing,
							    //number option(nullable) last_fixing,
							    number option(nullable) 	flt_fix_proxy, 			
							    logical  option(nullable) 	flt_allow_fwd_fix,
								
								error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{	
	try{
		//create empty parm
		CORE_SWAPLIB.fixflt_parm fifl = new CORE_SWAPLIB.fixflt_parm();

		//update parm with init_basic_static
		fifl.init_static(instr_def,spot_settle_code);

		//basic checks after instr_def update
		QL_FAIL_COND(fifl.is_cross_currency() && null(fix_disc_func), "invalid fix leg disc_func", E_INIT);	
				
		number pv01_not_unit  	= 100000;//rounding unit for notional when implied from pv01					
		number	fix_pv01 		= null;
		number	fix_start_stub 		= null;
		number	fix_end_stub 		= null;
		//update parm with current input
		fifl.set_plain(	name,trade_date, settle_date,  start_date,fwd_start_code,maturity, maturity_code,pay_fixed,						
						fix_r_rnd_dec,pv01_not_unit,
						fx_quote,fx_quote_baseccy, fx_quote_priceccy,pv_in_base_ccy,						
						fix_coupon_rate,fix_stub_type,fix_eff_date, fix_first_cpn_date,fix_last_reg_date,
						fix_notional, fix_pv01,						
						flt_spread, flt_stub_type, flt_eff_date, flt_first_cpn_date, flt_last_reg_date, flt_notional, 
						flt_curr_fix, flt_fixing_dates , flt_fixing_rates, flt_fix_proxy , flt_allow_fwd_fix,fix_start_stub,fix_end_stub);
		
		swap_fixflt swap = fifl.create_swap(fix_disc_func, flt_disc_func, flt_fwd_func);


		return swap;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt_plain");
		return null<swap_fixflt>;
	}
}


/*-----------------------------------------------------------------------
  func: swap_fixflt_plain (no instr_def)
  cross ccy ok
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt_plain(	string 						name  ,
								date option(nullable) 		trade_date ,
								date option(nullable) 		settle_date ,
								string option(nullable)		spot_settle_code,
								date option(nullable) 		start_date,
								string option(nullable) 	fwd_start_code,
								date  option(nullable)		maturity,
								string option(nullable)		maturity_code,
								//integer					cpn_roll_day,
								logical 					pay_fixed,
								bd_convention				pmt_bus_day,
								calendar					pmt_calendar,
								logical						end_of_month,								
								notional_exchg_style option(nullable)  not_exchange,
								number 		option(nullable) 	fx_quote,//as of settle
								string 		option(nullable) 	fx_quote_baseccy,
								string 		option(nullable) 	fx_quote_priceccy,
								logical 	option(nullable) 	pv_in_base_ccy,
								
								string							fix_currency,
								stub_type option(nullable)		fix_stub_type,
								date option(nullable) 			fix_eff_date,
					    		date  option(nullable)			fix_first_cpn_date,
					    		date  option(nullable)			fix_last_reg_date,						
								day_count_method				fix_day_count,
								interest_rule 					fix_ir_rule,
								sink_fund_style option(nullable) fix_sink_fund_style,	
								vector(date) option(nullable) 	fix_sink_fund_dates,
								vector(number) option(nullable) fix_sink_fund,
								integer 						fix_coupon_freq ,
								number  						fix_coupon_rate ,
								integer 						fix_r_rnd_dec,// < 0 --> no rounding
								number 							fix_notional,
								disc_func option(nullable) 		fix_disc_func,
								
								string							flt_currency,
								stub_type option(nullable)		flt_stub_type,
								date option(nullable) 			flt_eff_date,
								date  option(nullable)			flt_first_cpn_date,
					    		date  option(nullable)			flt_last_reg_date,
								day_count_method				flt_day_count,
								interest_rule 					flt_ir_rule,
								ir_index 						flt_ir_index,
								integer 						flt_coupon_freq ,
								sink_fund_style option(nullable) flt_sink_fund_style,	
								vector(date) option(nullable) 	flt_sink_fund_dates,
								vector(number) option(nullable) flt_sink_fund,
								number 	option(nullable) 		flt_notional,
								number							flt_spread,			
								number option(nullable) 		flt_curr_fix,							
								disc_func option(nullable) 		flt_disc_func,	
								fwd_func option(nullable) 		flt_fwd_func,	
								vector(date) option(nullable) 	flt_fixing_dates,	
								vector(number) option(nullable) flt_fixing_rates,
								//number option(nullable) first_fixing,
							    //number option(nullable) last_fixing,
							    number option(nullable) 		flt_fix_proxy, 
							    logical  option(nullable) 		flt_allow_fwd_fix,
								error_info option(nullable) 	error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{	
	try{
		CORE_INT.reset_single(error);

		//create empty parm
		CORE_SWAPLIB.fixflt_parm fifl = new CORE_SWAPLIB.fixflt_parm();

		//update parm with init_basic_static
		flt_stub_fwd_style flt_stub_rate_style = null;
		fifl.init_static(	spot_settle_code,pmt_bus_day,pmt_calendar,end_of_month,not_exchange,										
							fix_currency, fix_coupon_freq, fix_day_count, fix_ir_rule,null,null,null,null,
							flt_currency, flt_coupon_freq, flt_coupon_freq, flt_day_count, flt_ir_rule,flt_ir_index,
							null,null,null,null,null,null,flt_stub_rate_style);
		
		//update parm with current input					
		//fifl.imp_fix_rate_dec  = fix_r_rnd_dec;//rounding decimals of the implied cpn rate (on decimal value)
		//fifl.pv01_not_unit  = 100000;//rounding unit for notional when implied from pv01

		number	flt_fix_start_stub = null;
		number	flt_fix_end_stub = null;
			
		logical is_ccy = !equal_casei(fix_currency, flt_currency);	
		if(null(fix_sink_fund_style) && null(flt_sink_fund_style)){
			QL_FAIL_COND(is_ccy && null(fix_disc_func), "invalid fix leg disc_func", E_INIT);
			number	fix_pv01 = null;
			
			
			fifl.set_plain(	name,trade_date, settle_date,  start_date,fwd_start_code,maturity, maturity_code,
							pay_fixed, -1,-1,fx_quote,fx_quote_baseccy, fx_quote_priceccy,pv_in_base_ccy,
							fix_coupon_rate, fix_stub_type,fix_eff_date, fix_first_cpn_date,fix_last_reg_date, 
							fix_notional, fix_pv01,  flt_spread, 
							flt_stub_type,flt_eff_date,flt_first_cpn_date,flt_last_reg_date, flt_notional, 
							flt_curr_fix, flt_fixing_dates , flt_fixing_rates, flt_fix_proxy , flt_allow_fwd_fix,
							flt_fix_start_stub,flt_fix_end_stub);
		}
		else {
			QL_FAIL_COND(is_ccy, "amortizing swaps only supported for single currency", E_INIT);
			fifl.set_plain_amort_single_curr(name,trade_date, settle_date,  start_date,fwd_start_code,maturity, maturity_code,
											 pay_fixed,-1,fix_notional, fix_coupon_rate, fix_stub_type,fix_eff_date,
											 fix_first_cpn_date,fix_last_reg_date,
											 fix_sink_fund_style,
											 fix_sink_fund_dates,
											 fix_sink_fund,
											 flt_spread, flt_stub_type,flt_eff_date,flt_first_cpn_date,flt_last_reg_date,
											 flt_sink_fund_style ,
											 flt_sink_fund_dates ,
											 flt_sink_fund ,
											 flt_curr_fix, flt_fixing_dates , flt_fixing_rates, flt_fix_proxy , flt_allow_fwd_fix,
											flt_fix_start_stub,flt_fix_end_stub);
		}
		
		swap_fixflt swap = fifl.create_swap(fix_disc_func, flt_disc_func, flt_fwd_func);

		return swap;
				
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt_plain");
		return null<swap_fixflt>;
	}
}

/*-----------------------------------------------------------------------
  func: swap_fixflt_plain
  minimal constructor for single curr "screen"-swap,
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt_plain(	swap_fixflt 				sw,
								string 						instrument_name,	
								date option(nullable) 		trade_date,
								date option(nullable) 		settle_date,
								string option(nullable) 	spot_settle_code,
								date option(nullable) 		start_date,			//start_date has precedence over fwd_start_code
								string option(nullable) 	fwd_start_code,
								date option(nullable)		maturity,			//maturity has precedence over mat_code
								string 	option(nullable)	maturity_code,																												
								number option(nullable)		fix_coupon_rate ,	//if null --> will be set to par
								integer 					fix_r_rnd_dec,		// < 0 --> no rounding
								number 	 					notional,																										
								logical 					pay_fixed,
								..disc_func option(nullable) disc_func,								
								fwd_func option(nullable) 	flt_fwd_func,																																			
							    number option(nullable) 	curr_fix,
 							 
								error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{
	return CORE_INT_FIXFLT.swap_fixflt_plain_int(null,sw,null,instrument_name,trade_date,settle_date,spot_settle_code,start_date,
												 fwd_start_code,maturity,maturity_code,
												fix_coupon_rate ,fix_r_rnd_dec,notional,null, pay_fixed,disc_func,flt_fwd_func, null<disc_func>,
												curr_fix, error );
}

/*-----------------------------------------------------------------------
  func: swap_fixflt_plain
  minimal constructor for single curr "screen"-swap,
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt_plain(	swap_fixflt 				sw,
								string 						instrument_name,	
								date option(nullable) 		trade_date,
								date option(nullable) 		settle_date,
								string option(nullable) 	spot_settle_code,
								date option(nullable) 		start_date,			//start_date has precedence over fwd_start_code
								string option(nullable) 	fwd_start_code,
								date option(nullable)		maturity,			//maturity has precedence over mat_code
								string 	option(nullable)	maturity_code,																												
								number option(nullable)		fix_coupon_rate ,//if null --> will be set to par
								integer 					fix_r_rnd_dec,// < 0 --> no rounding
								number 	 					notional,																										
								logical 					pay_fixed,
								..disc_func option(nullable) disc_func,								
								disc_func option(nullable) 	flt_disc_func,																																			
							    number option(nullable) 	curr_fix, 
								error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{
	return CORE_INT_FIXFLT.swap_fixflt_plain_int(null,sw,null,instrument_name,trade_date,settle_date,spot_settle_code,start_date,
												 fwd_start_code,maturity,maturity_code,
												fix_coupon_rate ,fix_r_rnd_dec,notional, null,pay_fixed,disc_func,null<fwd_func>, flt_disc_func,
												curr_fix, error );
}

/*-----------------------------------------------------------------------
  func: swap_fixflt_plain
  minimal constructor for single curr "screen"-swap,
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt_plain(	instrument 					fixflt_sw,															
								number 	 					notional,																										
								logical 					pay_fixed,
								..disc_func option(nullable) disc_func,								
								fwd_func option(nullable) 	flt_fwd_func,																																		
							    number option(nullable) 	curr_fix,							    
								error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{
	error_info ee 			= new error_info(true,true);
	swap_fixflt sw 			= swap_fixflt(fixflt_sw,ee);
	string instrument_name 	= sw.name(ee);	
	date trade_date 		= sw.trade_date(ee);
	date settle_date 		= sw.settle_date(ee);
	date start_date 		= sw.issue_date(ee);
	string fwd_start_code	= null<string>;
	date maturity 			= sw.maturity(ee);			
	string maturity_code 	= null<string>;																												
	number fix_coupon_rate 	= sw.quote(ee)/100.0;
	string spot_settle_code	= sw.settle_code(ee);//assume instrument is a screenswap with spot_settle_code
	return CORE_INT_FIXFLT.swap_fixflt_plain_int(null,sw,null,instrument_name,trade_date,settle_date,spot_settle_code,start_date,fwd_start_code,maturity,maturity_code,
												fix_coupon_rate ,-1,notional,null, pay_fixed,disc_func,flt_fwd_func, null<disc_func>,
												curr_fix,error );
}


/*-----------------------------------------------------------------------
  carry		EXT SWAP: ok
  Note: cannot be a member-func because we use swap_fixflt_plain(...) --> possible to fix i.e. FIX_ME
  ----------------------------------------------------------------------*/
void carry(	swap_fixflt sw,
			date 		fwd_settle_date,			
			out number  carry,
			out number  run_carry,			
			error_info option(nullable) error )
{
	
	try {
		error_info ee 				= new error_info(true,true);
		date fwd_trade_date 		= null<date>;
		carry_result res 			= new carry_result();
		logical roll_compat 		= false;
		sw.upfront_carry(fwd_trade_date,fwd_settle_date, res, roll_compat ,ee);

		number sw_rate 			= sw.coupon_fix_leg(ee);
		disc_func df			= sw.disc_func_fix_leg(ee);
		tenor_surface ts 		= sw.tenor_surface_flt_leg(ee);
		tenor_code tc			= sw.index_tenor_code_flt_leg(ee) ;
		fwd_func ff				= ts.fwd_curve_f(tc);
		
		swap_fixflt sw_fwd_stub	= swap_fixflt_plain(sw, "fwd", sw.trade_date(ee),fwd_settle_date,null,fwd_settle_date,
													null<string>,sw.maturity(),null<string>, sw_rate ,-1,									
													sw.notional(), sw.pay_fix(),df,ff, null<number>,ee);
				
		number pv01_fwd		= sw_fwd_stub.pv01(null<number>,null<number>, null<disc_func>,ee);
		
		//swap_fixflt sw_fwd = new swap_fixflt(this);
		//sw_fwd = sw_fwd.set_date(this.trade_date(ee),fwd_settle_date,ee);

		//number _pv01 = sw_fwd.pv01();
		logical use_ai_adj	= true;
		carry 				= use_ai_adj ? res.carry_ai_adj : res.carry;
		run_carry 			= carry / pv01_fwd;
		/*number cpn = this.coupon(ee);
		number fwd_npv = npv_fwd_fix_leg+npv_fwd_flt_leg;
		number rf = sw_fwd.imp_rate_fix_leg(fwd_npv,false,ee);
		number ccy = (rf-cpn)*10000;*/
			
			
		return ;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "carry");
		carry = run_carry = null<number>;
		return ;
	}
}

/*-----------------------------------------------------------------------
  roll_down		EXT SWAP: ok
  Note: cannot be a member-func because we use swap_fixflt_plain(...) --> possible to fix i.e. FIX_ME
  ----------------------------------------------------------------------*/
void roll_down(	swap_fixflt sw,
				date 		fwd_settle_date,				
				out number  roll,
				out number  run_roll,				
				error_info option(nullable) error )
{
	
	try {
		error_info ee 				= new error_info(true,true);
		date fwd_trade_date 		= null<date>;		
		logical incl_reinv_in_roll 	= false;
		rolldown_result res			= new rolldown_result();
		sw.upfront_rolldown(fwd_trade_date,fwd_settle_date,	 res, incl_reinv_in_roll,ee);

		number sw_rate 		= sw.coupon_fix_leg(ee);
		disc_func df		= sw.disc_func_fix_leg(ee);
		tenor_surface ts 	= sw.tenor_surface_flt_leg(ee);
		tenor_code tc		= sw.index_tenor_code_flt_leg(ee) ;
		fwd_func ff			= ts.fwd_curve_f(tc);
		
		swap_fixflt sw_fwd_stub	= swap_fixflt_plain(sw, "fwd_stub", fwd_trade_date,fwd_settle_date,null,fwd_settle_date,
													null<string>,sw.maturity(),null<string>, sw_rate ,	-1,								
													sw.notional(), sw.pay_fix(),df,ff, null<number>,ee);		
		
		number pv01_fwd			= sw_fwd_stub.pv01(null<number>,null<number>, null<disc_func>,ee);

		logical use_ai_adj 		= true;
		roll 					= use_ai_adj ? res.roll_ai_adj : res.roll;
		run_roll 				= roll / pv01_fwd;
			
		return ;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "roll");
		roll = run_roll = null<number>;
		return ;
	}
}


/*-----------------------------------------------------------------------
  carry_roll_down		EXT SWAP: ok
  Note: cannot be a member-func because we use swap_fixflt_plain(...) --> possible to fix i.e. FIX_ME
  ----------------------------------------------------------------------*/
void carry_roll_down(	swap_fixflt sw,
						date 		fwd_settle_date,
						out number  carry_roll,
						out number  run_carry_roll,
						out number  carry,
						out number  run_carry,
						out number  carry_imp,
						out number  run_carry_imp,
						out number  roll,
						out number  run_roll,
						out logical ai_adjusted,
						error_info option(nullable) error )
{
	
	try {
		error_info ee 			= new error_info(true,true);
		date fwd_trade_date 	= null<date>;
		
		sw.upfront_carryroll(fwd_trade_date,fwd_settle_date,carry_roll,roll,carry_imp,ai_adjusted,ee );
		//roll --> roll_fwd_ai_adj
		..carry(sw,fwd_settle_date,carry,run_carry,ee );
		
		number sw_rate 		= sw.coupon_fix_leg(ee);
		disc_func df		= sw.disc_func_fix_leg(ee);
		tenor_surface ts 	= sw.tenor_surface_flt_leg(ee);
		tenor_code tc		= sw.index_tenor_code_flt_leg(ee) ;
		fwd_func ff			= ts.fwd_curve_f(tc);
		
		swap_fixflt sw_roll_fwd_stub = swap_fixflt_plain(sw, "fwd_stub", fwd_trade_date,fwd_settle_date,null,fwd_settle_date,
														null<string>,sw.maturity(),null<string>, sw_rate ,-1,									
														sw.notional(), sw.pay_fix(),df,ff, null<number>,ee);		
		
		number pv01_fwd = sw_roll_fwd_stub.pv01(null<number>,null<number>, null<disc_func>,ee);
		run_roll 		= roll / pv01_fwd;
		run_carry_roll	= carry_roll / pv01_fwd;
		run_carry_imp	= carry_imp/ pv01_fwd;
		
		return ;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "roll");
		carry_roll = run_carry_roll = carry = run_carry = roll = run_roll = carry_imp = run_carry_imp = null<number>;
		return ;
	}
}

/*-----------------------------------------------------------------------
  carry_roll_down		EXT SWAP: ok
  Note: cannot be a member-func because we use swap_fixflt_plain(...) --> possible to fix i.e. FIX_ME
  ----------------------------------------------------------------------*/
void carry_roll_down(	swap_fixflt sw_pay,
						swap_fixflt sw_rec,
						date 		fwd_settle_date,
						logical 	use_pv01_pay,
						out number  carry_roll,
						out number  run_carry_roll,
						out number  carry,
						out number  run_carry,
						out number  carry_imp,
						out number  run_carry_imp,
						out number  roll,
						out number  run_roll,
						out logical ai_adjusted,
						error_info option(nullable) error )
{
	
	try {
		error_info ee 				= new error_info(true,true);
		date fwd_trade_date 		= null<date>;

		QL_REQUIRE(sw_pay.notional() == sw_rec.notional(),"notional must be the same in both swaps");
		
		number carry_rf_pay,roll_f_pay,carry_imp_f_pay;
		logical ai_adj_pay;
		sw_pay.upfront_carryroll(fwd_trade_date,fwd_settle_date,carry_rf_pay,roll_f_pay,carry_imp_f_pay,ai_adj_pay,ee );
		//roll --> roll_fwd_ai_adj
		number carry_f_pay,run_carry_f_pay;
		..carry(sw_pay,fwd_settle_date,carry_f_pay,run_carry_f_pay,ee );

		number carry_rf_rec,roll_f_rec,carry_imp_f_rec;
		logical ai_adj_rec;		
		sw_rec.upfront_carryroll(fwd_trade_date,fwd_settle_date,carry_rf_rec,roll_f_rec,carry_imp_f_rec,ai_adj_rec,ee );
		//roll --> roll_fwd_ai_adj
		number carry_f_rec,run_carry_f_rec;
		..carry(sw_rec,fwd_settle_date,carry_f_rec,run_carry_f_rec,ee );

		carry_roll			= carry_rf_pay 		+ carry_rf_rec;
		carry				= carry_f_pay 		+ carry_f_rec;
		carry_imp			= carry_imp_f_pay	+ carry_imp_f_rec;
		roll				= roll_f_pay		+ roll_f_rec;

		swap_fixflt	sw_risk	= use_pv01_pay ? sw_pay: sw_rec;
		
		number sw_rate 		= sw_risk.coupon_fix_leg(ee);
		disc_func df		= sw_risk.disc_func_fix_leg(ee);
		tenor_surface ts 	= sw_risk.tenor_surface_flt_leg(ee);
		tenor_code tc		= sw_risk.index_tenor_code_flt_leg(ee) ;
		fwd_func ff			= ts.fwd_curve_f(tc);
		
		swap_fixflt sw_roll_fwd_stub = swap_fixflt_plain(sw_risk, "fwd_stub", fwd_trade_date,fwd_settle_date,null,fwd_settle_date,
														null<string>,sw_risk.maturity(),null<string>, sw_rate ,	-1,								
														sw_risk.notional(), sw_risk.pay_fix(),df,ff, null<number>,ee);		
		
		number pv01_roll_fwd = sw_roll_fwd_stub.pv01(null<number>,null<number>, null<disc_func>,ee);

		run_carry			= carry / pv01_roll_fwd;
		run_roll 			= roll / pv01_roll_fwd;
		run_carry_roll		= carry_roll / pv01_roll_fwd;
		run_carry_imp		= carry_imp/ pv01_roll_fwd;
		
		return ;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "roll");
		carry_roll = run_carry_roll = carry = run_carry = roll = run_roll = carry_imp = run_carry_imp = null<number>;
		return ;
	}
}

module CORE_INT_FIXFLT
{
	//-------------------------------------------------------
	//  carry_roll_fwd_swap_cf
	//-------------------------------------------------------
	void carry_roll_fwd_swap_cf(disc_func 	df_ois,					
								swap_fixflt swap_orig_fwd,
								swap_fixflt swap_stub_fwd,
								out number pv_ann_fix,
								out number pv_ann_flt,
								out number pv_ai_fix,
								out number pv_ai_flt)
	{
		error_info e 		= error_info(true,true);

		date carry_settle 	= swap_stub_fwd.settle_date(e);
		date td 			= swap_stub_fwd.trade_date(e);

		number ai_fix_fwd,ai_flt_fwd;
		swap_orig_fwd.accrued(false,ai_fix_fwd,ai_flt_fwd,e);
		
		vector(date) pmt_fix, pmt_flt;
		vector(number) cf_fix, cf_flt;
		swap_orig_fwd.cash_flow_data(true, ir_cf_code.TOTAL,pmt_fix, cf_fix, pmt_flt, cf_flt,e);

		vector(date) pmt_fix_f, pmt_flt_f;
		vector(number) cf_fix_f, cf_flt_f;
		swap_stub_fwd.cash_flow_data(true, ir_cf_code.TOTAL,pmt_fix_f, cf_fix_f, pmt_flt_f, cf_flt_f,e);

		
		integer size_fix = v_size(pmt_fix);
		integer size_flt = v_size(pmt_flt);
		
		integer size_fix_f = v_size(pmt_fix_f);
		integer size_flt_f = v_size(pmt_flt_f);

		vector(number) cf_diff_fix[size_fix_f];
		vector(number) cf_diff_flt[size_flt_f];

		vector(number) df_fix[size_fix_f];
		vector(number) df_flt[size_flt_f];
		
		vector(number) pv_cf_diff_fix[size_fix_f];
		vector(number) pv_cf_diff_flt[size_flt_f];
		
		number dfs_fwd 	= df_ois.disc((carry_settle-td)/365.0);
		integer j 		= 0;

		for(integer i=size_fix_f-1;i>=0;i--) {

			cf_diff_fix[i] 		= cf_fix_f[i] + cf_fix[size_fix-j-1];

			number t 			= (pmt_fix_f[i]-td)/365.0;
			df_fix[i] 			= df_ois.disc(t)/dfs_fwd;
			pv_cf_diff_fix[i]	= cf_diff_fix[i] * df_fix[i];
			j++;
		}
		pv_ai_fix 			= ai_fix_fwd * df_fix[0];
		pv_ann_fix 			= v_sum(pv_cf_diff_fix);

		j = 0;
		for(integer i=size_flt_f-1;i>=0;i--) {
			
			cf_diff_flt[i] 		= cf_flt_f[i] + cf_flt[size_flt-j-1];		
			number t 			= (pmt_flt_f[i]-td)/365.0;
			df_flt[i] 			= df_ois.disc(t)/dfs_fwd;
			pv_cf_diff_flt[i]	= cf_diff_flt[i] * df_flt[i];
			
			j++;
		}
		
		pv_ai_flt 			= ai_flt_fwd* df_flt[0];
		pv_ann_flt 			= v_sum(pv_cf_diff_flt);//carry: the first flt can differ due to different fixing rate in the fwd swap
	}

	//-------------------------------------------------------
	//  carry_fwd_swap
	//-------------------------------------------------------
	void carry_fwd_swap(swap_fixflt swap_orig,
						date carry_settle,
						disc_func 	df_ois,
						fwd_func 	fwd,
						out swap_fixflt swap_stub_fwd,
						out swap_fixflt swap_orig_fwd,
						out number sw_fwd_stub_be,
						out number pv01f_stub,
						out number pv_ann_fix,
						out number pv_ann_flt,
						out number pv_ai_fix,
						out number pv_ai_flt)
	{
		error_info e 		= error_info(true,true);
		
		date td 			= swap_orig.trade_date();	
		swap_stub_fwd 		= swap_fixflt_plain(swap_orig, "fwd_carry_stub", td,carry_settle,"BD2",carry_settle,
														null<string>, swap_orig.maturity(), null<string>, null ,									
														-1,swap_orig.notional(), !swap_orig.pay_fix(), df_ois,fwd, null<number>,e);
		
		pv01f_stub			= swap_stub_fwd.pv01(1000000,null<number>, null<disc_func>,e);
		sw_fwd_stub_be		= swap_stub_fwd.coupon_rate(e)*100;//sw_fwd_stub.imp_rate_fix_leg(0,false,e)*100;

		swap_orig_fwd 		= swap_orig.set_settle_date(carry_settle,e);

		carry_roll_fwd_swap_cf(df_ois,swap_orig_fwd,swap_stub_fwd,
							pv_ann_fix,pv_ann_flt,pv_ai_fix,pv_ai_flt);
	}

	//-------------------------------------------------------
	//  roll_fwd_swap
	//-------------------------------------------------------
	void roll_fwd_swap(	swap_fixflt swap_orig,
						date 		carry_trade_date,
						date 		carry_settle,
						disc_func 	df_ois,
						fwd_func 	fwd,
						out swap_fixflt swap_stub_fwd,
						out swap_fixflt swap_orig_fwd,
						out number sw_fwdr_stub_be,
						out number pv01r_stub,
						out number pv_ann_fix,
						out number pv_ann_flt,
						out number pv_ai_fix,
						out number pv_ai_flt )
	{
		error_info e 			= error_info(true,true);

		swap_stub_fwd 			= swap_fixflt_plain(swap_orig, "fwd_roll_stub", carry_trade_date,carry_settle,"BD2",carry_settle,
											null<string>, swap_orig.maturity(), null<string>, null ,									
											-1,swap_orig.notional(), !swap_orig.pay_fix(), df_ois,fwd, null<number>,e);

		pv01r_stub				= swap_stub_fwd.pv01(1000000,null<number>, null<disc_func>,e);
		sw_fwdr_stub_be			= swap_stub_fwd.coupon_rate(e)*100;//fwd_swap_stub.imp_rate_fix_leg(0,false,e)*100;

		date spot_start_date 	= swap_orig.start_date(e);
		number fix_cpn 			= swap_orig.coupon_rate(e);
		swap_orig_fwd 			= swap_fixflt_plain(swap_orig, "fwd_roll", carry_trade_date,carry_settle,"BD2",spot_start_date,
													null<string>, swap_orig.maturity(), null<string>, fix_cpn ,									
													-1,swap_orig.notional(), swap_orig.pay_fix(), df_ois,fwd, null<number>,e);

		date next_fix, next_flt;
		swap_orig.next_cpn_date(next_fix, next_flt,e);

		date next_fwd_fix, next_fwd_flt;
		swap_orig_fwd.next_cpn_date(next_fwd_fix, next_fwd_flt,e);

		if(next_fwd_flt > next_flt) {
			date prev_fwd_fix, prev_fwd_flt;
			swap_orig_fwd.previous_cpn_date(prev_fwd_fix, prev_fwd_flt,e);
			ir_index idx = swap_orig_fwd.ir_index_flt_leg(e);
			date fixd = idx.fixing_date(prev_fwd_flt);
			number per = (prev_fwd_flt-fixd)/365.0;
			number curr_fix = fwd.fwd(per);
			swap_orig_fwd.add_current_fixing(curr_fix);
		}
		else {
			date resetd;
			logical isfix;
			number curr_fix = swap_orig.current_fixing_flt_leg(resetd, isfix,e);
			swap_orig_fwd.add_current_fixing(curr_fix);
		}

		carry_roll_fwd_swap_cf(df_ois,swap_orig_fwd,swap_stub_fwd,
								pv_ann_fix,pv_ann_flt,pv_ai_fix,pv_ai_flt);

	}
}