option(null: hard);	
/*	
	instrument wrapper functions
	Developer: Algorithmica Research, Magnus Nyström			

	--------------------------------------------------------------------
	swap_il_zero
	--------------------------------------------------------------------
	Zero-coupon swap. A zero-coupon swap is the exchange of two  flows on one given date tp,
	which is a certain number of years n after the start date ts, on a reference notional N, with one
	flow a fixed amount. The amount is quoted through a compounded annual rate. For a rate R, the
	amount paid is N((1 + R)^n- X) with X = 0 if the notional is exchanged and 1 otherwise. The standard 
	is no exchange of notional. The inflation flow is given by the change of the price index. The amount paid is
	N * ( ReferenceIndexEnd / ReferenceIndexStart) - X
	
*/


//-----------------------------------------------------------------------
//  class swap_il_zero
//----------------------------------------------------------------------
class swap_il_zero: public swap_gen
option(category: "Instrument/Inflation Linked/Zero Swap")
//option(allow_undeclared_mfuncs)
{
public:

	//override string			instr_type_s();
	override instrument 	inst();
	override swap_il_zero 	clone();

	override void  					__dbg_print(__dbg_label);
    override void   				__dbg_browse(__dbg_split);

	/*----------ADD FUNCS [PUBLIC]---------------*/

	void 		add_quote(number,..quote_style );//???i.e. means add new cpn --> should be a new instrument
	override void 		add_quote(number);	
	void 		add_notional(number );
	void 		add_notional(number , number);
	void 		add_disc_func(disc_func option(nullable)); 
	void 		add_disc_func_zero_leg(disc_func option(nullable));
	void 		add_disc_func_infl_index_leg(disc_func option(nullable));	
	void 		add_cpi_link(..cpi_link);
	void 		add_cpi_link_forecast(..cpi_link);
	
	/*----------MISC FUNCS [PUBLIC]---------------*/
	logical		pay_inflation(error_info option(nullable) error = null<error_info>);
	number 		notional(error_info option(nullable) error = null<error_info>);
	
	/*----------CASHFLOW FUNCS [PUBLIC]---------------*/

	//vector(number)	cash_flow_nom_amounts(..cpi_link ,number option(nullable) nominal = null<number>,
											//logical post_settle = true,error_info option(nullable) error = null<error_info> );

	void 		data(out swap_fix_zero_leg_data,out swap_zero_infl_leg_data,error_info option(nullable) error = null<error_info>);
	
	
	/*----------PV FUNCS [PUBLIC]---------------*/
	number  	present_value(	disc_func option(nullable),out number ,
											out number ,number option(nullable) ,
											logical option(nullable) trade_date_pv = false,		
											error_info option(nullable) error = null<error_info>);
	
	number  	present_value(	disc_func option(nullable),out number ,
											out number,date option(nullable) ,date option(nullable) ,
											logical,logical  ,error_info option(nullable) error = null<error_info>);
	
	override number  	present_value(	out number ,out number ,
											logical option(nullable) ,error_info option(nullable) error = null<error_info> );

	number		pv01(	number option(nullable),number option(nullable),
									disc_func option(nullable) df = null<disc_func>,
									error_info option(nullable) error = null<error_info>);
	
	/*----------INFL-ZERO-RATE LEG FUNCS [PUBLIC]---------------*/

	disc_func 	disc_func_zero_leg(error_info option(nullable) error = null<error_info> ) ;			
	number		infl_zero_rate(error_info option(nullable) error = null<error_info> );
	number		imp_infl_zero_rate(number,logical option(nullable) trade_date_pv = false,
											 error_info option(nullable) error = null<error_info>);	
	/*----------INFL-INDEX-LEG FUNCS [PUBLIC]---------------*/
	number 		base_index(error_info option(nullable) error = null<error_info> );
	void 		implied_index(number option(nullable), vector(number) ,date,logical option(nullable),
							   out date,out date,out number ,out number,out number,error_info option(nullable) error = null<error_info> );
		
	void 		implied_index(number option(nullable),  out date,out date,out number ,out number,error_info option(nullable) error = null<error_info> );
		
	disc_func 	disc_func_infl_leg(error_info option(nullable) error = null<error_info> ) ;		
	integer 	index_freq(	error_info option(nullable) error = null<error_info> ); 
	//number 			index_factor(error_info option(nullable) error = null<error_info> ); 
	//number 			index_factor(date option(nullable),date,error_info option(nullable) error = null<error_info> );
	
	infl_index_method index_method(error_info option(nullable) error = null<error_info> );
	logical 		is_noip_index_method(error_info option(nullable) error = null<error_info>);
	
	..cpi_link 		cpi_link(error_info option(nullable) error = null<error_info> );
	..cpi_link 		cpi_link_forecast(error_info option(nullable) error = null<error_info> );	
	void  			cpi_knots(	logical ,out date ,out date  ,out logical 	,
								out logical  	,error_info option(nullable) error = null<error_info> );
	void  			cpi_values(logical,out date ,out date  ,out number  ,out number   ,error_info option(nullable) error = null<error_info> ); 
	
	//-----external swap specific-----
	override instrument_type 		swap_type(error_info option(nullable) error = null<error_info> );
	vector(ql_swap_leg) 	legs(logical,error_info option(nullable) error = null<error_info>);
	ql_zero_infl_leg 		infl_index_leg(logical,error_info option(nullable) error = null<error_info>);
	ql_fix_zero_leg 		infl_zero_rate_leg(logical,error_info option(nullable) error = null<error_info>);
	override ql_fixed_income_swap 	swap_instrument(error_info option(nullable) error = null<error_info>) ;

	/*---set functions-required--*/
	override swap_il_zero	set_quote(number option(nullable),error_info option(nullable) error = null<error_info>);
	override swap_il_zero	set_quote(string option(nullable),error_info option(nullable) error = null<error_info>);
	
	/*-----set functions-if applicable-----*/
	swap_il_zero			set_infl_rate_zero_leg(number option(nullable) ,error_info option(nullable) error = null<error_info>);
	swap_il_zero			set_pay_leg(logical ,error_info option(nullable) error = null<error_info>);
		
	//local set funcs
	swap_il_zero 			set_date(date,date option(nullable) settle_date=null<date>, error_info option(nullable) error = null<error_info>);
	override swap_il_zero 	set_date(date,date option(nullable),logical,logical , error_info option(nullable) error = null<error_info> );
	
	
	/*----------HIDDEN FUNCS [PUBLIC]-(mostly for backward compat)--------------*/
	swap_il_zero(__instrument);
	
	//override number 		nominal(error_info option(nullable) error = null<error_info> );					/*hidden, visible in root*/
	
	

	swap_il_zero(swap_il_zero);
protected:
	override swap_il_zero		create(__instrument option(nullable),out instr_error option(nullable),error_type type = E_INVALID_ARG);  
	swap_il_zero 				err_type(swap_il_zero option(nullable), error_info option(nullable) error = null<error_info>);

	
	override number 		pvbp(number option(nullable),number option(nullable),logical option(nullable),
								logical option(nullable),	error_info option(nullable) error = null<error_info>) ;
	
	override number 		pvbp(number option(nullable),number  option(nullable),
								 error_info option(nullable) error = null<error_info>);
	
	//set funcs
	override swap_il_zero	set_date(date,logical ,error_info option(nullable) error = null<error_info> );							/*legacy*/
	
	override swap_il_zero 	move_date(	date,date option(nullable) settle_date=null<date> ,
										error_info option(nullable) error = null<error_info>);
	
	override swap_il_zero	set_yield(number option(nullable) ,error_info option(nullable) error = null<error_info> );
	override swap_il_zero	set_quote_from_yield(number option(nullable),error_info option(nullable) error = null<error_info> );		/*legacy*/
	
	//logical ext_swap_;
	//logical ccy_swap_;
	//logical full_swap_;
 
};

//------------------------------------------------
// __dbg_print
//------------------------------------------------
void swap_il_zero.__dbg_print(__dbg_label l)
{
	error_info ee 	= new error_info(true,false);
    l.set_text(strcat([
						"name: ",
						name(ee),
						",  ext_swap: ",
						null(ext_swap_) ? "" : string(ext_swap_),
                        ", ccy_swap: ",
						null(ccy_swap_) ? "" : string(ccy_swap_),
						", valid: ",
						null(this.is_valid(ee)) ? "" : string(this.is_valid(ee))
					]));
}
//------------------------------------------------
// __dbg_browse
//------------------------------------------------
void swap_il_zero.__dbg_browse(__dbg_split s)
{
    s.resize(4);
	error_info ee 	= new error_info(true,false);
	
    s.set_text(0, "name");
    s.set_value(0, this.name(ee));

    s.set_text(1, "ext_swap");
    s.set_value(1, ext_swap_);

    s.set_text(2, "ccy_swap");
    s.set_value(2, ccy_swap_);

	s.set_text(3, "valid");
    s.set_value(3, is_valid(ee));
}
//------------------------------------------------
// copy constructor
//------------------------------------------------
swap_il_zero.swap_il_zero(swap_il_zero c) : swap_gen(c)
{}
//------------------------------------------------
// constructor [HIDDEN, cannot be protected]
//------------------------------------------------
swap_il_zero.swap_il_zero(__instrument i) option(hidden) : swap_gen(i)
{
	//try {
		error_info ee = new error_info(true,false);
		if(!this.is_valid(ee)) {
			ext_swap_ = ccy_swap_ = full_swap_ = false;
			return;
		}
		ext_swap_ = this.is_ext_swap(ee);
		ccy_swap_ = this.is_ccy_swap(ee);
		full_swap_	= this.verify_full_swap(true,ee);
		
			
		return;									
	//}
	//catch {		
	//	CORE_INT.write_warning_message(err.message());
	//}
}
//------------------------------------------------
// instr_type_s
//------------------------------------------------
//string swap_il_zero.instr_type_s()  { return string(..instr_type.IL_SWAP_ZERO);}
//------------------------------------------------
// inst
//------------------------------------------------
instrument 	swap_il_zero.inst()		{ return this;}

//------------------------------------------------
// copy constructor <FUNCTION>
//------------------------------------------------
swap_il_zero swap_il_zero(swap_il_zero b)			{ return new swap_il_zero(b);}
//------------------------------------------------
// clone
//------------------------------------------------
swap_il_zero swap_il_zero.clone() 	{ return new swap_il_zero(this);}
//------------------------------------------------
// create  [PROTECTED]
// create swap_il_zero from internal instrument
//------------------------------------------------
swap_il_zero swap_il_zero.create(	__instrument option(nullable) c,
									out instr_error option(nullable) error,
									error_type type )
{
	QL_FAIL_COND(null(c), "invalid/unknown (null) instrument",type );

	logical valid = c.is_valid(error);
	if( !valid && INSTR_CREATE_NULL_ERR)
		return null<swap_il_zero>; 
	
	..instr_type it = c.instr_type(error);
	QL_FAIL_COND(null(it), "invalid instrument type (unknown)",type );
	QL_FAIL_COND(it != ..instr_type.IL_SWAP_ZERO, "invalid instrument type",type );//should not happen
	
	return new swap_il_zero(c);
}
/*-----------------------------------------------------------------------
  create_swap_il_zero  <FUNCTION>
  ----------------------------------------------------------------------*/
swap_il_zero create_swap_il_zero(	__instrument option(nullable) c,
									out instr_error option(nullable) error,
									error_type type = E_INVALID_ARG)
option(com_name: 'INTERNAL_create_swap_il_zero')
{
	QL_FAIL_COND(null(c), "invalid/unknown (null) instrument",type );

	logical valid = c.is_valid(error);
	if(!valid && INSTR_CREATE_NULL_ERR)
		return null<swap_il_zero>; 
	
	..instr_type it = c.instr_type(error);
	QL_FAIL_COND(null(it), "invalid instrument type (unknown)",type );
	QL_FAIL_COND(it != ..instr_type.IL_SWAP_ZERO, "invalid instrument type",type );//should not happen
	
	return new swap_il_zero(c);
}

swap_il_zero create_swap_il_zero(	__instrument option(nullable) 		c,
									error_info option(nullable) 		error,
									error_type 							type = E_INIT)
option(com_name: 'INTERNAL_create_swap_il_zero_ei')
{
	instr_error ee 	= instr_error();
	swap_il_zero b 	= create_swap_il_zero(c,ee, E_INIT);
	if(ee.is_error())
		CORE_INT.add_error_info(error,ee.type(),ee.message(), "create_swap_il_zero");
	return b;
}
//-----------------------------------------------
// constructor <FUNCTION> dynamic cast  EXT SWAP: ok
//-----------------------------------------------
swap_il_zero	swap_il_zero(instrument i, error_info option(nullable) error = null<error_info>) 	
option (category: 'Instrument/Inflation Linked/Zero Swap')
option(com_name: 'swap_il_zero_dyncast') 
{ 
	try {
		swap_il_zero d = dynamic_cast<swap_il_zero>(i); 
		if(null(d))
			CORE_INT.add_error_info(error,ERR_T_INIT,"invalid cast (instrument is not an swap_il_zero)","swap_il_zero" );

		return d;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero");
		return null<swap_il_zero>;
	}
}

/*-----------------------------------------------------------------------
  err_type	EXT SWAP: ok
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.err_type(swap_il_zero option(nullable) nul, error_info option(nullable) error )
{	
	CORE_INT.add_error_info(error,ERR_T_CALC,"mwlib(ql) error: inapplicable function call for current instrument type");
	return null<swap_il_zero>;
}


/*-----------------------------------------------------------------------
  add_quote		EXT SWAP: ok
  ----------------------------------------------------------------------*/
void swap_il_zero.add_quote(	number  quote ) 
{	
	instrument.add_quote(quote, true);
	return ;						
}
/*-----------------------------------------------------------------------
  add_quote		EXT SWAP: ok
  ----------------------------------------------------------------------*/
void swap_il_zero.add_quote(	number  quote,
							..quote_style quote_style) 
{
	error_info ee = new error_info(true,true);
	..quote_style qs = instrument.quote_style_e(ee);
	
	if(quote_style == qs)
		this.add_quote(quote);
	else{
		if(quote_style == ..quote_style.YIELD)
			this.add_quote(quote*100);//instr quotestyle must be pct
		else if(quote_style == ..quote_style.YIELD_PCT)
			this.add_quote(quote/100);//instr quotestyle is not pct
		else
			QL_FAIL("invalid quote style",this,true);
	}					
}
/*-----------------------------------------------------------------------
  add_quote_from_yield
  ----------------------------------------------------------------------*/

/*void swap_il_zero.add_quote_from_yield(	number  yield,
										logical option(nullable) set_to_par )
{
	QL_FAIL_COND(!null(set_to_par) && !set_to_par,"set_to_par = false is not supported for swaps",this,true);
	instrument.add_quote_from_yield(yield, set_to_par );
}*/
/*-----------------------------------------------------------------------
  swap_il_zero: add_funcs   EXT SWAP: ok
  ----------------------------------------------------------------------*/
void swap_il_zero.add_notional(number notional)								{ swap_gen.swap_add_notional(notional);}		//if ccy swap: err
void swap_il_zero.add_notional(number notional1, number notional2)	{ swap_gen.swap_add_notional(notional1, notional2);}
void swap_il_zero.add_disc_func(disc_func option(nullable) df)				{ swap_gen.swap_add_disc_func(df); }	//if ccy swap: err
void swap_il_zero.add_disc_func_zero_leg(disc_func option(nullable) df)		{ swap_gen.swap_add_disc_func_leg1(df); } //if non ccy swap: both legs will be updated
void swap_il_zero.add_disc_func_infl_index_leg(disc_func  option(nullable) df) 	{ swap_gen.swap_add_disc_func_leg2(df); }	//if non ccy swap: both legs will be updated
/*-----------------------------------------------------------------------
  add_cpi_link	EXT SWAP: ok
  ----------------------------------------------------------------------*/
void swap_il_zero.add_cpi_link(..cpi_link cpi_link)
{	
	error_info ee = error_info(true,true);
	ql_zero_infl_leg l = infl_index_leg(false,ee );			
	l.set_cpi(cpi_link);
	return;
	
}
/*-----------------------------------------------------------------------
  add_cpi_link_forecast	EXT SWAP: ok
  ----------------------------------------------------------------------*/
void swap_il_zero.add_cpi_link_forecast(..cpi_link cpi_link)
{
	error_info ee = error_info(true,true);
	ql_zero_infl_leg l = infl_index_leg(false,ee );			
	l.set_cpi_forecast(cpi_link);
	return;
}
/*-----------------------------------------------------------------------
  notional	EXT SWAP: ok
  ----------------------------------------------------------------------*/
/*void  swap_il_zero.notional(out number notional_fix_leg, 
							out number notional_flt_leg, 
							error_info option(nullable) error ) 
{	
	try{	
		instr_error_type t;
   		string 			s;	
		i().__swap_notional(notional_fix_leg,notional_flt_leg,t, s);
		logical e = CORE_INT.add_error_info(error,t,s, "swap_il_zero.notional");
		if(e)
			notional_fix_leg = notional_flt_leg = null<number >;
		return;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.notional");
		notional_fix_leg = notional_flt_leg = null<number >;
		return ;
	}
}*/


/*-----------------------------------------------------------------------
  notional	EXT SWAP: ok
  ----------------------------------------------------------------------*/
number  swap_il_zero.notional(	error_info option(nullable) error ) 
{	
	return swap_gen.nominal(error);
}

/*-----------------------------------------------------------------------
  swap_type   EXT SWAP: ok
  ----------------------------------------------------------------------*/
instrument_type swap_il_zero.swap_type(error_info option(nullable) error ) 
{	
	return swap_gen.swap_type(error);
}

/*-----------------------------------------------------------------------
  pv01	EXT SWAP: ok  - always returns a positive number regardless of pay/receive and sign of notional
  Note: not a "true" measure of risk, does not take into account if the swap has a pv <> 0
   PV01 refers to present value of 1 basis point and it's the discounted value of the cashflows
  for a rate of 0.01% for all periods of a particular instrument, ie, the npv of the fixed leg with a rate of 0.01%
  ----------------------------------------------------------------------*/
number  swap_il_zero.pv01(	number option(nullable) notional,
							number option(nullable) delta,
							//logical option(nullable) trade_date_pv,
							disc_func option(nullable) df,
							error_info option(nullable) error ) 
{	
	try{	
		instr_error_type t;
   		string 			s;	

		error_info ee 		= new error_info(true,true);
		disc_func df_orig 	= this.disc_func_zero_leg(ee);
		logical upd 		= false;
		
		if(!null(df)){
			this.add_disc_func_zero_leg(df);
			upd = true;
		}
		else if(null(df_orig)) {	
			QL_FAIL("discount function not available",this,true);
		}
		
		number c = i().__swap_pvbp_fix_leg(notional,delta, true,t, s);
		if(!null(c))
			c = abs(c);
		
		if(upd )
			this.add_disc_func_zero_leg(df_orig);		

		logical e = CORE_INT.add_error_info(error,t,s, "swap_il_zero.pv01");	
		return e ? null<number>: c;								
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.pv01");
		return null<number>;
	}
}
/*-----------------------------------------------------------------------
  pvbp	EXT SWAP: ok
  ----------------------------------------------------------------------*/
number  swap_il_zero.pvbp(	number option(nullable) nominal,
							number option(nullable) delta,
							logical option(nullable) centered,
							logical option(nullable) disable_rounding,	
							error_info option(nullable) error ) 
{	
	try {
		error_info ee = new error_info(true,true);
		number pv01_ = this.pv01(nominal,delta,null<disc_func>,ee); //always positive
		if(!null(nominal) && nominal < 0)
			return -pv01_;
		else
			return pv01_;					
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.pvbp");
		return null<number>;
	}
}

/*-----------------------------------------------------------------------
  pvbp	EXT SWAP: ok
  ----------------------------------------------------------------------*/
number  swap_il_zero.pvbp(	number 	option(nullable) nominal,
							number  option(nullable) delta,																																		
							error_info option(nullable) error ) 
{	
	try{	
		error_info ee = new error_info(true,true);
		number pv01_ = this.pv01(nominal,delta, null<disc_func>,ee);//always positive
		if(!null(nominal) && nominal < 0)
			return -pv01_;
		else
			return pv01_;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.pvbp");
		return null<number>;
	}
}

/*-----------------------------------------------------------------------
  swap_instrument   EXT SWAP: ok
  ----------------------------------------------------------------------*/
ql_fixed_income_swap swap_il_zero.swap_instrument(error_info option(nullable) error )
{
	return swap_gen.swap_instrument(error);
}

/*-----------------------------------------------------------------------
  swap_leg   EXT SWAP: ok
  ----------------------------------------------------------------------*/
vector(ql_swap_leg) swap_il_zero.legs(logical clone,error_info option(nullable) error )
{	
	return swap_gen.swap_leg(clone,error);
}

/*-----------------------------------------------------------------------
  infl_index_leg   EXT SWAP: ok
  ----------------------------------------------------------------------*/
ql_zero_infl_leg swap_il_zero.infl_index_leg(logical clone,error_info option(nullable) error )
{
	
	vector(ql_zero_infl_leg) l = swap_gen.swap_zero_infl_leg(clone,error);
	return !null(l) ? l[0] : null<ql_zero_infl_leg>;
}

/*-----------------------------------------------------------------------
  infl_zero_rate_leg   EXT SWAP: ok
  ----------------------------------------------------------------------*/
ql_fix_zero_leg swap_il_zero.infl_zero_rate_leg(logical clone,error_info option(nullable) error )
{
	
	vector(ql_fix_zero_leg) l = swap_gen.swap_fix_zero_leg(clone,error);
	return !null(l) ? l[0] : null<ql_fix_zero_leg>;
}

/*-----------------------------------------------------------------------
  pay_inflation  EXT SWAP: ok
  ----------------------------------------------------------------------*/
logical	 swap_il_zero.pay_inflation(error_info option(nullable) error )
{
	return !swap_gen.swap_payleg1(error);
}

/*-----------------------------------------------------------------------
  present_value
  ----------------------------------------------------------------------*/
number swap_il_zero.present_value(	disc_func option(nullable)	disc_func,
									//disc_func option(nullable) 	disc_func_infl_leg,
									out number pv_zero_leg,
									out number pv_infl_leg,
									number option(nullable) 	notional,
									logical option(nullable) 	trade_date_pv,																								
									error_info option(nullable) error )
{	
	return swap_gen.present_value(disc_func,disc_func,pv_zero_leg,pv_infl_leg,notional,trade_date_pv,error);
}

/*-----------------------------------------------------------------------
  present_value
  ----------------------------------------------------------------------*/
number swap_il_zero.present_value(	disc_func option(nullable)	disc_func,
									//disc_func option(nullable) 	disc_func_infl_leg,
									out number pv_zero_leg,
									out number pv_infl_leg,
									date option(nullable) 		pv_date,
									date option(nullable) 		settle_date,
									logical  					incl_issue_cf,
									logical  					incl_mat_cf,	
									error_info option(nullable) error )
{	
	return swap_gen.present_value(disc_func,disc_func,pv_zero_leg,pv_infl_leg,pv_date, settle_date ,incl_issue_cf,incl_mat_cf,error );
}


number swap_il_zero.present_value(	out number pv_zero_leg,
									out number pv_infl_leg,
									logical option(nullable) 	trade_date_pv,	
									error_info option(nullable) error )
{	
	return swap_gen.present_value(pv_zero_leg,pv_infl_leg,trade_date_pv ,error );
}


/*-----------------------------------------------------------------------
  present_value override prot
  ----------------------------------------------------------------------*/
/*number  swap_il_zero.present_value(	number option(nullable) 	notional,
									logical 					trade_date_pv,
									disc_func option(nullable) 	disc_func_disc,
									error_info option(nullable) error )
{	
	try{	
		notional = conv_null_number_com(notional) ;		
		QL_FAIL_COND(!null(disc_func_disc),"input of disc_func (for trade_date_pv calculation) not supported for swaps", this, true);
		
		QL_FAIL_COND(!null(notional) && ccy_swap_,"input of notional not applicable for currency swaps", this, true);//use a version with 2 notionals or set notional to null	

		number pv_fix_leg,pv_flt_leg;
		return present_value(	null<disc_func>, pv_fix_leg, pv_flt_leg,
								notional,trade_date_pv,error);					
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.present_value");
		return null<number>;
	}
}*/

/*-----------------------------------------------------------------------
  present_value  override prot
  ----------------------------------------------------------------------*/
/*number  swap_il_zero.present_value(	disc_func					disc_func,
									number option(nullable) 	notional,
									logical 					trade_date_pv,
									disc_func option(nullable) 	disc_func_disc,
									error_info option(nullable) error ) 
{	
	try{	
		notional= conv_null_number_com(notional) ;
		QL_FAIL_COND(!null(disc_func_disc),"input of disc_func (for trade_date_pv calculation) not supported for swaps",this,true);
		
		QL_FAIL_COND(ccy_swap_,"inapplicable function call for cross currency swaps",this,true);//use a version with 2 disc_funcs	

		number pv_fix_leg,pv_flt_leg;
		return present_value(	disc_func, pv_fix_leg, pv_flt_leg,
								notional,trade_date_pv,error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.present_value");
		return null<number>;
	}
}*/
/*-----------------------------------------------------------------------
  data	EXT SWAP: ok
  ----------------------------------------------------------------------*/
void  swap_il_zero.data(out swap_fix_zero_leg_data zero_leg_data,
						out swap_zero_infl_leg_data	infl_leg_data,
						error_info option(nullable) error )
{
	try{
	
		error_info ee = new error_info(true,true);
		vector(ql_swap_leg) l = legs(false,ee);
		zero_leg_data = new swap_fix_zero_leg_data(l[0]);
		infl_leg_data = new swap_zero_infl_leg_data(l[1]);
		return ;						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.data");
		zero_leg_data = null<swap_fix_zero_leg_data>;
		infl_leg_data = null<swap_zero_infl_leg_data>;
		return ;
	}
}
/*-----------------------------------------------------------------------
  imp_infl_zero_rate    EXT SWAP: ok
  ----------------------------------------------------------------------*/
number  swap_il_zero.imp_infl_zero_rate(	number 							swap_pv,							
										logical option(nullable) 		trade_date_pv,
										error_info option(nullable) 	error ) 
{	
	try{	
		instr_error_type t;
   		string 			s;	
		number c = i().__swap_imp_fix_rate(swap_pv,trade_date_pv,true,t, s);
		logical e = CORE_INT.add_error_info(error,t,s, "swap_il_zero.imp_infl_zero_rate");	
		return e ? null<number>: c;							
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.imp_infl_zero_rate");
		return null<number>;
	}
}

/*-----------------------------------------------------------------------
  set_quote_from_yield  EXT SWAP: throws
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.set_quote_from_yield(number 	option(nullable) yield,
											error_info option(nullable) error )
{	
	instrument cc = instrument._set_quote_from_yield(yield,  true,error);
	return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);
}


/*-----------------------------------------------------------------------
  set_pay_leg   --> returns an external swap, a db swap cannot be changed in regards to pay vs receive 
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.set_pay_leg(logical pay_inflation,							
									error_info option(nullable) error )
{
	try{
		
		__instrument  c = i().__swap_set_pay_leg1(!pay_inflation );
		
		instr_error e = new instr_error();
		instrument cc 	= this.create( c, e, E_INIT);
		CORE_INT.instr_fail_check(null(cc), "invalid swap_il_zero", this, e);
		
		if(!cc.is_valid(error)) 
			return null<swap_il_zero>;	
				
		return dynamic_cast<swap_il_zero>(cc);					
	}	
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.set_pay_leg");
		return null<swap_il_zero>;
	}
}
/*-----------------------------------------------------------------------
  set_date 		EXT SWAP: ok
  ext_swap: if trade date is the same --> the settle date will be preserved (if not input)
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.set_date(	date  					trade_date, 
									date  option(nullable)	settle_date,
									error_info option(nullable) error )
{
	instrument cc;
	if(ext_swap_) {
		logical re_init = false; 
		logical init_quote = false;
		cc = instrument._set_date(trade_date,settle_date,re_init,init_quote,error);
	}
	else {
		QL_FAIL_COND(!null(settle_date),"input of settlement date only supported for non-db swaps" ,this,true);
		logical re_init = true; 
		logical init_quote = true;
		cc = instrument._set_date(trade_date,settle_date,re_init,init_quote,error);
	}
	return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);
	
}
/*-----------------------------------------------------------------------
  set_date <protected>
	if trade date is the same --> the settle date will be preserved (if not input)	
	NOTE: 	if coupon is not null in instr_def it is kept unchanged --> deposit will be non-par 
			when a deposit is defined in db the coupon is always null
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.set_date(	date  				trade_date, 
									date  option(nullable)	settle_date,
									logical 			re_init, //true->fail for ext_swap
									logical 			re_init_quote, //true->fail for ext_swap
									error_info option(nullable) error)
{	
	//instrument cc = instrument._set_date(trade_date,null<date>,true,re_init_quote,error);
	//return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);

	QL_FAIL_COND(ext_swap_ && (re_init || re_init_quote),"re_init/re_init_quote flags = true not supported for non-db swaps",this,true );
	QL_FAIL_COND(!ext_swap_ && !null(settle_date),"input of settlement date only supported for non-db swaps",this,true );
	instrument cc = instrument._set_date(trade_date,settle_date,re_init,re_init_quote,error);
	return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);
	
}

/*-----------------------------------------------------------------------
  set_date <protected/legacy>
	quote is NOT kept even if trade_date is unchanged
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.set_date(	date  				trade_date, 
									logical 			re_init, 
									error_info option(nullable) error) 

{	
	//instrument cc = instrument._set_date(trade_date,true,error);
	//return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);

	instrument cc;
	if(ext_swap_){
		QL_FAIL_COND(re_init,"re_init = true not supported for non-db swaps",this,true );
		logical init_quote = false;
		cc = instrument._set_date(trade_date,null<date>,re_init,init_quote,error);
	}
	else{		
		logical init_quote = true;
		cc = instrument._set_date(trade_date,null<date>,re_init,init_quote,error);
	}
	return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);	
}

/*-----------------------------------------------------------------------
   move_date  <protected/legacy>		EXT SWAP: ok
	changes the trade date of the instrument without retrieving the 
 	corresponding quote from the database or the real-time feed.
	ext_swap: if trade date is the same --> the settle date will be preserved (if not input)	
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.move_date(	date  				trade_date, 
									date option(nullable) settle_date,
									error_info option(nullable) error ) 
{
	QL_FAIL_COND(!ext_swap_ && !null(settle_date),"input of settlement date only supported for non-db swaps",this,true );
	instrument cc = instrument._move_date(trade_date,settle_date,error);
	return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);
}

/*-----------------------------------------------------------------------
  set_quote		EXT SWAP: ok
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.set_quote(	number 	option(nullable) quote,									
									error_info option(nullable) error )
{		
	instrument cc = instrument._set_quote(quote,null<date>, null<date>, true,error);
	return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);
}

/*-----------------------------------------------------------------------
  set_quote 		EXT SWAP: will throw because of quote_side
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.set_quote(	string 	option(nullable) quote_side,
									error_info option(nullable) error )
{	
	instrument cc = instrument._set_quote(quote_side,null<date>, null<date>, true,error);
	return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);
}

/*-----------------------------------------------------------------------
  set_yield   FIX-FLT SWAP: yield == fix_cpn_rate  backward compat
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.set_yield(	number 	option(nullable) yield,							
									error_info option(nullable) error )
{	
	return this.set_infl_rate_zero_leg(yield,error);
}

/*-----------------------------------------------------------------------
  set_infl_rate_zero_leg  EXT SWAP: ok
  ----------------------------------------------------------------------*/
swap_il_zero swap_il_zero.set_infl_rate_zero_leg(number 	option(nullable) rate,							
											error_info option(nullable) error )
{
	try{
	
		if(!ext_swap_) {
			instrument cc = instrument._set_yield(	rate, null<date>, null<date>,true, error);
			return null(cc) ? null<swap_il_zero>: dynamic_cast<swap_il_zero>(cc);
		}

		
		error_info ee = new error_info(true,true);
		..quote_style qs = this.quote_style_e(ee);
		
		number q;
		if(qs == ..quote_style.YIELD_PCT)
			q = rate * 100;		
		__instrument  c = i().__swap_ext_set_quote(q, null<date>, null<date> );

		instr_error e = new instr_error();
		instrument cc 	= this.create( c, e, E_INIT);
		CORE_INT.instr_fail_check(null(cc), "invalid swap_il_zero", this, e);
		
		if(!cc.is_valid(error)) 
			return null<swap_il_zero>;	
				
		return dynamic_cast<swap_il_zero>(cc);					
	}	
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.set_infl_rate_zero_leg");
		return null<swap_il_zero>;
	}
}

/*-----------------------------------------------------------------------
  base_index
  ----------------------------------------------------------------------*/
number swap_il_zero.base_index(	error_info option(nullable) error)
{	
	 
	try{
		error_info ee = error_info(true,true);
		ql_zero_infl_leg l = infl_index_leg(false,ee );				
		
		return l.base_idx();						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.base_cpi");		
		return null<number>;
	}
}


/*-----------------------------------------------------------------------
  index_freq
  ----------------------------------------------------------------------*/
integer swap_il_zero.index_freq(	error_info option(nullable) error)
{	
	 
	try{
		error_info ee = error_info(true,true);
		ql_zero_infl_leg l = infl_index_leg(false,ee );				
		
		return l.index_freq();						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.index_freq");		
		return err_int();
	}
}


/*-----------------------------------------------------------------------
  index_method
  ----------------------------------------------------------------------*/
infl_index_method swap_il_zero.index_method(error_info option(nullable) error)
{	
	 
	try{
		error_info ee = error_info(true,true);
		ql_zero_infl_leg l = infl_index_leg(false,ee );				
		
		return l.index_method();						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.index_method");		
		return null<infl_index_method>;
	}
}
/*-----------------------------------------------------------------------
  is_noip_index_method
  ----------------------------------------------------------------------*/
logical swap_il_zero.is_noip_index_method(error_info option(nullable) error)
{	
	 
	try{
		error_info ee = error_info(true,true);
		ql_zero_infl_leg l = infl_index_leg(false,ee );				
		
		return l.is_noip_index_method();						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.is_noip_index_method");		
		return null<logical>;
	}
}
/*-----------------------------------------------------------------------
  implied_index
  lag1 always less than lag2 --> idx_lag1 always after idx_lag2 in time
  i.e.
	idx_lag2		t-2
	idx_lag1		t-1
	current time	t
	forward			t+

im_use_forecast: if true and month_infl = null then
seas_fact is used if
1. null(month_infl) && im_use_forecast or
2. !null(month_infl);
  ----------------------------------------------------------------------*/
void swap_il_zero.implied_index(number option(nullable) zero_infl_rate,
								vector(number) 	seas_fact,
								date 			seas_base_date,
								logical option(nullable) use_forecast,
								out date  		idx_date1,
								out date   		idx_date2,
								out number  	idx_lag1,
								out number   	idx_lag2,
								out number   	imp_infl,
								error_info option(nullable) error)
{	
	try{
		
		error_info ee = error_info(true,true);
		integer freq = this.index_freq(ee);
		QL_REQUIRE(freq == 12,"only monthly indices are currently supported",this,true);
		
		ql_zero_infl_leg l = infl_index_leg(false,ee );
		if(null(zero_infl_rate))
			zero_infl_rate = infl_zero_rate(ee );
		QL_REQUIRE(!null(zero_infl_rate),"invalid zero inflation rate",this,true);

		if(null(use_forecast))
			use_forecast = false;
		l.imp_idx(zero_infl_rate,seas_fact,seas_base_date,use_forecast,idx_date1,idx_date2,idx_lag1,idx_lag2,imp_infl);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.implied_index");
		idx_lag1 = idx_lag2 = null<number>;
		return ;
	}
}

void swap_il_zero.implied_index(number option(nullable) zero_infl_rate,
								out date  		idx_date1,
								out date   		idx_date2,
								out number  	idx_lag1,
								out number   	idx_lag2,
								error_info option(nullable) error)
{	
	try{
		error_info ee = error_info(true,true);
		
		integer freq = this.index_freq(ee);
		QL_REQUIRE(freq == 12,"only monthly indices are currently supported",this,true);
		
		ql_zero_infl_leg l = infl_index_leg(false,ee );
		if(null(zero_infl_rate))
			zero_infl_rate = infl_zero_rate(ee );
		QL_REQUIRE(!null(zero_infl_rate),"invalid zero inflation rate",this,true);

		vector(number) seas_fact = [1,1,1,1,1,1,1,1,1,1,1,1];// no seasons
		date seas_base_date = #2010-01-01;//FIX_ME
		number   	imp_infl;
		l.imp_idx(zero_infl_rate,seas_fact,seas_base_date,false,idx_date1,idx_date2,idx_lag1,idx_lag2,imp_infl);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.implied_index");
		idx_lag1 = idx_lag2 = null<number>;
		return ;
	}
}

/*-----------------------------------------------------------------------
  disc_func_zero_leg / disc_func_infl_leg   EXT SWAP: ok
  ----------------------------------------------------------------------*/
disc_func swap_il_zero.disc_func_zero_leg(error_info option(nullable) error ) 	{ return swap_gen.swap_disc_func_leg1(error);}
disc_func swap_il_zero.disc_func_infl_leg(error_info option(nullable) error ) 	{ return swap_gen.swap_disc_func_leg2(error);}


/*-----------------------------------------------------------------------
  infl_zero_rate	DLL_SWAP_HANDLER: ok
  ----------------------------------------------------------------------*/
number  swap_il_zero.infl_zero_rate(error_info option(nullable) error ) 
{	
	try{	
		instr_error_type t;
   		string 			s;	
		number c1,c2;
		i().__swap_coupon(c1,c2,t, s);
		logical e = CORE_INT.add_error_info(error,t,s, "swap_il_zero.infl_zero_rate");	
		return e ? null<number>: c1;						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.infl_zero_rate");
		return null<number>;
	}
}


/*-----------------------------------------------------------------------
  cpi_link
  ----------------------------------------------------------------------*/
cpi_link swap_il_zero.cpi_link(error_info option(nullable) error)
{	
	try{
		error_info ee = error_info(true,true);
		ql_zero_infl_leg l = infl_index_leg(false,ee );			
		return l.cpi_link();						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.cpi_link");
		return null<..cpi_link> ;
	}
}

/*-----------------------------------------------------------------------
  cpi_link
  ----------------------------------------------------------------------*/
cpi_link swap_il_zero.cpi_link_forecast(error_info option(nullable) error)
{	
	try{
		error_info ee = error_info(true,true);
		ql_zero_infl_leg l = infl_index_leg(false,ee );			
		return l.cpi_link_fcast();						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.cpi_link_forecast");
		return null<..cpi_link> ;
	}
}
/*-----------------------------------------------------------------------
  cpi_knots
	lag1 always less than lag2 --> mth_lag1 always after mth_lag2 in time
	|-------|-------|--------
	   mth2    mth1    start
  ----------------------------------------------------------------------*/
void swap_il_zero.cpi_knots(	logical 	start_date,//if false will return data for maturity
								out date  	mth_lag1,//commonly 2m
								out date   	mth_lag2,//commonly 3m
								out logical	active_lag1,
								out logical	active_lag2,
								error_info option(nullable) error)
{	
	try{
		error_info ee = error_info(true,true);
		ql_zero_infl_leg l = infl_index_leg(false,ee );			
		l.cpi_knots(start_date,mth_lag1,mth_lag2,active_lag1,active_lag2);
		return;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.cpi_knots");
		mth_lag1 = mth_lag2 = null<date>;
		active_lag1 = active_lag2 = null<logical>;
		return  ;
	}
}

/*-----------------------------------------------------------------------
  cpi_values
	lag1 always less than lag2 --> mth_lag1 always after mth_lag2 in time
	|-------|-------|--------
	   mth2    mth1    start
  ----------------------------------------------------------------------*/
void swap_il_zero.cpi_values(	logical 	start_date,//if false will return data for maturity
								out date  	dt_lag1,//commonly 2m
								out date   	dt_lag2,//commonly 3m
								out number 	val_lag1,
								out number 	val_lag2,
								error_info option(nullable) error)
{	
	try{
		error_info ee = error_info(true,true);
		ql_zero_infl_leg l = infl_index_leg(false,ee );			
		l.cpi_values(start_date,dt_lag1,dt_lag2,val_lag1,val_lag2);
		return;
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_il_zero.cpi_value");
		dt_lag1 = dt_lag2 = null<date>;
		val_lag1 = val_lag2 = null<number>;
		return  ;
	}
}

//oooooooooooooooooo
//oooooooooooooooooo
//oooooooooooooooooo
//oooooooooooooooooo

/*-----------------------------------------------------------------------
  swap_il_zero: index_freq
  ----------------------------------------------------------------------*/
/*integer swap_il_zero.index_freq(	error_info option(nullable) error)
{	
	return instrument.il_index_freq(error);
}*/


/*-----------------------------------------------------------------------
  swap_il_zero: index_factor
  ----------------------------------------------------------------------*/
/*number swap_il_zero.index_factor(	error_info option(nullable) error)
{	
	return instrument.il_index_factor(error);//returns the base cpi value for swap_il_zero
}
*/
/*-----------------------------------------------------------------------
  swap_il_zero: il_index_factor
  ----------------------------------------------------------------------*/
/*number swap_il_zero.index_factor(	date option(nullable) 	trade_date,
									date 					settle_date,
									//logical 				check_publ_date,
									error_info option(nullable) error)
{	
	return instrument.il_index_factor(trade_date, settle_date, error);
}*/



/*-----------------------------------------------------------------------
  swap_il_zero: cpi_values

  ----------------------------------------------------------------------*/
/*void swap_il_zero.cpi_values(	out number  	value1,
								out number   	value2,	
								error_info option(nullable) error)
{	
	instrument.il_cpi_values(value1,value2,error);
}*/



/*-----------------------------------------------------------------------
  swap_il_zero: cash_flow_nom_amounts
  ----------------------------------------------------------------------*/
/*vector(number) swap_il_zero.cash_flow_nom_amounts(..cpi_link cpi_link,
													number option(nullable) nominal,
													logical post_settle,
													error_info option(nullable) error)
{	
	logical keep_size=false;
	return instrument.il_cash_flow_nom_amounts(cpi_link,nominal,post_settle,keep_size,error);
}*/
