option(null: hard);	

/*	
	swaplib wrapper functions
	Developer: Algorithmica Research, Magnus Nyström

	--------------------------------------------------------------------
	 	swap help functions for create funcs
	--------------------------------------------------------------------			
*/

/*-----------------------------------------------------------------------
  func: swap_fltois
  ----------------------------------------------------------------------*/
swap_fltois swap_fltois(ql_float_leg 	ql_float_leg,
						ql_ois_leg 		ql_ois_leg, 
						string 			id,
						error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Float vs ONIndex')
{	
	try{			
		
		__instrument c = __instrument_swap(ql_float_leg,ql_ois_leg,id);

		return create_swap_fltois( c, error);					
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fltois");
		return null<swap_fltois>;
	}
}

/*-----------------------------------------------------------------------
  func: swap_fltois
  ----------------------------------------------------------------------*/
swap_fltois swap_fltois(ql_fixed_income_swap ql_fixed_income_swap,
						error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Float vs ONIndex')
{	
	try{	
		QL_FAIL_COND(CORE_INT.swap_instr_type(ql_fixed_income_swap) != instr_type.SWAP_FLTOIS, "invalid instrument type");

		__instrument c = __instrument_swap(ql_fixed_income_swap);

		return create_swap_fltois( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fltois");
		return null<swap_fltois>;
	}
}
/*-----------------------------------------------------------------------
  func: swap_fltflt
  ----------------------------------------------------------------------*/
swap_fltflt swap_fltflt(ql_float_leg 	ql_float_leg1, 							
						ql_float_leg 	ql_float_leg2,
						string 			id,
						error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Float vs Float')
{	
	try{			
		
		__instrument c = __instrument_swap(ql_float_leg1,ql_float_leg2,id);
		return create_swap_fltflt( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fltflt");
		return null<swap_fltflt>;
	}
}

/*-----------------------------------------------------------------------
  func: swap_fltflt
  ----------------------------------------------------------------------*/
swap_fltflt swap_fltflt(ql_fixed_income_swap ql_fixed_income_swap,
						error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Float vs Float')
{	
	try{	
		QL_FAIL_COND(CORE_INT.swap_instr_type(ql_fixed_income_swap) != instr_type.SWAP_FLTFLT, "invalid instrument type");

		__instrument c = __instrument_swap(ql_fixed_income_swap);
		return create_swap_fltflt( c, error);							
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fltflt");
		return null<swap_fltflt>;
	}
}
/*-----------------------------------------------------------------------
  func: swap_fixflt
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt(ql_fixed_income_swap ql_fixed_income_swap,
						error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs Float')
{	
	try{	
		QL_FAIL_COND(CORE_INT.swap_instr_type(ql_fixed_income_swap) != instr_type.SWAP_FIXFLT, "invalid instrument type", E_INIT);

		__instrument c = __instrument_swap(ql_fixed_income_swap);

		return create_swap_fixflt(c, error);				
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt");
		return null<swap_fixflt>;
	}
}


/*-----------------------------------------------------------------------
  func: swap_fixflt
  ----------------------------------------------------------------------*/
swap_fixflt swap_fixflt(ql_fix_leg 		ql_fix_leg, 							
						ql_float_leg 	ql_float_leg,
						string 			id,
						error_info option(nullable) error = null<error_info>)
option (category: 'Curve and Instrument/Interest Rate Swap/Fixed vs Float')
{	
	try{			
		__instrument c = __instrument_swap(ql_fix_leg,ql_float_leg,id);

		return create_swap_fixflt(c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixflt");
		return null<swap_fixflt>;
	}
}



/*-----------------------------------------------------------------------
  func: swap_fixois
  ----------------------------------------------------------------------*/
swap_fixois swap_fixois(ql_fixed_income_swap ql_fixed_income_swap,
						error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs ONIndex')
{	
	try{	
		QL_FAIL_COND(CORE_INT.swap_instr_type(ql_fixed_income_swap) != instr_type.SWAP_FIXOIS, "invalid instrument type");

		__instrument c = __instrument_swap(ql_fixed_income_swap);

		return create_swap_fixois( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixois");
		return null<swap_fixois>;
	}
}

/*-----------------------------------------------------------------------
  func: swap_oisois
  ----------------------------------------------------------------------*/
swap_oisois swap_oisois(ql_ois_leg 		ql_ois_leg1,
						ql_ois_leg 		ql_ois_leg2, 
						string 			id,
						error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/ONIndex vs ONIndex')
{	
	try{			
		
		__instrument c = __instrument_swap(ql_ois_leg1,ql_ois_leg2,id);

		return create_swap_oisois( c, error);					
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_oisois");
		return null<swap_oisois>;
	}
}

/*-----------------------------------------------------------------------
  func: swap_oisois
  ----------------------------------------------------------------------*/
swap_oisois swap_oisois(ql_fixed_income_swap ql_fixed_income_swap,
						error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/ONIndex vs ONIndex')
{	
	try{	
		QL_FAIL_COND(CORE_INT.swap_instr_type(ql_fixed_income_swap) != instr_type.SWAP_OISOIS, "invalid instrument type");

		__instrument c = __instrument_swap(ql_fixed_income_swap);

		return create_swap_oisois( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_oisois");
		return null<swap_oisois>;
	}
}


/*-----------------------------------------------------------------------
  func: swap_fixois
  ----------------------------------------------------------------------*/
swap_fixois swap_fixois(ql_fix_leg 		ql_fix_leg,
						ql_ois_leg 		ql_ois_leg,
						string 			id,
						error_info option(nullable) error		= null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Fixed vs ONIndex')
{	
	try{			
		
		//QL_FAIL_COND(swap_instr_type(ql_swap_leg1,ql_swap_leg2) != INST_FIX_OIS, "invalid instrument type");

		__instrument c = __instrument_swap(ql_fix_leg,ql_ois_leg,id);

		return create_swap_fixois( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_fixois");
		return null<swap_fixois>;
	}
}

/*-----------------------------------------------------------------------
  func: swap_zeroflt
  ----------------------------------------------------------------------*/
swap_zeroflt swap_zeroflt(	ql_fix_zero_leg ql_fix_zero_leg, 							
							ql_float_leg 	ql_float_leg,
							string 			id,
							error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Zero vs Float')
{	
	try{			
		
		//QL_FAIL_COND(swap_instr_type(ql_swap_leg1,ql_swap_leg2) != INST_ZERO_FLT_SWAP, "invalid instrument type");

		__instrument c = __instrument_swap(ql_fix_zero_leg,ql_float_leg,id);

		return create_swap_zeroflt( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_zeroflt");
		return null<swap_zeroflt>;
	}
}
 

/*-----------------------------------------------------------------------
  func: swap_zeroflt
  ----------------------------------------------------------------------*/
swap_zeroflt swap_zeroflt(	ql_fixed_income_swap ql_fixed_income_swap,
							error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Zero vs Float')
{	
	try{	
		QL_FAIL_COND(CORE_INT.swap_instr_type(ql_fixed_income_swap) != instr_type.SWAP_ZEROFLT, "invalid instrument type");

		__instrument c = __instrument_swap(ql_fixed_income_swap);

		return create_swap_zeroflt( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_zeroflt");
		return null<swap_zeroflt>;
	}
}


/*-----------------------------------------------------------------------
  func: swap_zeroois
  ----------------------------------------------------------------------*/
swap_zeroois swap_zeroois(	ql_fix_zero_leg ql_fix_zero_leg, 							
							ql_ois_leg 	ql_ois_leg,
							string 			id,
							error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Zero vs ONIndex')
{	
	try{			
		
		__instrument c = __instrument_swap(ql_fix_zero_leg,ql_ois_leg,id);

		return create_swap_zeroois( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_zeroois");
		return null<swap_zeroois>;
	}
}
 

/*-----------------------------------------------------------------------
  func: swap_zeroois
  ----------------------------------------------------------------------*/
swap_zeroois swap_zeroois(	ql_fixed_income_swap ql_fixed_income_swap,
							error_info option(nullable) error = null<error_info>)
option (category: 'Instrument/Interest Rate Swap/Zero vs ONIndex')
{	
	try{	
		QL_FAIL_COND(CORE_INT.swap_instr_type(ql_fixed_income_swap) != instr_type.SWAP_ZEROOIS, "invalid instrument type");

		__instrument c = __instrument_swap(ql_fixed_income_swap);

		return create_swap_zeroois( c, error);						
	}
	catch {
		CORE_INT.catch_error_info(error,err.type(),err.message(), "swap_zeroois");
		return null<swap_zeroois>;
	}
}

module CORE_INT_SWAPLIB
{
	/*-----------------------------------------------------------------------
	  vector_is_null
	  ----------------------------------------------------------------------*/
	logical vector_is_null(vector(fwd_func) option(nullable) f)
	{
		if(null(f))
			return true;

		for(integer i=0;i<v_size(f);i++) {
			if(!null(f[i]))//if any element is not null the whole vector is considered not null
				return false;
		}
		return true;
	}
	/*-----------------------------------------------------------------------
	  vector_is_null
	  ----------------------------------------------------------------------*/
	logical vector_is_null(vector(disc_func) option(nullable) df)
	{
		if(null(df))
			return true;

		for(integer i=0;i<v_size(df);i++) {
			if(!null(df[i]))//if any element is not null the whole vector is considered not null
				return false;
		}
		return true;
	}
	//["ShortFirst","LongFirst","ShortLast","LongLast","SpecificFirst", "SpecificLast","SpecificBoth","FullCoupon","None"];
	stub_type stub_type_map(string rule)  
	{
		 stub_type e;
		 str_to_enum(rule,e);	 
		 return e;
	}

	vector(string) stub_type_list()
	{
		 vector(stub_type) e;
		 expand_enum(e);
		 return string(e);
	}
	
	void schedule_helper(	date 							start,
							date 							maturity,
							string  						tenor,
							number option(nullable) 		roll_day,
							calendar option(nullable) 		cal,
							bd_convention option(nullable) 	bus_day,
							bd_convention option(nullable) 	bus_day_mat,
							stub_type 						stub,
							logical option(nullable) 		eom,
							date option(nullable) 			eff_date,
							date option(nullable) 			first_cpn_date,
							date option(nullable) 			last_reg_cpn_date,
							out date 						eff_date_out,
							out date 						first_cpn_date_out,
							out date 						last_reg_cpn_date_out,
							out integer 					roll_day_out,
							out schedule_period_type 		first_type,
							out schedule_period_type 		last_type,
							integer	    					pmt_delay ,
							integer	    					pmt_delay_mat ,
							integer	    					min_dflt_period	=	5)
	{
		try
		{
			schedule sc = schedule(start, maturity, tenor, roll_day,cal,bus_day,bus_day_mat,
								   stub,eom,eff_date,first_cpn_date,last_reg_cpn_date,
								   pmt_delay,pmt_delay_mat,min_dflt_period);

			eff_date_out			= sc.start_date(true);
			first_cpn_date_out 		= sc.first_date(true);
			last_reg_cpn_date_out 	= sc.last_reg_date(true);
			roll_day_out 			= sc.roll_day();
			first_type 				= sc.first_period_type();		
			last_type 				= sc.last_period_type();		
		}
		catch {
			eff_date_out = first_cpn_date_out = last_reg_cpn_date_out = null;
			roll_day_out = 0;
			first_type = last_type = null;
		}
	}
	void schedule_helper(	date 							start,
							date 							maturity,
							integer							freq,
							number option(nullable) 		roll_day,
							calendar option(nullable) 		cal,
							bd_convention option(nullable) 	bus_day,
							bd_convention option(nullable) 	bus_day_mat,
							stub_type 						stub,
							logical option(nullable) 		eom,
							date option(nullable) 			eff_date,
							date option(nullable) 			first_cpn_date,
							date option(nullable) 			last_reg_cpn_date,
							out date 						eff_date_out,
							out date 						first_cpn_date_out,
							out date 						last_reg_cpn_date_out,
							out integer 					roll_day_out,
							out schedule_period_type 		first_type,
							out schedule_period_type 		last_type,
							integer	    					pmt_delay = 0,
							integer	    					pmt_delay_mat = 0,
							integer	    					min_dflt_period	= 5)
	{
		string tenor;
		switch (freq){
		case 1:
			tenor = "1Y";
			break;
		case 2:
			tenor = "6M";
			break;
		case 4:
			tenor = "3M";
			break;
		case 12:
			tenor = "1M";
			break;
		default:
			QL_FAIL("invalid frequency");
		}
		schedule_helper(start,maturity,tenor,roll_day,cal,bus_day,bus_day_mat,stub,eom,
						eff_date,first_cpn_date,last_reg_cpn_date,eff_date_out,first_cpn_date_out,
						last_reg_cpn_date_out,roll_day_out,first_type,last_type,pmt_delay,pmt_delay_mat,
						min_dflt_period);
	}
	/*-----------------------------------------------------------------------
	  set_ccy
	  ----------------------------------------------------------------------*/
	void set_ccy(logical is_ccy,
				 out number leg1_notional,
				 out number leg2_notional,
				 out number leg1_fx_mult ,
				 out number leg2_fx_mult)
	{
		if(is_ccy) {
			
			if(null(leg1_fx_mult)) 
				leg1_fx_mult = 1;			
			if(null(leg2_fx_mult)) 
				leg2_fx_mult = 1;

			if(!null(leg1_notional) && !null(leg2_notional))
				return;
			
			if(!null(leg1_notional))
				leg2_notional = leg1_notional*leg1_fx_mult/leg2_fx_mult;
			else if(!null(leg2_notional) ) 
				leg1_notional = leg2_notional*leg2_fx_mult/leg1_fx_mult;
			else
				QL_FAIL("missing notional");
		}
		else {
			if(!null(leg1_notional))
				leg2_notional		= leg1_notional;
			else if(!null(leg2_notional))
				leg1_notional		= leg2_notional;
			else
				QL_FAIL("missing notional");
			leg1_fx_mult 		= 1;
			leg2_fx_mult 		= 1;
		}
	}

	
	
	/*-----------------------------------------------------------------------
	  swap_start_date_helper
	  ----------------------------------------------------------------------*/
	void swap_start_date_helper(date_code option(nullable) 	spot_code,
								out date  					trade_date,
								calendar option(nullable) 	pmt_calendar,
								string option(nullable) 	fwd_start_code,
								logical	option(nullable) 	fwd_code_spot_offset,
								out date 					start_date,
								out date 					settle_date)
	{
		try {
			date spot_date = null;
			if(null(settle_date) && !null(trade_date) && !null(spot_code)){
				settle_date = spot_code.apply_fwd(trade_date,pmt_calendar);
				spot_date 	= settle_date;
				if(!null(start_date) && start_date > settle_date) {					
					settle_date = start_date;
					return;
				}
			}
			if(!null(settle_date) && null(trade_date) && !null(spot_code)){
				trade_date 	= spot_code.apply_back(settle_date,pmt_calendar);
				spot_date 	= settle_date;
			}
				
			if(null(start_date)){			
				if(!null(fwd_start_code)){
					QL_REQUIRE(!null(trade_date),"invalid trade date");
					QL_REQUIRE(!null(spot_code),"invalid spot settle code");

					if(null(fwd_code_spot_offset))
						fwd_code_spot_offset = true;

					date_code start_code = date_code(fwd_start_code);
					
					if(fwd_code_spot_offset) {
						if(null(spot_date))
							spot_date 	= spot_code.apply_fwd(trade_date,pmt_calendar);										
						start_date 		= start_code.apply_fwd(spot_date,pmt_calendar);
					}
					else {
						start_date 		= start_code.apply_fwd(trade_date,pmt_calendar);
					}
					start_date	= pmt_calendar.adjust_date (start_date, bd_convention.BD_FOLLOWING);
					QL_FAIL_COND(!null(settle_date) && settle_date > start_date,"invalid settle date (settle > startdate when using a fwd start code not allowed)");
					settle_date = start_date;
				}
				else if(!null(settle_date)){
					start_date = settle_date;
				}
				else if(!null(trade_date) && !null(spot_code)){
					QL_REQUIRE(!null(trade_date),"invalid trade date");
					settle_date = spot_code.apply_fwd(trade_date,pmt_calendar);
					start_date = settle_date;
				}
				/*Note: it is possible that settle_date and start_date ends up being null*/
			}
			else {
				if(null(settle_date) || start_date > settle_date) {
					settle_date = start_date;
				}				
			}
		}
		catch {
			start_date = settle_date = null<date>;
		}
	}
	/*-----------------------------------------------------------------------
	  swap_start_date_helper
	  ----------------------------------------------------------------------*/
	void swap_start_date_helper(string option(nullable) 	spot_code,
								out date 					trade_date,
								calendar option(nullable) 	pmt_calendar,
								string option(nullable) 	fwd_start_code,
								logical	option(nullable) 	fwd_code_spot_offset,
								out date 					start_date,
								out date 					settle_date)
	{
		date_code  	sc = null(spot_code) ? null : date_code(spot_code);
		swap_start_date_helper(sc,trade_date,pmt_calendar,fwd_start_code,fwd_code_spot_offset,start_date,settle_date);
	}
	/*-----------------------------------------------------------------------
	  swap_maturity_helper
	  ----------------------------------------------------------------------*/
	tenor_code swap_maturity_helper(instr_def option(nullable)	def,
									date option(nullable) maturity,
									string 	option(nullable) maturity_code)
	{		
		if(!null(maturity))
			return null<tenor_code>;	
		else if(!null(maturity_code))
			return tenor_code(maturity_code);		

		if(null(def))
			QL_FAIL("missing maturity/maturity_code", E_INIT);
		
		error_info ee = new error_info(true,true);
		string mc = def.maturity_code(ee);
		if(!null(mc))
			return tenor_code(mc);
		
		QL_FAIL("missing maturity/maturity_code", E_INIT);			
	}

	/*-----------------------------------------------------------------------
	check_surface_freq
	----------------------------------------------------------------------*/
	logical check_surface_freq(integer freq)
	{
		switch(freq){
		case 1://ann
		case 2://semi
		case 4://quart
		case 6://every 2m
		case 12://month
		case 26://every 2weeks
		case 52://weekly
		case 365://daily
			return true;
		}
		return false;
	}
	/*-----------------------------------------------------------------------
	mtm_data
	----------------------------------------------------------------------*/
	void mtm_data(date 	option(nullable) td,
				  string 		base_ccy,
				  calendar 		base_cal,
				  string 		price_ccy,
				  calendar 		price_cal,
				  calendar option(nullable) usd_calendar,
				  out integer 		fx_spot_days,
				  out fx_spot_rule 	fx_rule,
				  out date 			fx_spot_dt)
	{
		fx_spot_days 	= fx_spot_days_dflt(base_ccy, price_ccy);
		fx_rule			= fx_spot_rule_dflt(base_ccy, price_ccy, usd_calendar);
		if(null(td))
			fx_spot_dt = null;
		else
			fx_spot_dt		= fx_spot_date(td,base_cal, base_ccy,price_cal,price_ccy,fx_rule,fx_spot_days,usd_calendar, true);
	}
	/*-----------------------------------------------------------------------
	create_fixing_fwd_funcs
	-this function creates a vector of fwd_func for usage in a tenor_surface with the purpose to handle interpolation (of a start stub rate)
		of a non-historical fixing i.e. settledate >= tradedate. Technically it will work for any future settledate but is intended
		to be used for the case when resetdate == tradedate (spot starting swap, settledate normally t+2)
	-the vector contains a full fwd_func for the main tenor while the other fwd_funcs are limited for usage to get the rate for the settlement day
	-the limited fwd_funcs are only used for interpolation purposes when the swap requires an interpolated fixing
	-this function should only be used when the fwd_func for the main tenor is built with the fixing rate included
	 (and this does not need to be the case) (if the fixrate for the main tenor is included in the fix_rate it will be checked against its' fwd_func
	----------------------------------------------------------------------*/
	void create_fixing_fwd_funcs(date 				td,
								 vector(integer) 	main_freq,
								 vector(fwd_func) 	main_f,
								 vector(integer) 	fix_freq,
								 vector(number) 	fix_rate,				 
								 date 				fix_settle,
								 logical 			incl_restrict,
								 out vector(fwd_func) 	fwd_v,
								 out vector(tenor_code) tenor_v)
	{
		resize(fwd_v,0);
		resize(tenor_v,0);
		integer size = v_size(fix_freq);
		QL_REQUIRE(size == v_size(fix_rate), "invalid vector size" );
		QL_REQUIRE(fix_settle >= td, "invalid settlement date" );

		integer sizem = v_size(main_freq);
		QL_REQUIRE(sizem > 0 && sizem <= size, "invalid vector size" );
		QL_REQUIRE(sizem == v_size(main_f), "invalid vector size" );
		//QL_REQUIRE(check_surface_freq(main_freq),"invalid main freq");

		number max_t, min_t;
		if(incl_restrict) {
			max_t 		= fix_settle == td ? (fix_settle-td+1)/365.0 : (fix_settle-td)/365.0;  //because fwd_func_restrict does not allow 0 as max_t
			min_t 		= fix_settle == td ? 0 : (fix_settle-td-1)/365.0; //because fwd_func_restrict does not allow min_t == max_t
		}
		/* note:
		 1. max_t == min_t = 0 causes a quantlab crash
		 2. min_t == max_t is not allowed but is needed because we want to limit access to the fwd_func for 1 day only
			now we set min_t to be one day before which allows access for 2 days (probably not a problem even if we have fwd starting swaps)
		 
		 */
		vector(tenor_code) main_t[sizem];
		for(integer i=0; i<sizem;i++) {
			if(i>0){
				QL_REQUIRE(main_freq[i] < main_freq[i-1],"invalid freq order (must be sorted in descending order)");
			}

			QL_REQUIRE(check_surface_freq(main_freq[i]),"invalid freq");
			
			vector(integer) vf = v_find(fix_freq, [main_freq[i]]);
			QL_REQUIRE(v_size(vf) == 1,"main freq missing in fix freq vector");
			
			main_t[i] = new tenor_code.tenor_code(main_freq[i]);
		}
		
		integer j = 0;
		
		for(integer i=0; i<size;i++) {
			if(i>0){
				QL_REQUIRE(fix_freq[i] < fix_freq[i-1],"invalid freq order (must be sorted in descending order)");
			}
			
			if(fix_freq[i] > main_freq[j]){
				QL_REQUIRE(check_surface_freq(fix_freq[i]),"invalid freq");
				fwd_func f 	= fwd_func_interp([1],[fix_rate[i]],ip_linear());
				if(incl_restrict)
					f = fwd_func_restrict(f,min_t,max_t);			
				tenor_code t = new tenor_code.tenor_code(fix_freq[i]);
				push_back(fwd_v,f);
				push_back(tenor_v,t);
			}
			else if(j < sizem && fix_freq[i] == main_freq[j]){				
				push_back(fwd_v,main_f[j]);
				push_back(tenor_v,main_t[j]);
				number r = main_f[j].fwd((fix_settle-td)/365.0);
				QL_REQUIRE(CORE_INT.is_equal(r,fix_rate[i]),"main tenor fix rate does not match main fwd_func");
				j++;
			}
			else {//fix_freq[i] < main_freq[j]
				QL_REQUIRE(check_surface_freq(fix_freq[i]),"invalid freq");				
				fwd_func f 	= fwd_func_interp([1],[fix_rate[i]],ip_linear());
				if(incl_restrict)
					f = fwd_func_restrict(f,min_t,max_t);			
				tenor_code t = new tenor_code.tenor_code(fix_freq[i]);
				push_back(fwd_v,f);
				push_back(tenor_v,t);
			}
		}
	}
	/*-----------------------------------------------------------------------
	  create_fixing_fwd_funcs  <overload>
	  strings as tenor instead of integer as freq
	  ----------------------------------------------------------------------*/
	void create_fixing_fwd_funcs(date 				td,
								 vector(string) 	main_tenor,
								 vector(fwd_func) 	main_f,
								 vector(string) 	fix_tenor,
								 vector(number) 	fix_rate,				 
								 date 				fix_settle,
								 logical 			incl_restrict,
								 out vector(fwd_func) 	fwd_v,
								 out vector(tenor_code) tenor_v)
	{
		integer sizem 		= v_size(main_tenor);
		vector(integer) main_freq[sizem];
		
		for(integer i=0; i<sizem;i++) {
			tenor_code m_t 	= new tenor_code.tenor_code(main_tenor[i]);
			main_freq[i] 	= m_t.frequency();
		}
		
		integer size 		= v_size(fix_tenor);
		vector(integer) fix_freq[size];

		for(integer i=0; i<size;i++) {
			tenor_code t  	= new tenor_code.tenor_code(fix_tenor[i]);
			fix_freq[i] 	= t.frequency();
		}
		
		create_fixing_fwd_funcs(td,main_freq,main_f, fix_freq, fix_rate, fix_settle, incl_restrict, fwd_v, tenor_v);
	}
	/*-----------------------------------------------------------------------
	  create_fixing_sprd_fwd_funcs
	  ----------------------------------------------------------------------*/
	void create_fixing_sprd_fwd_funcs(	date 				td,
										vector(integer) 	main_freq,
										vector(fwd_func) 	main_f,
										vector(integer) 	fix_freq,
										vector(number) 		fix_rate,
										vector(number) 		fix_sprd,
										integer				sprd_base_freq,
										date 				fix_settle,										
										out vector(fwd_func) 	fwd_v,
										out vector(tenor_code) tenor_v)
	{
		integer size = v_size(fix_freq);
		QL_REQUIRE(size == v_size(fix_sprd), "invalid vector size" );

		create_fixing_fwd_funcs(td,main_freq,main_f,fix_freq,fix_rate,fix_settle,false, fwd_v, tenor_v);

		vector(integer) vf 	= v_find(main_freq,[sprd_base_freq]);
		QL_REQUIRE(v_size(vf) == 1, "sprd_base_freq not found in main freq vector" );
		fwd_func f_comp 	= main_f[vf[0]];

		vf = v_find(fix_freq,main_freq);
		integer sizem = v_size(main_freq);
		QL_REQUIRE(v_size(vf) == sizem, "main freq not found in fix freq vector" );

		number cutoff_t = (fix_settle-td)/365.0;
		
		integer j = 0;
		for(integer i=0; i<size;i++) {

			if(j < sizem && i == vf[j]){				
				j++;
				continue;
			}

			number sprd = fix_sprd[i];
			fwd_func fwd_post = fwd_func_add_spread( f_comp, sprd);
			fwd_v[i] = fwd_func_splice( fwd_v[i], fwd_post, cutoff_t);
		}
	}
	/*-----------------------------------------------------------------------
	  create_tenor_surface_flt
	  ----------------------------------------------------------------------*/
	void create_tenor_surface_flt(	date 								trade_date,
								   ..disc_func option(nullable) 		flt_disc_func,								
									vector(fwd_func) option(nullable) 	flt_fwd_func,
									vector(..disc_func) option(nullable) flt_fwd_disc_func,
									vector(tenor_code) option(nullable) flt_tenor_codes,
									ir_index 							flt_ir_index,
									logical option(nullable) 			allow_extrap,//default is to allow extrap when we have a surface (as opposed to a single tenor)
									out tenor_surface 					flt_tenor_surface)
	{
		try{
			if(!null(flt_fwd_func)) {
				integer size = v_size(flt_fwd_func);
				if(size <= 1) {
					if(null(allow_extrap))
						allow_extrap = false;//i.e. do not allow extrapolation if we have a surface with only one tenor
					flt_tenor_surface = tenor_surface(trade_date, flt_disc_func,flt_fwd_func,[flt_ir_index],allow_extrap);
				}
				else {
					QL_REQUIRE(size == v_size(flt_tenor_codes),"invalid tenor code vector");
					if(null(allow_extrap))
						allow_extrap = true;//i.e. allow extrapolation if we have a surface with more than one tenor
					flt_tenor_surface = tenor_surface(trade_date, flt_disc_func,flt_fwd_func,flt_tenor_codes,flt_ir_index,allow_extrap);
				}
			}
			else if (!null(flt_fwd_disc_func)) {
				integer size = v_size(flt_fwd_disc_func);
				if(size <= 1) {
					if(null(allow_extrap))
						allow_extrap = false;
					flt_tenor_surface = tenor_surface(trade_date, flt_disc_func,flt_fwd_disc_func,[flt_ir_index],allow_extrap);
				}
				else {
					QL_REQUIRE(size == v_size(flt_tenor_codes),"invalid tenor code vector");
					if(null(allow_extrap))
						allow_extrap = true;
					flt_tenor_surface = tenor_surface(trade_date, flt_disc_func,flt_fwd_disc_func,flt_tenor_codes,flt_ir_index,allow_extrap);
				}
			}
			else {
				if(null(allow_extrap))
					allow_extrap = false;
				flt_tenor_surface = tenor_surface(trade_date, flt_disc_func,flt_fwd_disc_func,[flt_ir_index],allow_extrap);
			}
		}
		catch {
			flt_tenor_surface = null<tenor_surface>;
		}
	}

	/*-----------------------------------------------------------------------
	 ----------------------------------------------------------------------
	 ----------------------------------------------------------------------
	 class ois_df_prio
	----------------------------------------------------------------------
	----------------------------------------------------------------------
	----------------------------------------------------------------------*/
	class ois_df_helper
	{
	public:
		ois_df_helper(logical 							is_endog,
						logical							is_approx,
						logical 						is_ccy,
						logical 						rev_leg_prio,
						logical							sync_endog_ok,
						..disc_func option(nullable) 	disc_func_rev_leg,
						..disc_func option(nullable) 	ois_disc_func,
						..fwd_func option(nullable) 	ois_fwd_func,
						..disc_func option(nullable) 	ois_fwd_disc_fun);
		void need_calc();
		
		disc_func 	get_ois_disc_func(date d, date m, day_count_method dc, calendar cal);
		fwd_func 	get_ois_fwd_func(date d, date m, day_count_method dc, calendar cal);
		disc_func 	get_ois_fwd_disc_func(date d, date m, day_count_method dc, calendar cal);
		disc_func 	get_disc_func_rev_leg(date d, date m, day_count_method dc, calendar cal);
		
		logical 	is_endog_;
		logical		is_approx_;
		logical 	is_ccy_;
		logical 	rev_leg_prio_;
		logical		sync_endog_ok_;
		..disc_func  disc_func_rev_leg_;
		..disc_func  ois_disc_func_;
		..fwd_func  ois_fwd_func_;
		..disc_func  ois_fwd_disc_func_;
		vector(logical) need_calc_;
	};

	/*-----------------------------------------------------------------------
	  ois_df_prio
	----------------------------------------------------------------------*/
	ois_df_helper.ois_df_helper(logical 							is_endog,
								logical								is_approx,
								logical 							is_ccy,
								logical 							rev_leg_prio,
								logical								sync_endog_ok,
								..disc_func option(nullable) 		disc_func_rev_leg,
								..disc_func option(nullable) 		ois_disc_func,
								..fwd_func option(nullable) 		ois_fwd_func,
								..disc_func option(nullable) 		ois_fwd_disc_func )
								:	is_endog_(is_endog),
									is_approx_(is_approx),
									is_ccy_(is_ccy),
									rev_leg_prio_(rev_leg_prio),
									sync_endog_ok_(sync_endog_ok),
									disc_func_rev_leg_(disc_func_rev_leg),
									ois_disc_func_(ois_disc_func),
									ois_fwd_func_(ois_fwd_func),
									ois_fwd_disc_func_(ois_fwd_disc_func)
								
	{
		need_calc();
	}

	/*-----------------------------------------------------------------------
	  get_disc_func_rev_leg
	----------------------------------------------------------------------*/
	disc_func ois_df_helper.get_disc_func_rev_leg(	date d,
													date m,
													day_count_method dc,
													calendar cal)
	{
		if(need_calc_[0]){

			if(is_ccy_)
				return null;
			
			if(!need_calc_[1]){
				disc_func_rev_leg_	= ois_disc_func_;
				need_calc_[0] = null(disc_func_rev_leg_);
				return disc_func_rev_leg_;
			}
			
			if(!is_endog_) 
				return null;

			if(!need_calc_[3]){
				disc_func_rev_leg_	= ois_fwd_disc_func_;
				need_calc_[0] = null(disc_func_rev_leg_);
				return disc_func_rev_leg_;
			}
			
			if(need_calc_[2])
				return null;
						
			disc_func_rev_leg_	= CORE_INT.create_disc_func_from_fwd_on(d,m,ois_fwd_func_,dc,cal,true, 0);	
			need_calc_[0] = null(disc_func_rev_leg_);
			
			ois_disc_func_ = disc_func_rev_leg_;
			need_calc_[1] = null(ois_disc_func_);
				
			ois_fwd_disc_func_ = ois_disc_func_;
			need_calc_[3] = null(ois_fwd_disc_func_);
			
		}
		
		return disc_func_rev_leg_;
	}
	
	/*-----------------------------------------------------------------------
	  get_ois_disc_func
	----------------------------------------------------------------------*/
	disc_func ois_df_helper.get_ois_disc_func(date d,
											  date m,
											  day_count_method dc,
											  calendar cal)
	{
		if(need_calc_[1]){

			if(is_endog_) {

				if(!need_calc_[3]){
					ois_disc_func_ = ois_fwd_disc_func_;
					ois_fwd_func_ = CORE_INT.create_fwd_func_from_df_on(d, m,cal,dc, ois_disc_func_,true, 0);					
				}
				else {
					if(need_calc_[2])
						return null;

					ois_fwd_disc_func_	= CORE_INT.create_disc_func_from_fwd_on(d,m,ois_fwd_func_,dc,cal,true, 0);	
					ois_disc_func_ 		= ois_fwd_disc_func_;
				}
			}
			
			need_calc_[1] = null(ois_disc_func_);
			need_calc_[2] = null(ois_fwd_func_);
			need_calc_[3] = null(ois_fwd_disc_func_);
			
			if(!is_ccy_ /*&& need_calc_[0]*/){
				disc_func_rev_leg_ = ois_disc_func_;
				need_calc_[0] = null(disc_func_rev_leg_);
			}
		}
		
		return ois_disc_func_;
		
	}
	/*-----------------------------------------------------------------------
	  get_ois_fwd_disc_func
	----------------------------------------------------------------------*/
	disc_func ois_df_helper.get_ois_fwd_disc_func(	date d,
													date m,
													day_count_method dc,
													calendar cal)
	{
		if(need_calc_[3]){
			if(is_endog_ && !need_calc_[1]) {
				ois_fwd_disc_func_ 	= ois_disc_func_ ;
				//ois_fwd_func_ 		= get_ois_fwd_func(d, m, dc, cal);
				ois_fwd_func_ 		= CORE_INT.create_fwd_func_from_df_on(d, m,cal,dc, ois_fwd_disc_func_,true, 0);
				need_calc_[2] 		= null(ois_fwd_func_);
				need_calc_[3] 		= null(ois_fwd_disc_func_);
				return ois_fwd_disc_func_;
			}
			
			if(need_calc_[2])
				return null;
						
			ois_fwd_disc_func_	= CORE_INT.create_disc_func_from_fwd_on(d,m,ois_fwd_func_,dc,cal,true, 0);			
			need_calc_[3] 		= null(ois_fwd_disc_func_);
			
			if(is_endog_) {
				ois_disc_func_ 	= ois_fwd_disc_func_;
				need_calc_[1] 	= null(ois_disc_func_);
			}
			
			if(!is_ccy_ /*&& need_calc_[0]*/){
				if(is_endog_) 
					disc_func_rev_leg_ = ois_fwd_disc_func_;
				need_calc_[0] = null(disc_func_rev_leg_);
			}
		}
		
		return ois_fwd_disc_func_;
	}
	/*-----------------------------------------------------------------------
	  get_ois_disc_func
	----------------------------------------------------------------------*/
	fwd_func ois_df_helper.get_ois_fwd_func(date d, date m, day_count_method dc, calendar cal)
	{
		if(need_calc_[2]){
			
			if(need_calc_[3])
				return null;
									
			ois_fwd_func_ = CORE_INT.create_fwd_func_from_df_on(d, m,cal,dc, ois_fwd_disc_func_,true, 0);
			need_calc_[2] = null(ois_fwd_func_);
		}
		
		return ois_fwd_func_;
	}
	/*-----------------------------------------------------------------------
	  need_calc
	----------------------------------------------------------------------*/
	void ois_df_helper.need_calc()
	{
		need_calc_ 			= [false,false,false,false];
		logical single_curr = !is_ccy_;
		if(single_curr) {
			if(is_endog_) {
				if(rev_leg_prio_ && !null(disc_func_rev_leg_)) {
					ois_disc_func_ = disc_func_rev_leg_;
					if(!sync_endog_ok_){
						ois_fwd_func_ = null;
						//ois_fwd_disc_func_ = disc_func_rev_leg_;
					}
					//else {
						ois_fwd_disc_func_ = disc_func_rev_leg_;
					//}
				}
				else if(!null(ois_disc_func_)){
					disc_func_rev_leg_ = ois_disc_func_;
					if(!sync_endog_ok_){
						ois_fwd_func_ = null;					
						//ois_fwd_disc_func_ = ois_disc_func_;  //hmmm?
					}
					//else {
						ois_fwd_disc_func_ = ois_disc_func_;
					//}
				}
				else if(!null(ois_fwd_disc_func_)){
					disc_func_rev_leg_ = ois_fwd_disc_func_;
					if(!sync_endog_ok_){
						ois_fwd_func_ = null;
					}
					ois_disc_func_ = ois_fwd_disc_func_;
				}
				else if(!null(disc_func_rev_leg_)) {
					ois_disc_func_ = disc_func_rev_leg_;
					if(!sync_endog_ok_){
						ois_fwd_func_ = null;
					}
					ois_fwd_disc_func_ = disc_func_rev_leg_;
				}							
			}
			else {
				if(rev_leg_prio_ && !null(disc_func_rev_leg_)) {
					ois_disc_func_ = disc_func_rev_leg_;					
				}
				else if(!null(ois_disc_func_)){
					disc_func_rev_leg_ = ois_disc_func_;
				}				
				else if(!null(disc_func_rev_leg_)) {
					ois_disc_func_ = disc_func_rev_leg_;
				}
				if(!sync_endog_ok_) {
					if(!null(ois_fwd_disc_func_))
						ois_fwd_func_ = null;
					else if(!null(ois_fwd_func_))
						ois_fwd_disc_func_ = null;
				}
			}
		}
		else {	//crosscurr
			if(is_endog_) {
				
				if(!null(ois_disc_func_)){
					if(!sync_endog_ok_){
						ois_fwd_func_ = null;
						ois_fwd_disc_func_ = null;
					}
					else {
						ois_fwd_disc_func_ = ois_disc_func_;
					}
				}
				else if(!null(ois_fwd_disc_func_)){
					if(!sync_endog_ok_) {
						ois_fwd_func_ = null;
					}
					ois_disc_func_ = ois_fwd_disc_func_;
				}				
			}
			else {
				if(!sync_endog_ok_) {
					if(!null(ois_fwd_disc_func_))
						ois_fwd_func_ = null;
					else if(!null(ois_fwd_func_))
						ois_fwd_disc_func_ = null;
				}
			}
		}
		
		need_calc_[0] = null(disc_func_rev_leg_);
		need_calc_[1] = null(ois_disc_func_);
		need_calc_[2] = null(ois_fwd_func_);
		need_calc_[3] = null(ois_fwd_disc_func_) ;
		return;
	}
	/*-----------------------------------------------------------------------
	  ois_df_check
	----------------------------------------------------------------------*/	
	logical ois_df_check(logical is_endog,
						 logical is_ccy,
						 ..disc_func option(nullable) disc_func_rev_leg,
						 ..disc_func option(nullable) ois_disc_func,
						 ..fwd_func option(nullable) ois_fwd_func,
						 ..disc_func option(nullable) ois_fwd_disc_func,
						 out string err_msg)
	{
		try {
			err_msg = "";
			if(!is_ccy) {
				if(is_endog) {
					if(!null(disc_func_rev_leg))
						QL_FAIL_COND(!null(ois_disc_func) || !null(ois_fwd_func) || !null(ois_fwd_disc_func), "ambigious input: only one disc_func/fwd_func required for single currency and self discounted ois leg", E_INIT);				
					else if(!null(ois_disc_func))
						QL_FAIL_COND(!null(ois_fwd_disc_func) || !null(ois_fwd_func), "ambigious input: only one disc_func/fwd_func required for single currency and self discounted ois leg", E_INIT);
					else if(!null(ois_fwd_func))
						QL_FAIL_COND(!null(ois_fwd_disc_func) , "ambigious input: only one disc_func/fwd_func required for single currency and self discounted ois leg", E_INIT);				
				}
				else {
					QL_FAIL_COND(!null(ois_fwd_func) && !null(ois_fwd_disc_func),"ambigious input (ois_fwd_func/ois_disc_func)");
					QL_FAIL_COND(!null(disc_func_rev_leg) && !null(ois_disc_func), "ambigious input: only one disc_func required for single currency", E_INIT);				
					
				}
			}
			else {
				if(is_endog) {
					 if(!null(ois_disc_func))
						QL_FAIL_COND(!null(ois_fwd_disc_func) || !null(ois_fwd_func), "ambigious input: only one disc_func/fwd_func required for self discounted ois leg", E_INIT);
					else if(!null(ois_fwd_func))
						QL_FAIL_COND(!null(ois_fwd_disc_func) , "ambigious input: only one disc_func/fwd_func required for self discounted ois leg", E_INIT);
				}
				else {
					QL_FAIL_COND(!null(ois_fwd_func) && !null(ois_fwd_disc_func),"ambigious input (ois_fwd_func/ois_disc_func)");
				}
			}
			return true;
		}
		catch {
			err_msg = err.message();
			return false;
		}
	}
	/*-----------------------------------------------------------------------
	  ois_df_fwd_helper
	  for single curr "screen"-swap,
	----------------------------------------------------------------------*/	
	void ois_df_fwd_helper(	logical 					is_ccy,
							..disc_func option(nullable) df_disc_func,//both legs
							fwd_func option(nullable) 	ois_fwd_f_func,
							disc_func option(nullable) 	ois_fwd_df_func,
							ir_index 					ois_ir_index,
							logical 					endog_df,
							logical 					is_approx_calc,
							date option(nullable)		trade_date,
							date option(nullable)		settle_date,
							date option(nullable) 		maturity,
							string option(nullable) 	maturity_code,
							out disc_func 				opp_leg_disc_func,
							out disc_func 				ois_disc_func,
							out tenor_surface 			ois_ts,
							logical 					sync_endog_ok = true)
	{
		ois_ts = null;
		
		if(null(trade_date)){
			opp_leg_disc_func = null<disc_func>;
			ois_disc_func = null<disc_func>;			
			return;
		}

		
		//QL_FAIL_COND(/*!endog_df &&*/ !null(ois_fwd_f_func) && !null(ois_fwd_df_func),"ambigious input");

		if(!is_ccy){
			if(!null(df_disc_func)) {
				opp_leg_disc_func = df_disc_func;
				ois_disc_func = df_disc_func;				
			}
			else if(!null(ois_disc_func)) {
				opp_leg_disc_func = ois_disc_func;
			}
			else if(!null(opp_leg_disc_func)) {
				ois_disc_func = opp_leg_disc_func;				
			}						
		}

		/*if(endog_df && !null(ois_disc_func)) {
			//in the single ccy case with self-discouting the priority order is 1) df_disc_func 2) ois_disc_func and 3) opp_leg_disc_func
			ois_fwd_df_func = ois_disc_func;
			ois_fwd_f_func = null;			
		}*/
		
		logical allow_extrap = false;
		
		if(endog_df) {

			//QL_FAIL_COND(!null(ois_disc_func) && (!null(ois_fwd_f_func) || !null(ois_fwd_df_func) ),"ambigious input");
			if(!null(ois_disc_func)) {
				if(!sync_endog_ok || null(ois_fwd_f_func)) {
					QL_FAIL_COND(null(maturity) &&  null(maturity_code ),"missing maturity/maturity_code (required when converting between fwd_func and disc_func)");
					date m = null(maturity) ? date_code(maturity_code).apply_fwd(trade_date+5,ois_ir_index.fixing_calendar()) : maturity;
					ois_fwd_f_func = CORE_INT.create_fwd_func_from_df_on(trade_date, m,ois_ir_index.fixing_calendar(),ois_ir_index.day_count_method(), ois_disc_func,false);
				}
			}
			else if(!null(ois_fwd_f_func)) {//convert fwd_func to disc_func
				if(!sync_endog_ok || null(ois_disc_func)) {
					QL_FAIL_COND(null(maturity) &&  null(maturity_code ),"missing maturity/maturity_code (required when converting between fwd_func and disc_func)");
					date m = null(maturity) ? date_code(maturity_code).apply_fwd(trade_date+5,ois_ir_index.fixing_calendar()) : maturity;
					ois_disc_func = CORE_INT.create_disc_func_from_fwd_on(trade_date,m,ois_fwd_f_func,ois_ir_index.day_count_method(),ois_ir_index.fixing_calendar(),true);
				}
			}
			else if(!null(ois_fwd_df_func)) {//convert disc_func to fwd_func
				
				ois_disc_func = ois_fwd_df_func;
				if(!sync_endog_ok || null(ois_fwd_f_func)) {
					//ois_fwd_f_func = fwd_func(ois_disc_func,365,trade_date,ois_ir_index.day_count_method(),ois_ir_index.fixing_calendar(),BD_FOLLOWING,false);
					QL_FAIL_COND(null(maturity) &&  null(maturity_code ),"missing maturity/maturity_code (required when converting between fwd_func and disc_func)");
					date m = null(maturity) ? date_code(maturity_code).apply_fwd(trade_date+5,ois_ir_index.fixing_calendar()) : maturity;
					ois_fwd_f_func = CORE_INT.create_fwd_func_from_df_on(trade_date, m,ois_ir_index.fixing_calendar(),ois_ir_index.day_count_method(), ois_disc_func,false);
				}
			}
			else if(!null(df_disc_func)) {//convert disc_func to fwd_func
				if(null(ois_disc_func))
					ois_disc_func = df_disc_func;
				if(!sync_endog_ok || null(ois_fwd_f_func)) {
					//ois_fwd_f_func = fwd_func(ois_disc_func,365,trade_date,ois_ir_index.day_count_method(),ois_ir_index.fixing_calendar(),BD_FOLLOWING,false);
					QL_FAIL_COND(null(maturity) &&  null(maturity_code ),"missing maturity/maturity_code (required when converting between fwd_func and disc_func)");
					date m = null(maturity) ? date_code(maturity_code).apply_fwd(trade_date+5,ois_ir_index.fixing_calendar()) : maturity;
					ois_fwd_f_func = CORE_INT.create_fwd_func_from_df_on(trade_date, m,ois_ir_index.fixing_calendar(),ois_ir_index.day_count_method(), ois_disc_func,false);
				}
			}
		}
		else {//independent fwd and disc
			if(!null(df_disc_func) && null(ois_disc_func)){
				ois_disc_func = df_disc_func;
			}
			if(null(ois_disc_func)) {
				
				if(!null(ois_fwd_f_func)) {//convert fwd_func to disc_func
					//QL_FAIL_COND(null(settle_date),"missing settle_date (required when converting fwd_func to disc_func)");
					date m = null(maturity) ? date_code(maturity_code).apply_fwd(trade_date+5,ois_ir_index.fixing_calendar()) : maturity;					
					ois_disc_func = CORE_INT.create_disc_func_from_fwd_on(trade_date,m,ois_fwd_f_func,ois_ir_index.day_count_method(),ois_ir_index.fixing_calendar(),true);

				}
				else if(!null(ois_fwd_df_func)){
					ois_disc_func = ois_fwd_df_func;
				}				
			}

			if(null(ois_fwd_f_func)) {
				
				if(!null(ois_fwd_df_func)){
					//ois_fwd_f_func = fwd_func(ois_fwd_df_func,365,trade_date,ois_ir_index.day_count_method(),ois_ir_index.fixing_calendar(),BD_FOLLOWING,false);
					QL_FAIL_COND(null(maturity) &&  null(maturity_code ),"missing maturity/maturity_code (required when converting between fwd_func and disc_func)");
					date m = null(maturity) ? date_code(maturity_code).apply_fwd(trade_date+5,ois_ir_index.fixing_calendar()) : maturity;
					ois_fwd_f_func = CORE_INT.create_fwd_func_from_df_on(trade_date, m,ois_ir_index.fixing_calendar(),ois_ir_index.day_count_method(), ois_disc_func,false);
				}
				else if(!null(ois_disc_func)){
					//ois_fwd_f_func = fwd_func(ois_disc_func,365,trade_date,ois_ir_index.day_count_method(),ois_ir_index.fixing_calendar(),BD_FOLLOWING,false);
					QL_FAIL_COND(null(maturity) &&  null(maturity_code ),"missing maturity/maturity_code (required when converting between fwd_func and disc_func)");
					date m = null(maturity) ? date_code(maturity_code).apply_fwd(trade_date+5,ois_ir_index.fixing_calendar()) : maturity;
					ois_fwd_f_func = CORE_INT.create_fwd_func_from_df_on(trade_date, m,ois_ir_index.fixing_calendar(),ois_ir_index.day_count_method(), ois_disc_func,false);
				}
			}
		}
		
		//create tenor surface
		if(is_approx_calc){
			if(!null(ois_fwd_f_func)){
				if(!null(ois_fwd_df_func)){
					//we know ois_fwd_f_func has been created from ois_fwd_df_func
					ois_ts = tenor_surface(trade_date, ois_fwd_df_func,[ois_fwd_f_func],[ois_ir_index],allow_extrap);
				}
				else if(!null(ois_disc_func)){
					//we know ois_fwd_f_func has been created from ois_disc_func
					ois_ts = tenor_surface(trade_date, ois_disc_func,[ois_fwd_f_func],[ois_ir_index],allow_extrap);
				}
				else {
					//QL_FAIL_COND(null(settle_date),"missing settle_date (required when converting fwd_func to disc_func)");
					date m = null(maturity) ? date_code(maturity_code).apply_fwd(trade_date+5,ois_ir_index.fixing_calendar()) : maturity;
					disc_func df = CORE_INT.create_disc_func_from_fwd_on(trade_date,m,ois_fwd_f_func,ois_ir_index.day_count_method(),ois_ir_index.fixing_calendar(),true);				
					ois_ts = tenor_surface(trade_date, df,[ois_fwd_f_func],[ois_ir_index],allow_extrap);
				}
			}
			else{
				ois_ts = null;
			}
		}
		else {
			//df is not used, when is_approx_calc = false, but cannot be null in tenor_surface(...)
			if(!null(ois_fwd_f_func)) {
				if(!null(ois_fwd_df_func)){
					//we know ois_fwd_f_func has been created from ois_fwd_df_func
					ois_ts = tenor_surface(trade_date, ois_fwd_df_func,[ois_fwd_f_func],[ois_ir_index],allow_extrap);
				}
				else if(!null(ois_disc_func)){
					//we know ois_fwd_f_func has been created from ois_disc_func
					ois_ts = tenor_surface(trade_date, ois_disc_func,[ois_fwd_f_func],[ois_ir_index],allow_extrap);
				}
				else {
					disc_func df = disc_func_interp([1],[1],ip_linear());//fejk disc_func ok because it will never be used
					ois_ts = tenor_surface(trade_date, df,[ois_fwd_f_func],[ois_ir_index],allow_extrap);
				}
			}
			else{
				ois_ts = null;
			}
		}
		
		if(!is_ccy)
			opp_leg_disc_func = ois_disc_func;
	}
}