/*	
	template classes
	Developer: Algorithmica Research, Magnus Nyström			
*/

option(null: hard);	

class enum mtm_code
{
	NONE 		option(str: "None"),
	LEG1_MTM 	option(str: "Mtm_leg1"),
	LEG2_MTM 	option(str: "Mtm_leg2")
};

logical is_mtm(logical 						is_cross_curr,
			   mtm_code option(nullable)	mtm,
			   out logical 					leg1)
{
	if( !is_cross_curr){
		leg1 = false;
		return false;
	}
	else if(null(mtm) || mtm == mtm_code.NONE){
		leg1 = false;
		return false;
	}
	else {
		leg1 = mtm == mtm_code.LEG1_MTM;
		return true;
	}
}


class enum quote_style_eq_opt
{
	VOL 		option(str: "volatility"),
	VOL_PCNT 	option(str: "volatility_pct"),
	PREM 		option(str: "premium"),
	PREM_PCNT	option(str: "premium_pct")
};

ir_index create_ir_index(string 							idx_name,
						 string 							idx_type,
						string 	option(nullable)			idx_tenor,
						number option(nullable)	 			idx_value_days,
						number option(nullable)	 			idx_fixing_days,
						day_count_method option(nullable) 	idx_day_count_method,
						bd_convention option(nullable) 		idx_bd_convention,
						logical option(nullable) 			idx_eom,
						calendar  							idx_calendar,//if libor: this is uk
						string option(nullable)				idx_calendar_name,
						string  							idx_currency,
						calendar option(nullable) 			idx_suppl_calendar,	//if libor: this is primary(ex.target), if ibor: this is null
						string option(nullable) 			idx_suppl_calendar_name,
						number option(nullable)				idx_spot_days,
						number option(nullable)				idx_spread)
{
	if(null(idx_calendar_name))
		idx_calendar_name = "idx_calendar";
	ir_index idx;
	if(equal_casei (idx_type, "IBOR")){
		QL_FAIL_COND(null(idx_value_days) || idx_value_days < 0, "invalid value days", E_INIT);
		QL_FAIL_COND(null(idx_day_count_method), "invalid daycount method", E_INIT);
		QL_FAIL_COND(null(idx_bd_convention), "invalid business day method", E_INIT);
		QL_FAIL_COND(null(idx_eom), "invalid end of month setting", E_INIT);
		QL_FAIL_COND(null(idx_spot_days), "invalid spot days", E_INIT);
		idx = ir_index_ibor(idx_name,idx_tenor,integer(idx_value_days),idx_fixing_days,idx_currency,
							idx_calendar,idx_calendar_name,idx_day_count_method,idx_bd_convention,
							idx_eom,integer(idx_spot_days));
	}
	else if(equal_casei (idx_type, "LIBOR")){
		QL_FAIL_COND(null(idx_value_days)|| idx_value_days < 0, "invalid value days", E_INIT);
		QL_FAIL_COND(null(idx_day_count_method), "invalid daycount method", E_INIT);
		if(null(idx_suppl_calendar_name))
			idx_suppl_calendar_name = "suppl_calendar";
		idx = ir_index_libor(idx_name,idx_tenor,integer(idx_value_days),idx_currency,idx_calendar,
							 idx_calendar_name,idx_suppl_calendar, idx_suppl_calendar_name,idx_day_count_method,
							 null(idx_spot_days) ? err_int() : integer(idx_spot_days));
	}
	else if(equal_casei (idx_type, "RFR") ){
		QL_FAIL_COND(null(idx_value_days)|| idx_value_days < 0, "invalid value days", E_INIT);
		QL_FAIL_COND(null(idx_day_count_method), "invalid daycount method", E_INIT);
		if(null(idx_spread))
			idx_spread = 0;
		idx = ir_index_rfr(idx_name,integer(idx_value_days),idx_fixing_days,idx_currency,idx_calendar,
							 idx_calendar_name,idx_day_count_method,idx_spread);
	}
	else if( equal_casei (idx_type, "RFRCOMPIDX")){
		QL_FAIL_COND(null(idx_value_days)|| idx_value_days < 0, "invalid value days", E_INIT);
		QL_FAIL_COND(null(idx_day_count_method), "invalid daycount method", E_INIT);
		idx = ir_index_rfr_idx(idx_name,integer(idx_value_days),idx_fixing_days,idx_currency,idx_calendar,
							 idx_calendar_name,idx_day_count_method);
	}
	else if(equal_casei (idx_type, "RB_REPO")){
		if(null(idx_day_count_method))
			idx_day_count_method = DC_ACT_360;
		idx = ir_index_rb_repo(idx_name,idx_currency,idx_calendar,idx_calendar_name);
	}
	else
		QL_FAIL("index type not applicable", E_INIT);

	return idx;
}

module INSTR_TMPL
{
	//logical display_warning = true;

	integer spot_days_dflt(string curr) 
	{
	    switch (str_to_upper(curr)) {
		case "GBP":
		case "AUD":
		case "CAD":                  
	    case "HKD":
		case "NZD": 
		case "RMB": //shibor, shanghai
		case "ZAR": 
	       return 0;
		default:
	       return 2;
	    }
    }

	
	//-------------------------------------------------------
	// class ir_index
	// for example libor and ibor indexes
	//-------------------------------------------------------
	public class ir_index_tmpl
	{
		public:
			ir_index_tmpl(	string,string,day_count_method, bd_convention,
							logical,calendar ,string,string, number option(nullable) v = null<number>,							
							calendar option(nullable) sc = null<calendar>,
							string option(nullable) scn = null<string>);
		
			string				name();	
			string				type();
			day_count_method 	day_count_method(); 						
			virtual bd_convention	bd_convention(); 	 					
			virtual logical			eom();				
			calendar			cal();//uk for libor index
			string				cal_name();
			calendar			suppl_cal();
			string				suppl_cal_name();	
			string				currency();
			integer 			spot_days();				
			logical				suppl_cal_adjust();

			//virtual string		name_tenor();
			//virtual string		tenor();
			//virtual integer  	fixing_days();
			//virtual integer  	value_days();

		protected:
			ir_index_tmpl(ir_index_tmpl);
			logical				is_libor_index();
			
			string				name_;
			string				type_;	
			day_count_method 	dc_; 						
			bd_convention 		bd_; 	 					
			logical				eom_;
			calendar 			cal_;
			string				cal_name_;
			string				curr_;	
			calendar 			suppl_cal_;
			string 				suppl_cal_name_;		
			integer				spot_days_;					
	};
	ir_index_tmpl.ir_index_tmpl(string				name,
								string				type,							
								..day_count_method 	dc, 						
								..bd_convention 	bd, 	 								
								logical				eom,										
								calendar 			cal,
								string 				cal_name,
								string				curr,								
								number option(nullable)	spot_days,
								calendar option(nullable) suppl_cal,
								string option(nullable) suppl_cal_name)
								: name_(name),type_(str_to_upper(type)),dc_(dc),bd_(bd), eom_(eom),cal_(cal),
								  cal_name_(cal_name),curr_(curr), suppl_cal_(suppl_cal), suppl_cal_name_(suppl_cal_name), 
								  spot_days_(null(spot_days)? spot_days_dflt(curr):integer(spot_days)) 
	{
		if(null(suppl_cal_))
			suppl_cal_name_ = null<string>;
		else if(null(suppl_cal_name_))
			suppl_cal_name_ = "noname";
			
	}

	ir_index_tmpl.ir_index_tmpl(ir_index_tmpl i)
								: name_(i.name_),type_(i.type_),dc_(i.dc_),bd_(i.bd_), eom_(i.eom_),cal_(i.cal_), 
								  cal_name_(i.cal_name_),curr_(i.curr_), suppl_cal_(i.suppl_cal_), suppl_cal_name_(i.suppl_cal_name_),
								  spot_days_(i.spot_days_) {}

	string				ir_index_tmpl.type() 				{ return type_;}
	string				ir_index_tmpl.name() 				{ return name_;}	
	day_count_method 	ir_index_tmpl.day_count_method()	{ return dc_;} 						
	bd_convention 		ir_index_tmpl.bd_convention()		{ return bd_;} 	 					
	logical				ir_index_tmpl.eom()					{ return eom_;}
	calendar			ir_index_tmpl.cal()					{ return null(cal_) ? calendar() : cal_;}
	string				ir_index_tmpl.cal_name() 			{ return cal_name_;}
	calendar			ir_index_tmpl.suppl_cal()			{ return suppl_cal_ ;}
	string				ir_index_tmpl.suppl_cal_name()		{ return suppl_cal_name_ ;}
	string				ir_index_tmpl.currency()			{ return curr_;}
	integer 			ir_index_tmpl.spot_days()			{ return spot_days_;}
	logical				ir_index_tmpl.is_libor_index() 		{ return equal_casei(type_, "LIBOR");}
	logical				ir_index_tmpl.suppl_cal_adjust() 	{ return is_libor_index();}
	//string				ir_index_tmpl.name_tenor()			{ return null<string>;}
	//string				ir_index_tmpl.tenor()				{ return null<string>;}
	//integer 			ir_index_tmpl.fixing_days()			{ return null<number>;}
	//integer 			ir_index_tmpl.value_days()			{ return null<number>;}

	//-------------------------------------------------------
	// class ir_index_tenor_tmpl
	//-------------------------------------------------------
	public class ir_index_tenor_tmpl: public ir_index_tmpl
	{
	public:
		ir_index_tenor_tmpl(ir_index_tmpl,string,string,integer ,
							number option(nullable) = null<number>,
							number option(nullable) = null<number>);

		string		name_tenor();
		string		tenor();
		integer 	fixing_days();
		integer 	value_days();
		logical		pre_spot();
		number		spread();
		override bd_convention	bd_convention(); 	 					
		override logical		eom();
		logical 	is_1d_tenor();
		logical		is_my_tenor();
	protected:
				
		string			name_;		
		string 			tenor_;	
		integer 		val_days_;
		integer 		fix_days_;
		number 			spread_;
	};

	ir_index_tenor_tmpl.ir_index_tenor_tmpl(ir_index_tmpl 	index,
											string			name,
											string 			tenor,
											integer 		val_days,
											number option(nullable)	fix_days,
											number option(nullable) spreadbp)
											:ir_index_tmpl(index), name_(name), tenor_(str_to_upper(tenor)),val_days_(val_days),
											 fix_days_(null(fix_days) ? val_days : integer(fix_days)),
											 spread_(null(spreadbp) ? 0 : spreadbp/10000.0)
	{
		if(equal_casei(tenor_, "CD1")) {
			tenor_ = "BD1";
		}
		else if(equal_casei(tenor_, "BD1")) {}
		else {
			QL_REQUIRE(!find_string(tenor_, "IMM"),"invalid tenor");
			logical ok = find_string(tenor_, "M") || find_string(tenor_, "Y") || find_string(tenor_, "W");
			QL_REQUIRE(ok,"invalid tenor");
		}
			
		QL_REQUIRE(val_days_ == 0 || val_days_ == 1 || val_days_ == 2, "invalid index value days (0, 1 or 2 are allowed)");

		if(equal_casei(type_, "RFR")) {
			QL_REQUIRE(equal_casei(tenor_, "BD1"),"RFR must have tenor = BD1");
			QL_REQUIRE(fix_days_ == -1 || fix_days_ == 0 || fix_days_ == 1 ||fix_days_ == 2, "invalid index fixing days (-1, 0, 1 or 2 for daily tenor allowed)");
		}
		else if(equal_casei(type_, "RFRCOMPIDX")) {
			QL_REQUIRE(equal_casei(tenor_, "BD1"),"RFRCOMPIDX must have tenor = BD1");
			QL_REQUIRE(fix_days_ == -1 || fix_days_ == 0 || fix_days_ == 1 ||fix_days_ == 2, "invalid index fixing days (-1, 0, 1 or 2 for daily tenor allowed)");
			QL_REQUIRE(CORE_INT.is_equal_zero(spread_),"index spread is only applicable to rfr") ;			
		}
		else {
			QL_REQUIRE(CORE_INT.is_equal_zero(spread_),"index spread is only applicable to rfr") ;
			if(equal_casei(tenor_, "BD1"))
				QL_REQUIRE(fix_days_ == -1 || fix_days_ == 0 || fix_days_ == 1 ||fix_days_ == 2, "invalid index fixing days (-1, 0, 1 or 2 for daily tenor allowed)");
			else
				QL_REQUIRE(fix_days_ == 0 || fix_days_ == 1 || fix_days_ == 2, "Invalid index fixing days (0, 1 or 2 for non-daily tenor allowed)");
		}
	}

	string		ir_index_tenor_tmpl.name_tenor()	{ return name_;}	
	string		ir_index_tenor_tmpl.tenor() 		{ return tenor_;}
	integer 	ir_index_tenor_tmpl.fixing_days()	{ return fix_days_;}
	integer 	ir_index_tenor_tmpl.value_days()	{ return val_days_;}	
	logical		ir_index_tenor_tmpl.pre_spot()		{ return val_days_ < spot_days_;}
	number 		ir_index_tenor_tmpl.spread()		{ return spread_;}
	logical 	ir_index_tenor_tmpl.is_1d_tenor()	{ return equal_casei(tenor_, "BD1") ;}
	
	logical		ir_index_tenor_tmpl.is_my_tenor()
	{
		return find_string(tenor_, "M") || find_string(tenor_, "Y");
	}
	
	..bd_convention  ir_index_tenor_tmpl.bd_convention()
	{
		if(bd_ == ..bd_convention.BD_MOD_FOLLOWING && !is_my_tenor()) {
			return ..bd_convention.BD_FOLLOWING;
		}
		else {
			return bd_;
		}
	}
	
	logical ir_index_tenor_tmpl.eom()
	{
		return !is_my_tenor() ? false : eom_;
	}
}

//-------------------------------------------------------
// create_ir_index
//-------------------------------------------------------
ir_index create_ir_index(INSTR_TMPL.ir_index_tenor_tmpl tmpl)
{
	return create_ir_index(	tmpl.name(),
							tmpl.type(),
							tmpl.tenor(),
							tmpl.value_days(),
							tmpl.fixing_days(),
							tmpl.day_count_method(),
							tmpl.bd_convention(),
							tmpl.eom(),
							tmpl.cal(),//if libor: this is uk
							tmpl.cal_name(),
							tmpl.currency(),
							tmpl.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
							tmpl.suppl_cal_name(),
							tmpl.spot_days(),
							tmpl.spread());
}
	
module INSTR_TMPL
{	
	
	//-------------------------------------------------------
	// class deposit_def_tmpl
	//-------------------------------------------------------
	public class instrument_def_tmpl 
	{
	public:
		instrument_def_tmpl(..instr_type option(nullable));
		instr_type instr_type();
	protected:
		instr_type type_;
	};

	instrument_def_tmpl.instrument_def_tmpl(..instr_type option(nullable) t) : type_(t){}
	instr_type instrument_def_tmpl.instr_type() { return type_;}

	//-------------------------------------------------------
	// class deposit_def_tmpl
	//-------------------------------------------------------
	public class deposit_def_tmpl: public  instrument_def_tmpl
	{
	   	public:
			deposit_def_tmpl(string,day_count_method, bd_convention,logical,string,string	option(nullable),..calendar,string,
							logical is_o = false, ..calendar option(nullable) sc = null<calendar>);
			
			string				name();	
			day_count_method 	day_count_method(); 						
			bd_convention 		bd_convention(); 	 					
			logical				eom();
			string				settle_code();
			string				maturity_code();
			calendar			calendar();
			calendar			suppl_cal();
			logical				adj_suppl_cal();
			string				currency();

		protected:
			string				name_;	
			day_count_method 	dc_; 						
			bd_convention 		bd_; 	 					
			logical				eom_;
			string				settle_code_;
			string				mat_code_;
			calendar			cal_;
			string				curr_;
			logical 			adj_suppl_cal_;
			calendar			suppl_cal_;	
			
	};

	deposit_def_tmpl.deposit_def_tmpl(	string			name,	
									..day_count_method 	dc, 						
									..bd_convention 	bd, 	 								
									logical				eom,
									string				settle_code,
									string	option(nullable) mat_code,
									..calendar				cal,
									string				curr,
									logical				adj_suppl_cal,
									..calendar option(nullable) suppl_cal)
									: instrument_def_tmpl(..instr_type.DEPOSIT), name_(name), dc_(dc),bd_(bd), eom_(eom), settle_code_(settle_code), 
									  mat_code_(mat_code),cal_(cal), curr_(curr), adj_suppl_cal_(adj_suppl_cal), suppl_cal_(suppl_cal) {}

	string				deposit_def_tmpl.name() 			{ return name_;}	
	day_count_method 	deposit_def_tmpl.day_count_method()	{ return dc_;} 						
	bd_convention 		deposit_def_tmpl.bd_convention()	{ return bd_;} 	 					
	logical				deposit_def_tmpl.eom()				{ return eom_;}
	string				deposit_def_tmpl.settle_code()		{ return settle_code_;}
	string				deposit_def_tmpl.maturity_code()	{ return mat_code_;}
	calendar			deposit_def_tmpl.calendar()			{ return cal_;}
	calendar			deposit_def_tmpl.suppl_cal()		{ return adj_suppl_cal_ ? suppl_cal_ : null<..calendar>;}
	string				deposit_def_tmpl.currency()			{ return curr_;}
	logical				deposit_def_tmpl.adj_suppl_cal() 	{ return adj_suppl_cal_;}

	//-------------------------------------------------------
	// class fra_def_tmpl
	//-------------------------------------------------------
	public class fra_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			fra_def_tmpl(string,date option(nullable),string option(nullable) ,date option(nullable) ,
						string option(nullable) ,logical option(nullable),
						day_count_method, bd_convention,logical,						
						..calendar,string, fra_calc_method ,ir_index_tenor_tmpl);
			
			string				name();	
			day_count_method 	day_count_method(); 						
			bd_convention 		bd_convention(); 	 					
			logical				eom();
			calendar			calendar();			
			string				currency();	
			date				expiry_settle_date();	
			string				expiry_settle_code();
			date				underl_maturity();
			string				underl_maturity_code();
			logical				mat_code_off_trade();
			logical				suppl_cal_adjust();
			//ir_index_tenor_tmpl	index_tenor();
			fra_calc_method fra_style();
			ir_index_tenor_tmpl	index();
			ir_index			ir_index();
		protected:
			string				name_;	
			day_count_method 	dc_; 						
			bd_convention 		bd_; 	 					
			logical				eom_;
			date				expiry_settle_date_;
			string 				expiry_settle_code_;
			date				underl_maturity_;
			string 				underl_maturity_code_;
			logical 			mat_code_off_trade_;
			..calendar				cal_;
			string				curr_;
			fra_calc_method 	fra_style_;			
			ir_index_tenor_tmpl	index_;			
			
	};

	fra_def_tmpl.fra_def_tmpl(	string						name,
								date option(nullable) 		expiry_settle_date,
								string option(nullable) 	expiry_settle_code,
								date option(nullable) 		underl_maturity,
								string option(nullable) 	underl_maturity_code,
								logical option(nullable) 	mat_code_off_trade,	
								..day_count_method 			dc, 						
								..bd_convention 			bd, 	 								
								logical						eom,
								..calendar					cal,
								string						curr,
								fra_calc_method 			fra_style,
								ir_index_tenor_tmpl			index)
							: 	instrument_def_tmpl(..instr_type.FRA), name_(name), dc_(dc),bd_(bd), eom_(eom),expiry_settle_date_(expiry_settle_date),
								expiry_settle_code_(expiry_settle_code),underl_maturity_(underl_maturity),
								underl_maturity_code_(underl_maturity_code),mat_code_off_trade_(mat_code_off_trade),
								cal_(cal), curr_(curr), 
								fra_style_(fra_style), index_(index) {}

	string				fra_def_tmpl.name() 				{ return name_;}	
	day_count_method 	fra_def_tmpl.day_count_method()		{ return dc_;} 						
	bd_convention 		fra_def_tmpl.bd_convention()		{ return bd_;} 	 					
	logical				fra_def_tmpl.eom()					{ return eom_;}
	string				fra_def_tmpl.expiry_settle_code()	{ return expiry_settle_code_;}
	date				fra_def_tmpl.expiry_settle_date()	{ return expiry_settle_date_;}
	string				fra_def_tmpl.underl_maturity_code()	{ return underl_maturity_code_;}
	date				fra_def_tmpl.underl_maturity()		{ return underl_maturity_;}
	logical				fra_def_tmpl.mat_code_off_trade()	{ return mat_code_off_trade_;}
	calendar			fra_def_tmpl.calendar()				{ return cal_;}
	string				fra_def_tmpl.currency()				{ return curr_;}
	logical				fra_def_tmpl.suppl_cal_adjust()		{ return index_.suppl_cal_adjust();}
	ir_index_tenor_tmpl	fra_def_tmpl.index()				{ return index_;}
	//ir_index_tenor_tmpl	fra_def_tmpl.index_tenor()	{ return index_;}
	fra_calc_method 	fra_def_tmpl.fra_style()			{ return fra_style_;}

	ir_index fra_def_tmpl.ir_index()
	{
		string t = index_.type();
		if(!equal(t, "IBOR") && !equal (t, "LIBOR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	index_.name(),
								t,
								index_.tenor(),
								index_.value_days(),
								index_.fixing_days(),
								index_.day_count_method(),
								index_.bd_convention(),
								index_.eom(),
								index_.cal(),//if libor: this is uk
								index_.cal_name(),
								index_.currency(),
								index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								index_.suppl_cal_name(),
								index_.spot_days(),0);
	}
	//-------------------------------------------------------
	// class irfut_def_tmpl  
	//-------------------------------------------------------
	public class irfut_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			irfut_def_tmpl(	string,date option(nullable),string option(nullable),string option(nullable),day_count_method, bd_convention,logical,
							..calendar option(nullable),string, irfut_calc_method,
							fut_tick_value_style,quote_style,number,number option(nullable), 
							ir_index_tenor_tmpl);
			
			string				name();
			string				underlying_mat_code();
			logical				mat_code_off_trade();		
			day_count_method 	day_count_method(); 						
			bd_convention 		bd_convention(); 	 					
			logical				eom();
			calendar			calendar();			
			string				currency();		
			logical				suppl_cal_adjust();
			irfut_calc_method	style();

			date 				expiration_settle_date(); 
			string 				expiration_settle_code();
			fut_tick_value_style tick_val_style();

			quote_style 		quote_style(); 
			ir_index_tenor_tmpl	index();
			number				contract_val();
			number				tick_size();
			ir_index 			ir_index();

		protected:
			string				name_;
			date  				expiration_settle_date_; 
			string 				expiration_settle_code_;
			string				mat_code_;	
			day_count_method 	dc_; 						
			bd_convention 		bd_; 	 					
			logical				eom_;
			..calendar			cal_;
			string				curr_;
			irfut_calc_method	style_;
			fut_tick_value_style tick_val_style_;
			quote_style			qs_;
			number				contract_val_;
			number				tick_size_;			
			ir_index_tenor_tmpl	index_;			
			
	};

	irfut_def_tmpl.irfut_def_tmpl(	string					name,
									date option(nullable) 		expiration_settle_date, 
									string option(nullable)		expiration_settle_code,
									string	option(nullable)	underlying_mat_code,	
									..day_count_method 		dc, 						
									..bd_convention 		bd, 	 								
									logical					eom,							
									..calendar option(nullable) cal,
									string					curr,
									irfut_calc_method 	style,
									fut_tick_value_style 	tick_val_style,
									..quote_style	qs,
									number					contract_val,
									number	option(nullable) tick_size,								
									ir_index_tenor_tmpl		index)
								: 	instrument_def_tmpl(..instr_type.IR_FUTURE), name_(name), expiration_settle_date_(expiration_settle_date),expiration_settle_code_(expiration_settle_code),
									mat_code_(underlying_mat_code), dc_(dc),bd_(bd), eom_(eom),cal_(cal), curr_(curr), 
									style_(style), tick_val_style_(tick_val_style),qs_(qs), contract_val_(contract_val), tick_size_(tick_size), index_(index) {}

	string				irfut_def_tmpl.name() 					{ return name_;}
	string				irfut_def_tmpl.underlying_mat_code()	{ return mat_code_;}
	logical				irfut_def_tmpl.mat_code_off_trade()		{ return false;}	
	day_count_method 	irfut_def_tmpl.day_count_method()		{ return dc_;} 						
	bd_convention 		irfut_def_tmpl.bd_convention()			{ return bd_;} 	 					
	logical				irfut_def_tmpl.eom()					{ return eom_;}
	calendar			irfut_def_tmpl.calendar()				{ return null(cal_) ? ..calendar() : cal_;}
	string				irfut_def_tmpl.currency()				{ return curr_;}
	logical				irfut_def_tmpl.suppl_cal_adjust()		{ return index_.suppl_cal_adjust();}
	irfut_calc_method	irfut_def_tmpl.style()			{ return style_;}
	quote_style 			irfut_def_tmpl.quote_style()	{ return qs_;}
	number				irfut_def_tmpl.contract_val()			{ return contract_val_;}
	number				irfut_def_tmpl.tick_size()				{ return tick_size_;}
	ir_index_tenor_tmpl	irfut_def_tmpl.index()					{ return index_;}
	date 				irfut_def_tmpl.expiration_settle_date()					{ return expiration_settle_date_;} 
	string 				irfut_def_tmpl.expiration_settle_code()					{ return expiration_settle_code_;}
	fut_tick_value_style 	irfut_def_tmpl.tick_val_style()	{ return tick_val_style_;}

	ir_index irfut_def_tmpl.ir_index()
	{
		string t = index_.type();
		if(!equal(t, "IBOR") && !equal (t, "LIBOR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	index_.name(),
								t,
								index_.tenor(),
								index_.value_days(),
								index_.fixing_days(),
								index_.day_count_method(),
								index_.bd_convention(),
								index_.eom(),
								index_.cal(),//if libor: this is uk
								index_.cal_name(),
								index_.currency(),
								index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								index_.suppl_cal_name(),
								index_.spot_days(),0);
	}
	//-------------------------------------------------------
	// class bill_def_tmpl
	//-------------------------------------------------------
	public class bill_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			bill_def_tmpl(	string,/*date option(nullable),date ,number option(nullable),*/ bill_yld_method ,
							day_count_method, string option(nullable),bill_cpn_type option(nullable), string ,
							..calendar, string, quote_style);
			
			string					name();
			//date					maturity();	
			//date					issue_date();
			//number				coupon();
			day_count_method 		day_count_method(); 				
			calendar				calendar();			
			string					currency();		
			string 					excpn_code();
			bill_yld_method		yld_method();
			string 					settle_code();
			quote_style quote_style();
			bill_cpn_type cpn_type(); 
			
		protected:
			string					name_;
			//date					issue_date_;
			//date					maturity_;
			//number				coupon_;
			bill_yld_method 	yld_method_;	
			day_count_method 		dc_; 						
			string 					excpn_code_; 
			bill_cpn_type		cpn_type_;	 					
			string					settle_code_;
			..calendar				cal_;
			string					curr_;			
			quote_style	qs_;			
			
	};

	bill_def_tmpl.bill_def_tmpl(string						name,
								//date option(nullable) 	issue_date,	
								//date 						maturity,
								//number option(nullable) 	coupon,
								bill_yld_method 	yld_method,
								..day_count_method 			dc, 						
								string option(nullable) 	exc, 	 								
								bill_cpn_type option(nullable) cpn_type,
								string 	 					settle_code,						
								..calendar	 				cal,
								string						curr,									
								..quote_style	qs)
								: 	instrument_def_tmpl(..instr_type.BILL), name_(name), yld_method_(yld_method), 
									dc_(dc),excpn_code_(exc), cpn_type_(cpn_type), settle_code_(settle_code), cal_(cal), curr_(curr), 
									qs_(qs) {}

	string					bill_def_tmpl.name() 				{ return name_;}
	//date					bill_def_tmpl.maturity()			{ return maturity_;}
	//date					bill_def_tmpl.issue_date()			{ return issue_date_;}
	day_count_method 		bill_def_tmpl.day_count_method()	{ return dc_;} 						
	//number 				bill_def_tmpl.coupon()				{ return coupon_;}
	bill_yld_method	bill_def_tmpl.yld_method()	{ return yld_method_;} 	 
	bill_cpn_type		bill_def_tmpl.cpn_type()	{ return cpn_type_;} 					
	string 					bill_def_tmpl.excpn_code()			{ return excpn_code_;} 
	string 					bill_def_tmpl.settle_code()			{ return settle_code_;} 
	calendar				bill_def_tmpl.calendar()			{ return null(cal_) ? ..calendar() : cal_;}
	string					bill_def_tmpl.currency()			{ return curr_;}	
	quote_style bill_def_tmpl.quote_style()			{ return qs_;}
	

	//-------------------------------------------------------
	// class bond_def_tmpl
	//-------------------------------------------------------
	public class bond_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			bond_def_tmpl(	string,bond_yld_method ,integer,day_count_method, logical,
							string option(nullable),first_cpn_type,last_per_yld_method,
							next_per_yld_method,odd_last_cpn_type,string ,..calendar,
							string, quote_style);
			
			string					name();
			integer					cpn_freq();
			day_count_method 		day_count_method(); 						
			logical					eom();
			calendar				calendar();			
			string					currency();		
			string 					excpn_code();
			first_cpn_type 			first_cpn_type();
    		last_per_yld_method		last_yld_method();
    		next_per_yld_method		next_yld_method(); 
    		odd_last_cpn_type		odd_last_cpn_type();	
			bond_yld_method			yld_method();
			string 					settle_code();
			quote_style 			quote_style();
			
			
		protected:
			string					name_;
			integer					cpn_freq_;
			bond_yld_method 		yld_method_;	
			day_count_method 		dc_;
			logical					eom_; 						
			string 					excpn_code_; 
			first_cpn_type 			first_cpn_type_;
    		last_per_yld_method	last_yld_method_;
    		next_per_yld_method	next_yld_method_; 
    		odd_last_cpn_type		odd_last_cpn_type_;				 					
			string					settle_code_;
			..calendar				cal_;
			string					curr_;			
			quote_style				qs_;			
			
	};

	bond_def_tmpl.bond_def_tmpl(string						name,									
								bond_yld_method 			yld_method,
								integer						cpn_freq,
								..day_count_method 			dc, 
								logical						eom,						
								string option(nullable) 	exc,
								..first_cpn_type 			first_cpn_type,
    							last_per_yld_method			last_yld_method,
    							next_per_yld_method			next_yld_method, 
    							..odd_last_cpn_type			odd_last_cpn_type, 	 																	
								string 	 					settle_code,						
								..calendar	 				cal,
								string						curr,									
								..quote_style				qs)
								: instrument_def_tmpl(..instr_type.FIXED_CPN_BOND), name_(name),cpn_freq_(cpn_freq), yld_method_(yld_method), dc_(dc),
								eom_(eom),excpn_code_(exc),first_cpn_type_(first_cpn_type),
								last_yld_method_(last_yld_method),next_yld_method_(next_yld_method),odd_last_cpn_type_(odd_last_cpn_type), 
								settle_code_(settle_code), cal_(cal), curr_(curr),  qs_(qs) {}

	string					bond_def_tmpl.name() 				{ return name_;}
	day_count_method 		bond_def_tmpl.day_count_method()	{ return dc_;} 						
	bond_yld_method			bond_def_tmpl.yld_method()			{ return yld_method_;} 	 
	logical					bond_def_tmpl.eom()					{ return eom_;} 
	integer					bond_def_tmpl.cpn_freq()			{ return cpn_freq_;} 					
	string 					bond_def_tmpl.excpn_code()			{ return excpn_code_;} 
	first_cpn_type 			bond_def_tmpl.first_cpn_type()		{ return first_cpn_type_;} 
    last_per_yld_method		bond_def_tmpl.last_yld_method()		{ return last_yld_method_;} 
    next_per_yld_method		bond_def_tmpl.next_yld_method()		{ return next_yld_method_;}  
    odd_last_cpn_type		bond_def_tmpl.odd_last_cpn_type() 	{ return odd_last_cpn_type_;}  
	string 					bond_def_tmpl.settle_code()			{ return settle_code_;} 
	calendar				bond_def_tmpl.calendar()			{ return null(cal_) ? ..calendar() : cal_;}
	string					bond_def_tmpl.currency()			{ return curr_;}	
	quote_style bond_def_tmpl.quote_style()						{ return qs_;}


	//-------------------------------------------------------
	// class frn_def_tmpl
	//-------------------------------------------------------
	public class frn_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			frn_def_tmpl(	string,integer,frn_roll_method ,day_count_method,bd_convention,..calendar,					
							logical,string,string,string, frn_neg_rate_rule, ir_index_tenor_tmpl );

			frn_def_tmpl(	string,integer,frn_roll_method ,day_count_method,bd_convention,..calendar,					
							logical,string,string,string, frn_neg_rate_rule, ir_index_tenor_tmpl,
							rfr_arrears_type ,
							number option(nullable) ,
							logical option(nullable) 	= true,
							//flt_sprd_comp_method option(nullable) = flt_sprd_comp_method.SC_SPRD_EXCL,
							//flt_avg_method option(nullable) 	= null<flt_avg_method>,
							integer payment_lag 				= 0,
							logical option(nullable) 			= true);
			
			string				name();
			integer				cpn_freq();
			frn_roll_method 	roll_method();
			day_count_method 	day_count_method(); 
			bd_convention		bd_convention();						
			logical				eom();
			calendar			calendar();			
			string				currency();		
			string 				excpn_code();			
			string 				settle_code();	
			frn_neg_rate_rule	neg_rate_rule();				
			ir_index_tenor_tmpl	index()	;
			ir_index 			ir_index();
			rfr_arrears_type 	arr_type();
			number 				arr_days();
			avg_comp_method 	comp_avg();
    		//flt_sprd_comp_method spread_comp_method();
    		//flt_avg_method 		avg_method();
			integer				payment_lag();
			logical				payment_lag_ex_mat();
			logical				is_rfr_frn();
		protected:
			string				name_;
			integer				cpn_freq_;
			frn_roll_method 	roll_method_;
			day_count_method 	dc_;
			bd_convention		bd_;
			logical				eom_; 						
			string 				excpn_code_; 			 					
			string				settle_code_;
			..calendar			cal_;
			string				curr_;
			frn_neg_rate_rule	neg_rate_rule_;			
			ir_index_tenor_tmpl	index_;
			rfr_arrears_type 	arr_type_;
			number 				arr_days_;
			avg_comp_method 	comp_avg_;
    		//flt_sprd_comp_method spread_comp_method_;
    		//flt_avg_method 		avg_method_;
			integer				payment_lag_;
			logical				payment_lag_ex_mat_;
	};

	frn_def_tmpl.frn_def_tmpl(	string				name,
							integer 				cpn_freq,						       
							frn_roll_method  		roll_method,   
							..day_count_method 		dc,
							..bd_convention 		bd,
							..calendar				cal,					
							logical 				eom,  
							string 					exc,
							string 					settle_code,						
							string 					curr,
							frn_neg_rate_rule	 	neg_rule,
							ir_index_tenor_tmpl		index )
							: instrument_def_tmpl(..instr_type.FRN), name_(name),cpn_freq_(cpn_freq), roll_method_(roll_method), dc_(dc),bd_(bd),
							eom_(eom),excpn_code_(exc),settle_code_(settle_code), cal_(cal), curr_(curr), 
							neg_rate_rule_(neg_rule), index_(index), arr_type_(null<rfr_arrears_type>), arr_days_(null<number>),comp_avg_(null<avg_comp_method>),
							payment_lag_(err_int()),payment_lag_ex_mat_(null<logical>)
	{
		QL_REQUIRE(equal_casei(index_.type(), "IBOR") || equal_casei (index_.type(), "LIBOR"),"invalid index type");
	}
	
	frn_def_tmpl.frn_def_tmpl(	string				name,
							integer 				cpn_freq,						       
							frn_roll_method  		roll_method,   
							..day_count_method 		dc,
							..bd_convention 		bd,
							..calendar				cal,					
							logical 				eom,  
							string 					exc,
							string 					settle_code,						
							string 					curr,
							frn_neg_rate_rule	 	neg_rule,
							ir_index_tenor_tmpl		index,
							rfr_arrears_type  		arr_type,
							number option(nullable) 	arr_days,
							logical option(nullable) 	comp,
							//flt_sprd_comp_method option(nullable) spread_comp_method,
							//flt_avg_method option(nullable) 	avg_method,
							integer  							payment_lag,
							logical option(nullable) 			payment_lag_ex_mat)
							: instrument_def_tmpl(..instr_type.FRN), name_(name),cpn_freq_(cpn_freq), roll_method_(roll_method), dc_(dc),bd_(bd),
							eom_(eom),excpn_code_(exc),settle_code_(settle_code), cal_(cal), curr_(curr), 
							neg_rate_rule_(neg_rule), index_(index), arr_type_(arr_type), arr_days_(arr_days),
							payment_lag_(payment_lag),payment_lag_ex_mat_(payment_lag_ex_mat)
	{
		if(null(comp) || comp)
			comp_avg_ = avg_comp_method.COMP_SPRD_EXCL;
		else
			comp_avg_ = avg_comp_method.AVG_WEIGHTED;
		
		QL_REQUIRE(equal_casei(index_.type(), "RFR"),"invalid index type");		
		
		if(null(arr_days_)){
			if(arr_type_ ==	 rfr_arrears_type.PLAIN)
				arr_days_ = 0;
			else
				QL_FAIL("arr_days missing");
		}
		//if(null(comp_avg_)) {
		//	comp_avg_ = avg_comp_method.COMP_SPRD_EXCL;
		//}
		
		if(payment_lag_ == err_int() || payment_lag_ < 0) {
			payment_lag_ = 0;
		}
		if(null(payment_lag_ex_mat)) {
			payment_lag_ex_mat = (arr_type_ ==	 rfr_arrears_type.LOCKOUT);
		}
	}

	string				frn_def_tmpl.name() 				{ return name_;}
	day_count_method 	frn_def_tmpl.day_count_method()		{ return dc_;} 						
	bd_convention 		frn_def_tmpl.bd_convention()		{ return bd_;} 
	logical				frn_def_tmpl.eom()					{ return eom_;} 
	integer				frn_def_tmpl.cpn_freq()				{ return cpn_freq_;} 					
	string 				frn_def_tmpl.excpn_code()			{ return excpn_code_;} 
	frn_roll_method 	frn_def_tmpl.roll_method()			{ return roll_method_;}
	string 				frn_def_tmpl.settle_code()			{ return settle_code_;} 
	calendar			frn_def_tmpl.calendar()				{ return null(cal_) ? ..calendar() : cal_;}
	string				frn_def_tmpl.currency()				{ return curr_;}
	frn_neg_rate_rule	frn_def_tmpl.neg_rate_rule() 		{ return neg_rate_rule_;}			
	ir_index_tenor_tmpl	frn_def_tmpl.index()				{ return index_;}

	rfr_arrears_type 	frn_def_tmpl.arr_type()				{ return arr_type_;}
	number 				frn_def_tmpl.arr_days()				{ return arr_days_;}
	avg_comp_method 	frn_def_tmpl.comp_avg()				{ return comp_avg_;}
	//flt_sprd_comp_method frn_def_tmpl.spread_comp_method()	{ return spread_comp_method_;}
	//flt_avg_method 		frn_def_tmpl.avg_method()			{ return avg_method_;}
	integer				frn_def_tmpl.payment_lag()			{ return payment_lag_;}
	logical				frn_def_tmpl.payment_lag_ex_mat()	{ return payment_lag_ex_mat_;}
	logical				frn_def_tmpl.is_rfr_frn()			{ return equal_casei(index_.type(), "RFR");}
	
	ir_index frn_def_tmpl.ir_index()
	{
		string t = index_.type();
		
		if(equal_casei(t, "RFR") || equal_casei(t, "IBOR") || equal_casei (t, "LIBOR")) {			
			return create_ir_index(	index_.name(),
								t,
								index_.tenor(),
								index_.value_days(),
								index_.fixing_days(),
								index_.day_count_method(),
								index_.bd_convention(),
								index_.eom(),
								index_.cal(),//if libor: this is uk
								index_.cal_name(),
								index_.currency(),
								index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								index_.suppl_cal_name(),
								index_.spot_days(),
								index_.spread());
		}
		else {
			QL_FAIL("index type not applicable", E_INIT);
		}
	}
//INSTR_TMPL.bondfut_def_tmpl nasdaq_2y_swe = new INSTR_TMPL.bondfut_def_tmpl(	"NASDAQ_2Y", "2Y_NONE", 1, 6, DC_30E_360, false, BFDM_FWD_DEL_SYNT_CPDEL,mtm, nopf, "SWEDEN","SEK",	YIELD_PCT);

	//-------------------------------------------------------
	// class bondfut_def_tmpl
	//-------------------------------------------------------
	public class bondfut_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			bondfut_def_tmpl(string,string,number,integer,day_count_method, logical,
							bond_fut_style, bond_fut_pf option(nullable) ,
							..calendar, string, quote_style, number, 
							number, string);
			
			string				name();
			string				underlying_mat_code();
			number				coupon();
			integer				cpn_freq();
			day_count_method 	day_count_method(); 						
			logical				eom();
			bond_fut_style 		style();
			bond_fut_pf 		pf_method();
			calendar			calendar();			
			string				currency();					
			quote_style 		quote_style();
			number				contract_val();
			number				tick_size();
			string				fixing_code();

		protected:
			string				name_;
			string				underlying_mat_code_;
			number 				coupon_;
			integer				cpn_freq_;				
			day_count_method 	dc_;
			logical				eom_; 						
			bond_fut_style 		style_;
			bond_fut_pf 		pf_method_;
			..calendar			cal_;
			string				curr_;			
			quote_style 		qs_;			
			number 				contract_val_;
			number				tick_size_;
			string				fixing_code_;
	};

	bondfut_def_tmpl.bondfut_def_tmpl(	string					name,									
										string					underlying_mat_code,
										number					coupon,
										integer					cpn_freq,
										..day_count_method 		dc, 
										logical					eom,						
										bond_fut_style			style,
										bond_fut_pf option(nullable) 	pf_method,
										..calendar	 			cal,
										string					curr,									
										..quote_style			qs,
										number 					contract_val,
										number 					tick_size,
										string					fixing_code)
										: 	instrument_def_tmpl(..instr_type.BOND_FUTURE), name_(name),underlying_mat_code_(underlying_mat_code), coupon_(coupon), cpn_freq_(cpn_freq),  dc_(dc),
											eom_(eom),	 style_(style), pf_method_(pf_method), cal_(cal), curr_(curr),  
											qs_(qs), contract_val_(contract_val), tick_size_(tick_size), fixing_code_(fixing_code) {}

	string				bondfut_def_tmpl.name() 				{ return name_;}
	string				bondfut_def_tmpl.underlying_mat_code()	{ return underlying_mat_code_;}
	number				bondfut_def_tmpl.coupon()				{ return coupon_;}
	day_count_method 	bondfut_def_tmpl.day_count_method()		{ return dc_;} 								
	logical				bondfut_def_tmpl.eom()					{ return eom_;} 
	bond_fut_style 		bondfut_def_tmpl.style()				{ return style_;} 
	bond_fut_pf 		bondfut_def_tmpl.pf_method()			{ return pf_method_;}
	integer				bondfut_def_tmpl.cpn_freq()				{ return cpn_freq_;} 					
	calendar			bondfut_def_tmpl.calendar()				{ return null(cal_) ? ..calendar() : cal_;}
	string				bondfut_def_tmpl.currency()				{ return curr_;}	
	quote_style 		bondfut_def_tmpl.quote_style()			{ return qs_;}
	number				bondfut_def_tmpl.contract_val()			{ return contract_val_;}
	number				bondfut_def_tmpl.tick_size()			{ return tick_size_;}
	string				bondfut_def_tmpl.fixing_code()			{ return fixing_code_;}


	
	//-------------------------------------------------------
	// class il_bond_def_tmpl
	//-------------------------------------------------------
	public class il_bond_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			il_bond_def_tmpl(	string,il_bond_yld_method ,infl_index_method,integer,day_count_method, logical,
								string option(nullable),first_cpn_type,last_per_yld_method,next_per_yld_method, 
    							odd_last_cpn_type,string ,..calendar, string, quote_style);
			
			string					name();
			integer					cpn_freq();
			day_count_method 		day_count_method(); 						
			logical					eom();
			calendar				calendar();			
			string					currency();		
			string 					excpn_code();
			first_cpn_type 			first_cpn_type();
    		last_per_yld_method		last_yld_method();
    		next_per_yld_method		next_yld_method(); 
    		odd_last_cpn_type		odd_last_cpn_type();	
			il_bond_yld_method		yld_method();
			infl_index_method		index_method();
			string 					settle_code();
			quote_style 			quote_style();
			
			
		protected:
			string					name_;
			integer					cpn_freq_;
			il_bond_yld_method		yld_method_;
			infl_index_method		index_method_;	
			day_count_method 		dc_;
			logical					eom_; 						
			string 					excpn_code_; 
			first_cpn_type 			first_cpn_type_;
    		last_per_yld_method		last_yld_method_;
    		next_per_yld_method		next_yld_method_; 
    		odd_last_cpn_type		odd_last_cpn_type_;				 					
			string					settle_code_;
			..calendar				cal_;
			string					curr_;			
			quote_style				qs_;			
			
	};

	il_bond_def_tmpl.il_bond_def_tmpl(	string					name,									
										il_bond_yld_method		yld_method,
										infl_index_method		index_method,
										integer					cpn_freq,
										..day_count_method 		dc, 
										logical					eom,						
										string option(nullable)	 exc,
										..first_cpn_type 		first_cpn_type,
		    							last_per_yld_method		last_yld_method,
		    							next_per_yld_method		next_yld_method, 
		    							..odd_last_cpn_type		odd_last_cpn_type, 	 																	
										string 	 				settle_code,						
										..calendar	 			cal,
										string					curr,									
										..quote_style			qs)
										: 	instrument_def_tmpl(..instr_type.IL_REAL_FIXED_CPN_BOND), name_(name),cpn_freq_(cpn_freq), yld_method_(yld_method), index_method_(index_method), dc_(dc),
											eom_(eom),excpn_code_(exc),first_cpn_type_(first_cpn_type),
											last_yld_method_(last_yld_method),next_yld_method_(next_yld_method),odd_last_cpn_type_(odd_last_cpn_type), 
											settle_code_(settle_code), cal_(cal), curr_(curr),  qs_(qs) {}

	string					il_bond_def_tmpl.name() 				{ return name_;}
	day_count_method 		il_bond_def_tmpl.day_count_method()		{ return dc_;} 						
	il_bond_yld_method		il_bond_def_tmpl.yld_method()			{ return yld_method_;} 
	infl_index_method	 	il_bond_def_tmpl.index_method()			{ return index_method_;} 	 
	logical					il_bond_def_tmpl.eom()					{ return eom_;} 
	integer					il_bond_def_tmpl.cpn_freq()				{ return cpn_freq_;} 					
	string 					il_bond_def_tmpl.excpn_code()			{ return excpn_code_;} 
	first_cpn_type 			il_bond_def_tmpl.first_cpn_type()		{ return first_cpn_type_;} 
    last_per_yld_method		il_bond_def_tmpl.last_yld_method()		{ return last_yld_method_;} 
    next_per_yld_method		il_bond_def_tmpl.next_yld_method()		{ return next_yld_method_;}  
    odd_last_cpn_type		il_bond_def_tmpl.odd_last_cpn_type() 	{ return odd_last_cpn_type_;}  
	string 					il_bond_def_tmpl.settle_code()			{ return settle_code_;} 
	calendar				il_bond_def_tmpl.calendar()				{ return null(cal_) ? ..calendar() : cal_;}
	string					il_bond_def_tmpl.currency()				{ return curr_;}	
	quote_style 			il_bond_def_tmpl.quote_style()			{ return qs_;}


	

	//-------------------------------------------------------
	// class zero_def_tmpl
	//-------------------------------------------------------
	public class zero_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			zero_def_tmpl(	string,zero_yld_method ,integer,day_count_method, 
							string ,..calendar, string, quote_style);
			
			string					name();
			integer					quasi_cpn_freq();
			day_count_method 		day_count_method(); 						
			calendar				calendar();			
			string					currency();		
			zero_yld_method 		yld_method();
			string 					settle_code();
			quote_style 			quote_style();
			
			
		protected:
			string					name_;
			integer					quasi_cpn_freq_;
			zero_yld_method 		yld_method_;		
			day_count_method 		dc_;																 					
			string					settle_code_;
			..calendar				cal_;
			string					curr_;			
			quote_style				qs_;			
			
	};

	zero_def_tmpl.zero_def_tmpl(string					name,									
								zero_yld_method 		yld_method,										
								integer					quasi_cpn_freq,
								..day_count_method 		dc, 																											    									 																	
								string 	 				settle_code,						
								..calendar	 			cal,
								string					curr,									
								..quote_style			qs)
								: instrument_def_tmpl(..instr_type.ZERO_CPN_BOND), name_(name),quasi_cpn_freq_(quasi_cpn_freq), yld_method_(yld_method), dc_(dc),
								settle_code_(settle_code), cal_(cal), curr_(curr),  qs_(qs) {}

	string					zero_def_tmpl.name() 						{ return name_;}
	day_count_method 		zero_def_tmpl.day_count_method()			{ return dc_;} 						
	zero_yld_method	 		zero_def_tmpl.yld_method()					{ return yld_method_;} 	 
	integer					zero_def_tmpl.quasi_cpn_freq()				{ return quasi_cpn_freq_;} 					 
	string 					zero_def_tmpl.settle_code()					{ return settle_code_;} 
	calendar				zero_def_tmpl.calendar()					{ return null(cal_) ? ..calendar() : cal_;}
	string					zero_def_tmpl.currency()					{ return curr_;}	
	quote_style 			zero_def_tmpl.quote_style()					{ return qs_;}



	//-------------------------------------------------------
	// class swap_fixflt_def_tmpl
	//-------------------------------------------------------
	public class swap_fixflt_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_fixflt_def_tmpl(	string,string ,string option(nullable),string option(nullable),integer,day_count_method,
									interest_rule, string,..calendar, bd_convention,logical,integer ,
									number option(nullable),day_count_method,interest_rule,string,..calendar,
									bd_convention,logical,flt_comp_avg_type option(nullable),
    								flt_sprd_comp_method option(nullable) 	,
    								flt_avg_method option(nullable) ,notional_exchg_style option(nullable),
									flt_stub_fwd_style option(nullable),ir_index_tenor_tmpl);

			void 				set_fwd_start_code(string);
			void 				set_maturity_code(string);
			
			string				name();
			string 				spot_settle_code();	
			string 				fwd_start_code();
			string 				maturity_code();
			notional_exchg_style not_exchange();
			logical				primary_quote_is_leg1();
			day_count_method 	fix_day_count_method();
			interest_rule		fix_ir_rule();
			bd_convention		fix_bd_convention();						
			logical				fix_eom();			
			integer				fix_cpn_freq();
			calendar			fix_calendar();
			string				fix_currency();	
			
			integer				flt_cpn_freq();
			integer				flt_reset_freq();			
			day_count_method 	flt_day_count_method();
			interest_rule		flt_ir_rule();
			calendar			flt_calendar();							
			string				flt_currency();
			bd_convention		flt_bd_convention();						
			logical				flt_eom();									
			flt_comp_avg_type 	flt_comp_avg();
    		flt_sprd_comp_method flt_spread_comp_method();
    		flt_avg_method 		flt_avg_method();							
			ir_index_tenor_tmpl	index()	;			
			ir_index			flt_ir_index();
			logical				is_cross_curr();
			flt_stub_fwd_style	flt_fwd_stub_rate_style();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
		protected:
			string				name_;
			integer				fix_cpn_freq_;
			day_count_method 	fix_dc_;
			interest_rule		fix_ir_rule_;
			string				fix_curr_;
			..calendar			fix_cal_;
			bd_convention		fix_bd_;
			logical				fix_eom_; 
			integer				flt_cpn_freq_;
			number 				flt_reset_freq_;
			day_count_method 	flt_dc_;
			interest_rule		flt_ir_rule_;
			string				flt_curr_;
			..calendar			flt_cal_;
			bd_convention		flt_bd_;
			logical				flt_eom_; 
			flt_comp_avg_type 	flt_comp_avg_;
    		flt_sprd_comp_method flt_spread_comp_method_;
    		flt_avg_method 		flt_avg_method_;												
			string				spot_code_;
			string				fwd_start_code_;				
			string				maturity_code_;
			ir_index_tenor_tmpl	index_;
			notional_exchg_style not_exchange_;
			flt_stub_fwd_style	flt_fwd_stub_rate_style_;
	};

	swap_fixflt_def_tmpl.swap_fixflt_def_tmpl(	string					name,
												string 					spot_settle_code,
												string 	option(nullable) fwd_start_code,
												string 	option(nullable) maturity_code,												
												integer 				fix_cpn_freq,	
												..day_count_method 		fix_dc,
												interest_rule			fix_ir_rule,
												string 					fix_curr,
												..calendar				fix_cal,
												..bd_convention 		fix_bd,												
												logical 				fix_eom, 						       
												integer 				flt_cpn_freq,
												number option(nullable) flt_reset_freq,	  
												..day_count_method 		flt_dc,
												interest_rule			flt_ir_rule,
												string 					flt_curr,
												..calendar				flt_cal,	
												..bd_convention 		flt_bd,												
												logical 				flt_eom,
												..flt_comp_avg_type option(nullable)	flt_comp_avg,
	    										..flt_sprd_comp_method option(nullable) flt_spread_comp_method,
	    										..flt_avg_method option(nullable) 		flt_avg_method, 
												notional_exchg_style option(nullable)	not_exchange,
												flt_stub_fwd_style	option(nullable)	flt_fwd_stub_rate_style,
												ir_index_tenor_tmpl		index)
								: 	instrument_def_tmpl(..instr_type.SWAP_FIXFLT), name_(name),fix_cpn_freq_(fix_cpn_freq),
									fix_dc_(fix_dc),fix_ir_rule_(fix_ir_rule),fix_curr_(fix_curr),fix_cal_(fix_cal),
									fix_bd_(fix_bd),fix_eom_(fix_eom),flt_cpn_freq_(flt_cpn_freq),flt_reset_freq_(flt_reset_freq),
									flt_dc_(flt_dc),flt_ir_rule_(flt_ir_rule),flt_curr_(flt_curr),flt_cal_(flt_cal), 
									flt_bd_(flt_bd),flt_eom_(flt_eom),flt_comp_avg_(flt_comp_avg),flt_spread_comp_method_(flt_spread_comp_method),
									flt_avg_method_(flt_avg_method),spot_code_(spot_settle_code),fwd_start_code_(fwd_start_code),maturity_code_(maturity_code), 
									index_(index), not_exchange_(not_exchange),flt_fwd_stub_rate_style_(flt_fwd_stub_rate_style) {}

	void 				swap_fixflt_def_tmpl.set_fwd_start_code(string c)	{ fwd_start_code_ = c;} 
	void 				swap_fixflt_def_tmpl.set_maturity_code(string c)	{ maturity_code_ = c;} 
	string				swap_fixflt_def_tmpl.name() 				{ return name_;}
	day_count_method 	swap_fixflt_def_tmpl.fix_day_count_method()	{ return fix_dc_;} 						
	day_count_method 	swap_fixflt_def_tmpl.flt_day_count_method()	{ return flt_dc_;}
	interest_rule		swap_fixflt_def_tmpl.fix_ir_rule()			{ return fix_ir_rule_;}
	interest_rule		swap_fixflt_def_tmpl.flt_ir_rule()			{ return flt_ir_rule_;}
	integer				swap_fixflt_def_tmpl.fix_cpn_freq()			{ return fix_cpn_freq_;} 					
	integer				swap_fixflt_def_tmpl.flt_cpn_freq()			{ return flt_cpn_freq_;} 
	integer 			swap_fixflt_def_tmpl.flt_reset_freq()		{ return null(flt_reset_freq_) ? flt_cpn_freq_ : integer(flt_reset_freq_);} 
	string				swap_fixflt_def_tmpl.fix_currency()			{ return fix_curr_;}
	string				swap_fixflt_def_tmpl.flt_currency()			{ return flt_curr_;}
	calendar			swap_fixflt_def_tmpl.fix_calendar()			{ return null(fix_cal_) ? ..calendar() : fix_cal_;}
	calendar			swap_fixflt_def_tmpl.flt_calendar()			{ return null(flt_cal_) ? ..calendar() : flt_cal_;}
	bd_convention 		swap_fixflt_def_tmpl.flt_bd_convention()	{ return flt_bd_;} 
	logical				swap_fixflt_def_tmpl.flt_eom()				{ return flt_eom_;} 
	bd_convention 		swap_fixflt_def_tmpl.fix_bd_convention()	{ return fix_bd_;} 
	logical				swap_fixflt_def_tmpl.fix_eom()				{ return fix_eom_;} 
	string 				swap_fixflt_def_tmpl.spot_settle_code()		{ return spot_code_;}
	string 				swap_fixflt_def_tmpl.fwd_start_code()		{ return fwd_start_code_;}  
	string 				swap_fixflt_def_tmpl.maturity_code()		{ return maturity_code_;} 
	flt_comp_avg_type 	swap_fixflt_def_tmpl.flt_comp_avg()			{ return flt_comp_avg_;} 
    flt_sprd_comp_method swap_fixflt_def_tmpl.flt_spread_comp_method() { return flt_spread_comp_method_;} 
    flt_avg_method 		swap_fixflt_def_tmpl.flt_avg_method()		{ return flt_avg_method_;} 					
	ir_index_tenor_tmpl	swap_fixflt_def_tmpl.index()				{ return index_;}
	logical				swap_fixflt_def_tmpl.is_cross_curr()		{ return !equal_casei(fix_curr_, flt_curr_);}
	logical 			swap_fixflt_def_tmpl.primary_quote_is_leg1() { return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),fix_curr_,flt_curr_,fix_cpn_freq_,flt_reset_freq());}
	flt_stub_fwd_style	swap_fixflt_def_tmpl.flt_fwd_stub_rate_style() { return flt_fwd_stub_rate_style_;}
	mtm_code 			swap_fixflt_def_tmpl.mtm() 						{ return mtm_code.NONE;}
	logical 			swap_fixflt_def_tmpl.is_mtm(out logical leg1)	{ return false; /*..is_mtm(is_cross_curr(), mtm_, leg1);*/}
	
	notional_exchg_style swap_fixflt_def_tmpl.not_exchange()	
	{ 
		if(null(not_exchange_)) {
			if( !is_cross_curr()) 
				return NE_NONE ;
			else 
				return NE_BOTH;
		}
		else
			return not_exchange_;
		
	}
	
	ir_index swap_fixflt_def_tmpl.flt_ir_index()
	{
		string t = index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	index_.name(),
								t,
								index_.tenor(),
								index_.value_days(),
								index_.fixing_days(),
								index_.day_count_method(),
								index_.bd_convention(),
								index_.eom(),
								index_.cal(),//if libor: this is uk
								index_.cal_name(),
								index_.currency(),
								index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								index_.suppl_cal_name(),
								index_.spot_days(),0);
	}

	//-------------------------------------------------------
	// class swap_fltflt_def_tmpl
	//-------------------------------------------------------
	public class swap_fltflt_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_fltflt_def_tmpl(	string					name,
									string 					spot_settle_code,
									string 	option(nullable) fwd_start_code,
									string 	option(nullable) maturity_code,																		
									notional_exchg_style option(nullable)	not_exchange,
									mtm_code option(nullable) 	mtm,
									integer 				flt1_cpn_freq,
									number option(nullable) flt1_reset_freq,	 	
									..day_count_method 		flt1_dc,
									interest_rule			flt1_ir_rule,
									string 					flt1_curr,
									..calendar				flt1_cal,
									..bd_convention 		flt1_bd,												
									logical 				flt1_eom,
									..flt_comp_avg_type option(nullable)	flt1_comp_avg,
									..flt_sprd_comp_method option(nullable) flt1_spread_comp_method,
									..flt_avg_method option(nullable) 		flt1_avg_method,
									flt_stub_fwd_style option(nullable) 		flt1_stub_fwd_style,
									ir_index_tenor_tmpl		flt1_index,	
									integer 				flt2_cpn_freq,
									number option(nullable) flt2_reset_freq,	  
									..day_count_method 		flt2_dc,
									interest_rule			flt2_ir_rule,
									string 					flt2_curr,
									..calendar				flt2_cal,	
									..bd_convention 		flt2_bd,												
									logical 				flt2_eom,
									..flt_comp_avg_type option(nullable)	flt2_comp_avg,
									..flt_sprd_comp_method option(nullable) flt2_spread_comp_method,
									..flt_avg_method option(nullable) 		flt2_avg_method,
									flt_stub_fwd_style option(nullable) 	flt2_stub_fwd_style,												
									ir_index_tenor_tmpl		flt2_index);
			
			string				name();
			string 				spot_settle_code();
			string 				fwd_start_code();
			string 				maturity_code()	;			
			logical				is_cross_curr();
			notional_exchg_style not_exchange();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
			logical 			primary_quote_is_leg1();
			
			integer				flt1_cpn_freq();
			integer 			flt1_reset_freq();
			day_count_method 	flt1_day_count_method();
			interest_rule		flt1_ir_rule();
			calendar			flt1_calendar();
			string				flt1_currency();
			bd_convention		flt1_bd_convention();						
			logical				flt1_eom();
			flt_comp_avg_type 	flt1_comp_avg();
    		flt_sprd_comp_method flt1_spread_comp_method();
    		flt_avg_method 		flt1_avg_method();
			ir_index_tenor_tmpl	flt1_index()	;
			ir_index			flt1_ir_index();
			flt_stub_fwd_style	flt1_fwd_stub_rate_style();
			
			integer				flt2_cpn_freq();			
			integer 			flt2_reset_freq();			 
			day_count_method 	flt2_day_count_method();
			interest_rule		flt2_ir_rule();
			calendar			flt2_calendar();								
			string				flt2_currency();
			bd_convention		flt2_bd_convention();						
			logical				flt2_eom();	
			flt_comp_avg_type 	flt2_comp_avg();
    		flt_sprd_comp_method flt2_spread_comp_method();
    		flt_avg_method 		flt2_avg_method();							
			ir_index_tenor_tmpl	flt2_index()	;
			ir_index			flt2_ir_index();
			flt_stub_fwd_style	flt2_fwd_stub_rate_style();
			
		protected:
			
			string				name_;
			integer				flt1_cpn_freq_;
			number 				flt1_reset_freq_;
			day_count_method 	flt1_dc_;
			interest_rule		flt1_ir_rule_;
			string				flt1_curr_;
			..calendar			flt1_cal_;
			bd_convention		flt1_bd_;
			logical				flt1_eom_; 
			flt_comp_avg_type 	flt1_comp_avg_;
    		flt_sprd_comp_method flt1_spread_comp_method_;
    		flt_avg_method 		flt1_avg_method_;
			flt_stub_fwd_style	flt1_fwd_stub_rate_style_;

			integer				flt2_cpn_freq_;
			number 				flt2_reset_freq_;
			day_count_method 	flt2_dc_;
			interest_rule		flt2_ir_rule_;
			string				flt2_curr_;
			..calendar			flt2_cal_;
			bd_convention		flt2_bd_;
			logical				flt2_eom_;
			flt_comp_avg_type 	flt2_comp_avg_;
    		flt_sprd_comp_method flt2_spread_comp_method_;
    		flt_avg_method 		flt2_avg_method_;
			flt_stub_fwd_style	flt2_fwd_stub_rate_style_;
			string				spot_code_;
			string				fwd_start_code_;
			string 				maturity_code_; 				
			ir_index_tenor_tmpl	flt1_index_;
			ir_index_tenor_tmpl	flt2_index_;
			notional_exchg_style not_exchange_;
			mtm_code			mtm_;
	};

	swap_fltflt_def_tmpl.swap_fltflt_def_tmpl(	string					name,
												string 					spot_settle_code,
												string 	option(nullable) fwd_start_code,
												string 	option(nullable) maturity_code,																		
												notional_exchg_style option(nullable)	not_exchange,
												mtm_code option(nullable) 	mtm,
												integer 				flt1_cpn_freq,
												number option(nullable) flt1_reset_freq,	 	
												..day_count_method 		flt1_dc,
												interest_rule			flt1_ir_rule,
												string 					flt1_curr,
												..calendar				flt1_cal,
												..bd_convention 		flt1_bd,												
												logical 				flt1_eom,
												..flt_comp_avg_type option(nullable)	flt1_comp_avg,
    											..flt_sprd_comp_method option(nullable) flt1_spread_comp_method,
    											..flt_avg_method option(nullable) 		flt1_avg_method,
												flt_stub_fwd_style option(nullable) 	flt1_stub_fwd_style,
												ir_index_tenor_tmpl		flt1_index,	
												integer 				flt2_cpn_freq,
												number option(nullable) flt2_reset_freq,	  
												..day_count_method 		flt2_dc,
												interest_rule			flt2_ir_rule,
												string 					flt2_curr,
												..calendar				flt2_cal,	
												..bd_convention 		flt2_bd,												
												logical 				flt2_eom,
												..flt_comp_avg_type option(nullable)	flt2_comp_avg,
    											..flt_sprd_comp_method option(nullable) flt2_spread_comp_method,
    											..flt_avg_method option(nullable) 		flt2_avg_method,
												flt_stub_fwd_style option(nullable) 	flt2_stub_fwd_style,												
												ir_index_tenor_tmpl		flt2_index)
								: 	instrument_def_tmpl(..instr_type.SWAP_FLTFLT), name_(name),flt1_cpn_freq_(flt1_cpn_freq), flt1_reset_freq_(flt1_reset_freq),
									flt1_dc_(flt1_dc),flt1_ir_rule_(flt1_ir_rule),
									flt1_curr_(flt1_curr),flt1_cal_(flt1_cal),  
									flt1_bd_(flt1_bd),flt1_eom_(flt1_eom),
									flt1_comp_avg_(flt1_comp_avg),flt1_spread_comp_method_(flt1_spread_comp_method),flt1_avg_method_(flt1_avg_method),
									flt1_fwd_stub_rate_style_(flt1_stub_fwd_style),
									flt2_cpn_freq_(flt2_cpn_freq),flt2_reset_freq_(flt2_reset_freq),
									flt2_dc_(flt2_dc),flt2_ir_rule_(flt2_ir_rule),flt2_curr_(flt2_curr),flt2_cal_(flt2_cal), 
									flt2_bd_(flt2_bd),flt2_eom_(flt2_eom),
									flt2_comp_avg_(flt2_comp_avg),flt2_spread_comp_method_(flt2_spread_comp_method),flt2_avg_method_(flt2_avg_method),
									flt2_fwd_stub_rate_style_(flt2_stub_fwd_style),
									spot_code_(spot_settle_code), fwd_start_code_(fwd_start_code),maturity_code_(maturity_code),
									flt1_index_(flt1_index),flt2_index_(flt2_index),not_exchange_(not_exchange), mtm_(mtm)
	{
		//check();
	}

	string				swap_fltflt_def_tmpl.name() 					{ return name_;}
	day_count_method 	swap_fltflt_def_tmpl.flt1_day_count_method()	{ return flt1_dc_;} 						
	day_count_method 	swap_fltflt_def_tmpl.flt2_day_count_method()	{ return flt2_dc_;}
	interest_rule		swap_fltflt_def_tmpl.flt1_ir_rule()				{ return flt1_ir_rule_;}
	interest_rule		swap_fltflt_def_tmpl.flt2_ir_rule()				{ return flt2_ir_rule_;}
	integer				swap_fltflt_def_tmpl.flt1_cpn_freq()			{ return flt1_cpn_freq_;} 					
	integer				swap_fltflt_def_tmpl.flt2_cpn_freq()			{ return flt2_cpn_freq_;} 
	
	string				swap_fltflt_def_tmpl.flt1_currency()			{ return flt1_curr_;}
	string				swap_fltflt_def_tmpl.flt2_currency()			{ return flt2_curr_;}
	calendar			swap_fltflt_def_tmpl.flt1_calendar()			{ return null(flt1_cal_) ? ..calendar() : flt1_cal_;}
	calendar			swap_fltflt_def_tmpl.flt2_calendar()			{ return null(flt2_cal_) ? ..calendar() : flt2_cal_;}
	bd_convention 		swap_fltflt_def_tmpl.flt2_bd_convention()		{ return flt2_bd_;} 
	logical				swap_fltflt_def_tmpl.flt2_eom()					{ return flt2_eom_;} 
	bd_convention 		swap_fltflt_def_tmpl.flt1_bd_convention()		{ return flt1_bd_;} 
	logical				swap_fltflt_def_tmpl.flt1_eom()					{ return flt1_eom_;} 
	string 				swap_fltflt_def_tmpl.spot_settle_code()			{ return spot_code_;} 
	string 				swap_fltflt_def_tmpl.fwd_start_code()			{ return fwd_start_code_;} 
	string 				swap_fltflt_def_tmpl.maturity_code()			{ return maturity_code_;} 

	integer 			swap_fltflt_def_tmpl.flt1_reset_freq()			{ return null(flt1_reset_freq_) ? flt1_cpn_freq_ : integer(flt1_reset_freq_);} 
	integer 			swap_fltflt_def_tmpl.flt2_reset_freq()			{ return null(flt2_reset_freq_) ? flt2_cpn_freq_ : integer(flt2_reset_freq_);} 
	flt_comp_avg_type 	swap_fltflt_def_tmpl.flt1_comp_avg()			{ return flt1_comp_avg_;} 
    flt_sprd_comp_method swap_fltflt_def_tmpl.flt1_spread_comp_method() { return flt1_spread_comp_method_;} 
    flt_avg_method 		swap_fltflt_def_tmpl.flt1_avg_method()			{ return flt1_avg_method_;} 
	flt_comp_avg_type 	swap_fltflt_def_tmpl.flt2_comp_avg()			{ return flt2_comp_avg_;} 
    flt_sprd_comp_method swap_fltflt_def_tmpl.flt2_spread_comp_method() { return flt2_spread_comp_method_;} 
    flt_avg_method 		swap_fltflt_def_tmpl.flt2_avg_method()			{ return flt2_avg_method_;} 

	ir_index_tenor_tmpl	swap_fltflt_def_tmpl.flt1_index()				{ return flt1_index_;}					
	ir_index_tenor_tmpl	swap_fltflt_def_tmpl.flt2_index()				{ return flt2_index_;}

	flt_stub_fwd_style	swap_fltflt_def_tmpl.flt1_fwd_stub_rate_style()	{ return flt1_fwd_stub_rate_style_;}	
	flt_stub_fwd_style	swap_fltflt_def_tmpl.flt2_fwd_stub_rate_style()	{ return flt2_fwd_stub_rate_style_;}
	
	logical				swap_fltflt_def_tmpl.is_cross_curr()			{ return !equal_casei(flt1_curr_, flt2_curr_);}	
	logical  			swap_fltflt_def_tmpl.primary_quote_is_leg1()	{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),flt1_curr_,flt1_curr_,flt1_reset_freq(),flt2_reset_freq());}	

	/*void 				swap_fltflt_def_tmpl.check()
	{
		if(is_cross_curr()){
		}
		else {
			QL_REQUIRE(flt1_reset_freq() > flt2_reset_freq(),"leg1 should have higher leg reset_freq (spread assumed to be added on leg1)");
		}
	}*/
	
	notional_exchg_style swap_fltflt_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) {
			if( equal_casei(flt1_curr_, flt2_curr_)) 
				return NE_NONE ;
			else 
				return NE_BOTH;
		}
		else
			return not_exchange_;
		
	}

	mtm_code swap_fltflt_def_tmpl.mtm() { return mtm_;}
	logical swap_fltflt_def_tmpl.is_mtm(out logical leg1)	{ return ..is_mtm(is_cross_curr(), mtm_, leg1);}	
	
	ir_index swap_fltflt_def_tmpl.flt1_ir_index()
	{
		string t = flt1_index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	flt1_index_.name(),
								t,
								flt1_index_.tenor(),
								flt1_index_.value_days(),
								flt1_index_.fixing_days(),
								flt1_index_.day_count_method(),
								flt1_index_.bd_convention(),
								flt1_index_.eom(),
								flt1_index_.cal(),//if libor: this is uk
								flt1_index_.cal_name(),
								flt1_index_.currency(),
								flt1_index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								flt1_index_.suppl_cal_name(),
								flt1_index_.spot_days(),0);
	}
	ir_index swap_fltflt_def_tmpl.flt2_ir_index()
	{
		string t = flt2_index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	flt2_index_.name(),
								t,
								flt2_index_.tenor(),
								flt2_index_.value_days(),
								flt2_index_.fixing_days(),
								flt2_index_.day_count_method(),
								flt2_index_.bd_convention(),
								flt2_index_.eom(),
								flt2_index_.cal(),//if libor: this is uk
								flt2_index_.cal_name(),
								flt2_index_.currency(),
								flt2_index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								flt2_index_.suppl_cal_name(),
								flt2_index_.spot_days(),0);
	}

	//-------------------------------------------------------
	// class swap_fixois_def_tmpl  -single currency
	//-------------------------------------------------------
	public class swap_fixois_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_fixois_def_tmpl(	string,string ,string option(nullable),string option(nullable),integer,day_count_method,
									..calendar, bd_convention,logical,string,interest_rule,string,
									rfr_arrears_type option(nullable),number option(nullable) ,flt_comp_avg_type option(nullable),
									flt_sprd_comp_method option(nullable),flt_avg_method option(nullable),logical,logical,
									integer,integer,ir_index_tenor_tmpl);

			
			void 				set_maturity_code(string);
			
			string				name();
			string 				spot_settle_code();
			string 				fwd_start_code();
			
			integer				payment_delay();
			integer				payment_delay_prin();					
			calendar			calendar();			
			bd_convention		bd_convention();						
			logical				eom();					
			string 				maturity_code()	;
			logical 			primary_quote_is_leg1();
			logical				is_cross_curr();
			notional_exchg_style not_exchange();

			//fix
			day_count_method 	fix_day_count_method();	
			integer				fix_cpn_freq();
			string				fix_currency();
			interest_rule		fix_ir_rule();
			day_count_method 	fix_day_count_method();	
			
			//ois
			day_count_method 	ois_day_count_method();	
			integer				ois_cpn_freq();
			string				ois_currency();
			ir_index_tenor_tmpl	on_index()	;
			ir_index			ois_ir_index();
			flt_comp_avg_type 	ois_comp_avg();
    		flt_sprd_comp_method ois_spread_comp_method();
    		flt_avg_method 		ois_avg_method();
			rfr_arrears_type 	ois_arr_type();
			number 				ois_arr_days();
			logical				ois_endog_df();					
			logical				ois_approx_calc();
			interest_rule		ois_ir_rule();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
		protected:
			string				name_;
			string				spot_code_;
			string				fwd_start_code_;
			string 				maturity_code_;
			integer				payment_lag_;
			integer				payment_lag_prin_;
			
			integer				cpn_freq_;
			day_count_method 	dc_;					
			..calendar			cal_;
			bd_convention		bd_;
			logical				eom_;
			
			string				fix_curr_;
			interest_rule		fix_ir_rule_;

			string 				ois_curr_;
			rfr_arrears_type 	ois_arr_type_;
			number 				ois_arr_days_;
			logical				ois_endog_df_;
			logical				ois_approx_calc_;
			flt_comp_avg_type 	ois_comp_avg_;
    		flt_sprd_comp_method ois_spread_comp_method_;
    		flt_avg_method 		ois_avg_method_;							
			ir_index_tenor_tmpl	on_index_;
			notional_exchg_style not_exchange_;
	};

	swap_fixois_def_tmpl.swap_fixois_def_tmpl(	string						name,
												string 						spot_settle_code,
												string 	option(nullable) 	fwd_start_code,
												string 	option(nullable) 	maturity_code,
												integer 					cpn_freq,	
												..day_count_method 			dc,																							
												..calendar					cal,
												..bd_convention 			bd,												
												logical 					eom,
												string 						fix_curr,
												interest_rule				fix_ir_rule,

												string 									ois_curr,
												rfr_arrears_type option(nullable) 		ois_arr_type,
												number option(nullable)					ois_arr_days,
												flt_comp_avg_type option(nullable)		ois_comp_avg,
				    							flt_sprd_comp_method option(nullable)	ois_spread_comp_method,
				    							flt_avg_method option(nullable)			ois_avg_method,
												logical 								ois_endog_df,
												logical									ois_approx_calc,												
												integer 								payment_delay,
												integer 								payment_delay_prin,
												ir_index_tenor_tmpl						on_index)
												: 	instrument_def_tmpl(..instr_type.SWAP_FIXOIS),name_(name),spot_code_(spot_settle_code),fwd_start_code_(fwd_start_code),
													maturity_code_(maturity_code),
													payment_lag_(payment_delay),payment_lag_prin_(payment_delay_prin),													
													cpn_freq_(cpn_freq), dc_(dc), cal_(cal),
													bd_(bd),eom_(eom),	fix_curr_(fix_curr), fix_ir_rule_(fix_ir_rule),	ois_curr_(ois_curr),											
													ois_arr_type_(ois_arr_type), ois_arr_days_(ois_arr_days),
													ois_endog_df_(ois_endog_df),ois_approx_calc_(ois_approx_calc),
													ois_comp_avg_(ois_comp_avg),
													ois_spread_comp_method_(ois_spread_comp_method),ois_avg_method_(ois_avg_method),
													on_index_(on_index), not_exchange_(null<notional_exchg_style>) {}

	
	void 				swap_fixois_def_tmpl.set_maturity_code(string c)	{ maturity_code_ = c;} 
	string				swap_fixois_def_tmpl.name() 					{ return name_;}
	//day_count_method 	swap_fixois_def_tmpl.day_count_method()			{ return dc_;}
	
	interest_rule		swap_fixois_def_tmpl.fix_ir_rule()				{ return fix_ir_rule_;}
	//integer				swap_fixois_def_tmpl.cpn_freq()					{ return cpn_freq_;} 					
	integer				swap_fixois_def_tmpl.payment_delay()			{ return payment_lag_;}
	integer				swap_fixois_def_tmpl.payment_delay_prin()		{ return payment_lag_prin_;}

	//fix
	day_count_method 	swap_fixois_def_tmpl.fix_day_count_method()		{ return dc_;}
	integer				swap_fixois_def_tmpl.fix_cpn_freq()				{ return cpn_freq_;} 
	//ois
	day_count_method 	swap_fixois_def_tmpl.ois_day_count_method()		{ return dc_;}
	integer				swap_fixois_def_tmpl.ois_cpn_freq()				{ return cpn_freq_;}
	
	flt_comp_avg_type 	swap_fixois_def_tmpl.ois_comp_avg()				{ return ois_comp_avg_;}
    flt_sprd_comp_method swap_fixois_def_tmpl.ois_spread_comp_method()	{ return ois_spread_comp_method_;}
    flt_avg_method 		swap_fixois_def_tmpl.ois_avg_method()			{ return ois_avg_method_;}
	rfr_arrears_type 	swap_fixois_def_tmpl.ois_arr_type()				{ return ois_arr_type_;}
	number 				swap_fixois_def_tmpl.ois_arr_days()				{ return ois_arr_days_;}
	logical				swap_fixois_def_tmpl.ois_endog_df()				{ return ois_endog_df_;}
	string				swap_fixois_def_tmpl.fix_currency()				{ return fix_curr_;}
	string				swap_fixois_def_tmpl.ois_currency()				{ return ois_curr_;}
	calendar			swap_fixois_def_tmpl.calendar()					{ return null(cal_) ? ..calendar() : cal_;}
	bd_convention 		swap_fixois_def_tmpl.bd_convention()			{ return bd_;} 
	logical				swap_fixois_def_tmpl.eom()						{ return eom_;} 
	string 				swap_fixois_def_tmpl.spot_settle_code()			{ return spot_code_;} 
	string 				swap_fixois_def_tmpl.fwd_start_code()			{ return fwd_start_code_;} 
	string 				swap_fixois_def_tmpl.maturity_code()			{ return maturity_code_;}  					
	ir_index_tenor_tmpl	swap_fixois_def_tmpl.on_index()					{ return on_index_;}
	logical				swap_fixois_def_tmpl.is_cross_curr()			{ return !equal_casei(fix_curr_, ois_curr_);}
	logical				swap_fixois_def_tmpl.ois_approx_calc()			{ return ois_approx_calc_;}
	interest_rule		swap_fixois_def_tmpl.ois_ir_rule()				{ return interest_rule.IR_ADJ;}
	logical  			swap_fixois_def_tmpl.primary_quote_is_leg1()	{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),fix_curr_,ois_curr_,cpn_freq_,365);}
	mtm_code 			swap_fixois_def_tmpl.mtm() 						{ return mtm_code.NONE;}
	logical 			swap_fixois_def_tmpl.is_mtm(out logical leg1)	{ return false; /*..is_mtm(is_cross_curr(), mtm_, leg1);*/}
	
	notional_exchg_style swap_fixois_def_tmpl.not_exchange()					
	{ 
		if(null(not_exchange_)) {
			if( !is_cross_curr()) 
				return NE_NONE ;
			else 
				return NE_BOTH;
		}
		else
			return not_exchange_;
		
	}
	
	ir_index swap_fixois_def_tmpl.ois_ir_index()
	{
		string t = on_index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR") && !equal_casei (t, "RFR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	on_index_.name(),t,
								on_index_.tenor(),
								on_index_.value_days(),
								on_index_.fixing_days(),
								on_index_.day_count_method(),
								on_index_.bd_convention(),
								on_index_.eom(),
								on_index_.cal(),//if libor: this is uk
								on_index_.cal_name(),
								on_index_.currency(),
								on_index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								on_index_.suppl_cal_name(),
								on_index_.spot_days(),
								on_index_.spread());
	}
	
	//-------------------------------------------------------
	// class swap_fltflt2s_def_tmpl
	//-------------------------------------------------------
	public class swap_fltflt2s_def_tmpl : public  instrument_def_tmpl
	{
		public:
			swap_fltflt2s_def_tmpl(	string name,
										string 	option(nullable) fwd_start_code,
										string 	option(nullable) maturity_code,
										swap_fixflt_def_tmpl fixflt_sprd,swap_fixflt_def_tmpl fixflt_flat);
			string					name();
			swap_fixflt_def_tmpl 	fixflt_sprd();
			swap_fixflt_def_tmpl 	fixflt_flat();
			string 					spot_settle_code();
			string 					fwd_start_code()	;
			string 					maturity_code();
			notional_exchg_style 	not_exchange();
			logical					is_cross_curr()	;
			logical 				primary_quote_is_leg1() ;
			string 					currency() ;
			calendar				calendar();

			integer					fix_cpn_freq();
			day_count_method 		fix_day_count_method();
			interest_rule			fix_ir_rule();
			//string					fix_currency();	
			bd_convention			fix_bd_convention();						
			logical					fix_eom();	
			//calendar				fix_calendar();

			integer				flt_sprd_cpn_freq();
			number 				flt_sprd_reset_freq();
			day_count_method 	flt_sprd_day_count_method();
			interest_rule		flt_sprd_ir_rule();
			//calendar			flt_sprd_calendar();
			//string				flt_sprd_currency();
			bd_convention		flt_sprd_bd_convention();						
			logical				flt_sprd_eom();
			flt_comp_avg_type 	flt_sprd_comp_avg()	 ;
		    flt_sprd_comp_method flt_sprd_spread_comp_method() ;
		    flt_avg_method 		flt_sprd_avg_method();
			ir_index_tenor_tmpl	flt_sprd_index()	;
			ir_index			flt_sprd_ir_index();
			flt_stub_fwd_style	flt_sprd_fwd_stub_rate_style();
			
			integer				flt_flat_cpn_freq();
			number 				flt_flat_reset_freq(); 
			day_count_method 	flt_flat_day_count_method();
			interest_rule		flt_flat_ir_rule();
			//calendar			flt_flat_calendar();											
			//string				flt_flat_currency();
			bd_convention		flt_flat_bd_convention();						
			logical				flt_flat_eom();										
			flt_comp_avg_type 	flt_flat_comp_avg()	;
		    flt_sprd_comp_method flt_flat_spread_comp_method() ;
		    flt_avg_method 		flt_flat_avg_method();				
			ir_index_tenor_tmpl	flt_flat_index()	;						
			ir_index			flt_flat_ir_index();						
			flt_stub_fwd_style	flt_flat_fwd_stub_rate_style();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
			void 				check();
			
		protected:
			string					name_;
			swap_fixflt_def_tmpl 	fixflt_sprd_;
			swap_fixflt_def_tmpl 	fixflt_flat_;
			string					fwd_start_code_;
			string					maturity_code_;
			notional_exchg_style	not_exchange_;

		
	};
	swap_fltflt2s_def_tmpl.swap_fltflt2s_def_tmpl(	string name,
													string 	option(nullable) fwd_start_code,
													string 	option(nullable) maturity_code,
													swap_fixflt_def_tmpl fixflt_sprd,
													swap_fixflt_def_tmpl fixflt_flat):
													instrument_def_tmpl(..instr_type.SWAP_FLTFLT2S), name_(name),
													fixflt_sprd_(fixflt_sprd), fixflt_flat_(fixflt_flat),
													fwd_start_code_(fwd_start_code),maturity_code_(maturity_code),
													not_exchange_(notional_exchg_style.NE_NONE)
	{
		check();
	}

	
	
	string					swap_fltflt2s_def_tmpl.name() 				{ return name_;}
	swap_fixflt_def_tmpl 	swap_fltflt2s_def_tmpl.fixflt_sprd()		{ return fixflt_sprd_;} 						
	swap_fixflt_def_tmpl 	swap_fltflt2s_def_tmpl.fixflt_flat()		{ return fixflt_flat_;}
	string 					swap_fltflt2s_def_tmpl.spot_settle_code() 	{ return fixflt_sprd_.spot_settle_code();}
	string 					swap_fltflt2s_def_tmpl.fwd_start_code()		{ return fwd_start_code_;}
	string 					swap_fltflt2s_def_tmpl.maturity_code()		{ return maturity_code_;}
	logical  				swap_fltflt2s_def_tmpl.primary_quote_is_leg1()	{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),currency(),currency(),fixflt_sprd_.flt_reset_freq(),fixflt_flat_.flt_reset_freq());}	
	logical					swap_fltflt2s_def_tmpl.is_cross_curr()		{ return false;}
	string					swap_fltflt2s_def_tmpl.currency()			{ return fixflt_sprd_.fix_currency();}
	calendar				swap_fltflt2s_def_tmpl.calendar()			{ return fixflt_sprd_.fix_calendar();}
	
		
	day_count_method 		swap_fltflt2s_def_tmpl.fix_day_count_method() { return fixflt_sprd_.fix_day_count_method();}
	interest_rule			swap_fltflt2s_def_tmpl.fix_ir_rule()		{ return fixflt_sprd_.fix_ir_rule();}
	integer					swap_fltflt2s_def_tmpl.fix_cpn_freq()		{ return fixflt_sprd_.fix_cpn_freq();} 
	//string					swap_fltflt2s_def_tmpl.fix_currency()				{ return currency();}
	//calendar				swap_fltflt2s_def_tmpl.fix_calendar()				{ return calendar();}
	bd_convention 			swap_fltflt2s_def_tmpl.fix_bd_convention()	{ return fixflt_sprd_.fix_bd_convention();} 
	logical					swap_fltflt2s_def_tmpl.fix_eom()			{ return fixflt_sprd_.fix_eom();}
	mtm_code 				swap_fltflt2s_def_tmpl.mtm() 						{ return mtm_code.NONE;}
	logical 				swap_fltflt2s_def_tmpl.is_mtm(out logical leg1)		{ return false; /*..is_mtm(is_cross_curr(), mtm_, leg1);*/}
	
	void swap_fltflt2s_def_tmpl.check()
	{
		QL_REQUIRE(flt_sprd_reset_freq() > flt_flat_reset_freq(),"fixflt_sprd-swap should have higher float leg reset_freq (spread assumed to be added on the swap with higher reset_freq)");
	}
	
	notional_exchg_style swap_fltflt2s_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) 
			return NE_NONE ;		
		else
			return not_exchange_;		
	}

	day_count_method 	swap_fltflt2s_def_tmpl.flt_sprd_day_count_method()	{ return fixflt_sprd_.flt_day_count_method();}
	interest_rule		swap_fltflt2s_def_tmpl.flt_sprd_ir_rule()			{ return fixflt_sprd_.flt_ir_rule();}
	integer				swap_fltflt2s_def_tmpl.flt_sprd_cpn_freq()			{ return fixflt_sprd_.flt_cpn_freq();} 	
	//string				swap_fltflt2s_def_tmpl.flt_sprd_currency()			{ return currency();}
	//calendar			swap_fltflt2s_def_tmpl.flt_sprd_calendar()			{ return fixflt_sprd_.flt_calendar();}
	bd_convention 		swap_fltflt2s_def_tmpl.flt_sprd_bd_convention()		{ return fixflt_sprd_.flt_bd_convention();}
	logical				swap_fltflt2s_def_tmpl.flt_sprd_eom()				{ return fixflt_sprd_.flt_eom();}
	number 				swap_fltflt2s_def_tmpl.flt_sprd_reset_freq()		{ return fixflt_sprd_.flt_reset_freq();} 
	flt_sprd_comp_method swap_fltflt2s_def_tmpl.flt_sprd_spread_comp_method() { return fixflt_sprd_.flt_spread_comp_method();}
	flt_comp_avg_type 	swap_fltflt2s_def_tmpl.flt_sprd_comp_avg()			{ return fixflt_sprd_.flt_comp_avg();} 
	flt_avg_method 		swap_fltflt2s_def_tmpl.flt_sprd_avg_method()		{ return fixflt_sprd_.flt_avg_method();}	
	ir_index 			swap_fltflt2s_def_tmpl.flt_sprd_ir_index()			{ return fixflt_sprd_.flt_ir_index();}
	ir_index_tenor_tmpl	swap_fltflt2s_def_tmpl.flt_sprd_index()				{ return fixflt_sprd_.index();}
	flt_stub_fwd_style	swap_fltflt2s_def_tmpl.flt_sprd_fwd_stub_rate_style(){ return fixflt_sprd_.flt_fwd_stub_rate_style();}

	
	day_count_method 	swap_fltflt2s_def_tmpl.flt_flat_day_count_method()	{ return fixflt_flat_.flt_day_count_method();}
	interest_rule		swap_fltflt2s_def_tmpl.flt_flat_ir_rule()			{ return fixflt_flat_.flt_ir_rule();}				
	integer				swap_fltflt2s_def_tmpl.flt_flat_cpn_freq()			{ return fixflt_flat_.flt_cpn_freq();} 
	//string				swap_fltflt2s_def_tmpl.flt_flat_currency()			{ return currency();}
	//calendar			swap_fltflt2s_def_tmpl.flt_flat_calendar()			{ return fixflt_flat_.flt_calendar();}
	bd_convention 		swap_fltflt2s_def_tmpl.flt_flat_bd_convention()		{ return fixflt_flat_.flt_bd_convention();} 
	logical				swap_fltflt2s_def_tmpl.flt_flat_eom()				{ return fixflt_flat_.flt_eom();} 
	number 				swap_fltflt2s_def_tmpl.flt_flat_reset_freq()		{ return fixflt_flat_.flt_reset_freq();} 
    flt_sprd_comp_method swap_fltflt2s_def_tmpl.flt_flat_spread_comp_method() { return fixflt_flat_.flt_spread_comp_method();} 
	flt_comp_avg_type 	swap_fltflt2s_def_tmpl.flt_flat_comp_avg()			{ return fixflt_flat_.flt_comp_avg();} 
    flt_avg_method 		swap_fltflt2s_def_tmpl.flt_flat_avg_method()		{ return fixflt_flat_.flt_avg_method();}
	ir_index 			swap_fltflt2s_def_tmpl.flt_flat_ir_index()			{ return fixflt_flat_.flt_ir_index();}
	ir_index_tenor_tmpl	swap_fltflt2s_def_tmpl.flt_flat_index()				{ return fixflt_flat_.index();}
	flt_stub_fwd_style	swap_fltflt2s_def_tmpl.flt_flat_fwd_stub_rate_style(){ return fixflt_flat_.flt_fwd_stub_rate_style();}
	
	
	//-------------------------------------------------------
	// class swap_fltois2s_def_tmpl
	//-------------------------------------------------------
	public class swap_fltois2s_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_fltois2s_def_tmpl(	string					name,									
									string 	option(nullable) fwd_start_code,
									string 	option(nullable) maturity_code,
									swap_fixflt_def_tmpl fixflt,
									swap_fixois_def_tmpl fixois);

			string				name();
			swap_fixflt_def_tmpl fixflt();					
			swap_fixois_def_tmpl fixois();
			string 				spot_settle_code();
			string 				fwd_start_code()	;
			string 				maturity_code();
			notional_exchg_style not_exchange();
			logical				is_cross_curr()	;
			string				currency();
			calendar			calendar();
			logical 			primary_quote_is_leg1() ;

			integer				fixo_cpn_freq();
			day_count_method 	fixo_day_count_method();
			interest_rule		fixo_ir_rule();
			bd_convention		fixo_bd_convention();						
			logical				fixo_eom();	
			//calendar			fixo_calendar();

			integer				fixf_cpn_freq();
			day_count_method 	fixf_day_count_method();
			interest_rule		fixf_ir_rule();
			bd_convention		fixf_bd_convention();						
			logical				fixf_eom();	
			//calendar			fixf_calendar();
			
			integer				ois_cpn_freq();
			number 				ois_reset_freq();
			day_count_method 	ois_day_count_method();
			interest_rule		ois_ir_rule();
			//calendar			ois_calendar();
			
			bd_convention		ois_bd_convention();						
			logical				ois_eom();
			flt_comp_avg_type 	ois_comp_avg();
    		flt_sprd_comp_method ois_spread_comp_method();
    		flt_avg_method 		ois_avg_method();
			ir_index_tenor_tmpl	ois_index()	;
			integer 			ois_payment_delay();
			integer 			ois_payment_delay_prin();
			logical				ois_endog_df();
			logical				ois_approx_calc();
			ir_index			ois_ir_index();
			rfr_arrears_type 	ois_arr_type();
			number 				ois_arr_days();
			
			integer				flt_cpn_freq();
			number 				flt_reset_freq(); 
			day_count_method 	flt_day_count_method();
			interest_rule		flt_ir_rule();
			//calendar			flt_calendar();											
			bd_convention		flt_bd_convention();						
			logical				flt_eom();										
			flt_comp_avg_type 	flt_comp_avg()	;
		    flt_sprd_comp_method flt_spread_comp_method() ;
		    flt_avg_method 		flt_avg_method();				
			ir_index_tenor_tmpl	flt_index()	;						
			ir_index			flt_ir_index();
			integer 			flt_payment_delay();
			integer 			flt_payment_delay_prin();
			flt_stub_fwd_style	flt_fwd_stub_rate_style();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
		protected:
			string					name_;
			swap_fixflt_def_tmpl 	fixflt_;
			swap_fixois_def_tmpl 	fixois_;
			string					fwd_start_code_;
			string					maturity_code_;
			notional_exchg_style	not_exchange_;

		
	};
	swap_fltois2s_def_tmpl.swap_fltois2s_def_tmpl(	string name,
													string 	option(nullable) fwd_start_code,
													string 	option(nullable) maturity_code,
													swap_fixflt_def_tmpl fixflt,
													swap_fixois_def_tmpl fixois):
													instrument_def_tmpl(..instr_type.SWAP_FLTOIS2S), name_(name),
													fixflt_(fixflt), fixois_(fixois),
													fwd_start_code_(fwd_start_code),maturity_code_(maturity_code),
													not_exchange_(notional_exchg_style.NE_NONE)
	{}

	string					swap_fltois2s_def_tmpl.name() 				{ return name_;}
	swap_fixflt_def_tmpl 	swap_fltois2s_def_tmpl.fixflt()				{ return fixflt_;} 						
	swap_fixois_def_tmpl 	swap_fltois2s_def_tmpl.fixois()				{ return fixois_;}
	string 					swap_fltois2s_def_tmpl.spot_settle_code() 	{ return fixflt_.spot_settle_code();}
	string 					swap_fltois2s_def_tmpl.fwd_start_code()		{ return fwd_start_code_;}
	string 					swap_fltois2s_def_tmpl.maturity_code()		{ return maturity_code_;}
	string					swap_fltois2s_def_tmpl.currency()			{ return fixflt_.fix_currency();}
	calendar				swap_fltois2s_def_tmpl.calendar()			{ return fixflt_.fix_calendar();}
	logical  				swap_fltois2s_def_tmpl.primary_quote_is_leg1()	{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),currency(),currency(),fixflt_.flt_reset_freq(),365);}	
	logical					swap_fltois2s_def_tmpl.is_cross_curr()		{ return false;}
	
	notional_exchg_style swap_fltois2s_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) 
			return NE_NONE ;		
		else
			return not_exchange_;		
	}

	day_count_method 	swap_fltois2s_def_tmpl.fixo_day_count_method()		{ return fixois_.fix_day_count_method();}
	interest_rule		swap_fltois2s_def_tmpl.fixo_ir_rule()				{ return fixois_.fix_ir_rule();}
	integer				swap_fltois2s_def_tmpl.fixo_cpn_freq()				{ return fixois_.fix_cpn_freq();} 
	bd_convention 		swap_fltois2s_def_tmpl.fixo_bd_convention()			{ return fixois_.bd_convention();} 
	logical				swap_fltois2s_def_tmpl.fixo_eom()					{ return fixois_.eom();}  

	day_count_method 	swap_fltois2s_def_tmpl.fixf_day_count_method()		{ return fixflt_.fix_day_count_method();}
	interest_rule		swap_fltois2s_def_tmpl.fixf_ir_rule()				{ return fixflt_.fix_ir_rule();}
	integer				swap_fltois2s_def_tmpl.fixf_cpn_freq()				{ return fixflt_.fix_cpn_freq();} 
	bd_convention 		swap_fltois2s_def_tmpl.fixf_bd_convention()			{ return fixflt_.fix_bd_convention();} 
	logical				swap_fltois2s_def_tmpl.fixf_eom()					{ return fixflt_.fix_eom();} 

	integer 			swap_fltois2s_def_tmpl.flt_payment_delay()		{ return 0;}
	integer 			swap_fltois2s_def_tmpl.flt_payment_delay_prin()	{ return 0;}
	day_count_method 	swap_fltois2s_def_tmpl.flt_day_count_method()	{ return fixflt_.flt_day_count_method();}
	interest_rule		swap_fltois2s_def_tmpl.flt_ir_rule()			{ return fixflt_.flt_ir_rule();}
	integer				swap_fltois2s_def_tmpl.flt_cpn_freq()			{ return fixflt_.flt_cpn_freq();}
	number 				swap_fltois2s_def_tmpl.flt_reset_freq()			{ return fixflt_.flt_reset_freq();} 
	bd_convention 		swap_fltois2s_def_tmpl.flt_bd_convention()		{ return fixflt_.flt_bd_convention();} 
	logical				swap_fltois2s_def_tmpl.flt_eom()				{ return fixflt_.flt_eom();} 
	ir_index_tenor_tmpl	swap_fltois2s_def_tmpl.flt_index()				{ return fixflt_.index();}
	flt_comp_avg_type 	swap_fltois2s_def_tmpl.flt_comp_avg()			{ return fixflt_.flt_comp_avg();} 
    flt_sprd_comp_method  swap_fltois2s_def_tmpl.flt_spread_comp_method() { return fixflt_.flt_spread_comp_method();} 
    flt_avg_method 		swap_fltois2s_def_tmpl.flt_avg_method()			{ return fixflt_.flt_avg_method();}
	ir_index 			swap_fltois2s_def_tmpl.flt_ir_index()			{ return fixflt_.flt_ir_index();}
	flt_stub_fwd_style	swap_fltois2s_def_tmpl.flt_fwd_stub_rate_style(){ return fixflt_.flt_fwd_stub_rate_style();}
	
		
	integer 			swap_fltois2s_def_tmpl.ois_payment_delay()		{ return fixois_.payment_delay();}
	integer 			swap_fltois2s_def_tmpl.ois_payment_delay_prin()	{ return fixois_.payment_delay_prin();}
	rfr_arrears_type 	swap_fltois2s_def_tmpl.ois_arr_type()			{ return fixois_.ois_arr_type();}
	number 				swap_fltois2s_def_tmpl.ois_arr_days()			{ return fixois_.ois_arr_days();}
	logical				swap_fltois2s_def_tmpl.ois_endog_df()			{ return fixois_.ois_endog_df();}
	logical				swap_fltois2s_def_tmpl.ois_approx_calc()		{ return fixois_.ois_approx_calc();}		
	day_count_method 	swap_fltois2s_def_tmpl.ois_day_count_method()	{ return fixois_.ois_day_count_method();} 						
	interest_rule		swap_fltois2s_def_tmpl.ois_ir_rule()			{ return fixois_.ois_ir_rule();}
	integer				swap_fltois2s_def_tmpl.ois_cpn_freq()			{ return fixois_.ois_cpn_freq();} 					
	bd_convention 		swap_fltois2s_def_tmpl.ois_bd_convention()		{ return fixois_.bd_convention();} 
	logical				swap_fltois2s_def_tmpl.ois_eom()				{ return fixois_.eom();}
	ir_index_tenor_tmpl	swap_fltois2s_def_tmpl.ois_index()				{ return fixois_.on_index();}				
	number 				swap_fltois2s_def_tmpl.ois_reset_freq()			{ return 365;} 
	flt_comp_avg_type 	swap_fltois2s_def_tmpl.ois_comp_avg()			{ return fixois_.ois_comp_avg();} 
    flt_sprd_comp_method  swap_fltois2s_def_tmpl.ois_spread_comp_method() { return fixois_.ois_spread_comp_method();} 
    flt_avg_method 		swap_fltois2s_def_tmpl.ois_avg_method()			{ return fixois_.ois_avg_method();} 
	ir_index 			swap_fltois2s_def_tmpl.ois_ir_index()			{ return fixois_.ois_ir_index();}
	mtm_code 			swap_fltois2s_def_tmpl.mtm() 						{ return mtm_code.NONE;}
	logical 			swap_fltois2s_def_tmpl.is_mtm(out logical leg1)		{ return false; /*..is_mtm(is_cross_curr(), mtm_, leg1);*/}
	//-------------------------------------------------------
	// class swap_fixfix_def_tmpl
	//-------------------------------------------------------
	public class swap_fixfix_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_fixfix_def_tmpl(	string,string,string 	option(nullable),string option(nullable),integer,day_count_method,interest_rule,
									string,..calendar, bd_convention,logical,integer ,
									day_count_method,interest_rule,string,..calendar,
									bd_convention,logical,notional_exchg_style option(nullable));
			
			string				name();
			string 				spot_settle_code();
			string 				fwd_start_code();
			string 				maturity_code();						
			notional_exchg_style not_exchange()	;
			logical				is_cross_curr();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
			
			integer				fix1_cpn_freq();
			bd_convention		fix1_bd_convention();						
			logical				fix1_eom();
			day_count_method 	fix1_day_count_method();
			interest_rule		fix1_ir_rule();
			calendar			fix1_calendar();
			string				fix1_currency();
			
			integer				fix2_cpn_freq();			
			day_count_method 	fix2_day_count_method();
			interest_rule		fix2_ir_rule();
			calendar			fix2_calendar();										
			string				fix2_currency();
			bd_convention		fix2_bd_convention();						
			logical				fix2_eom();	
								
		protected:
			string				name_;
			integer				fix1_cpn_freq_;
			day_count_method 	fix1_dc_;
			interest_rule		fix1_ir_rule_;
			string				fix1_curr_;
			..calendar			fix1_cal_;
			bd_convention		fix1_bd_;
			logical				fix1_eom_; 
			integer				fix2_cpn_freq_;
			day_count_method 	fix2_dc_;
			interest_rule		fix2_ir_rule_;
			string				fix2_curr_;
			..calendar			fix2_cal_;
			bd_convention		fix2_bd_;
			logical				fix2_eom_; 												
			string				spot_code_;
			string				fwd_start_code_;
			string				maturity_code_;
			notional_exchg_style not_exchange_;				
			
	};

	swap_fixfix_def_tmpl.swap_fixfix_def_tmpl(	string				name,
											string 					spot_settle_code,
											string 	option(nullable) fwd_start_code,
											string option(nullable) maturity_code,
											integer 				fix1_cpn_freq,	
											..day_count_method 		fix1_dc,
											interest_rule			fix1_ir_rule,
											string 					fix1_curr,
											..calendar					fix1_cal,
											..bd_convention 		fix1_bd,												
											logical 				fix1_eom, 						       
											integer 				fix2_cpn_freq,	  
											..day_count_method 		fix2_dc,
											interest_rule			fix2_ir_rule,
											string 					fix2_curr,
											..calendar					fix2_cal,	
											..bd_convention 		fix2_bd,												
											logical 				fix2_eom,  								
											notional_exchg_style option(nullable) not_exchange)
								: 	instrument_def_tmpl(..instr_type.SWAP_FIXFIX), name_(name),fix1_cpn_freq_(fix1_cpn_freq), fix1_dc_(fix1_dc),fix1_ir_rule_(fix1_ir_rule),
									fix1_curr_(fix1_curr),fix1_cal_(fix1_cal),
									fix1_bd_(fix1_bd),fix1_eom_(fix1_eom),fix2_cpn_freq_(fix2_cpn_freq),fix2_dc_(fix2_dc),fix2_ir_rule_(fix2_ir_rule),
									fix2_curr_(fix2_curr),fix2_cal_(fix2_cal),
									fix2_bd_(fix2_bd),fix2_eom_(fix2_eom),spot_code_(spot_settle_code),fwd_start_code_(fwd_start_code),maturity_code_(maturity_code),not_exchange_(not_exchange) {}

	string				swap_fixfix_def_tmpl.name() 					{ return name_;}
	day_count_method 	swap_fixfix_def_tmpl.fix1_day_count_method()	{ return fix1_dc_;} 						
	day_count_method 	swap_fixfix_def_tmpl.fix2_day_count_method()	{ return fix2_dc_;}
	interest_rule		swap_fixfix_def_tmpl.fix1_ir_rule()				{ return fix1_ir_rule_;}
	interest_rule		swap_fixfix_def_tmpl.fix2_ir_rule()				{ return fix2_ir_rule_;}
	integer				swap_fixfix_def_tmpl.fix1_cpn_freq()			{ return fix1_cpn_freq_;} 					
	integer				swap_fixfix_def_tmpl.fix2_cpn_freq()			{ return fix2_cpn_freq_;} 
	string				swap_fixfix_def_tmpl.fix1_currency()			{ return fix1_curr_;}
	string				swap_fixfix_def_tmpl.fix2_currency()			{ return fix2_curr_;}
	calendar			swap_fixfix_def_tmpl.fix1_calendar()			{ return null(fix1_cal_) ? ..calendar() : fix1_cal_;}
	calendar			swap_fixfix_def_tmpl.fix2_calendar()			{ return null(fix2_cal_) ? ..calendar() : fix2_cal_;}
	bd_convention 		swap_fixfix_def_tmpl.fix2_bd_convention()		{ return fix2_bd_;} 
	logical				swap_fixfix_def_tmpl.fix2_eom()					{ return fix2_eom_;} 
	bd_convention 		swap_fixfix_def_tmpl.fix1_bd_convention()		{ return fix1_bd_;} 
	logical				swap_fixfix_def_tmpl.fix1_eom()					{ return fix1_eom_;} 
	string 				swap_fixfix_def_tmpl.spot_settle_code()			{ return spot_code_;} 
	string 				swap_fixfix_def_tmpl.fwd_start_code()			{ return fwd_start_code_;} 
	string 				swap_fixfix_def_tmpl.maturity_code()			{ return maturity_code_;}
	logical				swap_fixfix_def_tmpl.is_cross_curr()			{ return !equal_casei(fix1_curr_, fix2_curr_);}	
	mtm_code 			swap_fixfix_def_tmpl.mtm() 						{ return mtm_code.NONE;}
	logical 			swap_fixfix_def_tmpl.is_mtm(out logical leg1)	{ return false; /*..is_mtm(is_cross_curr(), mtm_, leg1);*/}
	
	notional_exchg_style swap_fixfix_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) {
			if( equal_casei(fix1_curr_, fix2_curr_)) 
				return NE_NONE ;
			else 
				return NE_BOTH;
		}
		else
			return not_exchange_;
		
	}

	//-------------------------------------------------------
	// class swap_zeroflt_def_tmpl
	/* zero_comp_freq:
			1	= Annual compounding.
			2	= Semi-annual compounding.
			4	= Quarterly compounding
			12	= Montly compounding.
			-1	= Simple rate (no compounding)
			0	= Discount rate, (1 - coupon rate) * notional = interest. This is a "period-less" style.*/

	//-------------------------------------------------------
	public class swap_zeroflt_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_zeroflt_def_tmpl(	string,string ,string option(nullable),string option(nullable),day_count_method,string,..calendar, bd_convention,logical,
									integer ,integer,day_count_method,interest_rule,string,..calendar,
									bd_convention,logical,notional_exchg_style option(nullable) ,ir_index_tenor_tmpl);
			
			string				name();
			string 				spot_settle_code();
			string 				fwd_start_code();
			string 				maturity_code()	;
			notional_exchg_style not_exchange()	;
			logical 			primary_quote_is_leg1() ;		
			logical				is_cross_curr();
			
			day_count_method 	zero_day_count_method(); 
			integer 			zero_comp_freq(); 	
			bd_convention		zero_bd_convention();						
			logical				zero_eom();	
			calendar			zero_calendar();
			string				zero_currency();

			day_count_method 	flt_day_count_method();
			interest_rule		flt_ir_rule();
			calendar			flt_calendar();	
			integer				flt_cpn_freq();			
			string				flt_currency();
			bd_convention		flt_bd_convention();						
			logical				flt_eom();											
			ir_index_tenor_tmpl	index()	;			
			ir_index			flt_ir_index();
			flt_stub_fwd_style	flt_fwd_stub_rate_style();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
		protected:
			string				name_;
			
			day_count_method 	zero_dc_;			
			string				zero_curr_;
			..calendar			zero_cal_;
			bd_convention		zero_bd_;
			logical				zero_eom_;
			integer 			zero_comp_freq_; 
			integer				flt_cpn_freq_;
			integer				flt_reset_freq_;
			day_count_method 	flt_dc_;
			interest_rule		flt_ir_rule_;
			string				flt_curr_;
			..calendar			flt_cal_;
			bd_convention		flt_bd_;
			logical				flt_eom_;
			flt_stub_fwd_style	flt_fwd_stub_rate_style_;
			string				spot_code_;
			string				fwd_start_code_;
			string				maturity_code_;				
			ir_index_tenor_tmpl	index_;
			notional_exchg_style not_exchange_;
	};

	swap_zeroflt_def_tmpl.swap_zeroflt_def_tmpl(string					name,
												string 					spot_settle_code,
												string 	option(nullable) fwd_start_code,
												string option(nullable)	 maturity_code,												
												..day_count_method 		zero_dc,
												string 					zero_curr,
												..calendar				zero_cal,
												..bd_convention 		zero_bd,												
												logical 				zero_eom, 						       
												integer 				zero_comp_freq, 
												integer 				flt_cpn_freq,	  
												..day_count_method 		flt_dc,
												interest_rule			flt_ir_rule,
												string 					flt_curr,
												..calendar				flt_cal,	
												..bd_convention 		flt_bd,												
												logical 				flt_eom,  								
												notional_exchg_style option(nullable) not_exchange,	
												ir_index_tenor_tmpl	index)
											: 	instrument_def_tmpl(..instr_type.SWAP_ZEROFLT), name_(name), zero_dc_(zero_dc),zero_curr_(zero_curr),zero_cal_(zero_cal),  
												zero_bd_(zero_bd),zero_eom_(zero_eom),zero_comp_freq_(zero_comp_freq),flt_cpn_freq_(flt_cpn_freq),flt_reset_freq_(flt_cpn_freq),
												flt_dc_(flt_dc),flt_ir_rule_(flt_ir_rule),flt_curr_(flt_curr),flt_cal_(flt_cal), 
												flt_bd_(flt_bd),flt_eom_(flt_eom),flt_fwd_stub_rate_style_(null),
												spot_code_(spot_settle_code),fwd_start_code_(fwd_start_code),maturity_code_(maturity_code),
												index_(index),not_exchange_(not_exchange) {}

	string				swap_zeroflt_def_tmpl.name() 					{ return name_;}
	day_count_method 	swap_zeroflt_def_tmpl.zero_day_count_method()	{ return zero_dc_;} 						
	day_count_method 	swap_zeroflt_def_tmpl.flt_day_count_method()	{ return flt_dc_;}
	interest_rule 		swap_zeroflt_def_tmpl.flt_ir_rule()				{ return flt_ir_rule_;} 
	integer 			swap_zeroflt_def_tmpl.zero_comp_freq()			{ return zero_comp_freq_;} 					
	integer				swap_zeroflt_def_tmpl.flt_cpn_freq()			{ return flt_cpn_freq_;} 
	string				swap_zeroflt_def_tmpl.zero_currency()			{ return zero_curr_;}
	string				swap_zeroflt_def_tmpl.flt_currency()			{ return flt_curr_;}
	calendar			swap_zeroflt_def_tmpl.zero_calendar()			{ return null(zero_cal_) ? ..calendar() : zero_cal_;}
	calendar			swap_zeroflt_def_tmpl.flt_calendar()			{ return null(flt_cal_) ? ..calendar() : flt_cal_;}
	bd_convention 		swap_zeroflt_def_tmpl.flt_bd_convention()		{ return flt_bd_;} 
	logical				swap_zeroflt_def_tmpl.flt_eom()					{ return flt_eom_;}

	flt_stub_fwd_style	swap_zeroflt_def_tmpl.flt_fwd_stub_rate_style(){ return flt_fwd_stub_rate_style_;}
	
	bd_convention 		swap_zeroflt_def_tmpl.zero_bd_convention()		{ return zero_bd_;} 
	logical				swap_zeroflt_def_tmpl.zero_eom()				{ return zero_eom_;} 
	string 				swap_zeroflt_def_tmpl.spot_settle_code()		{ return spot_code_;} 
	string 				swap_zeroflt_def_tmpl.fwd_start_code()			{ return fwd_start_code_;} 
	string 				swap_zeroflt_def_tmpl.maturity_code()			{ return maturity_code_;}  					
	ir_index_tenor_tmpl	swap_zeroflt_def_tmpl.index()					{ return index_;}
	logical				swap_zeroflt_def_tmpl.is_cross_curr()			{ return !equal_casei(zero_curr_, flt_curr_);}
	logical  			swap_zeroflt_def_tmpl.primary_quote_is_leg1()	{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),zero_curr_,flt_curr_,0,flt_reset_freq_);}	
	mtm_code 			swap_zeroflt_def_tmpl.mtm() 						{ return mtm_code.NONE;}
	logical 			swap_zeroflt_def_tmpl.is_mtm(out logical leg1)		{ return false; /*..is_mtm(is_cross_curr(), mtm_, leg1);*/}
	
	notional_exchg_style swap_zeroflt_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) {
			if( equal_casei(zero_curr_, flt_curr_)) 
				return NE_NONE ;
			else 
				return NE_BOTH;
		}
		else
			return not_exchange_;
		
	}
	ir_index swap_zeroflt_def_tmpl.flt_ir_index()
	{
		string t = index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	index_.name(),t,
								index_.tenor(),
								index_.value_days(),
								index_.fixing_days(),
								index_.day_count_method(),
								index_.bd_convention(),
								index_.eom(),
								index_.cal(),//if libor: this is uk
								index_.cal_name(),
								index_.currency(),
								index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								index_.suppl_cal_name(),
								index_.spot_days(),0);
	}
	
	//-------------------------------------------------------
	// class swap_fltois_def_tmpl  
	//-------------------------------------------------------
	public class swap_fltois_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_fltois_def_tmpl(	string					name,
									string 					spot_settle_code,
									string 	option(nullable) fwd_start_code,
									string 	option(nullable) maturity_code,
									integer 				payment_delay,
									integer 				payment_delay_prin,
									notional_exchg_style 	not_exchange,
									mtm_code option(nullable) mtm,
									integer  flt_cpn_freq,
									number option(nullable)	flt_reset_freq,
									..day_count_method 		flt_dc,													
									..calendar				flt_cal,
									..bd_convention 		flt_bd,												
									logical 				flt_eom,												
									string					flt_curr,
									interest_rule			flt_ir_rule,
									flt_stub_fwd_style option(nullable)	flt_fwd_stub_rate_style,
									ir_index_tenor_tmpl		flt_index,
									integer 				ois_cpn_freq,												
									..day_count_method 		ois_dc,													
									..calendar				ois_cal,
									..bd_convention 		ois_bd,												
									logical 				ois_eom,
									string									ois_curr,
									rfr_arrears_type option(nullable) 		ois_arr_type,
									number option(nullable)					ois_arr_days,
									flt_comp_avg_type option(nullable)		ois_comp_avg,
									flt_sprd_comp_method option(nullable)	ois_spread_comp_method,
									flt_avg_method option(nullable)			ois_avg_method,
									interest_rule							ois_ir_rule,
									logical 								ois_endog_df,
									logical									ois_approx_calc,												
									ir_index_tenor_tmpl						ois_index);
			
			string				name();
			string 				spot_settle_code();
			string 				fwd_start_code();
			string 				maturity_code();
			integer				payment_delay();
			integer				payment_delay_prin();
			notional_exchg_style not_exchange();
			logical 			primary_quote_is_leg1() ;
			logical				is_cross_curr();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
			
			integer				flt_cpn_freq();
			integer				flt_reset_freq();	
			day_count_method 	flt_day_count_method();			
			calendar			flt_calendar();			
			bd_convention		flt_bd_convention();						
			logical				flt_eom();
	
			string				flt_currency();		
			ir_index_tenor_tmpl	flt_index()	;
			ir_index			flt_ir_index();
			interest_rule		flt_ir_rule();
			flt_stub_fwd_style	flt_fwd_stub_rate_style();

			integer				ois_cpn_freq();
			number				ois_reset_freq();	
			day_count_method 	ois_day_count_method();			
			calendar			ois_calendar();			
			bd_convention		ois_bd_convention();						
			logical				ois_eom();
			
			string				ois_currency();
			flt_comp_avg_type 	ois_comp_avg();
    		flt_sprd_comp_method ois_spread_comp_method();
    		flt_avg_method 		ois_avg_method();
			rfr_arrears_type 	ois_arr_type();
			number 				ois_arr_days();
			logical 			ois_endog_df();
			interest_rule		ois_ir_rule();
			logical				ois_approx_calc();
			ir_index_tenor_tmpl	ois_index()	;
			ir_index			ois_ir_index();						
			
		protected:
			string				name_;
			string				spot_code_;
			string				fwd_start_code_;
			string				maturity_code_;
			integer				payment_lag_;
			integer				payment_lag_prin_;
			notional_exchg_style not_exchange_;
			mtm_code			mtm_;
			
			integer				flt_cpn_freq_;
			number				flt_reset_freq_;
			day_count_method 	flt_dc_;						
			..calendar			flt_cal_;
			bd_convention		flt_bd_;
			logical				flt_eom_;
			string				flt_curr_;
			ir_index_tenor_tmpl	flt_index_;
			interest_rule		flt_ir_rule_;
			flt_stub_fwd_style	flt_fwd_stub_rate_style_;
			
			integer				ois_cpn_freq_;	
			day_count_method 	ois_dc_;						
			..calendar			ois_cal_;
			bd_convention		ois_bd_;
			logical				ois_eom_;
			string				ois_curr_;
			rfr_arrears_type 	ois_arr_type_;
			number 				ois_arr_days_;
			logical				ois_endog_df_;
			logical				ois_approx_calc_;
			flt_comp_avg_type 	ois_comp_avg_;
    		flt_sprd_comp_method ois_spread_comp_method_;
    		flt_avg_method 		ois_avg_method_;
			interest_rule		ois_ir_rule_;			
			ir_index_tenor_tmpl	on_index_;
			
	};

	swap_fltois_def_tmpl.swap_fltois_def_tmpl(	string					name,
												string 					spot_settle_code,
												string 	option(nullable) fwd_start_code,
												string 	option(nullable) maturity_code,
												integer 				payment_delay,
												integer 				payment_delay_prin,
												notional_exchg_style 	not_exchange,
												mtm_code option(nullable) mtm,
												
												integer 				flt_cpn_freq,
												number option(nullable)	flt_reset_freq,
												..day_count_method 		flt_dc,													
												..calendar				flt_cal,
												..bd_convention 		flt_bd,												
												logical 				flt_eom,												
												string					flt_curr,
												interest_rule			flt_ir_rule,
												flt_stub_fwd_style option(nullable)	flt_fwd_stub_rate_style,
												ir_index_tenor_tmpl		flt_index,

												integer 				ois_cpn_freq,												
												..day_count_method 		ois_dc,													
												..calendar				ois_cal,
												..bd_convention 		ois_bd,												
												logical 				ois_eom,
												string									ois_curr,
												rfr_arrears_type option(nullable) 		ois_arr_type,
												number option(nullable)					ois_arr_days,
												flt_comp_avg_type option(nullable)		ois_comp_avg,
				    							flt_sprd_comp_method option(nullable)	ois_spread_comp_method,
				    							flt_avg_method option(nullable)			ois_avg_method,
												interest_rule							ois_ir_rule,
												logical 								ois_endog_df,
												logical									ois_approx_calc,												
												ir_index_tenor_tmpl						ois_index)
												: 	instrument_def_tmpl(..instr_type.SWAP_FLTOIS),name_(name),spot_code_(spot_settle_code),
													fwd_start_code_(fwd_start_code), maturity_code_(maturity_code),
													payment_lag_(payment_delay),payment_lag_prin_(payment_delay_prin),
													not_exchange_(not_exchange), mtm_(mtm),

													flt_cpn_freq_(flt_cpn_freq), flt_reset_freq_(null(flt_reset_freq) ? flt_cpn_freq: flt_reset_freq), flt_dc_(flt_dc),flt_cal_(flt_cal),
													flt_bd_(flt_bd),flt_eom_(flt_eom),
													flt_curr_(flt_curr), flt_index_(flt_index),flt_ir_rule_(flt_ir_rule),flt_fwd_stub_rate_style_(flt_fwd_stub_rate_style),

													ois_cpn_freq_(ois_cpn_freq), ois_dc_(ois_dc),ois_cal_(ois_cal),
													ois_bd_(ois_bd),ois_eom_(ois_eom),
													
													ois_curr_(ois_curr),
													ois_arr_type_(ois_arr_type), ois_arr_days_(ois_arr_days),
													ois_endog_df_(ois_endog_df),ois_approx_calc_(ois_approx_calc),
													ois_comp_avg_(ois_comp_avg),
													ois_spread_comp_method_(ois_spread_comp_method),ois_avg_method_(ois_avg_method),
													ois_ir_rule_(ois_ir_rule), on_index_(ois_index) {}

	notional_exchg_style swap_fltois_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) {
			if( !is_cross_curr()) 
				return NE_NONE ;
			else 
				return NE_BOTH;
		}
		else
			return not_exchange_;
		
	}
	
	string				swap_fltois_def_tmpl.name() 					{ return name_;}
	integer				swap_fltois_def_tmpl.payment_delay()			{ return payment_lag_;}
	integer				swap_fltois_def_tmpl.payment_delay_prin()		{ return payment_lag_prin_;}
	string 				swap_fltois_def_tmpl.spot_settle_code()			{ return spot_code_;} 		
	string 				swap_fltois_def_tmpl.fwd_start_code()			{ return fwd_start_code_;}  
	string 				swap_fltois_def_tmpl.maturity_code()			{ return maturity_code_;} 
	mtm_code 			swap_fltois_def_tmpl.mtm()						{ return mtm_;}
	logical				swap_fltois_def_tmpl.is_cross_curr()			{ return !equal_casei(flt_curr_, ois_curr_);}

	
	day_count_method 	swap_fltois_def_tmpl.flt_day_count_method()		{ return flt_dc_;}
	interest_rule		swap_fltois_def_tmpl.flt_ir_rule()				{ return flt_ir_rule_;}
	integer				swap_fltois_def_tmpl.flt_cpn_freq()				{ return flt_cpn_freq_;}
	integer				swap_fltois_def_tmpl.flt_reset_freq()			{ return null(flt_reset_freq_) ? flt_cpn_freq_ : integer(flt_reset_freq_);} 
	calendar			swap_fltois_def_tmpl.flt_calendar()				{ return null(flt_cal_) ? ..calendar() : flt_cal_;}
	bd_convention 		swap_fltois_def_tmpl.flt_bd_convention()		{ return flt_bd_;} 
	logical				swap_fltois_def_tmpl.flt_eom()					{ return flt_eom_;}	
	string				swap_fltois_def_tmpl.flt_currency()				{ return flt_curr_;}
	ir_index_tenor_tmpl	swap_fltois_def_tmpl.flt_index()				{ return flt_index_;}
	flt_stub_fwd_style	swap_fltois_def_tmpl.flt_fwd_stub_rate_style()	{ return flt_fwd_stub_rate_style_;}

	day_count_method 	swap_fltois_def_tmpl.ois_day_count_method()		{ return ois_dc_;}
	integer				swap_fltois_def_tmpl.ois_cpn_freq()				{ return ois_cpn_freq_;}
	number				swap_fltois_def_tmpl.ois_reset_freq()			{ return 365;} 
	calendar			swap_fltois_def_tmpl.ois_calendar()				{ return null(ois_cal_) ? ..calendar() : ois_cal_;}
	bd_convention 		swap_fltois_def_tmpl.ois_bd_convention()		{ return ois_bd_;} 
	logical				swap_fltois_def_tmpl.ois_eom()					{ return ois_eom_;}	
	string				swap_fltois_def_tmpl.ois_currency()				{ return ois_curr_;}
	flt_comp_avg_type 	swap_fltois_def_tmpl.ois_comp_avg()				{ return ois_comp_avg_;}
    flt_sprd_comp_method swap_fltois_def_tmpl.ois_spread_comp_method()	{ return ois_spread_comp_method_;}
    flt_avg_method 		swap_fltois_def_tmpl.ois_avg_method()			{ return ois_avg_method_;}
	rfr_arrears_type 	swap_fltois_def_tmpl.ois_arr_type()				{ return ois_arr_type_;}
	number 				swap_fltois_def_tmpl.ois_arr_days()				{ return ois_arr_days_;}
	logical 			swap_fltois_def_tmpl.ois_endog_df()				{ return ois_endog_df_;}
	
	ir_index_tenor_tmpl	swap_fltois_def_tmpl.ois_index()				{ return on_index_;}	
	interest_rule		swap_fltois_def_tmpl.ois_ir_rule()				{ return interest_rule.IR_ADJ;}
	logical				swap_fltois_def_tmpl.ois_approx_calc()			{ return ois_approx_calc_;}
	logical  			swap_fltois_def_tmpl.primary_quote_is_leg1()	{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),flt_curr_,ois_curr_,flt_reset_freq(),365);}
	
	logical swap_fltois_def_tmpl.is_mtm(out logical leg1)	{ return ..is_mtm(is_cross_curr(), mtm_, leg1);}
	
	ir_index swap_fltois_def_tmpl.ois_ir_index()
	{
		string t = on_index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR") && !equal_casei (t, "RFR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	on_index_.name(),t,
								on_index_.tenor(),
								on_index_.value_days(),
								on_index_.fixing_days(),
								on_index_.day_count_method(),
								on_index_.bd_convention(),
								on_index_.eom(),
								on_index_.cal(),//if libor: this is uk
								on_index_.cal_name(),
								on_index_.currency(),
								on_index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								on_index_.suppl_cal_name(),
								on_index_.spot_days(),
								on_index_.spread());
	}
	ir_index swap_fltois_def_tmpl.flt_ir_index()
	{
		string t = flt_index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	flt_index_.name(),t,
								flt_index_.tenor(),
								flt_index_.value_days(),
								flt_index_.fixing_days(),
								flt_index_.day_count_method(),
								flt_index_.bd_convention(),
								flt_index_.eom(),
								flt_index_.cal(),//if libor: this is uk
								flt_index_.cal_name(),
								flt_index_.currency(),
								flt_index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								flt_index_.suppl_cal_name(),
								flt_index_.spot_days(),0);
	}


	//-------------------------------------------------------
	// class swap_zeroois_def_tmpl
	//-------------------------------------------------------
	public class swap_zeroois_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_zeroois_def_tmpl(	string,string ,string option(nullable),string option(nullable),day_count_method,
									string,..calendar, bd_convention,logical,integer,
									integer ,..day_count_method,string 	,..calendar	,..bd_convention ,logical ,
									rfr_arrears_type option(nullable), number option(nullable) ,
									flt_comp_avg_type option(nullable),flt_sprd_comp_method option(nullable),
									flt_avg_method option(nullable)	,logical,logical,integer,integer,notional_exchg_style option(nullable),ir_index_tenor_tmpl);
			
			string				name();			
			integer				payment_lag();
			integer				payment_lag_prin();
			notional_exchg_style not_exchange();
			string 				spot_settle_code();
			string 				fwd_start_code();
			string 				maturity_code()	;
			logical 			primary_quote_is_leg1() ;
			
			day_count_method 	zero_day_count_method(); 	
			calendar			zero_calendar();
			string				zero_currency();	
			integer 			zero_comp_freq(); 	
			bd_convention		zero_bd_convention();						
			logical				zero_eom();				

			integer				ois_cpn_freq();	
			day_count_method 	ois_day_count_method(); 			
			calendar			ois_calendar();	
			bd_convention		ois_bd_convention();						
			logical				ois_eom();				
			string				ois_currency();
			
			ir_index_tenor_tmpl	on_index()	;
			ir_index			ois_ir_index();
			flt_comp_avg_type 	ois_comp_avg();
    		flt_sprd_comp_method ois_spread_comp_method();
    		flt_avg_method 		ois_avg_method();
			rfr_arrears_type 	ois_arr_type();
			number 				ois_arr_days();
			logical				ois_endog_df();
			logical				ois_approx_calc();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
			
		protected:
			string				name_;
			string				spot_code_;
			string				fwd_start_code_;
			string 				maturity_code_;
			integer				payment_lag_;
			integer				payment_lag_prin_;
			
			day_count_method 	zero_dc_;
			string				zero_curr_;
			..calendar			zero_cal_;
			bd_convention		zero_bd_;
			logical				zero_eom_;
			integer 			zero_comp_freq_; 

			day_count_method 	ois_dc_;
			..calendar			ois_cal_;
			bd_convention		ois_bd_;
			logical				ois_eom_;
			integer 			ois_cpn_freq_; 
			string				ois_curr_;
			rfr_arrears_type 	ois_arr_type_;
			number 				ois_arr_days_;
			logical				ois_endog_df_;
			logical				ois_approx_calc_;
			flt_comp_avg_type 	ois_comp_avg_;
    		flt_sprd_comp_method ois_spread_comp_method_;
    		flt_avg_method 		ois_avg_method_;							
			ir_index_tenor_tmpl	on_index_;
			notional_exchg_style	not_exchange_;
			
	};

	swap_zeroois_def_tmpl.swap_zeroois_def_tmpl(string					name,
												string 					spot_settle_code,
												string 	option(nullable) fwd_start_code,
												string 	option(nullable) maturity_code,											
												..day_count_method 		zero_dc,
												string 					zero_curr,
												..calendar				zero_cal,
												..bd_convention 		zero_bd,												
												logical 				zero_eom, 						       
												integer 				zero_comp_freq,

												integer 				ois_cpn_freq,	  
												..day_count_method 		ois_dc,
												string 					ois_curr,
												..calendar				ois_cal,	
												..bd_convention 		ois_bd,												
												logical 				ois_eom,																						
												rfr_arrears_type option(nullable) 		ois_arr_type,
												number option(nullable)					ois_arr_days,
												flt_comp_avg_type option(nullable)		ois_comp_avg,
				    							flt_sprd_comp_method option(nullable)	ois_spread_comp_method,
				    							flt_avg_method option(nullable)			ois_avg_method,
												logical 								ois_endog_df,
												logical									ois_approx_calc,
												integer 								payment_lag,
												integer 								payment_lag_prin,
												notional_exchg_style option(nullable)	not_exchange, 
												ir_index_tenor_tmpl						on_index)
												: 	instrument_def_tmpl(..instr_type.SWAP_ZEROOIS),name_(name),spot_code_(spot_settle_code),fwd_start_code_(fwd_start_code),
													maturity_code_(maturity_code), payment_lag_(payment_lag), payment_lag_prin_(payment_lag_prin),
													zero_dc_(zero_dc),zero_curr_(zero_curr),zero_cal_(zero_cal),
													zero_bd_(zero_bd),zero_eom_(zero_eom),zero_comp_freq_(zero_comp_freq),

													ois_dc_(ois_dc),ois_cal_(ois_cal),ois_bd_(ois_bd),ois_eom_(ois_eom),
													ois_cpn_freq_(ois_cpn_freq), ois_curr_(ois_curr), ois_arr_type_(ois_arr_type),
													ois_arr_days_(ois_arr_days), ois_endog_df_(ois_endog_df), ois_approx_calc_(ois_approx_calc),
													ois_comp_avg_(ois_comp_avg),
													ois_spread_comp_method_(ois_spread_comp_method),ois_avg_method_(ois_avg_method),
													on_index_(on_index),not_exchange_(not_exchange) {}

	notional_exchg_style swap_zeroois_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) {
			if( equal_casei(zero_curr_, ois_curr_)) 
				return NE_NONE ;
			else 
				return NE_BOTH;
		}
		else
			return not_exchange_;
		
	}
	
	string				swap_zeroois_def_tmpl.name() 					{ return name_;}
	day_count_method 	swap_zeroois_def_tmpl.zero_day_count_method()	{ return zero_dc_;}
	integer 			swap_zeroois_def_tmpl.zero_comp_freq()			{ return zero_comp_freq_;}
	string				swap_zeroois_def_tmpl.zero_currency()			{ return zero_curr_;}
	calendar			swap_zeroois_def_tmpl.zero_calendar()			{ return null(zero_cal_) ? ..calendar() : zero_cal_;}
	bd_convention 		swap_zeroois_def_tmpl.zero_bd_convention()		{ return zero_bd_;} 
	logical				swap_zeroois_def_tmpl.zero_eom()				{ return zero_eom_;}
	integer				swap_zeroois_def_tmpl.payment_lag()				{ return payment_lag_;}
	integer				swap_zeroois_def_tmpl.payment_lag_prin()		{ return payment_lag_prin_;}
	day_count_method 	swap_zeroois_def_tmpl.ois_day_count_method()	{ return ois_dc_;}
	calendar			swap_zeroois_def_tmpl.ois_calendar()			{ return null(ois_cal_) ? ..calendar() : ois_cal_;}
	bd_convention		swap_zeroois_def_tmpl.ois_bd_convention()		{ return ois_bd_;}						
	logical				swap_zeroois_def_tmpl.ois_eom()					{ return ois_eom_;}							
	integer 			swap_zeroois_def_tmpl.ois_cpn_freq()			{ return ois_cpn_freq_;}
	string				swap_zeroois_def_tmpl.ois_currency()			{ return ois_curr_;}
	flt_comp_avg_type 	swap_zeroois_def_tmpl.ois_comp_avg()			{ return ois_comp_avg_;}
    flt_sprd_comp_method swap_zeroois_def_tmpl.ois_spread_comp_method()	{ return ois_spread_comp_method_;}
    flt_avg_method 		swap_zeroois_def_tmpl.ois_avg_method()			{ return ois_avg_method_;}
	rfr_arrears_type 	swap_zeroois_def_tmpl.ois_arr_type()			{ return ois_arr_type_;}
	number 				swap_zeroois_def_tmpl.ois_arr_days()			{ return ois_arr_days_;}
	logical				swap_zeroois_def_tmpl.ois_endog_df()			{ return ois_endog_df_;}
	logical				swap_zeroois_def_tmpl.ois_approx_calc()			{ return ois_approx_calc_;}
	string 				swap_zeroois_def_tmpl.spot_settle_code()		{ return spot_code_;} 
	string 				swap_zeroois_def_tmpl.fwd_start_code()			{ return fwd_start_code_;} 
	string 				swap_zeroois_def_tmpl.maturity_code()			{ return maturity_code_;}  					
	ir_index_tenor_tmpl	swap_zeroois_def_tmpl.on_index()				{ return on_index_;}
	logical  			swap_zeroois_def_tmpl.primary_quote_is_leg1()	{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),zero_curr_,ois_curr_,0,365);}
	mtm_code 			swap_zeroois_def_tmpl.mtm() 						{ return mtm_code.NONE;}
	logical 			swap_zeroois_def_tmpl.is_mtm(out logical leg1)		{ return false; /*..is_mtm(is_cross_curr(), mtm_, leg1);*/}
	
	ir_index swap_zeroois_def_tmpl.ois_ir_index()
	{
		string t = on_index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR") && !equal_casei (t, "RFR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	on_index_.name(),t,
								on_index_.tenor(),
								on_index_.value_days(),
								on_index_.fixing_days(),
								on_index_.day_count_method(),
								on_index_.bd_convention(),
								on_index_.eom(),
								on_index_.cal(),//if libor: this is uk
								on_index_.cal_name(),
								on_index_.currency(),
								on_index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								on_index_.suppl_cal_name(),
								on_index_.spot_days(),
								on_index_.spread());
	}
	
	//-------------------------------------------------------
	// class swap_oisois_def_tmpl
	//-------------------------------------------------------
	public class swap_oisois_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_oisois_def_tmpl(string	,
								 string,
								 string option(nullable),
								 string option(nullable) ,
								 integer,
								 integer ,
								 mtm_code option(nullable),
								 integer,
								..day_count_method 		ois1_dc,
								string 					ois1_curr,
								..calendar				ois1_cal,
								..bd_convention 		ois1_bd,												
								logical 				ois1_eom, 												
								rfr_arrears_type option(nullable) 		ois1_arr_type,
								number option(nullable)					ois1_arr_days,
								flt_comp_avg_type option(nullable)		ois1_comp_avg,
								flt_sprd_comp_method option(nullable)	ois1_spread_comp_method,
								flt_avg_method option(nullable)			ois1_avg_method,
								logical 								ois1_endog_df,
								logical									ois1_approx_calc,
								ir_index_tenor_tmpl						on_index1,
								integer 				ois2_cpn_freq,	
								..day_count_method 		ois2_dc,
								string 					ois2_curr,
								..calendar				ois2_cal,
								..bd_convention 		ois2_bd,												
								logical 				ois2_eom, 												
								rfr_arrears_type option(nullable) 		ois2_arr_type,
								number option(nullable)					ois2_arr_days,
								flt_comp_avg_type option(nullable)		ois2_comp_avg,
								flt_sprd_comp_method option(nullable)	ois2_spread_comp_method,
								flt_avg_method option(nullable)			ois2_avg_method,
								logical 								ois2_endog_df,
								logical									ois2_approx_calc,
								ir_index_tenor_tmpl						on_index2,
								notional_exchg_style option(nullable)	not_exchange);
			
			string				name();
			string 				spot_settle_code();
			string 				fwd_start_code();
			string 				maturity_code();
			integer				payment_delay();
			integer				payment_delay_prin();
			notional_exchg_style not_exchange();
			logical 			is_mtm(out logical);
			mtm_code 			mtm();
			logical				is_cross_curr();
			logical 			primary_quote_is_leg1() ;
			
			integer				ois1_cpn_freq();
			day_count_method 	ois1_day_count_method(); 		
			calendar			ois1_calendar();
			string				ois1_currency();		
			bd_convention		ois1_bd_convention();						
			logical				ois1_eom();
			flt_comp_avg_type 	ois1_comp_avg();
    		flt_sprd_comp_method ois1_spread_comp_method();
    		flt_avg_method 		ois1_avg_method();
			rfr_arrears_type 	ois1_arr_type();
			number 				ois1_arr_days();
			logical 			ois1_endog_df();
			logical				ois1_approx_calc();
			ir_index_tenor_tmpl	on_index1()	;
			ir_index			ois1_ir_index();
			interest_rule		ois1_ir_rule();
			
			integer				ois2_cpn_freq();
			day_count_method 	ois2_day_count_method(); 		
			calendar			ois2_calendar();
			string				ois2_currency();		
			bd_convention		ois2_bd_convention();						
			logical				ois2_eom();			
			flt_comp_avg_type 	ois2_comp_avg();
    		flt_sprd_comp_method ois2_spread_comp_method();
    		flt_avg_method 		ois2_avg_method();
			rfr_arrears_type 	ois2_arr_type();
			number 				ois2_arr_days();
			logical 			ois2_endog_df();
			logical				ois2_approx_calc();
			ir_index_tenor_tmpl	on_index2()	;			
			ir_index			ois2_ir_index();
			interest_rule		ois2_ir_rule();
		protected:
			string				name_;
			string				spot_code_;
			string				fwd_start_code_;
			string				maturity_code_;
			integer				payment_lag_;
			integer				payment_lag_prin_;
			notional_exchg_style not_exchange_;
			
			integer				ois1_cpn_freq_;
			day_count_method 	ois1_dc_;
			string				ois1_curr_;
			..calendar			ois1_cal_;
			bd_convention		ois1_bd_;
			logical				ois1_eom_;						
			rfr_arrears_type 	ois1_arr_type_;
			number 				ois1_arr_days_;
			logical 			ois1_endog_df_;
			logical				ois1_approx_calc_;
			flt_comp_avg_type 	ois1_comp_avg_;
    		flt_sprd_comp_method ois1_spread_comp_method_;
    		flt_avg_method 		ois1_avg_method_;
			ir_index_tenor_tmpl	on_index1_;

			integer				ois2_cpn_freq_;
			day_count_method 	ois2_dc_;
			string				ois2_curr_;
			..calendar			ois2_cal_;
			bd_convention		ois2_bd_;
			logical				ois2_eom_;	
			rfr_arrears_type 	ois2_arr_type_;
			number 				ois2_arr_days_;
			logical 			ois2_endog_df_;
			logical				ois2_approx_calc_;
			flt_comp_avg_type 	ois2_comp_avg_;
    		flt_sprd_comp_method ois2_spread_comp_method_;
    		flt_avg_method 		ois2_avg_method_;						
			ir_index_tenor_tmpl	on_index2_;
			mtm_code			mtm_;
			
	};

	swap_oisois_def_tmpl.swap_oisois_def_tmpl(	string					name,
												string 					spot_settle_code,
												string 	option(nullable) fwd_start_code,
												string 	option(nullable) maturity_code,
												integer 				payment_lag,
												integer 				payment_lag_prin,
												mtm_code option(nullable) mtm,
												integer 				ois1_cpn_freq,	
												..day_count_method 		ois1_dc,
												string 					ois1_curr,
												..calendar				ois1_cal,
												..bd_convention 		ois1_bd,												
												logical 				ois1_eom, 												
												rfr_arrears_type option(nullable) 		ois1_arr_type,
												number option(nullable)					ois1_arr_days,
												flt_comp_avg_type option(nullable)		ois1_comp_avg,
				    							flt_sprd_comp_method option(nullable)	ois1_spread_comp_method,
				    							flt_avg_method option(nullable)			ois1_avg_method,
												logical 								ois1_endog_df,
												logical									ois1_approx_calc,
												ir_index_tenor_tmpl						on_index1,

												integer 				ois2_cpn_freq,	
												..day_count_method 		ois2_dc,
												string 					ois2_curr,
												..calendar				ois2_cal,
												..bd_convention 		ois2_bd,												
												logical 				ois2_eom, 												
												rfr_arrears_type option(nullable) 		ois2_arr_type,
												number option(nullable)					ois2_arr_days,
												flt_comp_avg_type option(nullable)		ois2_comp_avg,
				    							flt_sprd_comp_method option(nullable)	ois2_spread_comp_method,
				    							flt_avg_method option(nullable)			ois2_avg_method,
												logical 								ois2_endog_df,
												logical									ois2_approx_calc,
												ir_index_tenor_tmpl						on_index2,												
												notional_exchg_style option(nullable)	not_exchange)
												: 	instrument_def_tmpl(..instr_type.SWAP_OISOIS),name_(name),spot_code_(spot_settle_code),fwd_start_code_(fwd_start_code),
													maturity_code_(maturity_code),
													payment_lag_(payment_lag),payment_lag_prin_(payment_lag_prin),not_exchange_(not_exchange),

													ois1_cpn_freq_(ois1_cpn_freq),
													ois1_dc_(ois1_dc),ois1_curr_(ois1_curr),ois1_cal_(ois1_cal),
													ois1_bd_(ois1_bd),ois1_eom_(ois1_eom),
													ois1_arr_type_(ois1_arr_type),ois1_arr_days_(ois1_arr_days),ois1_endog_df_(ois1_endog_df),
													ois1_approx_calc_(ois1_approx_calc), ois1_comp_avg_(ois1_comp_avg),
													ois1_spread_comp_method_(ois1_spread_comp_method),ois1_avg_method_(ois1_avg_method),
													on_index1_(on_index1),

													ois2_cpn_freq_(ois2_cpn_freq),
													ois2_dc_(ois2_dc),ois2_curr_(ois2_curr),ois2_cal_(ois2_cal),
													ois2_bd_(ois2_bd),ois2_eom_(ois2_eom),
													ois2_arr_type_(ois2_arr_type),ois2_arr_days_(ois2_arr_days),ois2_endog_df_(ois2_endog_df),																										
													ois2_approx_calc_(ois2_approx_calc), ois2_comp_avg_(ois2_comp_avg),
													ois2_spread_comp_method_(ois2_spread_comp_method),ois2_avg_method_(ois2_avg_method),
													on_index2_(on_index2), mtm_(mtm) {}

	notional_exchg_style swap_oisois_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) {
			if( !is_cross_curr()) 
				return NE_NONE ;
			else 
				return NE_BOTH;
		}
		else
			return not_exchange_;
		
	}
	
	logical 			swap_oisois_def_tmpl.primary_quote_is_leg1()		{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),ois1_curr_,ois2_curr_,365,365);}
	string				swap_oisois_def_tmpl.name() 						{ return name_;}
	integer				swap_oisois_def_tmpl.payment_delay()				{ return payment_lag_;}
	integer				swap_oisois_def_tmpl.payment_delay_prin()			{ return payment_lag_prin_;} 
	string 				swap_oisois_def_tmpl.spot_settle_code()				{ return spot_code_;} 
	string 				swap_oisois_def_tmpl.fwd_start_code()				{ return fwd_start_code_;}  
	string 				swap_oisois_def_tmpl.maturity_code()				{ return maturity_code_;}
	logical				swap_oisois_def_tmpl.is_cross_curr()				{ return !equal_casei(ois1_curr_, ois2_curr_);}

	string				swap_oisois_def_tmpl.ois1_currency()				{ return ois1_curr_;}			
	integer				swap_oisois_def_tmpl.ois1_cpn_freq()				{ return ois1_cpn_freq_;} 					
	calendar			swap_oisois_def_tmpl.ois1_calendar()				{ return null(ois1_cal_) ? ..calendar() : ois1_cal_;}
	bd_convention 		swap_oisois_def_tmpl.ois1_bd_convention()			{ return ois1_bd_;}
	day_count_method 	swap_oisois_def_tmpl.ois1_day_count_method()		{ return ois1_dc_;} 
	logical				swap_oisois_def_tmpl.ois1_eom()						{ return ois1_eom_;}

	
	flt_comp_avg_type 	swap_oisois_def_tmpl.ois1_comp_avg()				{ return ois1_comp_avg_;}
    flt_sprd_comp_method swap_oisois_def_tmpl.ois1_spread_comp_method()		{ return ois1_spread_comp_method_;}
    flt_avg_method 		swap_oisois_def_tmpl.ois1_avg_method()				{ return ois1_avg_method_;}
	rfr_arrears_type 	swap_oisois_def_tmpl.ois1_arr_type()				{ return ois1_arr_type_;}
	number 				swap_oisois_def_tmpl.ois1_arr_days()				{ return ois1_arr_days_;}
	logical 			swap_oisois_def_tmpl.ois1_endog_df()				{ return ois1_endog_df_;}
	logical				swap_oisois_def_tmpl.ois1_approx_calc()				{ return ois1_approx_calc_;}
	ir_index_tenor_tmpl	swap_oisois_def_tmpl.on_index1()					{ return on_index1_;}
	interest_rule		swap_oisois_def_tmpl.ois1_ir_rule()					{ return interest_rule.IR_ADJ;}

	string				swap_oisois_def_tmpl.ois2_currency()				{ return ois2_curr_;}			
	integer				swap_oisois_def_tmpl.ois2_cpn_freq()				{ return ois2_cpn_freq_;} 					
	calendar			swap_oisois_def_tmpl.ois2_calendar()				{ return null(ois2_cal_) ? ..calendar() : ois2_cal_;}
	bd_convention 		swap_oisois_def_tmpl.ois2_bd_convention()			{ return ois2_bd_;}
	day_count_method 	swap_oisois_def_tmpl.ois2_day_count_method()		{ return ois2_dc_;} 
	logical				swap_oisois_def_tmpl.ois2_eom()						{ return ois2_eom_;}
	
	flt_comp_avg_type 	swap_oisois_def_tmpl.ois2_comp_avg()				{ return ois2_comp_avg_;}
    flt_sprd_comp_method swap_oisois_def_tmpl.ois2_spread_comp_method()		{ return ois2_spread_comp_method_;}
    flt_avg_method 		swap_oisois_def_tmpl.ois2_avg_method()				{ return ois2_avg_method_;}
	rfr_arrears_type 	swap_oisois_def_tmpl.ois2_arr_type()				{ return ois2_arr_type_;}
	number 				swap_oisois_def_tmpl.ois2_arr_days()				{ return ois2_arr_days_;}
	logical 			swap_oisois_def_tmpl.ois2_endog_df()				{ return ois2_endog_df_;}
	logical				swap_oisois_def_tmpl.ois2_approx_calc()				{ return ois2_approx_calc_;}
	ir_index_tenor_tmpl	swap_oisois_def_tmpl.on_index2()					{ return on_index2_;}
	interest_rule		swap_oisois_def_tmpl.ois2_ir_rule()					{ return interest_rule.IR_ADJ;}
	mtm_code 			swap_oisois_def_tmpl.mtm()							{ return mtm_;}	
	logical swap_oisois_def_tmpl.is_mtm(out logical leg1) 					{ return ..is_mtm(is_cross_curr(), mtm_, leg1);}	
	
	ir_index swap_oisois_def_tmpl.ois1_ir_index()
	{
		string t = on_index1_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR") && !equal_casei (t, "RFR"))
			QL_FAIL("index1 type not applicable", E_INIT);
		
		return create_ir_index(	on_index1_.name(),t,
								on_index1_.tenor(),
								on_index1_.value_days(),
								on_index1_.fixing_days(),
								on_index1_.day_count_method(),
								on_index1_.bd_convention(),
								on_index1_.eom(),
								on_index1_.cal(),//if libor: this is uk
								on_index1_.cal_name(),
								on_index1_.currency(),
								on_index1_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								on_index1_.suppl_cal_name(),
								on_index1_.spot_days(),
								on_index1_.spread());
	}

	
	ir_index swap_oisois_def_tmpl.ois2_ir_index()
	{
		string t = on_index2_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR") && !equal_casei (t, "RFR"))
			QL_FAIL("index2 type not applicable", E_INIT);
		
		return create_ir_index(	on_index2_.name(),t,
								on_index2_.tenor(),
								on_index2_.value_days(),
								on_index2_.fixing_days(),
								on_index2_.day_count_method(),
								on_index2_.bd_convention(),
								on_index2_.eom(),
								on_index2_.cal(),//if libor: this is uk
								on_index2_.cal_name(),
								on_index2_.currency(),
								on_index2_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								on_index2_.suppl_cal_name(),
								on_index2_.spot_days(),
								on_index2_.spread());
	}

	
	//-------------------------------------------------------
	// class riba_def_tmpl
	//-------------------------------------------------------
	public class riba_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			riba_def_tmpl(string,date option(nullable),string option(nullable) ,date option(nullable) , 
						  string option(nullable) ,logical option(nullable),day_count_method, ..calendar,string);
			
			string				name();
			date 				accr_start_date()	;  
		    string 				accr_start_code()	;
		    date 				accr_end_date()	;
		    string 				accr_end_code()	;
		    logical 			accr_end_code_off_trade();	
			day_count_method 	day_count_method(); 
			calendar			calendar();			
			string				currency();
			ir_index 			rb_ir_index();

		protected:
			string				name_;
			date  				accr_start_date_;  
	    	string  			accr_start_code_; 
	    	date  				accr_end_date_; 
	    	string  			accr_end_code_;
	    	logical  			accr_end_code_off_trade_;	
			day_count_method 	dc_; 						
			..calendar			cal_;
			string				curr_;				
			
	};

	riba_def_tmpl.riba_def_tmpl(string						name,
								date option(nullable) 		accr_start_date,  
						    	string option(nullable) 	accr_start_code, 
						    	date option(nullable) 		accr_end_date, 
						    	string option(nullable) 	accr_end_code,
						    	logical option(nullable) 	accr_end_code_off_trade,	
								..day_count_method 			dc, 												
								..calendar					cal,
								string						curr)
								: 	instrument_def_tmpl(..instr_type.RIBA),name_(name), dc_(dc),cal_(cal), curr_(curr)  {}

	string				riba_def_tmpl.name() 					{ return name_;}	
	day_count_method 	riba_def_tmpl.day_count_method()		{ return dc_;} 						
	calendar			riba_def_tmpl.calendar()				{ return cal_;}
	string				riba_def_tmpl.currency()				{ return curr_;}
	date 				riba_def_tmpl.accr_start_date()			{ return accr_start_date_;}  
    string 				riba_def_tmpl.accr_start_code()			{ return accr_start_code_;} 
    date 				riba_def_tmpl.accr_end_date()			{ return accr_end_date_;} 
    string 				riba_def_tmpl.accr_end_code()			{ return accr_end_code_;}
    logical 			riba_def_tmpl.accr_end_code_off_trade()	{ return accr_end_code_off_trade_;}
	
	ir_index riba_def_tmpl.rb_ir_index()
	{		
		return create_ir_index(	"RB_REPO","RB_REPO",
								null<string>,null<number>,null<number>, dc_,
								null<bd_convention>,
								false,
								cal_,
								null<string>,
								"SEK",
								null<..calendar>,	
								null<string>,null<number>,0);
	}
	//-------------------------------------------------------
	// class fx_spot_def_tmpl
	//-------------------------------------------------------
	public class fx_spot_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			fx_spot_def_tmpl(string,string,string, ..calendar,..calendar,..calendar option(nullable),
							string, integer ,integer ,integer, fx_spot_rule );
			
			string 						name();
			string 						base_ccy();
			string  					price_ccy();							
			calendar					base_ccy_calendar();
			calendar					price_ccy_calendar();
			calendar 					usd_calendar();
			string 						spot_settle_code();
			integer 					pip_decimals();
			integer 					scale();
			integer						quote_decimals();
			fx_spot_rule	fx_rule();

		protected:
			string 						name_;
			string 						base_ccy_;
			string  					price_ccy_;							
			..calendar					base_ccy_calendar_;
			..calendar					price_ccy_calendar_;
			..calendar					usd_calendar_;
			string 						spot_settle_code_;
			integer 					pip_decimals_;
			integer 					scale_;
			integer						quote_decimals_;
			fx_spot_rule				fx_rule_;
			
	};

	fx_spot_def_tmpl.fx_spot_def_tmpl(	string 						instr_def_name,
										string 						base_ccy,
										string  					price_ccy,							
										..calendar					base_ccy_calendar,
										..calendar					price_ccy_calendar,
										..calendar option(nullable)	usd_calendar,
										string 						spot_settle_code,
										integer 					pip_decimals,
										integer 					scale,
										integer						quote_decimals,
										fx_spot_rule	fx_rule)
									: 	instrument_def_tmpl(..instr_type.FX_SPOT),name_(instr_def_name), base_ccy_(base_ccy),price_ccy_(price_ccy), base_ccy_calendar_(base_ccy_calendar),
										price_ccy_calendar_(price_ccy_calendar), usd_calendar_(usd_calendar), 
										spot_settle_code_(spot_settle_code), pip_decimals_(pip_decimals), scale_(scale),
										quote_decimals_(quote_decimals),fx_rule_(fx_rule)  {}
	
	string 						fx_spot_def_tmpl.name()					{ return name_;}	
	string 						fx_spot_def_tmpl.base_ccy()				{ return base_ccy_;}	
	string  					fx_spot_def_tmpl.price_ccy()			{ return price_ccy_;}								
	calendar					fx_spot_def_tmpl.base_ccy_calendar()	{ return base_ccy_calendar_;}	
	calendar					fx_spot_def_tmpl.price_ccy_calendar()	{ return price_ccy_calendar_;}	
	calendar 					fx_spot_def_tmpl.usd_calendar()			{ return null(usd_calendar_) ? ..calendar() : usd_calendar_;}	
	string 						fx_spot_def_tmpl.spot_settle_code()		{ return spot_settle_code_;}	
	integer 					fx_spot_def_tmpl.pip_decimals()			{ return pip_decimals_;}	
	integer 					fx_spot_def_tmpl.scale()				{ return scale_;}	
	integer						fx_spot_def_tmpl.quote_decimals()		{ return quote_decimals_;}	
	fx_spot_rule				fx_spot_def_tmpl.fx_rule()				{ return fx_rule_;}	

	//-------------------------------------------------------
	// class fx_swap_def_tmpl
	//-------------------------------------------------------
	public class fx_swap_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			fx_swap_def_tmpl(string, fx_spot_def_tmpl, bd_convention ,logical ,string option(nullable),string option(nullable),integer,logical  );
			
			string 						name();
			bd_convention 				bd_convention();
			logical 					eom();
			string						settle_code();
			string						maturity_code();
			string 						base_ccy();
			string  					price_ccy();							
			calendar					base_ccy_calendar();
			calendar					price_ccy_calendar();
			calendar 					usd_calendar();
			string 						spot_settle_code();
			integer 					pip_decimals();
			integer 					scale();
			integer						quote_decimals();
			fx_spot_rule	fx_rule();
			logical						matched_prin();

		protected:
			string 						name_;
			fx_spot_def_tmpl			fx_spot_;
			bd_convention 				bd_;
			logical 					eom_;
			string						settle_code_;
			string						mat_code_;
			integer 					quote_decimals_;
			logical						matched_prin_;			
	};

	fx_swap_def_tmpl.fx_swap_def_tmpl(	string 						instr_def_name,
										fx_spot_def_tmpl			fx_spot,
										..bd_convention 			bd,
										logical 					eom,
										string	option(nullable)	settle_code,
										string	option(nullable) 	mat_code,
										integer 					quote_decimals,
										logical 					matched_prin)
									: 	instrument_def_tmpl(..instr_type.FX_SWAP),name_(instr_def_name), fx_spot_(fx_spot), bd_(bd),
										eom_(eom), settle_code_(settle_code), mat_code_(mat_code),
										quote_decimals_(quote_decimals), matched_prin_(matched_prin)  {}
	
	string 						fx_swap_def_tmpl.name()					{ return name_;}
	..bd_convention 			fx_swap_def_tmpl.bd_convention()		{ return bd_;}
	logical 					fx_swap_def_tmpl.eom()					{ return eom_;}	
	string 						fx_swap_def_tmpl.base_ccy()				{ return fx_spot_.base_ccy();}	
	string  					fx_swap_def_tmpl.price_ccy()			{ return fx_spot_.price_ccy();}								
	calendar					fx_swap_def_tmpl.base_ccy_calendar()	{ return fx_spot_.base_ccy_calendar();}	
	calendar					fx_swap_def_tmpl.price_ccy_calendar()	{ return fx_spot_.price_ccy_calendar();}	
	calendar 					fx_swap_def_tmpl.usd_calendar()			{ return fx_spot_.usd_calendar();}	
	string 						fx_swap_def_tmpl.spot_settle_code()		{ return fx_spot_.spot_settle_code();}	
	integer 					fx_swap_def_tmpl.pip_decimals()			{ return fx_spot_.pip_decimals();}	
	integer 					fx_swap_def_tmpl.scale()				{ return fx_spot_.scale();}	
	integer						fx_swap_def_tmpl.quote_decimals()		{ return quote_decimals_;}	
	fx_spot_rule				fx_swap_def_tmpl.fx_rule()				{ return fx_spot_.fx_rule();}	
	logical 					fx_swap_def_tmpl.matched_prin()			{ return matched_prin_;}
	string						fx_swap_def_tmpl.settle_code()			{ return null(settle_code_) ? spot_settle_code() : settle_code_;}
	string						fx_swap_def_tmpl.maturity_code()		{ return mat_code_;}
	//-------------------------------------------------------
	// class fx_fwd_def_tmpl
	//-------------------------------------------------------
	public class fx_fwd_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			fx_fwd_def_tmpl(string, string option(nullable), fx_spot_def_tmpl, bd_convention ,logical ,integer );
			
			string 						name();
			string						fwd_settle_code();
			bd_convention 				bd_convention();
			logical 					eom();
			string 						base_ccy();
			string  					price_ccy();							
			calendar					base_ccy_calendar();
			calendar					price_ccy_calendar();
			calendar 					usd_calendar();
			string 						spot_settle_code();
			integer 					pip_decimals();
			integer 					scale();
			integer						quote_decimals();
			fx_spot_rule	fx_rule();			

		protected:
			string 						name_;
			string						fwd_settle_code_;
			fx_spot_def_tmpl			fx_spot_;
			bd_convention 				bd_;
			logical 					eom_;
			integer 					quote_decimals_;					
	};

	fx_fwd_def_tmpl.fx_fwd_def_tmpl(	string 						instr_def_name,
										string option(nullable)		fwd_settle_code,
										fx_spot_def_tmpl			fx_spot,
										..bd_convention 			bd,
										logical 					eom,
										integer 					quote_decimals)
										: 	instrument_def_tmpl(..instr_type.FX_FORWARD),name_(instr_def_name),
											fwd_settle_code_(fwd_settle_code), fx_spot_(fx_spot), bd_(bd), eom_(eom),
											quote_decimals_(quote_decimals)  {}
	
	string 						fx_fwd_def_tmpl.name()					{ return name_;}
	string						fx_fwd_def_tmpl.fwd_settle_code()		{ return fwd_settle_code_;}
	..bd_convention 			fx_fwd_def_tmpl.bd_convention()			{ return bd_;}
	logical 					fx_fwd_def_tmpl.eom()					{ return eom_;}	
	string 						fx_fwd_def_tmpl.base_ccy()				{ return fx_spot_.base_ccy();}	
	string  					fx_fwd_def_tmpl.price_ccy()				{ return fx_spot_.price_ccy();}								
	calendar					fx_fwd_def_tmpl.base_ccy_calendar()		{ return fx_spot_.base_ccy_calendar();}	
	calendar					fx_fwd_def_tmpl.price_ccy_calendar()	{ return fx_spot_.price_ccy_calendar();}	
	calendar 					fx_fwd_def_tmpl.usd_calendar()			{ return fx_spot_.usd_calendar();}	
	string 						fx_fwd_def_tmpl.spot_settle_code()		{ return fx_spot_.spot_settle_code();}	
	integer 					fx_fwd_def_tmpl.pip_decimals()			{ return fx_spot_.pip_decimals();}	
	integer 					fx_fwd_def_tmpl.scale()					{ return fx_spot_.scale();}	
	integer						fx_fwd_def_tmpl.quote_decimals()		{ return quote_decimals_;}	
	fx_spot_rule	fx_fwd_def_tmpl.fx_rule()				{ return fx_spot_.fx_rule();}	
	

	//-------------------------------------------------------
	// class swap_rr_fixflt_def_tmpl
	//-------------------------------------------------------
	public class swap_rr_fixflt_def_tmpl: public  instrument_def_tmpl 
	{
	   	public:
			swap_rr_fixflt_def_tmpl(string,string ,string 	option(nullable),string option(nullable),integer,
									day_count_method,interest_rule,string,..calendar, bd_convention,logical,integer ,
									number option(nullable),day_count_method,interest_rule,string,..calendar,
									bd_convention,logical,flt_comp_avg_type option(nullable),
    								flt_sprd_comp_method option(nullable) 	,
    								flt_avg_method option(nullable) ,flt_stub_fwd_style	option(nullable)  flt_fwd_stub_rate_style,
									notional_exchg_style option(nullable),
									infl_index_method	,ir_index_tenor_tmpl);
			
			string				name();
			logical 			primary_quote_is_leg1() ;
			integer				fix_cpn_freq();
			integer				flt_cpn_freq();
			integer				flt_reset_freq();
			day_count_method 	fix_day_count_method(); 
			day_count_method 	flt_day_count_method();
 			interest_rule		fix_ir_rule();
			interest_rule		flt_ir_rule();
			calendar			flt_calendar();			
			calendar			fix_calendar();
			string				fix_currency();		
			string				flt_currency();
			bd_convention		flt_bd_convention();						
			logical				flt_eom();	
			bd_convention		fix_bd_convention();						
			logical				fix_eom();			
			string 				spot_settle_code();
			string 				fwd_start_code();
			string 				maturity_code(); 	
			flt_comp_avg_type 		flt_comp_avg();
    		flt_sprd_comp_method  	flt_spread_comp_method();
    		flt_avg_method 			flt_avg_method();	
			infl_index_method		index_method();						
			ir_index_tenor_tmpl	index()	;
			notional_exchg_style not_exchange();
			ir_index 			flt_ir_index();
			flt_stub_fwd_style	flt_fwd_stub_rate_style();
		protected:
			string				name_;
			integer				fix_cpn_freq_;
			day_count_method 	fix_dc_;
			interest_rule		fix_ir_rule_;
			string				fix_curr_;
			..calendar				fix_cal_;
			bd_convention		fix_bd_;
			logical				fix_eom_; 
			integer				flt_cpn_freq_;
			number 				flt_reset_freq_;
			day_count_method 	flt_dc_;
			interest_rule		flt_ir_rule_;
			string				flt_curr_;
			..calendar			flt_cal_;
			bd_convention		flt_bd_;
			logical				flt_eom_; 
			flt_comp_avg_type 	flt_comp_avg_;
    		flt_sprd_comp_method flt_spread_comp_method_;
    		flt_avg_method 		flt_avg_method_;
			flt_stub_fwd_style	flt_fwd_stub_rate_style_;
			string				spot_code_;
			string				fwd_start_code_;
			string				maturity_code_;
			infl_index_method		index_method_;				
			ir_index_tenor_tmpl	index_;
			notional_exchg_style not_exchange_;
	};

	swap_rr_fixflt_def_tmpl.swap_rr_fixflt_def_tmpl(string					name,
													string 					spot_settle_code,
													string 	option(nullable) fwd_start_code,
													string 	option(nullable) maturity_code,
													integer 				fix_cpn_freq,	
													..day_count_method 		fix_dc,
													interest_rule			fix_ir_rule,
													string 					fix_curr,
													..calendar				fix_cal,
													..bd_convention 		fix_bd,												
													logical 				fix_eom, 						       
													integer 				flt_cpn_freq,
													number option(nullable) flt_reset_freq,	  
													..day_count_method 		flt_dc,
													interest_rule			flt_ir_rule,
													string 					flt_curr,
													..calendar				flt_cal,	
													..bd_convention 		flt_bd,												
													logical 				flt_eom,
													..flt_comp_avg_type option(nullable)	flt_comp_avg,
		    										..flt_sprd_comp_method option(nullable) flt_spread_comp_method,
		    										..flt_avg_method option(nullable) 		flt_avg_method,
													flt_stub_fwd_style	option(nullable)  flt_fwd_stub_rate_style,
													notional_exchg_style option(nullable)	not_exchange, 
													infl_index_method		index_method,	
													ir_index_tenor_tmpl	index)
								: 	instrument_def_tmpl(..instr_type.IL_SWAP_REAL_FIXFLT),name_(name),fix_cpn_freq_(fix_cpn_freq),
									fix_dc_(fix_dc),fix_ir_rule_(fix_ir_rule),fix_curr_(fix_curr),fix_cal_(fix_cal),
									fix_bd_(fix_bd),fix_eom_(fix_eom),flt_cpn_freq_(flt_cpn_freq),flt_reset_freq_(flt_reset_freq),
									flt_dc_(flt_dc),flt_ir_rule_(flt_ir_rule),flt_curr_(flt_curr),flt_cal_(flt_cal), 
									flt_bd_(flt_bd),flt_eom_(flt_eom),flt_comp_avg_(flt_comp_avg),flt_spread_comp_method_(flt_spread_comp_method),
									flt_avg_method_(flt_avg_method),flt_fwd_stub_rate_style_(flt_fwd_stub_rate_style),
									spot_code_(spot_settle_code), fwd_start_code_(fwd_start_code),maturity_code_(maturity_code),index_method_(index_method),
									index_(index),not_exchange_(not_exchange) {}

	string				swap_rr_fixflt_def_tmpl.name() 					{ return name_;}
	day_count_method 	swap_rr_fixflt_def_tmpl.fix_day_count_method()	{ return fix_dc_;} 						
	day_count_method 	swap_rr_fixflt_def_tmpl.flt_day_count_method()	{ return flt_dc_;}
	interest_rule		swap_rr_fixflt_def_tmpl.fix_ir_rule()			{ return fix_ir_rule_;}
	interest_rule		swap_rr_fixflt_def_tmpl.flt_ir_rule()			{ return flt_ir_rule_;}
	integer				swap_rr_fixflt_def_tmpl.fix_cpn_freq()			{ return fix_cpn_freq_;} 					
	integer				swap_rr_fixflt_def_tmpl.flt_cpn_freq()			{ return flt_cpn_freq_;} 
	integer 			swap_rr_fixflt_def_tmpl.flt_reset_freq()		{ return null(flt_reset_freq_) ? flt_cpn_freq_ : integer(flt_reset_freq_);} 
	string				swap_rr_fixflt_def_tmpl.fix_currency()			{ return fix_curr_;}
	string				swap_rr_fixflt_def_tmpl.flt_currency()			{ return flt_curr_;}
	calendar			swap_rr_fixflt_def_tmpl.fix_calendar()			{ return null(fix_cal_) ? ..calendar() : fix_cal_;}
	calendar			swap_rr_fixflt_def_tmpl.flt_calendar()			{ return null(flt_cal_) ? ..calendar() : flt_cal_;}
	bd_convention 		swap_rr_fixflt_def_tmpl.flt_bd_convention()		{ return flt_bd_;} 
	logical				swap_rr_fixflt_def_tmpl.flt_eom()				{ return flt_eom_;} 
	bd_convention 		swap_rr_fixflt_def_tmpl.fix_bd_convention()		{ return fix_bd_;} 
	logical				swap_rr_fixflt_def_tmpl.fix_eom()				{ return fix_eom_;} 
	string 				swap_rr_fixflt_def_tmpl.spot_settle_code()		{ return spot_code_;}
	string 				swap_rr_fixflt_def_tmpl.fwd_start_code()		{ return fwd_start_code_;}
	string 				swap_rr_fixflt_def_tmpl.maturity_code()			{ return maturity_code_;} 
	infl_index_method	swap_rr_fixflt_def_tmpl.index_method()			{ return index_method_;}	 
	flt_comp_avg_type 	swap_rr_fixflt_def_tmpl.flt_comp_avg()			{ return flt_comp_avg_;} 
    flt_sprd_comp_method swap_rr_fixflt_def_tmpl.flt_spread_comp_method() { return flt_spread_comp_method_;} 
    flt_avg_method 		swap_rr_fixflt_def_tmpl.flt_avg_method()		{ return flt_avg_method_;} 
					
	ir_index_tenor_tmpl	swap_rr_fixflt_def_tmpl.index()					{ return index_;}
	logical 			swap_rr_fixflt_def_tmpl.primary_quote_is_leg1()	{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),fix_curr_,flt_curr_,fix_cpn_freq_,flt_reset_freq());}
	flt_stub_fwd_style	swap_rr_fixflt_def_tmpl.flt_fwd_stub_rate_style() { return flt_fwd_stub_rate_style_;}
	
	notional_exchg_style swap_rr_fixflt_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) {
			if( equal_casei(fix_curr_, flt_curr_)) 
				return NE_END ;
			else 
				return NE_BOTH;
		}
		else {
			return not_exchange_;
		}
	}

	ir_index swap_rr_fixflt_def_tmpl.flt_ir_index()
	{
		string t = index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	index_.name(),t,
								index_.tenor(),
								index_.value_days(),
								index_.fixing_days(),
								index_.day_count_method(),
								index_.bd_convention(),
								index_.eom(),
								index_.cal(),//if libor: this is uk
								index_.cal_name(),
								index_.currency(),
								index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								index_.suppl_cal_name(),
								index_.spot_days(),0);
	}
	
	
	//-------------------------------------------------------
	// class swap_rr_fixois_def_tmpl
	//-------------------------------------------------------
	public class swap_rr_fixois_def_tmpl: public  instrument_def_tmpl 
	{
	   	public:
			swap_rr_fixois_def_tmpl(string,string ,string 	option(nullable),string option(nullable),integer,
									day_count_method,interest_rule,string,..calendar, bd_convention,logical,integer,
									day_count_method,string,..calendar,
									bd_convention,logical,
									rfr_arrears_type option(nullable)  , number option(nullable)	,
									flt_comp_avg_type option(nullable),
    								flt_sprd_comp_method option(nullable) 	,
    								flt_avg_method option(nullable) ,logical,logical,integer,integer,notional_exchg_style option(nullable),
									infl_index_method	,ir_index_tenor_tmpl);
			
			string				name();
			notional_exchg_style not_exchange();
			logical 			primary_quote_is_leg1();
			
			infl_index_method	index_method();
			integer				payment_lag();
			integer				payment_lag_prin();
			
			integer				fix_cpn_freq();
			day_count_method 	fix_day_count_method();
			interest_rule		fix_ir_rule();
			calendar			fix_calendar();
			string				fix_currency();	
			bd_convention		fix_bd_convention();						
			logical				fix_eom();
			
			integer				ois_cpn_freq();
			day_count_method 	ois_day_count_method(); 			
			calendar			ois_calendar();							
			string				ois_currency();
			bd_convention		ois_bd_convention();						
			logical				ois_eom();	
						
			string 				spot_settle_code();
			string 				fwd_start_code();
			string 				maturity_code(); 	
			flt_comp_avg_type 	ois_comp_avg();
    		flt_sprd_comp_method ois_spread_comp_method();
    		flt_avg_method 		ois_avg_method();	
			rfr_arrears_type 	ois_arr_type();
			number 				ois_arr_days();
			logical				ois_endog_df();
			logical				ois_approx_calc();
			ir_index_tenor_tmpl	on_index()	;
			ir_index			ois_ir_index();
	
		protected:
			string				name_;
			integer				fix_cpn_freq_;
			day_count_method 	fix_dc_;
			interest_rule		fix_ir_rule_;
			string				fix_curr_;
			..calendar			fix_cal_;
			bd_convention		fix_bd_;
			logical				fix_eom_;
			
			integer				ois_cpn_freq_;
			day_count_method 	ois_dc_;
			string				ois_curr_;
			..calendar			ois_cal_;
			bd_convention		ois_bd_;
			logical				ois_eom_;

			rfr_arrears_type 	ois_arr_type_;
			number 				ois_arr_days_;		
			flt_comp_avg_type 	ois_comp_avg_;
    		flt_sprd_comp_method ois_spread_comp_method_;
    		flt_avg_method 		ois_avg_method_;
			logical 			ois_endog_df_;
			logical				ois_approx_calc_;

			string				spot_code_;
			string				fwd_start_code_;
			string				maturity_code_;
			infl_index_method	index_method_;	
			ir_index_tenor_tmpl	on_index_;
			integer				payment_lag_;
			integer				payment_lag_prin_;
								
			notional_exchg_style not_exchange_;
	};

	swap_rr_fixois_def_tmpl.swap_rr_fixois_def_tmpl(string					name,
													string 					spot_settle_code,
													string 	option(nullable) fwd_start_code,
													string 	option(nullable) maturity_code,
													integer 				fix_cpn_freq,	
													..day_count_method 		fix_dc,
													interest_rule			fix_ir_rule,
													string 					fix_curr,
													..calendar				fix_cal,
													..bd_convention 		fix_bd,												
													logical 				fix_eom, 						       
													integer 				ois_cpn_freq,
													..day_count_method 		ois_dc,
													string 					ois_curr,
													..calendar				ois_cal,	
													..bd_convention 		ois_bd,												
													logical 				ois_eom,
													rfr_arrears_type option(nullable) 		ois_arr_type,
													number option(nullable)					ois_arr_days,
													flt_comp_avg_type option(nullable)		ois_comp_avg,
													flt_sprd_comp_method option(nullable)	ois_spread_comp_method,
													flt_avg_method option(nullable)			ois_avg_method,
													logical 								ois_endog_df,
													logical									ois_approx_calc,
													integer 				payment_lag,
													integer 				payment_lag_prin,
													notional_exchg_style option(nullable)	not_exchange, 
													infl_index_method		index_method,	
													ir_index_tenor_tmpl		on_index)
								: 	instrument_def_tmpl(..instr_type.IL_SWAP_REAL_FIXOIS),name_(name),
									fix_cpn_freq_(fix_cpn_freq), fix_dc_(fix_dc),fix_ir_rule_(fix_ir_rule),fix_curr_(fix_curr),fix_cal_(fix_cal),fix_bd_(fix_bd),fix_eom_(fix_eom),
									ois_cpn_freq_(ois_cpn_freq), ois_dc_(ois_dc),ois_curr_(ois_curr),ois_cal_(ois_cal), 
									ois_bd_(ois_bd),ois_eom_(ois_eom),ois_arr_type_(ois_arr_type),ois_arr_days_(ois_arr_days),
									ois_comp_avg_(ois_comp_avg),ois_spread_comp_method_(ois_spread_comp_method),ois_avg_method_(ois_avg_method),
									ois_endog_df_(ois_endog_df),ois_approx_calc_(ois_approx_calc),
									spot_code_(spot_settle_code), fwd_start_code_(fwd_start_code),maturity_code_(maturity_code),index_method_(index_method),
									on_index_(on_index),payment_lag_(payment_lag),payment_lag_prin_(payment_lag_prin),
									not_exchange_(not_exchange) {}

	string				swap_rr_fixois_def_tmpl.name() 					{ return name_;}
	day_count_method 	swap_rr_fixois_def_tmpl.fix_day_count_method()	{ return fix_dc_;}
	interest_rule		swap_rr_fixois_def_tmpl.fix_ir_rule()			{ return fix_ir_rule_;}
	day_count_method 	swap_rr_fixois_def_tmpl.ois_day_count_method()	{ return ois_dc_;} 
	integer				swap_rr_fixois_def_tmpl.fix_cpn_freq()			{ return fix_cpn_freq_;} 					
	integer				swap_rr_fixois_def_tmpl.ois_cpn_freq()			{ return ois_cpn_freq_;} 
	string				swap_rr_fixois_def_tmpl.fix_currency()			{ return fix_curr_;}
	string				swap_rr_fixois_def_tmpl.ois_currency()			{ return ois_curr_;}
	calendar			swap_rr_fixois_def_tmpl.fix_calendar()			{ return null(fix_cal_) ? ..calendar() : fix_cal_;}
	calendar			swap_rr_fixois_def_tmpl.ois_calendar()			{ return null(ois_cal_) ? ..calendar() : ois_cal_;}
	bd_convention 		swap_rr_fixois_def_tmpl.ois_bd_convention()		{ return ois_bd_;} 
	logical				swap_rr_fixois_def_tmpl.ois_eom()				{ return ois_eom_;} 
	bd_convention 		swap_rr_fixois_def_tmpl.fix_bd_convention()		{ return fix_bd_;} 
	logical				swap_rr_fixois_def_tmpl.fix_eom()				{ return fix_eom_;} 
	string 				swap_rr_fixois_def_tmpl.spot_settle_code()		{ return spot_code_;}
	string 				swap_rr_fixois_def_tmpl.fwd_start_code()		{ return fwd_start_code_;}
	string 				swap_rr_fixois_def_tmpl.maturity_code()			{ return maturity_code_;} 
	infl_index_method	swap_rr_fixois_def_tmpl.index_method()			{ return index_method_;}	 
	flt_comp_avg_type 	swap_rr_fixois_def_tmpl.ois_comp_avg()			{ return ois_comp_avg_;}
    flt_sprd_comp_method swap_rr_fixois_def_tmpl.ois_spread_comp_method(){ return ois_spread_comp_method_;}
    flt_avg_method 		swap_rr_fixois_def_tmpl.ois_avg_method()		{ return ois_avg_method_;}
	rfr_arrears_type 	swap_rr_fixois_def_tmpl.ois_arr_type()			{ return ois_arr_type_;}
	number 				swap_rr_fixois_def_tmpl.ois_arr_days()			{ return ois_arr_days_;} 
	logical				swap_rr_fixois_def_tmpl.ois_endog_df()			{ return ois_endog_df_;}
	logical				swap_rr_fixois_def_tmpl.ois_approx_calc()		{ return ois_approx_calc_;} 
	ir_index_tenor_tmpl	swap_rr_fixois_def_tmpl.on_index()				{ return on_index_;}
	integer				swap_rr_fixois_def_tmpl.payment_lag()			{ return payment_lag_;}
	integer				swap_rr_fixois_def_tmpl.payment_lag_prin()		{ return payment_lag_prin_;}
	logical 			swap_rr_fixois_def_tmpl.primary_quote_is_leg1()	{ return leg1_is_primary_quote(CORE_INT.conv_rev_instr_type(instr_type()),fix_curr_,ois_curr_,fix_cpn_freq_,365);}
	
	notional_exchg_style swap_rr_fixois_def_tmpl.not_exchange()			
	{ 
		if(null(not_exchange_)) {
			if( equal_casei(fix_curr_, ois_curr_)) 
				return NE_END ;
			else 
				return NE_BOTH;
		}
		else
			return not_exchange_;
		
	}

	ir_index swap_rr_fixois_def_tmpl.ois_ir_index()
	{
		string t = on_index_.type();
		if(!equal_casei(t, "IBOR") && !equal_casei (t, "LIBOR") && !equal_casei (t, "RFR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	on_index_.name(),t,
								on_index_.tenor(),
								on_index_.value_days(),
								on_index_.fixing_days(),
								on_index_.day_count_method(),
								on_index_.bd_convention(),
								on_index_.eom(),
								on_index_.cal(),//if libor: this is uk
								on_index_.cal_name(),
								on_index_.currency(),
								on_index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								on_index_.suppl_cal_name(),
								on_index_.spot_days(),
								on_index_.spread());
	}

	//-------------------------------------------------------
	// class onfut_def_tmpl  
	//-------------------------------------------------------
	public class onfut_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			onfut_def_tmpl(	string,date option(nullable),string	option(nullable),string,day_count_method, ..calendar option(nullable),
							string, onfut_calc_method, fut_tick_value_style, quote_style,number,number option(nullable), string,
							ir_index_tenor_tmpl);
			
			string					name();
			date					accr_start_date();
			string					accr_start_code();
			string					accr_end_code();
			logical					accr_end_code_off_trade();		
			day_count_method 		day_count_method(); 									
			calendar				calendar();			
			string					currency();	
			onfut_calc_method 		style();
			fut_tick_value_style 	tick_value_style();	
			logical					suppl_cal_adjust();
			quote_style 			quote_style(); 
			ir_index_tenor_tmpl		index();
			number					contract_val();
			number					tick_size();
			string					final_settle_code();
			ir_index 				ir_index();
				
		protected:		
			string					name_;
			date					accr_start_date_;
			string					accr_start_code_;
			string					accr_end_code_;	
			day_count_method 		dc_; 						
			..calendar				cal_;
			string					curr_;
			onfut_calc_method 		style_;
			fut_tick_value_style 	tick_style_;
			quote_style				qs_;
			number					contract_val_;
			number					tick_size_;
			string					final_settle_code_;			
			ir_index_tenor_tmpl		index_;			
			
	};

	onfut_def_tmpl.onfut_def_tmpl(	string							name,
									date	option(nullable)		accr_start_date,
									string	option(nullable)		accr_start_code,
									string							accr_end_code,	
									..day_count_method 				dc, 
									..calendar option(nullable) 	cal,
									string							curr,
									onfut_calc_method 				style,	
									fut_tick_value_style 			tick_style,							
									..quote_style					qs,
									number							contract_val,
									number	option(nullable) 		tick_size,
									string							final_settle_code,								
									ir_index_tenor_tmpl				index)
									: 	instrument_def_tmpl(..instr_type.OVERNIGHT_FUTURE),name_(name), accr_start_date_(accr_start_date),accr_start_code_(accr_start_code),accr_end_code_(accr_end_code), 
										dc_(dc),cal_(cal), curr_(curr),style_(style),
										tick_style_(tick_style),qs_(qs), contract_val_(contract_val), 
										 tick_size_(tick_size), final_settle_code_(final_settle_code),
										index_(index) {}

	ir_index onfut_def_tmpl.ir_index()
	{
		string t = index_.type();
		if(!equal(t, "IBOR") && !equal (t, "LIBOR") && !equal (t, "RFR"))
			QL_FAIL("index type not applicable", E_INIT);
		
		return create_ir_index(	index_.name(),t,
								"1D",
								index_.value_days(),
								index_.fixing_days(),
								index_.day_count_method(),
								BD_FOLLOWING,
								false,
								index_.cal(),//if libor: this is uk
								index_.cal_name(),
								index_.currency(),
								index_.suppl_cal(),	//if libor: this is primary(ex.target), if ibor: this is null
								index_.suppl_cal_name(),
								index_.spot_days(),
								index_.spread());
	}
	
	string				onfut_def_tmpl.name() 					{ return name_;}
	date				onfut_def_tmpl.accr_start_date() 		{ return accr_start_date_;}
	string				onfut_def_tmpl.accr_start_code() 		{ return accr_start_code_;}
	string				onfut_def_tmpl.accr_end_code()			{ return accr_end_code_;}
	logical				onfut_def_tmpl.accr_end_code_off_trade(){ return false;}	
	day_count_method 	onfut_def_tmpl.day_count_method()		{ return dc_;} 						
	calendar			onfut_def_tmpl.calendar()				{ return null(cal_) ? ..calendar() : cal_;}
	string				onfut_def_tmpl.currency()				{ return curr_;}
	onfut_calc_method 	onfut_def_tmpl.style()					{ return style_;}
	fut_tick_value_style onfut_def_tmpl.tick_value_style() 		{ return tick_style_;}
	logical				onfut_def_tmpl.suppl_cal_adjust()		{ return index_.suppl_cal_adjust();}
	quote_style 		onfut_def_tmpl.quote_style()			{ return qs_;}
	number				onfut_def_tmpl.contract_val()			{ return contract_val_;}
	number				onfut_def_tmpl.tick_size()				{ return tick_size_;}
	string				onfut_def_tmpl.final_settle_code()		{ return final_settle_code_;}
	ir_index_tenor_tmpl	onfut_def_tmpl.index()					{ return index_;}

	//-------------------------------------------------------
	// vanilla_option_fx_def_tmpl
	// expiry_cal should be a "global" calendar i.e. if at least one "marketplace" is open it is a good day
	//-------------------------------------------------------
	class vanilla_option_fx_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			vanilla_option_fx_def_tmpl(	string			currency_pair,
										exercise_type 	exercise_type,
										opt_type 		opt_type,
										string 			cut_off,
										..bd_convention bd, 	 								
										logical			eom,									
										string	option(nullable) expiry_code,
										..calendar		expiry_cal,
										string	option(nullable) strike_code,
										string	option(nullable) face_ccy,
										string	option(nullable) prem_ccy,
										logical			prem_at_spot,
										fx_delta_type option(nullable)	delta_type,
										fx_atm_type	option(nullable) atm_type,
										delivery_type 	del_type,
										logical	use_longterm_dflts = false);
			
			string			currency_pair();
			exercise_type 	exercise_type();
			opt_type 		opt_type();
			string 			cut_off();
			..bd_convention bd(); 	 								
			logical			eom();									
			string			expiry_code();
			..calendar		expiry_cal();
			string			strike_code();
			string			face_ccy();
			string			prem_ccy();
			logical			prem_at_spot();
			fx_delta_type	delta_type();
			fx_atm_type		atm_type();
			delivery_type 	del_type();
			logical			use_longterm_dflts();
			
		protected:
			string			currency_pair_;//format: FFFDDD, FFF=foreign ccy (=base ccy), DDD=domestic ccy (=price ccy)
			exercise_type 	exercise_type_;
			opt_type 		opt_type_;
			string 			cut_off_;
			..bd_convention bd_; 	 								
			logical			eom_;									
			string			expiry_code_;
			..calendar		expiry_cal_;
			string			strike_code_;
			string			face_ccy_;
			string			prem_ccy_;
			logical			prem_at_spot_;
			fx_delta_type	delta_type_;
			fx_atm_type		atm_type_;
			delivery_type 	del_type_;
			logical			use_longterm_dflts_;
			
	};

	vanilla_option_fx_def_tmpl.vanilla_option_fx_def_tmpl(string		currency_pair,//format: FFFDDD, FFF=foreign ccy (=base ccy), DDD=domestic ccy (=price ccy)
												..exercise_type exercise_type,
												..opt_type 		opt_type,
												string 			cut_off,
												..bd_convention bd, 	 								
												logical			eom,									
												string	option(nullable) expiry_code,
												..calendar		expiry_cal,
												string	option(nullable) strike_code,
												string	option(nullable) face_ccy,
												string	option(nullable) prem_ccy,
												logical			prem_at_spot,
												fx_delta_type option(nullable)	delta_type,
												fx_atm_type	option(nullable) atm_type,
												delivery_type 	del_type,
												logical			use_longterm_dflts )
												:   instrument_def_tmpl(null<..instr_type>),currency_pair_(currency_pair),
													exercise_type_(exercise_type),
													opt_type_(opt_type),
													cut_off_(cut_off),
													bd_(bd), 	 								
													eom_(eom),									
													expiry_code_(expiry_code),
													expiry_cal_(expiry_cal),
													strike_code_(strike_code),
													face_ccy_(face_ccy),
													prem_ccy_(prem_ccy),
													prem_at_spot_(prem_at_spot),
													delta_type_(delta_type),
													atm_type_(atm_type),
													del_type_(del_type),
													use_longterm_dflts_(use_longterm_dflts)
	{}

	string			vanilla_option_fx_def_tmpl.currency_pair()	{ return currency_pair_;}
	exercise_type 	vanilla_option_fx_def_tmpl.exercise_type()	{ return exercise_type_;}
	opt_type 		vanilla_option_fx_def_tmpl.opt_type()		{ return opt_type_;}
	string 			vanilla_option_fx_def_tmpl.cut_off()		{ return cut_off_;}
	..bd_convention vanilla_option_fx_def_tmpl.bd()				{ return bd_;} 	 								
	logical			vanilla_option_fx_def_tmpl.eom()			{ return eom_;}									
	string			vanilla_option_fx_def_tmpl.expiry_code()	{ return expiry_code_;}
	..calendar		vanilla_option_fx_def_tmpl.expiry_cal()		{ return expiry_cal_;}
	string			vanilla_option_fx_def_tmpl.strike_code()	{ return strike_code_;}
	string			vanilla_option_fx_def_tmpl.face_ccy()		{ return face_ccy_;}
	string			vanilla_option_fx_def_tmpl.prem_ccy()		{ return prem_ccy_;}
	logical			vanilla_option_fx_def_tmpl.prem_at_spot()	{ return prem_at_spot_;}
	fx_delta_type	vanilla_option_fx_def_tmpl.delta_type()		{ return delta_type_;}
	fx_atm_type		vanilla_option_fx_def_tmpl.atm_type()		{ return atm_type_;}
	delivery_type 	vanilla_option_fx_def_tmpl.del_type()		{ return del_type_;}
	logical			vanilla_option_fx_def_tmpl.use_longterm_dflts()	{ return use_longterm_dflts_;}


	//-------------------------------------------------------
	// class swap_il_fixyy_def_tmpl
	//-------------------------------------------------------
	public class swap_il_fixyy_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_il_fixyy_def_tmpl(	string,string,string option(nullable),integer,day_count_method,
									string,..calendar, bd_convention,logical,infl_index_method);
			
			string				name();
			string 				settle_code();	
			string 				maturity_code();									
			integer				fix_cpn_freq();
			bd_convention		fix_bd_convention();						
			logical				fix_eom();
			day_count_method 	fix_day_count_method(); 
			calendar			fix_calendar();
			string				fix_currency();
			infl_index_method	index_method();
			
		protected:
			string				name_;
			integer				fix_cpn_freq_;
			day_count_method 	fix_dc_;
			string				fix_curr_;
			..calendar			fix_cal_;
			bd_convention		fix_bd_;
			logical				fix_eom_; 												
			string				settle_code_;
			string				maturity_code_;
			infl_index_method	index_method_;				
			
	};

	swap_il_fixyy_def_tmpl.swap_il_fixyy_def_tmpl(	string					name,
													string 					settle_code,
													string option(nullable) maturity_code,
													integer 				fix_cpn_freq,	
													..day_count_method 		fix_dc,
													string 					fix_curr,
													..calendar				fix_cal,
													..bd_convention 		fix_bd,												
													logical 				fix_eom,
													infl_index_method		index_method)
												: 	instrument_def_tmpl(..instr_type.IL_SWAP_FIX_Y_Y),name_(name),fix_cpn_freq_(fix_cpn_freq), fix_dc_(fix_dc),
													fix_curr_(fix_curr),fix_cal_(fix_cal),
													fix_bd_(fix_bd),fix_eom_(fix_eom),
													settle_code_(settle_code),maturity_code_(maturity_code),
													index_method_(index_method){}

	string				swap_il_fixyy_def_tmpl.name() 					{ return name_;}
	day_count_method 	swap_il_fixyy_def_tmpl.fix_day_count_method()	{ return fix_dc_;} 						
	integer				swap_il_fixyy_def_tmpl.fix_cpn_freq()			{ return fix_cpn_freq_;} 					
	string				swap_il_fixyy_def_tmpl.fix_currency()			{ return fix_curr_;}
	calendar			swap_il_fixyy_def_tmpl.fix_calendar()			{ return null(fix_cal_) ? ..calendar() : fix_cal_;}
	bd_convention 		swap_il_fixyy_def_tmpl.fix_bd_convention()		{ return fix_bd_;} 
	logical				swap_il_fixyy_def_tmpl.fix_eom()				{ return fix_eom_;} 
	string 				swap_il_fixyy_def_tmpl.settle_code()			{ return settle_code_;}
	string 				swap_il_fixyy_def_tmpl.maturity_code()			{ return maturity_code_;}
	infl_index_method	swap_il_fixyy_def_tmpl.index_method()			{ return index_method_;}


	//-------------------------------------------------------
	// class il_zero_def_tmpl  (zero bond)
	//-------------------------------------------------------
	public class il_zero_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			il_zero_def_tmpl(	string,il_zero_yld_method ,infl_index_method,integer,day_count_method, 
								string ,..calendar, string, quote_style);
			
			string					name();
			integer					quasi_cpn_freq();
			day_count_method 		day_count_method(); 						
			calendar				calendar();			
			string					currency();		
			il_zero_yld_method		yld_method();
			infl_index_method		index_method();
			string 					settle_code();
			quote_style 			quote_style();			
			
		protected:
			string					name_;
			integer					quasi_cpn_freq_;
			il_zero_yld_method		yld_method_;
			infl_index_method		index_method_;	
			day_count_method 		dc_;																 					
			string					settle_code_;
			..calendar				cal_;
			string					curr_;			
			quote_style				qs_;			
			
	};

	il_zero_def_tmpl.il_zero_def_tmpl(	string					name,									
										il_zero_yld_method		yld_method,
										infl_index_method		index_method,
										integer					quasi_cpn_freq,
										..day_count_method 		dc, 																											    									 																	
										string 	 				settle_code,						
										..calendar	 			cal,
										string					curr,									
										..quote_style			qs)
										: 	instrument_def_tmpl(..instr_type.IL_REAL_ZERO_CPN_BOND),
											name_(name),quasi_cpn_freq_(quasi_cpn_freq), yld_method_(yld_method),
											index_method_(index_method), dc_(dc),
											settle_code_(settle_code), cal_(cal), curr_(curr),  qs_(qs) {}

	string					il_zero_def_tmpl.name() 				{ return name_;}
	day_count_method 		il_zero_def_tmpl.day_count_method()		{ return dc_;} 						
	il_zero_yld_method	 	il_zero_def_tmpl.yld_method()			{ return yld_method_;} 
	infl_index_method	 	il_zero_def_tmpl.index_method()			{ return index_method_;} 	 
	integer					il_zero_def_tmpl.quasi_cpn_freq()		{ return quasi_cpn_freq_;} 					 
	string 					il_zero_def_tmpl.settle_code()			{ return settle_code_;} 
	calendar				il_zero_def_tmpl.calendar()				{ return null(cal_) ? ..calendar() : cal_;}
	string					il_zero_def_tmpl.currency()				{ return curr_;}	
	quote_style 			il_zero_def_tmpl.quote_style()			{ return qs_;}

	
	//-------------------------------------------------------
	// class swap_il_zero_def_tmpl  
	//-------------------------------------------------------
	public class swap_il_zero_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			swap_il_zero_def_tmpl(	string, string ,..calendar,bd_convention,
									string, integer,infl_index_method);
			
			string					name();
			bd_convention  			bd_convention();
			logical					eom();
			day_count_method 		day_count_method(); 						
			calendar				calendar();			
			string					currency();				
			infl_index_method		index_method();
			string 					settle_code();
			integer 				maturity_years();
			
			
		protected:
			string					name_;
			..day_count_method 		dc_;
			..calendar				cal_;
			string					curr_;
	    	..bd_convention 		bd_;
	    	logical 				eom_;
			infl_index_method		index_method_;																				 					
			string					settle_code_;
			integer 				maturity_years_;
						
			
	};

	swap_il_zero_def_tmpl.swap_il_zero_def_tmpl(string					name,	
												string					curr,
			    								..calendar	 			cal,
			    								..bd_convention 		bd,
			    								string 	 				settle_code,
												integer 				maturity_years,
												infl_index_method		index_method)
												: 	instrument_def_tmpl(..instr_type.IL_SWAP_ZERO),
													name_(name),dc_(DC_ACT_365NL), cal_(cal), curr_(curr), bd_(bd), eom_(false), index_method_(index_method),
													settle_code_(settle_code), maturity_years_(maturity_years) {}

	string					swap_il_zero_def_tmpl.name() 					{ return name_;}
	day_count_method 		swap_il_zero_def_tmpl.day_count_method()		{ return dc_;} 						
	bd_convention  			swap_il_zero_def_tmpl.bd_convention()			{ return bd_;} 
	logical  				swap_il_zero_def_tmpl.eom()						{ return eom_;}
	infl_index_method	 	swap_il_zero_def_tmpl.index_method()			{ return index_method_;} 	 						 
	string 					swap_il_zero_def_tmpl.settle_code()				{ return settle_code_;}
	integer 				swap_il_zero_def_tmpl.maturity_years()			{ return maturity_years_;} 
	calendar				swap_il_zero_def_tmpl.calendar()				{ return null(cal_) ? ..calendar() : cal_;}
	string					swap_il_zero_def_tmpl.currency()				{ return curr_;}	
	


	/*

 nasdaq omx sek stock options

Type Of Contract: 		American call and put options contracts with physical delivery.

Contract Size: 			One hundred shares per contract.

Premium: 				Shall be expressed in Swedish Kronor and indicate the price for one contract share.

Premium Settlement Day: The first Swedish bank day following registration.

Expiration Day: 		The third Friday of the expiration month of the expiration year, or, where such day is not
						a Swedish bank day or is declared to be a half trading day, the preceding bank day.
						
Last Trading Day: 		The expiration day at the close of the electronic trading system for the series in question.

Exercise: 				Delivery of the relevant share occurs in exchange for an amount equivalent to the
						exercise price on the settlement day. Exercise may be requested up to the expiration day.
						
Settlement Day 			Delivery and payment for delivery occurs on the second Swedish bank day after expiration
						day, or on the second bank day following the exchange’s receipt of the exercise order.

 */
	 
	//-------------------------------------------------------
	// vanilla_option_eq_def_tmpl	
	//-------------------------------------------------------
	class vanilla_option_eq_def_tmpl : public  instrument_def_tmpl
	{
	   	public:
			vanilla_option_eq_def_tmpl(	exercise_type 	exercise_type,
										opt_type 		opt_type,
										number 			contract_size,
										..bd_convention bd, 	 								
										string 			prem_settle_code,
										string 			deliv_settle_code,
										string	option(nullable) expiry_code,
										..calendar		cal,
										string	option(nullable) strike_code,
										string	option(nullable) ccy,										
										logical			prem_at_spot,
										//eq_delta_type option(nullable)	delta_type,
										//eq_atm_type	option(nullable) atm_type,
										delivery_type 	del_type, quote_style_eq_opt);
			
			
			exercise_type 	exercise_type();
			opt_type 		opt_type();
			number 			contract_size();
			..bd_convention bd(); 	 								
			string 			prem_settle_code();
			string 			deliv_settle_code();						
			string			expiry_code();
			..calendar		cal();
			string			strike_code();
			string			ccy();			
			logical			prem_at_spot();
			//eq_delta_type	delta_type();
			//eq_atm_type		atm_type();
			delivery_type 	del_type();
			quote_style_eq_opt qs();
			
		protected:
			exercise_type 	exercise_type_;
			opt_type 		opt_type_;
			number 			contract_size_;
			..bd_convention bd_; 	 								
			string 			prem_settle_code_;
			string 			deliv_settle_code_;									
			string			expiry_code_;
			..calendar		cal_;
			string			strike_code_;
			string			ccy_;			
			logical			prem_at_spot_;
			//eq_delta_type	delta_type_;
			//eq_atm_type		atm_type_;
			delivery_type 	del_type_;
			quote_style_eq_opt qs_;
	};

	vanilla_option_eq_def_tmpl.vanilla_option_eq_def_tmpl(	..exercise_type exercise_type,
															..opt_type 		opt_type,
															number 			contract_size,
															..bd_convention bd, 	 								
															string 			prem_settle_code,
															string 			deliv_settle_code,									
															string	option(nullable) expiry_code,
															..calendar		cal,
															string	option(nullable) strike_code,
															string	option(nullable) ccy,															
															logical			prem_at_spot,
															//eq_delta_type option(nullable)	delta_type,
															//eq_atm_type	option(nullable) atm_type,
															delivery_type 	del_type,
															quote_style_eq_opt qs)
															:   instrument_def_tmpl(..instr_type.EQUITY_OPTION_VANILLA),
																exercise_type_(exercise_type),
																opt_type_(opt_type),
																contract_size_(contract_size),
																bd_(bd), 	 								
																prem_settle_code_(prem_settle_code),
																deliv_settle_code_(deliv_settle_code),								
																expiry_code_(expiry_code),
																cal_(cal),
																strike_code_(strike_code),
																ccy_(ccy),																
																prem_at_spot_(prem_at_spot),
																//delta_type_(delta_type),
																//atm_type_(atm_type),
																del_type_(del_type),
																qs_(qs)
	{}

	exercise_type 	vanilla_option_eq_def_tmpl.exercise_type()	{ return exercise_type_;}
	opt_type 		vanilla_option_eq_def_tmpl.opt_type()		{ return opt_type_;}
	number 			vanilla_option_eq_def_tmpl.contract_size()	{ return contract_size_;}
	..bd_convention vanilla_option_eq_def_tmpl.bd()				{ return bd_;}
	string 			vanilla_option_eq_def_tmpl.prem_settle_code()	{ return prem_settle_code_;}
	string 			vanilla_option_eq_def_tmpl.deliv_settle_code()	{ return deliv_settle_code_;}			
	string			vanilla_option_eq_def_tmpl.expiry_code()	{ return expiry_code_;}
	..calendar		vanilla_option_eq_def_tmpl.cal()			{ return cal_;}
	string			vanilla_option_eq_def_tmpl.strike_code()	{ return strike_code_;}
	string			vanilla_option_eq_def_tmpl.ccy()			{ return ccy_;}	
	logical			vanilla_option_eq_def_tmpl.prem_at_spot()	{ return prem_at_spot_;}
	//eq_delta_type	vanilla_option_eq_def_tmpl.delta_type()		{ return delta_type_;}
	//eq_atm_type		vanilla_option_eq_def_tmpl.atm_type()		{ return atm_type_;}
	delivery_type 	vanilla_option_eq_def_tmpl.del_type()		{ return del_type_;}
	quote_style_eq_opt vanilla_option_eq_def_tmpl.qs()			{ return qs_;}
}

