/*	
	User config for "dual" bootstrap
	Developed by Algorithmica Research, Magnus Nyström
	
	
	Comments:
	
	
*/

option(null: hard);	// Throw error if a null value is used in a non-nullable context

/*
	cb_dates;	//DZM_BOOT, DZM_MAX_SMOOTH, DZ_MAX_SMOOTH_SPREAD, DZM_TANGGAARD, DZM_TENSION: centralbank meeting dates, if null there will be no 
				//flattening of the ois curve
	apply_hyman 
	mc_force_pos 		//DZ_BOOT, DZ_ON_STEP_BOOT with ip_hagan_west(): positive forward rates are enforced
	allow_xpol			//DZ_BOOT, DZ_ON_STEP_BOOT: if true extrapolation is permitted
	step_end			//DZM_ON_STEP_BOOT: the end of the step period
	w_factor			//DZM_TANGGAARD: weight factor
	std_w 				//DZM_TANGGAARD: weights
	smooth_m 			//DZM_TANGGAARD, DZM_MAX_SMOOTH, DZ_MAX_SMOOTH_SPREAD:  smoothness measure
	spreads 			//DZ_MAX_SMOOTH_SPREAD, DZM_TENSION_W_SPREAD, DZM_TENSION_Y_SPREAD, DZM_TENSION_Z_SPREAD
	epsilon 			//DZM_TENSION_W
	tension_left_der	//DZM_TENSION_W, DZM_TENSION_Y
	tension_right_der	//DZM_TENSION_Y
*/
/*
	Model notes: 
	- tension sigma: When the sigma tends to zero, the tension spline degenerates into a regular cubic spline. 
					 On the other hand, when the sigma increases, the tension spline asymptotically reduces to piecewise 
					 linear interpolation.
					 Using too low a value for the tension parameter results in strongly non-local perturbation effects, 
					 As tension is increased the perturbation effects become increasingly local,
*/

//-------------------------------------------------------
// disc model defaults
//-------------------------------------------------------
disc_z_model_type 	g_disc_model			option(constant) = disc_z_model_type.DZ_MAX_SMOOTH;
string 				g_disc_model_s 			option(constant) = "Max_Smooth";
logical 			g_disc_mc_force_pos 	option(constant) = true; 
logical 			g_disc_apply_hyman		option(constant) = false;
disc_ip_type 		g_disc_ip				option(constant) = disc_ip_type.DIPT_LINEAR;	
string 				g_disc_ip_s				option(constant) = "Linear";
rate_type 			g_disc_rt				option(constant) = rate_type.RT_LOG_DF;
string  			g_disc_rt_s				option(constant) = "log_df";
number 				g_disc_left_der			option(constant) = 0;
number 				g_disc_hermite_right_der option(constant) = null<number>;
number 				g_disc_step_end			option(constant) = 365;
vector(number) 		g_disc_spreads()			 			 = [0.5];
z_mod_spread_type 	g_disc_spread_t			option(constant) = z_mod_spread_type.ZMS_BP;
number 				g_disc_epsilon			option(constant) = 0.3;
number 				g_disc_tension_left_der	option(constant) = 0;	//null<number> = natural boundary condition
number 				g_disc_tension_right_der option(constant) = 0;	//null<number> = natural boundary condition
number 				g_disc_w_factor			option(constant) = 0.1;
std_weights	 		g_disc_std_w			option(constant) = WS_PVBP;
smoothness_measure 	g_disc_smooth_m			option(constant) = SMOOTH_C2; 
vector(number) 		g_disc_sigma_x()			 			 = [1,10,30];
vector(number) 		g_disc_sigma_y()			 			 = [0,0,0];
disc_extrap_type 	g_disc_xpol_type		option(constant) = disc_extrap_type.DXP_ON_FLAT;
number 				g_disc_xpol_cut			option(constant) = -1;
//-------------------------------------------------------
// fwd model defaults
//-------------------------------------------------------
fwd_z_model_type 	g_fwd_model 			option(constant) = fwd_z_model_type.FZM_TENSION_EXACT;
string 				g_fwd_model_s 			option(constant) = "Tension_Exact";
logical 			g_fwd_apply_hyman		option(constant) = false;
number 				g_fwd_left_der			option(constant) = 0;
number 				g_fwd_hermite_right_der option(constant) = null<number>;
vector(number) 		g_fwd_spreads()			 				 = [0.5];
z_mod_spread_type 	g_fwd_spread_t			option(constant) = z_mod_spread_type.ZMS_BP;
number 				g_fwd_lambda			option(constant) = 0.1;
std_weights 		g_fwd_std_w				option(constant) = WS_PVBP;
vector(number) 		g_fwd_sigma_x()			 				 = [1,10,30];
vector(number) 		g_fwd_sigma_y()			 				 = [0,1,2];
number 				g_fwd_tension_left_der	option(constant) = 0;	//null<number> = natural boundary condition
number 				g_fwd_tension_right_der option(constant) = 0;	//null<number> = natural boundary condition
fwd_extrap_type 	g_fwd_xpol_type			option(constant) = fwd_extrap_type.FXP_TENOR_FLAT;
number 				g_fwd_xpol_cut			option(constant) = -1;


vector(string) disc_z_str() 	= disc_z_model_list();
vector(string) disc_ip_str()	= disc_ip_list();
vector(string) boot_rt_str()	= disc_rt_list();
vector(string) disc_extrap_str()= disc_extrap_list();
vector(string) curve_prio_str() = [	"SHORT","MIDDLE","LONG"];
vector(string) fwd_z_str() 		= fwd_z_model_list() ;
vector(string) fwd_extrap_str() =  fwd_extrap_list();

//-------------------------------------------------------
// check_tension_sigma
//-------------------------------------------------------
void check_tension_sigma(	vector(number) option(nullable) sigma_x,
							vector(number) option(nullable) sigma_y,
							string model) 
{
	QL_REQUIRE(!null(sigma_x) && v_size(sigma_x) > 0, strcat("invalid sigma_x ",model));
	QL_REQUIRE(!null(sigma_y) && v_size(sigma_y) > 0, strcat("invalid sigma_y ",model));	
	QL_REQUIRE(v_size(sigma_x) == v_size(sigma_y), strcat("invalid sigma_y/sigma_x (unequal sizes) ",model));
	for(integer i = 0; i < v_size(sigma_y); i++) {
		QL_REQUIRE(!null(sigma_y[i]) && sigma_y[i] >= 0, strcat("invalid sigma_y value (< 0) ",model));
		QL_REQUIRE(!null(sigma_x[i]) && sigma_x[i] >= 0, strcat("invalid sigma_x value (< 0) ",model));
		if( i > 0)
			QL_REQUIRE(sigma_x[i] > sigma_x[i-1], strcat("invalid sigma_x values (unordered) ",model));	
	}
}


//-------------------------------------------------------
// class disc_model_parm
//-------------------------------------------------------
class disc_model_parm option (category: 'Yield Curve/003 Curve Building')
{
   	public:

	disc_model_parm(disc_z_model_type ,calendar option(nullable) ,vector(date) option(nullable),vector(on_turns) option(nullable),logical option(nullable) ,
					logical option(nullable),disc_ip_type option(nullable),rate_type option(nullable) ,number option(nullable) ,
					number option(nullable),number option(nullable), vector(number) option(nullable) ,z_mod_spread_type option(nullable), vector(number) option(nullable) ,
					vector(number) option(nullable), number option(nullable),number option(nullable) ,number option(nullable),number option(nullable),								
					std_weights option(nullable),smoothness_measure option(nullable), disc_extrap_type ,number option(nullable));
	disc_model_parm(); 		


	void 	set_model(disc_z_model_type ) ;
	void 	set_model(string) ;
	void 	set_cal(calendar ) ;
	void 	set_cb_dates(vector(date) option(nullable),cb_step_type = cb_step_type.EXACT_FIT ) ;
	void 	set_turns(vector(on_turns) option(nullable) ) ;
	void 	set_turns(curve option(nullable)) ;
	void 	set_mc_force_pos(logical ) ;
	void 	set_apply_hyman(logical ) ;
	void 	set_ip(disc_ip_type ) ;
	void 	set_ip(string ) ;
	void 	set_rt(rate_type ) ;
	void 	set_rt(string) ;
	void 	set_left_der(number ) ;
	void 	set_hermite_right_der(number );
	void 	set_step_end(number) ;
	void 	set_spreads(vector(number) );
	void 	set_spread_type(z_mod_spread_type);
	void 	set_sigma(vector(number),vector(number)) ;
	void 	set_epsilon(number ) ;
	void 	set_tension_left_der(number ) ;
	void 	set_tension_right_der(number ) ;
	void 	set_w_factor(number ) ;								
	void 	set_std_w(std_weights) ;
	void 	set_smooth_m(smoothness_measure ) ;
	void 	set_xpol_type(disc_extrap_type);
	void 	set_xpol_type(string);
	void 	set_xpol_cut(number option(nullable));

	void 	clear_cb_dates();
	void 	clear_turns();
	
	//void 	set_model_boot_on(	disc_ip_type option(nullable),logical option(nullable),
	//						number option(nullable), number option(nullable),calendar option(nullable),vector(date) option(nullable),
	//						/*disc_extrap_type option(nullable) ,*/number option(nullable)) ;
	//void 	set_model_boot_on(	disc_ip_type option(nullable)) ;

	
	void 	set_model_boot(	disc_ip_type option(nullable),logical option(nullable),logical option(nullable),rate_type option(nullable),
							number option(nullable), number option(nullable),calendar option(nullable),vector(date) option(nullable),
							vector(..on_turns) option(nullable) ,disc_extrap_type option(nullable) ,number option(nullable)) ;
	void 	set_model_boot(	disc_ip_type option(nullable)) ;

	void 	set_model_step_boot(disc_ip_type option(nullable),logical option(nullable),logical option(nullable),rate_type option(nullable),
								number option(nullable),number option(nullable),calendar option(nullable),vector(date) option(nullable), 
								vector(..on_turns) option(nullable) ,number option(nullable),disc_extrap_type option(nullable) ,number option(nullable) ) ;
	void 	set_model_step_boot(disc_ip_type option(nullable)) ;

	void 	set_model_step() ;
	void 	set_model_step( calendar option(nullable), vector(date) option(nullable) ,vector(..on_turns) option(nullable) ,
							disc_extrap_type option(nullable) ,number option(nullable)) ;

	void 	set_model_ns();
	void	set_model_ns(	calendar option(nullable),vector(date) option(nullable),vector(..on_turns) option(nullable) ,std_weights option(nullable), 
							disc_extrap_type option(nullable) ,number option(nullable));
	
	void 	set_model_max_smooth();
	void	set_model_max_smooth(calendar option(nullable),vector(date) option(nullable),vector(..on_turns) option(nullable) ,smoothness_measure option(nullable), 
								 disc_extrap_type option(nullable) ,number option(nullable)); 

	void 	set_model_max_smooth_spread();
	void	set_model_max_smooth_spread(calendar option(nullable),vector(date) option(nullable),vector(..on_turns) option(nullable) ,smoothness_measure option(nullable),
										vector(number) option(nullable) ,z_mod_spread_type option(nullable) , 
								 		disc_extrap_type option(nullable) ,number option(nullable)) ;
	void 	set_model_tanggaard() ;
	void	set_model_tanggaard(calendar option(nullable) ,vector(date) option(nullable) ,vector(..on_turns) option(nullable) ,number option(nullable) ,std_weights option(nullable), 
								smoothness_measure option(nullable),disc_extrap_type option(nullable) ,number option(nullable)) ;

	void 	set_model_tension_w_exact();
	void	set_model_tension_w_exact(	calendar option(nullable),vector(date) option(nullable),vector(..on_turns) option(nullable) ,vector(number) option(nullable) ,vector(number) option(nullable),
										number option(nullable),disc_extrap_type option(nullable) ,number option(nullable),	
										number option(nullable) d = null<number> ) ;

	void 	set_model_tension_w_spread();
	void	set_model_tension_w_spread(	calendar option(nullable),vector(date) option(nullable),vector(..on_turns) option(nullable) ,vector(number) option(nullable) ,vector(number) option(nullable),
										number option(nullable),vector(number) option(nullable) ,z_mod_spread_type option(nullable), 
								 		disc_extrap_type option(nullable) ,number option(nullable),
										number option(nullable) d = null<number>);
	void 	set_model_tension_y_exact() ;
	void	set_model_tension_y_exact(	calendar option(nullable) ,vector(date) option(nullable), vector(..on_turns) option(nullable) ,vector(number) option(nullable) ,vector(number) option(nullable), 
								 		disc_extrap_type option(nullable) ,number option(nullable),
										number option(nullable) d = null<number>,number option(nullable) dd = null<number>);

	void 	set_model_tension_y_spread() ;
	void	set_model_tension_y_spread(	calendar option(nullable),vector(date) option(nullable),vector(..on_turns) option(nullable) ,vector(number) option(nullable) ,vector(number) option(nullable),
										vector(number) option(nullable) ,z_mod_spread_type option(nullable) , 
								 		disc_extrap_type option(nullable) ,number option(nullable),
										number option(nullable) d = null<number>,number option(nullable) dd = null<number>);
	void 	set_model_tension_z_exact();
	void	set_model_tension_z_exact(	calendar option(nullable) ,vector(date) option(nullable) ,vector(..on_turns) option(nullable) ,vector(number) option(nullable) ,vector(number) option(nullable), 
								 		disc_extrap_type option(nullable) ,number option(nullable));

	void 	set_model_tension_z_spread();
	void	set_model_tension_z_spread(	calendar option(nullable),vector(date) option(nullable),vector(..on_turns) option(nullable) ,vector(number) option(nullable) ,vector(number) option(nullable),
										vector(number) option(nullable) ,z_mod_spread_type option(nullable), 
								 		disc_extrap_type option(nullable) ,number option(nullable) );

	disc_z_model_type	model() ;
	calendar 			cal() ;
	vector(date)		cb_dates() ;
	cb_step_type		cb_step_type();
	vector(on_turns)	turns() ;
	logical 			mc_force_pos( ) ;
	logical 			apply_hyman() ;
	disc_ip_type		ip( ) ;
	rate_type			rt() ;
	number  			left_der() ;
	number  			hermite_right_der();
	number  			step_end() ;
	vector(number)		spreads();
	z_mod_spread_type	spread_type();
	vector(number) 		sigma_x() ;
	vector(number) 		sigma_y() ;
	number  			epsilon() ;
	number  			tension_left_der() ;
	number  			tension_right_der() ;	
	number  			w_factor() ;								
	std_weights			std_w() ;
	smoothness_measure	smooth_m() ;
	disc_extrap_type 	xpol_type();
	number 				xpol_cut();
	
	protected:

	void 				check();

	disc_z_model_type 	model_;
	//disc_z_model_type 	model2_;
	//number 				dual_model_cutoff_;
	calendar			cal_;
	vector(date)  		cb_dates_;
	cb_step_type		cb_step_type_;
	vector(..on_turns)  turns_;
	logical  			mc_force_pos_;
	logical  			apply_hyman_;
	disc_ip_type  		ip_;
	rate_type  			rt_;
	number  			left_der_;
	number  			hermite_right_der_;
	number				step_end_;
	vector(number)  	spreads_;
	z_mod_spread_type	spread_t_;
	vector(number)  	sigma_x_;
	vector(number)  	sigma_y_;
	number 				epsilon_;
	number				tension_left_der_;
	number				tension_right_der_;
	number 				w_factor_;								
	std_weights  		std_w_; 
	smoothness_measure 	smooth_m_;
	disc_extrap_type 	xpol_type_;
	number 				xpol_cut_;
};


disc_model_parm.disc_model_parm(disc_z_model_type 				model,
								calendar option(nullable) 		cal,
								vector(date) option(nullable) 	cb_dates,
								vector(..on_turns) option(nullable) 	turnv,
								logical option(nullable) 		mc_force_pos,								
								logical option(nullable) 		apply_hyman,
								disc_ip_type option(nullable) 	ip,
								rate_type option(nullable) 		rt,
								number option(nullable) 		left_der,
								number option(nullable) 		hermite_right_der,
								number option(nullable) 		step_end,
								vector(number) option(nullable) spreads,
								z_mod_spread_type option(nullable) spread_t,
								vector(number) option(nullable) sigma_x,
								vector(number) option(nullable) sigma_y,
								number option(nullable) 		epsilon,
								number option(nullable) 		tension_left_der,
								number option(nullable) 		tension_right_der,
								number option(nullable) 		w_factor,								
								std_weights option(nullable) 	std_w, 
								smoothness_measure option(nullable)	smooth_m,
								disc_extrap_type  				xpol_type,
								number option(nullable)			xpol_cut)
								:model_(model), cal_(cal), cb_dates_(clone_vector(cb_dates)), turns_(CORE_INT.clone_vector(turnv)),mc_force_pos_(mc_force_pos), 
								 apply_hyman_(apply_hyman), ip_(ip), rt_(rt), left_der_(left_der),hermite_right_der_(hermite_right_der), 
								 step_end_(step_end), spreads_(clone_vector(spreads)), spread_t_(spread_t), sigma_x_(sigma_x),
								 sigma_y_(sigma_y), epsilon_(epsilon), tension_left_der_(tension_left_der),tension_right_der_(tension_right_der),
								 w_factor_(w_factor), std_w_(std_w),smooth_m_(smooth_m),xpol_type_(xpol_type),
								 xpol_cut_(xpol_cut) 
{
	check();
}

disc_model_parm.disc_model_parm() : model_(g_disc_model), cal_(calendar()), cb_dates_(null<vector(date)>), turns_(null<vector(..on_turns)>),mc_force_pos_(g_disc_mc_force_pos), 
								 	apply_hyman_(g_disc_apply_hyman), ip_(g_disc_ip), rt_(g_disc_rt), left_der_(g_disc_left_der),
								 	hermite_right_der_(g_disc_hermite_right_der), step_end_(g_disc_step_end), spreads_(g_disc_spreads()), spread_t_(g_disc_spread_t), 
									sigma_x_(g_disc_sigma_x()),sigma_y_(g_disc_sigma_y()),epsilon_(g_disc_epsilon),tension_left_der_(g_disc_tension_left_der),
									tension_right_der_(g_disc_tension_right_der), w_factor_(g_disc_w_factor),
									std_w_(g_disc_std_w),smooth_m_(g_disc_smooth_m),xpol_type_(g_disc_xpol_type),xpol_cut_(g_disc_xpol_cut) 
{}


void disc_model_parm.check() 
{		
	switch (model_) 
	{
	//case disc_z_model_type.DZ_BOOT_ON :		
		//QL_REQUIRE(!null(ip_), "invalid interpolation type [boot_on disc model]");							
		//break;
	case disc_z_model_type.DZ_NS:
		QL_REQUIRE(!null(std_w_), "invalid weight factor [nelson_siegel disc model]");		
		break;
	case disc_z_model_type.DZ_BOOT :		
		QL_REQUIRE(!null(ip_), "invalid interpolation type [boot disc model]");		
		QL_REQUIRE(!null(rt_), "invalid rate type [boot disc model]");				
		break;		
	case disc_z_model_type.DZ_ON_STEP:		
		break;
	case disc_z_model_type.DZ_MAX_SMOOTH:	
		QL_REQUIRE(!null(smooth_m_), "invalid smoothness measure [max_smooth disc model]");
		break;						
	case disc_z_model_type.DZ_MAX_SMOOTH_SPREAD:
		QL_REQUIRE(!null( spread_t_), "invalid spread type [max_smooth_spread disc model]");
		QL_REQUIRE(!null(smooth_m_), "invalid smoothness measure [max_smooth_spread disc model]");
		break;
	case disc_z_model_type.DZ_ON_STEP_BOOT:			
		QL_REQUIRE(!null(ip_), "invalid interpolation type [on_step_boot disc model]");					
		QL_REQUIRE(!null(rt_), "invalid rate type [on_step_boot disc model]");	
		QL_REQUIRE(!null(step_end_), "invalid step end date [on_step_boot disc model]");									
		break;
	case disc_z_model_type.DZ_TANGGAARD:
		QL_REQUIRE(!null(w_factor_), "invalid weight factor [tanggaard disc model]");
		QL_REQUIRE(!null(smooth_m_), "invalid smoothness measure [tanggaard disc model]");		
		break;
	case disc_z_model_type.DZ_TENSION_W_EXACT: 
		check_tension_sigma(sigma_x_, sigma_y_, "[tension_w_exact disc model]");
		QL_REQUIRE(!null(epsilon_), "invalid epsilon [tension_w_exact disc model]");
		QL_REQUIRE(epsilon_ > 0, "invalid epsilon (must > 0)[tension_w_exact disc model]");		
		break;
	case disc_z_model_type.DZ_TENSION_W_SPREAD:
		QL_REQUIRE(!null( spread_t_), "invalid spread type [tension_w_spread disc model]");
		check_tension_sigma(sigma_x_, sigma_y_, "[tension_w_spread disc model]");
		QL_REQUIRE(!null(epsilon_), "invalid epsilon [tension_w_spread disc model]");	
		QL_REQUIRE(epsilon_ > 0, "invalid epsilon (must > 0)[tension_w_spread disc model]");	
		break;			
	case disc_z_model_type.DZ_TENSION_Y_EXACT:
	case disc_z_model_type.DZ_TENSION_Z_EXACT:			
		check_tension_sigma(sigma_x_, sigma_y_, "[tension_exact disc model]");		
		break;
	case disc_z_model_type.DZ_TENSION_Y_SPREAD: 		
	case disc_z_model_type.DZ_TENSION_Z_SPREAD:
		QL_REQUIRE(!null( spread_t_), "invalid spread type [tension_spread disc model]");			
		check_tension_sigma(sigma_x_, sigma_y_, "[tension_spread disc model]");		
		break;
	default:
		QL_FAIL("invalid disc model");		
	}
}

void 	disc_model_parm.set_model(disc_z_model_type t) 		{ model_ = t; }
void 	disc_model_parm.set_model(string t) 				{ model_ = disc_z_map(t); }
void 	disc_model_parm.set_cal(calendar t) 				{ cal_ = t;}
void 	disc_model_parm.set_cb_dates(vector(date) option(nullable) t, ..cb_step_type ty)  { cb_dates_ = (null(t) ? null : v_exclude_null(clone_vector(t))); cb_step_type_ = ty; }
void 	disc_model_parm.clear_cb_dates()  					{ cb_dates_ = null; }
void 	disc_model_parm.set_turns(vector(..on_turns) option(nullable) t) 	{ turns_ = (null(t) ? null : v_exclude_null(CORE_INT.clone_vector(t))); }
void 	disc_model_parm.set_turns(curve option(nullable) turn_crv)			{ turns_ = (null(turn_crv) ? null : CORE_INT.create_on_turn_v(turn_crv)); }
void 	disc_model_parm.clear_turns()						{ turns_ = null; }
void 	disc_model_parm.set_mc_force_pos(logical t ) 		{ mc_force_pos_ = t; }
void 	disc_model_parm.set_apply_hyman(logical t ) 		{ apply_hyman_ = t; }
void 	disc_model_parm.set_ip(disc_ip_type t) 				{ ip_ = t; }
void 	disc_model_parm.set_ip(string t) 					{ set_ip(disc_ip_type_map(t)); }
void 	disc_model_parm.set_rt(rate_type t) 				{ rt_ = t; }
void 	disc_model_parm.set_rt(string t) 					{ set_rt(disc_rt_map(t)); }
void 	disc_model_parm.set_left_der(number  t) 			{ left_der_ = t; }
void 	disc_model_parm.set_hermite_right_der(number  t) 	{ hermite_right_der_ = t; }
void 	disc_model_parm.set_step_end(number d) 				{ step_end_ = d; }
void 	disc_model_parm.set_spreads(vector(number) t) 		{ spreads_ = clone_vector(t); }
void 	disc_model_parm.set_spread_type(z_mod_spread_type t) 	{ spread_t_ = t; }
void 	disc_model_parm.set_sigma(vector(number) x,vector(number) y )  { sigma_x_ = clone_vector(x); sigma_y_ = clone_vector(y);}
void 	disc_model_parm.set_epsilon(number  t) 				{ epsilon_ = t; }
void 	disc_model_parm.set_tension_left_der(number t) 		{ tension_left_der_ = t;}
void 	disc_model_parm.set_tension_right_der(number t) 	{ tension_right_der_ = t;}
void 	disc_model_parm.set_w_factor(number  t) 			{ w_factor_ = t; }								
void 	disc_model_parm.set_std_w(std_weights t) 			{ std_w_ = t; }
void 	disc_model_parm.set_smooth_m(smoothness_measure t) 	{ smooth_m_ = t; }
void 	disc_model_parm.set_xpol_type(disc_extrap_type t) 	{ xpol_type_ = t; }
void 	disc_model_parm.set_xpol_type(string t) 			{ xpol_type_ = disc_extrap_map(t); }
void 	disc_model_parm.set_xpol_cut(number option(nullable) t) { xpol_cut_ = t; }

disc_z_model_type	disc_model_parm.model() 				{ return model_ ;}
calendar 			disc_model_parm.cal() 					{ return cal_ ;}
vector(date)		disc_model_parm.cb_dates() 				{ return null(cb_dates_) || v_size(cb_dates_) == 0 ? null<vector(date)> : clone_vector(cb_dates_) ;}
cb_step_type		disc_model_parm.cb_step_type() 			{ return cb_step_type_ ;}
vector(..on_turns)	disc_model_parm.turns() 				{ return null(turns_) || v_size(turns_) == 0 ? null<vector(..on_turns)> : clone_vector(turns_) ;}
logical 			disc_model_parm.mc_force_pos( ) 		{ return mc_force_pos_ ;}
logical 			disc_model_parm.apply_hyman()			{ return apply_hyman_ ;} 
disc_ip_type		disc_model_parm.ip( )					{ return ip_ ;} 
rate_type			disc_model_parm.rt() 					{ return rt_ ;}
number  			disc_model_parm.left_der() 				{ return left_der_ ;}
number  			disc_model_parm.hermite_right_der()		{ return hermite_right_der_ ;}
number  			disc_model_parm.step_end() 				{ return step_end_ ;}
vector(number)		disc_model_parm.spreads()				{ return null(spreads_) ? null : clone_vector(spreads_) ;}
z_mod_spread_type	disc_model_parm.spread_type()			{ return spread_t_ ;}
vector(number) 		disc_model_parm.sigma_x()				{ return null(sigma_x_) ? null : clone_vector(sigma_x_) ;} 
vector(number) 		disc_model_parm.sigma_y()				{ return null(sigma_y_) ? null : clone_vector(sigma_y_) ;} 
number  			disc_model_parm.epsilon() 				{ return epsilon_ ;}
number  			disc_model_parm.tension_left_der() 		{ return tension_left_der_ ;}
number  			disc_model_parm.tension_right_der() 	{ return tension_right_der_ ;}
number  			disc_model_parm.w_factor() 				{ return w_factor_ ;}								
std_weights			disc_model_parm.std_w() 				{ return std_w_ ;}
smoothness_measure	disc_model_parm.smooth_m() 				{ return smooth_m_ ;}
disc_extrap_type 	disc_model_parm.xpol_type() 			{ return xpol_type_ ; }
number  			disc_model_parm.xpol_cut() 				{ return xpol_cut_ ; }

/*void 	disc_model_parm.set_model_boot_on(	disc_ip_type option(nullable) 	ip,											
										logical option(nullable) 		apply_hyman,
										//logical option(nullable) 		mc_force_pos,										
										number option(nullable) 		left_der,
										number option(nullable) 		hermite_right_der,
										calendar option(nullable) 		cal,
										vector(date) option(nullable) 	cb_dates,
										//disc_extrap_type option(nullable) xpol_type,
										number option(nullable) 		xpol_cut) 				
{ 
	model_ = disc_z_model_type.DZ_BOOT_ON;
	if(!null(ip)) 			ip_ = ip; 
	if(!null(apply_hyman)) 	apply_hyman_ = apply_hyman;
	//if(!null(mc_force_pos)) mc_force_pos_ = mc_force_pos;
	if(!null(left_der)) 	left_der_ = left_der;
	if(!null(hermite_right_der)) hermite_right_der_ = hermite_right_der;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	//if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut;  
}*/

/*void 	disc_model_parm.set_model_boot_on(	disc_ip_type option(nullable) ip) 				
{ 
	model_ = disc_z_model_type.DZ_BOOT_ON;
	if(!null(ip)) ip_ = ip; 
}*/

void 	disc_model_parm.set_model_boot(	disc_ip_type option(nullable) 	ip,											
										logical option(nullable) 		apply_hyman,
										logical option(nullable) 		mc_force_pos,
										rate_type option(nullable) 		rt,
										number option(nullable) 		left_der,
										number option(nullable) 		hermite_right_der,
										calendar option(nullable) 		cal,
										vector(date) option(nullable) 	cb_dates,
										vector(..on_turns) option(nullable) turnv,
										disc_extrap_type option(nullable) xpol_type,
										number option(nullable) 		xpol_cut) 				
{ 
	model_ = disc_z_model_type.DZ_BOOT;
	if(!null(ip)) 			ip_ = ip; 
	if(!null(apply_hyman)) 	apply_hyman_ = apply_hyman;
	if(!null(mc_force_pos)) mc_force_pos_ = mc_force_pos;
	if(!null(rt)) 			rt_ = rt; 
	if(!null(left_der)) 	left_der_ = left_der;
	if(!null(hermite_right_der)) hermite_right_der_ = hermite_right_der;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut;  
}

void 	disc_model_parm.set_model_boot(	disc_ip_type option(nullable) ip) 				
{ 
	model_ = disc_z_model_type.DZ_BOOT;
	if(!null(ip))
		ip_ = ip;
}

void 	disc_model_parm.set_model_step_boot(disc_ip_type option(nullable) 	ip,										
											//logical option(nullable) 		allow_xpol,
											logical option(nullable) 		apply_hyman,
											logical option(nullable) 		mc_force_pos,
											rate_type option(nullable) 		rt,
											number option(nullable) 		left_der,
											number option(nullable) 		hermite_right_der,
											calendar option(nullable) 		cal,
											vector(date) option(nullable) 	cb_dates,
											vector(..on_turns) option(nullable) turnv,
											number option(nullable) 		step_end,											
											disc_extrap_type option(nullable) xpol_type,
											number option(nullable) 		xpol_cut) 				
{ 
	model_ = disc_z_model_type.DZ_ON_STEP_BOOT;
	if(!null(ip)) 			ip_ = ip; 
	if(!null(apply_hyman)) 	apply_hyman_ = apply_hyman;
	if(!null(mc_force_pos)) mc_force_pos_ = mc_force_pos;
	if(!null(rt)) 			rt_ = rt; 
	if(!null(left_der)) 	left_der_ = left_der;
	if(!null(hermite_right_der)) hermite_right_der_ = hermite_right_der;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(step_end)) 	step_end_ = integer(step_end); 
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut; 
}

void 	disc_model_parm.set_model_step_boot(disc_ip_type option(nullable) ip) 				
{ 
	model_ = disc_z_model_type.DZ_ON_STEP_BOOT;
	if(!null(ip)) ip_ = ip; 
}

void 	disc_model_parm.set_model_step(	calendar option(nullable) 			cal,
										vector(date) option(nullable) 		cb_dates,
										vector(..on_turns) option(nullable) turnv,
										disc_extrap_type option(nullable) 	xpol_type,
										number option(nullable) 			xpol_cut) 				
{ 
	model_ = disc_z_model_type.DZ_ON_STEP;
	if(!null(cal)) 			cal_ = cal;
	if(!null(cb_dates)) 	cb_dates_ 	= clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ 		= CORE_INT.clone_vector(turnv);
	if(!null(xpol_type))	xpol_type_ 	= xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ 	= xpol_cut;
}

void 	disc_model_parm.set_model_step() 				
{ 
	model_ = disc_z_model_type.DZ_ON_STEP;
}

void 	disc_model_parm.set_model_ns() 	{ model_ = disc_z_model_type.DZ_NS; }

void 	disc_model_parm.set_model_ns(	calendar option(nullable) 		cal,
										vector(date) option(nullable) 	cb_dates,
										vector(..on_turns) option(nullable) turnv,
										std_weights option(nullable)	std_w,										
										disc_extrap_type option(nullable) xpol_type,
										number option(nullable) 		xpol_cut) 			
{ 
	model_ = disc_z_model_type.DZ_NS; 
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(std_w)) 		std_w_ = std_w; 
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut;
}

void 	disc_model_parm.set_model_max_smooth() 			{ model_ = disc_z_model_type.DZ_MAX_SMOOTH; }

void 	disc_model_parm.set_model_max_smooth(	calendar option(nullable) 		cal,
												vector(date) option(nullable) 	cb_dates,
												vector(..on_turns) option(nullable) turnv,
												smoothness_measure option(nullable)	smooth_m,												
												disc_extrap_type option(nullable) xpol_type,
												number option(nullable) 		xpol_cut) 			
{ 
	model_ = disc_z_model_type.DZ_MAX_SMOOTH; 
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(smooth_m)) 	smooth_m_ = smooth_m; 
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut;
}

void 	disc_model_parm.set_model_max_smooth_spread() 	{ model_ = disc_z_model_type.DZ_MAX_SMOOTH_SPREAD;}

void 	disc_model_parm.set_model_max_smooth_spread(calendar option(nullable) 		cal,
													vector(date) option(nullable) 	cb_dates,
													vector(..on_turns) option(nullable) turnv,
													smoothness_measure option(nullable)	smooth_m,
													vector(number) option(nullable) spreads,
													z_mod_spread_type option(nullable) spread_t,													
													disc_extrap_type option(nullable) xpol_type,
													number option(nullable) 		xpol_cut) 			
{ 
	model_ = disc_z_model_type.DZ_MAX_SMOOTH_SPREAD; 
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(smooth_m)) 	smooth_m_ = smooth_m; 
	if(!null(spreads)) 		spreads_ = clone_vector(spreads); 
	if(!null(spread_t)) 	spread_t_ = spread_t; 
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut;
}

void 	disc_model_parm.set_model_tanggaard() 			{ model_ = disc_z_model_type.DZ_TANGGAARD; }

void 	disc_model_parm.set_model_tanggaard(calendar option(nullable) 		cal,
											vector(date) option(nullable) 	cb_dates,
											vector(..on_turns) option(nullable) turnv,
											number option(nullable) 		w_factor,								
											std_weights option(nullable) 	std_w, 
											smoothness_measure option(nullable)	smooth_m,
											disc_extrap_type option(nullable) xpol_type,
											number option(nullable) 		xpol_cut) 			
{ 
	model_ = disc_z_model_type.DZ_TANGGAARD;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(smooth_m)) 	smooth_m_ = smooth_m; 
	if(!null(std_w)) 		std_w_ = std_w; 
	if(!null(w_factor)) 	w_factor_ = w_factor;
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut;   
}


void 	disc_model_parm.set_model_tension_w_exact() 	{ model_ = disc_z_model_type.DZ_TENSION_W_EXACT;}

void 	disc_model_parm.set_model_tension_w_exact(	calendar option(nullable) 		cal,
													vector(date) option(nullable) 	cb_dates,
													vector(..on_turns) option(nullable) turnv,
													vector(number) option(nullable) sigma_x, 
													vector(number) option(nullable) sigma_y, 
													number option(nullable) 		epsilon,													
													disc_extrap_type option(nullable) xpol_type,
													number option(nullable) 		xpol_cut,
													number option(nullable) 		left_der) 	
{ 
	model_ = disc_z_model_type.DZ_TENSION_W_EXACT;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(sigma_x )) 	sigma_x_ = clone_vector(sigma_x) ;
	if(!null(sigma_y )) 	sigma_y_ = clone_vector(sigma_y) ;
	if(!null(epsilon)) 		epsilon_ = epsilon; 
	if(!null(left_der))		tension_left_der_ = left_der; 
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut; 
}

void 	disc_model_parm.set_model_tension_w_spread() 	{ model_ = disc_z_model_type.DZ_TENSION_W_SPREAD;}

void 	disc_model_parm.set_model_tension_w_spread(	calendar option(nullable) 		cal,
													vector(date) option(nullable) 	cb_dates,
													vector(..on_turns) option(nullable) turnv,
													vector(number) option(nullable) sigma_x, 
													vector(number) option(nullable) sigma_y, 
													number option(nullable) 		epsilon,													
													vector(number) option(nullable) spreads,
													z_mod_spread_type option(nullable) spread_t,
													disc_extrap_type option(nullable) xpol_type,
													number option(nullable) 		xpol_cut,
													number option(nullable) 		left_der)	
{ 
	model_ = disc_z_model_type.DZ_TENSION_W_SPREAD;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(sigma_x )) 	sigma_x_ = clone_vector(sigma_x) ;
	if(!null(sigma_y )) 	sigma_y_ = clone_vector(sigma_y) ;
	if(!null(epsilon)) 		epsilon_ = epsilon;
	if(!null(left_der))		tension_left_der_ = left_der; 
	if(!null(spreads)) 		spreads_ = clone_vector(spreads); 
	if(!null(spread_t)) 	spread_t_ = spread_t;
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut;   
}

void 	disc_model_parm.set_model_tension_y_exact() 	{ model_ = disc_z_model_type.DZ_TENSION_Y_EXACT;}

void 	disc_model_parm.set_model_tension_y_exact(	calendar option(nullable) 		cal,
													vector(date) option(nullable) 	cb_dates,
													vector(..on_turns) option(nullable) turnv,
													vector(number) option(nullable) sigma_x, 
													vector(number) option(nullable) sigma_y,
													disc_extrap_type option(nullable) xpol_type,
													number option(nullable) 		xpol_cut,
													number option(nullable) 		left_der,
													number option(nullable) 		right_der) 	
{ 
	model_ = disc_z_model_type.DZ_TENSION_Y_EXACT;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(sigma_x )) 	sigma_x_ = clone_vector(sigma_x) ;
	if(!null(sigma_y )) 	sigma_y_ = clone_vector(sigma_y) ;
	if(!null(left_der))		tension_left_der_ = left_der; 
	if(!null(right_der))	tension_right_der_ = right_der;
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut; 
}

void 	disc_model_parm.set_model_tension_y_spread() 	{ model_ = disc_z_model_type.DZ_TENSION_Y_SPREAD;}

void 	disc_model_parm.set_model_tension_y_spread(	calendar option(nullable) 		cal,
													vector(date) option(nullable) 	cb_dates,
													vector(..on_turns) option(nullable) turnv,
													vector(number) option(nullable) sigma_x, 
													vector(number) option(nullable) sigma_y,
													vector(number) option(nullable) spreads,
													z_mod_spread_type option(nullable) spread_t,
													disc_extrap_type option(nullable) xpol_type,
													number option(nullable) 		xpol_cut,
													number option(nullable) 		left_der,
													number option(nullable) 		right_der)	
{ 
	model_ = disc_z_model_type.DZ_TENSION_Y_SPREAD;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(sigma_x )) 	sigma_x_ = clone_vector(sigma_x) ;
	if(!null(sigma_y )) 	sigma_y_ = clone_vector(sigma_y) ;
	if(!null(left_der))		tension_left_der_ = left_der; 
	if(!null(right_der))	tension_right_der_ = right_der; 
	if(!null(spreads)) 		spreads_ = clone_vector(spreads); 
	if(!null(spread_t)) 	spread_t_ = spread_t;  
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut; 
}

void 	disc_model_parm.set_model_tension_z_exact() 	{ model_ = disc_z_model_type.DZ_TENSION_Z_EXACT;}

void 	disc_model_parm.set_model_tension_z_exact(	calendar option(nullable) 		cal,
													vector(date) option(nullable) 	cb_dates,
													vector(..on_turns) option(nullable) turnv,
													vector(number) option(nullable) sigma_x, 
													vector(number) option(nullable) sigma_y,
													disc_extrap_type option(nullable) xpol_type,
													number option(nullable) 		xpol_cut) 	
{ 
	model_ = disc_z_model_type.DZ_TENSION_Z_EXACT;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(sigma_x )) 	sigma_x_ = clone_vector(sigma_x) ;
	if(!null(sigma_y )) 	sigma_y_ = clone_vector(sigma_y) ;
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut; 
}

void 	disc_model_parm.set_model_tension_z_spread() 	{ model_ = disc_z_model_type.DZ_TENSION_Z_SPREAD;}

void 	disc_model_parm.set_model_tension_z_spread(	calendar option(nullable) 		cal,
													vector(date) option(nullable) 	cb_dates,
													vector(..on_turns) option(nullable) turnv,
													vector(number) option(nullable) sigma_x, 
													vector(number) option(nullable) sigma_y,
													vector(number) option(nullable) spreads,
													z_mod_spread_type option(nullable) spread_t,
													disc_extrap_type option(nullable) xpol_type,
													number option(nullable) 		xpol_cut)	
{ 
	model_ = disc_z_model_type.DZ_TENSION_Z_SPREAD;
	if(!null(cal)) 			cal_ = cal; 
	if(!null(cb_dates)) 	cb_dates_ = clone_vector(cb_dates);
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(sigma_x )) 	sigma_x_ = clone_vector(sigma_x) ;
	if(!null(sigma_y )) 	sigma_y_ = clone_vector(sigma_y) ;
	if(!null(spreads)) 		spreads_ = clone_vector(spreads); 
	if(!null(spread_t)) 	spread_t_ = spread_t; 
	if(!null(xpol_type)) 	xpol_type_ = xpol_type; 
	if(!null(xpol_cut)) 	xpol_cut_ = xpol_cut;  
}

//-------------------------------------------------------
// disc_model_parm
//-------------------------------------------------------
disc_model_parm disc_model_parm()
option (category: 'Yield Curve/003 Curve Building')
{			
	return new disc_model_parm();	
}	


//-------------------------------------------------------
// init_disc_model (internal)
// this is an init function for the disc model. The input arguments are the union of all inputarguments required for all
// disc models. 
//-------------------------------------------------------
disc_z_model init_disc_model(	disc_z_model_type 				disc_m_t,								
								calendar option(nullable) 		cal,
								vector(date) option(nullable) 	cb_dates,
								cb_step_type option(nullable)   cb_step_t,
								vector(..on_turns) option(nullable) turnv,
								logical option(nullable) 		boot_mc_force_pos,								
								logical option(nullable) 		boot_apply_hyman,
								disc_ip_type option(nullable) 	boot_ip,
								rate_type option(nullable) 		boot_rt,
								number option(nullable) 		boot_left_der,
								number option(nullable) 		hermite_right_der,
								number option(nullable) 		step_end,
								vector(number) option(nullable) spreads,
								z_mod_spread_type option(nullable) spread_t,
								vector(number) option(nullable) tension_sigma_x,
								vector(number) option(nullable) tension_sigma_y,
								number option(nullable) 		tension_epsilon,
								number option(nullable) 		tension_left_der,
								number option(nullable) 		tension_right_der,
								number option(nullable) 		tanggaard_w_factor,								
								std_weights option(nullable) 	std_w, 
								smoothness_measure option(nullable)	smooth_m,
								disc_extrap_type  				xpol_type,
								number option(nullable) 		xpol_cut)
option (category: 'Yield Curve/003 Curve Building')
{
	disc_z_model res;	
	switch (disc_m_t) 
	{
	/*case disc_z_model_type.DZ_BOOT_ON :
	{	
		QL_REQUIRE(!null(boot_ip), "invalid interpolation type [boot_on disc model]");
		QL_REQUIRE(boot_ip != disc_ip_type.DIPT_HAGAN_WEST, "Hagan-West model not applicable [boot_on disc model]");											
		res = disc_z_boot_on(cb_dates, cal,boot_ip,boot_apply_hyman,boot_left_der,hermite_right_der,xpol_cut);
		break;
	}*/
	case disc_z_model_type.DZ_NS :
	{	
		QL_REQUIRE(!null( std_w), "invalid std_weights [nelson_siegel disc model]");										
		res = disc_z_ns(std_w, cb_dates, cb_step_t,turnv,cal,xpol_type,xpol_cut);			
		break;
	}
	case disc_z_model_type.DZ_BOOT :
	{	
		QL_REQUIRE(!null(boot_ip), "invalid interpolation type [boot disc model]");
		if(boot_ip == disc_ip_type.DIPT_HAGAN_WEST){ 
			boot_rt = RT_CONT;
			QL_REQUIRE(!null(boot_mc_force_pos), "invalid hagan west force_pos flag [boot disc model - hagan west]");
		}				
		QL_REQUIRE(!null(boot_rt), "invalid rate type [boot disc model]");											
		res = disc_z_boot(cb_dates, cb_step_t,turnv,cal,boot_ip,boot_mc_force_pos,boot_apply_hyman,boot_left_der,hermite_right_der,boot_rt,xpol_type,xpol_cut);
		break;
	}	
	case disc_z_model_type.DZ_ON_STEP:
		res = disc_z_step(cb_dates,cb_step_t,turnv,cal,xpol_type,xpol_cut);
		//if(res.check_cb_dates(cb_dates))
			//QL_WARNING("cb_dates not used for on_step model");		
		break;
	case disc_z_model_type.DZ_MAX_SMOOTH:	
		QL_REQUIRE(!null(smooth_m), "invalid smoothness measure [max_smooth disc model]");	
		res = disc_z_max_smooth(disc_m_t,cb_dates,cb_step_t,turnv,cal,xpol_type,xpol_cut, smooth_m);
		break;			
	case disc_z_model_type.DZ_MAX_SMOOTH_SPREAD:
		QL_REQUIRE(!null(smooth_m), "invalid smoothness measure [max_smooth_spread disc model]");
		QL_REQUIRE(!null( spread_t), "invalid spread type [max_smooth_spread disc model]");
		res = disc_z_max_smooth(cb_dates,cb_step_t,turnv,cal, spreads, spread_t,xpol_type,xpol_cut, smooth_m);
		break;
	case disc_z_model_type.DZ_ON_STEP_BOOT:	
	{		
		QL_REQUIRE(!null(boot_ip), "invalid interpolation type [on_step_boot disc model]");
		if(boot_ip == disc_ip_type.DIPT_HAGAN_WEST){ 
			boot_rt = RT_CONT;
			QL_REQUIRE(!null(boot_mc_force_pos), "invalid hagan west force_pos flag [on_step_boot disc model - hagan west]");
		}			
		QL_REQUIRE(!null(boot_rt), "invalid rate type [on_step_boot disc model]");	
		QL_REQUIRE(!null(step_end), "invalid step end days [on_step_boot disc model]");								
		res = disc_z_step_boot(cb_dates,cb_step_t,turnv,cal,boot_ip,boot_mc_force_pos,boot_apply_hyman,boot_left_der,hermite_right_der,boot_rt,integer(step_end), xpol_type,xpol_cut);
		//if(res.check_cb_dates(cb_dates))
			//QL_WARNING("cb_dates not used for on_step_boot model");
		break;
	}
	case disc_z_model_type.DZ_TANGGAARD:
		QL_REQUIRE(!null(tanggaard_w_factor), "invalid weight factor [tanggaard disc model]");
		QL_REQUIRE(!null(smooth_m), "invalid smoothness measure [tanggaard disc model]");	
		res = disc_z_tanggaard(cb_dates, cb_step_t,turnv,cal, xpol_type,xpol_cut, tanggaard_w_factor, std_w, smooth_m);
		break;
	case disc_z_model_type.DZ_TENSION_W_EXACT: 
	{
		check_tension_sigma(tension_sigma_x, tension_sigma_y, "[tension_w_exact disc model]");	
		QL_REQUIRE(!null(tension_epsilon), "invalid epsilon [tension_w_exact disc model]");
		interpolation sigma = interpolation(ip_linear(0,0), tension_sigma_x, tension_sigma_y);
		QL_REQUIRE(!null(sigma), "failed to create sigma interpolation [tension_w_exact disc model]");
		res = disc_z_tension(disc_m_t, cb_dates,cb_step_t,turnv,cal,sigma, tension_epsilon, tension_left_der,tension_right_der, xpol_type,xpol_cut);
		break;
	}
	case disc_z_model_type.DZ_TENSION_W_SPREAD:
	{
		QL_REQUIRE(!null( spread_t), "invalid spread type [tension_w_spread disc model]");
		check_tension_sigma(tension_sigma_x, tension_sigma_y, "[tension_w_spread disc model]");
		QL_REQUIRE(!null(tension_epsilon), "invalid epsilon [tension_w_spread disc model]");
		interpolation sigma = interpolation(ip_linear(0,0), tension_sigma_x, tension_sigma_y);
		QL_REQUIRE(!null(sigma), "failed to create sigma interpolation [tension_w_spread disc model]");
		res = disc_z_tension(cb_dates,cb_step_t,turnv,cal,spreads, spread_t, sigma, tension_epsilon, tension_left_der,tension_right_der, xpol_type,xpol_cut);
		break;
	}
	case disc_z_model_type.DZ_TENSION_Y_EXACT:
	case disc_z_model_type.DZ_TENSION_Z_EXACT:
	{
		check_tension_sigma(tension_sigma_x, tension_sigma_y, "[tension_exact disc model]");
		interpolation sigma = interpolation(ip_linear(0,0), tension_sigma_x, tension_sigma_y);
		QL_REQUIRE(!null(sigma), "failed to create sigma interpolation [tension_exact disc model]");
		res = disc_z_tension(disc_m_t, cb_dates,cb_step_t,turnv,cal,sigma, tension_left_der,tension_right_der, xpol_type,xpol_cut);
		break;
	}
	case disc_z_model_type.DZ_TENSION_Y_SPREAD: 		
	case disc_z_model_type.DZ_TENSION_Z_SPREAD:
	{
		QL_REQUIRE(!null( spread_t), "invalid spread type [tension_spread disc model]");
		check_tension_sigma(tension_sigma_x, tension_sigma_y, "[tension_spread disc model]");
		interpolation sigma = interpolation(ip_linear(0,0), tension_sigma_x, tension_sigma_y);	
		QL_REQUIRE(!null(sigma), "failed to create sigma interpolation [tension_spread disc model]");	
		res = disc_z_tension(disc_m_t, cb_dates,cb_step_t,turnv,cal,spreads, spread_t, sigma, tension_left_der,tension_right_der, xpol_type,xpol_cut);
		break;
	}
	default:
		QL_FAIL("invalid disc model");		
	}
	QL_REQUIRE(!null(res), "failed to create disc model");
	return res;
}

//-------------------------------------------------------
// init_disc_model
//-------------------------------------------------------
disc_z_model init_disc_model(disc_model_parm p)
{
	return init_disc_model(	p.model(),
							p.cal(),
							p.cb_dates(),
							p.cb_step_type(),
							p.turns(),
							p.mc_force_pos(),							
							p.apply_hyman(),
							p.ip(),
							p.rt(),
							p.left_der(),
							p.hermite_right_der(),							
							p.step_end(),
							p.spreads(),
							p.spread_type(),
							p.sigma_x(),
							p.sigma_y(),
							p.epsilon(),
							p.tension_left_der(),
							p.tension_right_der(),
							p.w_factor(),								
							p.std_w(), 
							p.smooth_m(),
							p.xpol_type(),
							p.xpol_cut());
}

//-------------------------------------------------------
// class fwd_model_parm
//-------------------------------------------------------
class fwd_model_parm option (category: 'Yield Curve/003 Curve Building')
{
   	public:

	fwd_model_parm(fwd_z_model_type  ,vector(..fwd_turns)	option(nullable) ,logical option(nullable),number option(nullable) ,number option(nullable) ,
					vector(number) option(nullable) ,z_mod_spread_type option(nullable), vector(number) option(nullable), 
					vector(number) option(nullable), number option(nullable),number option(nullable),fwd_extrap_type, number option(nullable));
	fwd_model_parm(); 		

	void 	set_model(fwd_z_model_type ) ;
	void 	set_model(string) ;
	void 	set_turns(vector(fwd_turns) option(nullable)) ;
	void 	set_turns(curve option(nullable));
	void 	clear_turns();
	void 	set_apply_hyman(logical ) ;
	void 	set_left_der(number ) ;
	void 	set_hermite_right_der(number );
	void 	set_spreads(vector(number) );
	void 	set_spread_type(z_mod_spread_type );
	void 	set_sigma(vector(number),vector(number)) ;
	void 	set_tension_left_der(number ) ;
	void 	set_tension_right_der(number );
	void 	set_xpol_type(fwd_extrap_type);
	void 	set_xpol_type(string);
	void 	set_xpol_cut(number option(nullable));

	void 	set_lambda(number ) ;							
	void 	set_std_w(std_weights) ;

	void 	set_model_linear();
	void 	set_model_linear(vector(..fwd_turns) option(nullable),number option(nullable) ,fwd_extrap_type  option(nullable), number option(nullable) ); 

	void 	set_model_spline_nat();
	void 	set_model_spline_nat(	vector(..fwd_turns) option(nullable),logical option(nullable) ,number option(nullable),fwd_extrap_type  option(nullable), 
									number option(nullable) ); 

	void 	set_model_spline_fin();
	void 	set_model_spline_fin(	vector(..fwd_turns) option(nullable),logical option(nullable) ,number option(nullable),fwd_extrap_type  option(nullable), 
								 	number option(nullable) ); 

	void 	set_model_step();
	void 	set_model_step(vector(..fwd_turns) option(nullable),fwd_extrap_type  option(nullable), number option(nullable) ); 

	void 	set_model_hermite_akima();
	void 	set_model_hermite_akima(vector(..fwd_turns) option(nullable),logical option(nullable) ,number option(nullable) ,number option(nullable), 
									fwd_extrap_type  option(nullable),number option(nullable) ); 

	void 	set_model_hermite_cr();
	void 	set_model_hermite_cr(	vector(..fwd_turns) option(nullable),logical option(nullable) ,number option(nullable) ,number option(nullable) , 
									fwd_extrap_type  option(nullable),number option(nullable)); 

	void 	set_model_hermite_findiff();
	void 	set_model_hermite_findiff(	vector(..fwd_turns) option(nullable),logical option(nullable) ,number option(nullable) ,number option(nullable), 
										fwd_extrap_type  option(nullable),number option(nullable) ); 

	void 	set_model_hermite_fb();
	void 	set_model_hermite_fb(	vector(..fwd_turns) option(nullable),logical option(nullable) ,number option(nullable) ,number option(nullable), 
									fwd_extrap_type  option(nullable),number option(nullable) ); 

	void 	set_model_hermite_kruger();
	void 	set_model_hermite_kruger(	vector(..fwd_turns) option(nullable),logical option(nullable) ,number option(nullable) ,number option(nullable),
										fwd_extrap_type  option(nullable), number option(nullable) ); 

	void 	set_model_hermite_mono();
	void 	set_model_hermite_mono(	vector(..fwd_turns) option(nullable),logical option(nullable) ,number option(nullable) ,number option(nullable),
									fwd_extrap_type  option(nullable), number option(nullable) ); 

	void 	set_model_hermite_p();
	void 	set_model_hermite_p(vector(..fwd_turns) option(nullable),logical option(nullable) ,number option(nullable) ,number option(nullable),
								fwd_extrap_type  option(nullable), number option(nullable) ); 

	void 	set_model_step_linear();
	void 	set_model_step_linear(vector(..fwd_turns) option(nullable),number option(nullable),fwd_extrap_type  option(nullable), number option(nullable));

	void	set_model_tension_exact();
	void	set_model_tension_exact(vector(..fwd_turns) option(nullable),vector(number) option(nullable), vector(number) option(nullable),
									fwd_extrap_type  option(nullable), number option(nullable),
									number option(nullable) d = null<number>,number option(nullable) dd = null<number>);

	void	set_model_tension_spread();
	void	set_model_tension_spread(	vector(..fwd_turns) option(nullable),vector(number) option(nullable), vector(number) option(nullable), vector(number) option(nullable),
										z_mod_spread_type option(nullable),fwd_extrap_type  option(nullable), number option(nullable),
										number option(nullable) d = null<number>,number option(nullable) dd = null<number>);

	fwd_z_model_type	model() ;
	vector(..fwd_turns) turns();
	logical 			apply_hyman() ;		
	number	 			left_der() ;
	number  			hermite_right_der();	
	vector(number)		spreads();
	z_mod_spread_type	spread_type();
	vector(number)		sigma_x() ;
	vector(number)		sigma_y() ;
	number  			lambda() ;
	number				tension_left_der();
	number				tension_right_der();								
	std_weights			std_w() ;	
	fwd_extrap_type	xpol_type();
	number 				xpol_cut();

	protected:

	void 				check();
	void 				set_model_ip(	fwd_z_model_type,vector(..fwd_turns) option(nullable),logical option(nullable),number option(nullable) ,number option(nullable),
										fwd_extrap_type  option(nullable), number option(nullable) ) ;	
	
	fwd_z_model_type 	model_;
	vector(..fwd_turns) turns_;
	logical  			apply_hyman_;
	number  			left_der_;
	number  			hermite_right_der_;
	vector(number)  	spreads_;
	z_mod_spread_type	spread_t_;
	vector(number)		sigma_x_;
	vector(number)		sigma_y_;
	number				tension_left_der_;
	number				tension_right_der_;
	number 				lambda_;							
	std_weights  		std_w_; 
	fwd_extrap_type 	xpol_type_;
	number 				xpol_cut_;
};

fwd_model_parm.fwd_model_parm(	fwd_z_model_type 				model,
								vector(..fwd_turns)	option(nullable) turnv,
								logical option(nullable) 		apply_hyman,																
								number option(nullable) 		left_der,
								number option(nullable) 		hermite_right_der,								
								vector(number) option(nullable) spreads,
								z_mod_spread_type option(nullable) spread_t,
								vector(number) option(nullable) sigma_x,
								vector(number) option(nullable) sigma_y,
								//number option(nullable) 		lambda_  //not yet used
								number option(nullable) 		tension_left_der,
								number option(nullable) 		tension_right_der,	
								fwd_extrap_type  				xpol_type,
								number option(nullable)			xpol_cut)
								:model_(model), turns_(CORE_INT.clone_vector(turnv)),apply_hyman_(apply_hyman), left_der_(left_der),
								 hermite_right_der_(hermite_right_der), spreads_(clone_vector(spreads)), spread_t_(spread_t), sigma_x_(sigma_x),
								 sigma_y_(sigma_y),tension_left_der_(tension_left_der),tension_right_der_(tension_right_der),lambda_(g_fwd_lambda), 
								 std_w_(g_fwd_std_w), xpol_type_(xpol_type), xpol_cut_(xpol_cut)
{
	check();
}

fwd_model_parm.fwd_model_parm() : model_(g_fwd_model), turns_(null<vector(..fwd_turns)>), apply_hyman_(g_fwd_apply_hyman), left_der_(g_fwd_left_der),
								  hermite_right_der_(g_fwd_hermite_right_der), spreads_(g_fwd_spreads()),  spread_t_(g_fwd_spread_t), sigma_x_(g_fwd_sigma_x()),
								  sigma_y_(g_fwd_sigma_y()), tension_left_der_(g_fwd_tension_left_der),tension_right_der_(g_fwd_tension_right_der),lambda_(g_fwd_lambda), 
								  std_w_(g_fwd_std_w),xpol_type_(g_fwd_xpol_type), xpol_cut_(g_fwd_xpol_cut)
{}

void 	fwd_model_parm.set_model(fwd_z_model_type t) 		{ model_ = t; }
void 	fwd_model_parm.set_model(string t) 					{ model_ = fwd_z_map(t); }
void 	fwd_model_parm.set_turns(vector(..fwd_turns) option(nullable) t) 	{ turns_ = (null(t) ? null: v_exclude_null(CORE_INT.clone_vector(t))); }
void 	fwd_model_parm.set_turns(curve  option(nullable) turn_crv)			{ turns_ = (null(turn_crv) ? null: CORE_INT.create_fwd_turn_v(turn_crv)); }
void 	fwd_model_parm.clear_turns()										{ turns_ = null; }
void 	fwd_model_parm.set_apply_hyman(logical t ) 			{ apply_hyman_ = t; }
void 	fwd_model_parm.set_left_der(number  t) 				{ left_der_ = t; }
void 	fwd_model_parm.set_hermite_right_der(number  t) 	{ hermite_right_der_ = t; }
void 	fwd_model_parm.set_spreads(vector(number) t) 		{ spreads_ = clone_vector(t); }
void 	fwd_model_parm.set_spread_type(z_mod_spread_type t) 	{ spread_t_ = t; }
void 	fwd_model_parm.set_sigma(vector(number) x,vector(number) y )  { sigma_x_ = clone_vector(x); sigma_y_ = clone_vector(y);}
void 	fwd_model_parm.set_tension_left_der(number  t) 		{ tension_left_der_ = t; }
void 	fwd_model_parm.set_tension_right_der(number  t) 	{ tension_right_der_ = t; }
void 	fwd_model_parm.set_lambda(number  t) 				{ lambda_ = t; }							
void 	fwd_model_parm.set_std_w(std_weights t) 			{ std_w_ = t; }
void 	fwd_model_parm.set_xpol_type(fwd_extrap_type t)		{ xpol_type_ = t; }
void 	fwd_model_parm.set_xpol_type(string t) 				{ xpol_type_ = fwd_extrap_map(t); }
void 	fwd_model_parm.set_xpol_cut(number option(nullable) t) { xpol_cut_ = t; }

fwd_z_model_type	fwd_model_parm.model() 					{ return model_ ;}
vector(..fwd_turns)	fwd_model_parm.turns() 					{ return null(turns_) || v_size(turns_) == 0 ? null<vector(..fwd_turns)> : clone_vector(turns_);}
logical 			fwd_model_parm.apply_hyman()			{ return apply_hyman_ ;} 
number  			fwd_model_parm.left_der() 				{ return left_der_ ;}
number  			fwd_model_parm.hermite_right_der()		{ return hermite_right_der_ ;}
vector(number)		fwd_model_parm.spreads()				{ return null(spreads_) ? null : clone_vector(spreads_ );}
z_mod_spread_type	fwd_model_parm.spread_type()			{ return spread_t_ ;}
vector(number)		fwd_model_parm.sigma_x()				{ return null(sigma_x_) ? null : clone_vector(sigma_x_) ;}
vector(number)		fwd_model_parm.sigma_y()				{ return null(sigma_y_) ? null : clone_vector(sigma_y_) ;}  
number  			fwd_model_parm.tension_left_der() 		{ return tension_left_der_ ;}
number  			fwd_model_parm.tension_right_der()		{ return tension_right_der_ ;}
number  			fwd_model_parm.lambda() 				{ return lambda_ ;}								
std_weights			fwd_model_parm.std_w() 					{ return std_w_ ;}
fwd_extrap_type		fwd_model_parm.xpol_type()				{ return xpol_type_ ; }
number 				fwd_model_parm.xpol_cut() 				{ return xpol_cut_ ; }

void fwd_model_parm.check() 
{
	switch (model_) 
	{
	case fwd_z_model_type.FZM_LINEAR:
	case fwd_z_model_type.FZM_STEP:
	case fwd_z_model_type.FZM_STEP_LINEAR:
		break;			
	case fwd_z_model_type.FZM_SPLINE_NAT:
	case fwd_z_model_type.FZM_SPLINE_FIN:									
	case fwd_z_model_type.FZM_HERMITE_AKIMA:			
	case fwd_z_model_type.FZM_HERMITE_CR:			
	case fwd_z_model_type.FZM_HERMITE_FINDIFF:			
	case fwd_z_model_type.FZM_HERMITE_FB:			
	case fwd_z_model_type.FZM_HERMITE_KRUGER:			
	case fwd_z_model_type.FZM_HERMITE_MONO:			
	case fwd_z_model_type.FZM_HERMITE_P:		
		QL_REQUIRE(!null(apply_hyman_), "invalid hyman flag [fwd model]");				
		break;		
	case fwd_z_model_type.FZM_TENSION_EXACT:		
		check_tension_sigma(sigma_x_, sigma_y_, "[tension_exact fwd model]");		
		break;		
	case fwd_z_model_type.FZM_TENSION_SPREAD	:
		QL_REQUIRE(!null(spread_t_), "invalid spread type [tension_spread fwd model]");		
		check_tension_sigma(sigma_x_, sigma_y_, "[tension_spread fwd model]");
		break;
	default:
		QL_FAIL("invalid fwd model");
	}
}

//-------------------------------------------------------
// fwd_model_parm
//-------------------------------------------------------
fwd_model_parm fwd_model_parm()
option (category: 'Yield Curve/003 Curve Building')
{			
	return new fwd_model_parm();	
}

void 	fwd_model_parm.set_model_ip(fwd_z_model_type 				m,
									vector(..fwd_turns)	option(nullable) turnv,
									logical option(nullable) 		apply_hyman,
									number option(nullable) 		left_der,
									number option(nullable) 		hermite_right_der,
									fwd_extrap_type  option(nullable) xpol_type,
									number option(nullable)			xpol_cut) 				
{ 
	model_ = m; 
	if(!null(turnv)) 			turns_ = clone_vector(turnv);
	if(!null(apply_hyman)) 		apply_hyman_ = apply_hyman; 
	if(!null(left_der)) 		left_der_ = left_der;
	if(!null(hermite_right_der)) hermite_right_der_ = hermite_right_der;
	if(!null(xpol_type)) 		xpol_type_ = xpol_type;
	if(!null(xpol_cut)) 		xpol_cut_ = xpol_cut;
}

void 	fwd_model_parm.set_model_linear() 	{  model_ = fwd_z_model_type.FZM_LINEAR;}
void 	fwd_model_parm.set_model_linear(vector(..fwd_turns)	option(nullable) turnv,
										number option(nullable) 	left_der,
										fwd_extrap_type  option(nullable) xpol_type,
										number option(nullable)		xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_LINEAR,turnv,false,left_der,null<number>,xpol_type,xpol_cut);
}

void 	fwd_model_parm.set_model_spline_nat() 	{  model_ = fwd_z_model_type.FZM_SPLINE_NAT;}
void 	fwd_model_parm.set_model_spline_nat(vector(..fwd_turns)	option(nullable) turnv,
											logical option(nullable) 	apply_hyman,
											number option(nullable) 	left_der,
											fwd_extrap_type  option(nullable) xpol_type,
											number option(nullable)		xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_SPLINE_NAT,turnv,apply_hyman,left_der,null<number>,xpol_type,xpol_cut);
}

void 	fwd_model_parm.set_model_spline_fin() 	{  model_ = fwd_z_model_type.FZM_SPLINE_FIN;}
void 	fwd_model_parm.set_model_spline_fin(vector(..fwd_turns)	option(nullable) turnv,
											logical option(nullable) 	apply_hyman,
											number option(nullable) 	left_der,
											fwd_extrap_type  option(nullable) xpol_type,
											number option(nullable)		xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_SPLINE_FIN,turnv,apply_hyman,left_der,null<number>,xpol_type,xpol_cut);
}

void 	fwd_model_parm.set_model_step() 	{  model_ = fwd_z_model_type.FZM_STEP;}
void 	fwd_model_parm.set_model_step(	vector(..fwd_turns)	option(nullable) turnv,
										fwd_extrap_type  option(nullable) xpol_type,
										number option(nullable)			xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_STEP,turnv,false,null<number>,null<number>,xpol_type,xpol_cut);
}
			
void 	fwd_model_parm.set_model_hermite_akima() 	{  model_ = fwd_z_model_type.FZM_HERMITE_AKIMA;}
void 	fwd_model_parm.set_model_hermite_akima(	vector(..fwd_turns)	option(nullable) turnv,
												logical option(nullable) 	apply_hyman,
												number option(nullable) 	left_der,
												number option(nullable) 	right_der,
												fwd_extrap_type  option(nullable) xpol_type,
												number option(nullable)		xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_HERMITE_AKIMA,turnv,apply_hyman,left_der,right_der,xpol_type,xpol_cut);
}

void 	fwd_model_parm.set_model_hermite_cr() 	{  model_ = fwd_z_model_type.FZM_HERMITE_CR;}
void 	fwd_model_parm.set_model_hermite_cr(vector(..fwd_turns)	option(nullable) turnv,
											logical option(nullable) 	apply_hyman,
											number option(nullable) 	left_der,
											number option(nullable) 	right_der,
											fwd_extrap_type  option(nullable) xpol_type,
											number option(nullable)		xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_HERMITE_CR,turnv,apply_hyman,left_der,right_der,xpol_type,xpol_cut);
}

void 	fwd_model_parm.set_model_hermite_findiff() 	{  model_ = fwd_z_model_type.FZM_HERMITE_FINDIFF;}
void 	fwd_model_parm.set_model_hermite_findiff(vector(..fwd_turns)	option(nullable) turnv,
												 logical option(nullable) 	apply_hyman,
												number option(nullable) 	left_der,
												number option(nullable) 	right_der,
											fwd_extrap_type  option(nullable) xpol_type,
											number option(nullable)			xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_HERMITE_FINDIFF,turnv,apply_hyman,left_der,right_der,xpol_type,xpol_cut);
}				
			
void 	fwd_model_parm.set_model_hermite_fb() 	{  model_ = fwd_z_model_type.FZM_HERMITE_FB;}
void 	fwd_model_parm.set_model_hermite_fb(vector(..fwd_turns)	option(nullable) turnv,
											logical option(nullable) 	apply_hyman,
											number option(nullable) 	left_der,
											number option(nullable) 	right_der,
											fwd_extrap_type  option(nullable) xpol_type,
											number option(nullable)		xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_HERMITE_FB,turnv,apply_hyman,left_der,right_der,xpol_type,xpol_cut);
}			

void 	fwd_model_parm.set_model_hermite_kruger() 	{  model_ = fwd_z_model_type.FZM_HERMITE_KRUGER;}
void 	fwd_model_parm.set_model_hermite_kruger(vector(..fwd_turns)	option(nullable) turnv,
												logical option(nullable) 	apply_hyman,
												number option(nullable) 	left_der,
												number option(nullable) 	right_der,
												fwd_extrap_type  option(nullable) xpol_type,
												number option(nullable)		xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_HERMITE_KRUGER,turnv,apply_hyman,left_der,right_der,xpol_type,xpol_cut);
}		

void 	fwd_model_parm.set_model_hermite_mono() 	{  model_ = fwd_z_model_type.FZM_HERMITE_MONO;}
void 	fwd_model_parm.set_model_hermite_mono(	vector(..fwd_turns)	option(nullable) turnv,
												logical option(nullable) 	apply_hyman,
												number option(nullable) 	left_der,
												number option(nullable) 	right_der,
												fwd_extrap_type  option(nullable) xpol_type,
												number option(nullable)		xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_HERMITE_MONO,turnv,apply_hyman,left_der,right_der,xpol_type,xpol_cut);
}

void 	fwd_model_parm.set_model_hermite_p() 	{  model_ = fwd_z_model_type.FZM_HERMITE_P;}
void 	fwd_model_parm.set_model_hermite_p(	vector(..fwd_turns)	option(nullable) turnv,
											logical option(nullable) 	apply_hyman,
											number option(nullable) 	left_der,
											number option(nullable) 	right_der,
											fwd_extrap_type  option(nullable) xpol_type,
											number option(nullable)		xpol_cut) 
{
	set_model_ip(fwd_z_model_type.FZM_HERMITE_P,turnv,apply_hyman,left_der,right_der,xpol_type,xpol_cut);
}

void 	fwd_model_parm.set_model_step_linear() 	{  model_ = fwd_z_model_type.FZM_STEP_LINEAR;}
void 	fwd_model_parm.set_model_step_linear(	vector(..fwd_turns)	option(nullable) turnv,
												number option(nullable) 	left_der,
												fwd_extrap_type  option(nullable) xpol_type,
												number option(nullable)		xpol_cut) 
{
	model_ = fwd_z_model_type.FZM_STEP_LINEAR; 
	if(!null(turnv)) 		turns_ = clone_vector(turnv);
	if(!null(left_der)) 	left_der_ 	= left_der;	
	if(!null(xpol_type)) 	xpol_type_ 	= xpol_type;
	if(!null(xpol_cut)) 	xpol_cut_ 	= xpol_cut;
}

void 	fwd_model_parm.set_model_tension_exact() 	{  model_ = fwd_z_model_type.FZM_TENSION_EXACT;}
void 	fwd_model_parm.set_model_tension_exact(	vector(..fwd_turns)	option(nullable) turnv,
												vector(number) option(nullable) sigma_x,
												vector(number) option(nullable) sigma_y,
												fwd_extrap_type  option(nullable) xpol_type,
												number option(nullable)			xpol_cut,
												number option(nullable) 		left_der,
												number option(nullable) 		right_der) 
{
	model_ = fwd_z_model_type.FZM_TENSION_EXACT;
	if(!null(turnv))  turns_ 		= clone_vector(turnv);
	if(!null(sigma_x)) 	sigma_x_ 	= clone_vector(sigma_x);
	if(!null(sigma_y)) 	sigma_y_ 	= clone_vector(sigma_y);
	if(!null(left_der))	tension_left_der_ 	= left_der; 
	if(!null(right_der)) tension_right_der_ = right_der; 
	if(!null(xpol_type)) xpol_type_ = xpol_type;
	if(!null(xpol_cut)) xpol_cut_ 	= xpol_cut;

}

void 	fwd_model_parm.set_model_tension_spread() 	{  model_ = fwd_z_model_type.FZM_TENSION_SPREAD;}
void 	fwd_model_parm.set_model_tension_spread(vector(..fwd_turns)	option(nullable) turnv,
												vector(number) option(nullable) sigma_x,
												vector(number) option(nullable) sigma_y,
												vector(number) option(nullable) spreads,
												z_mod_spread_type option(nullable) spread_t,
												fwd_extrap_type  option(nullable) xpol_type,
												number option(nullable)			xpol_cut,
												number option(nullable) 		left_der,
												number option(nullable) 		right_der) 
{
	model_ = fwd_z_model_type.FZM_TENSION_SPREAD;
	if(!null(turnv))  turns_ 		= clone_vector(turnv);
	if(!null(sigma_x)) 	sigma_x_ 	= clone_vector(sigma_x);
	if(!null(sigma_y)) 	sigma_y_ 	= clone_vector(sigma_y);
	if(!null(spreads)) 	spreads_ 	= clone_vector(spreads);
	if(!null(spread_t)) spread_t_ 	= spread_t;
	if(!null(left_der))	tension_left_der_ 	= left_der; 
	if(!null(right_der)) tension_right_der_ = right_der; 
	if(!null(xpol_type)) xpol_type_ = xpol_type;
	if(!null(xpol_cut)) xpol_cut_ 	= xpol_cut;
}

//-------------------------------------------------------
// init_fwd_model (internal)
// this is an init function from from all the parameters for all fwd models
//-------------------------------------------------------
fwd_z_model init_fwd_model(	fwd_z_model_type 				fwd_model_t,
							vector(..fwd_turns) option(nullable) turnv,
							logical option(nullable) 		ip_apply_hyman,							
							number option(nullable) 		ip_left_der,
							number option(nullable) 		hermite_right_der,
							vector(number) option(nullable) tension_sigma_x,
							vector(number) option(nullable) tension_sigma_y,
							vector(number) option(nullable) tension_spreads,
							z_mod_spread_type option(nullable) tension_spread_t,
							number option(nullable) 		tension_left_der,
							number option(nullable) 		tension_right_der,							
							fwd_extrap_type 				xpol_type,
							number option(nullable) 		xpol_cut)
option (category: 'Yield Curve/003 Curve Building')
{
	fwd_z_model res;
	switch (fwd_model_t) 
	{
	case fwd_z_model_type.FZM_LINEAR:
	case fwd_z_model_type.FZM_STEP:
		res = fwd_z_ip(fwd_model_t, null<logical>, ip_left_der, null<number>, turnv,xpol_type, xpol_cut);
		break;			
	case fwd_z_model_type.FZM_SPLINE_NAT:
	case fwd_z_model_type.FZM_SPLINE_FIN:
		QL_REQUIRE(!null(ip_apply_hyman), "invalid hyman flag [spline fwd model]");						
		res = fwd_z_ip(fwd_model_t, ip_apply_hyman, ip_left_der, null<number>, turnv,xpol_type, xpol_cut);
		break;									
	case fwd_z_model_type.FZM_HERMITE_AKIMA:			
	case fwd_z_model_type.FZM_HERMITE_CR:			
	case fwd_z_model_type.FZM_HERMITE_FINDIFF:			
	case fwd_z_model_type.FZM_HERMITE_FB:			
	case fwd_z_model_type.FZM_HERMITE_KRUGER:			
	case fwd_z_model_type.FZM_HERMITE_MONO:			
	case fwd_z_model_type.FZM_HERMITE_P:	
		QL_REQUIRE(!null(ip_apply_hyman), "invalid hyman flag [hermite fwd model]");						
		res = fwd_z_ip(fwd_model_t, ip_apply_hyman, ip_left_der, hermite_right_der, turnv,xpol_type, xpol_cut);
		break;
	case fwd_z_model_type.FZM_STEP_LINEAR:			
		res = fwd_z_step_linear( ip_left_der, turnv,xpol_type, xpol_cut);
		break;	
	case fwd_z_model_type.FZM_TENSION_EXACT:
	{
		check_tension_sigma(tension_sigma_x, tension_sigma_y, "[tension_exact fwd model]");
		interpolation sigma = interpolation(ip_linear(0,0), tension_sigma_x, tension_sigma_y);
		QL_REQUIRE(!null(sigma), "failed to create sigma interpolation [tension_exact fwd model]");	
		res = fwd_z_tension(fwd_model_t, sigma, tension_left_der,tension_right_der,turnv,xpol_type, xpol_cut);
		break;
	}		
	case fwd_z_model_type.FZM_TENSION_SPREAD	:
	{
		check_tension_sigma(tension_sigma_x, tension_sigma_y, "[tension_spread fwd model]");
		interpolation sigma = interpolation(ip_linear(0,0), tension_sigma_x, tension_sigma_y);	
		QL_REQUIRE(!null(sigma), "failed to create sigma interpolation [tension_spread fwd model]");		
		QL_REQUIRE(!null(tension_spread_t), "invalid spread type [tension_spread fwd model]");
		res = fwd_z_tension(sigma, tension_spreads, tension_spread_t, tension_left_der,tension_right_der,turnv,xpol_type, xpol_cut);
		break;
	}
	default:
		QL_FAIL("invalid fwd model");
	}
	QL_REQUIRE(!null(res), "failed to create fwd model");
	return res;
}

//-------------------------------------------------------
// init_fwd_model
//-------------------------------------------------------
fwd_z_model init_fwd_model(fwd_model_parm p)
{
	return init_fwd_model(	p.model(),
							p.turns(),
							p.apply_hyman(),							
							p.left_der(),
							p.hermite_right_der(),														
							p.sigma_x(),
							p.sigma_y(),
							p.spreads(),
							p.spread_type(),
							p.tension_left_der(),
							p.tension_right_der(),							
							p.xpol_type(),
							p.xpol_cut());
}

