/*	
	curve template classes
	Developer: Algorithmica Research, Magnus Nyström			
*/

option(null: hard);	

module CURVE_TMPL
{
	//-------------------------------------------------------
	// class curve_tmpl <abstract>
	//-------------------------------------------------------
	public class curve_tmpl 
	{
	public:
		curve_tmpl(swap_curve_type);
		
		logical is_disc() ;
		logical is_base_fwd() ;
		logical is_basis_fwd() ;
		logical is_basis_2swp_fwd() ;
		virtual logical short_crv_is_fix();
		virtual fwd_func_tenor fwd_tenor() ;
		virtual instrument_name fixing_instr() ;
		
		virtual 				swap_curve_type type() 	= 0;
		virtual logical 		check(out error_info) 	= 0;
		virtual vector(string) 	curve_names() 			= 0;
		
		curve 					get_curve(	curve_name option(nullable) ,date, quote_side option(nullable),
											out error_info,
											logical  remove_err = true,
											logical  remove_nullq = false,
											logical  remove_expired = true,
											logical  remove_dup_mat = true,
											logical  remove_part_fixed = false) ;
		
		logical 				curve_diag(	curve_name option(nullable) ,date, quote_side option(nullable) ,
											out curve,
											out curve_status,						
											out vector(string),
											out vector(string) );
	protected:
		swap_curve_type type_;
	};
	
	curve_tmpl.curve_tmpl(swap_curve_type t): type_(t){}
	fwd_func_tenor curve_tmpl.fwd_tenor() 	{ return null<fwd_func_tenor>;}
	instrument_name curve_tmpl.fixing_instr() 				{ return null<instrument_name>;}
	
	logical curve_tmpl.is_disc() 			{ return type_ == swap_curve_type.SCE_DISC;}
	logical curve_tmpl.is_base_fwd() 		{ return type_ == swap_curve_type.SCE_FWD;}
	logical curve_tmpl.is_basis_fwd() 		{ return type_ == swap_curve_type.SCE_FWD_B1_F;}
	logical curve_tmpl.is_basis_2swp_fwd() 	{ return type_ == swap_curve_type.SCE_FWD_B2_F;}
		
	curve curve_tmpl.get_curve(	curve_name option(nullable) name,
								date 						td,
								quote_side option(nullable) qs,
								out error_info 				error,
								logical 					remove_err ,
								logical 					remove_nullq ,
								logical 					remove_expired ,
								logical 					remove_dup_mat,
								logical 					remove_part_fixed )
	{		
		if(null(name))
			return null<curve>;

		if(!__exist_curve(name)) {			
			CORE_INT.add_error_info(error,ERR_T_INIT,"curve name not in database","curve_tmpl.get_curve" );
			return null<curve>;
		}
		
		curve c ;
		try {
			logical use_prefetch = true;
			c = curve(name,td,qs,remove_err,remove_nullq, remove_expired, remove_dup_mat,use_prefetch, remove_part_fixed);
		}
		catch {
			c = null<curve>;
		}

		if(null(c)) {			
			CORE_INT.add_error_info(error,ERR_T_INIT,"null curve","curve_tmpl.get_curve" );
			return null<curve>;
		}
		if(c.empty()) {		
			CORE_INT.add_error_info(error,ERR_T_INIT,"empty curve","curve_tmpl.get_curve" );
			return null<curve>;
		}
		return c;
	}

	logical curve_tmpl.curve_diag(	curve_name option(nullable) name,
									date td,
									quote_side option(nullable) qs,
									out curve			orig_curve,
									out curve_status	status,						
									out vector(string) 	err_name,
									out vector(string) 	err_desc )
	{
		if(null(name)) {
			orig_curve 	= null<curve>;
			status 		= curve_status.NAME_NULL;					
			err_desc 	= err_name = null<vector(string)>;			
			return true;//because a null name is not an error in this context
		}
		
		vector(curve_instr_errs) err_e;
		curve_diagnostics(	name,td,qs,orig_curve,status,err_name,err_e,err_desc);
		
		return status == curve_status.OK;
	}

	logical curve_tmpl.short_crv_is_fix() { return null<logical>;}
	
	//-------------------------------------------------------
	// class disc_curve_tmpl
	//-------------------------------------------------------
	public class disc_curve_tmpl : public curve_tmpl
	{
	   	public:
		override swap_curve_type type() ;
		override logical check(out error_info) ;
		override vector(string) curve_names();
		
		disc_curve_tmpl(curve_name short_crv,quote_side);
		
		disc_curve_tmpl(curve_name option(nullable) short_crv,
						curve_name option(nullable) middle_crv,
						curve_name option(nullable)	long_crv,
						curve_name option(nullable) extrap_crv,						
						quote_side option(nullable) short_qs,
						quote_side option(nullable) middle_qs,
						quote_side option(nullable) long_qs);
		
		curve_name 		short_crv();
		curve_name  	middle_crv();
		curve_name 		long_crv();
		curve 			short_crv(date,quote_side option(nullable) , out error_info err );
		curve 			middle_crv(date,quote_side option(nullable) , out error_info err);
		curve 			long_crv(date,quote_side option(nullable) , out error_info err);
		curve_name		extrap_crv();
		curve 			extrap_crv(date,quote_side option(nullable) , out error_info err);
		//number  		long_first_fix();		
		quote_side  	short_qs();
		quote_side  	middle_qs();
		quote_side  	long_qs();
		quote_side  	extrap_qs();
		
		
		protected:
		
		curve_name 	short_crv_;
		curve_name  middle_crv_;
		curve_name 	long_crv_;
		curve_name	extrap_crv_;
		//number  	long_first_fix_;		
		quote_side  short_qs_;
		quote_side  middle_qs_;
		quote_side  long_qs_;
	};

	
	
	logical disc_curve_tmpl.check(out error_info err)
	{
		
		logical sh_in = !null(short_crv_)  ;
		logical mi_in = !null(middle_crv_)  ;
		logical lo_in = !null(long_crv_)  ;
		if(!sh_in  && !mi_in && ! lo_in){			
			CORE_INT.add_error_info(err,ERR_T_INIT,"missing curves","disc_curve_tmpl.check" );
			return false;
		}

		if( sh_in && !__exist_curve(short_crv_)){			
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (short end curve)","disc_curve_tmpl.check" );
			return false;
		}
		if( mi_in && !__exist_curve(middle_crv_)) {			
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (middle curve)","disc_curve_tmpl.check" );
			return false;
		}
		if( lo_in && !__exist_curve(long_crv_)){	
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (long end curve)","disc_curve_tmpl.check" );
			return false;
		}
		if( !null(extrap_crv_) && !__exist_curve(extrap_crv_)){			
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (extrapolation curve)","disc_curve_tmpl.check" );
			return false;
		}
		return true;
	}

	vector(string) disc_curve_tmpl.curve_names(){
		vector(string) res;
		if(!null(short_crv_))
			push_back(res,short_crv_);
		if(!null(middle_crv_)  )
			push_back(res,middle_crv_);
		if(!null(long_crv_)  )
			push_back(res,long_crv_);
		return res;
	}
	
	swap_curve_type disc_curve_tmpl.type() 			{ return type_;}
	curve_name 		disc_curve_tmpl.short_crv() 	{ return short_crv_;}
	curve_name  	disc_curve_tmpl.middle_crv()	{ return middle_crv_;}
	curve_name 		disc_curve_tmpl.long_crv()		{ return long_crv_;}
	curve_name 		disc_curve_tmpl.extrap_crv()	{ return extrap_crv_;}	
	//number  		disc_curve_tmpl.long_first_fix(){ return long_first_fix_;}	
	quote_side  	disc_curve_tmpl.short_qs()		{ return short_qs_;}
	quote_side  	disc_curve_tmpl.middle_qs()		{ return middle_qs_;}
	quote_side  	disc_curve_tmpl.long_qs()		{ return long_qs_;}
	quote_side  	disc_curve_tmpl.extrap_qs()
	{
		if(!null(long_qs_))
			return long_qs_;
		else if(!null(middle_qs_))
			return middle_qs_;
		else
			return short_qs_;
	}
			
	curve disc_curve_tmpl.short_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = short_qs_;
		return get_curve(short_crv_,td,qs, err);
	}
	
	curve  disc_curve_tmpl.middle_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = middle_qs_;
		return get_curve(middle_crv_,td,qs,err);
	}
	
	curve disc_curve_tmpl.long_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = long_qs_;
		return get_curve(long_crv_,td,qs,err);
	}
	
	curve disc_curve_tmpl.extrap_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = extrap_qs();
		return get_curve(extrap_crv_,td,qs,err);
	}
	
	disc_curve_tmpl.disc_curve_tmpl(curve_name option(nullable) 	short_crv,
									curve_name option(nullable) 	middle_crv,
									curve_name option(nullable)		long_crv,
									curve_name 	option(nullable) 	extrap_crv,
									quote_side option(nullable) 	short_qs,
									quote_side option(nullable) 	middle_qs,
									quote_side option(nullable) 	long_qs)
								:curve_tmpl(swap_curve_type.SCE_DISC),
								 short_crv_(null_curve_name(short_crv)),
								 middle_crv_(null_curve_name(middle_crv)),
								 long_crv_(null_curve_name(long_crv)),
								 extrap_crv_(null_curve_name(extrap_crv)),
								 //long_first_fix_(null<number>),								 
								 short_qs_(short_qs),
								 middle_qs_(middle_qs), long_qs_(long_qs)
	{}

	disc_curve_tmpl.disc_curve_tmpl(curve_name crv, quote_side qs)
								:curve_tmpl(swap_curve_type.SCE_DISC),
								 short_crv_(null_curve_name(crv)),
								 middle_crv_(null<curve_name>),
								 long_crv_(null<curve_name>),
								 extrap_crv_(null<curve_name>),
								 //long_first_fix_(null<number>),								 
								 short_qs_(qs),middle_qs_(qs), long_qs_(qs)
	{}


	//-------------------------------------------------------
	// class disc_basis_curve_tmpl
	//-------------------------------------------------------
	public class disc_basis_curve_tmpl : public curve_tmpl
	{
	   	public:
		override swap_curve_type type() ;
		override logical check(out error_info) ;
		override vector(string) curve_names();
		
		
		disc_basis_curve_tmpl(	curve_name option(nullable) short_crv,
								curve_name option(nullable) middle_crv,
								curve_name option(nullable) long_crv,	
								curve_name 					lib_swap_crv,
								curve_name 					ois_lib_basis_crv,													
								quote_side option(nullable) short_qs,
								quote_side option(nullable) middle_qs,
								quote_side option(nullable) long_qs,
								quote_side option(nullable) lib_swap_qs,
								quote_side option(nullable) ois_lib_basis_qs);
		
		curve_name 		short_crv();
		curve_name  	middle_crv();
		curve_name  	long_crv();
		curve_name 		ois_lib_basis_crv();
		curve_name 		lib_swap_crv();
		
		curve 			short_crv(date,quote_side option(nullable) , out error_info err );
		curve 			middle_crv(date,quote_side option(nullable) , out error_info err);
		curve 			long_crv(date,quote_side option(nullable) , out error_info err);
		curve 			ois_lib_basis_crv(date,quote_side option(nullable) , out error_info err);
		curve 			lib_swap_crv(date,quote_side option(nullable) , out error_info err);		
		
		//number  		long_first_fix();		
		quote_side  	short_qs();
		quote_side  	middle_qs();
		quote_side  	long_qs();
		quote_side  	ois_lib_basis_qs();
		quote_side  	lib_swap_qs();

		protected:
		
		curve_name 	short_crv_;
		curve_name  middle_crv_;
		curve_name	long_crv_;
		curve_name 	lib_swap_crv_;
		curve_name 	ois_lib_basis_crv_;
		
		quote_side  short_qs_;
		quote_side  middle_qs_;
		quote_side  long_qs_;
		quote_side  lib_swap_qs_;
		quote_side  ois_lib_basis_qs_;
		
	};

	
	
	logical disc_basis_curve_tmpl.check(out error_info err)
	{
		
		logical sh_in = !null(short_crv_)  ;
		logical mi_in = !null(middle_crv_)  ;
		logical lo_in = !null(long_crv_)  ;
		logical lob_in = !null(ois_lib_basis_crv_)  ;
		logical lo_ls_in = !null(lib_swap_crv_)  ;
		
		if(!sh_in  && !mi_in && ! lo_in && ! lob_in && !lo_ls_in){			
			CORE_INT.add_error_info(err,ERR_T_INIT,"missing curves","disc_basis_curve_tmpl.check" );
			return false;
		}

		if( sh_in && !__exist_curve(short_crv_)){			
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (short end curve)","disc_basis_curve_tmpl.check" );
			return false;
		}
		if( mi_in && !__exist_curve(middle_crv_)) {			
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (middle curve)","disc_basis_curve_tmpl.check" );
			return false;
		}
		if( lo_in && !__exist_curve(long_crv_)){			
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (long end curve)","disc_basis_curve_tmpl.check" );
			return false;
		}
		
		if( lob_in && !__exist_curve(ois_lib_basis_crv_)){	
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (ois_lib_basis curve)","disc_basis_curve_tmpl.check" );
			return false;
		}
		if( lo_ls_in && !__exist_curve(lib_swap_crv_)){	
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (lib_swap curve)","disc_basis_curve_tmpl.check" );
			return false;
		}
		
		return true;
	}

	vector(string) disc_basis_curve_tmpl.curve_names(){
		vector(string) res;
		if(!null(short_crv_))
			push_back(res,short_crv_);
		if(!null(middle_crv_)  )
			push_back(res,middle_crv_);
		if(!null(long_crv_)  )
			push_back(res,long_crv_);
		if(!null(lib_swap_crv_)  )
			push_back(res,lib_swap_crv_);
		if(!null(ois_lib_basis_crv_)  )
			push_back(res,ois_lib_basis_crv_);
		
		return res;
	}
	
	swap_curve_type disc_basis_curve_tmpl.type() 			{ return type_;}
	curve_name 		disc_basis_curve_tmpl.short_crv() 		{ return short_crv_;}
	curve_name  	disc_basis_curve_tmpl.middle_crv()		{ return middle_crv_;}
	curve_name 		disc_basis_curve_tmpl.long_crv()		{ return long_crv_;}
	curve_name 		disc_basis_curve_tmpl.ois_lib_basis_crv() { return ois_lib_basis_crv_;}
	curve_name 		disc_basis_curve_tmpl.lib_swap_crv()	{ return lib_swap_crv_;}
		
	//number  		disc_basis_curve_tmpl.long_first_fix(){ return long_first_fix_;}	
	quote_side  	disc_basis_curve_tmpl.short_qs()		{ return short_qs_;}
	quote_side  	disc_basis_curve_tmpl.middle_qs()		{ return middle_qs_;}
	quote_side  	disc_basis_curve_tmpl.long_qs()			{ return long_qs_;}
	quote_side  	disc_basis_curve_tmpl.ois_lib_basis_qs() { return ois_lib_basis_qs_;}
	quote_side  	disc_basis_curve_tmpl.lib_swap_qs()		{ return lib_swap_qs_;}
			
	curve disc_basis_curve_tmpl.short_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = short_qs_;
		return get_curve(short_crv_,td,qs, err);
	}
	
	curve  disc_basis_curve_tmpl.middle_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = middle_qs_;
		return get_curve(middle_crv_,td,qs,err);
	}
	curve  disc_basis_curve_tmpl.long_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = long_qs_;
		return get_curve(long_crv_,td,qs,err);
	}
	
	curve disc_basis_curve_tmpl.ois_lib_basis_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = ois_lib_basis_qs_;
		return get_curve(ois_lib_basis_crv_,td,qs,err);
	}
	curve disc_basis_curve_tmpl.lib_swap_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = lib_swap_qs_;
		return get_curve(lib_swap_crv_,td,qs,err);
	}

	
	disc_basis_curve_tmpl.disc_basis_curve_tmpl(curve_name option(nullable) 	short_crv,
												curve_name option(nullable) 	middle_crv,
												curve_name option(nullable)		long_crv,	
												curve_name 						lib_swap_crv,
												curve_name 						ois_lib_basis_crv,	
												quote_side option(nullable) 	short_qs,
												quote_side option(nullable) 	middle_qs,
												quote_side option(nullable) 	long_qs,
												quote_side option(nullable) 	lib_swap_qs,
												quote_side option(nullable) 	ois_lib_basis_qs)
												:curve_tmpl(swap_curve_type.SCE_DISC_B1_FLTOIS),
												 short_crv_(null_curve_name(short_crv)),
												 middle_crv_(null_curve_name(middle_crv)),
												 long_crv_(null_curve_name(long_crv)),
												 lib_swap_crv_(null_curve_name(lib_swap_crv)),
												 ois_lib_basis_crv_(null_curve_name(ois_lib_basis_crv)),
												 								 
												 short_qs_(short_qs),
												 middle_qs_(middle_qs),long_qs_(long_qs),
												 lib_swap_qs_(lib_swap_qs),
												 ois_lib_basis_qs_(ois_lib_basis_qs)
	{}


	//-------------------------------------------------------
	// class fixing_curve_tmpl
	//-------------------------------------------------------
	public class fixing_curve_tmpl : public curve_tmpl
	{
	   	public:
		fixing_curve_tmpl(	curve_name );

		override swap_curve_type type() ;
		override logical 		check(out error_info) ;		
		override vector(string) curve_names();
		curve_name 				fix_crv();
		fixing_curve 			fix_crv(date,out error_info err);

		protected:
		
		curve_name 	fix_crv_;
	};

	logical fixing_curve_tmpl.check(out error_info err)
	{
		if( !__exist_curve(fix_crv_)){
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name","fixing_curve_tmpl.check" );
			return false;
		}
		return true;
	}

	fixing_curve_tmpl.fixing_curve_tmpl(curve_name fix_crv)
										: curve_tmpl(swap_curve_type.SCE_FIX), fix_crv_(null_curve_name(fix_crv))								 
	{}

	vector(string) fixing_curve_tmpl.curve_names() 	{ return [fix_crv_];}
	swap_curve_type fixing_curve_tmpl.type() 		{ return type_;}
	curve_name 		fixing_curve_tmpl.fix_crv() 	{ return fix_crv_;}
	
	fixing_curve fixing_curve_tmpl.fix_crv(date td, out error_info err)
	{		
		curve c = get_curve(fix_crv_,td,null<quote_side>,err);
		if(err.is_error())
			return null<fixing_curve>;
		return fixing_curve(c);
	}

	//-------------------------------------------------------
	// class fwd_curve_tmpl
	//-------------------------------------------------------
	public class fwd_curve_tmpl : public curve_tmpl
	{
	   	public:

		//fwd_curve_tmpl(curve_name short_crv,fwd_func_tenor ,quote_side );
		override swap_curve_type type() ;
		override logical check(out error_info) ;
		override vector(string) curve_names();
		fwd_curve_tmpl(	curve_name option(nullable)	short_crv,
						curve_name option(nullable) middle_crv,
						curve_name option(nullable)	long_crv,												
						fwd_func_tenor fwd_tenor,						
						quote_side option(nullable) short_qs,
						quote_side option(nullable) middle_qs,
						quote_side option(nullable) long_qs,
						logical option(nullable) short_crv_is_fix,
						instrument_name option(nullable) fixing_instr);
		
		
		curve_name 		short_crv();
		curve_name  	middle_crv();
		curve_name 		long_crv();
		curve 			short_crv(date,quote_side option(nullable) , out error_info err);
		curve 			middle_crv(date,quote_side option(nullable) , out error_info err);
		curve 			long_crv(date,quote_side option(nullable), out error_info err);
		fixing_curve 	fixing_crv(date,quote_side option(nullable) , out error_info err);		
		override fwd_func_tenor fwd_tenor();		
		quote_side  	short_qs();
		quote_side  	middle_qs();
		quote_side  	long_qs();		
		override logical short_crv_is_fix();
		override instrument_name fixing_instr();
		
		protected:
		
		curve_name 	short_crv_;
		curve_name  middle_crv_;
		curve_name 	long_crv_;		
		fwd_func_tenor fwd_tenor_;	
		quote_side  short_qs_;
		quote_side  middle_qs_;
		quote_side 	long_qs_;
		logical 	short_crv_is_fix_;
		instrument_name fixing_instr_;
	};

	logical fwd_curve_tmpl.check(out error_info err)
	{
		logical sh_in = !null(short_crv_)  ;
		logical mi_in = !null(middle_crv_)  ;
		logical lo_in = !null(long_crv_)  ;
		if(!sh_in  && !mi_in && ! lo_in){			
			CORE_INT.add_error_info(err,ERR_T_INIT,"missing curves","fwd_curve_tmpl.check" );
			return false;
		}

		if( sh_in && !__exist_curve(short_crv_)){
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (short end curve)","fwd_curve_tmpl.check" );
			return false;
		}
		if( mi_in && !__exist_curve(middle_crv_)) {
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (middle curve)","fwd_curve_tmpl.check" );
			return false;
		}
		if( lo_in && !__exist_curve(long_crv_)){
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (long end curve)","fwd_curve_tmpl.check" );
			return false;
		}
		if( sh_in && null(short_crv_is_fix_)){
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid fixing flag for short end curve","fwd_curve_tmpl.check" );
			return false;
		}
		return true;
	}

	vector(string) fwd_curve_tmpl.curve_names(){
		vector(string) res;
		if(!null(short_crv_))
			push_back(res,short_crv_);
		if(!null(middle_crv_)  )
			push_back(res,middle_crv_);
		if(!null(long_crv_)  )
			push_back(res,long_crv_);
		return res;
	}
	
	fwd_curve_tmpl.fwd_curve_tmpl(	curve_name option(nullable) short_crv,
									curve_name option(nullable) middle_crv,
									curve_name option(nullable)	long_crv,																		
									fwd_func_tenor fwd_tenor,									
									quote_side option(nullable) short_qs,
									quote_side option(nullable) middle_qs,
									quote_side option(nullable) long_qs,
									logical option(nullable) short_crv_is_fix,
									instrument_name option(nullable) fixing_instr)
									:curve_tmpl(swap_curve_type.SCE_FWD),
									 short_crv_(null_curve_name(short_crv)),
									 middle_crv_(null_curve_name(middle_crv)),
									 long_crv_(null_curve_name(long_crv)),
									 fwd_tenor_(fwd_tenor),
									 short_qs_(short_qs),middle_qs_(middle_qs), long_qs_(long_qs),
									 short_crv_is_fix_(short_crv_is_fix), fixing_instr_(fixing_instr)
								 
	{}
	swap_curve_type fwd_curve_tmpl.type() 				{ return type_;}
	curve_name 		fwd_curve_tmpl.short_crv() 			{ return short_crv_;}	
	curve_name  	fwd_curve_tmpl.middle_crv()			{ return middle_crv_;}
	curve_name 		fwd_curve_tmpl.long_crv()			{ return long_crv_;}
	fwd_func_tenor fwd_curve_tmpl.fwd_tenor()	{ return fwd_tenor_;}	
	quote_side  	fwd_curve_tmpl.short_qs()			{ return short_qs_;}
	quote_side  	fwd_curve_tmpl.middle_qs()			{ return middle_qs_;}
	quote_side  	fwd_curve_tmpl.long_qs()			{ return long_qs_;}
	logical 		fwd_curve_tmpl.short_crv_is_fix()	{ return short_crv_is_fix_;}
	instrument_name fwd_curve_tmpl.fixing_instr()		{ return fixing_instr_;}
	curve fwd_curve_tmpl.short_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = short_qs_;
		return get_curve(short_crv_,td,qs,err);
	}
	
	curve fwd_curve_tmpl.middle_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = middle_qs_;
		return get_curve(middle_crv_,td,qs,err);
	}
	
	curve fwd_curve_tmpl.long_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = long_qs_;
		return get_curve(long_crv_,td,qs,err);
	}
	
	fixing_curve fwd_curve_tmpl.fixing_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(!short_crv_is_fix_ || null(short_crv_))
			null<fixing_curve>;
		
		if(null(qs))
			qs = short_qs_;
		curve c = get_curve(short_crv_,td,qs,err);
		if(err.is_error())
			return null<fixing_curve>;
		return fixing_curve(c);
	}

	//-------------------------------------------------------
	// class fwd_basis_curve_tmpl
	//-------------------------------------------------------
	public class fwd_basis_curve_tmpl : public curve_tmpl
	{
	   	public:
		override swap_curve_type type() ;
		override logical check(out error_info) ;
		override vector(string) curve_names();
		
		fwd_basis_curve_tmpl(	fwd_curve_tmpl,
								curve_name option(nullable)	short_crv,
								curve_name option(nullable) middle_crv,
								curve_name 					basis_crv,
								fwd_func_tenor fwd_tenor,						
								quote_side option(nullable) short_qs,
								quote_side option(nullable) middle_qs,
								quote_side option(nullable) basis_qs,
								logical option(nullable) short_crv_is_fix,
								logical option(nullable) ip_to_swap_base,
								instrument_name option(nullable) fixing_instr);

		
		fwd_curve_tmpl 	fwd_curve_tmpl_base();
		curve_name 		short_crv();
		curve_name  	middle_crv();
		curve_name 		basis_crv();
		curve_name 		ip_crv();
		curve 			short_crv(date,quote_side option(nullable) , out error_info err);
		curve 			middle_crv(date,quote_side option(nullable) , out error_info err);
		curve 			basis_crv(date,quote_side option(nullable) , out error_info err);
		curve 			ip_crv(date,out error_info err);
		fixing_curve 	fixing_crv(date,quote_side option(nullable) , out error_info err);		
		override fwd_func_tenor fwd_tenor();
		fwd_func_tenor fwd_tenor_base();
		quote_side  	short_qs();
		quote_side  	middle_qs();
		quote_side  	basis_qs();		
		override logical  short_crv_is_fix();
		logical			ip_to_swap_base();
		override instrument_name fixing_instr();
		protected:

		fwd_curve_tmpl 	fwd_in_;
		curve_name 		short_crv_;
		curve_name  	middle_crv_;
		curve_name 		basis_crv_;
		fwd_func_tenor fwd_tenor_;	
		quote_side  	short_qs_;
		quote_side  	middle_qs_;
		quote_side  	basis_qs_;
		logical 		short_crv_is_fix_;
		logical			ip_to_swap_base_;
		instrument_name fixing_instr_;
		
	};

	logical fwd_basis_curve_tmpl.check(out error_info err)  {
		

		if(!fwd_in_.check(err))
			return false;
		
		logical sh_in = !null(short_crv_)  ;
		logical mi_in = !null(middle_crv_)  ;
		logical lo_in = !null(basis_crv_)  ;
		if(!sh_in  && !mi_in && ! lo_in){			
			CORE_INT.add_error_info(err,ERR_T_INIT,"missing curves","fwd_basis_curve_tmpl.check" );
			return false;
		}

		if( sh_in && !__exist_curve(short_crv_)){	
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (short end curve)","fwd_basis_curve_tmpl.check" );
			return false;
		}
		if( mi_in && !__exist_curve(middle_crv_)) {
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (middle curve)","fwd_basis_curve_tmpl.check" );
			return false;
		}
		if( lo_in && !__exist_curve(basis_crv_)){
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (basis curve)","fwd_basis_curve_tmpl.check" );
			return false;
		}
		
		return true;
	}

	vector(string) fwd_basis_curve_tmpl.curve_names(){
		vector(string) res = fwd_in_.curve_names();
		if(!null(short_crv_))
			push_back(res,short_crv_);
		if(!null(middle_crv_)  )
			push_back(res,middle_crv_);
		if(!null(basis_crv_)  )
			push_back(res,basis_crv_);
		return res;
	}
	
	fwd_basis_curve_tmpl.fwd_basis_curve_tmpl(	fwd_curve_tmpl 				fwd_in,
												curve_name option(nullable) short_crv,
												curve_name option(nullable) middle_crv,
												curve_name 					basis_crv,
												fwd_func_tenor fwd_tenor,									
												quote_side option(nullable) short_qs,
												quote_side option(nullable) middle_qs,
												quote_side option(nullable) basis_qs,
												logical option(nullable) 	short_crv_is_fix,
												logical option(nullable) 	ip_to_swap_base,
												instrument_name option(nullable) fixing_instr)
												:curve_tmpl(swap_curve_type.SCE_FWD_B1_F),
												 fwd_in_(fwd_in),
												 short_crv_(null_curve_name(short_crv)),
												 middle_crv_(null_curve_name(middle_crv)),
												 basis_crv_(null_curve_name(basis_crv)),											
												 fwd_tenor_(fwd_tenor),
												 short_qs_(short_qs),middle_qs_(middle_qs), basis_qs_(basis_qs),
												 short_crv_is_fix_(short_crv_is_fix),
												 ip_to_swap_base_(ip_to_swap_base), fixing_instr_(fixing_instr)
	{}
	
	swap_curve_type fwd_basis_curve_tmpl.type() 			{ return type_;}
	curve_name 		fwd_basis_curve_tmpl.short_crv() 		{ return short_crv_;}	
	curve_name  	fwd_basis_curve_tmpl.middle_crv()		{ return middle_crv_;}
	curve_name 		fwd_basis_curve_tmpl.basis_crv()		{ return basis_crv_;}
	curve_name 		fwd_basis_curve_tmpl.ip_crv()			{ return ip_to_swap_base_? fwd_in_.long_crv() : null<curve_name>;}
	fwd_func_tenor fwd_basis_curve_tmpl.fwd_tenor_base() { return fwd_in_.fwd_tenor();}
	fwd_func_tenor fwd_basis_curve_tmpl.fwd_tenor() { return fwd_tenor_;}	
	quote_side  	fwd_basis_curve_tmpl.short_qs()			{ return short_qs_;}
	quote_side  	fwd_basis_curve_tmpl.middle_qs()		{ return middle_qs_;}
	quote_side  	fwd_basis_curve_tmpl.basis_qs()			{ return basis_qs_;}
	logical 		fwd_basis_curve_tmpl.short_crv_is_fix()	{ return short_crv_is_fix_;}
	fwd_curve_tmpl 	fwd_basis_curve_tmpl.fwd_curve_tmpl_base(){ return fwd_in_;}
	logical			fwd_basis_curve_tmpl.ip_to_swap_base()	{ return ip_to_swap_base_;}
	instrument_name fwd_basis_curve_tmpl.fixing_instr()		{ return fixing_instr_;}
	curve fwd_basis_curve_tmpl.short_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = short_qs_;
		return get_curve(short_crv_,td,qs,err);
	}
	
	curve fwd_basis_curve_tmpl.middle_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = middle_qs_;
		return get_curve(middle_crv_,td,qs,err);
	}
	
	curve fwd_basis_curve_tmpl.basis_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = basis_qs_;
		return get_curve(basis_crv_,td,qs,err);
	}

	curve fwd_basis_curve_tmpl.ip_crv(date td, out error_info err)
	{
		if(!ip_to_swap_base_)
			return null<curve>;
				
		return get_curve(ip_crv(),td,basis_qs_,err);
	}
	
	fixing_curve fwd_basis_curve_tmpl.fixing_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(!short_crv_is_fix_ || null(short_crv_))
			null<fixing_curve>;
		
		if(null(qs))
			qs = short_qs_;
		
		curve c = get_curve(short_crv_,td,qs,err);
		if(err.is_error())
			return null<fixing_curve>;
		
		return fixing_curve(c);
	}



	//-------------------------------------------------------
	// class fwd_basis_2swp_curve_tmpl
	//-------------------------------------------------------
	public class fwd_basis_2swp_curve_tmpl : public curve_tmpl
	{
	   	public:

		override swap_curve_type type() ;
		override logical check(out error_info) ;
		override vector(string) curve_names();
		fwd_curve_tmpl 	fwd_curve_tmpl_base();
		fwd_basis_2swp_curve_tmpl(	fwd_curve_tmpl fwd_in,
									curve_name option(nullable)	short_crv,
									curve_name option(nullable) middle_crv,
									curve_name 	basis_crv,
									fwd_func_tenor fwd_tenor,						
									quote_side option(nullable) short_qs,
									quote_side option(nullable) middle_qs,
									quote_side option(nullable) basis_qs,
									logical option(nullable) short_crv_is_fix,
									instrument_name option(nullable) fixing_instr);
		
		
		curve_name 		short_crv();
		curve_name  	middle_crv();
		curve_name 		long_base_crv();
		curve_name 		basis_crv();
		curve 			short_crv(date,quote_side option(nullable) , out error_info err);
		curve 			middle_crv(date,quote_side option(nullable) , out error_info err);
		curve 			long_base_crv(date,quote_side option(nullable) , out error_info err);
		curve 			basis_crv(date,quote_side option(nullable) , out error_info err);
		fixing_curve 	fixing_crv(date,quote_side option(nullable) , out error_info err);		
		override fwd_func_tenor fwd_tenor();		
		quote_side  	short_qs();
		quote_side  	middle_qs();
		quote_side  	basis_qs();		
		override logical  short_crv_is_fix();
		override instrument_name fixing_instr();
		protected:
		
		fwd_curve_tmpl 	fwd_in_;
		curve_name 		short_crv_;
		curve_name  	middle_crv_;		
		curve_name 		basis_crv_;
		fwd_func_tenor fwd_tenor_;	
		quote_side  	short_qs_;
		quote_side  	middle_qs_;
		quote_side  	basis_qs_;
		logical 		short_crv_is_fix_;
		instrument_name fixing_instr_;
	};

	logical fwd_basis_2swp_curve_tmpl.check(out error_info err)
	{

		if(!fwd_in_.check(err))
			return false;
		
		if(null(fwd_in_.long_crv())) {		
		   CORE_INT.add_error_info(err,ERR_T_INIT,"invalid long end base curve","fwd_basis_2swp_curve_tmpl.check" );
		   return false;
		}
		
		logical sh_in = !null(short_crv_)  ;
		logical mi_in = !null(middle_crv_)  ;
		logical lo_in = !null(basis_crv_)  ;
		if(!sh_in  && !mi_in && ! lo_in){
			CORE_INT.add_error_info(err,ERR_T_INIT,"missing curves","fwd_basis_2swp_curve_tmpl.check" );
			return false;
		}

		if( sh_in && !__exist_curve(short_crv_)){
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (short end curve)","fwd_basis_2swp_curve_tmpl.check" );
			return false;
		}
		if( mi_in && !__exist_curve(middle_crv_)) {
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (middle curve)","fwd_basis_2swp_curve_tmpl.check" );
			return false;
		}
		if( lo_in && !__exist_curve(basis_crv_)){
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (basis curve)","fwd_basis_2swp_curve_tmpl.check" );
			return false;
		}
		
		return true;
	}

	vector(string) fwd_basis_2swp_curve_tmpl.curve_names(){
		vector(string) res = fwd_in_.curve_names();
		if(!null(short_crv_))
			push_back(res,short_crv_);
		if(!null(middle_crv_)  )
			push_back(res,middle_crv_);
		if(!null(basis_crv_)  )
			push_back(res,basis_crv_);
		return res;
	}
	
	fwd_basis_2swp_curve_tmpl.fwd_basis_2swp_curve_tmpl(	fwd_curve_tmpl 				fwd_in,
															curve_name option(nullable) short_crv,
															curve_name option(nullable) middle_crv,															
															curve_name 					basis_crv,
															fwd_func_tenor fwd_tenor,									
															quote_side option(nullable) short_qs,
															quote_side option(nullable) middle_qs,
															quote_side option(nullable) basis_qs,
															logical option(nullable) 	short_crv_is_fix,
															instrument_name option(nullable) fixing_instr)
															:curve_tmpl(swap_curve_type.SCE_FWD_B2_F),
															 fwd_in_(fwd_in),
															 short_crv_(null_curve_name(short_crv)),
															 middle_crv_(null_curve_name(middle_crv)),														
															 basis_crv_(null_curve_name(basis_crv)),
															 fwd_tenor_(fwd_tenor),
															 short_qs_(short_qs),middle_qs_(middle_qs), basis_qs_(basis_qs),
															 short_crv_is_fix_(short_crv_is_fix), fixing_instr_(fixing_instr)								
	{}
	
	swap_curve_type fwd_basis_2swp_curve_tmpl.type() 			{ return type_;}
	curve_name 		fwd_basis_2swp_curve_tmpl.short_crv() 		{ return short_crv_;}	
	curve_name  	fwd_basis_2swp_curve_tmpl.middle_crv()		{ return middle_crv_;}
	curve_name 		fwd_basis_2swp_curve_tmpl.long_base_crv()	{ return fwd_in_.long_crv();}
	curve_name 		fwd_basis_2swp_curve_tmpl.basis_crv()		{ return basis_crv_;}
	fwd_func_tenor fwd_basis_2swp_curve_tmpl.fwd_tenor()	{ return fwd_tenor_;}	
	quote_side  	fwd_basis_2swp_curve_tmpl.short_qs()		{ return short_qs_;}
	quote_side  	fwd_basis_2swp_curve_tmpl.middle_qs()		{ return middle_qs_;}
	quote_side  	fwd_basis_2swp_curve_tmpl.basis_qs()		{ return basis_qs_;}
	logical 		fwd_basis_2swp_curve_tmpl.short_crv_is_fix(){ return short_crv_is_fix_;}
	fwd_curve_tmpl 	fwd_basis_2swp_curve_tmpl.fwd_curve_tmpl_base()	{ return fwd_in_;}
	instrument_name fwd_basis_2swp_curve_tmpl.fixing_instr()	{ return fixing_instr_;}
	
	curve fwd_basis_2swp_curve_tmpl.short_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = short_qs_;
		return get_curve(short_crv_,td,qs,err);
	}
	
	curve fwd_basis_2swp_curve_tmpl.middle_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = middle_qs_;
		return get_curve(middle_crv_,td,qs,err);
	}
	
	curve fwd_basis_2swp_curve_tmpl.long_base_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = basis_qs_;
		return get_curve(long_base_crv(),td,qs,err);
	}

	curve fwd_basis_2swp_curve_tmpl.basis_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = basis_qs_;
		return get_curve(basis_crv_,td,qs,err);
	}
	
	fixing_curve fwd_basis_2swp_curve_tmpl.fixing_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(!short_crv_is_fix_ || null(short_crv_))
			null<fixing_curve>;
		
		if(null(qs))
			qs = short_qs_;

		curve c = get_curve(short_crv_,td,qs,err);
		if(err.is_error())
			return null<fixing_curve>;
		
		return fixing_curve(c);
	}
	
//**********************************
//**********************************

	//-------------------------------------------------------
	// class currbasis_curve_tmpl
	//-------------------------------------------------------
	public class currbasis_curve_tmpl : public curve_tmpl
	{
	   	public:

		override swap_curve_type type() ;
		override logical check(out error_info) ;
		override vector(string) curve_names();
		curve_tmpl 	fwd_curve_tmpl_flat();
		curve_tmpl 	fwd_curve_tmpl_sprd();
		
		currbasis_curve_tmpl(curve_tmpl option(fixedref) fwd_flat,
							curve_tmpl option(fixedref) fwd_sprd,
							curve_name option(nullable)	fx_swap_crv,								
							curve_name 	option(nullable) basis_crv,
							instrument_name fx_spot_name,
							quote_side option(nullable) fx_swap_qs,								
							quote_side option(nullable) basis_qs,
							quote_side option(nullable) fx_spot_qs);
		
		
		curve_name 		fx_swap_crv();
		curve_name 		basis_crv();
		curve 			fx_swap_crv(date,quote_side option(nullable) , out error_info );
		curve 			basis_crv(date,quote_side option(nullable) , out error_info );
		quote_side  	fx_swap_qs();
		quote_side  	basis_qs();
		quote_side 		fx_spot_qs();
		instrument_name fx_spot_name();
		fx_spot 		fx_sp(date , quote_side option(nullable) qs, out error_info );
		instrument_name fixing_instr_flat();
		instrument_name fixing_instr_sprd();
		quote_side 		fixing_instr_qs();
		protected:
		
		curve_tmpl 	option(fixedref) 	fwd_flat_;
		curve_tmpl 	option(fixedref) 	fwd_sprd_;
		curve_name 		fx_swap_crv_;		
		curve_name 		basis_crv_;
		instrument_name fx_spot_name_;
		quote_side  	fx_swap_qs_;
		quote_side  	basis_qs_;
		quote_side 		fx_spot_qs_;
		quote_side		fixing_instr_qs_;
	};

	logical currbasis_curve_tmpl.check(out error_info err)
	{

		if(!fwd_flat_.check(err))
			return false;

		if(!fwd_sprd_.check(err))
			return false;

		if(null(fwd_flat_.fixing_instr())){
			CORE_INT.add_error_info(err,ERR_T_INIT,"fixing instrument missing (flat leg)","currbasis_curve_tmpl.check" );
			return false;
		}

		if(null(fwd_sprd_.fixing_instr())){
			CORE_INT.add_error_info(err,ERR_T_INIT,"fixing instrument missing (spread leg)","currbasis_curve_tmpl.check" );
			return false;
		}
		
		/*if(null(fwd_in_.swap_crv())) {		
		   logical e = CORE_INT.add_error_info(err,ERR_T_INIT,"invalid swap base curve","fwd_basis_2swp_curve_tmpl.check" );
		   return false;
		}*/
		
		logical fx_in = !null(fx_swap_crv_)  ;		
		logical lo_in = !null(basis_crv_)  ;
		if(!fx_in && ! lo_in){
			CORE_INT.add_error_info(err,ERR_T_INIT,"missing curves","currbasis_curve_tmpl.check" );
			return false;
		}

		if( fx_in && !__exist_curve(fx_swap_crv_)){
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (fx swap curve)","currbasis_curve_tmpl.check" );
			return false;
		}
		
		if( lo_in && !__exist_curve(basis_crv_)){
			CORE_INT.add_error_info(err,ERR_T_INIT,"invalid name (basis curve)","currbasis_curve_tmpl.check" );
			return false;
		}
		
		return true;
	}

	vector(string) currbasis_curve_tmpl.curve_names(){
		vector(string) res = fwd_flat_.curve_names();
		vector(string) res_s = fwd_sprd_.curve_names();
		if(null(res))
			res = res_s;
		else if(!null(res_s))
			res = concat(res,res_s);
		
		if(!null(fx_swap_crv_))
			push_back(res,fx_swap_crv_);	
		if(!null(basis_crv_)  )
			push_back(res,basis_crv_);
		return res;
	}
	
	currbasis_curve_tmpl.currbasis_curve_tmpl(	curve_tmpl option(fixedref) fwd_flat,
											curve_tmpl option(fixedref) fwd_sprd,
											curve_name option(nullable)	fx_swap_crv,								
											curve_name 	option(nullable) basis_crv,
											instrument_name fx_spot_name,
											quote_side option(nullable) fx_swap_qs,								
											quote_side option(nullable) basis_qs,
											quote_side option(nullable) fx_spot_qs)
											:curve_tmpl(swap_curve_type.SCE_CURR_B1_F),
											 fwd_flat_(fwd_flat),
											 fwd_sprd_(fwd_sprd),
											 fx_swap_crv_(null_curve_name(fx_swap_crv)),																							
											 basis_crv_(null_curve_name(basis_crv)),
											 fx_spot_name_(fx_spot_name),
											 fx_swap_qs_(fx_swap_qs),
											 basis_qs_(basis_qs),
											 fx_spot_qs_(fx_spot_qs),fixing_instr_qs_("ASK")
	{}
	
	swap_curve_type currbasis_curve_tmpl.type() 			{ return type_;}
	curve_name 		currbasis_curve_tmpl.fx_swap_crv() 		{ return fx_swap_crv_;}	
	
	//curve_name 	currbasis_curve_tmpl.swap_crv()			{ return fwd_in_.swap_crv();}
	curve_name 		currbasis_curve_tmpl.basis_crv()		{ return basis_crv_;}
	instrument_name currbasis_curve_tmpl.fx_spot_name()		{ return fx_spot_name_;}
	quote_side  	currbasis_curve_tmpl.fx_swap_qs()		{ return fx_swap_qs_;}
	quote_side  	currbasis_curve_tmpl.basis_qs()			{ return basis_qs_;}
	quote_side  	currbasis_curve_tmpl.fx_spot_qs()		{ return fx_spot_qs_;}
	curve_tmpl 		currbasis_curve_tmpl.fwd_curve_tmpl_flat()	{ return fwd_flat_;}
	curve_tmpl 		currbasis_curve_tmpl.fwd_curve_tmpl_sprd()	{ return fwd_sprd_;}
	instrument_name currbasis_curve_tmpl.fixing_instr_flat()	{ return fwd_flat_.fixing_instr();}
	instrument_name currbasis_curve_tmpl.fixing_instr_sprd()	{ return fwd_sprd_.fixing_instr();}
	quote_side  	currbasis_curve_tmpl.fixing_instr_qs()		{ return fixing_instr_qs_;}
	
	curve currbasis_curve_tmpl.fx_swap_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = fx_swap_qs_;
		return get_curve(fx_swap_crv_,td,qs,err);
	}

	curve currbasis_curve_tmpl.basis_crv(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = basis_qs_;
		return get_curve(basis_crv_,td,qs,err);
	}

	fx_spot currbasis_curve_tmpl.fx_sp(date td, quote_side option(nullable) qs, out error_info err)
	{
		if(null(qs))
			qs = fx_spot_qs_;
		return ..fx_spot(fx_spot_name_,td,qs,err);
	}
	
	
}